//
// "$Id: Fl_win32.cxx,v 1.33.2.37 2001/04/27 15:43:38 easysw Exp $"
//
// OS/2-specific code for the Fast Light Tool Kit (FLTK).
//
// Copyright 1998-2001 by Bill Spitzak and others.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.
//
// Please report all bugs and problems to "fltk-bugs@fltk.org".
//

// This file contains OS/2-specific code for fltk which is always linked
// in.  Search other files for "OS2PM" or filenames ending in _os2.cxx
// for other system-specific code.

#include <config.h>
#include <FL/Fl.H>
#include <FL/os2.H>
#include <FL/Fl_Window.H>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <time.h>
#include <ctype.h>

////////////////////////////////////////////////////////////////
// interface to poll/select call:

static fd_set fdsets[3];

#define POLLIN 1
#define POLLOUT 4
#define POLLERR 8

static int nfds = 0;
static int fd_array_size = 0;

static struct FD
{
  int fd;
  short events;
  void (*cb)(int, void*);
  void* arg;
} *fd = 0;

void Fl::add_fd(int n, int events, void (*cb)(int, void*), void *v)
{
  remove_fd(n,events);
  int i = nfds++;
  if (i >= fd_array_size)
  {
    fd_array_size = 2*fd_array_size+1;
    fd = (FD*)realloc(fd, fd_array_size*sizeof(FD));
  }
  fd[i].fd = n;
  fd[i].events = events;
  fd[i].cb = cb;
  fd[i].arg = v;

  if (events & POLLIN) FD_SET(n, &fdsets[0]);
  if (events & POLLOUT) FD_SET(n, &fdsets[1]);
  if (events & POLLERR) FD_SET(n, &fdsets[2]);
}

void Fl::add_fd(int fd, void (*cb)(int, void*), void* v)
{
  Fl::add_fd(fd, POLLIN, cb, v);
}

void Fl::remove_fd(int n, int events)
{
  int i,j;
  for (i=j=0; i<nfds; i++)
  {
    if (fd[i].fd == n)
    {
      int e = fd[i].events & ~events;
      if (!e) continue; // if no events left, delete this fd
      fd[i].events = e;
    }
    // move it down in the array if necessary:
    if (j<i)
    {
      fd[j]=fd[i];
    }
    j++;
  }
  nfds = j;

  if (events & POLLIN) FD_CLR(unsigned(n), &fdsets[0]);
  if (events & POLLOUT) FD_CLR(unsigned(n), &fdsets[1]);
  if (events & POLLERR) FD_CLR(unsigned(n), &fdsets[2]);
}

void Fl::remove_fd(int n)
{
  remove_fd(n, -1);
}

QMSG fl_msg;
HAB fl_pm_hab;

// This is never called with time_to_wait < 0.0.
// It *should* return negative on error, 0 if nothing happens before
// timeout, and >0 if any callbacks were done.  This version only
// returns zero if nothing happens during a 0.0 timeout, otherwise
// it returns 1.
int fl_wait(double time_to_wait)
{
  int have_message = 0;
  int timerid;

  if (nfds)
  {
    // For OS/2 we need to poll for socket input FIRST, since
    // the event queue is not something we can select() on...
    timeval t;
    t.tv_sec = 0;
    t.tv_usec = 0;

    fd_set fdt[3];
    fdt[0] = fdsets[0];
    fdt[1] = fdsets[1];
    fdt[2] = fdsets[2];

    if (::select(0,&fdt[0],&fdt[1],&fdt[2],&t))
    {
      // We got something - do the callback!
      for (int i = 0; i < nfds; i ++)
      {
        int f = fd[i].fd;
        short revents = 0;
        if (FD_ISSET(f,&fdt[0])) revents |= POLLIN;
        if (FD_ISSET(f,&fdt[1])) revents |= POLLOUT;
        if (FD_ISSET(f,&fdt[2])) revents |= POLLERR;
        if (fd[i].events & revents) fd[i].cb(f, fd[i].arg);
      }
      time_to_wait = 0.0; // just peek for any messages
    }
    else
    {
      // we need to check them periodically, so set a short timeout:
      if (time_to_wait > .001) time_to_wait = .001;
    }
  }

  if (time_to_wait < 2147483.648)
  {
    // Perform the requested timeout...
    have_message = WinPeekMsg(fl_pm_hab, &fl_msg, NULLHANDLE, 0, 0, PM_REMOVE);
    if (!have_message)
    {
      int t = (int)(time_to_wait * 1000.0 + .5);
      if (t <= 0) return 0; // too short to measure
      timerid = WinStartTimer(fl_pm_hab, NULLHANDLE, timerid, t);
      have_message = WinGetMsg(fl_pm_hab, &fl_msg, NULLHANDLE, 0, 0);
      WinStopTimer(fl_pm_hab,NULLHANDLE,timerid);
    }
  }
  else
  {
    have_message = WinGetMsg(fl_pm_hab, &fl_msg, NULLHANDLE, 0, 0);
//    have_message = WinPeekMsg(fl_pm_hab, &fl_msg, NULLHANDLE, 0, 0, PM_REMOVE);
  }

  // Execute the message we got, and all other pending messages:
  while (have_message)
  {
    //TranslateMessage(&fl_msg);
    WinDispatchMsg(fl_pm_hab, &fl_msg);
    have_message = WinPeekMsg(fl_pm_hab, &fl_msg, NULLHANDLE, 0, 0, PM_REMOVE);
  }

  // This should return 0 if only timer events were handled:
  return 1;
}

// fl_ready() is just like fl_wait(0.0) except no callbacks are done:
int fl_ready()
{
  if (WinPeekMsg(fl_pm_hab, &fl_msg, NULL, 0, 0, PM_NOREMOVE)) return 1;
  timeval t;
  t.tv_sec = 0;
  t.tv_usec = 0;
  fd_set fdt[3];
  fdt[0] = fdsets[0];
  fdt[1] = fdsets[1];
  fdt[2] = fdsets[2];
  return ::select(0,&fdt[0],&fdt[1],&fdt[2],&t);
}

////////////////////////////////////////////////////////////////
// On Win32, there is a notion of a workarea that might not be
// the entire screen.  I won't implement this notion on OS/2,
// though it may apply with regards to the Warp Center.
int Fl::x()
{
  return 0;
}

int Fl::y()
{
  LONG lValue;

  lValue = WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN);
  return(int)lValue;
}

int Fl::h()
{
  LONG lValue;

  lValue = WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN);
  return(int)lValue;
}

int Fl::w()
{
  LONG lValue;

  lValue = WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN);
  return(int)lValue;
}

void Fl::get_mouse(int &x, int &y)
{
  POINTL ptl;
  WinQueryPointerPos(HWND_DESKTOP, &ptl);
  //WinMapWindowPoints(HWND_DESKTOP, hwnd, &ptl, 1);
  x = ptl.x;
  y = ptl.y;
}

////////////////////////////////////////////////////////////////

HWND fl_capture;

static int mouse_event(Fl_Window *window, int what, int button,
                       LONG wParam, LONG lParam)
{

  static int px, py, pmx, pmy;
  POINTL pt;
  WinQueryPointerPos(HWND_DESKTOP, &pt);
  Fl::e_x_root = pt.x;
  Fl::e_y_root = Fl::h() - pt.y; // Need to subtract off the desktop height...?
  WinMapWindowPoints(HWND_DESKTOP, fl_xid(window), &pt, 1);
  Fl::e_x = pt.x;
  Fl::e_y = window->h() - pt.y; // Need to subtract off the window height...?
  while (window->parent())
  {
    Fl::e_x += window->x();
    Fl::e_y += window->y();
    window = window->window();
  }
  ulong state = Fl::e_state & 0xff0000; // keep shift key states
  if (button == 1) state |= FL_BUTTON1;
  if (button == 2) state |= FL_BUTTON2;
  if (button == 3) state |= FL_BUTTON3;
  Fl::e_state = state;

  switch (what)
  {
    case 1: // double-click
      if (Fl::e_is_click)
      {
        Fl::e_clicks++; goto J1;
      }
    case 0: // single-click
      Fl::e_clicks = 0;
      J1:
      if (!fl_capture) WinSetCapture(HWND_DESKTOP,fl_xid(window));
      Fl::e_keysym = FL_Button + button;
      Fl::e_is_click = 1;
      px = pmx = Fl::e_x_root; py = pmy = Fl::e_y_root;
      return Fl::handle(FL_PUSH,window);

    case 2: // release:
      if (!fl_capture) WinSetCapture(HWND_DESKTOP,NULLHANDLE);
      Fl::e_keysym = FL_Button + button;
      return Fl::handle(FL_RELEASE,window);

    case 3: // move:
    default:
      // MSWindows produces extra events even if mouse does not move, ignore em:
      if (Fl::e_x_root == pmx && Fl::e_y_root == pmy) return 1;
      pmx = Fl::e_x_root; pmy = Fl::e_y_root;
      if (abs(Fl::e_x_root-px)>5 || abs(Fl::e_y_root-py)>5) Fl::e_is_click = 0;
      return Fl::handle(FL_MOVE,window);
  }
}

// convert a MSWindows VK_x to an Fltk (X) Keysym:
// See also the inverse converter in Fl_get_key_win32.C
// This table is in numeric order by VK:
static const struct
{
  unsigned short vk, fltk, extended;
} vktab[] = {
  {VK_BACKSPACE,FL_BackSpace},
  {VK_TAB,  FL_Tab},
  {VK_NEWLINE,  FL_Enter, FL_KP_Enter},
  {VK_SHIFT,  FL_Shift_L, FL_Shift_R},
  {VK_CTRL, FL_Control_L, FL_Control_R},
  {VK_PAUSE,  FL_Pause},
  {VK_CAPSLOCK, FL_Caps_Lock},
  {VK_ESC,  FL_Escape},
  {VK_SPACE,  ' '},
  {VK_PAGEUP, FL_KP+'9',  FL_Page_Up},
  {VK_PAGEDOWN, FL_KP+'3',  FL_Page_Down},
  {VK_END,  FL_KP+'1',  FL_End},
  {VK_HOME, FL_KP+'7',  FL_Home},
  {VK_LEFT, FL_KP+'4',  FL_Left},
  {VK_UP, FL_KP+'8',  FL_Up},
  {VK_RIGHT,  FL_KP+'6',  FL_Right},
  {VK_DOWN, FL_KP+'2',  FL_Down},
  {VK_PRINTSCRN,FL_Print},
  {VK_INSERT, FL_KP+'0',  FL_Insert},
  {VK_DELETE, FL_KP+'.',  FL_Delete},
  {VK_SCRLLOCK, FL_Scroll_Lock},
  {VK_NUMLOCK,  FL_Num_Lock},
  {VK_ENTER, FL_Enter, FL_KP_Enter},
  {VK_CLEAR,  FL_KP+'5',  0xff0b/*XK_Clear*/},
/*
  {VK_F10,  FL_Menu},
  {VK_MENU, FL_Alt_L, FL_Alt_R},
  {VK_LWIN,	FL_Meta_L},
  {VK_RWIN,	FL_Meta_R},
  {VK_APPS,	FL_Menu},
  {VK_MULTIPLY,	FL_KP+'*'},
  {VK_ADD,	FL_KP+'+'},
  {VK_SUBTRACT,	FL_KP+'-'},
  {VK_DECIMAL,	FL_KP+'.'},
  {VK_DIVIDE,	FL_KP+'/'},
  {0xba,  ';'},
  {0xbb,  '='},
  {0xbc,  ','},
  {0xbd,  '-'},
  {0xbe,  '.'},
  {0xbf,  '/'},
  {0xc0,  '`'},
  {0xdb,  '['},
  {0xdc,  '\\'},
  {0xdd,  ']'},
  {0xde,  '\''}
*/
};

static int os2kfltk(int vk, int extended)
{
  static unsigned short vklut[256];
  static unsigned short extendedlut[256];
  if (!vklut[1])
  { // init the table
    unsigned int i;
    for (i = 0; i < 256; i++) vklut[i] = tolower(i);
    for (i=VK_F1; i<=VK_F16; i++) vklut[i] = i+(FL_F-(VK_F1-1));
//    for (i=VK_NUMPAD0; i<=VK_NUMPAD9; i++) vklut[i] = i+(FL_KP+'0'-VK_NUMPAD0);
    for (i = 0; i < sizeof(vktab)/sizeof(*vktab); i++)
    {
      vklut[vktab[i].vk] = vktab[i].fltk;
      extendedlut[vktab[i].vk] = vktab[i].extended;
    }
    for (i = 0; i < 256; i++) if (!extendedlut[i]) extendedlut[i] = vklut[i];
  }
  return extended ? extendedlut[vk] : vklut[vk];
}

extern HPAL fl_select_palette(void); // in fl_color_OS2.C

static Fl_Window* resize_bug_fix;

////////////////////////////////////////////////////////////////
// This function gets the dimensions of the top/left borders and
// the title bar, if there is one, based on the FL_BORDER, FL_MODAL
// and FL_NONMODAL flags, and on the window's size range.
// It returns the following values:
//
// value | border | title bar
//   0   |  none  |   no
//   1   |  fix   |   yes
//   2   |  size  |   yes

int Fl_X::fake_X_wm(const Fl_Window* w,int &X,int &Y, int &bt,int &bx, int &by)
{
  int W, H, xoff, yoff, dx, dy;
  int ret = bx = by = bt = 0;
  if (w->border() && !w->parent())
  {
    if (w->maxw != w->minw || w->maxh != w->minh)
    {
      ret = 2;
      bx = WinQuerySysValue(HWND_DESKTOP, SV_CXSIZEBORDER);
      by = WinQuerySysValue(HWND_DESKTOP, SV_CYSIZEBORDER);
    }
    else
    {
      ret = 1;
      bx = WinQuerySysValue(HWND_DESKTOP, SV_CXDLGFRAME);
      by = WinQuerySysValue(HWND_DESKTOP, SV_CYDLGFRAME);
    }
    bt = WinQuerySysValue(HWND_DESKTOP, SV_CYTITLEBAR);
  }
  //The coordinates of the whole window, including non-client area
  xoff = bx;
  yoff = by + bt;
  dx = 2*bx;
  dy = 2*by + bt;
  X = w->x()-xoff;
  Y = w->y()-yoff;
  W = w->w()+dx;
  H = w->h()+dy;

  //Proceed to positioning the window fully inside the screen, if possible
  //Make border's lower right corner visible
  if (Fl::w() < X+W) X = Fl::w() - W;
  if (Fl::h() < Y+H) Y = Fl::h() - H;
  //Make border's upper left corner visible
  if (X<0) X = 0;
  if (Y<0) Y = 0;
  //Make client area's lower right corner visible
  if (Fl::w() < X+dx+ w->w()) X = Fl::w() - w->w() - dx;
  if (Fl::h() < Y+dy+ w->h()) Y = Fl::h() - w->h() - dy;
  //Make client area's upper left corner visible
  if (X+xoff < 0) X = -xoff;
  if (Y+yoff < 0) Y = -yoff;
  //Return the client area's top left corner in (X,Y)
  X+=xoff;
  Y+=yoff;

  return ret;
}

////////////////////////////////////////////////////////////////

void Fl_Window::resize(int X,int Y,int W,int H)
{
  Fl_Window *fred = Fl_Window::current();
  if (fred)
  {
    int parentHeight = Fl_Window::current()->h() - 1;

    Fl_Group::resize(X,parentHeight - Y,W,H);
    if (shown())
    {
      redraw(); i->wait_for_expose = 1;
    }
  }
}

////////////////////////////////////////////////////////////////

void fl_fix_focus(); // in Fl.C

/*
 * Window procedure
 */

MRESULT EXPENTRY WinProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{

  Fl_Window *window = fl_find(hwnd);
      int iValidUSCH,
        iShiftActive,
        iCtrlActive,
        iAltActive,
        iKeyUp,
        iDeadKey,
        iLoneKey;
      UCHAR ucrepeat;
      UCHAR ucscancode;
      USHORT usch;
      USHORT usvk;
      ulong state;

#include "../../pmmsgs.cpp"

  switch (msg)
  {
    case WM_CLOSE: // user clicked close box
      Fl::handle(FL_CLOSE, window);
      return 0;
      break;
    case WM_PAINT:
      {
        Fl_X *i = Fl_X::i(window);

        i->wait_for_expose = 0;
        // We need to merge this damage into fltk's damage.  I do this in
        // reverse, telling Win32 about fltk's damage and then reading back
        // the new accumulated region.
        if (window->damage())
        {
          if (i->region)
          {
            WinInvalidateRegion(hwnd,i->region,FALSE); 
            WINGETLASTERROR("WinInvaliateRegion");
            WinQueryUpdateRegion(hwnd,i->region);
            WINGETLASTERROR("WinQueryUpdateRegion");
          }
/*
          else
          {
            i->region = XRectangleRegion(0,0,0,0);
            WINGETLASTERROR("GpiCreateRegion");
            WinQueryUpdateRegion(hwnd,i->region);
            WINGETLASTERROR("WinQueryUpdateRegion");
          }
*/
        }
        else
        {
          // If there is no region, the window damage comes from PM alone
          WinQueryUpdateRegion(hwnd,i->region);
        }
        window->clear_damage(window->damage()|FL_DAMAGE_EXPOSE);
        // These next two statements should not be here, so that all update
        // is deferred until Fl::flush() is called during idle.  However Win32
        // apparently is very unhappy if we don't obey it and draw right now.
        // Very annoying!
        i->flush();
        window->clear_damage();
        // This convinces PM that we have painted whatever they wanted
        // us to paint, and stops it from sending WM_PAINT messages:
        WinValidateRegion(hwnd,i->region,TRUE);
      }
      break;

    case 1055: //0x041e: WM_MOUSELEAVE:
      Fl::handle(FL_LEAVE, window);
      break;
    case WM_BUTTON1DOWN:
      mouse_event(window, 0, 1, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON1DBLCLK:
      mouse_event(window, 1, 1, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON1UP:    
      mouse_event(window, 2, 1, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON3DOWN:  
      mouse_event(window, 0, 2, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON3DBLCLK:
      mouse_event(window, 1, 2, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON3UP:    
      mouse_event(window, 2, 2, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON2DOWN:  
      mouse_event(window, 0, 3, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON2DBLCLK:
      mouse_event(window, 1, 3, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_BUTTON2UP:    
      mouse_event(window, 2, 3, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
    case WM_MOUSEMOVE:
      mouse_event(window, 3, 0, LONGFROMMP(mp1), LONGFROMMP(mp2));
      return 0;
      break;

    case WM_SETFOCUS:
      Fl::handle(FL_FOCUS, window);
      break;

    case WM_FOCUSCHANGE:
      if (SHORT1FROMMP(mp2))
      {
        Fl::handle(FL_UNFOCUS, window);
        Fl::flush(); // it never returns to main loop when deactivated...
      }
      break;

    case WM_CHAR:
      iValidUSCH = CHARMSG(&msg)->fs & KC_CHAR;
      iShiftActive = CHARMSG(&msg)->fs & KC_SHIFT;
      iCtrlActive = CHARMSG(&msg)->fs & KC_CTRL;
      iAltActive = CHARMSG(&msg)->fs & KC_ALT;
      iKeyUp = CHARMSG(&msg)->fs & KC_KEYUP;
      iDeadKey = CHARMSG(&msg)->fs & KC_DEADKEY;
      iLoneKey = CHARMSG(&msg)->fs & KC_LONEKEY;
      ucrepeat = CHAR3FROMMP(mp1);
      ucscancode = CHAR4FROMMP(mp1);
      usch = SHORT1FROMMP(mp2);
      usvk = SHORT2FROMMP(mp2);

      state = Fl::e_state & 0xff000000; // keep the mouse button state
      if (WinGetKeyState(HWND_DESKTOP,VK_SHIFT) & 0x8000) state |= FL_SHIFT;
      if (WinGetKeyState(HWND_DESKTOP,VK_CAPSLOCK) & 0x8000) state |= FL_CAPS_LOCK;
      if (WinGetKeyState(HWND_DESKTOP,VK_CTRL) & 0x8000) state |= FL_CTRL;
      if (WinGetKeyState(HWND_DESKTOP,VK_NUMLOCK) & 0x8000) state |= FL_NUM_LOCK;
      if (WinGetKeyState(HWND_DESKTOP,VK_SCRLLOCK) & 0x8000) state |= FL_SCROLL_LOCK;
      if (WinGetKeyState(HWND_DESKTOP,VK_ALT) & 0x8000) state |= FL_ALT;
      Fl::e_state = state;

      // save the keysym until we figure out the characters:
      Fl::e_keysym = os2kfltk(usvk,iAltActive);

      if ((CHARMSG(&msg)->fs & KC_KEYUP) == 0) // ignore up events after fixing shift
      {
          static char buffer[2];
          if (CHARMSG(&msg)->fs & KC_CHAR)
          {
            buffer[0] = CHARMSG(&msg)->chr;
            Fl::e_length = 1;
          }
          else if (Fl::e_keysym >= FL_KP && Fl::e_keysym <= FL_KP_Last)
          {
            buffer[0] = Fl::e_keysym-FL_KP;
            Fl::e_length = 1;
          }
          else
          {
            buffer[0] = 0;
            Fl::e_length = 0;
          }
          Fl::e_text = buffer;
          while (window->parent()) window = window->window();
          if (Fl::handle(FL_KEYBOARD,window)) return 0;
      }
      break;
    case WM_CREATE:
      fl_GetDC(hwnd);
      fl_select_palette();
      break;
    case WM_SIZE:
      if (!window->parent())
      {
        if (LONGFROMMP(mp1) != 0L)
        {
          Fl::handle(FL_SHOW, window);
          resize_bug_fix = window;
          window->size(SHORT1FROMMP(mp2), SHORT2FROMMP(mp2));
        }
      }
      break;

    default:
      if (Fl::handle(0,0))
      {
        return 0;
      }
      break;
  }
  return WinDefWindowProc(hwnd, msg, mp1, mp2);
}


char fl_show_iconic;  // hack for Fl_Window::iconic()

HWND fl_default_cursor;
int fl_disable_transient_for; // secret method of removing TRANSIENT_FOR

Fl_X* Fl_X::make(Fl_Window* w)
{
  Fl_Group::current(0); // get rid of very common user bug: forgot end()

  int parentHeight = 0;
  int needBorder = 0;

  HMQ hmq;

  if (!Fl::initYet)
  {
    // Initialize the Presentation Manager - needs to happen once
    // per thread.  Not sure this is the right place to do it...
    fl_pm_hab = WinInitialize(0);
    hmq = WinCreateMsgQueue(fl_pm_hab, 0);
    APIRET rc = WinRegisterClass(fl_pm_hab,
                                 (PSZ)"FLTK",
                                 WinProc,
                                 WS_VISIBLE,
                                 NULL);
    Fl::initYet = 1;
    Fl::resid = 2;
  }

  Fl_X* x = new Fl_X;

  fl_default_cursor = WinQuerySysPointer(HWND_DESKTOP, SPTR_ARROW, FALSE);
  HWND parent = HWND_DESKTOP;
  ULONG 
    showStyle = SWP_SIZE|SWP_ACTIVATE|SWP_SHOW,
    flStyle = WS_VISIBLE,
    flFrameFlags = FCF_TITLEBAR | FCF_SYSMENU | FCF_TASKLIST | FCF_SHELLPOSITION;

  int xp = w->x();
  int yp = w->y();
  int wp = w->w();
  int hp = w->h();

  int showit = 1;

  if (w->parent())
  {
    parentHeight = w->parent()->h();
    //style |= WS_CHILD;
    //styleEx |= WS_EX_WINDOWEDGE | WS_EX_CONTROLPARENT;
    flFrameFlags |= FCF_SIZEBORDER | FCF_MAXBUTTON;
    // We'd like to be able to give the user the opportuntity to tab among
    // this new child window and the parent window, but we'll see...
    parent = fl_xid(w->window());
  }
  else
  {
    if (!w->size_range_set)
    {
      if (w->resizable())
      {
        Fl_Widget *o = w->resizable();
        int minw = o->w(); if (minw > 100) minw = 100;
        int minh = o->h(); if (minh > 100) minh = 100;
        w->size_range(w->w() - o->w() + minw, w->h() - o->h() + minh, 0, 0);
      }
      else
      {
        w->size_range(w->w(), w->h(), w->w(), w->h());
      }
    }
    int xwm = xp , ywm = yp , bt, bx, by;
    switch (fake_X_wm(w, xwm, ywm, bt, bx, by))
    {
      // No border (used for menus)
      case 0: needBorder = 0; break;
        // Thin border and title bar
      case 1: needBorder = 1; flFrameFlags |= FCF_DLGBORDER; break;
        // Thick, resizable border and title bar, with maximize button
      case 2: needBorder = 1; flFrameFlags |= FCF_SIZEBORDER | FCF_MAXBUTTON; break;
    }
    if (by+bt)
    {
      if (!w->modal()) flFrameFlags |= FCF_MINBUTTON;
      wp += 2*bx;
      hp += 2*by+bt;
    }
    if (!(w->flags() & Fl_Window::FL_FORCE_POSITION))
    {
      showStyle |= SWP_MOVE;
    }
    else
    {
      if (!Fl::grab())
      {
        xp = xwm; yp = ywm;
        w->x(xp);w->y(yp);
      }
      xp -= bx;
      yp -= by+bt;
    }
    parent = 0;
    if (w->non_modal() && Fl_X::first && !fl_disable_transient_for)
    {
      // find some other window to be "transient for":
      Fl_Window* w = Fl_X::first->w;
      while (w->parent()) w = w->window();
      parent = fl_xid(w);
      if (!w->visible()) showit = 0;
    }
  }

  x->other_xid = 0;
  x->setwindow(w);
  x->region = 0;
  x->private_dc = 0;
  x->cursor = fl_default_cursor;
  x->next = Fl_X::first;
  Fl_X::first = x;
  x->wait_for_expose = 1;

  if (needBorder)
  {
    x->frame = WinCreateStdWindow(HWND_DESKTOP,
                                  0,
                                  &flFrameFlags,
                                  (PSZ)"FLTK",
                                  (PSZ)w->label(),
                                  0,
                                  NULLHANDLE,
                                  0,
                                  &x->xid);

    WinSetWindowPos(x->frame,
                    NULLHANDLE,
                    xp, yp,
                    wp, hp, 
                    SWP_SIZE|SWP_ACTIVATE|SWP_SHOW);

    WinSetActiveWindow(HWND_DESKTOP,x->frame);

    WinSendMsg(x->frame,
               WM_SETICON,
               (MPARAM)WinQuerySysPointer(HWND_DESKTOP, SPTR_PROGRAM, FALSE),
               NULL);

  }
  else
  {
    yp = parentHeight - hp;
    x->xid = WinCreateWindow(parent,        /* Parent window             */
                             (PSZ)"FLTK",   /* Class name                */
                             (PSZ)w->label(),  /* Window text            */
                             WS_VISIBLE,    /* Window style              */
                             xp, yp,        /* Position (x,y)            */
                             wp, hp,        /* Size (width,height)       */
                             parent,        /* Owner window              */
                             HWND_TOP,      /* Sibling window            */
                             Fl::resid++,   /* Window id                 */
                             NULL,          /* Control data              */
                             NULL);         /* Pres parameters           */
    /* We aren't getting an initial paint message... send it one(?) */
    RECTL rect = { xp, yp, wp, hp};
    Region tempRegion = GpiCreateRegion(fl_gc,1,&rect);
    WINGETLASTERROR("GpiCreateRegion");
    WinInvalidateRegion(x->xid,tempRegion,FALSE);
    GpiDestroyRegion(fl_gc,tempRegion);
    WINGETLASTERROR("GpiDestroyRegion");
    WinSendMsg(x->xid,
               WM_PAINT,
               NULL,
               NULL);
//    WinShowWindow(x->xid,TRUE);
    WinSetActiveWindow(HWND_DESKTOP,x->xid);

  }

  /* Moved some x-> stuff up a bit... */

  if (fl_show_iconic)
  {
    showit = 0; fl_show_iconic = 0;
  }
  if (showit)
  {
    w->set_visible();
    w->handle(FL_SHOW); // get child windows to appear
    w->redraw(); // force draw to happen
  }
  if (w->modal())
  {
    Fl::modal_ = w; fl_fix_focus();
  }
  return x;
}


////////////////////////////////////////////////////////////////

void Fl_Window::size_range_()
{
  size_range_set = 1;
}

void Fl_X::set_minmax(LPMINMAXINFO minmax)
{
  int td, wd, hd, dummy;

  fake_X_wm(w, dummy, dummy, td, wd, hd);
  wd *= 2;
  hd *= 2;
  hd += td;

  minmax->ptMinTrackSize.x = w->minw + wd;
  minmax->ptMinTrackSize.y = w->minh + hd;
  if (w->maxw)
  {
    minmax->ptMaxTrackSize.x = w->maxw + wd;
    minmax->ptMaxSize.x = w->maxw + wd;
  }
  if (w->maxh)
  {
    minmax->ptMaxTrackSize.y = w->maxh + hd;
    minmax->ptMaxSize.y = w->maxh + hd;
  }
}

////////////////////////////////////////////////////////////////

#include <FL/filename.H> // need so FL_EXPORT filename_name works

// returns pointer to the filename, or null if name ends with '/'
const char *filename_name(const char *name)
{
  const char *p,*q;
  q = name;
  if (q[0] && q[1]==':') q += 2; // skip leading drive letter
  for (p = q; *p; p++) if (*p == '/' || *p == '\\') q = p+1;
  return q;
}

void Fl_Window::label(const char *name,const char *iname)
{
  Fl_Widget::label(name);
  iconlabel_ = iname;
  if (shown() && !parent())
  {
    if (!name) name = "";
    WinSetWindowText(i->xid, (PSZ)name);
  }
}

////////////////////////////////////////////////////////////////
// Implement the virtual functions for the base Fl_Window class:

// If the box is a filled rectangle, we can make the redisplay *look*
// faster by using X's background pixel erasing.  We can make it
// actually *be* faster by drawing the frame only, this is done by
// setting fl_boxcheat, which is seen by code in fl_drawbox.C:
// For WIN32 it looks like all windows share a background color, so
// I use FL_GRAY for this and only do this cheat for windows that are
// that color.
// Actually it is totally disabled.
// Fl_Widget *fl_boxcheat;
//static inline int can_boxcheat(uchar b) {return (b==1 || (b&2) && b<=15);}

void Fl_Window::show()
{
  if (!shown())
  {
    Fl_X::make(this);
  }
  else
  {
    // Once again, we would lose the capture if we activated the window.
    //if (IsIconic(i->xid)) OpenIcon(i->xid);
    //if (!fl_capture) BringWindowToTop(i->xid);
    WinSetActiveWindow(HWND_DESKTOP,i->frame);
  }
}

Fl_Window *Fl_Window::current_;
// The current device context
HDC fl_pm_hdc = NULLHANDLE;
// The current presentation space
HPS fl_gc = NULLHANDLE;
// the current window handle, initially set to -1 so we can correctly
// allocate fl_GetDC(0)
HWND fl_window = (HWND)-1;

// Here we ensure only one presentation space is ever in place.
HPS fl_GetDC(HWND w)
{
  SIZEL pagesize = {0,0};
  if (fl_pm_hdc == NULLHANDLE)
  {
    fl_pm_hdc = WinOpenWindowDC(w);
    WINGETLASTERROR("WinOpenWindowDC");
  }
  if (fl_gc)
  {
    if (w == fl_window) return fl_gc;
    WinReleasePS(fl_gc);
    WINGETLASTERROR("WinReleasePS");
  }
  fl_gc = WinGetPS(w);
  WINGETLASTERROR("WinGetPS");
  int rc = GpiCreateLogColorTable(fl_gc,0,LCOLF_RGB,0,0,NULL);
  WINGETLASTERROR("GpiCreateLogColorTable");
  fl_window = w;
  // calling GetDC seems to always reset these: (?)
  //SetTextAlign(fl_gc, TA_BASELINE|TA_LEFT);
  //SetBkMode(fl_gc, TRANSPARENT);
  return fl_gc;
}

// make X drawing go into this window (called by subclass flush() impl.)
void Fl_Window::make_current()
{
  fl_GetDC(fl_xid(this));

  fl_select_palette();

  current_ = this;
  fl_clip_region(0);
}

//
// End of "$Id: Fl_win32.cxx,v 1.33.2.37 2001/04/27 15:43:38 easysw Exp $".
//
