
#include "Xlib_private.h"

extern struct {
	char *alias, *fontname;
} FontAlias[];

Xlib_FontHints *Xlib_ParseFontHints(HPS, char*, int);

FONTMETRICS *Xlib_LoadFont(char* name, Xlib_Font *xfont)
{
	DBUG_ENTER("Xlib_LoadFont")
	HPS hps;
	int i, plen, tmp;
	LONG cFonts, lTemp = 0;
	char pattern[256], tempname[256];
	Xlib_FontHints *hints = NULL;
	FONTMETRICS *pfm, *cpfm, *npfm;

	if (name)
		strcpy(pattern, (char*)name);
	else
		memset(pattern, 0, sizeof(pattern));

	strlwr(pattern); plen = strlen(pattern);
	/*fprintf(stderr,"pattern = '%s',\n",pattern);*/
	
	/* First, we go through the aliases */
	if (plen)
	for (i=0; FontAlias[i].alias; i++) 
	if (!plen || PatternMatch(pattern, plen, FontAlias[i].alias, strlen(FontAlias[i].alias))) {
		strcpy(pattern, (char*) FontAlias[i].fontname);
		/*fprintf(stderr, "Alias for '%s',\n",pattern);*/
		break;
	}

	if(!(plen = strlen(pattern)))
		DBUG_RETURN(NULL);

	/* Now, we go through the available fonts */

	hps = WinGetScreenPS(hwndDesktop);	
	hints = Xlib_ParseFontHints(hps, pattern, plen);

	cFonts = GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE, NULL, 
			&lTemp, sizeof(FONTMETRICS), NULL);
	cpfm = pfm = malloc( cFonts * sizeof(FONTMETRICS) );
	GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE, NULL,
			&cFonts, sizeof(FONTMETRICS), pfm);
	WinReleasePS(hps);

	/* Now try to find first matching font */
	for (lTemp = 0; lTemp < cFonts; lTemp++, cpfm++) {
		tmp = RenderFontName(cpfm, tempname, hints, xfont);
		if (PatternMatch(pattern, plen, tempname, tmp-1)) {
			npfm = malloc(sizeof(FONTMETRICS));
			memcpy(npfm, cpfm, sizeof(FONTMETRICS));
			free(pfm); if (hints) free(hints);
			/*fprintf(stderr,"Pattern matches font '%s'\n",tempname);*/
			DBUG_RETURN(npfm);
		}
	}
	/*fprintf(stderr,"No font from %d matched pattern!\n",cFonts);*/
	free(pfm); if (hints) free(hints);
	DBUG_RETURN(NULL);
}

Font XLoadFont(Display* dpy, _Xconst char* name)
{
	DBUG_ENTER("XLoadFont")
	Xlib_Font *font = malloc(sizeof(Xlib_Font));
	FONTMETRICS *pfm;

	memset(font, 0, sizeof(Xlib_Font));

	pfm = Xlib_LoadFont((char*)name, font);
	
	SyncHandle();
	if (pfm) {
		free(pfm);
		/*fprintf(stderr,"Font %x loaded '%s'\n",(Font)font, name);*/
		DBUG_RETURN((Font) font);
	}
	free(font);
	DBUG_RETURN((Font) 0);
}

int XUnloadFont(Display* dpy, Font font)
{
	DBUG_ENTER("XUnloadFont")
	Xlib_Font *xfnt = (Xlib_Font*) font;
	if (!font) DBUG_RETURN(0);
	
	Xlib_FreeFontAtom(xfnt);
	Xlib_InvalidateResource((XID)font);

	SyncHandle();
	free(xfnt);
	DBUG_RETURN(1);
};

XFontProp *Xlib_NewFontProp(HPS hps, FONTMETRICS *pfm, int *n_properties)
{
	DBUG_ENTER("Xlib_NewFontProp")
	XFontProp *prop, *p;
	int propcount = 10;
	
	prop = p = malloc( propcount * sizeof(XFontProp) );

	p->name = XA_SUPERSCRIPT_X; p->card32 = pfm->lSuperscriptXOffset; p++;
	p->name = XA_SUPERSCRIPT_Y; p->card32 = pfm->lSuperscriptYOffset; p++;
	p->name = XA_SUBSCRIPT_X; p->card32 = pfm->lSubscriptXOffset; p++;
	p->name = XA_SUBSCRIPT_Y; p->card32 = pfm->lSubscriptYOffset; p++;
	p->name = XA_UNDERLINE_POSITION; p->card32 = pfm->lUnderscorePosition; p++;
	p->name = XA_UNDERLINE_THICKNESS; p->card32 = pfm->lUnderscoreSize; p++;
	p->name = XA_STRIKEOUT_ASCENT; p->card32 = pfm->lStrikeoutPosition + pfm->lStrikeoutSize - 1; p++;
	p->name = XA_STRIKEOUT_DESCENT; p->card32 = pfm->lStrikeoutPosition - pfm->lStrikeoutSize +1; p++;
	p->name = XA_X_HEIGHT; p->card32 = pfm->lMaxAscender - pfm->lInternalLeading; p++;
	p->name = XA_POINT_SIZE; p->card32 = pfm->lEmHeight * 10;
	
	*n_properties = propcount;
	DBUG_RETURN(prop);
}

XCharStruct *Xlib_NewCharBounds(HPS hps, FONTMETRICS *pfm, XCharStruct *min, XCharStruct *max)
{
	DBUG_ENTER("Xlib_NewCharBounds")
	XCharStruct *cstr, *p;
	LONG limit, i, *widths;

	max->lbearing = min->lbearing = 0;
	max->rbearing = min->rbearing =
	max->width = min->width = pfm->lMaxCharInc;
	min->ascent = max->ascent = pfm->lMaxAscender;
	min->descent = max->descent = pfm->lMaxDescender;
	min->attributes = max->attributes = 0;
	
	limit = pfm->sLastChar - pfm->sFirstChar;

	cstr = p = malloc( limit * sizeof(XCharStruct) );
	memset(cstr, 0, limit * sizeof(XCharStruct) );

	for (i=pfm->sFirstChar; i<pfm->sLastChar; i++, p++) {
		POINTL aptl[TXTBOX_COUNT];

		if (!GpiQueryTextBox(hps, i<0xff?1:2, (PSZ)&i, TXTBOX_COUNT, aptl)) {
			memcpy(p, &cstr[pfm->sDefaultChar], sizeof(XCharStruct));
			continue;
		}
		p->lbearing = aptl[TXTBOX_BOTTOMLEFT].x;
		p->rbearing = aptl[TXTBOX_TOPRIGHT].x;
		p->width = p->rbearing - p->lbearing;
		p->ascent =  aptl[TXTBOX_TOPLEFT].y;
		p->descent =-aptl[TXTBOX_BOTTOMRIGHT].y;

		if (p->lbearing > max->lbearing) max->lbearing = p->lbearing;
		if (p->rbearing < min->rbearing) min->rbearing = p->rbearing;
		if (p->width < min->width) min->width = p->width;
		if (p->ascent < min->ascent) min->ascent = p->ascent;
		if (p->descent < min->descent) min->descent = p->descent;
	}
	DBUG_RETURN(cstr);
}

static XFontStruct *_XQueryFont (dpy, fid, pfm)
    register Display *dpy;
    Font fid;
    FONTMETRICS *pfm;
{
	DBUG_ENTER("_XQueryFont")
	register XFontStruct *fs = malloc(sizeof(XFontStruct));
	register FONTMETRICS *fm = pfm?pfm:malloc(sizeof(FONTMETRICS));
	Xlib_Font *font = (Xlib_Font *)fid;
	HPS hps = NULLHANDLE;
	
	if (!fid || !fm) {
		free(fs);
		if (!pfm && fm) free(fm);
		DBUG_RETURN(0);
	}

	hps = WinGetScreenPS(hwndDesktop);
	GpiCreateLogFont(hps, NULL, 1, &font->fattrs);
	GpiSetCharSet(hps, 1);
	if (font->psmode)
		GpiSetCharBox(hps, &font->sizef);
	if (!pfm)
	GpiQueryFontMetrics(hps, sizeof(FONTMETRICS), fm);

	fs->ext_data		= NULL;
	fs->fid			= fid;
	fs->direction		= FontLeftToRight;
	fs->min_char_or_byte2	= fm->sFirstChar & 0xff;
	fs->max_char_or_byte2	= fm->sLastChar & 0xff;
	fs->min_byte1		= fm->sFirstChar >> 8;
	fs->max_byte1		= fm->sLastChar >> 8;
	fs->default_char	= fm->sFirstChar + fm->sDefaultChar;
	fs->all_chars_exist	= (fm->fsType & FM_TYPE_FIXED) ? True : False;
	fs->properties		= Xlib_NewFontProp(hps, fm, &fs->n_properties);
	fs->per_char		= Xlib_NewCharBounds(hps, fm, &fs->min_bounds, &fs->max_bounds);
	fs->ascent		= fm->lMaxAscender;
	fs->descent		= fm->lMaxDescender;

	GpiDeleteSetId(hps,1);
	WinReleasePS(hps);
	if (!pfm) free(fm);
	DBUG_RETURN(fs);
}


XFontStruct *XLoadQueryFont(Display* dpy, _Xconst char* name)
{
	DBUG_ENTER("XLoadQueryFont")
	Xlib_Font *font = malloc(sizeof(Xlib_Font));
	FONTMETRICS *pfm;

	memset(font, 0, sizeof(Xlib_Font));

	pfm = Xlib_LoadFont((char*)name, font);
	if (pfm) {
                XFontStruct *xfs = _XQueryFont(dpy, (Font)font, pfm);
		free(pfm);
		SyncHandle();
		/*fprintf(stderr,"Font %x loaded '%s'\n",(Font)font, name);*/
		DBUG_RETURN(xfs);
        }
	free(font);
	SyncHandle();
	DBUG_RETURN(NULL);
}

int XFreeFont(Display* dpy, XFontStruct* fs)
{
	DBUG_ENTER("XFreeFont")
	_XExtension *ext;
	if (!fs) DBUG_RETURN(0);

	LockDisplay(dpy);
	for (ext = dpy->ext_procs; ext; ext = ext->next)
	if (ext->free_Font) (*ext->free_Font)(dpy, fs, &ext->codes);

	XUnloadFont(dpy, fs->fid);
	UnlockDisplay(dpy);
	SyncHandle();
	_XFreeExtData(fs->ext_data);
	if (fs->per_char) Xfree((char*)fs->per_char);
	if (fs->properties) Xfree((char*)fs->properties);
	Xfree((char*)fs);
	DBUG_RETURN(1);
}

XFontStruct *XQueryFont (dpy, fid)
    register Display *dpy;
    Font fid;
{
    DBUG_ENTER("XQueryFont")
    XFontStruct *font_result;

    LockDisplay(dpy);
    font_result = _XQueryFont(dpy, fid, NULL);
    UnlockDisplay(dpy);
    SyncHandle();
    DBUG_RETURN(font_result);
}
