/* This file contains the definition of the abstract Loader class, which
	attempts to provide a relatively machine independent interface for
	programs that read binary files.

	The abstraction is as follows. The named file is loaded as if at the
	native address implied by the file (if not, then at a default address).
	When loaded with the Load() function, it is relocated appropriately.
	The loader object maintains a list of sections. (If a format, such
	as the Windows NE format, is not really organised into sections,
	then sections will be created, one for the symbol table information,
	one for each loadable segment, and so on). Section information is
	available as an array of SECTIONINFO structs; each struct has infor-
	mation about where the section is loaded, whether it is text or
	data, and whether it is a primary or non primary section. Primary
	sections are the ones that most loader applications would be
	interested in (e.g. .text and .plt sections, but not .init sections,
	even though the latter may contain code).

	The Loader object keeps track of the total size of the code and
	data sections, and the mapping between native and host addresses.
	The program using the Loader object can read the whole image by
	processing all the code and data.

	If the binary file is an archive file with multiple parts, only the
	first part is initially loaded. Subsequent parts may be accessed
	by calling the NextPart() function. Load() and NextPart() return
	1 for success, and 0 for failure.

	Symbol tables are treated as a list of (string, value) pairs, where
	the value is the native address associated with the string. Names
	and values in the symbol table are in no particular order.

	The called program must call UnLoad() when done with the loaded
	image; otherwise the image will not be properly freed. This is
	true even if Load() fails (returns 0).

	To summarise usage:
		1) Instantiate the appropriate object (derived from this abstract
			class; e.g. to load elf files, instantiate an ElfLoader obj)

		2) Call Load() with the full name of the file

		3) Call GetNumSections() and GetSectionInfo() and examine the
			section information. Use the pointers in the SECTIONINFO structs
			to access the loaded image

		4) If required, call GetNumSyms() and GetSymTable() for symbol
			table information

		5) If required, call NextPart(), and if it returns true, repeat
			steps 3-5

		6) Call UnLoad(). Do not access the loaded image after this call.


	It should be possible to use this code with any c++ compiler under
	Unix or Windows, since no fancy classes like String were used.
*/

#include <string.h>		/* memcpy(), strcat(), strlen() */


// Section Type flag values. See below for usage
#define ST_CODE		0x01		// Section contains program code
#define ST_DATA		0x02		// Section contains program data
#define ST_BSS		0x04		// Section is not initialised (no file data)
#define ST_PRIMARY	0x08		// Section is important (e.g. .text, not .init)
#define ST_READONLY	0x10		// Section has read-only data
#define ST_HEADER	0x20		// Pseudo-Section is the header
#define ST_RELOC	0x40		// Pseudo-Section with reloc info (.EXE only)

// SECTIONINFO structure. GetSectionInfo returns a pointer to an array of
// these structs. All information about the sections is contained in these
// structures.

typedef struct sectioninfo_tag
{
	char*		pSectionName;		// Name of section
	unsigned	fSectionFlags;		// Flags, bitwise OR of ST_XXX flags
	unsigned	uNativeAddr;	// Logical or native load address
	unsigned	uHostAddr;		// Host or actual address of data
	unsigned	uSectionSize;		// Size of section in bytes
	unsigned	uSectionEntrySize;	// Size of one section entry (if applic)
} SECTIONINFO;

typedef SECTIONINFO* PSECTIONINFO;
	
typedef struct symtab_tag
{
	char*		pSymName;		// Name of the symbol
	unsigned	uSymValue;		// Value of the symbol
} LD_SYMTAB;

typedef LD_SYMTAB* PLD_SYMTAB;
	
class Loader
{
public:
				Loader();					// Default constructor
  virtual int	Load(const char* sName) = 0;// Load the file; pure virtual
  virtual int	GetNextPart() = 0;			// Get next part of archive
  virtual void	UnLoad() = 0;				// Unload file; pure virtual
	int			GetNumSections() const;		// Return number of sections
	PSECTIONINFO GetSectionInfo() const;		// Return array of section structs
	int			GetNumSyms() const;			// Return number of sections
	PLD_SYMTAB 	GetSymTable() const;		// Return array of section structs
				// Find section index given name
	int			GetSectionIndexByName(const char* sName);
				// Find section info given name
	PSECTIONINFO GetSectionInfoByName(const char* sName);
	unsigned	GetInitPC() const;			// Get the initial PC
	unsigned	GetInitSP() const;			// Get the initial SP

protected:
	int			m_iNumSections;				// Number of sections
	PSECTIONINFO m_pSections;				// The section info
	int			m_iNumSymbols;				// Number of symbols
	PLD_SYMTAB	m_pSymbols;					// The symbol table
	unsigned	m_uInitPC;					// Initial program counter
	unsigned	m_uInitSP;					// Initial stack pointer
};

