/*
************************************************************************
** Copyright: (C) IBM BocaRaton Ltd. 1994
**
** Classification:  IBM Test Tool
**
** Original Author
** and Date: 	    Kishan Kasety, March 1994
**
** Filename:        DDTPCMC.CPP
**
** Description:     Device driver test functions, PCMCIA socket services
**
** Revisions:       Jesse Chen, 6/15/1994
**
** Comments:        Bitwise outputs now reflect the data returned
**		    and are end-user readable
**
*************************************************************************
*/

#include "apicover.h"
#include "ddtpcmc.h"
#include "clspcmc.h"
#include <istring.hpp>

#define  CATEGORY     200

// Global variables

ULONG       ulRc;
CHAR        OutBuff[256];

/*
*************************************************************************
** Name: APIRET _export PCMC_Open( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_Open( Kwd_List& param )
{
   IString    s1;
   ULONG     ulAction = 0;
   ULONG     ulOpenFlag;
   ULONG     ulOpenMode;
   HFILE     hfdrvhandle;

   ulOpenFlag = OPEN_ACTION_OPEN_IF_EXISTS;
   ulOpenMode = OPEN_FLAGS_FAIL_ON_ERROR | OPEN_SHARE_DENYWRITE;

   ulRc = ddtDosOpen( "PCMCIA$ ", &hfdrvhandle, &ulAction,
		      0, 0, ulOpenFlag, ulOpenMode, NULL,
		      "PCMC_Open", "only call",
		      param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
     s1 = "Device failed to open.  Aborting Script";
     param.files()->out1<<s1;
     param.files()->out1.flush();
     DosExit(0,ulRc);
   }
   param.set("DRIVEHANDLE", (IString)(long)hfdrvhandle );
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_Close( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_Close( Kwd_List& param )
{
   HFILE   hfdrvhandle = param.getInt("DRIVEHANDLE");

   ulRc = ddtDosClose( hfdrvhandle,
		       "PCMC_Close",
		       "only call",
		       param.files()->out1);
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetAdapterCount( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetAdapterCount( Kwd_List & param )
{
  struct
  {
    BYTE          TotalAdapters;
    BYTE          Signature[2];
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x80, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetAdapterCount", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)  // ddtDosDevIOCtl failed
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Total Adapters : ";
   s1 += (IString)(LONG)paramblock.TotalAdapters + (IString)"\n\tSignature      : ";
   sprintf(OutBuff, "%c%c", paramblock.Signature[0], paramblock.Signature[1]);
   s1 += OutBuff;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_AckInterrupt( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_AckInterrupt( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          Socket;
    USHORT        NumInterrupts;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Socket = param.getInt("SOCKET");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x9E, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_AckInterrupt", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter         : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tSocket          : ";
   s1 += (IString)(LONG)paramblock.Socket + (IString)"\n\t# of Interrupts : ";
   s1 += (IString)(LONG)paramblock.NumInterrupts;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_PriorHandler( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_PriorHandler( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          Mode;
  } paramblock;

   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Mode = param.getInt("MODE");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x9F, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_PriorHandler", "only call",
			  param.files()->out1);

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_SSAddr( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_SSAddr( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          ProcMode;
    BYTE          Function;
    BYTE          NumAddData;
  } paramblock;

   IString         s1;
   ULONG          data[8];
   ULONG          ulDataSize = sizeof(data);
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.ProcMode = param.getInt("PROCMODE");
   paramblock.Function = param.getInt("FUNCTION");
   paramblock.NumAddData = param.getInt("NUM_SEGS");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0xA0, &paramblock,
			  ulParamSize, &ulParamSize,
			  &data, ulDataSize, &ulDataSize,
			  "PCMC_SSAddr", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tProc Mode : ";
   s1 += (IString)(LONG)paramblock.ProcMode + (IString)"\n\tFunction  : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Function);
   s1 += OutBuff;

   s1 += "\n\t# of Additional data Segments : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.NumAddData);
   s1 += OutBuff;
   s1 += "\n\tData returned into DataArea : ";
   for (int i=0; i<8; i++ )
   {
      s1 += "\n";
      sprintf(OutBuff, "\t\t0x%4.4x: 0x%4.4x",(i-1)*4, data[i] );
      s1 += OutBuff;
   }

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_AccessOffsets( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_AccessOffsets( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          ProcMode;
    BYTE          NumDesired;
    BYTE          NumAvail;
  } paramblock;

   IString         s1;
   USHORT         data[18];
   ULONG          ulDataSize = sizeof(data);
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.ProcMode = param.getInt("PROCMODE");
   paramblock.NumDesired = param.getInt("NUM_DESIRED");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0xA1, &paramblock,
			  ulParamSize, &ulParamSize,
			  &data, ulDataSize, &ulDataSize,
			  "PCMC_AccessOffsets", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tProc Mode : ";
   s1 += (IString)(LONG)paramblock.ProcMode + (IString)"\n\tNumAvail  : ";
   s1 += (IString)(LONG)paramblock.NumAvail;

   s1 += "\n\tData returned into DataArea : ";
   for (int i=0; i<18; i++ )
   {
      s1 += "\n";
      sprintf(OutBuff, "\t\t0x%2.2x: 0x%4.4x", (i-1)*2, data[i] );
      s1 += OutBuff;
   }

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetAdapter( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetAdapter( Kwd_List & param )
{
  struct
  {
    BYTE	Adapter;
    int	        PowerCon        :1;
    int         PreserveState   :1;
    BYTE        Reserved1       :6;
   // AdpAttribs bits:
    BYTE 	IRQlevel        :4;
    BYTE	Reserved2	:2;
    int         InterruptLevel  :1;
    int         InterruptEnab   :1;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t     ");
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x85, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetAdapter", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 += "Adapter #  : ";
   s1 += (IString)(LONG)paramblock.Adapter;
   s1 += "\n\n\tAdapter Attribs: ";
   s1.CheckBit(paramblock.PowerCon, 3, "Adapter Reduce Power Consumption Mode is ",
		"ON", "OFF");
   s1.CheckBit(paramblock.PreserveState, 3, "Adapter Preserve State Mode is ",
		"ON", "OFF");

   if(  paramblock.Reserved1 )
   {
     s1 += "\n\t     ERROR: some reserved bits in Adapter attributes are on";
     sprintf(OutBuff,"\n\t          Adapter attributes reserved bits 2-7 = 0x%2.2x", paramblock.Reserved1);
     s1 += OutBuff;
   }
   else
     s1 += "\n\t     Adapter attributes Reserved bits (2-7) are all RESET";

   s1 += "\n\n\tIRQ: ";
   sprintf(OutBuff, "\n\t     IRQ Level: 0x%2.2x", paramblock.IRQlevel);
   s1 += OutBuff;
   s1.CheckBit(paramblock.InterruptLevel, 3, "Status Change Interrupt Level is ",
		"ACTIVE HIGH", "ACTIVE LOW");
   s1.CheckBit(paramblock.InterruptEnab, 3, "Status Change Interrupt is ",
		"ENABLED", "DISABLED");
   if (paramblock.Reserved2)
   {
	s1 += "\n\t     ERROR: Reserved bits 4 and 5 on (should always be zero)";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 4 and 5 = 0x%2.2x", paramblock.Reserved2);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and reset to ZERO";

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetSSInfo( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetSSInfo( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    USHORT        Compliance;
    BYTE          NumAdapters;
    BYTE          FirstAdapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x83, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetSSInfo", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter : ";
   s1 += (IString)(LONG)paramblock.Adapter;
   s1 += "\n\tCompliance    : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Compliance);
   s1 += OutBuff;
   s1 += "\n\t# of Adapters : ";
   s1 += (IString)(LONG)paramblock.NumAdapters + (IString)"\n\tFirst Adapter : ";
   s1 += (IString)(LONG)paramblock.FirstAdapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetVendorInfo( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetVendorInfo( Kwd_List & param )
{
   struct
   {
     BYTE          Adapter;
     BYTE          Type;
     USHORT        Release;
   } paramblock;

   struct
   {
     USHORT        BytesAlloc;
     USHORT        BytesRet;
     CHAR          Asciidata[1024];
   } data;

   IString         s1;
   ULONG          ulDataSize = sizeof(data);
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Type = 0;
   data.BytesAlloc = 1024;

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x9D, &paramblock,
			  ulParamSize, &ulParamSize,
			  &data, ulDataSize, &ulDataSize,
			  "PCMC_GetVendorInfo", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tRelease   : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Release);
   s1 += OutBuff;
   s1 += "\n\tData returned into DataArea : ";
   s1 += "\n\tBytes allocated : ";
   sprintf(OutBuff, "0x%4.4x", data.BytesAlloc);
   s1 += OutBuff;
   s1 += "\n\tBytes returned  : ";
   sprintf(OutBuff, "0x%4.4x", data.BytesRet);
   s1 += OutBuff;
   s1 += "\n\n";
   s1 += (IString)data.Asciidata;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_InqAdapter( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_InqAdapter( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          NumSockets;
    BYTE          NumWindows;
    BYTE          NumEDCs;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t\t   ");
   struct
   {
	USHORT  data1[2];
   //   Adapter Capabilities (bit-mapped) bits
	USHORT  ACIndicator	:1;
	USHORT  ACPowerMan	:1;
	USHORT	ACDbw		:1;
	USHORT	ACReserved	:13;
	USHORT	data2[13];
   }    data;
   ULONG          ulDataSize = sizeof(data);
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   data.data1[0] = 32;

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x84, &paramblock,
			  ulParamSize, &ulParamSize,
			  &data, ulDataSize, &ulDataSize,
			  "PCMC_InqAdapter", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 +=  "Adapter #    : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\t# of Sockets : ";
   s1 += (IString)(LONG)paramblock.NumSockets + (IString)"\n\t# of Windows : ";
   s1 += (IString)(LONG)paramblock.NumWindows + (IString)"\n\t# of EDCs    : ";
   s1 += (IString)(LONG)paramblock.NumEDCs;

   s1 += "\n\n\tData returned into DataArea : ";
   s1 += "\n\n\tBytes allocated : ";
   sprintf(OutBuff, "0x%4.4x",data.data1[0]);
   s1 += OutBuff;

   s1 += "\n\tBytes returned  : ";
   sprintf(OutBuff, "0x%4.4x",data.data1[1]);
   s1 += OutBuff;

   sprintf(OutBuff, "\n\n\t0x%2.2x: ", 0);
   s1 += OutBuff;
   s1 += "   Adapter Capabilities (bit mapped):";
   s1.CheckBit(data.ACIndicator, 3, "Indicators are ",
		"SET", "RESET");
   s1.CheckBit(data.ACPowerMan, 3, "Power Management is ",
		"SET", "RESET");
   s1.CheckBit(data.ACDbw, 3, "Data Bus Width is ",
		"SET", "RESET");
   if(data.ACReserved)
   {
	s1 += "\n\t\t   ERROR...reserved bits 3-7 NOT zero";
	sprintf(OutBuff, "\n\t\t   -- Reserved bits 3-7 = 0x%4.4x", data.ACReserved);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t\t   All other bits are reserved and set to ZERO";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (3-2)*2, data.data2[0] );
   s1 += OutBuff;
   s1 += "   Active HIGH IRQs Bit 0 = IRQ0 and Bit 15 = IRQ15";
   sprintf(OutBuff, "\n\t0x%2.2x: 0x%4.4x", (4-2)*2, data.data2[1] );
   s1 += OutBuff;
   if (data.data2[1])
	s1 += "   <-ERROR...All bits are reserved and should be RESET";
   else
	s1 += "   Addl active HIGH IRQs all bits are reserved, RESET to zero";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (5-2)*2, data.data2[2] );
   s1 += OutBuff;
   s1 += "   Active LOW IRQs Bit 0 = IRQ0 and Bit 15 = IRQ15";
   sprintf(OutBuff, "\n\t0x%2.2x: 0x%4.4x", (6-2)*2, data.data2[3] );
   s1 += OutBuff;
   if (data.data2[3])
	s1 += "   <-ERROR...All bits are reserved and should be RESET";
   else
	s1 += "   Addl active LOW IRQs all bits are reserved, RESET to zero";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (7-2)*2, data.data2[4] );
   s1 += OutBuff;
   s1 += "   # of Power Management entries";

   s1.CheckEntry(data.data2[4], &(data.data2[5]));

   for (int i= 8 + data.data2[4]; i<16; i++ )
   {
      sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (i-2)*2, data.data2[i-3] );
      s1 += OutBuff;
   }

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_SetAdapter( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_SetAdapter( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          State;
    BYTE          IRQ;
  } paramblock;

   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.State = param.getInt("ADPSTATE");
   paramblock.IRQ = param.getInt("IRQ");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x86, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_SetAdapter", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc !=0)
   {
      printerror(ulRc, param);
      return ulRc;
   }
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_VendorSpecific( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_VendorSpecific( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          FunctionCode;
    BYTE          SubFunction;
  } paramblock;

   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.SubFunction = param.getInt("SUBFUNCTION");
   paramblock.FunctionCode = 1;

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0xAE, &paramblock,
                          ulParamSize, &ulParamSize, NULL, 0, NULL,
                          "PCMC_VendorSpecific", "only call",
                          param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc !=0)
   {
      printerror(ulRc, param);
      return ulRc;
   }
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetSocket( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetSocket( Kwd_List & param )
{
  struct
  {
    BYTE        Adapter;
    BYTE        Socket;
//  InterruptMask variables
    int         MaskWriteProtect	:1;
    int 	MaskCardLock		:1;
    int		MaskEjectReq		:1;
    int		MaskInsertReq		:1;
    int		MaskBatteryDead		:1;
    int		MaskBatteryWarn		:1;
    int		MaskReadyChange		:1;
    int		MaskCardDetect		:1;

    BYTE        VccLevel;
    BYTE        VppLevel;
//  SocketState variables
    int         StateWriteProtect	:1;
    int		StateCardLock		:1;
    int		StateEjectReq		:1;
    int		StateInsertReq		:1;
    int		StateBatteryDead	:1;
    int		StateBatteryWarn	:1;
    int		StateReadyChange	:1;
    int		StateCardDetect		:1;

//  SocketControl variables
    BYTE	CtrlReserved1		:4;
    int 	CtrlLockState           :1;
    BYTE	CtrlReserved2		:1;
    int		CtrlBusyState		:1;
    BYTE	CtrlReserved3		:1;

//  USHORT IFIRQ variables
    USHORT      IFIRQlevel		:4;
    USHORT      IFIRQReserved1		:2;
    USHORT      IFIRQInvertEnab		:1;
    USHORT      IFIRQSteerEnab		:1;
    USHORT      IFIRQMemOnlyInter	:1;
    USHORT      IFIRQMemIOInter		:1;
    USHORT      IFIRQReserved2		:6;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t     ");
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Socket = param.getInt("SOCKET");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x8D, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetSocket", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 += "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tSocket #  : ";
   s1 += (IString)(LONG)paramblock.Socket;
   s1 += "\n\n\tInterrupt Mask (reset to zero - masked):";
   s1.CheckBit(paramblock.MaskWriteProtect, 3, "Write Protect Change is ", "SET", "RESET");
   s1.CheckBit(paramblock.MaskCardLock, 3, "Card Lock Change is ", "SET", "RESET");
   s1.CheckBit(paramblock.MaskEjectReq, 3, "Ejection Request is ",
		"SET (Enable for PS/2-E)", "RESET (Disable for PS/2-E)");
   s1.CheckBit(paramblock.MaskInsertReq, 3, "Insertion Request is ", "SET", "RESET");
   s1.CheckBit(paramblock.MaskBatteryDead, 3, "Battery Dead Change is ",
		"SET (Enabled)", "RESET (Disabled)");
   s1.CheckBit(paramblock.MaskBatteryWarn, 3, "Battery Warning Change is ",
		"SET (Enabled)", "RESET (Disabled)");
   s1.CheckBit(paramblock.MaskReadyChange, 3, "Ready Change is ",
		"SET (Enabled)", "RESET (Disabled)");
   s1.CheckBit(paramblock.MaskCardDetect, 3, "Card Detect Change is ",
		"SET (Enabled)", "RESET (Disabled)");

   s1 += "\n\n\tVcc Level      : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.VccLevel);
   s1 += OutBuff;
   s1 += "\n\n\tVpp Level      : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.VppLevel);
   s1 += OutBuff;
   s1 += "   Upper Nibble--Vpp1 level, Lower Nibble--Vpp2 level";

// ----- Socket State -----

   s1 += "\n\n\tSocket State (reset to zero=false, set to one=true):";
   s1.CheckBit(paramblock.StateWriteProtect, 3, "Write Protect Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.StateCardLock, 3, "Card Lock Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.StateEjectReq, 3, "Ejection Request(For PS/2-E) is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.StateInsertReq, 3, "Insetion Request is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.StateBatteryDead, 3, "Battery Dead Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.StateBatteryWarn, 3, "Battery Warning Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.StateReadyChange, 3, "Ready Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.StateCardDetect, 3, "Card Detect Change is ",
		"SET", "RESET");

// ----- Socket Control -----

   s1 += "\n\n\tSocket Control (set = on, reset = off): ";
   s1.CheckBit(paramblock.CtrlLockState, 3, "Lock State is ",
		"SET (locked)", "RESET (unlocked)");
   s1.CheckBit(paramblock.CtrlBusyState, 3, "Busy State is ", "SET", "RESET");
   if (paramblock.CtrlReserved1 || paramblock.CtrlReserved2 || paramblock.CtrlReserved3)
   {
	s1 += "\n\t     ERROR: Some reserved bits are NOT set to zero!!";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 0-3   = 0x%2.2x", paramblock.CtrlReserved1);
	s1 += OutBuff;
	sprintf(OutBuff, "\n\t\t-- Reserved bit 5      = 0x%1x", paramblock.CtrlReserved2);
	s1 += OutBuff;
	sprintf(OutBuff, "\n\t\t-- Reserved bit 7      = 0x%1x", paramblock.CtrlReserved3);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and are set to ZERO";

// ----- IFIRQ -----

   s1 += "\n\n\tIFIRQ Level Steering(I/O Cards only): ";
   sprintf(OutBuff, "\n\t     IFIRQ Level is : 0x%2.2x", paramblock.IFIRQlevel);
   s1 += OutBuff;
   s1.CheckBit(paramblock.IFIRQInvertEnab, 3, "IRQ Inverter is ",
		"ENABLED", "DISABLED");
   s1.CheckBit(paramblock.IFIRQSteerEnab, 3, "IRQ Steering is ",
		"ENABLED", "DISABLED");
   s1.CheckBit(paramblock.IFIRQMemOnlyInter, 3, "Memory Only Interface is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.IFIRQMemIOInter, 3, "IRQ I/O and Memory Interface is ",
		"SET", "RESET");

   if (paramblock.IFIRQReserved1 || paramblock.IFIRQReserved2)
   {
	s1 += "\n\t     ERROR: Some of the Reserved bits are NOT set to zero!!";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 4 and 5 = 0x%2.2x", paramblock.IFIRQReserved1);
	s1 += OutBuff;
	sprintf(OutBuff, "\n\t\t-- Reserved bits 10-15   = 0x%4.4x", paramblock.IFIRQReserved2);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and set to ZERO";

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetStatus( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetStatus( Kwd_List & param )
{
  struct
  {
    BYTE        Adapter;
    BYTE        Socket;
//  CardAttribs bits
    int         AttrWriteProtect	:1;
    BYTE	AttrReserved		:3;
    int		AttrBatteryDead		:1;
    int		AttrBatteryWarn		:1;
    int		AttrReadyBusy		:1;
    int		AttrCardDetect		:1;

//  SocketState bits
    int		StateWriteProtect	:1;
    int		StateCardLock		:1;
    int		StateEjectReq		:1;
    int		StateInsertReq		:1;
    int		StateBatteryDead	:1;
    int		StateBatteryWarn	:1;
    int		StateReadyChange	:1;
    int		StateCardDetect		:1;

//  SocketControl bits

    BYTE        CtrlReserved1		:4;
    int         CtrlLockState           :1;
    BYTE	CtrlReserved2		:1;
    int		CtrlBusyState		:1;
    BYTE  	CtrlReserved3		:1;

//  IFRQ bits

    USHORT      IFIRQlevel		:4;
    USHORT      IFIRQReserved1		:2;
    USHORT      IFIRQInvertEnab		:1;
    USHORT      IFIRQSteerEnab		:1;
    USHORT      IFIRQMemOnlyInter	:1;
    USHORT      IFIRQMemIOInter		:1;
    USHORT      IFIRQReserved2		:6;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t     ");
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Socket = param.getInt("SOCKET");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x8F, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetStatus", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 += "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tSocket #  : ";
   s1 += (IString)(LONG)paramblock.Socket;

   s1 += "\n\n\tCard Attribs :";
   s1.CheckBit(paramblock.AttrWriteProtect, 3, "Write Protect is ",
		"ON (protected)", "OFF (not protected)");
   if (paramblock.AttrReserved)
   {
	s1 += "\n\t     ERROR: Reserved bits 1-3 are NOT zero!!";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 1-3 = 0x%1x",paramblock.AttrReserved);
	s1 += OutBuff;
   }

   s1.CheckBit(paramblock.AttrBatteryDead, 3, "Battery Dead is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.AttrBatteryWarn, 3, "Battery Warning is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.AttrReadyBusy, 3, "Ready/Busy Bit is ",
		"ON (ready)", "OFF (not ready)");
   s1.CheckBit(paramblock.AttrCardDetect, 3, "Card Detect is ",
		"DETECTED", "Not Detected");

   s1 += "\n\n\tSocket State   : ";
   s1 += "   (reset to zero = false,  set to zero = true)";
   s1.CheckBit(paramblock.StateWriteProtect, 3, "Write Protect Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.StateCardLock, 3, "Card Lock Change is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.StateEjectReq, 3, "Ejection Request(for PS/2-E) is ",
		"SET (enable for PS/2-E)", "RESET");
   s1.CheckBit(paramblock.StateInsertReq, 3, "Insertion Request is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.StateBatteryDead, 3, "Battery Dead Change is ",
		"SET (enabled)", "RESET (disabled)");
   s1.CheckBit(paramblock.StateBatteryWarn, 3, "Battery Warning Change is ",
		"SET (enabled)", "RESET (disabled)");
   s1.CheckBit(paramblock.StateReadyChange, 3, "Ready Change is ",
		"SET (enabled)", "RESET (disabled)");
   s1.CheckBit(paramblock.StateCardDetect, 3, "Card Detect Change is ",
		"SET", "RESET");

   s1 += "\n\n\tSocket Control : ";
   s1 += "   (set = on, reset = off)";

   s1.CheckBit(paramblock.CtrlLockState, 3, "Lock State is ",
		"SET (=locked)", "RESET (unlocked)");
   s1.CheckBit(paramblock.CtrlBusyState, 3, "Busy State Indicator is ",
		"ON", "OFF");
   if (paramblock.CtrlReserved1 || paramblock.CtrlReserved2
				|| paramblock.CtrlReserved3)
   {
	s1 += "\n\t     ERROR...reserved bits 0-3, 5, 7 are NOT zero";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 0-3   = 0x%1x",paramblock.CtrlReserved1);
	s1 += OutBuff;
	sprintf(OutBuff, "\n\t\t-- Reserved bit 5      = 0x%1x",paramblock.CtrlReserved2);
	s1 += OutBuff;
	sprintf(OutBuff, "\n\t\t-- Reserved bit 7      = 0x%1x",paramblock.CtrlReserved3);
   }
   else
	s1 += "\n\t     All other bits are reserved and set to ZERO";

   s1 += "\n\n\tIFIRQ: ";
   sprintf(OutBuff, "\n\t     IRQ Level : 0x%2.2x", paramblock.IFIRQlevel);
   s1 += OutBuff;

   s1.CheckBit(paramblock.IFIRQInvertEnab, 3, "IRQ Inverter is ",
		"ENABLED", "DISABLED");
   s1.CheckBit(paramblock.IFIRQSteerEnab, 3, "IRQ Steering is ",
		"ENABLED", "DISABLED");
   s1.CheckBit(paramblock.IFIRQMemOnlyInter, 3, "Memory only interface is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.IFIRQMemIOInter, 3, "I/O and Memory interface is ",
		"SET", "RESET");

   if (paramblock.IFIRQReserved1 || paramblock.IFIRQReserved2)
   {
	s1 += "\n\t     ERROR...reserved bits 4, 5 and 10-15 are NOT zero";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 4 and 5 = 0x%1x", paramblock.IFIRQReserved1);
	s1 += OutBuff;
	sprintf(OutBuff, "\n\t\t-- Reserved bits 10-15   = 0x%4.4x", paramblock.IFIRQReserved2);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and are set to ZERO";

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_InqSocket( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_InqSocket( Kwd_List & param )
{
  struct
  {
    BYTE        Adapter;
    BYTE        Socket;
//  InterruptCaps bits
    int         InterWpc	:1;
    int         InterClc 	:1;
    int		InterEr         :1;
    int 	InterIr		:1;
    int		InterBd		:1;
    int		InterBw		:1;
    int		InterRc		:1;
    int		InterCdc	:1;
//  ReportCaps bits
    int		ReportWpc	:1;
    int         ReportClc	:1;
    int		ReportEr	:1;
    int		ReportIr	:1;
    int 	ReportBd	:1;
    int		ReportBw	:1;
    int		ReportRc	:1;
    int		ReportCdc	:1;
//  ControlCaps bits
    int		CtrlWps		:1;
    int		CtrlCls		:1;
    int		CtrlMcec	:1;
    int		CtrlMcic	:1;
    int		CtrlClc		:1;
    int		CtrlBsi		:1;
    int		CtrlBs		:1;
    int		CtrlXips	:1;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t     ");
   struct
   {
	USHORT	data1[2];
	USHORT	MemOnly		:1;
	USHORT	IOMem		:1;
	USHORT	Reserved	:14;
	USHORT	data2[4];
   } data;

   ULONG          ulDataSize = sizeof(data);
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Socket = param.getInt("SOCKET");
   data.data1[0] = 14;

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x8C, &paramblock,
			  ulParamSize, &ulParamSize,
			  &data, ulDataSize, &ulDataSize,
			  "PCMC_InqSocket", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 += "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tSocket #  : ";
   s1 += (IString)(LONG)paramblock.Socket +
	 (IString)"\n\n\tStatus Change Interrupt Capabilities : ";

   s1.CheckBit(paramblock.InterWpc, 3, "Write Protect Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.InterClc, 3, "Card Lock Change is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.InterEr, 3, "Ejection Request is ",
		"SET: for PS/2-E", "RESET");
   s1.CheckBit(paramblock.InterIr, 3, "Insertion Request is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.InterBd, 3, "Battery Dead Change is ",
		"SET", "RESET:ERROR...should always be one");
   s1.CheckBit(paramblock.InterBw, 3, "Battery Warning Change is ",
		"SET", "RESET:ERROR...should always be one");
   s1.CheckBit(paramblock.InterRc, 3, "Ready Change is ",
		"SET", "RESET:ERROR...should always be one");
   s1.CheckBit(paramblock.InterCdc, 3, "Card Detect Change is ",
		"SET", "RESET:ERROR...should always be one");

   s1 += "\n\n\tStatus Change Reporting Capabilities : ";
   s1.CheckBit(paramblock.ReportWpc, 3, "Write Protect Change is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.ReportClc, 3, "Card Lock Change is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.ReportEr, 3, "Ejection Request is ",
		"ON: for PS/2-E", "RESET");
   s1.CheckBit(paramblock.ReportIr, 3, "Insertion Request is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.ReportBd, 3, "Battery Dead Change is ",
		"SET", "RESET:ERROR...should always be one");
   s1.CheckBit(paramblock.ReportBw, 3, "Battery Warning Change is ",
		"SET", "RESET:ERROR...should always be one");
   s1.CheckBit(paramblock.ReportRc, 3, "Ready Change is ",
		"SET", "RESET:ERROR...should always be one");
   s1.CheckBit(paramblock.ReportCdc, 3, "Card Detect Change is ",
		"SET", "RESET:ERROR...should always be one");

   s1 += "\n\n\tControl and Indicators Capabilities : ";
   s1.CheckBit(paramblock.CtrlWps, 3, "Write Protect Status is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.CtrlCls, 3, "Card Lock Status is ",
		"ON:for PS/2-E", "OFF");
   s1.CheckBit(paramblock.CtrlMcec, 3, "Motorized Card Ejection Control is ",
		"SET:ERROR...must be zero", "RESET");
   s1.CheckBit(paramblock.CtrlMcic, 3, "Motorized Card Insertion Control is ",
		"SET:ERROR...must be zero", "RESET");
   s1.CheckBit(paramblock.CtrlClc, 3, "Card Lock Control is ",
		"ON: for PS/2-E", "OFF");
   s1.CheckBit(paramblock.CtrlBsi, 3, "Battery Status Indicator is ",
		"SET:ERROR...should always be zero", "RESET");
   s1.CheckBit(paramblock.CtrlBs, 3, "Busy Status is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.CtrlXips, 3, "XIP status is ",
		"SET:ERROR...should always be zero", "RESET");

   s1 += "\n\n\tData returned into DataArea : ";
   s1 += "\n\tBytes allocated : ";
   sprintf(OutBuff, "0x%4.4x",data.data1[0]);
   s1 += OutBuff;

   s1 += "\n\tBytes returned  : ";
   sprintf(OutBuff, "0x%4.4x",data.data1[1]);
   s1 += OutBuff;

   sprintf(OutBuff, "\n\n\t0x%2.2x:", 0);
   s1 += OutBuff;
   s1 += "  Interface Type Supported (bit mapped): ";
   s1.CheckBit(data.MemOnly, 3, "Memory Only bit is ",
		"SET", "RESET");
   s1.CheckBit(data.IOMem, 3, "I/O and Memory bit is ",
		"SET", "RESET");
   if (data.Reserved)
   {
	s1 += "\n\t     ERROR...reserved bits 2-15 are NOT zero";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 2-15 = 0x%4.4x", data.Reserved);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and set to ZERO";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (3-2)*2, data.data2[0] );
   s1 += OutBuff;
   s1 += "   Active HIGH IRQs Bit 0 = IRQ0 and Bit 15 = IRQ15";
   sprintf(OutBuff, "\n\t0x%2.2x: 0x%4.4x", (4-2)*2, data.data2[1] );
   s1 += OutBuff;
   if(data.data2[1])
	s1 += "   <- ERROR...all bits are reserved and should be set to zero";
   else
	s1 += "   Addl active HIGH IRQs all bits are reserved and set to ZERO";
   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (5-2)*2, data.data2[2] );
   s1 += OutBuff;
   s1 += "   Active LOW IRQs Bit 0 = IRQ0 and Bit 15 = IRQ15";
   sprintf(OutBuff, "\n\t0x%2.2x: 0x%4.4x", (6-2)*2, data.data2[3] );
   s1 += OutBuff;
   if (data.data2[3])
	s1 += "   <- ERROR...all bits are reserved and should be set to zero";
   else
	s1 += "   Addl active LOW IRQs all bits are reserved, reset to ZERO";

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_ResetSocket( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_ResetSocket( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          Socket;
  } paramblock;

   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Socket  = param.getInt("SOCKET");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x90, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_ResetSocket", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc !=0)
   {
      printerror(ulRc, param);
      return ulRc;
   }
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_SetSocket( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_SetSocket( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          Socket;
    BYTE          InterruptMask;
    BYTE          VccLevel;
    BYTE          VppLevel;
    BYTE          SocketState;
    BYTE          SocketControl;
    USHORT        IFIRQ;
  } paramblock;

   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Socket = param.getInt("SOCKET");
   paramblock.InterruptMask = param.getInt("INTERRUPTMASK");
   paramblock.VccLevel = param.getInt("VCCLEVEL");
   paramblock.VppLevel = param.getInt("VPPLEVEL");
   paramblock.SocketState = param.getInt("SOCKETSTATE");
   paramblock.SocketControl = param.getInt("SOCKETCONTROL");
   paramblock.IFIRQ = param.getInt("IFIRQ");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x8E, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_SetSocket", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc !=0)
   {
      printerror(ulRc, param);
      return ulRc;
   }
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetPage( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetPage( Kwd_List & param )
{
  struct
  {
    BYTE        Adapter;
    BYTE        Window;
    BYTE        Page;
//  PageAttribs bits
    int         AttribCommon	:1;
    int		EnabDis		:1;
    int		PageWrite	:1;
    BYTE	Reserved	:5;
    USHORT      Offset;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t     ");
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Window = param.getInt("WINDOW");
   paramblock.Page = param.getInt("PAGE");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x8A, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetPage", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 += "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tWindow #  : ";
   s1 += (IString)(LONG)paramblock.Window + (IString)"\n\tPage #    : ";
   s1 += (IString)(LONG)paramblock.Page;

   s1 += "\n\tPage Attributes  : ";
   s1.CheckBit(paramblock.AttribCommon, 3, "",
		"Attribute Memory", "Common Memory");
   s1.CheckBit(paramblock.EnabDis, 3, "",
		"Enable", "Disable");
   s1.CheckBit(paramblock.PageWrite, 3, "Page Write Protect by h/w is ",
		"SET:ERROR...should always be zero", "RESET");
   if(paramblock.Reserved)
   {
	s1 += "\n\t     ERROR...reserved bits 3-7 are NOT zero";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 3-7 = 0x%4.4x", paramblock.Reserved);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and set to ZERO";

   s1 += "\n\tOffset : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Offset);
   s1 += OutBuff;
   s1 += "\t(4 KByte units)";

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetWindow( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetWindow( Kwd_List & param )
{
  struct
  {
    BYTE        Adapter;
    BYTE        Window;
    BYTE        Socket;
    USHORT      Size;
//  State bits
    int         StateMio	:1;
    int 	StateDe		:1;
    int		StateDpw	:1;
    int		StateSip	:1;
    int		StateEisa	:1;
    int		StateEa		:1;
    BYTE	StateReserved	:2;

    BYTE        Speed;
    USHORT      Base;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t     ");
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Window = param.getInt("WINDOW");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x88, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetWindow", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 +=  "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tSocket #  : ";
   s1 += (IString)(LONG)paramblock.Socket + (IString)"\n\tWindow #  : ";
   s1 += (IString)(LONG)paramblock.Window + (IString)"\n\tSize      : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Size);
   s1 += OutBuff;
   s1 += "\n\t   (Bytes for I/O, 4 KByte units for Memory)";

   s1 += "\n\n\tState  : ";
   s1.CheckBit(paramblock.StateMio, 3, "",
		"I/O ", "Memory ");
   s1.CheckBit(paramblock.StateDe, 3, "",
		"ENABLED", "DISABLED");
   s1.CheckBit(paramblock.StateDpw, 3, "Data Path Width = ",
		"16 bits", "8 bits");
   s1.CheckBit(paramblock.StateSip, 3, "Subdivided into Pages is ",
		"SET:ERROR...should always set to zero", "RESET");
   s1.CheckBit(paramblock.StateEisa, 3, "EISA I/O  mapping is ",
		"SET:ERROR...should always set to zero", "RESET");
   s1.CheckBit(paramblock.StateEa, 3, "Enable access to IO ports in EISA common I/O area is ",
		"SET:ERROR...should always set to zero", "RESET");
   if (paramblock.StateReserved)
   {
	s1 += "\n\t     ERROR: Reserved bits are NOT zero";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 6 and 7 = 0x%2.2x", paramblock.StateReserved);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and are set to ZERO";

   s1 += "\n\tSpeed  : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Speed);
   s1 += OutBuff;

   s1 += "\n\tBase   : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Base);
   s1 += OutBuff;
   s1 += "\n\t   (Bytes for I/O, 4Kbyte units for Memory)";
   s1 += "\n\t   First addressable byte = 160 (0xa0)";
   s1 += "\n\t   Last addressable byte  = 255 (0xff)";

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_InqWindow( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_InqWindow( Kwd_List & param )
{
  struct
  {
    BYTE        Adapter;
    BYTE        Window;
//  Cpabilities bits
    int         CommonMem	:1;
    int		AttribMem	:1;
    int		IOspace		:1;
    BYTE	Reserved	:5;
    BYTE        Sockets;
  } paramblock;

   IStringPcmc     s1("\n", "\n\t", "\n\t     ");
   struct
   {
	USHORT	data1[2];
   //   Characteristics(bit-mapped)
	USHORT	ProgBa		:1;
	USHORT  ProgWinSize	:1;
	USHORT  WinEnab		:1;
	USHORT	EightDataBus	:1;
	USHORT	SixteenDataBus	:1;
	USHORT	ReqBa		:1;
	USHORT	PowerTsg	:1;
	USHORT	ReqCardOff	:1;
	USHORT	PageHa		:1;
	USHORT  PageHs		:1;
	USHORT	PageEnab	:1;
	USHORT	SoftWriteP	:1;
	USHORT  Reserved	:4;

	USHORT	data2[13];
   } data;

   ULONG          ulDataSize = sizeof(data);
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Window = param.getInt("WINDOW");
   data.data1[0] = 32;

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x87, &paramblock,
			  ulParamSize, &ulParamSize,
			  &data, ulDataSize, &ulDataSize,
			  "PCMC_InqWindow", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 += "Adapter #    : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tWindow #     : ";
   s1 += (IString)(LONG)paramblock.Window + (IString)"\n\tCapabilities : ";
   s1.CheckBit(paramblock.CommonMem, 3, "Common Memory is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.AttribMem, 3, "Attribute Memeory is ",
		"SET", "RESET");
   s1.CheckBit(paramblock.IOspace, 3, "I/O Space is ",
		"SET", "RESET");
   if(paramblock.Reserved)
   {
	s1 += "\n\t     ERROR...reserved bits 3-7 are NOT zero";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 3-7 = 0x%4.4x", paramblock.Reserved);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and always set to ZERO";

   s1 += "\n\tSockets      : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Sockets);
   s1 += OutBuff;
   s1 += "\t(bitmap of assignable Sockets)";

   s1 += "\n\n\tWindow Characteristics Table : ";
   s1 += "\n\tBytes allocated : " ;
   sprintf(OutBuff, "0x%4.4x",data.data1[0]);
   s1 += OutBuff;

   s1 += "\n\tBytes returned  : " ;
   sprintf(OutBuff, "0x%4.4x",data.data1[1]);
   s1 += OutBuff;

   sprintf(OutBuff, "\n\n\t0x%2.2x:", 0);
   s1 += OutBuff;
   s1 += "  Characteristics(bit-mapped):";
   s1.CheckBit(data.ProgBa, 3, "Programmable Base Address is ",
		"SET", "RESET");
   s1.CheckBit(data.ProgWinSize, 3, "Programmable Window Size is ",
		"SET", "RESET");
   s1.CheckBit(data.WinEnab, 3, "Window Disable/Enable Supported is ",
		"SET", "RESET");
   s1.CheckBit(data.EightDataBus, 3, "Eight-bit Data Bus Supported is ",
		"SET", "RESET");
   s1.CheckBit(data.SixteenDataBus, 3, "Sixteen-bit Data Bus Supported is ",
		"SET", "RESET");
   s1.CheckBit(data.ReqBa, 3, "Requires Base Address Alignment on Size Boundary is ",
		"SET", "RESET");
   s1.CheckBit(data.PowerTsg, 3, "Power of Two Size Granularity is ",
		"SET", "RESET");
   s1.CheckBit(data.ReqCardOff, 3, "Requires Card Offset Alignment on Size Boundary is ",
		"SET", "RESET");
   s1.CheckBit(data.PageHa, 3, "Paging Hardware Available is ",
		"SET", "RESET");
   s1.CheckBit(data.PageHs, 3, "Paging Hardware Shared is ",
		"SET", "RESET");
   s1.CheckBit(data.PageEnab, 3, "Page Disable/Enable Supported is ",
		"SET", "RESET");
   s1.CheckBit(data.SoftWriteP, 3, "Software Write-Protect Available is ",
		"SET", "RESET");
   if (data.Reserved)
   {
	s1 += "\n\t     ERROR...Reserved bits 12-15 are NOT zero";
	sprintf(OutBuff, "\n\t\t-- Reserved bits 12-15 = 0x%4.4x", data.Reserved);
	s1 += OutBuff;
   }
   else
	s1 += "\n\t     All other bits are reserved and set to ZERO";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (3-2)*2, data.data2[0]);
   s1 += OutBuff;
   s1 += "   First Byte addressable (4Kbytes Block)  0x00A0 ";
   sprintf(OutBuff, "\n\t0x%2.2x: 0x%4.4x", (4-2)*2, data.data2[1]);
   s1 += OutBuff;
   s1 += "   Last Byte addressable (4Kbytes Block)  0x00FF ";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (5-2)*2, data.data2[2]);
   s1 += OutBuff;
   s1 += "   Minimum Window Size (4Kbytes Block)  0x0001 ";
   sprintf(OutBuff, "\n\t0x%2.2x: 0x%4.4x", (6-2)*2, data.data2[3]);
   s1 += OutBuff;
   s1 += "   Maximum Window Size (4Kbytes Block)  0x0080 ";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (7-2)*2, data.data2[4]);
   s1 += OutBuff;
   s1 += "   Required Window Size Granularity(4Kbytes Blocks)  0x0001 ";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (8-2)*2, data.data2[5]);
   s1 += OutBuff;
   s1 += "   Required Base Address Alignment(4Kbytes Blocks)  0x0001 ";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (9-2)*2, data.data2[6]);
   s1 += OutBuff;
   s1 += "   Required Card Offset Alignment(4Kbytes Blocks)  0x0001 ";

   sprintf(OutBuff, "\n\n\t0x%2.2x: 0x%4.4x", (10-2)*2, data.data2[7]);
   s1 += OutBuff;
   s1 += "   Access Speed Supported (Upper Nibble--Slowest, Lower Nibble--Fastest)";

   for (int i=11; i<16; i++ )
   {
      sprintf(OutBuff, "\n\t0x%2.2x: 0x%4.4x", (i-2)*2, data.data2[i-3]);
      s1 += OutBuff;
   }

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_SetPage( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_SetPage( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          Window;
    BYTE          Page;
    BYTE          State;
    USHORT        Offset;
  } paramblock;

   IString         s1, s2, s3;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Window = param.getInt("WINDOW");
   paramblock.State = param.getInt("PGSTATE");
   paramblock.Offset = param.getInt("OFFSET");
   paramblock.Page = 0;

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x8B, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_SetPage", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc !=0)
   {
      printerror(ulRc, param);
      return ulRc;
   }
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_SetWindow( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_SetWindow( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          Window;
    BYTE          Socket;
    USHORT        Size;
    BYTE          State;
    BYTE          Speed;
    USHORT        Base;
  } paramblock;

   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.Window = param.getInt("WINDOW");
   paramblock.Socket = param.getInt("SOCKET");
   paramblock.Size = param.getInt("SIZE");
   paramblock.State = param.getInt("WINSTATE");
   paramblock.Speed = param.getInt("SPEED");
   paramblock.Base = param.getInt("BASE");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x89, &paramblock,
                          ulParamSize, &ulParamSize, NULL, 0, NULL,
                          "PCMC_SetWindow", "only call",
                          param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc !=0)
   {
      printerror(ulRc, param);
      return ulRc;
   }
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_GetEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:
**
**************************************************************************
*/

APIRET _export PCMC_GetEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
    BYTE          EDC;
    BYTE          Socket;
    BYTE          State;
    BYTE          Type;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");
   paramblock.EDC = param.getInt("EDC");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_GetEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter + (IString)"\n\tSocket  : ";
   s1 += (IString)(LONG)paramblock.Socket + (IString)"\n\tEDC     : ";
   s1 += (IString)(LONG)paramblock.EDC + (IString)"\n\tState : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.State);
   s1 += OutBuff;

   s1+= "\n\tType : ";
   sprintf(OutBuff, "0x%4.4x", paramblock.Type);
   s1+= OutBuff;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_InquireEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:    using the same function code as in GetEDC
**
**************************************************************************
*/

APIRET _export PCMC_InquireEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_InquireEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_PauseEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:    using the same function code as in GetEDC
**
**************************************************************************
*/

APIRET _export PCMC_PauseEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_PauseEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_ReadEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:    using the same function code as in GetEDC
**
**************************************************************************
*/

APIRET _export PCMC_ReadEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_ReadEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_ResumeEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:    using the same function code as in GetEDC
**
**************************************************************************
*/

APIRET _export PCMC_ResumeEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_ResumeEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_SetEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:    using the same function code as in GetEDC
**
**************************************************************************
*/

APIRET _export PCMC_SetEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");

   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_SetEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_StartEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:    using the same function code as in GetEDC
**
**************************************************************************
*/

APIRET _export PCMC_StartEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_StartEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}

/*
*************************************************************************
** Name: APIRET _export PCMC_StopEDC( Kwd_List & )
**
** Description:
**
** Parameters:  Kwd_List &.
**
** Returns:     APIRET.
**
** Cautions:    using the same function code as in GetEDC
**
**************************************************************************
*/

APIRET _export PCMC_StopEDC( Kwd_List & param )
{
  struct
  {
    BYTE          Adapter;
  } paramblock;

   IString         s1;
   ULONG          ulParamSize = sizeof(paramblock);
   HFILE          hfDrvHandle = param.getInt("DRIVEHANDLE");
   paramblock.Adapter = param.getInt("ADAPTER");

   ulRc = ddtDosDevIOCtl( hfDrvHandle, CATEGORY, 0x96, &paramblock,
			  ulParamSize, &ulParamSize, NULL, 0, NULL,
			  "PCMC_StopEDC", "only call",
			  param.files()->out1 );

   param.files()->out1.flush();
   if (ulRc != 0)
   {
      printerror(ulRc, param);
      return ulRc;
   }

   s1 = "Adapter # : ";
   s1 += (IString)(LONG)paramblock.Adapter ;

   param.files()->out1<<s1;
   param.files()->out1.flush();
   return ulRc;
}


// new class IStringPcmc function codes; added 4/94 //

void IStringPcmc::CheckBit(int bit,
			  int AlignIndicator,
			  IString BitStr,
			  IString TrueStr,
			  IString FalseStr)
{
	if (bit)
	{
		if (AlignIndicator == 1)
			*this += align1 + BitStr + TrueStr;
		else if (AlignIndicator == 2)
			*this += align2 + BitStr + TrueStr;
		else
			*this += align3 + BitStr + TrueStr;
	}
	else
	{
		if (AlignIndicator == 1)
			*this += align1 + BitStr + FalseStr;
		else if (AlignIndicator == 2)
			*this += align2 + BitStr + FalseStr;
		else
			*this += align3 + BitStr + FalseStr;
	}

}


void IStringPcmc::CheckEntry(int EntryCount, USHORT *data)
{
	CHAR	OutBuff[256];
	PMEntry *pPMEntry;
	pPMEntry = (PMEntry *)data;

	for (int i=0; i<EntryCount; i++)
	{
		sprintf(OutBuff, "\n\n\t   Entry #%1d: ", i);
		*this += OutBuff;
		if (pPMEntry->Vcc)
		{
			sprintf(OutBuff, "%3.1d VDC", (pPMEntry->Voltage)/10);
			(*this).CheckBit(pPMEntry->Voltage, 3, "Vcc  has available: ",
					 OutBuff, "0 VDC");
		}
		if (pPMEntry->Vpp1)
		{
			sprintf(OutBuff, "%3.1d VDC", (pPMEntry->Voltage)/10);
			(*this).CheckBit(pPMEntry->Voltage, 3, "Vpp1 has available: ",
					 OutBuff, "0 VDC");
		}
		if (pPMEntry->Vpp2)
		{
			sprintf(OutBuff, "%3.1d VDC", (pPMEntry->Voltage)/10);
			(*this).CheckBit(pPMEntry->Voltage, 3, "Vpp2 has available: ",
					 OutBuff, "0 VDC");
		}
		pPMEntry++; // jump to next entry //
	} // end for //

}


/*
******************************************************************
* printerror(ULONG)
*
* checks the error code returned by Socket Services calls
******************************************************************
*/
VOID printerror(ULONG &ul, Kwd_List &param)
{
 IString s1;
 s1 = "ERROR: ";
 switch (ul)
 {
   case 0xff01:
      s1 += "Specified adapter does not exist";
      break;
   case 0xff02:
      s1 += "Specified attribute is invalid";
      break;
   case 0xff03:
      s1 += "Specified base system memory address is invalid";
      break;
   case 0xff04:
      s1 += "Specified EDC generator is invalid (N/A)";
      break;

   // code 5 is reserved //

   case 0xff06:
      s1 += "Specified IRQ level is invalid";
      break;
   case 0xff07:
      s1 += "Specified PC Card offset is invalid";
      break;
   case 0xff08:
      s1 += "Specified page is invalid";
      break;
   case 0xff09:
      s1 += "Unable to complete read request";
      break;
   case 0xff0a:
      s1 += "Specified window size is invalid";
      break;
   case 0xff0b:
      s1 += "Specified socket does not exist";
      break;

   // code 12 is reserved

   case 0xff0d:
      s1 += "Window type specified is not available";
      break;
   case 0xff0e:
      s1 += "Specified Vcc power level index is invalid";
      break;
   case 0xff0f:
      s1 += "Specified Vpp1 or Vpp2 power level index is invalid";
      break;

   // codes 16 and 19 are reserved

   case 0xff11:
      s1 += "Specified window does not exist";
      break;
   case 0xff12:
      s1 += "Unable to complete write request";
      break;
   case 0xff14:
      s1 += "No PC Card in socket";
      break;
   case 0xff15:
      s1 += "Socket Services implementation does not support function";
      break;
   case 0xff16:
      s1 += "Request processor mode not supported";
      break;
   case 0xff17:
      s1 += "Specified speed is unavailable";
      break;
   case 0xff18:
      s1 += "Unable to process request at this time --retry later please";
      break;
   case 0xff19:
      s1 += "An undefined error has occurred";
      break;
   default: 
      s1 += "Error code unknown to PCMCIA Socket Services";
      break;
   }
   param.files()->out1<<s1;
   ul = 0;
   return;
}


