/*
************************************************************************
** Copyright: (C) IBM BocaRaton Ltd. 1994
**
** Classification: IBM Test Tool
**
** Original Author and Date: Richard Copeland, June 1994.
**
** Revisions:  03/05/05 - Adam Loving porint to IBM CSet
**
** Comments:
**
**
** Public Class/ Functions:
**
*************************************************************************
*/

#include "ddtkbd.h"
#include "kbd.h"
#include "apicover.h"

// DDTKBD.CPP defines the stub functions for testing the keyboard IDC's for
//	the device dependent keyboard driver.  The stub functions require
//	the test device driver, TESTKBD.SYS, to be loaded at boot time.

// Structures used by stub functions

struct CapStruc { // Used in query_caps stubs
  USHORT DevFlags;
  USHORT KeyCount;
  USHORT MaxTypa;
  USHORT MinTypa;
  USHORT MaxDelay;
  USHORT MinDelay;
};

struct IDstruc { // Used in query_IDs stubs
  USHORT idlen;
  char idbytes[8];
};

struct CmdStruc { // Used in send_generic_cmd stub
  USHORT wait;
  USHORT strlength;
  UCHAR bytes[256];
};

// Internally used function prototypes -- each of these
// 	is an implementation of the interrupt/non-interrupt
//	pair of stub functions.  The second parameter is a
//	boolean value representing whether the IDC is an
//	interrupt IDC or not.

// Function: diskey
// Description: Implement disable_keyboard and
//	disable_keyboard_int.
APIRET diskey(Kwd_List&, int);

// Function: enkey
// Description: Implement enable_keyboard and
//	enable_keyboard_int.
APIRET enkey(Kwd_List&, int);

// Function: reshdw
// Description: Implement reset_hardware and
//	reset_hardware_int.
APIRET reshdw(Kwd_List&, int);

// Function: settypa
// Description: Implement set_typematic and
//	set_typematic_int.
APIRET settypa(Kwd_List&, int);

// Function: setled
// Description: Implement set_LEDs and set_LEDs_int.
APIRET setled(Kwd_List&, int);

// ***************************************************
// ***************************************************
// ************** Device Dependent IDCs **************
// ***************************************************
// ***************************************************

// **************************
// * Open / Close functions *
// **************************


// Function: open
// Description:  Opens the TESTKBD.SYS device driver, returns a handle
//	in keyword "HANDLE", and calls the OPEN IDC of KBDDD.SYS.
APIRET _export open(Kwd_List& param) {
  HFILE hf; 	// Holds the new handle to TESTKBD.SYS
  ULONG action; // Dummy variable for ddtDosOpen
  // Open TESTKBD.SYS
  ULONG ulRC = ddtDosOpen("TESTKBD", &hf, &action, 0, 0,
	OPEN_ACTION_OPEN_IF_EXISTS, OPEN_FLAGS_FAIL_ON_ERROR |
	OPEN_SHARE_DENYNONE | OPEN_ACCESS_READWRITE, NULL,
	"DdtKbd Open", "only call", param.files()->out1);
  if(ulRC) return ulRC;
  param.set("HANDLE", hf); // Set the HANDLE keyword
  KbdStruc ks;	// KbdStrucs are used in all IDC commands
  ks.func = 0; 	// Open
  ks.var1 = 0;	// Not used
  ks.var2 = 0;	// Not used
  ULONG ksl = sizeof(ks); // Necessary for ddtDosDevIOCtl
  // Call OPEN IDC.
  ulRC = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Open", "only call",
	param.files()->out1);
  if(ulRC) return ulRC;
  return ks.func;	// ks.func holds the IDC return value
}

// Function: close
// Description: Calls the CLOSE IDC of KBDDD.SYS and closes the
//    	TESTKBD.SYS device driver.  Also removes "HANDLE" from
//	the keyword list.
APIRET _export close(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE"); // Get the TESTKBD.SYS handle
  KbdStruc ks;	// KbdStrucs are used in all IDC commands
  ks.func = 1; 	// Close code
  ks.var1 = 0;  // Not used
  ks.var2 = 0;  // Not used
  ULONG ksl = sizeof(ks); // Necessary for ddtDosDevIOCtl
  // Call CLOSE IDC
  ULONG ulRC = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Close", "only call",
	param.files()->out1);
  if(ulRC) return ulRC;
  // Close the TESTKBD.SYS device driver
  ulRC = ddtDosClose(hf, "DdtKbd Close", "only call", param.files()->out1);
  if(ulRC) return ulRC;
  param.unset("HANDLE"); // Handle is no longer valid
  return ulRC;
}

// *******************
// * Query functions *
// *******************

// Function: query_caps
// Description: Calls the QUERY CAPABILITIES IDC of KBDDD.SYS and
//	outputs the results to the logfile.  Also sets the
//	keywords: MAXTYPA, MINTYPA, MAXDELAY, MINDELAY, KEYCOUNT, and
//	NUMLEDS.
APIRET _export query_caps(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE"); // Get the TESTKBD.SYS handle
  KbdStruc ks;	// KbdStrucs are used in all IDC commands
  ks.func = 2; 	// Query Capabilities function code
  ks.var1 = 0;  // Not used
  ks.var2 = 0;  // Not used
  ULONG ksl = sizeof(ks);  // Necessary for ddtDosDevIOCtl
  CapStruc cs;  // Structure which holds the capabilities of the keyboard
  ULONG csl = sizeof(cs); // Necessary for ddtDosDevIOCtl
  // Call QUERY CAPABILITES IDC
  ULONG ulRC = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	&cs, csl, &csl, "DdtKbd Query_Caps", "only call",
	param.files()->out1);
  if(ulRC) return ulRC;
  DdtOStream& out1 = param.files()->out1;  // Acquire an alias for the logfile
  // Stream capabilites to the logfile
  IString temp = (IString)"Keyboard Capabilities:";		//AL
  out1<<temp;							//AL
  if(cs.DevFlags & 0x1) // Bit 0 of DevFlags maps True Keyboard status
  { temp = (IString)"     True Keyboard";			//AL
    out1<< temp;}						//AL
  else
  { temp = (IString)"     Not True Keyboard";			//AL
    out1<< temp;}
  if(cs.DevFlags & 0x2) // Bit 1 of DevFlags maps Hotplug detection
  { temp = (IString)"     Hotplug Detection";			//AL
    out1<< temp;}			
  else
  { temp = (IString)"     No Hotplug Detection";		//AL
    out1<< temp;}
  {
    IString leds        =(IString)"     Number of LEDs: " + (IString)((cs.DevFlags & 0x70)/0x10); //AL introducing temps
    IString keycount    =(IString)"     Key Count:      " + (IString)cs.KeyCount;
    IString maxtypa     =(IString)"     Max Typematic:  " + (IString)cs.MaxTypa;
    IString mintypa     =(IString)"     Min Typematic:  " + (IString)cs.MinTypa;
    IString maxdelay    =(IString)"     Max Delay:      " + (IString)cs.MaxDelay;
    IString mindelay    =(IString)"     Min Delay:      " + (IString)cs.MinDelay;

    out1<<(leds);
    out1<<(keycount);
    out1<<(maxtypa);
    out1<<(mintypa);
    out1<<(maxdelay);
    out1<<(mindelay);
  }
  // Set keywords for later use
  param.set("MAXTYPA", cs.MaxTypa);
  param.set("MINTYPA", cs.MinTypa);
  param.set("MAXDELAY", cs.MaxDelay);
  param.set("MINDELAY", cs.MinDelay);
  param.set("KEYCOUNT", cs.KeyCount);
  param.set("NUMLEDS", (cs.DevFlags & 0x70) >> 0x4);
  return ulRC;
}

// Function: query_typematic
// Description: Calls the QUERY TYPEMATIC IDC of KBDDD.SYS and
// 	outputs the results to the logfile.  Also sets the keywords:
//	TYPASEL and DELAYSEL.
APIRET _export query_typematic(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 3; 	// Query typematic rate & delay function code
  ks.var1 = 0;  // Will hold AX, which maps the rate and delay
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG ulRC = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Query_Typematic", "only call",
	param.files()->out1);
  if(ulRC) return ulRC;
  int delaysel = (ks.var1 & 0x60) >> 5;  		// Mask and shift the delay selector
  IString typematic, delay;
  int typematicsel = (ks.var1 & 0x1F);		// Mask the typematic selector
  switch(typematicsel) {
  // This switch statement is a table to look up the typematic rate in
  // characters per second of the keyboard driver.
    case 0x00: typematic = "30.0"; break;  case 0x10: typematic = "7.5"; break;
    case 0x01: typematic = "26.7"; break;  case 0x11: typematic = "6.7"; break;
    case 0x02: typematic = "24.0"; break;  case 0x12: typematic = "6.0"; break;
    case 0x03: typematic = "21.8"; break;  case 0x13: typematic = "5.5"; break;
    case 0x04: typematic = "20.0"; break;  case 0x14: typematic = "5.0"; break;
    case 0x05: typematic = "18.5"; break;  case 0x15: typematic = "4.6"; break;
    case 0x06: typematic = "17.1"; break;  case 0x16: typematic = "4.3"; break;
    case 0x07: typematic = "16.0"; break;  case 0x17: typematic = "4.0"; break;
    case 0x08: typematic = "15.0"; break;  case 0x18: typematic = "3.7"; break;
    case 0x09: typematic = "13.3"; break;  case 0x19: typematic = "3.3"; break;
    case 0x0a: typematic = "12.0"; break;  case 0x1a: typematic = "3.0"; break;
    case 0x0b: typematic = "10.9"; break;  case 0x1b: typematic = "2.7"; break;
    case 0x0c: typematic = "10.0"; break;  case 0x1c: typematic = "2.5"; break;
    case 0x0d: typematic = "9.2" ; break;  case 0x1d: typematic = "2.3"; break;
    case 0x0e: typematic = "8.6" ; break;  case 0x1e: typematic = "2.1"; break;
    case 0x0f: typematic = "8.0" ; break;  case 0x1f: typematic = "2.0"; break;
  }
  switch(delaysel) {
  // This switch statement is a table to look up the delay value
  // in milliseconds of the keyboard driver.
    case 0x00: delay = "250 ms"; break;
    case 0x01: delay = "500 ms"; break;
    case 0x02: delay = "750 ms"; break;
    case 0x03: delay = "1000 ms"; break;
  }
  // Stream results to logfile
  IString typemessage = (IString)"Typematic rate and delay:"; 		//AL
  IString typerate    = (IString)"     Typematic Rate:  "+typematic+" chars/sec"; //AL
  IString typedelay   = (IString)"     Delay value:     "+delay;

  param.files()->out1<<typemessage;
  param.files()->out1<<typerate;
  param.files()->out1<<typedelay;
  // Set keywords
  param.set("TYPASEL", typematicsel);
  param.set("DELAYSEL", delaysel);
  return ulRC;
}

// Function: query_LEDs
// Description: Calls the QUERY LEDs IDC of KBDDD.SYS and
//	outputs the results to the logfile.  This IDC queries
//	which LEDs are currently lit.
APIRET _export query_LEDs(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 4; 	// Query LEDs function code
  ks.var1 = 0;  // Will hold AX, which maps the lit LEDs
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG ulRC = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Query_LEDs", "only call",
	param.files()->out1);
  if(ulRC) return ulRC;
  // Stream results to logfile
  IString temp = (IString)"Query LEDs:";	//AL
  param.files()->out1<<temp;			//AL intro'd following temps
  IString scroll_on 	=(IString)"     Scroll Lock is ON";	
  IString scroll_off	=(IString)"     Scroll Lock is OFF";	
  IString num_on	=(IString)"     Num Lock is ON";	
  IString num_off	=(IString)"     Num Lock is OFF";	
  IString caps_on	=(IString)"     Caps Lock is ON";	
  IString caps_off	=(IString)"     Caps Lock is OFF";	
  if(ks.var1 & 0x1) param.files()->out1<<scroll_on;		
  else param.files()->out1<<scroll_off;				
  if(ks.var1 & 0x2) param.files()->out1<<num_on;		
  else param.files()->out1<<num_off;				
  if(ks.var1 & 0x4) param.files()->out1<<caps_on;		
  else param.files()->out1<<caps_off;				
  return ulRC;
}

// Function: query_IDs
// Description: Calls the QUERY IDs IDC of KBDDD.SYS and outputs
//	the results to the logfile.  Also sets the keywords:
//	ID and KEYCOUNT.  This IDC returns the byte ID of the
//	keyboard as well as the number of keys.
APIRET _export query_IDs(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 5; 	// Query IDs function code
  ks.var1 = 0;  // Will hold AX, which maps the number of keys
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  IDstruc ids;
  ULONG idsl = sizeof(ids);
  ULONG ulRC = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	&ids, idsl, &idsl, "DdtKbd Query_IDs", "only call",
	param.files()->out1);
  if(ulRC) return ulRC;
  // Stream results to logfile
  IString queryid 	= (IString)"Query IDs:";		//AL
  IString keycount1	=(IString)"     Key Count:  ???";
  IString keycount2	=(IString)"     Key Count:  "+(IString)ks.var1;
  IString idcodelen	=(IString)"     ID Code Len:"+(IString)ids.idlen;
  param.files()->out1<<queryid;
  // ks.var1 is the key count.  If ==0, the key count is unknown.
  if(ks.var1 == 0) param.files()->out1<<keycount1;
  else param.files()->out1<<keycount2;
  param.files()->out1<<idcodelen;
  // Format ID code to an IString
  IString idcode;
  {
    char buffer[40];
    for(int i = 0; i<ids.idlen && i<8; i++) {
      unsigned int i1 = ids.idbytes[i];
      i1 = i1 & 0xFF;  // Mask off 1 byte
      sprintf(buffer, "%.2x", i1);
      idcode = idcode + buffer;
    }
  }
  IString temp = (IString)"     ID Code:    0x"+idcode;
  param.files()->out1<<temp;
  param.set("ID", idcode);
  if(ks.var1) param.set("KEYCOUNT", ks.var1);
  return ulRC;
}

// Function: query_disabled
// Description: Calls the QUERY DISABLED IDC of KBDDD.SYS and
//	outputs the results to the logfile.  This function simply
//	returns whether the keyboard has been disabled by a previous
//	call to disable_keyboard.
APIRET _export query_disabled(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 6; 	// Query disabled function code
  ks.var1 = 0;  // Will hold AX, the disabled flag
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG ulRC = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Query_Disabled", "only call",
	param.files()->out1);
  if(ulRC) return ulRC;
  // Stream result to logfile
  IString disabled =(IString)"Keyboard disabled";
  IString enabled  =(IString)"Keyboard enabled";
  if(ks.var1) param.files()->out1<<disabled;
  else param.files()->out1<<enabled;
  return ulRC;
}

// Function: query_kbd_ready
// Description: Calls the QUERY KBD READY IDC of KBDDD.SYS and
//	outputs the results to the logfile.
APIRET _export query_kbd_ready(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 0xf; 	// Query keyboard ready function code
  ks.var1 = 0;		// Will hold ready flag
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Query_Kbd_Ready", "only call",
	param.files()->out1);
  if(rc) return rc;
  // Stream result to logfile
  IString ready = (IString)"Keyboard ready";		//AL
  IString notready = (IString)"Keyboard not ready";	//AL
  if(ks.var1) param.files()->out1<<ready;
  else param.files()->out1<<notready;
  return rc;
}

// ****************************************************
// * The following functions change the state of the  *
// * keyboard.  Each of these IDCs can be executed in *
// * interrupt or non-interrupt mode.  Interrupt mode *
// * is specified by the tag "_int" at the end of the *
// * function name.                                   *
// ****************************************************

// Function: disable_kbd and disable_kbd_int
// Description: Calls the DISABLE KBD IDC to disable the
//	keyboard.  WARNING: always follow a call to this
//	function with a call to enable_kbd.
APIRET _export disable_kbd(Kwd_List& param) {
  return diskey(param, 0);
}

APIRET _export disable_kbd_int(Kwd_List& param) {
  return diskey(param, 1);
}

// Function: enable_kbd and enable_kbd_int
// Description: Calls the ENABLE KBD IDC to enable the
// 	keyboard.
APIRET _export enable_kbd(Kwd_List& param) {
  return enkey(param, 0);
}

APIRET _export enable_kbd_int(Kwd_List& param) {
  return enkey(param, 1);
}

// Function: reset_hardware and reset_hardware_int
// Description: Calls the RESET HARDWARE IDC to reset the
//	hardware to some base, start state.
APIRET _export reset_hardware(Kwd_List& param) {
  return reshdw(param, 0);
}

APIRET _export reset_hardware_int(Kwd_List& param) {
  return reshdw(param, 1);
}

// Function: set_typematic and set_typematic_int
// Description:  Calls the SET TYPEMATIC IDC to set the
//	typematic rate and delay.  Takes two keywords,
//	TYPASEL (0-31) and DELAYSEL (0-3) in addition to
//	the TESTKBD.SYS handle.
APIRET _export set_typematic(Kwd_List& param) {
  return settypa(param, 0);
}

APIRET _export set_typematic_int(Kwd_List& param) {
  return settypa(param, 1);
}

// Function: set_LEDs and set_LEDs_int
// Description: Calls the SET LEDS IDC to set the LEDs.  The
//	one keyword is the flag LEDS, which should consist of
//	a three-digit binary number denoting which LEDs to set
//	and which to clear:
//		xxx
//		123
//    	1: Caps Lock
//	2: Num Lock
//	3: Scroll Lock
APIRET _export set_LEDs(Kwd_List& param) {
  return setled(param, 0);
}

APIRET _export set_LEDs_int(Kwd_List& param) {
  return setled(param, 1);
}

// Function: diskey
// Description: Implement disable_keyboard and
//	disable_keyboard_int.
APIRET diskey(Kwd_List& param, int interrupt) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  // Set function code according to interrupt value
  if(!interrupt) ks.func = 7;	// Disable keyboard fucntion
  else ks.func = 0x8007;
  ks.var1 = 0;
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc =  ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Disable_Kbd", "only call",
	param.files()->out1);
  return rc;
}

// Function: enkey
// Description: Implement enable_keyboard and
//	enable_keyboard_int.
APIRET enkey(Kwd_List& param, int interrupt) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  // Set function code according to interrupt value
  if(!interrupt) ks.func = 8; 	// Enable keyboard function code
  else ks.func = 0x8008;
  ks.var1 = 0;
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Enable_Kbd", "only call",
	param.files()->out1);
  return rc;
}

// Function: reshdw
// Description: Implement reset_hardware and
//	reset_hardware_int.
APIRET reshdw(Kwd_List& param, int interrupt) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  // Set function code according to interrupt value
  if(!interrupt) ks.func = 9; 	// Reset hardware function code
  else ks.func = 0x1009;
  ks.var1 = 0;
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Reset_Hardware", "only call",
	param.files()->out1);
  return rc;
}

// Function: settypa
// Description: Implement set_typematic and
//	set_typematic_int.
APIRET settypa(Kwd_List& param, int interrupt) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  int rate = param.getInt("TYPASEL");
  int delay = param.getInt("DELAYSEL");
  // Set function code according to interrupt value
  if(!interrupt) ks.func = 0xa; 	// Set typematic function code
  else ks.func = 0x800a;
  ks.var1 = ((rate % 32) << 0) | ((delay % 4) << 5); // Map selectors
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Set_Typematic", "only call",
	param.files()->out1);
  return rc;
}

// Function: setled
// Description: Implement set_LEDs and set_LEDs_int.
APIRET setled(Kwd_List& param, int interrupt) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  int flags = param.getInt("LEDS");
  // Map flags to binary
  flags = (flags / 100) * 0x4 + ((flags % 100) / 10) * 0x2+(flags % 10);
  // Set function code according to interrupt value
  if(!interrupt) ks.func = 0xb; 	// Set LEDs function code
  else ks.func = 0x800b;
  ks.var1 = flags;
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Set_LEDs", "only call",
	param.files()->out1);
  return rc;
}

// ****************************************
// * Miscellaneous KBDDD.SYS IDC commands *
// ****************************************

// Function: send_generic
// Description: Calls the SEND GENERIC IDC to send a generic
//	command string to the keyboard.  The command string
//	is specified by the "COMMAND" keyword, and the device
//	driver will wait for an acknowlegement if the "ACK"
//	keyword is set to a nonzero value.
APIRET _export send_generic(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  BYTE ack = param.getInt("ACK"); // Is an acknowlegement expected?
  KbdStruc ks;
  ks.func = 0xe; 	// Send generic command function code
  ks.var1 = 0;
  ks.var2 = 0;
  CmdStruc cs;
  cs.wait = (ack!=0);
  // Currently, this function supports only string generic commands.
  // Since most keyboards do not respond to generic commands, this
  // limitation is most likely unnoticeable.
  cs.strlength = strlen(param["COMMAND"]);
//  strncpy(cs.bytes, param["COMMAND"], 255);  // AL
  strncpy((char*)&(cs.bytes[0]), param["COMMAND"], 255);
  cs.bytes[255] = 0;				//AL this is probably broken
  ULONG ksl = sizeof(ks);
  ULONG csl = sizeof(cs);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	&cs, csl, &csl, "DdtKbd Send_Generic", "only call",
	param.files()->out1);
  return rc;
}

// Function: flush_partial_keys
// Description: Calls the FLUSH PARTIAL KEYS IDC to reset the
//	device driver to a home state.
APIRET _export flush_partial_keys(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 0x10; 	// Flush partial keys function code
  ks.var1 = 0;
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Flush_Partial_Keys", "only call",
	param.files()->out1);
  return rc;
}

// Function: save_state and restore_state
// Description: Calls the SAVE STATE or RESTORE STATE IDC to
//	be used with Advanced Power Management.
APIRET _export save_state(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 0x12; 	// Save state function code
  ks.var1 = 0;
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Save_State", "only call",
	param.files()->out1);
  return rc;
}

APIRET _export restore_state(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc ks;
  ks.func = 0x13; 	// Restore state function code
  ks.var1 = 0;
  ks.var2 = 0;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFF, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd Restore_State", "only call",
	param.files()->out1);
  return rc;
}

//-----------------------------------------------------------------------------
// ***************************************************
// ***************************************************
// ************* Device Independent IDCs *************
// ***************************************************
// ***************************************************

// Function: open_di
// Description: Assuming that TESTKBD.SYS has already been opened
//	by the stub open, open a channel for IDC with KBDDI.SYS.  Sets
//	the keyword IDCHANDLE for a handle to the DI IDC.
APIRET _export open_di(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc2 ks;  // KbdStruc2s are used for DI IDCs.
  ks.func = 0x0000;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFE, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd open_di", "only call",
	param.files()->out1);
  if(rc) return rc;
  if(ks.ax==0xFFFF) return ks.ax; // The only DI IDC error coed
  param.set("IDCHANDLE", ks.ax);
  return 0;
}

// Function: close_di
// Description: Close the IDC channel with KBDDI.SYS.
APIRET _export close_di(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc2 ks;
  ks.func = 0x0001;
  ks.handle = param.getInt("IDCHANDLE");
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFE, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd close_di", "only call",
	param.files()->out1);
  if(rc) return rc;
  if(ks.ax==0xFFFF) return ks.ax;
  return 0;
}

// Function: process_keystroke
// Description: Send the PROCESS KEYSTROKE IDC to KBDDI.SYS and
//	process a keystroke.  This requires the keywords:
//	XCHAR, SCAN, STATUS, DSHIFT, SHIFT, and TIME.
APIRET _export process_keystroke(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc2 ks;
  char keystroke[20];
  CharData * cd = (CharData *) keystroke;
  // Get the CharData structure
  cd->XChar = param.getInt("XCHAR");
  cd->Scan = param.getInt("SCAN");
  cd->Status = param.getInt("STATUS");
  cd->DShift = param.getInt("DSHIFT");
  cd->Shift = param.getInt("SHIFT");
  cd->Time = param.getInt("TIME");
  // First, we must move the keystroke buffer into the
  //	TESTKBD.SYS data segment so we can use the IDC later.
  ks.func = 0x0005; // Code to set the keystroke buffer
  ULONG ksl = sizeof(ks);
  ULONG keystrokelen = sizeof(keystroke);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFE, &ks, ksl, &ksl,
	keystroke, keystrokelen, &keystrokelen,
	"DdtKbd process_keystroke", "set buffer call",
	param.files()->out1);
  if(rc) return rc;
  // Now, we can actually use the IDC.
  ks.func = 0x0002;
  ks.handle = param.getInt("IDCHANDLE");
  rc = ddtDosDevIOCtl(hf, 0xFC, 0xFE, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd process_keystroke", "process keystroke call",
	param.files()->out1);
  if(rc) return rc;
  if(ks.ax==0xFFFF) return ks.ax;
  return 0;
}

// Function: process_reinit
// Description: Sent the PROCESS REINIT IDC to KBDDI.SYS to
//	process a reinit.
APIRET _export process_reinit(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  KbdStruc2 ks;
  ks.func = 0x0003;
  ks.handle = param.getInt("IDCHANDLE");
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFE, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd process_reinit", "only call",
	param.files()->out1);
  if(rc) return rc;
  if(ks.ax==0xFFFF) return ks.ax;
  return 0;
}

// Function: print_last_idc
// Description: Print the last IDC call to TESTKBD.SYS.  This
//	does not enter the keyboard driver at all.
APIRET _export print_last_idc(Kwd_List& param) {
  HFILE hf = param.getInt("HANDLE");
  // This stub will call the TESTKBD.SYS which is supposed to
  // 	record all IDCs processed by it.
  KbdStruc2 ks;
  ks.func = 0x0004;
  ULONG ksl = sizeof(ks);
  ULONG rc = ddtDosDevIOCtl(hf, 0xFC, 0xFE, &ks, ksl, &ksl,
	NULL, 0, NULL, "DdtKbd print_last_idc", "only call",
	param.files()->out1);
  if(rc) return rc;
  char tempbuffer[60];
  if(ks.ax != 0xFF)
    sprintf(tempbuffer, "The last IDC processed was %x", ks.ax);
  else
    sprintf(tempbuffer, "No IDCs processed");
  IString tempbuffertemp = (IString)tempbuffer;			//AL
  param.files()->out1<<tempbuffertemp;
  return 0;
}

