/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
#ifdef DCAF
/**********************************************************************/
/*                                                                    */
/*   Module          = SCRAREA.C                                      */
/*                                                                    */
/*   Description     = Screen Change Area Entry Points.               */
/*                                                                    */
/*   Function        = OpenScreenChangeArea                           */
/*                     GetScreenChangeArea                            */
/*                     CloseScreenChangeArea                          */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
#define INCL_DDICOMFLAGS
#define INCL_GRE_DCS
#define INCL_GRE_REGIONS
#define INCL_GRE_SCREEN
#define INCL_SHLERRORS
#include <eddinclt.h>
#include <memman.h>
#include <eddgextf.h>
#include <dcaf.h>
#include <dcafextf.h>

PSCA    pscaStart;

extern  SCA scaSeamless;


ULONG APIENTRY InnerGre32Entry5(HDC,ULONG,ULONG,PDC,ULONG);
ULONG APIENTRY InnerGre32Entry7(HDC,HRGN,PRECTL,HRGN,ULONG,PDC,ULONG);

/**********************************************************************/
/* OpenScreenChangeArea                                               */
/*                                                                    */
/* This routine will allocate a data area internal to the display     */
/* driver in which the driver will accumulate screen changes.  It     */
/* returns a 32 bit handle which is required to identify the area in  */
/* GetScreenChangeArea and CloseScreenChangeArea calls.               */
/*                                                                    */
/**********************************************************************/


DDIENTRY OpenScreenChangeArea( HDC       hdc,
                               PDC       pdcArg,
                               ULONG     FunN )

{
    PSCA    psca;
    HSCA    rc;

    /******************************************************************/
    /* Enter Driver.                                                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Allocate memory for the new ScreenChangeArea                   */
    /******************************************************************/
    psca = AllocateMemory( sizeof(SCA),
                           MT_SCA,
                           MO_SHARED );

    if (psca != NULL)
    {
        /**************************************************************/
        /* If this is the first SCA to be opened then we must start   */
        /* accumulating screen bounds.                                */
        /* This is done by telling the engine to turn on the          */
        /* COM_SCR_BOUND flag.                                        */
        /**************************************************************/
        if (pscaStart == NULL)
        {
            InnerGre32Entry5 ( hdc,
                               COM_SCR_BOUND,
                               COM_SCR_BOUND,
                               pdcArg,
                               NGreSetProcessControl );
        }

        /**************************************************************/
        /* Add the new area to the linked list.                       */
        /**************************************************************/
        psca->pscaNext = pscaStart;
        psca->cRects = 0;
        pscaStart = psca;

        /**************************************************************/
        /* Return the new handle.                                     */
        /**************************************************************/
        rc = (HSCA)psca;
    }
    else
    {
        /**************************************************************/
        /* There was no memory available, so return an error.         */
        /* The memory allocation failure will have been logged by the */
        /* memory allocation routine.                                 */
        /**************************************************************/
        rc = (HSCA)GPI_ERROR;
    }

    /******************************************************************/
    /* ExitDriver.                                                    */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(rc);

} /* OpenScreenChangeArea */



/**********************************************************************/
/* CloseScreenChangeArea                                              */
/*                                                                    */
/* This routine frees the data area internal to the display driver,   */
/* identified by the SCA handle, which was accumulating screen        */
/* changes.  It returns a Boolean value indicating success or failure.*/
/*                                                                    */
/* NB. Error conditions are not obvious here, but we will return TRUE */
/*     if passed a handle we recognise, and false in all other cases. */
/**********************************************************************/

DDIENTRY CloseScreenChangeArea( HDC      hdc,
                                HSCA     hsca,
                                PDC      pdcArg,
                                ULONG    FunN )

{
    PSCA    psca;
    BOOL    rc;

    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Enter Driver.                                                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* We need to take slightly different action if the supplied      */
    /* SCA is the first in the list                                   */
    /******************************************************************/
    if (pscaStart == (PSCA)hsca)
    {
        /**************************************************************/
        /* The supplied SCA is the first in the list.                 */
        /* Update pscaStart and free the memory.                      */
        /**************************************************************/
        pscaStart = pscaStart->pscaNext;
        FreeMemory( (PVOID)hsca );
        rc = TRUE;
    }
    else
    {
        /**************************************************************/
        /* Track along the linked list until we either find the SCA   */
        /* to be closed, or reach the end of the list.                */
        /**************************************************************/
        for ( psca = pscaStart;
              (psca != NULL) && (psca->pscaNext != (PSCA)hsca);
              psca = psca->pscaNext )
        {
            /**********************************************************/
            /* Do nothing - all the work is done in the FOR statement!*/
            /**********************************************************/
        }

        /**************************************************************/
        /* If we have found the SCA then update the linked list and   */
        /* free the memory.                                           */
        /**************************************************************/
        if (psca->pscaNext == (PSCA)hsca)
        {
            psca->pscaNext = psca->pscaNext->pscaNext;
            FreeMemory( (PVOID)hsca );
            rc = TRUE;
        }
        else
        {
            /**********************************************************/
            /* We did not find the SCA.                               */
            /* Return error.                                          */
            /**********************************************************/
            LogError(PMERR_NO_HANDLE);
            rc = FALSE;
        }

    }

    /******************************************************************/
    /* ExitDriver.                                                    */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(rc);

} /* CloseScreenChangeArea */




/**********************************************************************/
/* GetScreenChangeArea                                                */
/*                                                                    */
/* This routine takes a Screen Change Area handle, and for the Screen */
/* Change Area identified adds its rectangles to the region pointed   */
/* to by the phrgn parameter.                                         */
/*                                                                    */
/* The SCA is reset to NULL as a result of this call.                 */
/*                                                                    */
/**********************************************************************/

DDIENTRY GetScreenChangeArea( HDC        hdc,
                              HSCA       hsca,
                              PHRGN      phrgn,
                              PDC        pdcArg,
                              ULONG      FunN )

{
    PSCA    psca;
    BOOL    rc;
    ULONG   iRect;

    /******************************************************************/
    /* EnterDriver                                                    */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* If there are any rects in the Seamless SCA then merge them     */
    /* now with all of the currently active SCAs.                     */
    /******************************************************************/
    for (iRect = 0; iRect < scaSeamless.cRects; iRect++)
    {
        AccumulateScreenBounds(&(scaSeamless.arcl[iRect]));
    }

    /******************************************************************/
    /* Reset the Seamless SCA.                                        */
    /******************************************************************/
    scaSeamless.cRects = 0;

    /******************************************************************/
    /* Track along the linked list until we either find the SCA to be */
    /* returned, or reach the end of the list.                        */
    /******************************************************************/
    for ( psca = pscaStart;
          (psca != NULL) && (psca != (PSCA)hsca);
          psca = psca->pscaNext )
    {
        /**************************************************************/
        /* Do nothing - all the work is done in the FOR statement!    */
        /**************************************************************/
    }

    /******************************************************************/
    /* If we found the SCA, then process it.                          */
    /******************************************************************/
    if (psca == (PSCA)hsca)
    {
        /**************************************************************/
        /* Initialise the return code to TRUE (in case there are no   */
        /* rects to return)                                           */
        /**************************************************************/
        rc = TRUE;

        /**************************************************************/
        /* We found the passed handle.                                */
        /* Add the SCA rectangles into the passed region.             */
        /**************************************************************/
        for ( iRect = 0; iRect < psca->cRects; iRect++ )
        {
            rc = InnerGre32Entry7( hdc,
                                   *phrgn,
                                   &psca->arcl[iRect],
                                   *phrgn,
                                   CRGN_OR,
                                   pdcArg,
                                   NGreCombineRectRegion );

            /**********************************************************/
            /* Break out of loop if the call failed.                  */
            /**********************************************************/
            if (rc == FALSE)
            {
                haltproc();
                break;
            }
        }

        /**************************************************************/
        /* Check if we were able to set the region sucessfully.       */
        /**************************************************************/
        if (rc != FALSE)
        {
            /**********************************************************/
            /* Ensure that the return value is correctly set to TRUE. */
            /* The region call can return other non-zero values       */
            /* that also indicate success.                            */
            /**********************************************************/
            rc = TRUE;

            /**********************************************************/
            /* We set the region OK, so reset the SCA.                */
            /**********************************************************/
            ((PSCA)hsca)->cRects = 0;
        }
    }
    else
    {
        /**************************************************************/
        /* We do not recognise the supplied handle, so log an error   */
        /* and return FALSE.                                          */
        /**************************************************************/
        LogError(PMERR_NO_HANDLE);
        rc = FALSE;
    }

    /******************************************************************/
    /* ExitDriver.                                                    */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(rc);

} /* GetScreenChangeArea */


#endif /* DCAF */

