/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**********************************************************************/
/*                                                                    */
/*   Module          = EDDVUPDC                                       */
/*                                                                    */
/*   Description     = Display Device Driver Function:                */
/*                     UpdateCursor.                                  */
/*                                                                    */
/*   Function        = UpdateCursor sets the cursor shape, position   */
/*                     and visibility                                 */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
#define INCL_DDICOMFLAGS
#define INCL_DDIMISC
#define INCL_NOSHIELDPROCS
#include <eddinclt.h>

#include <edddtypt.h>

#include <eddvcone.h>
#include <eddmcone.h>

#include <eddaextf.h>
#include <eddgextf.h>
#include <eddmextf.h>
#include <eddvextf.h>
#ifdef DCAF                                                               //DCAF
#include <dcafextf.h>                                                     //DCAF
#endif                                                                    //DCAF

#include <eddhcone.h>
#include <eddhtype.h>
#include <eddhmacr.h>

#include <cursor.h>

extern DDTType          DDT;
extern AVIOPB           AIxfer;
extern CURSORDATA       cursor_data;
extern RGNRECT          GetClipsControl;
extern ClipRectangle    GetClipsBounds;

/**********************************************************************/
/* InvertCursor inverts a rectangle on the screen, the shape and      */
/* position of which are defined by the private fields in VioPS.      */
/*                                                                    */
/* do not call this with mouse cursor disabled because it handles     */
/* mouse exclusion itself, and returns with mouse not disabled        */
/**********************************************************************/
VOID DRIVERCALL InvertAVIOCursor(PVIOPS  VioPS)
{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    DevRect     ExcludeRect;

    /******************************************************************/
    /* Calculate the rectangle that needs to be inverted in AI        */
    /* coordinates.                                                   */
    /******************************************************************/
    AIxfer.usCursorX = pdc->DCIOrigin.X + VioPS->XLatch;
    AIxfer.usCursorL = VioPS->WidthLatch - 1;
    AIxfer.usCursorY = DDT.ScreenHeight-1 -
                                 (pdc->DCIOrigin.Y + VioPS->YLatch);
    AIxfer.usCursorH = VioPS->HeightLatch - 1;

    /******************************************************************/
    /* exclude the mouse cursor if needed                             */
    /******************************************************************/
    if (cursor_data.cursor_status & CURSOR_SOFTWARE)
    {
        ExcludeRect[0].X = AIxfer.usCursorX;
        ExcludeRect[0].Y = AIxfer.usCursorY;
        ExcludeRect[1].X = AIxfer.usCursorX + AIxfer.usCursorL;
        ExcludeRect[1].Y = AIxfer.usCursorY + AIxfer.usCursorH;
        eddm_ExcludeCursor(ExcludeRect,COORD_AI);
    }

    /******************************************************************/
    /* Set up the parameters for the cursor routine.                  */
    /******************************************************************/
    AIxfer.pbmhDest = pdc->DCISelListEntry;
    AIxfer.pFirstClipRect = pdc->DCIClipRects;

    /******************************************************************/
    /* If there are less than 4 clip rectangles and we need to get    */
    /* them into the cache ensure that the bounding clip rectangle is */
    /* the size of the screen. This ensures that we get all the clip  */
    /* rectangles into the cache. Otherwise we can optimise the       */
    /* clups so we only get those which intersectng the AVIO PS.      */
    /******************************************************************/
    if (pdc->ClipChanged)
    {
        if (pdc->DCIEngineClips <= CACHED_CLIPS)
        {
            GetClipsBounds.X0 = pdc->DCIBoundingClip[0].X;
            GetClipsBounds.X1 = pdc->DCIBoundingClip[1].X + 1;
            GetClipsBounds.Y0 = pdc->DCIBoundingClip[0].Y;
            GetClipsBounds.Y1 = pdc->DCIBoundingClip[1].Y + 1;
        }
        else
        {
            GetClipsBounds.X0 = max ((pdc->DCIOrigin.X +
                                      VioPS->XLatch),
                                     0);
            GetClipsBounds.X1 = min ((pdc->DCIOrigin.X +
                                      VioPS->XLatch +
                                      VioPS->WidthLatch),
                                     pdc->DCIBoundingClip[1].X + 1);
            GetClipsBounds.Y0 = max((pdc->DCIOrigin.Y +
                                     VioPS->YLatch -
                                     VioPS->HeightLatch),
                                    0);
            GetClipsBounds.Y1 = min((pdc->DCIOrigin.Y +
                                     VioPS->YLatch + 1),
                                    (pdc->DCIBoundingClip[1].Y + 1));
        }
    }

    /******************************************************************/
    /* Cycle through all of the current clip rectangles.              */
    /******************************************************************/
    GetClipsControl.ircStart = 1;
    do /* for each rectangle comprising the clip region */
    {
        /**************************************************************/
        /* set the GetClips control structure number of rectangles    */
        /* returned to zero so that we will exit the loop if all      */
        /* rectangles processed                                       */
        /**************************************************************/
        GetClipsControl.crcReturned = 0;

        /**************************************************************/
        /* if the total number of clips comprising the region exceed  */
        /* the size of the dc cache or notify clip change has been    */
        /* called then get the cache updated from the engine          */
        /**************************************************************/
        if ( (pdc->DCIEngineClips > CACHED_CLIPS) || pdc->ClipChanged )
        {
            edda_GetClipRectangles();

            /**********************************************************/
            /* reset the start rectangle for next iteration           */
            /**********************************************************/
            GetClipsControl.ircStart += GetClipsControl.crcReturned;
        }

        /**************************************************************/
        /* Call the 386 routine                                       */
        /**************************************************************/
        if (pdc->DCIClipNum)
        {
            AIxfer.cClipRects = pdc->DCIClipNum;
            AVIOCursor();
        }

    } while ( GetClipsControl.crcReturned &&
                  (GetClipsControl.ircStart <= pdc->DCIEngineClips));

    /******************************************************************/
    /* put back the mouse cursor                                      */
    /******************************************************************/
    reenable_cursor();
}


/**********************************************************************/
/*                                                                    */
/*   This sets the cursor shape, position and visibility.  It is      */
/* called twice during a flash cycle; with a flash cycle flag on      */
/* and then off.  The cursor is clipped to the window and the         */
/* current clip regions.                                              */
/*                                                                    */
/**********************************************************************/
DDIENTRY eddv_UpdateCursor (HDC         hdc,
                            PVIOPS      VioPS,
                            PDC         pdcArg,
                            ULONG       FunN)

{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    SHORT              Height;      /* Height of cursor               */
    DevRect            CursorPos;   /* Cursor rectangle definition    */

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD);

    /******************************************************************/
    /* Apply the DC command mask to the command bits                  */
    /******************************************************************/
    COMMANDBITS(FunN) &= pdc->DCICommandMask;

    /******************************************************************/
    /* Check the cell sizes in the VioPS, and if necessary coerce     */
    /* them to match one of the available font sizes. Also set up     */
    /* the AI fonts if necessary.                                     */
    /******************************************************************/
    CheckAVIOFonts(VioPS);

    /******************************************************************/
    /* Make sure that the AVIO fonts are still cached (haven't been   */
    /* evicted)                                                       */
    /******************************************************************/
    CheckAVIOFontsCached( VioPS );

    /******************************************************************/
    /* Return immediately if this is not a Direct DC, if the COM_DRAW */
    /* bit is not set, or if there are no clip regions. Note that     */
    /* none if these cases return an error.                           */
    /******************************************************************/
    if ( (pdc->DCIDCType != OD_DIRECT) ||
         !(FunNTest(COM_DRAW))         ||
         !(pdc->DCIEngineClips) )
    {
        goto UPDATECURSOR_OK_EXIT;
    }

    /******************************************************************/
    /* Test rgfShieldStates to check if any drawing is required.      */
    /******************************************************************/
    if ( (VioPS->rgfShieldStates & fServeNobody) ||
         ( (VioPS->rgfShieldStates & fServeShieldOnly) &&
           !(VioPS->rgfShieldStates & fShieldCalling) ) )
    {
        goto UPDATECURSOR_OK_EXIT;
    }

    /******************************************************************/
    /* Compare the current cursor details with the stored cursor      */
    /* details.                                                       */
    /******************************************************************/
    if ( (VioPS->WindowOriginRow     != VioPS->RowOrgLatch)    ||
         (VioPS->WindowOriginColumn  != VioPS->ColOrgLatch)    ||
         (VioPS->TextCursorWidth     != VioPS->CursorWidth)    ||
         (VioPS->TextCursorRow       != VioPS->CursorRow)      ||
         (VioPS->TextCursorColumn    != VioPS->CursorCol)      ||
         (VioPS->TextCursorStartLine != VioPS->CursorStartLine)||
         (VioPS->TextCursorEndLine   != VioPS->CursorEndLine) )
    {
        /**************************************************************/
        /* One or more of the parameters have changed. We need to     */
        /* copy the new parameters across, validating them as we go.  */
        /* If the cursor is currently visible, we need to remove it   */
        /* from the display before we overwrite the details of its    */
        /* shape and position.                                        */
        /**************************************************************/
        if (VioPS->rgfShieldStates & fCursorShowing)
        {
            /**********************************************************/
            /* The cursor is visible so invert it.                    */
            /**********************************************************/
            InvertAVIOCursor(VioPS);

            /**********************************************************/
            /* Clear the CursorShowing flag                           */
            /**********************************************************/
            VioPS->rgfShieldStates &= ~fCursorShowing;
        }

        /**************************************************************/
        /* Now validate and copy the new parameters.                  */
        /* Start with the cursor width.                               */
        /**************************************************************/
        if ( (VioPS->TextCursorWidth == 0) ||
             (VioPS->TextCursorWidth > VioPS->CellImageWidth) )
        {
            /**********************************************************/
            /* The cursor width is invalid. Force it to be equal to   */
            /* the cell width.                                        */
            /**********************************************************/
            VioPS->TextCursorWidth = VioPS->CellImageWidth;
        }
        /**************************************************************/
        /* Store the new cursor width in CursorWidth and WidthLatch.  */
        /**************************************************************/
        VioPS->CursorWidth = (BYTE)
                           (VioPS->WidthLatch = VioPS->TextCursorWidth);

        /**************************************************************/
        /* Store the new cursor endline in CursorEndLine.             */
        /**************************************************************/
        VioPS->CursorEndLine = VioPS->TextCursorEndLine;

        /**************************************************************/
        /* Validate the cursor end line.                              */
        /**************************************************************/
        if (VioPS->CursorEndLine >= VioPS->CellImageHeight)
        {
            /**********************************************************/
            /* Cursor endline is invalid. Force it to be the bottom   */
            /* line of the cell.                                      */
            /**********************************************************/
            VioPS->CursorEndLine = VioPS->CellImageHeight - 1;
        }

        /**************************************************************/
        /* Store the new cursor startline in CursorStartLine.         */
        /**************************************************************/
        VioPS->CursorStartLine = VioPS->TextCursorStartLine;

        /**************************************************************/
        /* Validate the cursor start line.                            */
        /**************************************************************/
        if (VioPS->CursorStartLine >= VioPS->CellImageHeight)
        {
            /**********************************************************/
            /* Cursor startline is invalid. Force it to be the bottom */
            /* line of the cell.                                      */
            /**********************************************************/
            VioPS->CursorStartLine = VioPS->CellImageHeight - 1;
        }
        /**************************************************************/
        /* Now calculate the height of the cursor rectangle and       */
        /* store it in HeightLatch.                                   */
        /**************************************************************/
        Height = VioPS->CursorEndLine - VioPS->CursorStartLine + 1;
        VioPS->HeightLatch = (Height > 0) ?
                             Height : VioPS->CellImageHeight;

        /**************************************************************/
        /* Validate the cursor row                                    */
        /**************************************************************/
        if (VioPS->TextCursorRow >= VioPS->BufferRowCount)
        {
            /**********************************************************/
            /* Cursor is outside buffer, so move it inside.           */
            /**********************************************************/
            VioPS->TextCursorRow = VioPS->BufferRowCount - 1;
        }
        VioPS->CursorRow = VioPS->TextCursorRow;

        /**************************************************************/
        /* Copy the row origin.                                       */
        /**************************************************************/
        VioPS->RowOrgLatch = VioPS->WindowOriginRow;

        /**************************************************************/
        /* Now calculate the vertical pixel offset of the cursor from */
        /* the window origin, and store it in YLatch.                 */
        /**************************************************************/
        if (Height > 0)
        {
            /**********************************************************/
            /* The cursor is solid.                                   */
            /**********************************************************/
            VioPS->YLatch =
               ((VioPS->TextCursorRow - VioPS->WindowOriginRow + 1) *
                VioPS->CellImageHeight) - 1 - VioPS->CursorStartLine +
                                  (signed CHAR)VioPS->PartialCellAdjust;
        }
        else
        {
            /**********************************************************/
            /* The cursor is split.                                   */
            /**********************************************************/
            VioPS->YLatch =
                (VioPS->TextCursorRow - VioPS->WindowOriginRow + 1) *
                                             VioPS->CellImageHeight - 1;
        }

        /**************************************************************/
        /* Validate the cursor column.                                */
        /**************************************************************/
        if (VioPS->TextCursorColumn >= VioPS->BufferColumnCount)
        {
            /**********************************************************/
            /* Cursor is outside buffer, so move it inside.           */
            /**********************************************************/
            VioPS->TextCursorColumn = VioPS->BufferColumnCount - 1;
        }

        VioPS->CursorCol = VioPS->TextCursorColumn;

        /**************************************************************/
        /* Copy the column origin.                                    */
        /**************************************************************/
        VioPS->ColOrgLatch = VioPS->WindowOriginColumn;

        /**************************************************************/
        /* Now calculate the horizontal pixel offset of the cursor    */
        /* from the window origin, and store it in XLatch.            */
        /**************************************************************/
        VioPS->XLatch =
         ((SHORT)VioPS->TextCursorColumn - VioPS->WindowOriginColumn) *
                                                  VioPS->CellImageWidth;
    }

    /******************************************************************/
    /* All of the necesssary cursor fields have now been copied       */
    /* into the private area of the VioPS.                            */
    /* Decide whether the cursor is visible based upon the flag       */
    /* settings in rgfShieldStates.                                   */
    /******************************************************************/
    if ( (VioPS->rgfShieldStates & fHasTheFocus) &&
         (VioPS->rgfShieldStates & fCursorIsOn)  &&
         (VioPS->TextCursorVisible) )
    {
        /**************************************************************/
        /* The cursor needs to be displayed.                          */
        /**************************************************************/
        if (!(VioPS->rgfShieldStates & fCursorShowing))
        {
            /**********************************************************/
            /* The cursor is currently off, so it needs to be drawn.  */
            /**********************************************************/
            /**********************************************************/
            /* Do the drawing.                                        */
            /**********************************************************/
            InvertAVIOCursor(VioPS);

            /**********************************************************/
            /* Set the CursorShowing flag.                            */
            /**********************************************************/
            VioPS->rgfShieldStates |= fCursorShowing;
        }
    }
    else
    {
        /**************************************************************/
        /* The cursor should be off.                                  */
        /**************************************************************/
        if (VioPS->rgfShieldStates & fCursorShowing)
        {
            /**********************************************************/
            /* The cursor is currently on, so it needs to be          */
            /* removed.                                               */
            /**********************************************************/
            /**********************************************************/
            /* Do the drawing.                                        */
            /**********************************************************/
            InvertAVIOCursor(VioPS);

            /**********************************************************/
            /* Reset the CursorShowing flag.                          */
            /**********************************************************/
            VioPS->rgfShieldStates &= ~fCursorShowing;
        }
    }

    /******************************************************************/
    /* If bounds required                                             */
    /******************************************************************/
#ifdef DCAF                                                               //DCAF
    if ( FunNTest(COM_BOUND | COM_ALT_BOUND) ||                           //DCAF
         DCAFBoundsRequired(FunN) )                                       //DCAF
#else                                                                     //DCAF
    if (FunNTest(COM_BOUND | COM_ALT_BOUND))
#endif                                                                    //DCAF
    {
        CursorPos[0].X = VioPS->XLatch;
        CursorPos[1].X = CursorPos[0].X + VioPS->WidthLatch - 1;
        CursorPos[1].Y = VioPS->YLatch;
        CursorPos[0].Y = CursorPos[1].Y - (VioPS->HeightLatch - 1);

#ifdef DCAF                                                               //DCAF
        if (FunNTest(COM_BOUND | COM_ALT_BOUND))                          //DCAF
        {                                                                 //DCAF
#endif                                                                    //DCAF
        /**************************************************************/
        /* Update the global bounds                                   */
        /**************************************************************/
        eddg_AddBounds ((pDevRect)CursorPos,
                        FunN,
                        COORD_DEVICE_WORD);
#ifdef DCAF                                                               //DCAF
        }                                                                 //DCAF
                                                                          //DCAF
        /**************************************************************/  //DCAF
        /* Accumulate DCAF screen bounds if required                  */  //DCAF
        /**************************************************************/  //DCAF
        if (DCAFBoundsRequired(FunN))                                     //DCAF
        {                                                                 //DCAF
            AccumulateScreenBoundsThroughClips( (pDevRect)CursorPos,      //DCAF
                                                COORD_DEVICE_WORD );      //DCAF
        }                                                                 //DCAF
#endif                                                                    //DCAF
    }


UPDATECURSOR_OK_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD);

    return(OK_ZERO);
}
