/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**********************************************************************/
/*                                                                    */
/*   Module          = EDDVSETF                                       */
/*                                                                    */
/*   Description     = Display Device Driver Function:                */
/*                     DeviceSetAVIOFont                              */
/*                                                                    */
/*   Function        = This puts the address of a loadable AVIO font  */
/*                     in the DC instance data                        */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
#define INCL_DDIFONTSTRUCS
#define INCL_DDIMISC
#define INCL_GRE_FONTS
#define INCL_NOSHIELDPROCS
#include <eddinclt.h>

#include <eddacone.h>
#include <eddvcone.h>

#include <edddtypt.h>
#include <eddtcone.h>
#include <eddhcone.h>
#include <eddhtype.h>

#include <eddgextf.h>
#include <eddvextf.h>
#include <eddtextf.h>

extern PFONTCACHEINFO      pFontCacheInfo;

/**********************************************************************/
/*                                                                    */
/* DeviceSetAVIOFont is passed a font that is associated with         */
/* one of the AVIO logical fonts 1..3. These character sets are then  */
/* used to draw buffer cells whose definition is four bytes long and  */
/* whose font id is 1, 2 or 3.                                        */
/*                                                                    */
/* The function checks that the supplied character set is fixed       */
/* pitch, single plane, and that its size matches the base AVIO font  */
/* cell size.                                                         */
/*                                                                    */
/* NOTE when the base font changes size (but not when it changes      */
/* codepage) all the loadable fonts are cleared.                      */
/*                                                                    */
/**********************************************************************/

DDIENTRY eddv_DeviceSetAVIOFont2 (HDC           hdc,
                                  PFATTRS       pFattrs,
                                  PFOCAFONT     FontDef,
                                  LONG          LCIDindex,
                                  PDC           pdcArg,
                                  ULONG         FunN)

{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    LONG               AvioLCID;     /* Converted AVIO LCID (1..3)    */
    USHORT             usCodePage;
    PUSHORT            pCodePageVector;

    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Apply the DC command mask to the command bits                  */
    /******************************************************************/
    COMMANDBITS(FunN) &= pdc->DCICommandMask;

    /******************************************************************/
    /* Return an error if this is not a Direct DC                     */
    /******************************************************************/
    if ( pdc->DCIDCType == OD_INFO )
    {
        /**************************************************************/
        /* Just return OK                                             */
        /**************************************************************/
        goto SETAVIOFONT_OK_EXIT;
    }
    else if (pdc->DCIDCType != OD_DIRECT)
    {
        LogError(PMERR_INCORRECT_DC_TYPE);
        goto SETAVIOFONT_ERR_EXIT;
    }

    /******************************************************************/
    /* The AVIO LCID index that we are passed is LCID_AVIO_1/2/3 -    */
    /* which are -2, -3, -4. Make sure the lcid is in range.          */
    /******************************************************************/
    AvioLCID = -(SHORT)(LCIDindex + 1) ;
    if ( (AvioLCID < 1) || (AvioLCID > CNT_LOADABLE_LCIDS) )
    {
        /**************************************************************/
        /* The AVIO LCID is invalid. Return an error.                 */
        /**************************************************************/
        LogError(PMERR_INV_SETID);
        goto SETAVIOFONT_ERR_EXIT;
    }

    /******************************************************************/
    /* If the FontDef parameter is null then we just have to clear    */
    /* out the specified AVIO LCID.                                   */
    /******************************************************************/
    if (FontDef == FNULL)
    {
        /**************************************************************/
        /* AH-HA! we need to free the cached font if it exists...     */
        /**************************************************************/
        if (pdc->DCIAvioFonts[AvioLCID].pFocaFont)
        {
            eddt_FreeCachedFont(
                     pdc->DCIAvioFonts[AvioLCID].usFontID,
                     pdc->DCIAvioFonts[AvioLCID].usCachedFontIndex );
        }
        /**************************************************************/
        /* Clear out this AVIO LCID and leave.                        */
        /**************************************************************/
        pdc->DCIAvioFonts[AvioLCID].pFocaFont = FNULL;
        pdc->DCIAvioInfo.pfciTable[AvioLCID] = FNULL;
        goto SETAVIOFONT_OK_EXIT;
    }


    /******************************************************************/
    /* Check that the font is fixed pitch, single plane, and that its */
    /* size matches one of the supported AVIO sizes.                  */
    /******************************************************************/
    if (FontDef->fdDefinitions.fsFontdef != FONTDEFFONT1)
    {
        /**************************************************************/
        /* The font is proportionally-spaced or multiplane, so we     */
        /* cannot use it. Return an error.                            */
        /**************************************************************/
        LogError(PMERR_INV_FONT_ATTRS);
        goto SETAVIOFONT_ERR_EXIT;
    }

    /******************************************************************/
    /* Size must be same as base font!                                */
    /******************************************************************/
    if ( (pdc->DCIAvioFonts[0].pFocaFont->fdDefinitions.yCellHeight !=
                                 FontDef->fdDefinitions.yCellHeight) ||
         (pdc->DCIAvioFonts[0].pFocaFont->fdDefinitions.xCellWidth !=
                                 FontDef->fdDefinitions.xCellWidth )   )
    {
        /**************************************************************/
        /* No match was found, so return an error.                    */
        /**************************************************************/
        LogError(PMERR_INV_FONT_ATTRS);
        goto SETAVIOFONT_ERR_EXIT;
    }

    /******************************************************************/
    /* Font is OK and we're going ahead, so lets free up any old      */
    /* cached font for this lcid.                                     */
    /******************************************************************/
    if (pdc->DCIAvioFonts[AvioLCID].pFocaFont)
    {
        eddt_FreeCachedFont(
                     pdc->DCIAvioFonts[AvioLCID].usFontID,
                     pdc->DCIAvioFonts[AvioLCID].usCachedFontIndex );
        /**************************************************************/
        /* in case somthing goes wrong, clear the lcid                */
        /**************************************************************/
        pdc->DCIAvioFonts[AvioLCID].pFocaFont = FNULL;
        pdc->DCIAvioInfo.pfciTable[AvioLCID]  = FNULL;
    }

    /******************************************************************/
    /* Now get the codepage, using the fattrs value if it is passed   */
    /* in.                                                            */
    /******************************************************************/
    usCodePage = FontDef->fmMetrics.usCodePage;
    if ( pFattrs )
    {
        /**************************************************************/
        /* If Fattrs codepage is zero, use the DC current codepage.   */
        /**************************************************************/
        if ( !(usCodePage = pFattrs->usCodePage) )
        {
            usCodePage = (USHORT)pdc->DCICodePage;
        }
    }

    /******************************************************************/
    /* Get the codepage vector from the engine                        */
    /******************************************************************/
    pCodePageVector = NULL;
    if (usCodePage)
    {
        pCodePageVector = (PUSHORT)GreQueryCodePageVector(usCodePage);
    }

    pdc->DCIAvioFonts[AvioLCID].pFocaFont = FontDef;
    pdc->DCIAvioFonts[AvioLCID].usCodePage = usCodePage;
    pdc->DCIAvioFonts[AvioLCID].pCodePageVector = pCodePageVector;

    /******************************************************************/
    /* Now ensure this is cacheable. This sets the FontId and         */
    /* FontCacheIndex fields.                                         */
    /******************************************************************/
    eddt_LocateCachedFont( &(pdc->DCIAvioFonts[AvioLCID]) );

    /******************************************************************/
    /* Setup pointer in AVIOParms for this font                       */
    /******************************************************************/
    pdc->DCIAvioInfo.pfciTable[AvioLCID] = &( pFontCacheInfo[
                   pdc->DCIAvioFonts[AvioLCID].usCachedFontIndex ] );


SETAVIOFONT_OK_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(OK);

SETAVIOFONT_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(ERROR_ZERO);
}
