/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/

/**********************************************************************/
/*                                                                    */
/*   Module / File   = EDDGACCB                                       */
/*                                                                    */
/*   Description     = Display Device Driver Minor Functions:         */
/*                     AccumulateBounds,                              */
/*                     ResetBounds,                                   */
/*                     GetBoundsData.                                 */
/*                                                                    */
/*   Function        = AccumulateBounds updates both sets of current  */
/*                     Bounds (GPI and User) with a given bounding    */
/*                     rectangle.                                     */
/*                     ResetBounds resets current bounds held in the  */
/*                     DC instance data.                              */
/*                     GetBoundsData returns either the current User  */
/*                     bounds or the current GPI bounds               */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/

#define INCL_DDIMISC
#define INCL_DDICOMFLAGS
#define INCL_DOSMEMMGR
#define INCL_DOSSEMAPHORES
#include <eddinclt.h>

#include <eddgcone.h>
#include <eddgextf.h>

#include <semaphor.h>





/**********************************************************************/
/*                                                                    */
/*   Given a bounding rectangle AccumulateBounds                      */
/* updates both sets of current bounds in the DC instance data by     */
/* calling AddBounds.                                                 */
/*                                                                    */
/*   The given rectangle will be in Device or Model co-ords           */
/* depending on the Com_Transform bit.  GPI bounds are held in the DC */
/* instance data in Model co-ords; User bounds are held in Device     */
/* co-ords.                                                           */
/*                                                                    */
/**********************************************************************/

DDIENTRY eddg_AccumulateBounds (HDC              hdc,
                                pWcsRect         ArgBoundsRect,
                                PDC              pdcArg,
                                ULONG            FunN)

{
#define TFUNC "eddg_AccumBnds"

    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    USHORT         Result;              /* function call result       */
    USHORT         CoordFlag;           /* flag for which co-ords     */
    WcsRect        DeviceBounds;        /* bounds in device coords    */

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_ENTER(2);

    /******************************************************************/
    /* We can look at the flags in the high word of the function      */
    /* number to see what type of bounds we have been supplied either-*/
    /* User bounds (COM_ALT_BOUND set ) or                            */
    /* GPI  bounds (COM_BOUND set )                                   */
    /* User bounds are always supplied in screen coordinates. GPI     */
    /* bounds are in screen coordinates (COM_TRANSFORM set) or model  */
    /* coordinates (COM_TRANSFORM not set)                            */
    /******************************************************************/
    if (FunNTest(COM_TRANSFORM | COM_ALT_BOUND))
    {
        /**************************************************************/
        /* Convert the bounds from screen to device                   */
       /***************************************************************/
        CoordFlag = COORD_DEVICE;
        DeviceBounds[0].X = (*ArgBoundsRect)[0].X - pdc->DCIOrigin.X;
        DeviceBounds[0].Y = (*ArgBoundsRect)[0].Y - pdc->DCIOrigin.Y;
        DeviceBounds[1].X = (*ArgBoundsRect)[1].X - pdc->DCIOrigin.X;
        DeviceBounds[1].Y = (*ArgBoundsRect)[1].Y - pdc->DCIOrigin.Y;

        /**************************************************************/
        /* Update both sets of current bounds by calling AddBounds    */
        /**************************************************************/
        Result = eddg_AddBounds( (pDevRect)DeviceBounds,
                                 COM_BOUND | COM_ALT_BOUND,
                                 CoordFlag);
    }
    else
    {
        /**************************************************************/
        /* The bounds are in model coordinates                        */
        /**************************************************************/
        CoordFlag = COORD_MODEL;
        Result = eddg_AddBounds( (pDevRect)ArgBoundsRect,
                                 COM_BOUND | COM_ALT_BOUND,
                                 CoordFlag);
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(2);
    return ((ULONG)Result);
}
#undef TFUNC






/**********************************************************************/
/*                                                                    */
/*   GetBoundsData returns a bounding rectangle to the Engine in      */
/* BoundsRect.  Which bounding rectangle is returned depends on       */
/* Style: if Style is set to GBD_USER then User bounds are returned;  */
/* if Style is set to GBD_GPI then GPI bounds are returned;           */
/*                                                                    */
/*   User bounds are returned in Screen co-ords; GPI bounds are       */
/* returned in Model co-ords.                                         */
/*                                                                    */
/*   If user bounds are got, then they are also reset                 */
/*                                                                    */
/**********************************************************************/

DDIENTRY eddg_GetBoundsData (HDC            hdc,
                             ULONG          ArgStyle,
                             pWcsRect       ArgBoundsRect,
                             PDC            pdcArg,
                             ULONG          FunN)

{
#define TFUNC "eddg_GetBoundsData"

    /******************************************************************/
    /* Get driver semaphore and perform entry checks.                 */
    /* Do NOT cache the DC instance data.                             */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_ENTER(45);

    /******************************************************************/
    /* Return the current bounds indicated by Style                   */
    /******************************************************************/
    if (ArgStyle == GBD_USER)
    {
        /**************************************************************/
        /* Return user bounds.                                        */
        /**************************************************************/
        if (pdc->DCIDefUserBounds == TRUE)
        {
            /**********************************************************/
            /* Return default bounds                                  */
            /**********************************************************/
            (*ArgBoundsRect)[0].X = DFLT_BOUNDS_MIN;
            (*ArgBoundsRect)[0].Y = DFLT_BOUNDS_MIN;
            (*ArgBoundsRect)[1].X = DFLT_BOUNDS_MAX;
            (*ArgBoundsRect)[1].Y = DFLT_BOUNDS_MAX;
        }
        else
        {
            /**********************************************************/
            /* Return current bounds and reset them.                  */
            /**********************************************************/
            (*ArgBoundsRect)[0].X = pdc->DCIUserBounds[0].X +
                                                 pdc->DCIOrigin.X;
            (*ArgBoundsRect)[0].Y = pdc->DCIUserBounds[0].Y +
                                                 pdc->DCIOrigin.Y;
            (*ArgBoundsRect)[1].X = pdc->DCIUserBounds[1].X +
                                                 pdc->DCIOrigin.X;
            (*ArgBoundsRect)[1].Y = pdc->DCIUserBounds[1].Y +
                                                 pdc->DCIOrigin.Y;

            pdc->DCIDefUserBounds = TRUE;
        }
    }
    else
    {
        /**************************************************************/
        /* Return GPI bounds.                                         */
        /**************************************************************/
        if (pdc->DCIDefGPIBounds == TRUE)
        {
            /**********************************************************/
            /* Return default bounds.                                 */
            /**********************************************************/
            (*ArgBoundsRect)[0].X = DFLT_BOUNDS_MIN;
            (*ArgBoundsRect)[0].Y = DFLT_BOUNDS_MIN;
            (*ArgBoundsRect)[1].X = DFLT_BOUNDS_MAX;
            (*ArgBoundsRect)[1].Y = DFLT_BOUNDS_MAX;
        }
        else
        {
            /**********************************************************/
            /* Return current bounds.                                 */
            /**********************************************************/
            (*ArgBoundsRect)[0].X = pdc->DCIGPIBounds[0].X;
            (*ArgBoundsRect)[0].Y = pdc->DCIGPIBounds[0].Y;
            (*ArgBoundsRect)[1].X = pdc->DCIGPIBounds[1].X;
            (*ArgBoundsRect)[1].Y = pdc->DCIGPIBounds[1].Y;
        }
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(45);
    return (OK);
}
#undef TFUNC






/**********************************************************************/
/*                                                                    */
/*   ResetBounds resets current bounds.  Separate bits in Flags       */
/* indicate whether each type of bounds are to be reset or not.       */
/* The bounds are reset to the default values.                        */
/*                                                                    */
/**********************************************************************/

DDIENTRY eddg_ResetBounds (HDC            hdc,
                           ULONG          ArgFlags,
                           PDC            pdcArg,
                           ULONG          FunN)

{
#define TFUNC "eddg_ResetBnds"
    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_ENTER(44);

    /******************************************************************/
    /* Set User bounds default flag if relevant bit in Flags is set   */
    /******************************************************************/
    if (ArgFlags & RB_USER)
    {
        pdc->DCIDefUserBounds = TRUE;
    }

    /******************************************************************/
    /* Set GPI bounds default flag if relevant bit in Flags is set    */
    /******************************************************************/
    if (ArgFlags & RB_GPI)
    {
        pdc->DCIDefGPIBounds = TRUE;
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(44);
    return (OK);
}
#undef TFUNC
