/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/

/**********************************************************************/
/*                                                                    */
/*   Module          = EDDGPICK                                       */
/*                                                                    */
/*   Description     = Display Device Driver minor Function:          */
/*                     SetPickWindow and GetPickWindow                */
/*                                                                    */
/*   Function        = SetPickWindow stores the pick window.          */
/*                     GetPickWindow returns the pick window.         */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/

#define INCL_DDICOMFLAGS

#include <eddinclt.h>
#include <eddicone.h>
#include <eddaextf.h>
#include <eddgextf.h>
#include <memman.h>

extern RGNRECT             GetClipsControl;
extern ClipRectangle       GetClipsBounds;

/**********************************************************************/
/* private functions                                                  */
/**********************************************************************/
USHORT PASCAL NEAR intersect_rectangles (pDevPoint,
                                         pDevPoint);


/**********************************************************************/
/* SetPickWindow                                                      */
/*                                                                    */
/* HDC          DC Handle                                             */
/* pWcsRect     pointer to PickWindow rectangle in page coordinates   */
/* PDC          DC Instance Data                                      */
/* ULONG        Function Number                                       */
/*                                                                    */
/* The supplied pick rectangle is converted to device coordinates to  */
/* and stored in the DC Instance.                                     */
/* ClipPickWindow is called to reset the correlation rectangles.      */
/*                                                                    */
/**********************************************************************/

DSPENTRY eddg_SetPickWindow (HDC        hdc,
                             pWcsRect   Pwindow,
                             PDC        pdcArg,
                             ULONG      FunN)

{
#define TFUNC "eddg_SetPickWindow"
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    DevRect     PickWindowDevice; /* New pick window in device coords */

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* If the function is being called within an area or path         */
    /* definition then we raise an error and leave immediately.       */
    /******************************************************************/
    if (FunNTest(COM_AREA | COM_PATH))
    {
        if (FunNTest(COM_AREA))
        {
            LOGERR(TFUNC, "Invalid within area", FNULL, 0,
                                                     PMERR_INV_IN_AREA);
        }

        else /* COM_PATH */
        {
            LOGERR(TFUNC, "Invalid within path", FNULL, 0,
                                                     PMERR_INV_IN_PATH);
        }
        goto SETPICKWIN_ERR_EXIT;
    }

    /******************************************************************/
    /* Convert the pick window to device coordinates and store it in  */
    /* a local variable (so that we can recover if anything goes      */
    /* wrong).                                                        */
    /******************************************************************/
    if ( OK != eddg_Convert( (PULONG) Pwindow,
                             (PULONG) PickWindowDevice,
                             COORD_PAGE,
                             COORD_DEVICE_WORD,
                             2,
                             COM_TRANSFORM
                           )
       )
    {
        /**************************************************************/
        /* eddg_Convert detected an error.                            */
        /**************************************************************/
        LOGERR(TFUNC, "Convert failed", FNULL, 0, PMERR_INV_COORDINATE);
        goto SETPICKWIN_ERR_EXIT;
    }

    /******************************************************************/
    /* Everything looks good, so go ahead and store new pick window.  */
    /* Store new pick window in device coordinates.                   */
    /******************************************************************/
    pdc->DCIPickWindowDevice[0] = PickWindowDevice[0];
    pdc->DCIPickWindowDevice[1] = PickWindowDevice[1];

    /******************************************************************/
    /* Store new pick window in page coordinates.                     */
    /******************************************************************/
    pdc->DCIPickWindowPage[0] = (*Pwindow)[0];
    pdc->DCIPickWindowPage[1] = (*Pwindow)[1];


    /******************************************************************/
    /* Mark correlation rectangles as invalid                         */
    /******************************************************************/
    pdc->DCICorrInvalid = TRUE;

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(OK);


SETPICKWIN_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(ERROR_ZERO);
}
#undef TFUNC




/**********************************************************************/
/* GetPickWindow                                                      */
/*                                                                    */
/* HDC          DC Handle                                             */
/* pWcsRect     pointer to PickWindow rectangle in page coordinates   */
/* PDC          DC Instance Data                                      */
/* ULONG        Function Number                                       */
/*                                                                    */
/* Convert the pick window stored in the DC Instance to page          */
/* coordinates and return it.                                         */
/*                                                                    */
/**********************************************************************/

DSPENTRY eddg_GetPickWindow (HDC        hdc,
                             pWcsRect   Pwindow,
                             PDC        pdcArg,
                             ULONG      FunN)

{
#define TFUNC "eddg_GetPickWindow"


    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Write back the current pick window in page coordinates.        */
    /******************************************************************/
    (*Pwindow)[0] = pdc->DCIPickWindowPage[0];
    (*Pwindow)[1] = pdc->DCIPickWindowPage[1];

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(OK);

}
#undef TFUNC



/**********************************************************************/
/*                                                                    */
/*   ClipPickWindow takes the pickwindow in device coordinates        */
/* and intersects it with the clip rectangles in AI space.            */
/*                                                                    */
/*   Memory is obtained to hold the correlation rectangles. If the    */
/* allocation fails this routine returns an error.                    */
/*                                                                    */
/**********************************************************************/


USHORT PASCAL eddg_ClipPickWindow(VOID)

{
#define TFUNC "eddg_ClipPickWindow"
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    USHORT             Corrn;
    USHORT             i;
    DevRect            AIPickWindow;
    DevRect            CurrentClip;
    DevRect            CurrentCorr;
    DevRect *          OldCorrRects;


    /******************************************************************/
    /* Convert from Device Space to AI Space                          */
    /******************************************************************/

    AIPickWindow[0].X = pdc->DCIPickWindowDevice[0].X +
                                                   pdc->DCIOrigin.X;
    AIPickWindow[1].X = pdc->DCIPickWindowDevice[1].X +
                                                   pdc->DCIOrigin.X;
    AIPickWindow[0].Y = pdc->DCIConvFactor -
                                      pdc->DCIPickWindowDevice[1].Y;
    AIPickWindow[1].Y = pdc->DCIConvFactor -
                                      pdc->DCIPickWindowDevice[0].Y;


    /******************************************************************/
    /* Check if more memory needs to be allocated. If it does then    */
    /* allocate the same amount as required for the clip rectangles   */
    /******************************************************************/
    if (pdc->DCICorrSize < pdc->DCIEngineClips)
    {
        OldCorrRects = pdc->DCICorrRects;

        pdc->DCICorrRects =
            AllocateMemory( pdc->DCIEngineClips * sizeof(DevRect),
                            MT_CORRRECTS,
                            MO_SHARED );
        if (pdc->DCICorrRects == NULL)
        {
            /**********************************************************/
            /* Allocation has failed.                                 */
            /**********************************************************/
            goto CLIPPICK_ERR_EXIT;
        }
    }

    /******************************************************************/
    /* Calculate the intersection of the pick window with the clip    */
    /* rectangles                                                     */
    /******************************************************************/
    Corrn = 0;

    /******************************************************************/
    /* Ensure that the bounding clip rectangle is the size of the     */
    /* bitmap. This ensures that we get all the clip rectangles       */
    /* through the cache.                                             */
    /******************************************************************/
    GetClipsBounds.X0 = pdc->DCIBoundingClip[0].X;
    GetClipsBounds.X1 = pdc->DCIBoundingClip[1].X + 1;
    GetClipsBounds.Y0 = pdc->DCIBoundingClip[0].Y;
    GetClipsBounds.Y1 = pdc->DCIBoundingClip[1].Y + 1;

    /******************************************************************/
    /* Cycle through all of the current clip rectangles.              */
    /******************************************************************/
    GetClipsControl.ircStart = 1;
    do /* for each rectangle comprising the clip region */
    {
        /**************************************************************/
        /* set the GetClips control structure number of rectangles    */
        /* returned to zero so that we will exit the loop if all      */
        /* rectangles processed                                       */
        /**************************************************************/
        GetClipsControl.crcReturned = 0;

        if ( (pdc->DCIEngineClips > CACHED_CLIPS) ||
                                                   pdc->ClipChanged )
        /**************************************************************/
        /* if the total number of clips comprising the region exceed  */
        /* the size of the dc cache or notify clip change has been    */
        /* called then get the cache updated from the engine          */
        /**************************************************************/
        {
            if ( !edda_GetClipRectangles() )
                goto CLIPPICK_ERR_EXIT;

            /**********************************************************/
            /* reset the start rectangle for next iteration           */
            /**********************************************************/
            GetClipsControl.ircStart += GetClipsControl.crcReturned;
        }

        for (i = 0; i < pdc->DCIClipNum; i++)
        {
            /**********************************************************/
            /* Set up pick window to be modified by clip rectangle    */
            /**********************************************************/
            CurrentCorr[0].X = AIPickWindow[0].X;
            CurrentCorr[0].Y = AIPickWindow[0].Y;
            CurrentCorr[1].X = AIPickWindow[1].X;
            CurrentCorr[1].Y = AIPickWindow[1].Y;

            CurrentClip[0].X = pdc->DCIClipRects[i].X0;
            CurrentClip[0].Y = pdc->DCIClipRects[i].Y0;
            CurrentClip[1].X = pdc->DCIClipRects[i].X1;
            CurrentClip[1].Y = pdc->DCIClipRects[i].Y1;

            /**********************************************************/
            /* Get the intersection                                   */
            /**********************************************************/
            if (INTERSECTION == intersect_rectangles(CurrentClip,
                                                          CurrentCorr) )
            {
                /******************************************************/
                /* keep count of number used and get index to next    */
                /******************************************************/
                pdc->DCICorrRects[Corrn][0].X = CurrentCorr[0].X;
                pdc->DCICorrRects[Corrn][0].Y = CurrentCorr[0].Y;
                pdc->DCICorrRects[Corrn][1].X = CurrentCorr[1].X;
                pdc->DCICorrRects[Corrn][1].Y = CurrentCorr[1].Y;

                Corrn++;
            }
        } /* for each rectangle in the dc cache */
    } while ( GetClipsControl.crcReturned &&
                  (GetClipsControl.ircStart <= pdc->DCIEngineClips));

    /******************************************************************/
    /* If meory was previously allocated and some more was allocated  */
    /* above then release the old                                     */
    /******************************************************************/
    if ((pdc->DCICorrSize) &&
        (pdc->DCICorrSize < pdc->DCIEngineClips) )
    {
        FreeMemory(OldCorrRects);
    }

    /******************************************************************/
    /* Reset Number of correlation rectangles and mark them as not    */
    /* invalid.                                                       */
    /******************************************************************/
    pdc->DCICorrNum     = Corrn;
    pdc->DCICorrSize    = pdc->DCIEngineClips;
    pdc->DCICorrInvalid = FALSE;

CLIPPICK_OK_EXIT:
    return(OK);

CLIPPICK_ERR_EXIT:
    /******************************************************************/
    /* Cannot assign new memory                                       */
    /******************************************************************/
    return(ERROR_ZERO);

}
#undef TFUNC





/**********************************************************************/
/* intersect_rectangles intersects the given rectangles. The          */
/* intersection rectangle is returned in the given target rectangle.  */
/* IntersectRectangles return value indicates whether there was       */
/* an intersection                                                    */
/**********************************************************************/
USHORT PASCAL NEAR intersect_rectangles (pDevPoint SrcRect,
                                         pDevPoint TrgRect)

{
#define TFUNC "intersect_rectangles"

    /******************************************************************/
    /* check for no intersection                                      */
    /******************************************************************/
    if (TrgRect[0].X > SrcRect[1].X ||
        TrgRect[1].X < SrcRect[0].X ||
        TrgRect[0].Y > SrcRect[1].Y ||
        TrgRect[1].Y < SrcRect[0].Y)
        return (NO_INTERSECTION);

    /******************************************************************/
    /* Set bottom left hand corner - highest of x and y values        */
    /******************************************************************/
    if (SrcRect[0].X > TrgRect[0].X)
        TrgRect[0].X = SrcRect[0].X;

    if (SrcRect[0].Y > TrgRect[0].Y)
        TrgRect[0].Y = SrcRect[0].Y;

    /******************************************************************/
    /* Set top right hand corner - lowest of x and y values           */
    /******************************************************************/
    if (SrcRect[1].X < TrgRect[1].X)
        TrgRect[1].X = SrcRect[1].X;

    if (SrcRect[1].Y < TrgRect[1].Y)
        TrgRect[1].Y = SrcRect[1].Y;

    return(INTERSECTION);

} /* intersect_rectangles */
#undef TFUNC
