/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/

/**********************************************************************/
/*                                                                    */
/*   Module          = EDDGCORR                                       */
/*                                                                    */
/*   Description     = Display Device Driver minor function handler   */
/*                     line and rectangle correlation.                */
/*                                                                    */
/*   Function        = Intersects line with Pick Window               */
/*                     Intersects rectangle with Pick Window          */
/*                     Clips and Correlates Complex Line with Pick    */
/*                     Window                                         */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/

#include <eddinclt.h>
#include <eddgextf.h>
#include <eddicone.h>

/**********************************************************************/
/* private functions                                                  */
/**********************************************************************/
VOID PASCAL NEAR adjust_line (pDevPoint ClipRect,
                              pDevPoint Begin,
                              pDevPoint End,
                              USHORT    StartValue);

USHORT PASCAL NEAR get_outcode (DevPoint,
                                pDevPoint);

/**********************************************************************/
/* Note: These routines all return OK for failed correlation, and     */
/*                                 OK_CORR for successful correlation */
/*       Both OK and OK_CORR != FALSE, so the routines sould be used  */
/*        in the form                                                 */
/*              CorrResult = RoutineName;                             */
/*        and not by                                                  */
/*              if (RoutineName)                                      */
/*                 CorrResult = OK_CORR;                              */
/**********************************************************************/

/**********************************************************************/
/* Subroutine CheckLineIntersection                                   */
/* Given the endpoints of a line check whether it intersects with any */
/* of the correlation rectangles. Note that there is at least one     */
/* correlation rectangle or else this routine would not have been     */
/* called.                                                            */
/**********************************************************************/
USHORT PASCAL eddg_CheckLineCorrelation(pDevPoint  Startpt,
                                        pDevPoint  Endpt)

{
#define TFUNC "eddg_CheckLineCorrelation"
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    USHORT    StartValue;
    USHORT    EndValue;
    USHORT    i;


    /******************************************************************/
    /* Check each correlation rectangle for line intersection         */
    /******************************************************************/
    for (i = 0;i < pdc->DCICorrNum; i++)
    {
        /**************************************************************/
        /* return correlated if startpoint is inside corr rectangle   */
        /**************************************************************/
        StartValue = get_outcode(*Startpt, pdc->DCICorrRects[i]);
        if (StartValue == 0)
        {
            return(OK_CORR);
        }

        else
        {
            /**********************************************************/
            /* return correlated if endpoint is inside corr rectangle */
            /**********************************************************/
            EndValue = get_outcode(*Endpt,
                                    pdc->DCICorrRects[i]);
            if (EndValue == 0)
            {
               return(OK_CORR);
            }
            else if ((StartValue & EndValue) != 0)
            {
                /**********************************************************/
                /* return no correlation if line is completlely outside   */
                /* correlation rectangle                                  */
                /**********************************************************/
                return(OK);
            }
            else
            {
                /**********************************************************/
                /* Perform cohen-sutherland clipping on line until a      */
                /* definate answer is obtained                            */
                /**********************************************************/
                return( eddg_ClipAndCorrLine(
                               StartValue,
                               EndValue,
                               *Startpt,
                               *Endpt,
                               (pDevRect)pdc->DCICorrRects[i]));
            }
        }
    }
}
#undef TFUNC

/**********************************************************************/
/* Subroutine CheckRectIntersection                                   */
/* Given a rectangle check whether it intersects with any of the      */
/* correlation rectangles. Note that there is at least one correlation*/
/* rectangle or else this routine would not have been called          */
/* called.                                                       */
/**********************************************************************/
USHORT PASCAL eddg_CheckRectCorrelation(pDevRect Rectangle)

{
#define TFUNC "eddg_CheckRectCorrelation"
    /******************************************************************/
    /* local variables                                                */
    /******************************************************************/
    USHORT   i;
    SHORT    temp;


    /******************************************************************/
    /* ensure target rectangle is well-ordered                        */
    /******************************************************************/
    if ((*Rectangle)[0].X > (*Rectangle)[1].X)
    {
        /**************************************************************/
        /* switch x                                                   */
        /**************************************************************/
        temp = (*Rectangle)[0].X;
        (*Rectangle)[0].X = (*Rectangle)[1].X;
        (*Rectangle)[1].X = temp;
    }

    if ((*Rectangle)[0].Y > (*Rectangle)[1].Y)
    {
        /**************************************************************/
        /* switch y                                                   */
        /**************************************************************/
        temp = (*Rectangle)[0].Y;
        (*Rectangle)[0].Y = (*Rectangle)[1].Y;
        (*Rectangle)[1].Y = temp;
    }


    /******************************************************************/
    /* loop through correlation rectangles stopping when hit. Test is */
    /* if we are not outside of correlation rectangle                */
    /******************************************************************/
    for (i=0; i < pdc->DCICorrNum; i++)
    {
        if  (!(((*Rectangle)[0].X > pdc->DCICorrRects[i][1].X) ||
              ((*Rectangle)[1].X < pdc->DCICorrRects[i][0].X) ||
              ((*Rectangle)[0].Y > pdc->DCICorrRects[i][1].Y) ||
              ((*Rectangle)[1].Y < pdc->DCICorrRects[i][0].Y)))
        {
            return(OK_CORR);
        }

    }
    /******************************************************************/
    /* if we have reached here then we have not correlated            */
    /******************************************************************/
    return(OK);

}
#undef TFUNC

/**********************************************************************/
/* Subroutine ClipAndCorrLine                                         */
/* Having discovered that a particular line does not have its end     */
/* points in a correlation rectangle, nor is it completely outside    */
/* of the correlation rectangle, we must discover whether it crosses  */
/* the correlation rectangle. To do this we step along the line from  */
/* the startpoint, intersecting it with the boundaries until we       */
/* discover that the remaining line intersects the rectangle, or lies */
/* completely outside.                                                */
/**********************************************************************/
USHORT PASCAL eddg_ClipAndCorrLine(USHORT       StartValue,
                                   USHORT       EndValue,
                                   DevPoint     Startpt,
                                   DevPoint     Endpt,
                                   pDevRect     Rectangle)

{
#define TFUNC "eddg_ClipAndCorrLine"

    /******************************************************************/
    /* Adjust Line. this returns a new start point                    */
    /******************************************************************/
    adjust_line ((pDevPoint)Rectangle,
                 &Startpt,
                 &Endpt,
                 StartValue
                 );

    /******************************************************************/
    /* Calculate new start value for start point                      */
    /******************************************************************/
    StartValue = get_outcode(Startpt, (pDevPoint)Rectangle);

    if (StartValue == 0)
    {
        /**************************************************************/
        /* Correlate if start point is now in line                    */
        /**************************************************************/
        return(OK_CORR);
    }
    else
    {
        /**************************************************************/
        /* return no correlation if line is completlely outside       */
        /* correlation rectangle                                      */
        /**************************************************************/
        if ((StartValue & EndValue) != 0)
        {
            return(OK);
        }

        else
        {
            /**************************************************************/
            /* Recursively call ourself to perform cohen-sutherland on    */
            /* line until a definite answer is obtained.                  */
            /**************************************************************/
            return( eddg_ClipAndCorrLine(StartValue,
                                         EndValue,
                                         Startpt,
                                         Endpt,
                                         Rectangle) );
        }
    }
}
#undef TFUNC




/**********************************************************************/
/* Clips Line to one boundary of edge of rectangle                    */
/**********************************************************************/

VOID PASCAL NEAR adjust_line (pDevPoint ClipRect,
                              pDevPoint Begin,
                              pDevPoint End,
                              USHORT    StartValue)

{
#define TFUNC "adjust_line"
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    SHORT    Xmax;
    SHORT    Ymax;

    Xmax = ClipRect[1].X;
    Ymax = ClipRect[1].Y;


    if (StartValue & CS_ABOVE)
    {
        if (Begin->X != End->X)
        {
            Begin->X = (USHORT)(Begin->X +
                               (LONG)((LONG)(End->X - Begin->X) *
                                       (LONG)(Ymax - Begin->Y)) /
                                              (SHORT)(End->Y - Begin->Y));
        }
        Begin->Y = Ymax;
    }
    else if (StartValue & CS_BELOW)
    {
        if (Begin->X != End->X)
        {
            Begin->X = (USHORT)(Begin->X +
                          (LONG)((LONG)(End->X - Begin->X) *
                           (LONG)((SHORT)ClipRect[0].Y - Begin->Y)) /
                                              (SHORT)(End->Y - Begin->Y));
        }
        Begin->Y = ClipRect[0].Y;
    }
    else if (StartValue & CS_RIGHT)
    {
         if (Begin->Y != End->Y)
         {
            Begin->Y = (USHORT)(Begin->Y +
                            (LONG)((LONG)(End->Y - Begin->Y) *
                                    (LONG)(Xmax - Begin->X)) /
                                              (SHORT)(End->X - Begin->X));
         }
         Begin->X = Xmax;
    }
    else if (StartValue & CS_LEFT)
    {
        if (Begin->Y != End->Y)
        {
            Begin->Y = (USHORT)(Begin->Y +
                        (LONG)((LONG)(End->Y - Begin->Y) *
                           (LONG)((SHORT)ClipRect[0].X - Begin->X)) /
                                         (SHORT)((SHORT)End->X - Begin->X));
        }
        Begin->X = ClipRect[0].X;
    }

} /* adjust_line */
#undef TFUNC



USHORT PASCAL NEAR get_outcode (DevPoint  Point,
                                pDevPoint ClipRect)

/**********************************************************************/
/* Calculates a Cohen-Sutherland clipping algorithm outcode. Point is */
/* deemed to be inside the rectangle if it lies on the edge,          */
/**********************************************************************/

{
    USHORT Outcode;          /* the outcode calculated                */

    if (Point.X  < (ClipRect[0].X))
    {
        Outcode = CS_LEFT;
    }

    else if (Point.X  > (int)(ClipRect[1].X))
    {
        Outcode = CS_RIGHT;
    }

    else
    {
        Outcode = 0;
    }



    if (Point.Y  < (int)(ClipRect[0].Y))
    {
        return(Outcode | CS_BELOW);
    }

    else if (Point.Y > (int)(ClipRect[1].Y))
    {
        return(Outcode | CS_ABOVE);
    }

    else
    {
        return(Outcode);
    }

} /* get_outcode */
#undef TFUNC
