/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/

/**********************************************************************/
/*                                                                    */
/*   Module          = EDDCQTAB                                       */
/*                                                                    */
/*   Description     = Display Device Driver minor function handler   */
/*                     QueryColorData, QueryLogColorTable.            */
/*                                                                    */
/*   Function        = Returns information about the current logical  */
/*                     color table.                                   */
/*                     Returns the contents of the current logical    */
/*                     color table                                    */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/


#include <eddinclt.h>
#include <edddtypt.h>

#include <eddccone.h>

#include <eddcextf.h>
#include <eddgextf.h>


DDIENTRY eddc_QueryColorData (HDC       hdc,
                              ULONG     ArgCount,
                              PULONG    ArgArray,
                              PDC       pdcArg,
                              ULONG     FunN)

{
#define TFUNC "eddc_QueryColorData"

    /******************************************************************/
    /*                                                                */
    /* QueryColorData returns information about the current logical   */
    /* color table. The routine loads ArgArray with the following     */
    /* details about the current logical color table.                 */
    /*                                                                */
    /* Format:     LCOLF_DEFAULT, _INDRGB, _RGB, or _PALETTE          */
    /* LowIndex:   Smallest color index loaded (always zero)          */
    /* HighIndex:  Highest color index loaded                         */
    /* Options:    color table (or palette) options                   */
    /*                                                                */
    /* Information is only returned for the number of elements        */
    /* supplied (given in ArgCount). Any extra elements supplied      */
    /* that are not overwritten with results must be zeroed.          */
    /*                                                                */
    /******************************************************************/

    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG        i;                   /* Loop variable                */

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_ENTER(31);

    /******************************************************************/
    /* Presentation Driver Interface says that ArgCount is a LONG so  */
    /* check for a signed value                                       */
    /******************************************************************/
    if (ArgCount & 0x80000000)
    {
        LOGERR(TFUNC, "-ve ArgCount invalid", &ArgCount, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
        goto QRYCOLORDATA_ERR_EXIT;
    }

    /******************************************************************/
    /* Return each item that there is room for.                       */
    /******************************************************************/
    if (ArgCount > 0)
    {
        /**************************************************************/
        /* there is room for the format - this is common for both     */
        /* color tables and palettes                                  */
        /**************************************************************/
        ArgArray[0] = pdc->DCIColFormat;
        if (ArgCount > 1)
        {
            /**********************************************************/
            /* there is room for the Min value                        */
            /* this is stored in the pdc for color tables          */
            /* and is always 0 for palettes                           */
            /**********************************************************/
#ifdef PALETTE_MGR
            if (pdc->DCIColFormat == LCOLF_PALETTE)
            {
                ArgArray[1] = 0;
            }
            else
            {
#endif /* PALETTE_MGR */
                ArgArray[1] = pdc->DCILowIndex;
#ifdef PALETTE_MGR
            }
#endif /* PALETTE_MGR */
            if (ArgCount > 2)
            {
                /******************************************************/
                /* there is room for the Max value                    */
                /* this is stored in the DCIData for color tables     */
                /* and in the palette structure for palettes          */
                /******************************************************/
#ifdef PALETTE_MGR
                if (pdc->DCIColFormat == LCOLF_PALETTE)
                {
                    ArgArray[2] = pdc->Palette->usMax;
                }
                else
                {
#endif /* PALETTE_MGR */
                    ArgArray[2] = pdc->DCIHighIndex;
#ifdef PALETTE_MGR
                }
#endif /* PALETTE_MGR */
#ifdef COLORDATA
                if (ArgCount > 3)
                {
                    /**************************************************/
                    /* there is room for the flags                    */
                    /* for color tables these come from the pdc       */
                    /* for palettes they come from the palette data   */
                    /**************************************************/
#ifdef PALETTE_MGR
                    if (pdc->DCIColFormat == LCOLF_PALETTE)
                    {
                        /**********************************************/
                        /* we may store additional internal flags in  */
                        /* the palette structure so make sure that we */
                        /* mask to return just the user flags         */
                        /**********************************************/
                        ArgArray[3] = pdc->Palette->usFlags &
                                      (LCOL_PURECOLOR |
                                       LCOL_OVERRIDE_DEFAULT_COLORS);
                    }
                    else
                    {
#endif /* PALETTE_MGR */
                        ArgArray[3] = pdc->DCIColStatus;
#ifdef PALETTE_MGR
                    }
#endif /* PALETTE_MGR */

                    /**************************************************/
                    /* Zero any array elements that are not been used */
                    /**************************************************/
                    for (i = 4; i < ArgCount; i++)
                    {
                        ArgArray[i] = 0;
                    }
                }
#else /* COLORDATA */
                /******************************************************/
                /* Zero any array elements that have not been used.   */
                /******************************************************/
                for (i = 3; i < ArgCount; i++)
                {
                    ArgArray[i] = 0;
                }
#endif /* COLORDATA */
            }
        }
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(31);
    return(OK);

QRYCOLORDATA_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(31);

    return(ERROR_ZERO);
}
#undef TFUNC



/**********************************************************************/
/**********************************************************************/



DDIENTRY eddc_QueryLogColorTable (HDC           hdc,
                                  ULONG         ArgOptions,
                                  ULONG         ArgStart,
                                  ULONG         ArgCount,
                                  PULONG        ArgArray,
                                  PDC           pdcArg,
                                  ULONG         FunN)

{
#define TFUNC "eddc_QueryLogColorTable"

    /******************************************************************/
    /*                                                                */
    /* Returns the values stored in the current logical color table.  */
    /* The only valid option is:                                      */
    /*                                                                */
    /* LCOLOPT_INDEX   - The routine must return the index for each   */
    /*                   RGB value.                                   */
    /*                                                                */
    /******************************************************************/

    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG          i;            /* Loop count                        */
    ULONG          Index;        /* Index into ArgArray               */
    ULONG          Result;       /* Value returned                    */


    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_ENTER(32);

#ifdef PALETTE_MGR
    /******************************************************************/
    /* this call cannot be used if there is a palette selected into   */
    /* this DC                                                        */
    /******************************************************************/
    if (pdc->DCIColFormat == LCOLF_PALETTE)
    {
        LOGERR(TFUNC, "palette error", 0L, 0 ,PMERR_PALETTE_SELECTED);
        goto QRYLOGCOLTABLE_ERR_EXIT;
    }
#endif /* PALETTE_MGR */

    /******************************************************************/
    /* Check whether start index is valid                             */
    /******************************************************************/
    if (ArgStart > pdc->DCIHighIndex)
    {
        LOGERR(TFUNC, "Invalid start index", &ArgStart, 1,
                                           PMERR_INV_COLOR_START_INDEX);
        goto QRYLOGCOLTABLE_ERR_EXIT;
    }

    /******************************************************************/
    /* Check whether the count is a reasonable value                  */
    /******************************************************************/
    if (ArgCount & 0xFFFF0000)
    {
        LOGERR(TFUNC, "Invalid count", &ArgCount, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
        goto QRYLOGCOLTABLE_ERR_EXIT;
    }

    /******************************************************************/
    /* Check whether the options are valid. The only acceptable       */
    /* option is LCOLOPT_INDEX                                        */
    /******************************************************************/
    if (ArgOptions & ~(ULONG)LCOLOPT_INDEX)
    {
        LOGERR(TFUNC, "Invalid options", &ArgOptions, 1,
                                               PMERR_INV_COLOR_OPTIONS);
        goto QRYLOGCOLTABLE_ERR_EXIT;
    }

    /******************************************************************/
    /* If current mode is RGB mode then no elements are returned.     */
    /******************************************************************/
    if (pdc->DCIColFormat == LCOLF_RGB)
    {
        Result = QLCT_RGB;
        goto QRYLOGCOLTABLE_OK_EXIT;
    }

    if (ArgOptions & LCOLOPT_INDEX)
    {
        /**************************************************************/
        /* ArgCount must be even, as we are returning pairs of values */
        /**************************************************************/
        ArgCount &= 0xFFFFFFFE;
    }

    /******************************************************************/
    /* Reset index into results array                                 */
    /******************************************************************/
    Index = 0;

    /******************************************************************/
    /* Enter loop which writes values into results array              */
    /******************************************************************/
    for (i = ArgStart;
         (Index < ArgCount) && (i <= pdc->DCIHighIndex);
         i++)
    {
        if (pdc->DCIColorTable[i].PhyIndex != CLR_NOPHYINDEX)
        {
            /**********************************************************/
            /* Entry in logical color table is valid                  */
            /**********************************************************/
            if (ArgOptions & LCOLOPT_INDEX)
            {
                /******************************************************/
                /* Write index                                        */
                /******************************************************/
                ArgArray[Index++] = i;
            }
            /**********************************************************/
            /* Write RGB value                                        */
            /**********************************************************/
            ArgArray[Index++] = URGB(pdc->DCIColorTable[i].LogRGB);
        }
        else
        {
            /**********************************************************/
            /* The entry in the logical color table is invalid. If    */
            /* LCOLOPT_INDEX is set then skip the entry.              */
            /**********************************************************/
            if ( !(ArgOptions & LCOLOPT_INDEX) )
            {
                /******************************************************/
                /* Write a value to indicate that the current RGB     */
                /* value is invalid.                                  */
                /******************************************************/
                ArgArray[Index++] = QLCT_NOTLOADED;
            }
        }
    }

    /******************************************************************/
    /* Return the number of values written to results array           */
    /******************************************************************/
    Result = Index;

QRYLOGCOLTABLE_OK_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(32);
    return(Result);


QRYLOGCOLTABLE_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(32);

    return(QLCT_ERROR);
}
#undef TFUNC
