/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
#define XGADEBUG
/**********************************************************************/
/*                                                                    */
/*   Module          = EDDBCREA                                       */
/*                                                                    */
/*   Description     = Display Device Driver Function :               */
/*                     DeviceCreateBitmap                             */
/*                                                                    */
/*   Function        = Acquires memory for a bitmap and a bitmap      */
/*                     header and returns the address of the header.  */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/* Revisions:                                                         */
/*                                                                    */
/* 76226  12/01/93 JOHNB  Changed check for old or new bitmap         */
/*                        structure for ArgInitInfo to separate       */
/*                        check to initialize the ArgInitInfo2        */
/*                        variable. The ArgBitmapInfo and ArgInitInfo */
/*                        structures could be of different types.     */
/*                                                                    */
/**********************************************************************/

#define INCL_DEV
#define INCL_DDIMISC
#define INCL_DOSMEMMGR
#define INCL_DOSSEMAPHORES

#include <eddinclt.h>

#include <eddbcone.h>
#include <eddhcone.h>

#include <eddbtypt.h>
#include <edddtypt.h>

#include <eddbextf.h>
#include <eddgextf.h>

#include <errors.h>
#include <bitmaps.h>

#include <memman.h>
#include <semaphor.h>


extern DDTType        DDT;




/**********************************************************************/
/*                                                                    */
/*   DeviceCreateBitmap acquires memory for the requested bitmap,     */
/* and returns the bitmap handle (the address of the header).         */
/*                                                                    */
/*   The function does the following:                                 */
/*                                                                    */
/*   - The device driver will create a (1,1) bitmap if this is        */
/*     requested; otherwise the default format will be used. The      */
/*     requested number of planes and bits are not checked against    */
/*     the Winthorn Standard Formats, because we want to be as        */
/*     flexible as possible.                                          */
/*                                                                    */
/*   - Calls AllocMemForBitmap to allocate memory for the bitmap      */
/*     and initialise the bitmap to a background colour.  This also   */
/*     allocates memory for the header and initialises it.            */
/*                                                                    */
/*   - If the Usage parameter is set to Initialise the bitmap is      */
/*     initialised to the given data.                                 */
/*                                                                    */
/**********************************************************************/

DDIENTRY eddb_DeviceCreateBitmap (HDC                   hdc,
                                  PBITMAPINFOHEADER     ArgBitmapInfo,
                                  ULONG                 ArgUsage,
                                  PBYTE                 ArgInitBits,
                                  PBITMAPINFO           ArgInitInfo,
                                  PDC                   pdcArg,
                                  ULONG                 FunN)

{
#define TFUNC "eddb_DevCreBmap"

    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    pBitmapHeader    InternalHeader;  /* pointer to internal header   */
    POINTL           ptlStart;        /* Start position.              */
    USHORT           usScanLinesSet;  /* used by conversion function  */
    USHORT           BitCount;        /* bits per pixel in new bm     */
    PBITMAPINFO2     ArgInitInfo2;
    PBITMAPINFO2     ArgBitmapInfo2;
    BOOL             fUncompressed;

    /******************************************************************/
    /* Get exclusive access to the driver                             */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* assume no conversion required                                  */
    /******************************************************************/
    fUncompressed = FALSE;

    /******************************************************************/
    /* validate the BITMAPINFOHEADER parameter                        */
    /* - width must be greater than 0                                 */
    /* - height must be greater than 0                                */
    /******************************************************************/
    if (ArgBitmapInfo->cbFix == sizeof(BITMAPINFOHEADER))
    {
        /**************************************************************/
        /* old format - easy to verify                                */
        /**************************************************************/
        if ( (ArgBitmapInfo->cx == 0) ||
             (ArgBitmapInfo->cy == 0) )
        {
            LOGERR(TFUNC, "Null bitmap", FNULL, 0,
                                            PMERR_INV_BITMAP_DIMENSION);
            goto CREATEBITMAP_ERR_EXIT;
        }
    }
    else
    {
        /**************************************************************/
        /* new format, so set up our copies of the pointers           */
        /**************************************************************/
        ArgBitmapInfo2 = (PBITMAPINFO2)ArgBitmapInfo;

        if ( (ArgBitmapInfo2->cx == 0) ||
             (ArgBitmapInfo2->cy == 0) )
        {
            LOGERR(TFUNC, "Null bitmap", FNULL, 0,
                                            PMERR_INV_BITMAP_DIMENSION);
            goto CREATEBITMAP_ERR_EXIT;
        }
    }

    /******************************************************************/
    /* If bitmap is to be initialised then check the format           */
    /* parameters                                                     */
    /******************************************************************/
    if (ArgUsage & CBM_INIT)
    {
        /**************************************************************/
        /* @76226 Check the ArgInitInfo header seperately, it can     */
        /*        be different                                        */
        /**************************************************************/
        if (ArgInitInfo->cbFix > sizeof(BITMAPINFOHEADER))
            ArgInitInfo2   = (PBITMAPINFO2)ArgInitInfo;
    
        /**************************************************************/
        /* validate the complex header structure                      */
        /**************************************************************/
        if (!ValidHeader((PBITMAPINFO2)ArgInitInfo))
        {
            /**********************************************************/
            /* Validation failed.                                     */
            /* Log the error that was noted.                          */
            /**********************************************************/
            CheckForErrors();
            goto CREATEBITMAP_ERR_EXIT;
        }

        /**************************************************************/
        /* the header is valid                                        */
        /* now it may be compressed                                   */
        /* we call a routine which on return ensures that our         */
        /* pointers point at an uncompressed external bitmap          */
        /**************************************************************/
        if (!uncompress_bitmap((PBITMAPINFO2 FAR *)&ArgInitInfo,
                               (PBYTE FAR *)&ArgInitBits,
                               &fUncompressed))
        {
            /**********************************************************/
            /* convert failed (probably due to lack of memory)        */
            /**********************************************************/
            goto CREATEBITMAP_ERR_EXIT;
        }
    }

    /******************************************************************/
    /* Calculate format bitmap will be created in. This is            */
    /*   - (1,1) if requested                                         */
    /*   - default (colour) format otherwise.                         */
    /* Note the device driver only works with no of Planes = 1        */
    /******************************************************************/
    BitCount = DDT.BitCount;            /* assume colour format       */

    if (ArgBitmapInfo->cbFix == sizeof(BITMAPINFOHEADER))
    {
        /**************************************************************/
        /* old format                                                 */
        /**************************************************************/
        if ((ArgBitmapInfo->cBitCount == 1) &&
            (ArgBitmapInfo->cPlanes   == 1) )
        {
            BitCount = 1;
        }
    }
    else
    {
        /**************************************************************/
        /* new format                                                 */
        /**************************************************************/
        if ((ArgBitmapInfo2->cBitCount == 1) &&
            (ArgBitmapInfo2->cPlanes   == 1) )
        {
            BitCount = 1;
        }
    }

    /******************************************************************/
    /* Allocate memory for the bitmap header                          */
    /******************************************************************/
    InternalHeader = eddb_AllocBitmapHeader();
    if (InternalHeader == NULL)
    {
        goto CREATEBITMAP_ERR_EXIT;
    }


    /******************************************************************/
    /* initialise the bitmap header fields needed for                 */
    /* AllocMemForBitmap                                              */
    /******************************************************************/
    if (ArgBitmapInfo->cbFix == sizeof(BITMAPINFOHEADER))
    {
        /**************************************************************/
        /* old format                                                 */
        /**************************************************************/
        InternalHeader->Info.Width  = ArgBitmapInfo->cx;
        InternalHeader->Info.Height = ArgBitmapInfo->cy;
    }
    else
    {
        /**************************************************************/
        /* new format                                                 */
        /**************************************************************/
        InternalHeader->Info.Width  = (USHORT)ArgBitmapInfo2->cx;
        InternalHeader->Info.Height = (USHORT)ArgBitmapInfo2->cy;
    }
    InternalHeader->Info.BitCount = BitCount;


    /******************************************************************/
    /* allocate memory for the bitmap                                 */
    /******************************************************************/
    if ( OK != eddb_AllocMemForBitmap(InternalHeader) )
    {
        goto CREATEBITMAP_ERR_EXIT;
    }

    /******************************************************************/
    /* Initialise bitmap if required                                  */
    /******************************************************************/
    if (ArgUsage & CBM_INIT)
    {
        /**************************************************************/
        /* Set up the number of rows to be converted.                 */
        /**************************************************************/
        if (ArgInitInfo->cbFix == sizeof(BITMAPINFOHEADER))
        {
            /**********************************************************/
            /* old format                                             */
            /**********************************************************/
            usScanLinesSet = min(InternalHeader->Info.Height,
                                 ArgInitInfo->cy);
        }
        else
        {
            /**********************************************************/
            /* new format                                             */
            /**********************************************************/
            usScanLinesSet = min(InternalHeader->Info.Height,
                                 (USHORT)ArgInitInfo2->cy);
        }
        ptlStart.x = 0;
        ptlStart.y = 0;

        /**************************************************************/
        /* Transfer bits from application storage to the bitmap       */
        /* using a conversion function                                */
        /**************************************************************/
        ConvertExtToInt( ArgInitBits,
                         (PBITMAPINFO2)ArgInitInfo,
                         InternalHeader,
                         &ptlStart,
                         usScanLinesSet);
    }


    if (fUncompressed)
    {
        FreeMemory(ArgInitInfo);
    }
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Return the bitmap header address as our bitmap handle.         */
    /******************************************************************/
    PROFILE_EXIT(9);
    return((ULONG)(PVOID)InternalHeader);


CREATEBITMAP_ERR_EXIT:
    if (fUncompressed)
    {
        FreeMemory(ArgInitInfo);
    }
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_EXIT(9);
    return(ERROR_ZERO);

}
#undef TFUNC

/**********************************************************************/
/* The 32 bit, 8 parameter entry into the graphics engine. Prototype  */
/* here so that the compiler can check the types.                     */
/**********************************************************************/
ULONG APIENTRY InnerGre32Entry9(HDC,ULONG,ULONG,PBITMAPINFO2,PBYTE,
                                PBITMAPINFO2,PBYTE,ULONG,ULONG);

/**********************************************************************/
/* This routine uncompresses bitmap data if needed                    */
/*                                                                    */
/* Parameters are:                                                    */
/*     ArgInfo          info for the supplied/returned bitmap         */
/*     ArgBits          bits of the supplied/returned bitmap          */
/*     options          options to pass to ConvertBitmap routine      */
/*     converted        returns whether bitmap was converted or not   */
/*                                                                    */
/* Returns TRUE if no error, FALSE if an error was logged             */
/**********************************************************************/

BOOL uncompress_bitmap(PBITMAPINFO2 * ArgInfo2,
                       PBYTE        * ArgBits,
                       PBOOL          converted)

{
    PBITMAPINFO2        ConvertInfo;
    PBYTE               ConvertBits;
    ULONG               decompsize;
    USHORT              headersize;

    /******************************************************************/
    /* assume no conversion is done                                   */
    /******************************************************************/
    *converted = FALSE;

    /******************************************************************/
    /* see if the passed bitmap is compressed                         */
    /* only do this if the header is big enough to include the        */
    /* compression field                                              */
    /******************************************************************/
    if (((*ArgInfo2)->cbFix >= SIZE_BM2_TO_COMPRESS(*ArgInfo2)) &&
        ((*ArgInfo2)->ulCompression != BCA_UNCOMP))
    {
        /**********************************************************/
        /* size is number of scanlines * bytes per line           */
        /* remembering that all scanlines are dword padded        */
        /**********************************************************/
        decompsize = (*ArgInfo2)->cy *
             (((((*ArgInfo2)->cx * (*ArgInfo2)->cBitCount) + 31L) / 32L) * 4L);

        headersize = SIZE_BM2_TO_IMAGE(*ArgInfo2);

        if ((*ArgInfo2)->cBitCount != 24)
        {
            /**************************************************/
            /* increase the header size to allow for the      */
            /* colour table                                   */
            /* each colour table entry is 4 bytes             */
            /**************************************************/
            headersize += (4 << (*ArgInfo2)->cBitCount);
        }

        ConvertInfo = AllocateMemory( decompsize + headersize,
                                      MT_BITMAP,
                                      MO_SHARED );
        if (ConvertInfo == NULL)
        {
            /**************************************************/
            /* Allocation failed.                             */
            /**************************************************/
            return FALSE;
        }

        /******************************************************/
        /* set the return pointers                            */
        /******************************************************/
        ConvertBits = ((PBYTE)ConvertInfo) + headersize;

        ConvertInfo->cbFix         = SIZE_BM2_TO_IMAGE(*ArgInfo2);
        ConvertInfo->cx            = (*ArgInfo2)->cx;
        ConvertInfo->cy            = (*ArgInfo2)->cy;
        ConvertInfo->cPlanes       = (*ArgInfo2)->cPlanes;
        ConvertInfo->cBitCount     = (*ArgInfo2)->cBitCount;
        ConvertInfo->ulCompression = BCA_UNCOMP;
        ConvertInfo->cbImage       = decompsize;

        /******************************************************/
        /* got the memory now make the call                   */
        /******************************************************/
        if (OK != InnerGre32Entry9((HDC)pdc->DCIhdc,
                                   CBD_DECOMPRESSION,
                                   0L,
                                   (PBITMAPINFO2)*ArgInfo2,
                                   *ArgBits,
                                   ConvertInfo,
                                   ConvertBits,
                                   0L,
                                   NGreConvertBitmapData))
        {
            /**************************************************/
            /* convert failed                                 */
            /* free the memory and return                     */
            /**************************************************/
            FreeMemory(ConvertInfo);
            return FALSE;
        }
        /**************************************************************/
        /* adjust the input pointers to point at the converted bitmap */
        /**************************************************************/
        *ArgInfo2 = (PBITMAPINFO2)ConvertInfo;
        *ArgBits  = ConvertBits;

        /**************************************************************/
        /* we have a converted bitmap                                 */
        /**************************************************************/
        *converted = TRUE;

    }
    /******************************************************************/
    /* no errors                                                      */
    /******************************************************************/
    return TRUE;
}
