/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = VVVGA.C
 *
 * DESCRIPTIVE NAME = Virtual Video VGA Specific routines
 *
 *
 * VERSION = V2.0
 *
 * DATE      11/10/88
 *
 * DESCRIPTION
 *              This module contains the VVD's VGA specific routines and data.
 *
 *
 * FUNCTIONS
 *              vvVGADisableVRAM()      Disable VRAM thru Misc. Output
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/


#define  IO8BIT          /* CL386 Version 6.00.054 FLAG:               */
#include <mvdm.h>
#include <vvd.h>
#include "vvdp.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif

#pragma  BEGIN_SWAP_DATA

extern IOH aiohCGAUnknown [2];                                  /*          */

extern PLE pleMiscOutWrite;                                     /*          */
extern PLE pleFeatureWrite;                                     /*          */
extern PLE pleATC0;
extern PLE pleATCx;                                             /*          */
extern PLE pleATC1;

#ifdef SVGA                                                     /*          */
extern PLE pleATIVGAIndx;                                       /*          */
#endif /* SVGA */                                               /*          */

#ifdef SVGA
extern SA8514INFO sA8514Info;
#endif /* SVGA */

BOOL flXGAPresent = FALSE;                                      /*          */

 /*  Adapter-specific data
 **
 **  For the Genoa SuperVGA (model 5100), their ROM relies on obtaining
 **  the true setting of an undocumented register (Sequencer register #7)
 **  in order to initialize properly as a VGA.  We don't have to do anything
 **  special when the VDM is initializing in full-screen mode, because the
 **  I/O access goes to the real hardware;  but when a windowed VDM is
 **  created, we don't know how to virtualize that register, so the ROM
 **  mistakenly initializes the card as an EGA, and any attempt to make the
 **  VDM full-screen later really screws the display up (even when you try to
 **  switch back to PM).  The simplest workaround is to record the values of
 **  all the registers during initialization, and stuff them into the per-VDM
 **  data every time we create a new VDM.
 */

BYTE aregSEQInit [MAX_SEQREGS];
BYTE aregGDCInit [MAX_GDCREGS];                                 /*          */
BYTE aregCRTInit [MAX_CRTREGS];                                 /*          */

IOH aiohVSE [2] =                       /* 03c3 */              /*          */
{                                                               /*          */
  {                                                             /*          */
    NULL,                                                       /*          */
    &VVWriteVSEFgnd,                                            /*          */
  },                                                            /*          */
  {                                                             /*          */
    &VVReadVSEBgnd,                                             /*          */
    &VVWriteVSEBgnd,                                            /*          */
  },                                                            /*          */
};                                                              /*          */

IOH aiohVGADACMask [2] =                /* 03c6 */
{
  {
    NULL,
  },
  {
    &VVReadDACMaskBgnd,
    &VVWriteDACMaskBgnd,
  },
};

IOH aiohVGADACRead [2] =                /* 03c7 */
{
  {
    NULL,
    &VVWriteDACReadFgnd,
  },
  {
    &VVReadDACStateBgnd,
    &VVWriteDACReadBgnd,
  },
};

IOH aiohVGADACWrite [2] =               /* 03c8 */
{
  {
    NULL,
    &VVWriteDACWriteFgnd,
  },
  {
    &VVReadDACIndxBgnd,
    &VVWriteDACWriteBgnd,
  },
};

IOH aiohVGADACData [2] =                /* 03c9 */
{
  {
    &VVReadDACDataFgnd,
    &VVWriteDACDataFgnd,
  },
  {
    &VVReadDACDataBgnd,
    &VVWriteDACDataBgnd,
  },
};

IOH aiohVVReserved [2] =                /* 03d6 */
{
  {
    &VVReadVVReservedFgnd,                                      /*          */
    &VVWriteVVReservedFgnd,                                     /*          */
  },
  {
    &VVReadVVReservedBgnd,                                      /*          */
    &VVWriteVVReservedBgnd,                                     /*          */
  },
};

#ifdef SVGA                                                     /*            */
IOH aiohVSEAlternate [2] =              /* 46e8 */              /*          */
{
  {
    NULL,
    &VVWriteVSEAlternateFgnd,
  },
  {
    &VVReadVSEAlternateBgnd,
    &VVWriteVSEAlternateBgnd,
  },
};
#endif                                                          /*            */

/* Although some VGAs can read this port, some cannot. */       /*          */
/* So this port is best treated as write only internally. */    /*          */
PLE pleVSE =                                                    /*          */
{                                                               /*          */
  {                                                             /*          */
    PORT_VGAENABLE,                     /* 03c3 */              /*          */
    &VDMData.regVSE,                                            /*          */
    &VDMData.regVSE,                                            /*          */
    WRITE_MANY | READ_NONE | R_NE_W,                            /*          */
    aiohVSE,                                                    /*          */
  },                                                            /*          */
  0,                                                            /*          */
  NULL,                                                         /*          */
};                                                              /*          */

PLE pleVGADACMask =                                             /*          */
{
  {                                                             /*          */
    PORT_VGADACMASK,                    /* 3C6h (R/W) */
    &VDMData.regDACMask,
    &VDMData.regDACMask,
    WRITE_MANY | READ_MANY | R_EQ_W,                            /*          */
    aiohVGADACMask,                                             /*          */
  },                                                            /*          */
  0,                                    /* non-Index register */
  NULL,
};

PLE pleVGADACRead =                                             /*          */
{
  {                                                             /*          */
    PORT_VGADACREAD,                    /* 3C7h (W/O) */
    &VDMData.regDACIndx,
    &VDMData.regDACIndx,
    WRITE_MANY | READ_NONE | R_NE_W,                            /*          */
    aiohVGADACRead,                                             /*          */
  },                                                            /*          */
  0,                                    /* non-Index register */
  NULL,
};

PLE pleVGADACWrite =                                            /*          */
{
  {                                                             /*          */
    PORT_VGADACWRITE,                   /* 3C8h (R/W) */
    &VDMData.regDACIndx,                                        /*          */
    &VDMData.regDACIndx,                                        /*          */
    WRITE_MANY | READ_MANY | R_EQ_W,                            /*          */
    aiohVGADACWrite,                                            /*          */
  },                                                            /*          */
  0,                                                            /*          */
  NULL,                                                         /*          */
};

PLE pleVGADACAuto =                                             /*          */
{
  {                                                             /*          */
    PORT_VGADACDATA,                    /* 3C9h (R/W) */
    &VDMData.adacDACData[0].dac_bRed,
    &VDMData.adacDACData[0].dac_bRed,
    PORTF_AUTOINC                       /* auto-increment register */
      | WRITE_INDX | READ_INDX | R_EQ_W,                        /*          */
    aiohVGADACData,                                             /*          */
  },                                                            /*          */
  TOTAL_DACREGS*sizeof(DACREG),         /* 256 (R,G,B) values */
  &pleVGADACWrite,                                              /*          */
};

PLE pleVGADACDatax =                                            /*          */
{
  {                                                             /*          */
    PORT_VGADACDATA,                    /* 3c9h (R/W) */
    &VDMData.stateDAC,
    &VDMData.stateDAC,
    WRITE_MANY | READ_MANY | R_EQ_W,                            /*          */
    aiohVGADACData,                                             /*          */
  },                                                            /*          */
  0,                                    /* non-Index register */
  NULL,
};

PLE pleVGADACData =                                             /*          */
{
  {                                                             /*          */
    PORT_VGADACDATA,                    /* 3C9h (R/W) */
    &VDMData.dacDACDataCur,
    &VDMData.dacDACDataCur,
    WRITE_INDX | READ_INDX | R_EQ_W,                            /*          */
    aiohVGADACData,                                             /*          */
  },                                                            /*          */
  DAC_DATA2 + 1,                        /* non-Index register */
  &pleVGADACDatax,
};

PLE pleVVReserved =                                             /*          */
{                                                               /*          */
  {                                                             /*          */
    0x03d6,                             /* 03d6 (R/W) */        /*          */
    NULL,                                                       /*          */
    NULL,                                                       /*          */
    WRITE_MANY | READ_MANY | R_NE_W,                            /*          */
    aiohVVReserved,                                             /*          */
  },                                                            /*          */
  0,                                    /* non-Index register *//*          */
  NULL,                                                         /*          */
};                                                              /*          */

/* FLAG: 26-Jul-90               */
/* More complete virtualization? */
PLE pleIBMP70Indx =                                             /*          */
{                                                               /*          */
  {                                                             /*          */
    PORT_IBMP70_INDX,                   /* 0d00 (R/W) */        /*          */
    NULL,                                                       /*          */
    NULL,                                                       /*          */
    WRITE_MANY | READ_MANY | R_EQ_W,                            /*          */
    aiohCGAUnknown,                                             /*          */
  },                                                            /*          */
  0,                                    /* non-Index register *//*          */
  NULL,                                                         /*          */
};

/* FLAG: 26-Jul-90               */
/* More complete virtualization? */
PLE pleIBMP70Data =                                             /*          */
{                                                               /*          */
  {                                                             /*          */
    PORT_IBMP70_DATA,                   /* 0d01 (R/W) */        /*          */
    NULL,                                                       /*          */
    NULL,                                                       /*          */
    WRITE_MANY | READ_MANY | R_EQ_W,                            /*          */
    aiohCGAUnknown,                                             /*          */
  },                                                            /*          */
  0,                                    /* non-Index register *//*          */
  NULL,                                                         /*          */
};

#ifdef SVGA                                                     /*            */
/* Although some VGAs can read this port, some cannot. */       /*          */
/* So this port is best treated as write only internally. */    /*          */
PLE pleVSEAlternate =
{
  {
    A8514_ROMPAGESELECT,                /* 46e8 */
    &VDMData.regVSEAlternate,
    &VDMData.regVSEAlternate,
    WRITE_MANY | READ_NONE | R_NE_W,
    aiohVSEAlternate,
  },
  0,
  NULL,
};
#endif                                                          /*            */

CHAR pszVGAAdapterName [] = "VGA";                              /*          */

PCHAR ppszVGAChipNames [] =                                     /*          */
{  /* As produced by SVGA.EXE! */                               /*          */
  pszVGAAdapterName,                                            /*          */
};                                                              /*          */

SADAPTERINFO sVGAAdapterInfo;           /* Forward */           /*          */
PSADAPTERINFO psCurAdapterInfo = &sVGAAdapterInfo;              /*          */
PSDACINFO psCurDacInfo;                 /* Forward */           /*          */

#pragma  END_SWAP_DATA

#pragma  BEGIN_SWAP_CODE
                                                                /*          */
/***********************************************************************
 *
 * FUNCTION NAME = vvVGAEnableBankAddr()
 *
 * DESCRIPTION   = Enable Bank addressing. Linear addressing should be
 *                 enabled as well as default aperture at A0000.
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvSetFgnd
 *      vvSetBgnd
 *
 ***********************************************************************/

VOID    PRIVENTRY vvVGAEnableBankAddr(
  HVDM hvdm )
{
  USHORT i;

  /*
  ** Clear Aperture Control setting for all XGAs
  */
  if( flXGAPresent )                              /*          *//*          */
    for( i = 0;
         i < 8;
         i++ )
      OUTB( 0x2100 | (i << 4) | AXGA_APERTURE_CONTROL,
            0 );
}

/***************************************************************************
 *
 * FUNCTION NAME = vvVGADisableVRAM(hvdm)
 *
 * DESCRIPTION   = Disable VRAM
 *
 *                 Disable address decode for the display buffer from
 *                 the system.  This is necessary for situations where
 *                 a Seamless application is running on the XGA desktop
 *                 interfering with the secondary VGA screen because
 *                 the Seamless XGA         driver is using the 64K
 *                 aperture to update the application window.
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvDisableVRAM(register HVDM hvdm)                /*          */
{                                                               /*          */
  register PVDMDATA pvd = pVDMData(hvdm);                       /*          */
                                                                /*          */
  if ( flXGAPresent                                             /*          */
       && (pvd->flVDMVideo & VDM_FGND) )                        /*          */
    vvOutput( INVALID_HVDM,                                     /*          */
              &pleMiscOutWrite,                                 /*          */
              (BYTE) (pvd->regMiscOut & ~MISCOUT_ENABLERAM) );  /*          */
}                                                               /*          */

/***************************************************************************
 *
 * FUNCTION NAME = vvSaveATCState()                                       
 *
 * DESCRIPTION   = Update virtual ATC register state from hardware
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        =
 *                 None
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * PSEUDO-CODE
 *      Find out how many ATCs in use.
 *      Save ATC index
 *      Save all ATC registers
 *      Re-enable display
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGAATCSaveState(
  HVDM hvdm )
{
  INT i;
  BYTE bSaveIndex;
  register PVDMDATA pvd = pVDMData(hvdm);

  if((pvd->flVDMXVideo & VDMX_NOIOTRAPPING) &&                  /*            */
     (pvd->flVDMVideo & VDM_IOINIT))
  {
  /* set flip-flop */
  vvInput( hvdm,                                              /*          */
           &pleFeatureWrite );        /* =Status1 read */     /*          */
  bSaveIndex = vvInput( hvdm,                                 /*          */
                        &pleATC0 );   /* save index    */     /*          */
  for( i = 0;
       i < pleATCx.ulBRegCount;                               /*          */
       i++ )
  {
#ifdef SVGA
//  if (pleATCx.pbBRegReadMask[i >> 3] & (1 << (i & 7)))        //          
    if( pleATCx.pbBRegReadMask [i] )                            /*@V3.0YEE01*/
    {
#endif
        vvInput( hvdm,                                            /*          */
                &pleFeatureWrite );      /* =Status1 read */     /*          */
        vvOutput( hvdm,                                           /*          */
                &pleATC0,                                       /*          */
                (BYTE) i );
        vvInput( hvdm,                                            /*          */
                &pleATC1 );                                      /*          */
#ifdef SVGA
    }
#endif
  }
  vvInput( hvdm,                                              /*          */
           &pleFeatureWrite );        /* =Status1 read */     /*          */
  vvOutput( hvdm,                                             /*          */
            &pleATC0,                                         /*          */
            bSaveIndex );                                     /*          */
  }

}

/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacStdSave()                                        
 *
 * DESCRIPTION   = Save DAC state from hardware
 *
 *                 Save the DAC's internal state.  Specifically, this is:
 *                     - Read/Write mode
 *                     - 0 (normal), 1 or 2 bytes read/written
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacStdSave(                                 /*          */
  HVDM hvdm )
{
  INT n;
  BYTE bIndx;                                                   /*          */
  register INT i;
  register PVDMDATA pvd = pVDMData(hvdm);
  register PBYTE pbDACData;

  if (pvd->flVDMXVideo & VDMX_NOIOTRAPPING)
  {
    pvd->regDACIndx =
      vvInput( INVALID_HVDM,                                    /*          */
               &pleVGADACWrite );                               /*          */
    pvd->stateDAC =
      (vvInput( INVALID_HVDM,                                   /*          */
                &pleVGADACRead )                                /*          */
       ? DAC_READ
       : DAC_WRITE);
    n = DAC_DATA0;                      /* now read until index changes */
    pvd->dacDACDataCur.dac_bRed =
      vvInput( INVALID_HVDM,                                    /*          */
               &pleVGADACData );                                /*          */
    if( vvInput( INVALID_HVDM,                                  /*          */
                 &pleVGADACWrite ) == pvd->regDACIndx )         /*          */
    {
      n++;
      pvd->dacDACDataCur.dac_bGreen =
        vvInput( INVALID_HVDM,                                  /*          */
                 &pleVGADACData );                              /*          */
      if( vvInput( INVALID_HVDM,                                /*          */
                   &pleVGADACWrite ) == pvd->regDACIndx )       /*          */
        n++;
    }
    pvd->stateDAC |= DAC_DATA2 - n;     /* update state with rgb index    */
    vvOutput( INVALID_HVDM,                                     /*          */
              &pleVGADACRead,                                   /*          */
              0 );                      /* prepare to read out entire DAC */
    pbDACData = (PBYTE)pvd->adacDACData;
    for( i = 0;
         i < MAX_DACREGS*3;
         i++ )
      *pbDACData++ =
        vvInput( INVALID_HVDM,                                  /*          */
                 &pleVGADACData );                              /*          */
  }
}
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacStdRestore()
 *
 * DESCRIPTION   = Restore DAC state to hardware
 *
 *                 Restore the DAC's internal state.  Specifically, this is:
 *                     - Read/Write mode
 *                     - 0 (normal), 1 or 2 bytes read/written
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacStdRestore(                              /*          */
  register HVDM hvdm )
{
  BYTE bIndx;                                                   /*          */
  register INT i,n;
  register PVDMDATA pvd = pVDMData(hvdm);

  i = pvd->regDACIndx;
  n = pvd->stateDAC&DAC_DATAMASK;

  if ((pvd->stateDAC&DAC_RWMASK) == DAC_READ)
  {
    /*
    ** Left in READ state, so write to the READ port,
    ** and do the appropriate number of IN instructions
    */
    vvOutput( INVALID_HVDM,                                     /*          */
              &pleVGADACRead,                                   /*          */
              (BYTE) i );
    if( n > DAC_DATA0 )
      vvInput( INVALID_HVDM,                                    /*          */
               &pleVGADACData );                                /*          */
    if( n > DAC_DATA1 )
      vvInput( INVALID_HVDM,                                    /*          */
               &pleVGADACData );                                /*          */
  }
  else
  {
    /*
    ** Left in WRITE state, so write to the WRITE port,
    ** and do the appropriate number of OUT instructions
    */
    vvOutput( INVALID_HVDM,                                     /*          */
              &pleVGADACWrite,                                  /*          */
              (BYTE) i );
    i &= MAX_DACREGS-1;                 /* insurance! (never assume the VDM  */
                                        /* always uses index values in-range)*/
    if( n > DAC_DATA0 )
      vvOutput( INVALID_HVDM,                                   /*          */
                &pleVGADACData,                                 /*          */
                pvd->dacDACDataCur.dac_bRed );
    if( n > DAC_DATA1 )
      vvOutput( INVALID_HVDM,                                   /*          */
                &pleVGADACData,                                 /*          */
                pvd->dacDACDataCur.dac_bGreen );
  }
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAVertDspEnd
 *
 * DESCRIPTION   = Get Vertical Display Length
 *
 * INPUT         = hvdm
 *
 * OUTPUT        = Vertical Display Width
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT PRIVENTRY vvVGAVertDspEnd(
  HVDM hvdm )
{
  register PVDMDATA pvd = pVDMData(hvdm);

  return( vvEGAVertDspEnd( hvdm )
          + ((pvd->aregCRTData [REG_CRTOVERFLOW] & CRTOVFL_VERTDSPEND2)
             >> ZEROBITS( CRTOVFL_VERTDSPEND2 ) << 9) );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGABitsPerPixel
 *
 * DESCRIPTION   = Get Bits Per Pixel
 *
 * INPUT         = hvdm
 *
 * OUTPUT        = Bits Per Pixel
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT PRIVENTRY vvVGABitsPerPixel(
  HVDM hvdm )
{
  register PVDMDATA pvd = pVDMData(hvdm);

  return( (((PSEGAINFO) psCurAdapterInfo)->                     /*          */
             pbfnEGAPackedMode( hvdm )                          /*          */
           ? 8
           : vvEGABitsPerPixel( hvdm )) );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAPrepareSetMode()
 *
 * DESCRIPTION   = Fix Registers & return mode number
 *
 * INPUT         = HVDM
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvInt10SetMode
 *
 **************************************************************************/

USHORT PRIVENTRY vvVGAPrepareSetMode(
  HVDM hvdm,
  PCRF pcrf )
{
  USHORT usMode = (AX( pcrf ) == 0x4F02)        /* VESA mode detect */  /*          */
          ? BX( pcrf )                                          /*          */
          : vvCGAPrepareSetMode( hvdm, pcrf );
  /*
  ** V2.2SEN08 the attempt to freeze the 2f in the background here was
  ** a disaster since vdh is not going to schedule our foreground context
  ** while VDM is still executing an INT10 context hook. This may also
  ** be happening when we freeze/suspend the vdm in the mode set return hook,
  ** but the consequences are smaller, since we atleast have a state to restore
  ** and the setmode would not be executing on top of the shadow (IO hooks for
  ** the background are still in since the foreground context didn't execute).
  */
  return( usMode );                /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAVertLineCmp
 *
 * DESCRIPTION   = Get Vertical Display Compare
 *
 * INPUT         = hvdm
 *
 * OUTPUT        = Vertical Line Compare (for split screen)
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT PRIVENTRY vvVGAVertLineCmp(
  HVDM hvdm )
{
  register PVDMDATA pvd = pVDMData(hvdm);

  return( vvEGAVertLineCmp( hvdm )
          + ((pvd->aregCRTData [REG_CRTMAXSCAN] & CRTMAXSCAN_LINECMP)
             >> ZEROBITS(CRTMAXSCAN_LINECMP) << 9) );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacStdIPFClear()
 *
 * DESCRIPTION   = Clear DAC IPF (Internal Programming Flag)
 *
 * INPUT         = NONE
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacStdIPFClear(                             /*          */
  HVDM hvdm )
{
  hvdm;
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacStdIPFSet()
 *
 * DESCRIPTION   = Set DAC IPF (Internal Programming Flag)
 *
 * INPUT         = NONE
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacStdIPFSet(                               /*          */
  HVDM hvdm )
{
  hvdm;
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacSave()
 *
 * DESCRIPTION   = Save DAC state
 *
 * INPUT         = NONE
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacSave(                                    /*          */
  HVDM hvdm )
{
  /* Add save and restore RS[3:2] if necessary */
  psCurDacInfo->pfnDacSave( hvdm );                             /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacRestore()
 *
 * DESCRIPTION   = Restore DAC state
 *
 * INPUT         = NONE
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacRestore(                                 /*          */
  HVDM hvdm )
{
  /* Add save and restore RS[3:2] if necessary */
  psCurDacInfo->pfnDacRestore( hvdm );                          /*          */
}

#ifdef SVGA                                                     /*          */
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAEditTables()
 *
 * DESCRIPTION   = Called once to do SVGA-specific initialisation
 *
 * INPUT         =
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 **************************************************************************/

VOID PRIVENTRY vvVGAEditTables( VOID )
{
  /* fSVGASeqFixups = TRUE; */ /* Default! */                   /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAInterlaced
 *
 * DESCRIPTION   = Get Interlaced Status
 *
 * INPUT         = hvdm
 *
 * OUTPUT        = True = interlacing on
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL PRIVENTRY vvVGAInterlaced(
  HVDM hvdm )
{
  return( FALSE );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAUpdateIoState()
 *
 * DESCRIPTION   = Update SVGA I/O state
 *
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGAUpdateIoState(
  HVDM hvdm )
{
  hvdm;
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGARestoreIoState()
 *
 * DESCRIPTION   = Do adapter-specific register restores that can't be
 *                 handled in the usual way.
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGARestoreIoState(
  HVDM hvdm )
{
  vvAcceleratorRestoreIOState( hvdm );                          /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGASetBank()
 *
 * DESCRIPTION   = Set bank register
 *
 * INPUT         = hvdm
 *                 ulBank
 *                 fSetWriteBank
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGASetBank(
  HVDM hvdm,
  ULONG ulBank,
  BOOL fSetWriteBank )
{
  hvdm;                                                         /*          */
  ulBank;                                                       /*          */
  fSetWriteBank;                                                /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAGetBank()
 *
 * DESCRIPTION   = Get bank register
 *
 *
 * INPUT         = hvdm
 *                 fGetWriteBank
 *
 * OUTPUT        = ulBank
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG PRIVENTRY vvVGAGetBank(
  HVDM hvdm,
  BOOL fGetWriteBank )
{
  hvdm;                                                         /*          */
  fGetWriteBank;                                                /*          */
  return( 0 );                                                  /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAFixSetMode()
 *
 * DESCRIPTION   = Fix up registers based on the mode set
 *
 *                 This subroutine fixes up various register states
 *                 after an int 10 set mode request completes.
 *                 Usually this is to fix BIOS bugs.
 *
 *                 This routine must get called before vvUpdateAll.
 *
 * INPUT         = None
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvSVGAFixSetMode
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGAFixSetMode(
  HVDM hvdm,
  ULONG ulMode )
{
  hvdm;
  ulMode;
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAFixSetBgnd()
 *
 * DESCRIPTION   = Fix Registers
 *
 * INPUT         = HVDM
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvSVGAFixSetBgnd
 *
 **************************************************************************/

VOID PRIVENTRY vvVGAFixSetBgnd(
  HVDM hvdm )
{
  hvdm;
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAWriteGDCDataFgnd()
 *
 * DESCRIPTION   = Do gymnastics associated with emulating the
 *                 Western Digital and Cirrus GDC regs in the background.
 *
 * INPUT         = bNewValue = new GDC reg value
 *
 * OUTPUT        = Appropriate GDC data value in shadow updated
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvWriteGDCDataFgnd
 *      vvWriteGDCDataBgnd
 *
 **************************************************************************/

VOID PRIVENTRY vvVGAWriteGDCDataFgnd(
  BYTE bNewValue )
{

  /*
  ** Don't overindex.           
  */
  if (VDMData.regGDCIndx >= MAX_GDCREGS)
    return;
  VDMData.aregGDCData [VDMData.regGDCIndx] = bNewValue;
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAReadSEQDataBgnd()
 *
 * DESCRIPTION   = Do gymnastics associated with emulating the
 *                 Trident reg defs in the background.
 *
 * INPUT         = Shadow area
 *
 * OUTPUT        = Appropriate SEQ data value from shadow
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvReadSEQDataBgnd
 *
 **************************************************************************/

BYTE PRIVENTRY vvVGAReadSEQDataBgnd( VOID )
{
  return( VDMData.aregSEQData [VDMData.regSEQIndx] );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGAWriteSEQDataBgnd()
 *
 * DESCRIPTION   = Do gymnastics associated with emulating the
 *                 Trident reg defs in the background.
 *
 * INPUT         = bNewValue = new SEQ reg value
 *
 * OUTPUT        = Appropriate SEQ data value in shadow updated
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvWriteSEQDataBgnd
 *
 **************************************************************************/

VOID PRIVENTRY vvVGAWriteSEQDataBgnd(
  BYTE bNewValue )
{
  /*
  ** Don't overindex.           
  */
  if (VDMData.regSEQIndx >= MAX_SEQREGS)
    return;
  VDMData.aregSEQData [VDMData.regSEQIndx] = bNewValue;
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacRS2Clear()
 *
 * DESCRIPTION   = Clear DAC RS2
 *
 * INPUT         = NONE
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacRS2Clear(                                /*          */
  HVDM hvdm )
{
  /* No real RS2 here, so try the DAC IPF */                    /*          */
  psCurDacInfo->pfnDacIPFClear( hvdm );                         /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacRS2Set()
 *
 * DESCRIPTION   = Reset DAC RS2
 *
 * INPUT         = NONE
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacRS2Set(                                  /*          */
  HVDM hvdm )
{
  /* No real RS2 here, so try the DAC IPF */                    /*          */
  psCurDacInfo->pfnDacIPFSet( hvdm );                           /*          */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvVGADacRS3()
 *
 * DESCRIPTION   = Set DAC RS3 State
 *
 * INPUT         = NONE
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvVGADacRS3(                                     /*          */
  HVDM hvdm,
  BOOL bRS3 )
{
  hvdm;
  bRS3;
}

#endif /* SVGA */                                               /*          */

#pragma  END_SWAP_CODE

#pragma  BEGIN_SWAP_DATA                                        /*          */
                                                                /*          */
SDACINFO sVGADacStdInfo =                                       /*          */
{                                                               /*          */
#ifdef SVGA                                                     /*          */
  VGADAC_NAME,                                                  /*          */
#else /* SVGA */                                                /*          */
  /* same as VGADAC_NAME, but SVGADEFS.H is SVGA only! */       /*          */
  "VGA",                                                        /*          */
#endif /* SVGA */                                               /*          */
  &vvVGADacStdIPFClear,                                         /*          */
  &vvVGADacStdIPFSet,                                           /*          */
  &vvVGADacStdSave,                                             /*          */
  &vvVGADacStdRestore,                                          /*          */
};                                                              /*          */

PSDACINFO psCurDacInfo = &sVGADacStdInfo;                       /*          */

SADAPTERINFO sVGAAdapterInfo =
{
#ifdef SVGA                                                     /*          */
  { /* VGAINFO */
#endif /* SVGA */                                               /*          */
    { /* EGAINFO */
      { /* SCGAINFO */
        &vvEGAAdjustedPort,
        &vvCGAStartAddr,
        &vvEGAOddEvenMode,
        &vvEGAHighResMode,
        &vvEGAHorzDspEnd,
        &vvEGAHorzLogEnd,
        &vvVGAVertDspEnd,
        &vvVGABitsPerPixel,
        &vvVGAPrepareSetMode,                                   /*          */
      },
      &vvEGAPackedMode,
      &vvVGAVertLineCmp,
    },
    &vvVGADacSave,                                              /*          */
    &vvVGADacRestore,                                           /*          */
#ifdef SVGA                                                     /*          */
  },
  &vvVGAEditTables,
  &vvVGAInterlaced,
  &vvVGAUpdateIoState,
  &vvVGARestoreIoState,
  &vvVGASetBank,
  &vvVGAGetBank,
  &vvVGAFixSetMode,
  &vvVGAFixSetBgnd,
  &vvVGAWriteGDCDataFgnd,
  &vvVGAReadSEQDataBgnd,
  &vvVGAWriteSEQDataBgnd,
  &vvVGADacRS2Clear,                                            /*          */
  &vvVGADacRS2Set,                                              /*          */
  &vvVGADacRS3,                                                 /*          */
  &sA8514Info.sAcceleratorInfo,
  pszVGAAdapterName,
  1,
  ppszVGAChipNames,
#endif /* SVGA */                                               /*          */
};

#pragma  END_SWAP_DATA                                          /*          */
