/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/************************************************************************
 *
 * SOURCE FILE NAME = VVTRIDNT.C
 *
 * DESCRIPTIVE NAME = Virtual Video Trident Specific Processing
 *
 *
 * VERSION = V2.1
 *
 * DATE      04/30/93
 *
 * DESCRIPTION  This module contains all accelerator specific SVGA code and data.
 *
 *
 * FUNCTIONS
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/


#define  IO8BIT           /* CL386 Version 6.00.054 FLAG:              */
#include <mvdm.h>
#include <vvd.h>
#include <vvdp.h>

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif
#define  INCL_DOSERRORS
#include <bseerr.h>

#pragma  BEGIN_SWAP_DATA

/*
**     Externals
*/
extern ULONG ulSVGAChipType;
extern BYTE aregSEQInit[];
extern PLE pleSEQIndx;
extern PLE pleSEQData;
extern PLE pleGDCIndx;                                          /*          */
extern PLE pleGDCData;                                          /*          */
extern PLE pleCRTIndx;                                          /*          */
extern PLE pleCRTData;                                          /*          */
extern PSSVGAINFO psCurAdapterInfo;                             /*          */
extern SA8514INFO sA8514Info;

CHAR pszTridentAdapterName [] = "TRIDENT" ;                     /*          */

PCHAR ppszTridentChipNames [MAX_TRIDENT_CHIP] =                 /*          */
{  /* As produced by SVGA.EXE! */                               /*          */
  "TR8800",
  "TR8900",
};                                                              /*          */

#pragma  END_SWAP_DATA

#pragma  BEGIN_SWAP_CODE
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvTridentInterlaced
 *
 * DESCRIPTION   = Get Interlaced Status
 *
 * INPUT         = hvdm
 *
 * OUTPUT        = True = interlacing on
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL PRIVENTRY vvTridentInterlaced(
  HVDM hvdm )
{
  register PVDMDATA pvd = pVDMData(hvdm);

  /*!!Does this explain the following fixup?! */                /*          */
//  case TRIDENT_ADAPTER:           /*                             */
//    if (pvd->vvMode.vvm_nRows == 384)
//        pvd->vvMode.vvm_nRows *= 2;
//    }
//    break;
  return( (pvd->aregCRTData [0x1e] & 0x04) >> ZEROBITS( 0x04 ) );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvTridentStartAddr
 *
 * DESCRIPTION   = Get Display frame start address
 *
 * INPUT         = hvdm
 *
 * OUTPUT        = offset into VRAM for display frame
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG PRIVENTRY vvTridentStartAddr(
  HVDM hvdm )
{
  register PVDMDATA pvd = pVDMData(hvdm);

  return(  vvCGAStartAddr( hvdm )
           | ((pvd->aregCRTData [0x1e] & 0x20)
              >> ZEROBITS( 0x20) << 16) );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvTridentHorzLogEnd
 *
 * DESCRIPTION   = Get Horizontal Logical End
 *
 * INPUT         = hvdm
 *
 * OUTPUT        = Horizontal Logical End
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT PRIVENTRY vvTridentHorzLogEnd(
  HVDM hvdm )
{
  register PVDMDATA pvd = pVDMData(hvdm);

  return( (((pvd->mstateVideo == MEMORY_GRFX)                   /*          */
            && (pvd->vvMode.vvm_nCols == 1024))
           /* This required with certain  */
           /* DRAM configurations where   */
           /* CRT offset reflects double. */
           ? (vvEGAHorzLogEnd( hvdm ) >> 1)
           : (((pvd->mstateVideo == MEMORY_GRFX256)             /*          */
               && (vvEGAHorzDspEnd( hvdm ) < 128)               /*          */
               && ((pvd->ulBIOSMode & ~BIOSVINFO_DONTCLEAR)
                   == BIOSVMODE_CO320X200X256))                 /*          */
              ? (vvEGAHorzLogEnd( hvdm ) << 1)                  /*          */
              : vvEGAHorzLogEnd( hvdm ))) );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvTridentUpdateIoState()
 *
 * DESCRIPTION   = Update SVGA I/O state
 *
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvTridentUpdateIoState(
  HVDM hvdm )
{
  BYTE bSEQIndx;
  BYTE bTmp;

  bSEQIndx = vvInput( hvdm,                                     /*          */
                      &pleSEQIndx );    /* restore reg defs   *//*          */
  pVDMData(hvdm)->VdmSVGA.stateTridentRegDef =
    VVGetTridentRegDef();
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            INDX_TRIDENT_REGDEF );
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            0x01 );                     /* set 'old' defs     */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            INDX_TRIDENT_MODECTRL );
  pVDMData(hvdm)->VdmSVGA.regTridentModeCtrl2 =
    vvInput( hvdm,                                              /*          */
             &pleSEQData );                                     /*          */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            INDX_TRIDENT_REGDEF );
  vvInput( hvdm,                                                /*          */
           &pleSEQData );               /* set new for save   *//*          */
  vvOutput( hvdm,
            &pleSEQIndx,                                        /*          */
            0x0e );                     /* get bank select    *//*          */
  bTmp = (vvInput( hvdm,                                        /*          */
                   &pleSEQData )                                /*          */
          ^ 0x02 ) | 0x80;
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            bTmp );                     /* unprotect 0x0c&0x0f*/
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            bSEQIndx );                 /* restore index */
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvTridentRestoreIoState()
 *
 * DESCRIPTION   = Do adapter-specific register restores that can't be
 *                 handled in the usual way.
 *
 * INPUT         = hvdm -> VDM
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvTridentRestoreIoState(
  HVDM hvdm )
{
  BYTE bSEQIndx;
  BYTE bTmp;

  bSEQIndx = vvInput( hvdm,                                     /*          */
                      &pleSEQIndx );    /* restore reg defs */  /*          */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            INDX_TRIDENT_REGDEF );
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            0x01 );                     /* set 'old' defs */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            INDX_TRIDENT_MODECTRL );
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            pVDMData(hvdm)->VdmSVGA.regTridentModeCtrl2 );
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            INDX_TRIDENT_REGDEF );
  vvInput( hvdm,                                                /*          */
           &pleSEQData );               /* set new for restore*//*          */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            0x0e );                     /* get bank select */
  bTmp = (vvInput( hvdm,                                        /*          */
                   &pleSEQData)                                 /*          */
          ^ 0x02 ) | 0x80;
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            bTmp );                     /* unprotect 0x0c&0x0f*/
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            bSEQIndx );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvTridentPrepareSetMode()
 *
 * DESCRIPTION   = Fix Registers & return mode number
 *
 * INPUT         = HVDM
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvInt10SetMode
 *
 **************************************************************************/

USHORT PRIVENTRY vvTridentPrepareSetMode(
  HVDM hvdm,
  PCRF pcrf )
{
  BYTE bSeqIndx;                                                /*          */

  /*
  ** Setup sequencers to default values
  */
  bSeqIndx = vvInput( CURRENT_VDM,
                      &pleSEQIndx );  /* save index           */
  vvOutput( CURRENT_VDM,
            &pleSEQIndx,
            0x0e );                   /* unprotect 0x0c & 0x0f*/
  vvOutput( CURRENT_VDM,
            &pleSEQData,
            (BYTE) ((vvInput( CURRENT_VDM,
                              &pleSEQData )
                    | 0x80) ^ 0x02) );
  vvOutput( CURRENT_VDM,
            &pleSEQIndx,
            0x0c );                   /* Select register 0x0c */
  vvOutput( CURRENT_VDM,
            &pleSEQData,
            aregSEQInit[0x0c] );      /* and set default value*/
  vvOutput( CURRENT_VDM,
            &pleSEQIndx,
            bSeqIndx );               /* restore index        */
  (*psCurAdapterInfo->pfnSVGASetBank)                           /*          */
    ( CURRENT_VDM,                                              /*          */
      BANK0,
      TRUE );
  return( vvVGAPrepareSetMode( hvdm,
                               pcrf ) );
}

/***************************************************************************
 *
 * FUNCTION NAME = vvTridentSetBank()
 *
 * DESCRIPTION   = Set Trident bank register
 *
 *                 Register 0bh at port 3c4h (Hardware Version Register) is
 *                 read-only.
 *
 *                 BUT, By writing to the register the Mode Control
 *                      registers will assume their 'old' definitions.
 *                      Reading the register causes the Mode Control
 *                      registers to assume 'new' definitions.  The
 *                      register's previous state therefore can't be
 *                      restored.  All we need in a multi-tasking
 *                      environment......  Note also that Page bit (bit 1)
 *                      is inverted when setting the write bank and doesn't
 *                      read back as written.
 *
 *                 Mode Control registers 1 & 2 are at 3c4/5 index
 *                 0Dh and 0Eh.
 *
 *                 Write operations require bit 1 (Page Select) to be
 *                 inverted.
 *
 * INPUT         = hvdm
 *                 ulBank
 *                 fSetWriteBank
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID PRIVENTRY vvTridentSetBank(
  HVDM hvdm,
  ULONG ulBank,
  BOOL fSetWriteBank )
{
  register BYTE bBank,bSeqIndx,bGDCIndx;
  BOOL fNewState;
  PVDMDATA pvd = pVDMData(hvdm);


  if (!(pvd->flVDMVideo&VDM_FGND))      /*                                      */
    return ;                            /*                                      */

  if (pVDMData(hvdm)->mstateVideo >= MEMORY_GRFX)
  {
    bGDCIndx = vvInput( hvdm,                                   /*          */
                        &pleGDCIndx );                          /*          */
    vvOutput( hvdm,                                             /*          */
              &pleGDCIndx,                        /*          *//*          */
              0x06 );                                           /*          */
    if( !(vvInput( hvdm,                                        /*          */
                   &pleGDCData)                                 /*          */
          & 0x08) )                                             /*          */
    {                                   /* if A000 referenced */
      bSeqIndx = vvInput( hvdm,                                 /*          */
                          &pleSEQIndx );/* save SEQ index */    /*          */
      fNewState = VVGetTridentRegDef();
      vvOutput( hvdm,                                           /*          */
                &pleGDCIndx,                                    /*          */
                0x06 );                 /* memory addr mode   */
      vvOutput( hvdm,                                           /*          */
                &pleGDCData,                                    /*          */
                0x05 );                 /* set to 64k         */
      vvOutput( hvdm,                                           /*          */
                &pleSEQIndx,                                    /*          */
                0x0b );                 /* This chgs Mode Ctrl*/
      vvInput( hvdm,                                            /*          */
               &pleSEQData );           /* reg's def to 'new' *//*          */
      vvOutput( hvdm,                                           /*          */
                &pleSEQIndx,                                    /*          */
                0x0e );                 /* select Mode Ctrl #1*/
      bBank = (BYTE)ulBank;
      bBank ^= 0x02;                    /* invert Page bit    */
      vvOutput( hvdm,                                           /*          */
                &pleSEQData,                                    /*          */
                bBank );                /* set the bank       */
      if( !fNewState )
      {
        vvOutput( hvdm,                                         /*          */
                  &pleSEQIndx,                                  /*          */
                  0x0b );               /* This chgs Mode Ctrl*/
        vvOutput( hvdm,                                         /*          */
                  &pleGDCData,                                  /*          */
                  0x01 );               /* set to 'old' mode  */
      }
      vvOutput( hvdm,                                           /*          */
                &pleSEQIndx,                                    /*          */
                bSeqIndx );             /* restore index      */
    }
    vvOutput( hvdm,                                             /*          */
              &pleGDCIndx,                                      /*          */
              bGDCIndx );               /* as opposed to 128k */
  }
}

/***************************************************************************
 *
 * FUNCTION NAME = vvTridentGetBank()
 *
 * DESCRIPTION   = Get Trident bank register
 *
 *                 Tseng Bank Select is done via port 0x3cd which contains
 *                 bits to select read and write banks.
 *                 The read and write banks are set up here.
 *
 *                 On the ET3000 the layout is:
 *                         bits 0-2 - write bank (64k bank)
 *                         bits 3-5 - read bank (64k bank)
 *
 *                 On the ET4000 the layout is:
 *                         bits 0-3 - write bank (64k bank)
 *                         bits 4-7 - read bank (64k bank)
 *
 * INPUT         = hvdm
 *                 fGetWriteBank
 *
 * OUTPUT        = ulBank
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG PRIVENTRY vvTridentGetBank(HVDM hvdm,BOOL fGetWriteBank)
{
  ULONG ulBank = 0;
  register BYTE bSeqIndx;
  BOOL fNewState;
  PVDMDATA pvd = pVDMData(hvdm);

  if( pvd->mstateVideo >= MEMORY_GRFX )
  {
    if( pvd->flVDMVideo & VDM_FGND )
    {
      bSeqIndx = vvInput( hvdm,                                 /*          */
                          &pleSEQIndx); /* save SEQ index */    /*          */
      fNewState = VVGetTridentRegDef();
      vvOutput( hvdm,                                           /*          */
                &pleSEQIndx,                                    /*          */
                0x0b );                 /* Chgs Mode Ctrl regs*/
      vvInput( hvdm,                                            /*          */
               &pleSEQData );           /* def 'old' to 'new' *//*          */
      vvOutput( hvdm,
                &pleSEQIndx,                                    /*          */
                0x0e );                 /* select Mode Ctrl #1*/
      ulBank = vvInput( hvdm,                                   /*          */
                        &pleSEQData );                          /*          */
      if( !fNewState )
      {
        vvOutput( hvdm,                                         /*          */
                  &pleSEQIndx,                                  /*          */
                  0x0b );               /* This chgs Mode Ctrl*/
        vvOutput( hvdm,                                         /*          */
                  &pleGDCData,                                  /*          */
                  0x01 );               /* set to 'old' mode  */
      }
      vvOutput( hvdm,                                           /*          */
                &pleSEQIndx,                                    /*          */
                bSeqIndx );             /* restore index      */
    }
    else
      ulBank = pvd->aregSEQData [0x0e] ^ 0x02;
  }
  return( ulBank & 0x0f );
}
                                                                /*          */
/***************************************************************************
 *
 * FUNCTION NAME = vvTridentFixSetMode()
 *
 * DESCRIPTION   = Fix up registers based on the mode set
 *
 *                 This subroutine fixes up various register states
 *                 after an int 10 set mode request completes.
 *                 Usually this is to fix BIOS bugs.
 *
 *                 This routine must get called before vvUpdateAll.
 *
 * INPUT         = None
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvSVGAFixRegs
 *
 ****************************************************************************/

VOID PRIVENTRY vvTridentFixSetMode(
  HVDM hvdm,
  ULONG ulMode )
{
  if( ulMode <= 0x13 )                                          /*          */
    (*psCurAdapterInfo->pfnSVGASetBank)                         /*          */
      ( hvdm,                                                   /*          */
        BANK0,                                                  /*          */
        TRUE );                                                 /*          */
}

/***************************************************************************
 *
 * FUNCTION NAME = vvTridentFixSetBgnd()
 *
 * DESCRIPTION   = Fix Trident Registers
 *
 *                 Make sure interlacing gets fixed up going background.
 *
 *                 Also flip a bit in the bank select register. Since we are
 *                 going background and UpdateIOState has already happened
 *                 this is a 'safe' place to put it. This bit is toggled on
 *                 a write to the bank select register and therefore is read
 *                 back in the opposite state to which it was written.
 *
 *           - EXTRA SUPPORT FOR CHIP LEVEL C WITH READ/WRITE
 *                              REGISTER AT INDEX 0X0C.
 *
 * INPUT         = HVDM
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvSVGAFixSetBgnd
 *
 **************************************************************************/

/*!!Shouldn't this be in the PMI file? */                       /*          */
VOID PRIVENTRY vvTridentFixSetBgnd(
  HVDM hvdm )
{
  BYTE bSeqIndx,bCRTIndx,bChipLevel,bConfig,bNMIEnable,bTmp;

  pVDMData(hvdm)->aregSEQData[0x0e] ^= 0x02;     /* flip this for restore   */
  bSeqIndx = vvInput( hvdm,                                     /*          */
                      &pleSEQIndx );    /* save SEQ index */    /*          */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            0x0b );                     /* This chgs Mode Ctrl*/
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            0x01 );                     /* set to 'old' mode  */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            0x0d );                     /* Old Mode Ctrl reg  */
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            0x00 );
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            0x0e );                     /* 128k page mode reg */
  bNMIEnable = vvInput( hvdm,                                   /*          */
                        &pleSEQData );  /* bit 5: disable NMI *//*          */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            0x0b );                     /* This chgs Mode Ctrl*/
  bChipLevel = vvInput( hvdm,                                   /*          */
                        &pleSEQData );  /* reg's def to 'new' *//*          */
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            0x0d );                     /* New Mode Ctrl reg  */
  vvOutput( hvdm,                                               /*          */
            &pleSEQData,                                        /*          */
            (BYTE) (vvInput( hvdm,                              /*          */
                            &pleSEQData)                        /*          */
                    & ~ 0x01) );        /* clear ext CLK sel  */
  if( (bChipLevel >= 3)                 /* 8900B is v3        */
      && (bNMIEnable&0x20) )            /* 8900C is           */
  {
    vvOutput( hvdm,                                             /*          */
              &pleSEQIndx,                                      /*          */
              0x0c );                   /* Config port reg 1  */
    bConfig = vvInput( hvdm,                                    /*          */
                       &pleSEQData );                           /*          */
    if( bConfig & 0x60 )
    {                                   /* if both bits set   */
      bConfig &= ~0x20;                 /* clear bit 5        */
      vvOutput( hvdm,                                           /*          */
                &pleSEQIndx,                                    /*          */
                0x0e );                 /* unprotect 0x0c     */
      bTmp = vvInput( hvdm,                                     /*          */
                      &pleSEQData)                              /*          */
             ^ 0x02;                    /* get bank sel (New Mode) */
      vvOutput( hvdm,                                           /*          */
                &pleSEQData,                                    /*          */
                (BYTE) (bTmp | 0x80) ); /* set to unprotect   */
      vvOutput( hvdm,                                           /*          */
                &pleSEQIndx,                                    /*          */
                0x0c );                 /* Config port reg 1  */
      vvOutput( hvdm,                                           /*          */
                &pleSEQData,                                    /*          */
                bConfig );              /* set new DRAM config*/
      vvOutput( hvdm,
                &pleSEQIndx,                                    /*          */
                0x0e );                 /* restore 0x0e       */
      vvOutput( hvdm,                                           /*          */
                &pleSEQData,                                    /*          */
                bTmp );                 /* restore bank select*/
    }
  }
  vvOutput( hvdm,                                               /*          */
            &pleSEQIndx,                                        /*          */
            bSeqIndx );                 /* restore index      */
  bCRTIndx = vvInput( hvdm,                                     /*          */
                      &pleCRTIndx );    /* save index         *//*          */
  vvOutput( hvdm,                                               /*          */
            &pleCRTIndx,                                        /*          */
            0x1e );                     /* CRTC Module Test   */
  vvOutput( hvdm,                                               /*          */
            &pleCRTData,                                        /*          */
            (BYTE) (vvInput( hvdm,                              /*          */
                             &pleCRTData )                      /*          */
                    & ~ 0x04) );        /* clear interlace    *//*          */
  vvOutput( hvdm,                                               /*          */
            &pleCRTIndx,                                        /*          */
            bCRTIndx );                 /* restore index      */
}

/***************************************************************************
 *
 * FUNCTION NAME = vvTridentReadSEQDataBgnd()
 *
 * DESCRIPTION   = Do gymnastics associated with emulating the
 *                 Trident reg defs in the background.
 *
 * INPUT         = Shadow area
 *
 * OUTPUT        = Appropriate SEQ data value from shadow
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvReadSEQDataBgnd
 *
 **************************************************************************/

BYTE PRIVENTRY vvTridentReadSEQDataBgnd( VOID )
{
  if( VDMData.regSEQIndx == INDX_TRIDENT_REGDEF )
    VDMData.VdmSVGA.stateTridentRegDef = 0x01;  /* 'new' definition. */
  return( ((VDMData.regSEQIndx == INDX_TRIDENT_MODECTRL)
           && !VDMData.VdmSVGA.stateTridentRegDef) /* 'old' definition? */
          ? VDMData.VdmSVGA.regTridentModeCtrl2
          : VDMData.aregSEQData [VDMData.regSEQIndx] );
}

/***************************************************************************
 *
 * FUNCTION NAME = vvTridentWriteSEQDataBgnd()
 *
 * DESCRIPTION   = Do gymnastics associated with emulating the
 *                 Trident reg defs in the background.
 *
 * INPUT         = bNewValue = new SEQ reg value
 *
 * OUTPUT        = Appropriate SEQ data value in shadow updated
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 * CALLED BY
 *      vvWriteSEQDataBgnd
 *
 **************************************************************************/

VOID PRIVENTRY vvTridentWriteSEQDataBgnd(
  BYTE bNewValue )
{
  if( VDMData.regSEQIndx == INDX_TRIDENT_REGDEF )
    VDMData.VdmSVGA.stateTridentRegDef = 0x00;  /* 'old' definition. */
  *(((VDMData.regSEQIndx == INDX_TRIDENT_MODECTRL)
     && !VDMData.VdmSVGA.stateTridentRegDef)    /* 'old' definition? */
    ? &VDMData.VdmSVGA.regTridentModeCtrl2
    : &VDMData.aregSEQData [VDMData.regSEQIndx] ) =
    bNewValue;
}

#pragma  END_SWAP_CODE

#pragma  BEGIN_SWAP_DATA

SSVGAINFO sTridentAdapterInfo =                                 /*          */
{
  { /* SVGAINFO */
    { /* SEGAINFO */                                            /*          */
      { /* SCGAINFO */
        &vvEGAAdjustedPort,
        &vvTridentStartAddr,
        &vvEGAOddEvenMode,
        &vvEGAHighResMode,
        &vvEGAHorzDspEnd,
        &vvTridentHorzLogEnd,
        &vvVGAVertDspEnd,
        &vvVGABitsPerPixel,
        &vvTridentPrepareSetMode,                               /*          */
      },
      &vvEGAPackedMode,
      &vvVGAVertLineCmp,
    },
    &vvVGADacSave,                                              /*          */
    &vvVGADacRestore,                                           /*          */
  },                                                            /*          */
  &vvVGAEditTables,
  &vvTridentInterlaced,
  &vvTridentUpdateIoState,
  &vvTridentRestoreIoState,
  &vvTridentSetBank,
  &vvTridentGetBank,
  &vvTridentFixSetMode,
  &vvTridentFixSetBgnd,
  &vvVGAWriteGDCDataFgnd,
  &vvTridentReadSEQDataBgnd,
  &vvTridentWriteSEQDataBgnd,
  &vvVGADacRS2Clear,                                            /*          */
  &vvVGADacRS2Set,                                              /*          */
  &vvVGADacRS3,                                                 /*          */
  NULL,                 //           was &sA8514Info.sAcceleratorInfo,
  pszTridentAdapterName,
  MAX_TRIDENT_CHIP,
  ppszTridentChipNames,
  NULL,                                                         /*          */
};

#pragma  END_SWAP_DATA
