/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Microsoft Corporation, 1989                                 */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/*static char *SCCSID = "@(#)vlpti05.c  6.1 90/11/19";*/
/* static char *SCCSID = "@(#) @(#)vlpti05.c  6.11 90/01/24"; */
#pragma linesize(132)
#pragma pagesize(60)
#pragma title("VLPTI05.C")
/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  VLPTI05.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  VDM interrupt 05H handler                             */
/*                                                                            */
/*   FUNCTION: These routines service the VDM software interrupt 05H.         */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: Spooler is running                                      */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             VLPTInt05    - VDM interrupt 05H handler                       */
/*             VLPTIRet5    - VDM interrupt 05H IRET handler                  */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*             VDHArmReturnHook - Set an IRET or RETF hook                    */
/*             VDHPopInt - Remove interrupt frame from client's stack         */
/*             VDHQueryHookData - Query interrupt hook data                   */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include <mvdm.h>                       /* VDH services, etc.        */
#include "vlptp.h"                      /* VLPT data defines         */
#include "vlptdata.h"                   /* VLPT external data        */


#pragma BEGIN_SWAP_CODE

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  VLPTINT05                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Virtual LPT Interrupt 05H Handler               */
/*                                                                    */
/* FUNCTION:  The function of this routine is to service the VDM      */
/*            software interrupt 05h. This routine saves the current  */
/*            per VDM data area contents for LPT1, and then           */
/*            it initializes the area so that the data from the       */
/*            print-screen request can be eventually printed.         */
/*            The direct mode PDB and the direct mode flag are not    */
/*            saved and initialized, they are not touched by          */
/*            this routine. VDHArmReturnHook is called so that when   */
/*            the CBIOS is done servicing the interrupt VLPT will be  */
/*            notified to allow for closing of the print file and     */
/*            clean-up.                                               */
/*                                                                    */
/* NOTES: This routine does not emulate CBIOS interrupt 05H. It       */
/*        simply sets up the necessary data areas so that the         */
/*        CBIOS interrupt 05H will not stomp all over the virtual     */
/*        state and contents of LPT1 when it issues interrupt 17Hs.   */
/*        When returning control this routine returns a TRUE when     */
/*        a print-screen is in progress and calls VDHPopInt to eat    */
/*        the interrupt. The return value turns off chaining.         */
/*        It returns FALSE after completing normally, i.e. allows     */
/*        for chaining to the CBIOS interrupt 05H handler.            */
/*                                                                    */
/* CONTEXT: VDM task time                                             */
/*                                                                    */
/* ENTRY POINT:  VLPTInt05                                            */
/*    LINKAGE:  CALL NEAR 32                                          */
/*                                                                    */
/* INPUT:  (parameters)  (passed on the stack)                        */
/*               pcrf  -  client register frame pointer               */
/*                                                                    */
/* EXIT-NORMAL:  FALSE (chaining enabled)                             */
/*                                                                    */
/* EXIT-ERROR:  TRUE (chaining disabled) - currently servicing        */
/*              print-screen                                          */
/*                                                                    */
/* EFFECTS:  LPT1 information in the per VDM data area.               */
/*           Print-Screen in progress flag in LPT1's per VDM data     */
/*           area.                                                    */
/*                                                                    */
/* INTERNAL REFERENCES:  chk_offset - validate LPT1 offset            */
/*                                                                    */
/* EXTERNAL REFERENCES:  VDHArmReturnHook - Set an IRET or RETF hook  */
/*                       VDHPopInt - Remove interrupt frame from      */
/*                                   client's stack                   */
/*                       VDHQueryHookData - query int. hook data      */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL HOOKENTRY VLPTInt05(pcrf)
PCRF pcrf;
{

    INT rc;                        /* return code                    */
    UINT port_id;                  /* port id                        */

    /*---------------------------------------------------------------*/
    /*- initialize return code to disable chaining (TRUE)           -*/
    /*---------------------------------------------------------------*/
    rc = TRUE;
    port_id = LPTONE;              /* initialize port_id to LPT1     */

#ifdef DEBUG
   vdhInt3();
#endif

    /*---------------------------------------------------------------*/
    /*- Make sure that port_id is still the offset of the correct   -*/
    /*- port address. This check is done by calling chk_offset. If  -*/
    /*- for some reason the user has changed the order of the       -*/
    /*- parallel port addresses in the VDMs 40:08 area then         -*/
    /*- chk_offset will return the correct offset into the VLPT     -*/
    /*- port address table.                                         -*/
    /*---------------------------------------------------------------*/
    if ((port_id = chk_offset(port_id))<(MAXPRTS+1)) {

    /*---------------------------------------------------------------*/
    /*- if the print screen in-progress flag for LPT1 in the per VDM-*/
    /*-     data area is not set then                               -*/
    /*---------------------------------------------------------------*/
        if (!(pvd_area[port_id].flags & PRT_SC_IN_PROG)) {

    /*---------------------------------------------------------------*/
    /*- copy LPT1 contents of the per VDM data area to the temporary */
    /*- location of the per VDM data area                            */
    /*---------------------------------------------------------------*/
            pvd_area[PORT_HOLD].flags = pvd_area[port_id].flags;
            pvd_area[PORT_HOLD].data = pvd_area[port_id].data;
            pvd_area[PORT_HOLD].status = pvd_area[port_id].status;
            pvd_area[PORT_HOLD].fshandle = pvd_area[port_id].fshandle;
            pvd_area[PORT_HOLD].bufcount = pvd_area[port_id].bufcount;
            pvd_area[PORT_HOLD].stream = pvd_area[port_id].stream;
            pvd_area[PORT_HOLD].pdb_spool = pvd_area[port_id].pdb_spool;
            pvd_area[PORT_HOLD].hhookLptEventContext = NULL;

    /*---------------------------------------------------------------*/
    /*- initialize the flags to zero and maintain the setting of    -*/
    /*- the direct mode flag                                        -*/
    /*---------------------------------------------------------------*/
            pvd_area[port_id].flags &= DIRECT_MODE_FLAG;

    /*---------------------------------------------------------------*/
    /*- set the print screen in-progress flag for LPT1 in the per   -*/
    /*- VDM data area                                               -*/
    /*---------------------------------------------------------------*/
            pvd_area[port_id].flags |= PRT_SC_IN_PROG;

    /*---------------------------------------------------------------*/
    /*- initialize LPT1 area of the per VDM data area to initialize -*/
    /*- state                                                       -*/
    /*---------------------------------------------------------------*/
            pvd_area[port_id].data = FLAG_NOT_SET;
            pvd_area[port_id].status = (BYTE)POWER_ON_STATUS;
            pvd_area[port_id].fshandle = INVALID_HANDLE;
            pvd_area[port_id].bufcount = 0;
            pvd_area[port_id].stream = NULL;
            pvd_area[port_id].pdb_spool = INVALID_PDB;

    /*---------------------------------------------------------------*/
    /*- call VDHQueryHookData to save the offset of the port_id that */
    /*- was used for the print screen.                               */
    /*---------------------------------------------------------------*/
            *((PUINT) VDHQueryHookData(hhookVLPTIRet5)) = port_id;

    /*---------------------------------------------------------------*/
    /*- call VDHArmReturnHook with VLPTIRet5 so that VLPT will get   */
    /*- control when CBIOS interrupt 05H has completed               */
    /*---------------------------------------------------------------*/
            VDHArmReturnHook(hhookVLPTIRet5, VDHARH_NORMAL_IRET);

    /*---------------------------------------------------------------*/
    /*- set return code to enable chaining (FALSE)                  -*/
    /*---------------------------------------------------------------*/
            rc = FALSE;

    /*---------------------------------------------------------------*/
    /*- if print-screen is in progress eat the interrupt            -*/
    /*---------------------------------------------------------------*/
        } else {

            VDHPopInt();

        } /* endif */

    /*---------------------------------------------------------------*/
    /*- A port address match was not found in the VLPT port address -*/
    /*- table for the given port_id. So, eat the interrupt.         -*/
    /*---------------------------------------------------------------*/
    } else {

        VDHPopInt();

    } /* endif */

    return(rc);

}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  VLPTIRET5                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Virtual LPT Interrupt 05H IRET Handler          */
/*                                                                    */
/* FUNCTION:  The function of this routine is to close the file       */
/*            created by interrupt 05H so that the Spooler can        */
/*            get the file printed. It also restores the original     */
/*            port id's contents in the per VDM data area. And clears */
/*            the print-screen in progress flag. The direct mode PDB  */
/*            and the direct mode flag are not touched by this        */
/*            routine.                                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: VDM task time                                             */
/*                                                                    */
/* ENTRY POINT:  VLPTIRet5                                            */
/*    LINKAGE:  CALL NEAR 32                                          */
/*                                                                    */
/* INPUT:  (parameters)  (passed on the stack)                        */
/*               vlptinst - ptr to offset portid of per VDM data area */
/*               pcrf  -  client register frame pointer               */
/*                                                                    */
/* EXIT-NORMAL:  NONE                                                 */
/*                                                                    */
/* EXIT-ERROR:  NONE                                                  */
/*                                                                    */
/* EFFECTS:  Port id information in the per VDM data area.            */
/*           Print-Screen in progress flag in port id's per VDM data  */
/*           area.                                                    */
/*                                                                    */
/* INTERNAL REFERENCES:  doCloseSpool                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:  NONE                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
VOID HOOKENTRY VLPTIRet5(pport_id,pcrf)
UINT *pport_id;
PCRF pcrf;
{
    register UINT port_id = *pport_id;

#ifdef DEBUG
    vdhInt3();
#endif
    /*---------------------------------------------------------------*/
    /*- call doCloseSpool with ptr to offset portid of the per VDM  -*/
    /*- data area that was used for the print screen.               -*/
    /*---------------------------------------------------------------*/
    doCloseSpool(&pvd_area[port_id]);

    /*---------------------------------------------------------------*/
    /*- clear the print screen in-progress flag for LPT1 in the per -*/
    /*- VDM data area and copy the previous contents of LPT1 in the -*/
    /*- per VDM data area from the temporary location of the per    -*/
    /*- VDM data area, making sure to maintain the direct mode flag -*/
    /*- setting in the port_id area of the per VDM data area        -*/
    /*---------------------------------------------------------------*/
    pvd_area[port_id].flags &= DIRECT_MODE_FLAG;
    pvd_area[port_id].flags |=
        (pvd_area[PORT_HOLD].flags & (~(DIRECT_MODE_FLAG)));
    pvd_area[port_id].data = pvd_area[PORT_HOLD].data;
    pvd_area[port_id].status = pvd_area[PORT_HOLD].status;
    pvd_area[port_id].fshandle = pvd_area[PORT_HOLD].fshandle;
    pvd_area[port_id].bufcount = pvd_area[PORT_HOLD].bufcount;
    pvd_area[port_id].stream = pvd_area[PORT_HOLD].stream;
    pvd_area[port_id].pdb_spool = pvd_area[PORT_HOLD].pdb_spool;

    return;

}
#pragma END_SWAP_CODE
