/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Microsoft Corporation, 1989                                 */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
#pragma  pagesize(55)

/**************************************************************************
 *
 * SOURCE FILE NAME = PATHS.C
 *
 * DESCRIPTIVE NAME PLOTTER DRIVER SOURCE
 *
 *
 * VERSION = V2.0
 *
 * DATE      09/18/88
 *
 * DESCRIPTION = Path related functions
 *
 *
 * FUNCTIONS
 *              draw_pattern_lines
 *              draw_patsym_diag
 *              draw_patsym_horiz
 *              draw_patsym_vert
 *              clear_patsym_area
 *              fill_the_clip_rectangles
 *              fill_polygon
 *              fill_clip_area
 *              BeginArea
 *              EndArea
 *              DevicePolygonSet
 *              PolygonSet
 *              PaintRegion
 *              FillPath
 *
 *
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/
#include "plotters.h"
#include "paths.h"
#include "color.h"
#include "box.h"
#include "dispatch.h"
#include "error.h"
#include "outpline.h"
#include "output.h"
#include "dosio.h"
#include "utils.h"
#include "xforms.h"
#include "init.h"
#include "lockddc.h"


/*
**   ADJUST PATTERN SPACING
**
**   added for ptr SM something ???
**   Fill patterns were too (emphasis) large with P7 pen type.
**
**   dlr                                     @MGX-SYSDEV, 18-JUNE-91 14:55HRS
*/

//#define  ADJUSTSPACING(spacing)  \
//         if (spacing > 4) { spacing = (spacing * 3) / 4; }

/*
** Local Functions
*/

LOCAL VOID  draw_patsym_diag(HDC,PRECTL,SHORT,SHORT,PDDC);
LOCAL VOID  draw_patsym_horiz(HDC,PRECTL,SHORT,PDDC);
LOCAL VOID  draw_patsym_vert(HDC,PRECTL,SHORT,PDDC);
LOCAL VOID  clear_patsym_area(HDC,PRECTL,SHORT,USHORT,PDDC);
LOCAL BOOL  fill_the_clip_rectangles(HDC,USHORT,PDDC);
LOCAL SHORT fill_polygon(HDC,LONG,ULONG,USHORT,PDDC);
LOCAL BOOL  fill_clip_area( HDC,PRECTL,USHORT,PDDC);
LOCAL VOID  draw_pattern_lines(HDC,PPOINTL,SHORT,BOOL,LONG,PDDC);

/***************************************************************************
** Local Defines
****************************************************************************/
/*
** area types
*/
#define AT_REGION   0
#define AT_PATH     1

#define POLY_NOTFILLED        0
#define POLY_FILLED           1
#define POLY_FILLEDBACKGROUND 2

/***************************************************************************
 *
 * FUNCTION NAME = draw_pattern_lines
 *
 * DESCRIPTION   = Draw a series of line patterns
 *
 *
 * INPUT    HDC     hDC        - device context handle
 *          PPOINTL &Points[2] - line end points
 *          PBOUNDS pBounds    - object's bounding box
 *          SHORT   iSpacing   - vertical spacing of lines
 *          BOOL    bIncX      - TRUE = increment X values FALSE = Inc Y
 *          LONG    lLines     - Number of lines to draw.
 *          PDDC    pDDC       - pointer to device driver context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
*//*
** Local Defines for this function
*/
LOCAL VOID  draw_pattern_lines(HDC hDC, PPOINTL pPoints,
                                          SHORT iSpacing, BOOL bIncX,
                                          LONG lLines, PDDC pDDC)
#define  STACK_ARRAY_SIZE 20
#define  MAX_32K       32768
{
  LONG lPoints = lLines * 2;
  LONG lMemSize;                       /* Size of allocated mem in bytes    */
  PPOINTL pPtl;                        /* pointer to line area mem area     */
  BOOL bFreeMem = FALSE;               /* TRUE if mem has been allocated    */
  BOOL bBottomToTop;                   /* TRUE = draw line bottom to top    */
  LONG nArraySize;                     /* Size of point aray being used     */
  POINTL aLinePoints[STACK_ARRAY_SIZE];/* line point pairs                  */

  /*
  ** Use stack for work area by default
  */
  pPtl = &aLinePoints[0];
  nArraySize = STACK_ARRAY_SIZE;

  /*
  ** if number of points greater than the stack array
  ** try to alloc some memory
  */
  if (lPoints > STACK_ARRAY_SIZE)
  {
    PPOINTL pTempPtl;                  /* Temp Pointer to alloc mem area    */

    /*
    ** if we expect more than 32k, just do it in 32k pieces
    */
    if ((lMemSize = sizeof(POINTL) * lPoints) > MAX_32K)
    {
      lMemSize = MAX_32K;
    }

    /*
    ** Alloc the mem. If it fails don't worry. We will just use the
    ** small stack array(slower but it works).
    */
    if (pTempPtl = (PPOINTL)GplMemoryAlloc(pDDC->pPDevice->hmcbHeap,
                                                   lMemSize))
    {
      pPtl = pTempPtl;
      nArraySize = (lMemSize/sizeof(POINTL))&~1;/* &~1 = even pairs         */
      bFreeMem = TRUE;
    }
  }                                    /* endif                             */

  /*
  **  Draw lines
  */
  /*
  ** defect 44143-SetCurrentPosition should be in world Coordinates
  ** Set flag to identify the coordinates as device.  MarkV
  */
  lPoints = 0L;
  bBottomToTop = TRUE;

  while (lLines)
  {
    /*
    ** Draw lines alternating the direction of the lines.
    ** This will connect the lines outside the bounding box
    ** and allow us to pass a large buffer of lines to polyline.
    */
    if (bBottomToTop)
    { /* Draw Bottom to Top */
      pPtl[lPoints++] = pPoints[0];
      pPtl[lPoints++] = pPoints[1];
      bBottomToTop = FALSE;
    }
    else
    { /* Draw Top To Bottom */
      pPtl[lPoints++] = pPoints[1];
      pPtl[lPoints++] = pPoints[0];
      bBottomToTop = TRUE;
    }

    /*
    ** if buffer full, send it to polyline
    **/
    if (lPoints >= nArraySize)
    {
      SetCurrentPosition(hDC, (PPOINTL)&pPtl[0], pDDC,
                         MAKEULONG(NGreSetCurrentPosition, 0));
      PolyLine(hDC, (PPOINTL)&pPtl[1], lPoints - 1L, pDDC,
                     MAKEULONG(NGrePolyLine, HIUSHORT(COM_DRAW)));
      lPoints = 0L;
    }

    /*
    ** Move line by the line spacing
    ** Line is moved to the right if bIncX is true.
    ** else the line is moved up.
    */
    if (bIncX)
    {
      pPoints[0].x += iSpacing;
      pPoints[1].x += iSpacing;
    }
    else
    {
      pPoints[0].y += iSpacing;
      pPoints[1].y += iSpacing;
    }

    lLines--;
  }

  /*
  ** If we have points in the last buffer,
  ** send it
  */
  if (lPoints)
  {
    SetCurrentPosition(hDC, (PPOINTL)&pPtl[0], pDDC,
                       MAKEULONG(NGreSetCurrentPosition, 0));
    PolyLine(hDC, (PPOINTL)&pPtl[1], lPoints - 1L, pDDC,
                       MAKEULONG(NGrePolyLine, HIUSHORT(COM_DRAW)));
  }


  /*
  ** Free memory only if we allocated it.
  */
  if (bFreeMem)
  {
    GplMemoryFree ((PVOID)pPtl);
  }
}

/***************************************************************************
 *
 * FUNCTION NAME = draw_patsym_diag
 *
 * DESCRIPTION   = Draw a series of diagonal lines into the current clip
 *                 path etc.
 *
 * INPUT         = hDC     - device context handle
 *                 pBounds - object's bounding box
 *                 Spacing - vertical spacing of lines
 *                 Slope   - pattern line slope x 100
 *                 pDDC    - pointer to device driver context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
 **************************************************************************/

LOCAL VOID  draw_patsym_diag(HDC hDC,PRECTL pBounds,
                             SHORT iSpacing, SHORT Slope,
                             PDDC pDDC)
{
  POINTL Points[2];                    /* line end points                   */
  POINTL rp;                           /* pattern offset reference point    */
  PPDEVICE pPDevice = pDDC->pPDevice;
  POINTL ptlRefPoint;                  /* Device Pattern reference point    */
  LONG   lLines;                       /* Number of lines to be drawn       */

  iSpacing *= pDDC->pPDevice->CurrentPenThickness;
  //ADJUSTSPACING(iSpacing);

  /*
  ** Put pattern reference point in device coordinates
  ** Note: We can not rely on the COM_TRANSFORM flag here.
  ** we must always assume we must do the convert because
  ** this function may be called with COM_TRANSFORM OFF and
  ** the reference point may be in world coordinates.          
  */
  ptlRefPoint.x = pDDC->DCState.abnd.ptlRefPoint.x;
  ptlRefPoint.y = pDDC->DCState.abnd.ptlRefPoint.y;
  convert_world_to_device(hDC, &ptlRefPoint, 1L, pDDC);

  /*
  **  Make pattern reference point modulus spacing
  **
  **  Calculate offset for starting point based on the difference
  **  between the reference point and upper left corner of the bounds.
  **
  */
  rp.x = iSpacing - (pBounds->xLeft - ptlRefPoint.x) % iSpacing;
  rp.y = iSpacing - (pBounds->yTop  - ptlRefPoint.y) % iSpacing;

  /*
  **      Calculate endpoints of sloped pattern line long enough
  **  to span bounding box aligned to pattern reference point
  */
  Points[0].x = pBounds->xLeft - iSpacing - rp.x;
  Points[0].y = pBounds->yBottom + rp.y;
  Points[1].x = pBounds->xRight + (iSpacing - ((pBounds->xRight -
                                   ptlRefPoint.x) % iSpacing));
  Points[1].y = Points[0].y + (((Points[1].x - Points[0].x) * Slope) / 100);

  /*
  **  Shift pattern line down until below bounding box
  */
  while (Points[0].y > pBounds->yBottom || Points[1].y > pBounds->yBottom)
  {
    Points[0].y -= iSpacing;
    Points[1].y -= iSpacing;
  }

  /*
  ** calc number of points
  ** for memory usage determination
  */
  lLines = (((pBounds->yTop + iSpacing) -
                       MIN(Points[0].y,Points[1].y)) / iSpacing);

  draw_pattern_lines(hDC, Points, iSpacing, FALSE, lLines, pDDC);


  return ;
}

/***************************************************************************
 *
 * FUNCTION NAME = draw_patsym_horiz
 *
 * DESCRIPTION   = Draw a series of horizontal lines into the current
 *                 clip path etc.
 *
 * INPUT         = hDC     - device context handle
 *                 pBounds - object's boundary rectangle
 *                 Spacing - vertical spacing interval
 *                 pDDC    - pointer to device driver context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
 **************************************************************************/

LOCAL VOID  draw_patsym_horiz(HDC hDC,PRECTL pBounds,
                              SHORT iSpacing, PDDC pDDC)
{
  POINTL Points[2];                    /* line end points                   */
  POINTL ptlRefPoint;                  /* Device Pattern reference point    */
  SHORT  iOffSet;                      /* pattern offset                    */
  LONG   lLines;                       /* number of lines to draw           */

  /*
  **   Setup line endpoints
  **   add 1 to the right and subtract 1 from the left to
  **   ensure joining lines are clipped out
  **
  **       bounds clip box
  **     Ŀ                                     Result
  **     ſ----line joins are clipped out     
  **                                       
  **    ſ                                   
  **                                        
  **     
  **  Lines are joined to allow us to work with a buffer of polylines.
  **
  */

  Points[0].x = pBounds->xLeft - 1;
  Points[0].y = pBounds->yBottom;
  Points[1].x = pBounds->xRight + 1;
  Points[1].y = pBounds->yBottom;

  iSpacing *= pDDC->pPDevice->CurrentPenThickness;
  //ADJUSTSPACING(iSpacing);

  /*
  ** Put pattern reference point in device coordinates
  ** Note: We can not rely on the COM_TRANSFORM flag here.
  ** we must always assume we must do the convert because
  ** this function may be called with COM_TRANSFORM OFF and
  ** the reference point may be in world coordinates.          
  */
  ptlRefPoint.x = pDDC->DCState.abnd.ptlRefPoint.x;
  ptlRefPoint.y = pDDC->DCState.abnd.ptlRefPoint.y;
  convert_world_to_device(hDC, &ptlRefPoint, 1L, pDDC);

  /*
  **    Apply pattern offset
  */
  iOffSet = iSpacing - (Points[0].y - ptlRefPoint.y) % iSpacing;
  Points[0].y += iOffSet;
  Points[1].y += iOffSet;

  /*
  ** calc number of points
  ** for memory usage determination
  */
  lLines = (((pBounds->yTop + iSpacing) - Points[0].y) / iSpacing);

  draw_pattern_lines(hDC, Points, iSpacing, FALSE, lLines, pDDC);

  return ;
}

/***************************************************************************
 *
 * FUNCTION NAME = draw_patsym_vert
 *
 * DESCRIPTION   = Draw a series of vertical lines into the current
 *                 clip path etc.
 *
 * INPUT         = hDC      - device context handle
 *                 pBounds  - object's boundary rectangle
 *                 iSpacing - horizontal spacing interval
 *                 pDDC     - pointer to device driver context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
 **************************************************************************/

LOCAL VOID  draw_patsym_vert(HDC hDC,PRECTL pBounds,SHORT iSpacing,
                                         PDDC pDDC)
{
  POINTL Points[2];                    /* line end points                   */
  POINTL ptlRefPoint;                  /* Device Pattern reference point    */
  SHORT  iOffSet;                      /* pattern offset                    */
  LONG   lLines;                       /* number of line to draw            */


  /*
  **  Setup line endpoints
  **   add 1 to the right and subtract 1 from the left to
  **   ensure joining lines are clipped out
  **
  **
  **     ڿڿڿڿڿ----line joins are clipped out
  **    ſ                                     Result
  **                                       
  **     -bounds clip box                  
  **                                       
  **                                       
  **    
  **      
  **  Lines are joined to allow us to work with a buffer of polylines.
  **
  */
  Points[0].x = pBounds->xLeft;
  Points[0].y = pBounds->yBottom - 1;
  Points[1].x = pBounds->xLeft;
  Points[1].y = pBounds->yTop + 1;

  iSpacing *= pDDC->pPDevice->CurrentPenThickness;
  //ADJUSTSPACING(iSpacing);

  /*
  ** Put pattern reference point in device coordinates
  ** Note: We can not rely on the COM_TRANSFORM flag here.
  ** we must always assume we must do the convert because
  ** this function may be called with COM_TRANSFORM OFF and
  ** the reference point may be in world coordinates.          
  */

  ptlRefPoint.x = pDDC->DCState.abnd.ptlRefPoint.x;
  ptlRefPoint.y = pDDC->DCState.abnd.ptlRefPoint.y;
  convert_world_to_device(hDC, &ptlRefPoint, 1L, pDDC);

  /*
  **   Apply pattern offset
  */
  iOffSet = iSpacing - (Points[0].x - ptlRefPoint.x) % iSpacing;
  Points[0].x += iOffSet;
  Points[1].x += iOffSet;

  /*
  ** calc number of points
  ** for memory usage determination
  */
  lLines = (((pBounds->xRight + iSpacing) - Points[0].x) / iSpacing);

  draw_pattern_lines(hDC, Points, iSpacing, TRUE, lLines, pDDC);

  return ;
}

/***************************************************************************
 *
 * FUNCTION NAME = clear_patsym_area
 *
 * DESCRIPTION   = Called for HPGL2 Devices only.
 *                 Depends on the code in color.c "set_color_palette()" to
 *                 set pen 0 to white.
 *                 If the area is not complex fill the area with a rectangle
 *                 command else draw a series of horizontal white lines
 *                 into the current clip path etc..
 *
 *
 * INPUT         = hDC     - device context handle
 *                 pBounds - object's boundary rectangle
 *                 iSpacing - horizontal spacing interval
 *                 pDDC    - pointer to device driver context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 * RETURN-ERROR  = NONE
 *
 **************************************************************************/

LOCAL VOID  clear_patsym_area(HDC hDC,PRECTL pBounds,SHORT iSpacing,
                                         USHORT usAreaType, PDDC pDDC)
{
  USHORT usBackMixMode = pDDC->DCState.abnd.usBackMixMode;

  /*
  ** if we are filling an area or path and the
  ** background mix mode is not simple overpaint do not
  ** clear the area.
  */
  if (usAreaType == AT_PATH &&
      usBackMixMode != BM_DEFAULT &&
      usBackMixMode != BM_OVERPAINT)
  {
    return;
  }
  /*
  ** else it's a region or the area background
  ** mix mode is a simple overpaint
  */
  else
  {
    POINTL   ptlStart;
    POINTL   ptlEnd;
    PPDEVICE pPDevice = pDDC->pPDevice;
    LONG     lColorWhite;
    LONG     lCurrentRealColor = pPDevice->CurrentRealColor;

    if (pDDC->DCState.usColorFormat != LCOLF_RGB)
        lColorWhite = CLR_WHITE;
    else
        lColorWhite = COLOR_WHITE;
    /*
    ** select pen zero which was reserved for white fills
    */
    //output_bytes(pDDC, "SP");
    //output_number(pDDC, 0L);
    select_fill_pen(pDDC, lColorWhite);

    /*
    ** set a flag which will cause the color not to change
    */
    pDDC->DCState.bClearingArea = TRUE;

    pDDC->pPDevice->bLineInConstruction = FALSE;

    ptlStart.x = pBounds->xLeft;
    ptlStart.y = pBounds->yBottom;
    move_pen(pDDC, &ptlStart, FALSE);

    /*
    ** if area is a simple rectangle
    */
    if (pDDC->DCState.ClipComplexity == 1L)
    {
      ptlEnd.x = pBounds->xRight;
      ptlEnd.y = pBounds->yTop;

      /*
      ** fill rectangle
      */
      output_bytes(pDDC, "RA");
      construct_point(pDDC, &ptlEnd, FALSE);

      /*
      ** We must turn of bLineInConstruction because continuing with
      ** points after a box command will not work.
      */
      pDDC->pPDevice->bLineInConstruction = FALSE;
    }
    /*
    ** Area is complex
    ** we must fill with lines and let the engine
    ** do the clipping.
    */
    else
    {
      if ((pBounds->xLeft - pBounds->xRight) >
          (pBounds->yTop - pBounds->yBottom))
      {
        /*
        ** Area is wider than tall
        ** draw horizontal lines.
        */
        draw_patsym_horiz(hDC, pBounds, iSpacing, pDDC);
      }
      else
      {
        /*
        ** Area is taller than wide
        ** draw horizontal lines.
        */
        draw_patsym_vert(hDC, pBounds, iSpacing, pDDC);
      }

      /*
      **   Setup starting line endpoints
      */
      ptlEnd.x = pBounds->xRight;
      ptlEnd.y = pBounds->yBottom;

    }

    ///*
    //** get current pen back
    //** using the pen palette pen number
    //*/
    //output_bytes(pDDC, "SP");
    //output_number(pDDC,
    //              (LONG)pPDevice->iHPGL2PenNumber[pPDevice->CurrentCarousel]
    //              [pPDevice->CurrentPen-1]);

    pDDC->DCState.bClearingArea = FALSE;

    /*
    ** get current pen back
    */
    select_fill_pen(pDDC, lCurrentRealColor);
  }

  return;
}

/***************************************************************************
 *
 * FUNCTION NAME = fill_the_clip_rectangles
 *
 * DESCRIPTION   = Fills the area by filling the clip rectangles.
 *
 *                 Code improved the speed of filling large simple
 *                 areas.
 *
 *                 6 clip rectangles was chosen for plotters because
 *                 it included wide line boxes which produce 4 clip
 *                 rectangles.  We do not fill more complex areas on
 *                 plotters because filling with boxes does not
 *                 optimize pen up commands on normal plotters.
 *
 *                 On HPGL2 devices we fill complex areas entirely
 *                 with this method.
 *
 * INPUT         = hDC  - device context handle
 *                 pDDC - pointer to device driver context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = TRUE if the area was filled
 * RETURN-ERROR  = NONE
 *
 **************************************************************************/

#define  STACK_ARTL_SZ 6
LOCAL BOOL  fill_the_clip_rectangles(HDC hDC, USHORT usAreaType,
                                                PDDC pDDC)
{
  BOOL bResult = TRUE;
  USHORT usBackMixMode = pDDC->DCState.abnd.usBackMixMode;

  /*
  ** if the device can not handle filling rectangles
  ** we can not fill the area with rectangles
  */
  if (!(pDDC->pPDevice->usHPGLType & HPGL_FILLRECT))
  {
    return (FALSE);
  }

  /*
  ** check for null region
  */
  if (pDDC->DCState.ClipComplexity == 0)
  {
    /*
    ** nothing to fill return we filled it, OK
    */
    return (TRUE);
  }

  ///*
  //** if we are filling a path we must check the
  //** background mix mode.  Note: regions do not use
  //** the background mix mode.
  //*/
  //if (usAreaType == AT_PATH)
  //{
  //  /*
  //  ** if fill type is not solid and
  //  ** area BACKGROUND mix mode is not simple overpaint we can not let
  //  ** the device handle the fill. We can only let the background
  //  ** see through the pattern if we fill the area with forground lines.
  //  */
  //  if (check_fill_type( pDDC ) &&
  //      (usBackMixMode != BM_DEFAULT && usBackMixMode != BM_OVERPAINT))
  //  {
  //    return (FALSE);                             /* did not fill the area */
  //  }
  //}

  /*
  ** Check if we are filling a relatively simple area.         @MV1
  ** and the device can handle fill rectangle commands
  */
  if (pDDC->DCState.ClipComplexity < 7L &&     /* simple area      */
      PlotterClass[pDDC->pPDevice->Plotter].usHPGLCaps & HPGL_FILLRECT)
  {
    /*
    ** fill clip rectangles with boxes
    */
    BOOL    bFreeMem = FALSE;
    POINTL  ptlEnd;
    RGNRECT Control;
    USHORT  cArraySize;
    PRECTL  prcl,prclArray,prclTemp;
    LONG    lMemSize;
    RECTL   rclBounds[STACK_ARTL_SZ];
    USHORT  usSymbol = pDDC->DCState.abnd.usSymbol;
    LONG    lColor   = pDDC->DCState.abnd.lColor;
    USHORT  usSet;          // saves instructions in while loops.

    /*
    ** set up to use the stack rectl array
    */
    prclArray = &rclBounds[0];
    cArraySize = STACK_ARTL_SZ;

    /*
    ** Check if number of clip rectangles is larger than our stack array
    */
    if (pDDC->DCState.ClipComplexity > STACK_ARTL_SZ)
    {

      /*
      ** Set MAX mem alloc to 8k buffer
      */
      if (pDDC->DCState.ClipComplexity > 0x400)/* > 1k RECTLS               */
          lMemSize = 0x2000;           /* 8k buffer                         */
      else
        lMemSize = pDDC->DCState.ClipComplexity *sizeof(RECTL);

      /*
      ** Alloc the mem. If it fails don't worry. We will just use the
      ** small stack array(slower but it works).
      */
      if (prclTemp = (PRECTL)GplMemoryAlloc(pDDC->pPDevice->hmcbHeap,
                                                           lMemSize))
      {
        prclArray = prclTemp;
        cArraySize = (USHORT)(lMemSize/sizeof(RECTL));
        bFreeMem = TRUE;
      }
    }                                  /* endif array bigger than stack array */

    Control.crc = cArraySize;
    Control.crcReturned = 0;

    /*
    ** Set direction from bottom to top which gives us positive
    ** increments to avoid sending the negative sign digit.
    **
    ** Note: If you change the direction you should also change
    **       the loop below that draws small boxes with lines to
    **       draw in the same direction.
    */
    Control.ulDirection = RECTDIR_LFRT_BOTTOP;
    Control.ircStart = 1;

    /*
    **  Guess we can call setup_raster_fill_index here once and use the same
    **  RFindex to fill all the rectangles if it is a PATSYM_USERDEFINED.
    **  Note: For a user defined pattern default back ground mix mode is
    **  BM_OVERPAINT. So If we have a FM_LEAVEALONE for foreground mix mode
    **  then we call check_area_background_color in setup_raster_fill_index
    **  and return to here. We then set usPatSym to PATSYM_SOLID and lColor
    **  to lBackColor and fill the rectangles with abnd.lBackColor and get
    **  out.
    **  Kran.
    */
    usSet = pDDC->DCState.abnd.usSet;
    if (usSet)  // user defined pattern
    {
      usSymbol = PATSYM_USERDEFINED;
      setup_raster_fill_index(hDC,pDDC,NULL,NULL);
      if (pDDC->DCState.abnd.usMixMode == FM_LEAVEALONE)
      {
        usSymbol = PATSYM_SOLID;
        /*
        ** If we are here then we don't need lColor. So use it.
        ** we pass this to draw_box later.
        */
        lColor = pDDC->DCState.abnd.lBackColor;
      }
    }
    while (bResult && (ULONG)Control.ircStart <= pDDC->DCState.ClipComplexity)
    {
      /*
      **  Call the engine to get the next clip rectangle (in device
      **  coordinates but in LONG format).
      */
      prcl = prclArray;
      bResult = GreGetClipRects(hDC, (PRECTL)NULL, &Control, prclArray);

      if (bResult)
      {
        Control.ircStart += Control.crcReturned;/* set next start pos       */

        /*
        ** finally draw the boxes
        */
        while (Control.crcReturned--)
        {
          /*
          ** Adjust box to make bottom left part of the
          ** interior.
          ** Clip rectangles are inclusive on the bottom left
          ** and exclusinve on the top right.
          */
          pDDC->DCState.LogPosition.x = --prcl->xLeft;
          pDDC->DCState.LogPosition.y = --prcl->yBottom;
          ptlEnd.x = prcl->xRight;
          ptlEnd.y = prcl->yTop;


          //
          //  Cant fill with lines if we are filling non solid areas
          //
          ///*
          //** if box is less than 5 lines high, it is
          //** faster to just draw the box as a series of lines
          //*/
          //if ((prcl->yTop-prcl->yBottom) < 5)
          //{
          //  POINTL ptlRight,ptlLeft;
          //
          //  ptlLeft.x = prcl->xLeft;
          //  ptlLeft.y = prcl->yBottom;
          //  ptlRight.x = prcl->xRight;
          //  ptlRight.y = prcl->yBottom;
          //
          //  /*
          //  ** Draw the lines from the closest end.
          //  */
          //  do
          //  {
          //    if (get_closest_point(&pDDC->pPDevice->PhysPosition,
          //                          &ptlLeft, &ptlRight))
          //    {
          //      move_pen(pDDC, &ptlLeft, FALSE);
          //      plot_line(pDDC, &ptlRight);
          //    }
          //    else
          //    {
          //      move_pen(pDDC, &ptlRight, FALSE);
          //      plot_line(pDDC, &ptlLeft);
          //    }                        /* endif                             */
          //
          //    ptlRight.y++;
          //    ptlLeft.y++;
          //
          //  } while (ptlLeft.y <= ptlEnd.y);

          //}
          //else
          {
            /*
            **    Make Two calls here - the reason is that the Background
            **  may be a different color to the Pattern of the box.
            */
            if (!usSet)    // standard pattern
            {
              if( check_area_background_color(pDDC) )
                draw_box( pDDC, &ptlEnd, pDDC->DCState.abnd.lBackColor, TRUE,
                          PATSYM_SOLID);

              draw_box(pDDC, &ptlEnd, lColor, TRUE, usSymbol);
            }
            else  // user defined pattern. we need to call draw_box only once.
            {     // Reason: both background and foreground is reprasented by
                  // the pattern bitmap bits.
              draw_box(pDDC, &ptlEnd, lColor, TRUE, usSymbol);
            }
          }
          prcl++;
        }                              /* endwhile                          */
      }
    }                                  /* endwhile                          */

    if (bFreeMem)
    {
      GplMemoryFree ((PVOID)prclArray);
    }

    return (bResult);
  }
  else
  {
    /*
    ** we did not fill it
    */
    return (FALSE);
  }
}                                      /* end fill_the_clip_rectangles      */
#undef   STACK_ARTL_SZ

/***************************************************************************
 *
 * FUNCTION NAME = fill_polygon
 *
 *      Current plotters can only handle 1 clip rectangle.
 *      Note: Older plotters have small polygon
 *            buffers this method may not always work.
 *
 *   NOTE: On the 32bit Engine (ver 2.02 or greater )
 *         calling GreOutlinePath with the new option (NO_CLIPPING_REQD )
 *         results in a call to our DrawLinesInPath
 *         function for each subpath-unclipped.
 *         On the 16bit Engine Calling GreOutlinePath results in
 *         many calls to our PolyShortLine function for each subpath.
 *         Because the end of the subpath is too hard to predect on the
 *         16bit engine or older versions of the 32bit engine we will
 *         only use the polygon filling method on engine versions 202
 *         or greater.
 *
 * DESCRIPTION   = Fills the area by querying the outline of the path and
 *                 filling the outline as a HPGL polygon.
 *                 Function only fills the area if:
 *                 clipping is simple. (1 clip)
 *                 fill mode is alternate (FPATH_ALTERNATE)
 *                 engine version is 202 or greater
 *
 * INPUT         = hDC
 *                 pathId
 *                 flOptions
 *                 usComFlags
 *                 pDDC
 *
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = 0 POLY_NOTFILLED  polygon was not filled
 *                 1 POLY_FILLED     Polygon was filled
 *
 **************************************************************************/

#ifndef  OPTH_NO_CLIPPING_REQD         /* = Don't clip: def. in 2.0 eng ver
                                          202+                              */
  #define  OPTH_NO_CLIPPING_REQD 0x01L
#endif

#ifndef  OPTH_QRY_PATH_POINTS          /* = ret # of pts: def. in 2.0 eng
                                          ver 202+                          */
  #define  OPTH_QRY_PATH_POINTS 0x02L
#endif
LOCAL SHORT  fill_polygon(HDC hDC,LONG PathId,ULONG flOptions,USHORT
                                     usComFlags,PDDC pDDC)
{
  SHORT    sResult    = POLY_FILLED;
  PRECTL   pClipRect  = &pDDC->DCState.ClipRect;
  PPDEVICE pPDevice   = pDDC->pPDevice;
  LONG     lNumPoints;
  LONG     lMaxPoints;
  LONG     hSavedDC   = 0;
  LONG     lColor;
  USHORT   usSymbol   = pDDC->DCState.abnd.usSymbol;

  /*
  ** If the Engine version in < 0x202 GreOutlinePath will not return
  ** the outline in a form we can use.
  ** or filling option is not FPATH_ALTERNATE or
  ** not an HPGL2 device.
  ** Return without filling the area.
  */
  if ( flOptions != FPATH_ALTERNATE ||
       pDDC->lEngineVersion < 0x202L ||
       !(PlotterClass[pDDC->pPDevice->Plotter].usHPGLCaps & HPGL2) )
  {
    return (POLY_NOTFILLED);
  }

  /*
  ** Check for complex areas.
  ** We do not want to fill complex areas because the HPGL device
  ** can not handle complex clipping.
  */
  if (pDDC->DCState.ClipComplexity > 1)
  {
    /*
    ** return and indicate we did not fill the area
    */
    return (POLY_NOTFILLED);
  }

  /*
  ** Check if the polygon is too big to fit in the device polygon buffer.
  ** Note:OPTH_QRY_PATH_POINTS on GreOutlinePath only works on GRE
  **      Versions 202 or greater
  */
  lNumPoints = GreOutlinePath(hDC, PathId, OPTH_QRY_PATH_POINTS);
  lMaxPoints = (pPDevice->pSetup->bGECOption ?
                 PlotterClass[pPDevice->Plotter].usMaxPolygonSizeGEC :
                 PlotterClass[pPDevice->Plotter].usMaxPolygonSize);

  if ((lNumPoints == GPI_ERROR) || (lNumPoints > lMaxPoints))
  {
    return (POLY_NOTFILLED);
  }

  /*
  ** if NULL region delete the path and return success.
  */
  if (pDDC->DCState.ClipComplexity == 0)
  {
    /*
    ** Delete the Path
    */
    (*daFillPath)(hDC, PathId, flOptions, pDDC,
                  MAKEULONG(NGreFillPath, (usComFlags &
                  ~(HIUSHORT(COM_DRAW | COM_BOUND)))));
    return(POLY_FILLED);
  }

  /************************************************************************
  ** Fill the polygon
  ** by outlining the area and sending the outline to
  ** the device in polygon mode.
  ** See code in DrawLinesInPath.
  *************************************************************************/
  pDDC->DCState.bFilling = TRUE;
  pPDevice->bFillingPolygon = TRUE;
  pPDevice->bFirstPolygonPtSet = FALSE;
  /*
  ** Outline the polygon.
  ** GreOutlinePath will outline the path without clipping and
  ** delete the path
  */
  if (GPI_ERROR == GreOutlinePath(hDC, PathId, OPTH_NO_CLIPPING_REQD))
  {
    sResult = POLY_NOTFILLED;
  }                                    /* endif                             */

  /*
  ** if we started the polygon fill it
  */
  if (pDDC->pPDevice->bFirstPolygonPtSet)
  {
    POINTL ptlBottomLeft,ptlTopRight;
    USHORT usMixMode;

    /*
    ** Finish the current polygon
    ** PM2=Exit Polygon Mode
    */
    output_bytes(pDDC, "PM2");

    /*
    ** We must turn of bLineInConstruction because continuing with
    ** points after ending a polygon will not work.
    */
    pPDevice->bLineInConstruction = FALSE;

    /*
    ** We do not know the size and don't realy care about the
    ** fill direction in  HPGL2 devices so we just set the
    ** bounds to null.
    */
    ptlBottomLeft.x = 0;
    ptlBottomLeft.x = 0;
    ptlTopRight.x = 0;
    ptlTopRight.y = 0;

    if(pDDC->DCState.abnd.usSet)     // user defined pattern
    {
      usSymbol = PATSYM_USERDEFINED;
      setup_raster_fill_index(hDC,pDDC,NULL,NULL);
    }
    /*
    ** If the BackGround mix  mode is overpaint and
    ** the foreground pattern is not solid we must
    ** Paint the background in polygon mode too.
    ** We dont need to do this if it is a user defined pattern. Kran
    */
    else if(!pDDC->DCState.abnd.usSet && check_area_background_color(pDDC) )
    {
      select_fill_pen(pDDC, pDDC->DCState.abnd.lBackColor);
      select_fill_type(pDDC, PATSYM_SOLID, &ptlBottomLeft, &ptlTopRight);
      output_bytes(pDDC, "FP");
    }


    /*
    ** Special case stuff for clearing areas with PATSYM_NOSHADE
    ** or PATSYM_BLANK.  Note: On HPGL2 devices we reserve pen
    ** zero for clearing areas.
    */
    usMixMode = pDDC->DCState.abnd.usMixMode;

    if (pDDC->DCState.abnd.usSet ||  check_area_color(pDDC))
    {
      /*
      ** If we are on an HPGL2 device and
      ** PATSYM is clearing the area- select pen zero
      **
      ** Should also check if area mix mode is a simple overpaint
      */
      //if ((PlotterClass[pDDC->pPDevice->Plotter].usHPGLCaps & HPGL2) &&
      //   (usSymbol == PATSYM_NOSHADE ||
      //    usSymbol == PATSYM_BLANK))
      //{
      //  /*
      //  ** select pen zero which was reserved for white fills
      //  */
      //  /*
      //  ** Force color to White
      //  */
      //  if (pDDC->DCState.usColorFormat != LCOLF_RGB)
      //    lColor = CLR_WHITE;
      //  else
      //    lColor = COLOR_WHITE;
      //}
      //else
      {
        lColor = pDDC->DCState.abnd.lColor;
      }

      if((pDDC->DCState.abnd.usSet) && (usMixMode == FM_LEAVEALONE))
      {
        lColor = pDDC->DCState.abnd.lBackColor;
        usSymbol = PATSYM_SOLID;
      }

      /*
      ** set fill type to the area forground pattern
      */
      select_fill_pen(pDDC, lColor);

      if(!pDDC->DCState.abnd.usSet)  // standard pattern.  Kran
        select_fill_type(pDDC, usSymbol, &ptlBottomLeft, &ptlTopRight);
      else
        select_fill_type(pDDC, usSymbol, &ptlBottomLeft, &ptlTopRight);

      /*
      ** Tell the plotter to fill the polygon
      ** FP=Fill polygon
      ** Fill polygon (FP) in HPGL only fills the interior.
      ** Since the engine fill is inclusive we must also edge the
      ** polygon to include the edge. However, edge polygon (EP) draws
      ** a solid line which will not match the area fill pattern if the
      ** area fill was not solid.  Therefore, we only edge the polygon
      ** if the area is solid and accept that area fills of non solid
      ** patterns will not include the edge.
      ** EP=Edge Polygon (since the engine fill is inclusive of the border)
      */
      output_bytes(pDDC, "FP");

      if (usSymbol == PATSYM_DEFAULT ||
          usSymbol == PATSYM_SOLID)
      {
        output_bytes(pDDC, "EP");
      }
    }

    /*
    **  Paintjet HPGL2 gets out of sync with our penup pendown
    **  So make sure we are in sync by sending the command to put the
    **  pen in the position we think its in.
    */
    if (pPDevice->bPenIsUp)
      output_bytes(pDDC, "PU");
    else
      output_bytes(pDDC, "PD");

    /*
    ** We force a move to the current position here because the plotter
    ** (designjet 650C) will leave the current position at the polygon
    ** close point which may cause us to get out of sync with the application.
    ** Lotus Feelance 2.0 is using 5 points to draw a box and closes the
    ** box on the 4th point. We will think the current position is the
    ** 5th point. However, the plotter leaves the current position at the
    ** 4th point. MV
    */
    move_pen(pDDC, &pDDC->DCState.LogPosition, TRUE);
  }

  pPDevice->bFirstPolygonPtSet = FALSE;
  pPDevice->bFillingPolygon = FALSE;
  pDDC->DCState.bFilling = FALSE;

  /*
  ** if we saved the DC and painted the polygon background
  ** restore the DC
  */
  if (hSavedDC)
    GreRestoreDC(hDC, hSavedDC);

  return (sResult);
}                                      /* end fill_polygon                  */

/***************************************************************************
 *
 * FUNCTION NAME = fill_clip_area
 *
 * DESCRIPTION   = Fills the currently selected clip path or clip region.
 *
 *
 * INPUT         = hDC  - device context handle
 *                 pDDC - pointer to device driver context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = TRUE if the area was filled
 * RETURN-ERROR  = FALSE
 *
 *   11/28/92                      Added Hook for PaintRegion
 **************************************************************************/
LOCAL BOOL  fill_clip_area( HDC  hDC, PRECTL prclBounds,
                                       USHORT usAreaType, PDDC pDDC)
{
  /*
  **      At this stage,  we have accumulated bounds (for the
  **  clip path defining our area) in PLOTTER pels.  The
  **  following function calls work on the basis of plotter
  **  pels,  and call any other functions with COM_TRANSFORM
  **  disabled,  so all drawing is done in plotter pels.
  **  This results in line spacing on the plotter being done
  **  in a consistent manner,  regardless of any transforms
  **  operating in the system.
  */

  /*
  ** Local vars
  */
  USHORT LineType;
  RECTL  Bounds;

  Bounds = *prclBounds;
  pDDC->DCState.bFilling = TRUE;

  /*
  **  Temporarily change the line type - restore when done
  */
  LineType = pDDC->DCState.lbnd.usType;
  pDDC->DCState.lbnd.usType = LINETYPE_SOLID;

  /*
  ** preselect fill pen
  */
  select_fill_pen(pDDC, pDDC->DCState.abnd.lColor);

  /*
  ** Try to fill the area by filling the clip rectangles
  ** with boxes. if not, proceed with normal fill.
  */
  if (!fill_the_clip_rectangles(hDC, usAreaType, pDDC))
  {
    /*
    ** we do not need to do fill optimization if the area
    ** is simple. The lines are already in closest line order
    **/
    BOOL bDoFillOpt = (pDDC->DCState.ClipComplexity > 1);

    /*
    ** Complex area or (not a solid fill and device does not
    ** support fill types)
    ** Fill with lines.
    */

    /*
    ** IF WE ARE ON AN HPGL2 DEVICE AND THE PATTERN
    ** IS NOT SOLID WE MUST ZERO THE AREA WITH
    ** WHITE SPACE BEFORE APPLYING THE
    ** PATTERN. MARKV
    */
    if (pDDC->DCState.abnd.usSymbol != PATSYM_DEFAULT &&
        pDDC->DCState.abnd.usSymbol != PATSYM_SOLID &&
        PlotterClass[pDDC->pPDevice->Plotter].usHPGLCaps & HPGL2)
    {
      /*
      ** Zero or white out the area
      */
      clear_patsym_area(hDC, &Bounds, 1, usAreaType, pDDC);
    }

    /*
    ** initialize line drawing optimization
    */
    if (bDoFillOpt)
      fill_opt_init(pDDC);

    switch (pDDC->DCState.abnd.usSymbol)
    {
      case  PATSYM_DENSE1 :
        draw_patsym_horiz(hDC, &Bounds, 4, pDDC);
        draw_patsym_vert(hDC, &Bounds, 4, pDDC);
        break;

      case  PATSYM_DENSE2 :
        draw_patsym_horiz(hDC, &Bounds, 5, pDDC);
        draw_patsym_vert(hDC, &Bounds, 5, pDDC);
        break;

      case  PATSYM_DENSE3 :
        draw_patsym_horiz(hDC, &Bounds, 6, pDDC);
        draw_patsym_vert(hDC, &Bounds, 6, pDDC);
        break;

      case  PATSYM_DENSE4 :
        draw_patsym_horiz(hDC, &Bounds, 7, pDDC);
        draw_patsym_vert(hDC, &Bounds, 7, pDDC);
        break;

      case  PATSYM_DENSE5 :
        draw_patsym_horiz(hDC, &Bounds, 8, pDDC);
        draw_patsym_vert(hDC, &Bounds, 8, pDDC);
        break;

      case  PATSYM_DENSE6 :
        draw_patsym_horiz(hDC, &Bounds, 10, pDDC);
        draw_patsym_vert(hDC, &Bounds, 10, pDDC);
        break;

      case  PATSYM_DENSE7 :
        draw_patsym_horiz(hDC, &Bounds, 13, pDDC);
        draw_patsym_vert(hDC, &Bounds, 13, pDDC);
        break;

      case  PATSYM_DENSE8 :
        draw_patsym_horiz(hDC, &Bounds, 16, pDDC);
        draw_patsym_vert(hDC, &Bounds, 16, pDDC);
        break;

      case  PATSYM_HORIZ :
        draw_patsym_horiz(hDC, &Bounds, 10, pDDC);
        break;

      case  PATSYM_VERT :
        draw_patsym_vert(hDC, &Bounds, 10, pDDC);
        break;

      case  PATSYM_DIAG1 :
        draw_patsym_diag(hDC, &Bounds, 16, 100, pDDC);
        break;

      case  PATSYM_DIAG2 :
        draw_patsym_diag(hDC, &Bounds, 15, 60, pDDC);
        break;

      case  PATSYM_DIAG3 :
        draw_patsym_diag(hDC, &Bounds, 16, -100, pDDC);
        break;

      case  PATSYM_DIAG4 :
        draw_patsym_diag(hDC, &Bounds, 15, -60, pDDC);
        break;

      case  PATSYM_HALFTONE :
        draw_patsym_diag(hDC, &Bounds, 9, 100, pDDC);
        draw_patsym_diag(hDC, &Bounds, 9, -100, pDDC);
        break;

      case  PATSYM_DEFAULT :
      case  PATSYM_SOLID :
        if (Bounds.xRight - Bounds.xLeft >=
            Bounds.yTop   - Bounds.yBottom)
          draw_patsym_horiz(hDC, &Bounds, 1, pDDC);

        else
          draw_patsym_vert(hDC, &Bounds, 1, pDDC);
        break;

      case  PATSYM_NOSHADE :
      case  PATSYM_BLANK :
      default  :
        break;
    }

    /*
    ** flush any buffered lines
    */
    if (bDoFillOpt)
      fill_opt_end(pDDC);
  }                            /* end else                          */

  /*
  ** Restore the line type
  */
  pDDC->DCState.lbnd.usType = LineType;

  /*
  ** no longer filling
  */
  pDDC->DCState.bFilling = FALSE;

  return(TRUE);

}                                                       /* end fill_clip_area */

/*
** Exported Routines
*/
/***************************************************************************
 *
 * FUNCTION NAME = StrokePath
 *
 * DESCRIPTION   = This function converts the path to a wide line.
 *                 Note: The select_pen function in color.c will
 *                 set the pen width to the geometric width if
 *                 instance varable  pDDC->DCState.bWideLines is true.
 *                 Setting of lineend and linejoin will be done in
 *                 set_line_type if pDDC->DCState.bWideLines is set.
 *
 *
 * INPUT         = hDC     - device context handle
 *                 pathId  - filling path
 *                 flOptions - filling control options
 *                 pDDC    - pointer to device driver context
 *                 FunN    - function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = GPI_OK    - Success
 * RETURN-ERROR  = GPI_ERROR - Failure
 *
 ****************************************************************************/

LONG  StrokePath(HDC hDC,LONG PathId,ULONG flOptions,PDDC pDDC,ULONG
                            FunN)
{
  LONG lResult = GPI_ERROR;


  if(!(EnterDriver(pDDC)))
    return(GPI_ERROR);

  if ((PlotterClass[pDDC->pPDevice->Plotter].usHPGLCaps & HPGL2) &&
     (pDDC->DCState.abnd.usSymbol == PATSYM_DEFAULT ||
      pDDC->DCState.abnd.usSymbol == PATSYM_SOLID) &&
      check_area_color(pDDC) &&
      (FunN & COM_DRAW))
  {
    /*
    ** Setting of lineend and linejoin will be done in set_line_type
    ** if DCState.bWideLines is set.
    ** set_line_type is called from the line drawing routines.
    */
    pDDC->DCState.bWideLines = TRUE;

    /*
    ** use the area attributes
    */
    pDDC->DCState.bFilling = TRUE;

    /*
    ** Force pen selection to set pen width
    */
    pDDC->pPDevice->bPenIsSelected = FALSE;

    /*
    ** Calling the engine outline path function before the path
    ** has been modified (widened) results in line calls which
    ** we can widen in the HPGL2 device.
    */
    lResult = GreOutlinePath(hDC, PathId, 0L);

    pDDC->DCState.bWideLines = FALSE;
    pDDC->DCState.bFilling   = FALSE;

    /*
    ** force pen to be set back to normal width upon next use.
    */
    pDDC->pPDevice->bPenIsSelected = FALSE;
  }
  else
  {
    /*
    ** let the engine handle it
    */
    lResult = (*daStrokePath)(hDC, PathId, flOptions, pDDC, FunN);
  }

  LeaveDriver(pDDC);
  return (lResult);
}                                      /* end StrokePath                    */

/***************************************************************************
 *
 * FUNCTION NAME = BeginArea
 *
 * DESCRIPTION   = Function indicating the beginning of a set of
 *                 drawing operations to define an area boundary.
 *                 Areas us the attributes at begin area to fill
 *                 the area.
 *
 * INPUT         = hDC   - device context handle
 *                 Flags - determines filling and boundary operations
 *                 pDDC  - pointer to device driver context
 *                 FunN  - function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = GPI_OK    - Success
 * RETURN-ERROR  = GPI_ERROR - Already in area, or engine return code
 *
 ****************************************************************************/

LONG  BeginArea(HDC hDC,ULONG Flags,PDDC pDDC,ULONG FunN)
{
  LONG Result = GPI_OK;


  if(!(EnterDriver(pDDC)))
      return(GPI_ERROR);
  if (pDDC->DCState.bInArea)
  {

    /*
    **  Not allowed,  so record the error and tell the caller
    */
    GplErrSetError(PMERR_ALREADY_IN_AREA);
    LeaveDriver(pDDC);
    return  GPI_ERROR;
  }

  if ((check_area_color(pDDC)||check_area_background_color(pDDC)) &&
     (FunN & COM_DRAW))
  {
    /*
    **   We are actually doing it this time,  so on with the show.
    */

    pDDC->DCState.bInArea = TRUE;
    pDDC->DCState.ulAreaFlags = Flags;

    /*
    ** save the begin area attributes
    */
    pDDC->DCState.abndBeginArea = pDDC->DCState.abnd;

    GreBeginPath(hDC, 1L);

    //GreSetProcessControl(hDC, COM_AREA|COM_PATH, COM_AREA);
  }
  else
    Result = (*daBeginArea)(hDC, Flags, pDDC, FunN);

  LeaveDriver(pDDC);
  return  Result;
}

/***************************************************************************
 *
 * FUNCTION NAME = EndArea
 *
 * DESCRIPTION   = Indicate the end of the drawing operations defining
 *                 the area.
 *
 * INPUT         = hDC    - device context handle
 *                 Cancel - cancel area if zero
 *                 pDDC   - pointer to device driver context
 *                 FUnN   - function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = GPI_OK    - Success
 * RETURN-ERROR  = GPI_ERROR - Failure
 *
 **************************************************************************/

LONG  EndArea(HDC hDC,ULONG Cancel,PDDC pDDC,ULONG FunN)
{
  LONG Result = GPI_ERROR;
  AREABUNDLE  abndTemp;


  //if (pDDC->DCState.bInArea)
    //GreSetProcessControl(hDC, COM_AREA|COM_PATH, COM_PATH);

  if(!(EnterDriver(pDDC)))
    return(GPI_ERROR);
  if (pDDC->DCState.bInArea)
  {
    ULONG flOptions;                     /* Winding or Alternate path options */

    /*
    ** Close the figure if the BOUNDARY attrib set.
    **
    ** Moved from above if() to match begin area logic
    ** and fix a play metafile error PMERR_NOT_IN_PATH 0x20E1
    ** during PTT color test.  MarkV
    **
    ** Change to be like the engine and always close the path
    ** Changed back ????
    */
    if (pDDC->DCState.ulAreaFlags & BA_BOUNDARY)
      GreCloseFigure(hDC);

    /*
    ** It's for us,  so perform the magic required of us
    */
    flOptions = (pDDC->DCState.ulAreaFlags & BA_WINDING) ? FPATH_WINDING:
                                                         FPATH_ALTERNATE;

    if (GreEndPath(hDC, Cancel) && Cancel != EA_CANCEL)
    {
      LONG hSavedDC;

      /*
      ** must save the DC to preserve the path if we
      ** are going to draw the boundary
      */
      if (pDDC->DCState.ulAreaFlags & BA_BOUNDARY)
        hSavedDC = GreSaveDC(hDC);

      /*
      ** Set Area attrubutes to the begin area attributes we saved
      */
      abndTemp = pDDC->DCState.abnd;
      pDDC->DCState.abnd  =  pDDC->DCState.abndBeginArea;

      Result = GreFillPath(hDC, 1L, flOptions);

      /*
      ** Put area attrs back to what the user has left them at
      */
      pDDC->DCState.abnd  =  abndTemp;

      if (pDDC->DCState.ulAreaFlags & BA_BOUNDARY)
      {
        GreRestoreDC(hDC, hSavedDC);
        Result = GreOutlinePath(hDC, 1L, 0L);
      }
    }
  }
  else
    Result = (*daEndArea)(hDC, Cancel, pDDC, FunN);

  pDDC->DCState.bInArea = FALSE;

  LeaveDriver(pDDC);
  return  Result;
}

/***************************************************************************
 *
 * FUNCTION NAME = PolygonSet
 *
 * DESCRIPTION   = This Function draws a set of closed polygons. The
 *                 polygons are filled using the current AREABUNDLE
 *                 structure values. For the first polygon, the current
 *                 position is the first point. For all subsequent polygons
 *                 all points that define the polygon are given explicity.
 *                 The Polygons are automaticlly closed, if necessary, by
 *                 drawing a line from the last vertex to the first. Note:
 *                 Polygons can overlap if needed.
 *                 Note: New function for 2.0
 *                 Note: GrePolygonSet is not valid when the COM_AREA or
 *                       COM_PATH flag is set.
 *
 *                 This function is called as a result of GpiPolygonSet
 *                 and internaly be DevicePolygonSet.
 *
 *
 * INPUT         = hDC    - device context handle
 *                 flModel- Determines inclusive/exclusive border rules
 *                          POLYGON_INCL - 0L Default Setting.
 *                                         Fill is inclusive of bottom right
 *                          POLYGON_EXCL - 8L Fill is exclusive of bottim right
 *                                         Aids migration from other graphics
 *                                         models.
 *                 flOptions - Determines filling Rules
 *                          POLYGON_ALTERNATE - 0L Alternate fill
 *                          POLYGON_WINDING - 2L Winding fill
 *                          POLYGON_BOUNDARY - 1L Draw boundary
 *                          POLYGON_NOBOUNDARY - 0L Don't Draw boundary
 *                 paWorldPolygon - Ptr to an aray of Polygon structure
 *                          ulPoints- Number of points
 *                          aPointl - Aray of points
 *                 cPolygons Number of polygon structures in the aray
 *                 pDDC   - pointer to device driver context
 *                 FUnN   - function number
 *
 *
 * RETURN-NORMAL = GPI_OK    - Success
 * RETURN-ERROR  = GPI_ERROR - Failure
 *
 *   04/25/93               Added Hook for PolygonSet
 *
 **************************************************************************/

LONG  PolygonSet( HDC hDC, LONG flModel, LONG flOptions,
                            PPOLYGON paWorldPolygon, LONG cPolygons,
                            PDDC  pDDC, ULONG FunN)
{
  PPDEVICE pPDevice;
  LONG lResult      = GPI_OK;
  LONG lNumPoints   = 0L;   /* Number of points in polygon structures */
  ULONG ulSize      = 0L;   /* Size of aPolygon passed in */
  LONG lMaxPoints;          /* Max # of polygon points this ploter can do */
  LONG lPolyIndex;          /* Loop counters for #polygons  */
  ULONG ulPointIndex;       /* Loop counters for #Points in polygon  */
  LONG  lColor;
  USHORT usSymbol;

  if(!(EnterDriver(pDDC)))
    return(GPI_ERROR);

  pPDevice = pDDC->pPDevice;

  /*
  ** Get the maximum number of points that this plotter
  ** can handle in polygon mode.
  */
  lMaxPoints = (pPDevice->pSetup->bGECOption ?
                   PlotterClass[pPDevice->Plotter].usMaxPolygonSizeGEC :
                   PlotterClass[pPDevice->Plotter].usMaxPolygonSize);

  /*
  ** Get the number of points in the polygon structrue
  */
  for (lPolyIndex = 0; lPolyIndex < cPolygons ; lPolyIndex++ )
  {
    lNumPoints += paWorldPolygon[lPolyIndex].ulPoints;
  }

  /*
  ** if we can not do the polygons let the engine handle it
  */
  if (!(FunN & COM_DRAW) ||
     (FunN  & (COM_BOUND | COM_PATH | COM_AREA)) ||
     (!check_area_color(pDDC)) ||
     (pDDC->DCState.ClipComplexity > 1L) ||
     (lNumPoints > lMaxPoints) ||
     (flModel & POLYGON_EXCL) ||
     (flOptions & POLYGON_WINDING))
  {
    LeaveDriver(pDDC);
    return ((LONG)(*daPolygonSet)(hDC, flModel, flOptions, paWorldPolygon,
                                  cPolygons, pDDC, FunN));
  }
  /*
  ** else Let's do it
  */
  else
  {
    PPOLYGON paPolygon;       /* ptr to an array of polygons */
    PBYTE    pByte;           /* ptr to allocated memory */
    POINTL   ptlBottomLeft,   /* Bounding rectangle points for determining */
             ptlTopRight;     /* fill drection - Note: set to 0 (not used) */
    POINTL   ptlStart;        /* First point of polygon */
    USHORT   usMixMode;       /* area mix mode */
    BOOL     bResetPen = FALSE;
    BOOL     bFreeMem  = FALSE;

    pDDC->pPDevice->bLineInConstruction = FALSE;

    /*
    **  If the data is supplied in world coordinates,  transform
    **  to device.  Copy data to local buffer,  then transform the
    **  copy.
    */
    if (FunN & COM_TRANSFORM)
    {

       /*
       ** Calculate the size of the polygons and polygon points
       ** then allocate some memory.
       */
       ULONG  ulAPolySize = cPolygons * sizeof(POLYGON);
       ulSize = lNumPoints * sizeof(POINTL);
       ulSize += ulAPolySize;
       if (pByte = (PBYTE)GplMemoryAlloc(pDDC->pPDevice->hmcbHeap,
                                                          ulSize))
       {
          PPOINTL  pPointl;
          bFreeMem = TRUE;

          /*
          ** Copy the array of polygon structures
          */
          paPolygon = (PPOLYGON)pByte;
          CopyMem((PBYTE)paPolygon, (PBYTE)paWorldPolygon,
                  (USHORT)ulAPolySize);
          /*
          ** Write point arrays just beyond the array of polygon structs
          */
          pPointl = (PPOINTL)&pByte[ulAPolySize];

          /*
          ** Loop through each polygon structure Copying each set
          ** of polygon points, fix up the pointer to
          ** the polygon points, and
          ** convert the points for each polygon
          */
          for (lPolyIndex = 0; lPolyIndex < cPolygons ; lPolyIndex++ )
          {
             /*
             ** Copy polygon points
             */
             CopyMem( (PBYTE)pPointl, (PBYTE)paPolygon->aPointl,
                       (USHORT)paPolygon->ulPoints * sizeof(POINTL));
             /*
             ** Fix up polygon points pointer to point to the new area
             */
             paPolygon->aPointl = pPointl;

             convert_world_to_device(hDC, paPolygon->aPointl,
                                          paPolygon->ulPoints,
                                          pDDC);
             /*
             ** Move pointer to the free area just beyond the current
             ** array
             */
             pPointl += paPolygon->ulPoints * sizeof(POINTL);

             paPolygon++;
          }  /* end for */

          /*
          ** Put polygon pointer back at the beginning
          */
          paPolygon = (PPOLYGON)pByte;
       }
       else
       {
         /*
         ** ERROR PMERR_INSUFFICIENT_MEMORY
         */
         LeaveDriver(pDDC);
         return(GPI_ERROR);
       }
    }
    else
    {
      paPolygon = paWorldPolygon;
    }

    lColor = pDDC->DCState.abnd.lColor;

    /*
    ** Check if this guy wants to use user defined pattern to fill the polygons
    ** and if so then call setup_raster_fill_index to down load the user pattern
    ** on to the device.          Kran
    */
    if (pDDC->DCState.abnd.usSet)
    {
      usSymbol = PATSYM_USERDEFINED;
      setup_raster_fill_index(hDC,pDDC,NULL,NULL);
      if (pDDC->DCState.abnd.usMixMode == FM_LEAVEALONE)
      {
        usSymbol = PATSYM_SOLID;
        lColor = pDDC->DCState.abnd.lBackColor;
      }
    }

    /*
    ** Select the fill pen
    ** Note: must select fill type after selecting fill pen
    */
    Select_fill_pen(pDDC, lColor);

    /*
    ** Special case stuff for clearing areas with PATSYM_NOSHADE
    ** or PATSYM_BLANK.  Note: On HPGL2 devices we reserve pen
    ** zero for clearing areas.
    */
    usMixMode = pDDC->DCState.abnd.usMixMode;

    /*
    ** PATSYM is clearing the area- select pen zero
    **
    ** Shuold also check if area mix mode is a simple overpaint
    */
    if (pDDC->DCState.abnd.usSymbol == PATSYM_NOSHADE ||
        pDDC->DCState.abnd.usSymbol == PATSYM_BLANK)
    {
      /*
      ** select pen zero which was reserved for white fills
      */
      output_bytes(pDDC, "SP");
      output_number(pDDC, 0L);
      bResetPen = TRUE;
    }

    /*
    ** We do not know the size.
    ** But, we could acumulate the bounds and set the
    ** fill direction.
    ** for now we just set the
    ** bounds to null.
    */
    ptlBottomLeft.x = 0;
    ptlBottomLeft.x = 0;
    ptlTopRight.x = 0;
    ptlTopRight.y = 0;

    /*
    ** set fill type
    */
    if (!pDDC->DCState.abnd.usSet)   // standard pattern
      select_fill_type(pDDC, pDDC->DCState.abnd.usSymbol,
                       &ptlBottomLeft, &ptlTopRight);
    else                             // user pattern       Kran
      select_fill_type(pDDC, usSymbol,
                       &ptlBottomLeft, &ptlTopRight);

    set_line_type(pDDC, LINETYPE_SOLID);


    /*
    ** Force move to the first point. We must guarantee
    ** we start at the first point for polygon
    ** mode. MV.
    **
    ** PM0 and PM1 must have the first point set.
    ** PM0 - first point is set before PM0
    ** PM1 - first point is set after PM1
    */
    move_pen(pDDC, &pDDC->DCState.LogPosition, TRUE);
    ptlStart = pDDC->DCState.LogPosition;

    output_bytes(pDDC, "PM0");

    /*
    ** loop through the polygons
    */
    lPolyIndex = 0L;          /* Loop counters for #polygons  */
    while (lPolyIndex < cPolygons)
    {

      /*
      ** loop through the points for the polygon
      */
      ulPointIndex = 0L;      /* Loop counters for #Points in polygon  */
      while (ulPointIndex < paPolygon->ulPoints)
      {
        if (ulPointIndex == 0L && lPolyIndex > 0L)
        {
          move_pen(pDDC, &paPolygon->aPointl[ulPointIndex], FALSE);
          ptlStart = paPolygon->aPointl[ulPointIndex];
        }
        else
        {
          plot_line(pDDC, &paPolygon->aPointl[ulPointIndex]);
        }

        ulPointIndex++;
      }

      /*
      ** Close the polygon if necessary
      */
      if (ptlStart.x != pDDC->DCState.LogPosition.x &&
          ptlStart.y != pDDC->DCState.LogPosition.y)
      {
         plot_line(pDDC, &ptlStart);
      }

      /*
      ** Close the current polygon and remain in polygon mode
      ** The point specified after PM1 becomes the first point of
      ** the next subpolygon
      */
      output_bytes(pDDC, "PM1");
      pPDevice->bLineInConstruction = FALSE;

      paPolygon++;
      lPolyIndex++;           /* Loop counters for #polygons  */
    } /* endwhile */

    /*
    ** Finish the current polygon
    ** PM2=Exit Polygon Mode
    */
    output_bytes(pDDC, "PM2");
    /*
    ** Tell the plotter to fill the polygon
    ** FP=Fill polygon
    ** Fill polygon (FP) in HPGL only fills the interior.
    */
    output_bytes(pDDC, "FP");

    /*
    ** draw the edge of the polygon if necessary
    */
    if (flOptions & POLYGON_BOUNDARY)
    {
      output_bytes(pDDC, "EP");
    }

    /*
    ** We must turn of bLineInConstruction because continuing with
    ** points after ending a polygon will not work.
    */
    pPDevice->bLineInConstruction = FALSE;

    /*
    ** if pen was set to zero, reset the pen to the original value
    */
    if (bResetPen)
    {
      output_bytes(pDDC, "SP");
      output_number(pDDC,
                    (LONG)pPDevice->iHPGL2PenNumber[pPDevice->CurrentCarousel]
                       [pPDevice->CurrentPen-1]);
    }

    /*
    **  Paintjet HPGL2 gets out of sync with our penup pendown
    **  So make sure we are in sync by sending the command to put the
    **  pen in the position we think its in.
    */
    if (pPDevice->bPenIsUp)
    {
      output_bytes(pDDC, "PU");
    }
    else
    {
      output_bytes(pDDC, "PD");
    }

    /*
    ** Free memory only if we allocated it.
    */
    if (bFreeMem)
    {
      GplMemoryFree ((PVOID)pByte);
    }
  }

  LeaveDriver(pDDC);
  return (lResult);
}
/***************************************************************************
 *
 * FUNCTION NAME = PaintRegion
 *
 * DESCRIPTION   = Paints the specified region using the current area
 *                 foreground and background colors.   Mixing is controlled
 *                 by the area foreground mix only.
 *
 *
 *                 We could handle the PaintRegion by calling
 *                 GreGetRegionRects and filling the rectangles
 *                 ourselves.
 *
 * INPUT         = hDC    - device context handle
 *                 hRgn   - Region Handle
 *                 pDDC   - pointer to device driver context
 *                 FUnN   - function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = GPI_OK    - Success
 * RETURN-ERROR  = GPI_ERROR - Failure
 *
 *   11/28/92                      Added Hook for PaintRegion
 *
 **************************************************************************/

LONG  PaintRegion( HDC   hDC, HRGN  hRgn, PDDC  pDDC, ULONG FunN)
{
  LONG lResult = GPI_ERROR;

  if(!(EnterDriver(pDDC)))
    return(GPI_ERROR);

  /*
  ** let the gre do the bounds accumulation
  */
  if (FunN & COM_BOUND)
    lResult = (*daPaintRegion)(hDC, hRgn, pDDC, FunN & ~COM_DRAW);

  if (check_area_color(pDDC) && (FunN & COM_DRAW))
  {
    HRGN hrgnNew = (HRGN)NULL;
    HRGN hrgnOld;
    HRGN hrgnTemp;

    /*
    ** Check if the user has a clip region selected.
    ** If the previously selected region returned in hrgnOld
    ** is NULL, there was no clip region.
    */
    GreSelectClipRegion(hDC, (HRGN)NULL, &hrgnOld);
    if (hrgnOld)
    {
      /*
      ** Create null region
      */
      if (!(hrgnNew = GreCreateRectRegion(hDC, (PRECTL)NULL, 0L)) )
      {
        LeaveDriver(pDDC);
        return(lResult);
      }

      /*
      ** Combine the users clip region with the region
      ** being painted to get the intersection of the
      ** two regios.
      */
      GreCombineRegion(hDC, hrgnNew, hRgn, hrgnOld, CRGN_AND);

      hRgn = hrgnNew;                 /* use new interseting region */
    }

    /*
    ** Select the region as the clip region,
    ** if it's not a null region, fill it.
    */
    if ( GreSelectClipRegion(hDC, hRgn, &hrgnTemp) != RGN_NULL)
    {
      USHORT usSaveMixMode;
      /*
      ** Fill the clip area.
      ** Use the bounds set in NotifyClipChange by the
      ** GreSelectClipRegion. pDDC->DCState.ClipRect.
      ** Mixing is controlled by the area foreground mix only.
      ** Therefore we set the background mix to the foreground while
      ** filling the region.
      */
      usSaveMixMode = pDDC->DCState.abnd.usBackMixMode;
      pDDC->DCState.abnd.usBackMixMode = pDDC->DCState.abnd.usMixMode;
      fill_clip_area(hDC, &pDDC->DCState.ClipRect, AT_REGION, pDDC);
      pDDC->DCState.abnd.usBackMixMode = usSaveMixMode;

    }
    /*
    ** Unselect the region and put the old region
    ** back before restoring the DC irrespective of whether the region is
    ** empty or not. Kran
    */
    GreSelectClipRegion(hDC, hrgnOld, &hrgnOld);

    /*
    ** if we created a region destroy it
    */
    if (hrgnNew)
    {
      GreDestroyRegion(hDC, hrgnNew);
    }

    lResult = GPI_OK;
  }
  //else
  //{
  //  /*
  //  ** Call the engine to Paint the region
  //  **
  //  ** This will result in a call to bitblt for each
  //  ** region rectangle.
  //  */
  //  lResult = (*daPaintRegion)(hDC, hRgn, pDDC, FunN);
  //}

  LeaveDriver(pDDC);
  return  lResult;
}

/***************************************************************************
 *
 * FUNCTION NAME = FillPath
 *
 * DESCRIPTION   = Fill the path specified by the user.
 *
 * INPUT         = hDC     - device context handle
 *                 PathId  - currently restricted to 1
 *                 flOptions - controls filling
 *                 pDDC    - pointer to device driver context
 *                 FunN    - function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = GPI_OK    - Success
 * RETURN-ERROR  = GPI_ERROR - Failure!
 *
 **************************************************************************/

LONG  FillPath(HDC hDC, LONG PathId, ULONG flOptions, PDDC pDDC, ULONG FunN)
{
  LONG  lResult = GPI_ERROR;
  SHORT sPolyResult;


  if(!(EnterDriver(pDDC)))
    return(GPI_ERROR);

  if (pDDC->DCState.bDoFills &&
      (check_area_color(pDDC) ||
       check_area_background_color(pDDC)) &&
      (FunN & COM_DRAW))
  {
    ULONG hSavedDC = 0L;
    PRECTL prclClipBounds = &pDDC->DCState.ClipRect;

    /*
    ** Try to fill the area with the polygon commands
    ** by outlining the area in polygon mode.
    ** If fill_polygon returns FALSE(area not filled), no problem,
    ** use another filling method. Older plotters have small polygon
    ** buffers this method may not always work
    */
    sPolyResult = fill_polygon(hDC, PathId, flOptions, FunN, pDDC);
    if (sPolyResult == POLY_FILLED)
    {
      lResult = GPI_OK;
    }
    else
    {
      /*
      **  It is for us,  so process.  Basic operation is to save the DC
      **  (because we are going to clobber it),  then select the path as
      **  a clip path, get the bounds from the notify clip change in
      **  PLOTTER units.  Fill the clip area. Restore the DC. Delete
      **  the path.
      */
      if ( (hSavedDC = GreSaveDC(hDC)) &&
         /*
         ** The SaveDC worked,  so onto the real meat of the matter!
         **
         ** Select the path as a clip path and check its bounds.
         ** if its OK fill it.
         */
         ((*daSelectClipPath)(hDC, PathId, (flOptions|SCP_AND), 0L,
                              MAKEULONG(NGreSelectClipPath,
                               HIUSHORT(FunN & ~COM_TRANSFORM)))))
      {

        /*
        ** MarkV
        ** Verify the bounds
        **
        ** The bounds are actually the clip bounds updated in
        ** NotifyClipChange by the previous call (SelectClipPath)
        **
        ** The clip bounds were found to be faster
        ** than the additional step to accumulate the bounds.
        **
        ** The clip bounds yTop and xRight are 1 greater than the
        ** bounds(exclusive on the top and right). Therefore, we
        ** must subtract 1 from the top and right.
        */
        if ( (prclClipBounds->xLeft <= --prclClipBounds->xRight) &&
             (prclClipBounds->yBottom <= --prclClipBounds->yTop) )
        {
          /*
          ** convert clip rect to bounds
          */
          RECTL  Bounds;

          Bounds = *prclClipBounds;
          //Bounds.xRight -= 1;
          //Bounds.yTop   -= 1;

          /*
          ** Fill the selected clip area with the current area pattern
          ** and color.
          **/
          fill_clip_area(hDC, &Bounds, AT_PATH, pDDC);
          lResult = GPI_OK;

        }
        else
        {
          /*
          ** Null area / Null bounds
          ** No need to fill a null area, Just clean up
          ** and get out.
          ** This should not return an error.
          ** **fix for metafile play error ***
          */
          lResult = GPI_OK;
        }

        if (hSavedDC)
            GreRestoreDC(hDC, hSavedDC);

        /*
        ** Delete the path.
        ** We only want to delete the
        ** the specified path, but there is no direct way to
        ** to do that.  no need to calculate bounds as before.
        */
        lResult = (*daFillPath)(hDC, PathId, flOptions, pDDC,
                        MAKEULONG(NGreFillPath,
                        HIUSHORT(FunN & ~(COM_DRAW | COM_BOUND))));
      }                                /* end if savedc & selectclippath OK */
      /*
      ** SaveDC or selectclippath failed. Selectclippath may fail
      ** if the area is too complex to be a clip path. Our only
      ** chance here is call the engine and let the engine fill the
      ** area.
      */
      else
      {
        fill_opt_init(pDDC);
        lResult = (*daFillPath)(hDC, PathId, flOptions, pDDC, FunN);
        fill_opt_end(pDDC);
      }
    }
    lift_pen(pDDC);
  }                                    /* end if do fills                   */
  else
    lResult = (*daFillPath)(hDC, PathId, flOptions, pDDC, FunN);

  LeaveDriver(pDDC);
  return  lResult;
}

