/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
#if INCLUDE_COMMENTS
/******************************************************************************
 *
 * SOURCE FILE NAME = GPLMEM.H
 *
 * DESCRIPTION = Public output memory header for genplib.lib
 *
 *
 * VERSION = V1.0
 * DATE    = 02/18/94
 * AUTHOR  = Lee Tartak
 *
 *
 * FUNCTIONS
 *
 *
 *
 * NOTES
 *
 * The GPLMEM functions provide memory management routines for device driver
 * writers.  The package will create allocate memory directly from the system
 * or from a heap.  The memory can be shared or process private.  The package
 * frees the developer from having to keep track of where the different memory
 * objects came from and how to free them.  Debugging and diagnostic support
 * are also provided.
 *
 * Usage:
 *
 * 1) Initialize the memory system
 * The first function called is GplMemoryCreateInstance which initializes a
 * memory instance.  The user passes the sizes of the primary and secondary
 * heaps.  Allocations come from from the primary heap first.  When memory from
 * the first heap is exhausted the subsystem creates a secondary heap(s).  The
 * idea is one can create a "smaller" heap for general use but have extra for
 * large requests.
 *
 * The third parameter is the threshold value.  Any requests over this value
 * will be routed to DosAllocMem instead of the heap manager.  If you do not
 * want any request going to the DosAlloc then make the threshold the same size
 * as the heap.
 *
 * The fourth parameter is the type of memory - either per process memory
 * spareable within the process or shared memory which can be shared between
 * processes.  Shared memory has more associated overhead than per process
 * memory and should be used only when specifically needed.  The call returns a
 * handle to the Memory Control Block (MCB).  This handle is used for memory
 * request.  A user can create multiple memory instances if needed by calling
 * GplMemoryCreateInstance the desired number of times.  For example one may a
 * DC heap, a overall process heap, and a spareable heap.
 *
 * 2) Allocate the memory
 * The GplMemoryAlloc call has two parameters.  The first is the HMCB returned
 * by the GplMemoryCreateInstance call.  The second is size in bytes of the
 * memory.  If the first parm is NULL the system will allocate from DosAllocMem
 * no matter what the threshold is.  This might be useful if one has just a
 * small amount of memory needed and does not want a heap set up.  The call
 * returns a pointer to the block if succesful, NULL if there is a failure.
 *
 * 3) Free the memory
 * GplMemoryFree will free the memory allocated by GplMemoryAlloc.  The only
 * parm is pointer returned by GplMemoryAlloc.  If you have a problem with
 * memory (ex over writing the ends or changing the pointer) this call will
 * return with an error.  If you are using the kernel debugger and have linked
 * with the debug version of the library you will get some messages if there is
 * a problem.
 *
 * 4) Delete the memory instance
 * Call GplMemoryDeleteInstance with the HMCB returned by
 * GplMemoryCreateInstance to free us the heaps.  If you are using the debug
 * version the system will dump out any unfreed memory blocks.  This indicates
 * unmatched alloc/free calls and could be a memory leak.
 *
 * Debug hints
 * We keep our bookkeeping info in several bytes before data area:
 * +------------+---------------------------------+
 * |control info|user data.............           |
 * +------------+---------------------------------+
 *              |<-- Pointer returned to user
 *
 * In the retail version we can only check 1) the pointer given to us by
 * GplMemoryFree points to the right spot, and 2) the control info has not been
 * corrupted.  In the debug version more tests are added.  Signature LONGs are
 * added to beginning and end to check for over write upon the free.  The
 * signatures are changed slightly so you will be notified upon a double free
 * and can tell if it is in use during a memory dump.
 *
 * +----+------------+------------------------------+----+
 * |INFO|control info|user data.............        |FADE|
 * +----+------------+------------------------------+----+
 *                   |<-- Pointer returned to user
 *
 * Also each entree is stored in a table which the un-freed entries are dumped
 * upon the delete instance.
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/#endif

typedef PVOID HMCB;   /* Define for Handle to Memory Control Block */

/*
** Flags for memory type
** ---------------------
** PROCESS_MEMORY comes from DosAllocMem.  It is shareable within a process.
** This good for most needs.
**
** SHARED_MEMORY comes from SSAllocMem.  It can be shared between processes.
** This should be used with care since it involves more overhead and is in
** limited amounts.
*/
#define PROCESS_MEMORY   1
#define SHARED_MEMORY    2


/*
** Extern interface to Memory functions
*/

/*****************************************************************************\
**
** GplMemoryCreateInstance
** -----------------------
** PARM1 - A ULONG containing the initial size of the heap. If zero you will
**         get 64K heap.
** PARM2 - A ULONG containing the size of the secondary heaps.  If the primary
**         heap fills a secondary heap will be created.  The subsystem will
**         chain the heaps transparently to the user.  Zero creates 64K
**         secondarys.
** PARM3 - A ULONG containing the threshold value.  Any memory requests larger
**         than this are passed to DOS/SSAllocMem bypassing the heap.  Zero
**         sets a 4K threshold.
** PARM4 - A ULONG containing the type of memory (see above).  Zero is Process
**         memory.
** RETURNS - HMCB - Handle to Memory Control Block.  The HMCB is used for
**           allocs and frees.
**           NULL is returned if an error occurs.
**
\*****************************************************************************/

extern HMCB APIENTRY  GplMemoryCreateInstance( ULONG, ULONG, ULONG, ULONG );

/*****************************************************************************\
**
** GplMemoryDeleteInstance
** -----------------------
** PARM1 - HMCB of the Memory instance to delete.
** RETURNS - NULL if ok
**           ERROR - DosCall return values or ERROR_INVALID_PARAMETER
**
\*****************************************************************************/

extern APIRET APIENTRY  GplMemoryDeleteInstance( HMCB );

/*****************************************************************************\
**
** GplMemoryAlloc
** --------------
** PARM1 - HMCB of memory instance to allocate from. If this parm is NULL then
**         memory is allocated from DosAllocMem.
** PARM2 - A ULONG containing the size of the request.
** RETURNS - A pointer to memory if OK, NULL if an error occured.
**
\*****************************************************************************/

extern PVOID APIENTRY GplMemoryAlloc( HMCB, ULONG );


/*****************************************************************************\
**
** GplMemoryFree
** -------------
** PARM1 - Pointer to memory from GplMemoryAlloc to free
** RETURNS - NULL if ok
**           ERROR - DosCall return values or ERROR_INVALID_PARAMETER
**
\*****************************************************************************/

extern APIRET APIENTRY GplMemoryFree( PVOID );

/*****************************************************************************\
**
** GplMemoryGetObjectSize
** ----------------------
** Return size of memory object data
**
** PARM1 - Pointer to memory object
** RETURNS - size of memory
**
\*****************************************************************************/
extern LONG APIENTRY GplMemoryGetObjectSize( PVOID );

