/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                                                                          */
/* FILENAME       : GPLJRNL.H                                               */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 10-07-93                                                */
/* DESCRIPTION    : This module contains all MACROS, IDs, constants and     */
/*                  data type definitions used by the journalling functions */
/*                                                                          */
/***********************************************************************************/
/*                                                                                 */
/* Public functions                                                                */
/* ----------------                                                                */
/*                                                                                 */
/* These functions are used for journalling:                                       */
/*    JournalCreateInstance                                                        */
/*    JournalPlayInstance                                                          */
/*    JournalDeleteInstance                                                        */
/*    JournalCalcBandSize                                                          */
/*                                                                                 */
/* What is journalling?                                                            */
/* --------------------                                                            */
/*                                                                                 */
/* A journal file is a record of all the GreXXX function calls that your driver    */
/* has received between the time that it starts journal file recording and the     */
/* time it stops journal file recording.  Typically, you start recording at the    */
/* beginning of a page and stop recording at the end of a page.  After you have    */
/* stopped recording, you can replay the journal file as many times as you wish.   */
/* There are many reasons why you might want to record the GreXXX calls for a      */
/* particular HDC (Handle to a Drawing Context).  One example is for color         */
/* sorting.  Another is for banding.  For color sorting, you output printer        */
/* commands for a particular color while you record the journal file.  Then,       */
/* you play back the journal file for as many colors as you have left.  You        */
/* only output printer commands when the current color is the color you are        */
/* working with.  For banding, you allow a portion of the page as a bitmap         */
/* and only draw into that area while you play back the journal file.  Then,       */
/* you move the DC origin up the size of the band and replay the journal file.     */
/* You do this for every band on the page.                                         */
/*                                                                                 */
/* Flow control                                                                    */
/* ------------                                                                    */
/*                                                                                 */
/* The first function you would call is JournalCalcBandSize.  This is a optional   */
/* helper function.  Its goal is to tell you whether or not you need to band given */
/* your memory requirements and band size.  The next function you call is          */
/* JournalCreateInstance.  This function initializes an instance or can reuse an   */
/* instance.  Next, you call JournalPlayInstance.  This function will replay       */
/* the journal file either on a per page basis or on a per band basis.  This       */
/* function can also preserve the attributes of the page or change them.  Finally, */
/* you call JournalDeleteInstance to clean up your instance data.                  */
/*                                                                                 */
/* JournalCalcBandSize                                                             */
/* -------------------                                                             */
/* NOTES:  This is a helper function for banding.                                  */
/*                                                                                 */
/* Inputs:                                                                         */
/*    PIJOURNAL   pIJournal;     // Journal input structure                        */
/*                   - usXLength      // Page size in pels                         */
/*                   - usYLength      // Page size in pels                         */
/*    USHORT      usBitsPerPel;  // Page bitmap # bits per pel                     */
/*    USHORT      usNumPlanes;   // Page bitmap # planes                           */
/*    USHORT      usModulus;     // Make sure that the band size is a              */
/*                                  multiple of this.                              */
/*    ULONG       ulMemoryLimit; // Memory limit in bytes                          */
/*                                                                                 */
/* Outputs:                                                                        */
/*    FALSE  - You do not need to journal.                                         */
/*    TRUE   - You need to journal.  The following is filled in and valid.         */
/*                                                                                 */
/*    PIJOURNAL   pIJournal;     // Journal input structure                        */
/*                   - usBandLength   // Band length in pels                       */
/*                                                                                 */
/* ulMemoryLimit is the limit of the number of bytes that your band bitmap cannot  */
/* exceed.  Based on the page size, the number of bits per pel, and the number of  */
/* color planes, it calculates the number of bytes needed for the page bitmap.     */
/* If that amount exceeds the memory limit, it will calculate the number of bands  */
/* to break the page into.  If usModulus is not 0 or 1, then it will make sure     */
/* that the band size is a multiple of usModulus.  This is helpful if you need     */
/* to send the bits in a certain column height.                                    */
/*                                                                                 */
/* JournalCreateInstance                                                           */
/* ---------------------                                                           */
/*                                                                                 */
/* Inputs:                                                                         */
/*    HJOURNAL     hJournal      // Handle to the journal file instance            */
/*    PIJOURNAL    pIJournal     // A pointer to the input structure               */
/*                                                                                 */
/* Outputs: (see below)                                                            */
/*    TRUE   - Function successful                                                 */
/*    FALSE  - Function not successful                                             */
/*                                                                                 */
/* This function will perform different actions based on the input parameters.     */
/*                                                                                 */
/* Case #1)  hJournal  = NULL                                                      */
/*           pIJournal = NULL                                                      */
/*           ulFlags   = 0                                                         */
/*           The function will return the size of a handle (in the number of       */
/*           bytes).  It is the responsibility of the of caller to allocate and    */
/*           free the memory for a journal handle.                                 */
/*                                                                                 */
/* Case #2)  hJournal  = address of newly allocated memory                         */
/*           pIJournal = address of the input structure (has following format)     */
/*              ulSize           - size of this structure.                         */
/*              hModule          - module handle of the calling program.           */
/*              hdc              - current HDC.                                    */
/*              usXLength        - page size in pels.                              */
/*              usYLength        - page size in pels.                              */
/*              bPortrait        - Orientation of page. TRUE  = portrait           */
/*                                                      FALSE = landscape          */
/*              usBandLength     - band height in pels.                            */
/*              pfunBandCallback - pointer to a callback function with the         */
/*                                 following prototype.  This function is only     */
/*                                 called at the end of each band.                 */
/*                                 ( _System * )( PVOID, PRECTL )                  */
/*              pfunArg          - Any pointer.  This will be passed to the first  */
/*                                 parameter of the previous call back function.   */
/*           ulFlags   = create options                                            */
/*                                                                                 */
/*           This function will create an engine journal file and start            */
/*           recording to it.  It also starts the accumulation of bounds.  Call    */
/*           JournalPlayInstance to stop recording and play the journal file.      */
/*                                                                                 */
/* Case #3)  hJournal  = a handle to a journal file instance                       */
/*           pIJournal = address of the input structure (has the previous format)  */
/*           ulFlags   = create options                                            */
/*                                                                                 */
/*           The function performs the same functions as the 2nd case above.       */
/*           However, you can change any of the parameters in the input            */
/*           structure from the previous call to JournalCreateInstance.            */
/*                                                                                 */
/* Case #4)  hJournal  = a handle to a journal file instance                       */
/*           pIJournal = NULL                                                      */
/*           ulFlags   = create options                                            */
/*                                                                                 */
/*           For this case, a new page has been seen and you want to restart       */
/*           the entire process (case 2).  It uses the same parameters as          */
/*           was given to the original call to JournalCreateInstance.              */
/*                                                                                 */
/*    where ulFlags can be                                                         */
/*          CREATE_DRAW_WHILE_RECORDING  - Turn on drawing while the               */
/*                                         journal file is being recorded          */
/*                                                                                 */
/* JournalPlayInstance                                                             */
/* -------------------                                                             */
/*                                                                                 */
/* Inputs:                                                                         */
/*    HJOURNAL    hJournal       // Handle to the journal file instance            */
/*    ULONG       ulFlags        // Flags (see below)                              */
/*                                                                                 */
/* Outputs:                                                                        */
/*    TRUE   - Function successful                                                 */
/*    FALSE  - Function not successful                                             */
/*                                                                                 */
/* This function will perform different actions based on the ulFlags.              */
/*                                                                                 */
/* ulFlags is a bit vector (individual bits have different meanings).  Note        */
/* that the options are broken up into mutually exclusive groups that must         */
/* have one member set!                                                            */
/*                                                                                 */
/* PLAY_JOURNAL_BAND   - Play all bands at this time.  The                         */
/*                       callback function that was defined at                     */
/*                       JournalCreateInstance time will be                        */
/*                       called after each band has been played.                   */
/*  or                                                                             */
/* PLAY_JOURNAL_PAGE   - Plays the journal file once for this page                 */
/*                                                                                 */
/* PLAY_NONDESTRUCTIVE - The DC state will not be altered after                    */
/*  or                   the call has been completed.                              */
/* PLAY_DESTRUCTIVE    - The DC state will now have the final                      */
/*                       attributes of the page.                                   */
/*                                                                                 */
/* PLAY_TOP_TO_BOTTOM  - Bands are played from the maximum Y value                 */
/*  or                   to 0.                                                     */
/* PLAY_BOTTOM_TO_TOP  - Bands are played from 0 to the maximum Y                  */
/*                       value.                                                    */
/*                                                                                 */
/* JournalDeleteInstance                                                           */
/* ---------------------                                                           */
/*                                                                                 */
/* Inputs:                                                                         */
/*    HJOURNAL     hJournal      // Handle to the journal file instance            */
/*                                                                                 */
/* Outputs:                                                                        */
/*    TRUE   - Function successful                                                 */
/*    FALSE  - Function not successful                                             */
/*                                                                                 */
/* This function will take a journal instance, clean up any resources it uses,     */
/* and deletes it.                                                                 */
/*                                                                                 */
/***********************************************************************************/
/*--------------------------------------------------------------------------*/
/* Modification/Change log                                                  */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                                                          */
/****************************************************************************/

#ifndef __journal_h
#define __journal_h

/****************************************************************************/
/* This is the type of the callback function.  The function returns a       */
/* boolean value which indicates success for failure.  The linkage is       */
/* defined to be _System.  It takes two arguments.  The first is what was   */
/* in pfunArg during the call to CreateJournalFunction.  The second is the  */
/* current band rectangle location.  This is done in two steps since C      */
/* compilers can't always handle the entire thing at once.                  */
/****************************************************************************/
typedef BOOL ( _System JFUNC )( PVOID, PRECTL );
typedef JFUNC *PJFUN;

typedef BOOL ( _System OFFSETFUNC )( PVOID, RECTL, POINTL );
typedef OFFSETFUNC *POFFSETFUNC;

/****************************************************************************/
/* This is what is input to the CreateJournalFunction call.                 */
/****************************************************************************/
struct _JournalInput {
   ULONG       ulSize;              // Size of the structure
   // Input parameters...
   HMODULE     hModule;             // Calling module's handle
   HDC         hdc;                 // GRE hdc value
   ULONG       usXLength;           // length along x axis
   ULONG       usYLength;           // length along y axis
   BOOL        bPortrait;           // portrait or landscape
   ULONG       usBandLength;        // length of a band
   PJFUN       pfunBandCallback;    // Callback function
   PVOID       pfunArg;             // ... and its first argument
   POFFSETFUNC pfunOfsetCallback;   // Callback function for page rectangle
                                    // size & DC offset
   PVOID       pfunOCBArg;          // ... and its first argument
};
typedef struct _JournalInput IJOURNAL, *PIJOURNAL;

typedef VOID *HJOURNAL;

/* Flags to GplJournalCreateInstance ...
 */
#define CREATE_DRAW_WHILE_RECORDING 0x80000000L

/* Flags to GplPlayJournalInstance ...
 */
#define PLAY_JOURNAL_BAND      0x0001L      // either this
#define PLAY_JOURNAL_PAGE      0x0002L      // ... or this must be present
#define PLAY_NONDESTRUCTIVE    0x0004L      // either this
#define PLAY_DESTRUCTIVE       0x0008L      // ... or this must be present
#define PLAY_TOP_TO_BOTTOM     0x0010L      // either this
#define PLAY_BOTTOM_TO_TOP     0x0020L      // ... or this must be present

/* Function prototypes ...
 */
APIRET APIENTRY GplJournalCreateInstance ( HJOURNAL  hJournal,
                                           PIJOURNAL pIJournal,
                                           ULONG     ulFlags);
APIRET APIENTRY GplJournalPlayInstance   ( HJOURNAL  hJournal,
                                           ULONG     ulFlags);
APIRET APIENTRY GplJournalDeleteInstance ( HJOURNAL  hJournal);
APIRET APIENTRY GplJournalCalcBandSize   ( PIJOURNAL pIJournal,
                                           USHORT    usBitsPerPel,
                                           USHORT    usNumPlanes,
                                           USHORT    usModulus,
                                           ULONG     ulMemoryLimit);
APIRET APIENTRY GplJournalAbortDoc       ( HJOURNAL hJournal);
APIRET APIENTRY GplJournalResetAbortDoc  ( HJOURNAL hJournal);

#endif
