/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Lexmark Corporation, 1989                                   */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = PRDUPATT
 *
 * DESCRIPTIVE NAME =
 *
 *
 * VERSION
 *
 * DATE
 *
 * DESCRIPTION
 *
 * FUNCTIONS   prdu_GetPatternInfo
 *             prdu_GetDriverPattern
 *             prdu_GetBitmapPattern
 *             prdu_ExpandPattern
 *             prdu_GetGreyScaleIndex
 *             prdu_AllocPatternMemory
 *             prdu_FreePatternMemory
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#define INCL_32                        /* CON3201 */
#define INCL_DOSPROCESS                /* CON3201 */
#define INCL_DOSSEMAPHORES             /* CON3201 */
#define INCL_GPIBITMAPS
#define INCL_GPIPRIMITIVES
#define INCL_WINHEAP
#define INCL_FONTFILEFORMAT
#define INCL_GPILOGCOLORTABLE
#include <os2.h>
#undef INCL_DOSPROCESS                  /* CON3201 */
#undef INCL_DOSSEMAPHORES               /* CON3201 */
#undef INCL_GPILOGCOLORTABLE
#undef INCL_FONTFILEFORMAT
#undef INCL_WINHEAP
#undef INCL_GPIPRIMITIVES
#undef INCL_GPIBITMAPS

#define INCL_DDIMISC
#define INCL_DDIBUNDLES
#include <pmddi.h>
#undef INCL_DDIBUNDLES
#undef INCL_DDIMISC

#define NO_CONSTANT_INCL
#include <prdinclt.h>
#undef NO_CONSTANT_INCL
#undef INCL_32                               /* CON3201 */
#include <prdbcone.h>
#include <prdbtyp1.h>
#include <prddcone.h>
#include <prdbextf.h>
#include <prdccone.h>
#include <prdcextf.h>
#include <prdconse.h>
#include <prdgextf.h>
#include <prduextf.h>
#include <prdupatt.h>

/******************************************************************************/
/*  Now for a few externals...                                                */
/******************************************************************************/
extern PatternInfoType  PatternInfoTable[];
extern USHORT           PatternTabSize;

/******************************************************************************/
/*  FUNCTION:  prdu_GetPatternInfo                                            */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  PBYTE  lpPatData;  pointer to pattern data                                */
/*  lpDCI  DCIData;                                                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function sets up the pattern information into the pattern data       */
/*  structure supplied. Currently used only for fast path PolyScanline, this  */
/*  should become the only way of getting the pattern.                        */
/******************************************************************************/
/*CON3201
USHORT  prdu_GetPatternInfo(DCIData)

lpDCI  DCIData;*/
USHORT prdu_GetPatternInfo(lpDCI DCIData)
{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT        usResult;            /* result of function                  */
    USHORT        usXOrigin;           /* pattern origin                      */
    USHORT        usYOrigin;           /* pattern origin                      */
    USHORT        usWidth;             /* pattern width                       */
    USHORT        usHeight;            /* pattern height                      */
    USHORT        usConvPat0;          /* pattern 0 resultant value           */
    USHORT        usConvPat1;          /* pattern 1 resultant value           */
    PDAREABUNDLE  AreaAts;             /* current area attributes             */
    pPatType      lpPD;                /* pointer to pattern data             */
    pBMListEntry  lpLE;                /* pointer to current bitmap           */

    /**************************************************************************/
    /*  Set up local variables                                                */
    /*                                                                        */
    /*  PD00616 : Removed confusing duplicate code for local variables below. */
    /*  The code apparently was left over from a block copy and appeared just */
    /*  below the call to prdu_memset() and set the same variables to the same*/
    /*  values there.                                                         */
    /**************************************************************************/
    usResult = OK;
    AreaAts  = DCIData->DCICurPtnAts;
    lpPD     = (pPatType)DCIData->lpPD;
    lpLE     = (pBMListEntry)DCIData->DCISelBitmap;

    /**************************************************************************/
    /*  Free any memory already attached to this pattern information          */
    /**************************************************************************/
    if (lpPD->usPatFlags & PAT_MEM_ALLOCATED)
    {
        (VOID)prdg_FreeHeapItem(DCIData, lpPD->usPatMemSize,
                             /* (PUSHORT)&lpPD->lpPatBM); */
                                (PUSHORT)lpPD->lpPatBM);           /* CON3201 */
    }

    /**************************************************************************/
    /*  reset the pattern fields                                              */
    /**************************************************************************/
    prdu_memset((PBYTE)lpPD, 0x00, sizeof(PatType));

    /**************************************************************************/
    /*  If the pattern set is '0' then use our default pattern set.           */
    /**************************************************************************/
    if (AreaAts->adef.defSet == 0)
    {

        /**********************************************************************/
        /*  One of our default patterns                                       */
        /**********************************************************************/
        (VOID)prdu_GetDriverPattern((PBYTE)lpPD, DCIData);
    }
    else
    {

        /**********************************************************************/
        /*  The pattern is not one of our default ones so we must get its     */
        /*  bitmap handle and parameters.                                     */
        /**********************************************************************/
        if (AreaAts->adef.fFlags & CDEF_GENERIC)
        {

            /******************************************************************/
            /*  This is a font.  Don't apply fast paths to this.  This falls  */
            /*  out with a pattern width of zero and therefore does not go    */
            /*  into the fast path code.                                      */
            /******************************************************************/
            goto EXIT_FUNC;
        }
        else
        {

            /******************************************************************/
            /*  This is a bitmap.  This is the easy case.                     */
            /******************************************************************/
            (VOID)prdu_GetBitmapPattern((PBYTE)lpPD, DCIData);
        }
    }

    /**************************************************************************/
    /*  Set up the pattern colors and mixes.                                  */
    /**************************************************************************/
    if (lpPD->usPatFlags & PAT_GREYSCALE)
    {

        /**********************************************************************/
        /*  If we're grey scaling, then the foreground and background colors  */
        /*  are black and white respectively and the pattern just needs       */
        /*  copying and the mix modes must be identical.                      */
        /**********************************************************************/
        lpPD->usPatMode = PAT_COPY_BITS;
        lpPD->usFGMix   = AreaAts->abnd.usMixMode;
        lpPD->usBGMix   = AreaAts->abnd.usMixMode;

        /**********************************************************************/
        /*  Switch from the default to the real mix.                          */
        /**********************************************************************/
        if (lpPD->usFGMix == FM_DEFAULT)
        {
            lpPD->usFGMix = FM_OVERPAINT;
            lpPD->usBGMix = FM_OVERPAINT;
        }
    }
    else
    {

        /**********************************************************************/
        /*  Use current pattern attributes.  Careful about brackets           */
        /**********************************************************************/
        usConvPat0 = prdc_ColorToPelBits(AreaAts->abnd.lBackColor, DCIData,
                                         lpLE->DCTPtr);
        usConvPat1 = prdc_ColorToPelBits(AreaAts->abnd.lColor, DCIData,
                                         lpLE->DCTPtr);

        /**********************************************************************/
        /*  PD00462 : If both foreground and background colors for the pattern*/
        /*  go to black, we'll let the foreground color go to black and we'll */
        /*  take the background to white.                                     */
        /**********************************************************************/
        if ((usConvPat0 == PRD_FOREGROUND) && (usConvPat1 == PRD_FOREGROUND))
            usConvPat0 = PRD_BACKGROUND;

        /**********************************************************************/
        /*  Set up the pattern copy mode and the mix modes.                   */
        /**********************************************************************/
        lpPD->usPatMode = (usConvPat1 << 1) | usConvPat0;
        lpPD->usFGMix   = AreaAts->abnd.usMixMode;
        lpPD->usBGMix   = AreaAts->abnd.usBackMixMode;

        /**********************************************************************/
        /*  Switch from the default                                           */
        /**********************************************************************/
        if (lpPD->usFGMix == FM_DEFAULT)
            lpPD->usFGMix = FM_OVERPAINT;

        if (lpPD->usBGMix == BM_DEFAULT)
            lpPD->usBGMix = BM_LEAVEALONE;
    }

    /**************************************************************************/
    /*  Will only do a fast polyscanline if both mixes are overpaint.  or     */
    /*  foreground is overpaint and background is leavealone.                 */
    /**************************************************************************/
    if ((lpPD->usFGMix == FM_OVERPAINT) &&
        ((lpPD->usBGMix == BM_OVERPAINT) || (lpPD->usBGMix == BM_LEAVEALONE)))
    {
        lpPD->usPatFlags |= PAT_MIX_OK;
    }
    else
    {
        goto EXIT_FUNC;
    }

    /**************************************************************************/
    /*  A width of 8, 16 or 32 may need expanding and or shifting to allow us */
    /*  to use the fast path.                                                 */
    /**************************************************************************/
    usWidth  = lpPD->usPatWidth;
    usHeight = lpPD->usPatHeight;
    if ((usWidth ==  8) || (usWidth == 16) || (usWidth == 32))
    {

        /**********************************************************************/
        /*  Can fastpath this                                                 */
        /**********************************************************************/
        lpPD->usPatFlags |= PAT_WIDTH_OK;

        /**********************************************************************/
        /*  Get the x origin of the pattern adjusting for the DC origin.      */
        /**********************************************************************/
        usXOrigin = DCIData->DCIPattOriDev.X + lpLE->XOrigin;
        usXOrigin %= usWidth;
        usYOrigin = DCIData->DCIPattOriDev.Y + lpLE->XOrigin;
        usYOrigin %= usHeight;

        if ((usXOrigin + usYOrigin) || (usWidth != 32))
        {

            /******************************************************************/
            /*  Copy it across and rotate it around if necessary.             */
            /******************************************************************/
            usResult = prdu_ExpandPattern(DCIData->lpPD, DCIData);
        }
    }

    /**************************************************************************/
    /*  save a pointer to the top row of the pattern.                         */
    /**************************************************************************/
    lpPD->usTopOffset = (lpPD->usPatHeight - 1) * lpPD->usPatBytesPerRow;

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/

EXIT_FUNC:

    return(usResult);
}

/******************************************************************************/
/*  FUNCTION:  prdu_GetDriverPattern                                          */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  PBYTE  lpPatData;  pointer to pattern data                                */
/*  lpDCI  DCIData;                                                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function sets up the pattern information for a driver defined        */
/*  pattern.                                                                  */
/******************************************************************************/
/*CON3201
USHORT prdu_GetDriverPattern(lpPatternPointer, DCIData)

PBYTE  lpPatternPointer;
lpDCI  DCIData;
                */

USHORT prdu_GetDriverPattern(PBYTE  lpPatternPointer,
                             lpDCI  DCIData)
{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    PDAREABUNDLE       AreaAts;        /* current area attributes             */
    USHORT             ii;             /* loop variable                       */
    USHORT             usPatNumber;    /* index into pattern table            */
    USHORT             usPatIndex;     /* into pattern info table             */
    USHORT             usPatType;      /* GetGreyScaleInfo flag               */
    ULONG              ulGSColor;      /* color to greyscale                  */
    BOOL               fGreyScale;     /* true is greyscaling                 */
    pPatType           lpPatData;      /* pointer to pattern data             */
    lpPatternInfoType  lpPatternInfo;
    lpDDTType          pDDT;           /* DDT Pointer                         */

    /**************************************************************************/
    /*  Set up local variables                                                */
    /**************************************************************************/
    lpPatData   = (pPatType)lpPatternPointer;
    pDDT        = &DCIData->DCIPdbInstance->DDT;
    AreaAts     = DCIData->DCICurPtnAts;
    usPatNumber = AreaAts->abnd.usSymbol;

    /**************************************************************************/
    /*  This is one of our patterns                                           */
    /**************************************************************************/
    lpPatData->cPatType = PAT_IS_DRIVERS;

    /**************************************************************************/
    /*  Later on we wish to use usPatNumber as an index into our pattern      */
    /*  table.  Unfortunately the DDI interface does not use contiguous       */
    /*  identifiers.  So correct for that here.  Also, if the pattern fill is */
    /*  all ones or all zeroes then store that information for later use.     */
    /**************************************************************************/
    switch (usPatNumber)
    {
        case PATSYM_DEFAULT  :
            usPatNumber = PATSYM_SOLID;
            lpPatData->usPatFlags |= PAT_SOLID;
            break;
        case PATSYM_BLANK    :
            usPatNumber = PATSYM_NOSHADE;
            lpPatData->usPatFlags |= PAT_BLANK;
            break;
        case PATSYM_NOSHADE  :
            usPatNumber = PATSYM_NOSHADE;
            lpPatData->usPatFlags |= PAT_BLANK;
            break;
        case PATSYM_SOLID    :
            usPatNumber = PATSYM_SOLID;
            lpPatData->usPatFlags |= PAT_SOLID;
            break;
        case PATSYM_HALFTONE :         /* PD00599...                          */
            usPatNumber     = PATSYM_DENSE4;
            break;
        default              :
            if (usPatNumber > PatternTabSize)
            {
                usPatNumber = PATSYM_SOLID;
                lpPatData->usPatFlags |= PAT_SOLID;
            }
    }

    /**************************************************************************/
    /*  PD00462 : Have to use background color for PATSYM_NOSHADE.  Moved the */
    /*  color assignment to here so we can test for RGB_BLACK - code was in   */
    /*  the following fGreyScale block just before the call to get the        */
    /*  greyscale index.                                                      */
    /**************************************************************************/
    if (usPatNumber == PATSYM_NOSHADE)
        ulGSColor = AreaAts->abnd.lBackColor;
    else
        ulGSColor = AreaAts->abnd.lColor;

    /**************************************************************************/
    /*  PD00462 : We'll greyscale if we meet the conditions specified in the  */
    /*  prdb_GetGreyScaleInfo() function, the color to be greyscaled is not   */
    /*  RGB_BLACK and the pattern number is PATSYM_SOLID or PATSYM_NOSHADE.   */
    /*                                                                        */
    /*  GREY:  Changed call to prdb_GetGreyScaleInfo()                        */
    /**************************************************************************/
    fGreyScale = ((usPatNumber == PATSYM_SOLID) ||
                  (usPatNumber == PATSYM_NOSHADE)) &&
                 (prdb_GetGreyScaleInfo(ulGSColor, &usPatType,
                                        &(lpPatData->abGreyScale), DCIData));

    /**************************************************************************/
    /*  Different set up for greyscaling                                      */
    /**************************************************************************/
    if (fGreyScale)
    {

        /**********************************************************************/
        /*  When we have finished we will return the greyscale flag.  Switch  */
        /*  off the pattern description flags.  The correct one is set on     */
        /*  below.                                                            */
        /**********************************************************************/
        lpPatData->usPatFlags &= ~PAT_SOLID;
        lpPatData->usPatFlags |= PAT_GREYSCALE;

        /**********************************************************************/
        /*  Mix modes should be the same                                      */
        /**********************************************************************/
        lpPatData->usBGMix = lpPatData->usFGMix;

        /**********************************************************************/
        /*  Save whether this finally ends up as a solid pattern or a blank   */
        /*  pattern.  Remember, the greyscale pattern is inverted later on.   */
        /*                                                                    */
        /*  PD00447 : Changed the following if to check for the value 0x100   */
        /*  instead of 32 since the final index returned as the usPatIndex    */
        /*  is the actual index multiplied by 8, which gives you the actual   */
        /*  memory offset to the beginning of the pattern.  Since that is     */
        /*  not what we're working with here, we need to change it so that    */
        /*  the fastpath for the BLANK case works.                            */
        /*                                                                    */
        /*  PD00462 : Changed the following again - now that we're calling    */
        /*  prdb_GetGreyScaleInfo(), we don't have usPatIndex.  However, we do*/
        /*  have the usPatType flags from that function which serves the same */
        /*  basic purpose here.                                               */
        /**********************************************************************/
        if (usPatType == PAT_ALL_ZEROS)
        {
            lpPatData->usPatFlags |= PAT_BLANK;
        }
        else
        {
            if (usPatType == PAT_ALL_ONES)
            {
                lpPatData->usPatFlags |= PAT_SOLID;
            }
        }

        /**********************************************************************/
        /*  Now the rest of the fields.                                       */
        /**********************************************************************/
        lpPatData->lpPatBM          = lpPatData->abGreyScale;
        lpPatData->usPatWidth       = GS_PATTERN_WIDTH;
        lpPatData->usPatHeight      = GS_PATTERN_HEIGHT;
        lpPatData->usPatBytesPerRow = GS_PATTERN_BYTES_PER_ROW;
    }
    else
    {

        /**********************************************************************/
        /*  Default patterns differ depending on orientation                  */
        /**********************************************************************/
        if (DCIData->DCIPdbInstance->Orientation == PORTRAIT)
        {
            usPatIndex = pDDT->DDTRasterMode->PortPattIndex;
        }
        else
        {
            usPatIndex = pDDT->DDTRasterMode->LandPattIndex;
        }

        /**********************************************************************/
        /*  This gives us the pattern structure                               */
        /**********************************************************************/
        lpPatternInfo = &PatternInfoTable[usPatIndex];

        /**********************************************************************/
        /*  Pattern number enumerates from zero in our table but one in the   */
        /*  DDI interface.  So correct for that.                              */
        /**********************************************************************/
        usPatNumber--;

        /**********************************************************************/
        /*  Fill in the pattern dimensions                                    */
        /**********************************************************************/
        lpPatData->lpPatBM          = lpPatternInfo->PatternTable[usPatNumber];
        lpPatData->usPatWidth       = LOUSHORT(lpPatternInfo->PatternWidth);
        lpPatData->usPatHeight      = LOUSHORT(lpPatternInfo->PatternHeight);
        lpPatData->usPatBytesPerRow = LOUSHORT(lpPatternInfo->PatternBytesPerRow);
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return(OK);
}

/******************************************************************************/
/*  FUNCTION:  prds_GetBitmapPattern                                          */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  PBYTE  lpPatData;  pointer to pattern data                                */
/*  lpDCI  DCIData;                                                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function sets up the pattern information for a bitmap defined        */
/*  pattern.                                                                  */
/******************************************************************************/
/* CON3201
USHORT prdu_GetBitmapPattern(lpPatternPointer, DCIData)

PBYTE  lpPatternPointer;
lpDCI  DCIData;
               */

USHORT prdu_GetBitmapPattern(PBYTE  lpPatternPointer,
                             lpDCI  DCIData)

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    pPatType      lpPatData;           /* pointer to pattern data             */
    pBMListEntry  lpListEntry;         /* pointer to bitmap list entry        */

    /**************************************************************************/
    /*  Set up local variables                                                */
    /**************************************************************************/
    lpPatData   = (pPatType)lpPatternPointer;
    lpListEntry = (pBMListEntry)DCIData->DCICurPtnAts->adef.defSet;
    lpPatData->lpPatBM          = lpListEntry->Bitmap;
    lpPatData->usPatHeight      = LOUSHORT(lpListEntry->Parms.Height);
    lpPatData->usPatWidth       = LOUSHORT(lpListEntry->Parms.Width);
    lpPatData->usPatBytesPerRow = LOUSHORT(lpListEntry->Parms.BytesPerRow);

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return(OK);
}

/******************************************************************************/
/*  FUNCTION:  prdu_ExpandPattern                                             */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  PBYTE  lpPattern;  pointer to new pattern memory                          */
/*  PBYTE  lpPatData;  pointer to pattern data                                */
/*  lpDCI  DCIData;                                                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function copies, rotates and expands the pattern in the pattern      */
/*  data putting the result in lpPattern.                                     */
/******************************************************************************/
/*CON3201
USHORT prdu_ExpandPattern(lpPatternPointer, DCIData)

PBYTE  lpPatternPointer;
lpDCI  DCIData;
               */

USHORT prdu_ExpandPattern(PBYTE  lpPatternPointer,
                          lpDCI  DCIData)

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT        usResult;            /* result of function                  */
    USHORT        usXOrigin;           /* x origin of pattern                 */
    USHORT        usYOrigin;           /* y origin of pattern                 */
    USHORT        usWidth;             /* pattern width                       */
    USHORT        usHeight;            /* pattern height                      */
    USHORT        ii;                  /* loop variable                       */
    pPatType      lpPatData;           /* pointer to pattern data             */
    pBMListEntry  lpListEntry;         /* pointer to current bitmap           */
    PBYTE         lpBuffer;            /* new pattern memory                  */
    PBYTE         lpSrc;               /* ... place to copy from              */
    PBYTE         lpDst;               /* ... place to copy to                */
    PBYTE         lpFirst;             /* first pattern row                   */
    PBYTE         lpLast;              /* last pattern row                    */

    /**************************************************************************/
    /*  Some useful variables                                                 */
    /**************************************************************************/
    lpListEntry = (pBMListEntry) DCIData->DCISelBitmap;
    usResult    = OK;

    /**************************************************************************/
    /*  Other useful local variables                                          */
    /**************************************************************************/
    lpPatData = (pPatType)lpPatternPointer;
    usWidth   = lpPatData->usPatWidth;
    usHeight  = lpPatData->usPatHeight;
    lpFirst   = lpPatData->lpPatBM;
    lpLast    = lpPatData->lpPatBM + (usHeight - 1) *
                lpPatData->usPatBytesPerRow;

    /**************************************************************************/
    /*  Get the x origin of the pattern adjusting for the DC origin.          */
    /**************************************************************************/
    usXOrigin  = DCIData->DCIPattOriDev.X + lpListEntry->XOrigin;
    usXOrigin %= usWidth;

    /**************************************************************************/
    /*  If origin was zero then it should stay zero, otherwise the origin     */
    /*  needs subtracting from the width.                                     */
    /**************************************************************************/
    if (usXOrigin)
    {
        usXOrigin  = usWidth - usXOrigin;
    }

    /**************************************************************************/
    /*  Get the y origin of the pattern adjusting for the DC origin           */
    /**************************************************************************/
    usYOrigin  = DCIData->DCIPattOriDev.Y + lpListEntry->YOrigin;
    usYOrigin %= usHeight;

    /**************************************************************************/
    /*  If origin was zero then it should stay zero, otherwise the origin     */
    /*  needs subtracting from the width.                                     */
    /**************************************************************************/
    if (usYOrigin)
    {
        usYOrigin  = usHeight - usYOrigin;
    }

    /**************************************************************************/
    /*  Set up the size                                                       */
    /**************************************************************************/
    lpPatData->usPatMemSize = 4 * usHeight;
    if (prdg_AllocHeapItem(DCIData, lpPatData->usPatMemSize,
                           (PUSHORT *)&lpBuffer) != OK)
    {
        usResult = ERROR_ZERO;
        goto EXIT_FUNC;
    }

    /**************************************************************************/
    /*  Get a local pointer to it                                             */
    /**************************************************************************/
    lpDst = lpBuffer;
    lpSrc = lpFirst + usYOrigin * lpPatData->usPatBytesPerRow;

    /**************************************************************************/
    /*  Do clever rotation and expansion.                                     */
    /**************************************************************************/
    for (ii = 0; ii < usHeight; ii++)
    {

        /**********************************************************************/
        /*  Get the pattern bits                                              */
        /**********************************************************************/
        prdu_bitcopy((PBYTE)lpDst, lpSrc, 4, usXOrigin, usWidth);
        if (lpSrc == lpLast)
        {
            lpSrc = lpFirst;
        }
        else
        {
            lpSrc += lpPatData->usPatBytesPerRow;
        }
        lpDst += 4;
    }

    /**************************************************************************/
    /*  Done the rotation, store this in the pattern memory field now         */
    /**************************************************************************/
    lpPatData->lpPatBM           = lpBuffer;
    lpPatData->usPatWidth        = 32;
    lpPatData->usPatBytesPerRow  = 4;
    lpPatData->usPatFlags       |= PAT_MEM_ALLOCATED;

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/

EXIT_FUNC:

    return(usResult);
}

/******************************************************************************/
/*  FUNCTION:  prdu_GetGreyScaleIndex                                         */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  ULONG  Color;                                                             */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function sets returns the index into the greyscale pattern table     */
/*  for the pattern that corresponds to the color passed in.                  */
/******************************************************************************/
/*CON3201
USHORT prdu_GetGreyScaleIndex(ulRGBColor, Resolution)

ULONG   ulRGBColor;
USHORT  Resolution;
                    */
USHORT prdu_GetGreyScaleIndex(ULONG   ulRGBColor,
                              USHORT  Resolution)

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT  Grey;
    USHORT  Red;
    USHORT  Green;
    USHORT  Blue;

    /**************************************************************************/
    /*  Split it into it's Red, Green and Blue components.                    */
    /**************************************************************************/
    Red   = (USHORT)((ulRGBColor >> 16) & 0xff);
    Green = (USHORT)((ulRGBColor >> 8) & 0xff);
    Blue  = (USHORT)(ulRGBColor & 0xff);

    /**************************************************************************/
    /*  We're going to convert the RGB color to a grey scale value, which will*/
    /*  be in the range of 0 to 255.  The Red, Green and Blue values that make*/
    /*  up the RGB value will be weighted to convert to an appropriate grey   */
    /*  scale value as follows:                                               */
    /*                                                                        */
    /*    Red    0.30                                                         */
    /*    Green  0.59                                                         */
    /*    Blue   0.11                                                         */
    /*                                                                        */
    /*  PD00462 : Slight tweak to the greyscale pattern index algorithm to    */
    /*  closely approximate the HP's indices.  Note that the new weighted     */
    /*  values are as follows:                                                */
    /*                                                                        */
    /*    Red    0.27                                                         */
    /*    Green  0.59                                                         */
    /*    Blue   0.14                                                         */
    /**************************************************************************/
    Grey = ((Red * 27) + (Green * 59) + (Blue * 14)) / 100;

    /**************************************************************************/
    /*  PD00462 : Tweak Grey here... add 3 to ensure that we get one and only */
    /*  one value that maps to solid black.  Note that the offset we are      */
    /*  factoring in slowly goes away as the grey value increases in value.   */
    /*  This is to avoid having one pattern selected for 15 or so colors as   */
    /*  well as running out of patterns prematurely (again, closely           */
    /*  approximates the HP's mechanism, though not exactly - empirical       */
    /*  analysis only).                                                       */
    /**************************************************************************/
    Grey += (3 - (Grey / NUM_GREY_PATTERNS));

    /**********************************************************************/
    /*  Now, use the value in Grey (in the range of 0 to 255) to derive a */
    /*  value in the range of 0 to 64.                                    */
    /**********************************************************************/
    Grey = NUM_GREY_PATTERNS - (Grey * NUM_GREY_PATTERNS / 255);

    /**************************************************************************/
    /*  And finished.  Multiply by eight to get the index into the table (each*/
    /*  pattern is eight bytes).                                              */
    /**************************************************************************/
    return(Grey * 8);
}

/******************************************************************************/
/*  FUNCTION:  prdu_AllocPatternMemory                                        */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpDCI  DCIData;                                                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function allocates pattern memory for the device                     */
/*  context.                                                                  */
/******************************************************************************/
/*CON3201
USHORT prdu_AllocPatternMemory(DCIData)

lpDCI  DCIData;     */
USHORT prdu_AllocPatternMemory(lpDCI DCIData)
{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT    usResult;                /* Result of function                  */
    USHORT    usSize;                  /* size of pattern data                */
    pPatType  lpPD;                    /* pointer to the scan line info       */

    /**************************************************************************/
    /*  Set up local variables first of all.                                  */
    /**************************************************************************/
    usResult = OK;

    /**************************************************************************/
    /*  Allocate memory for the scan pattern data.                            */
    /**************************************************************************/
    if (prdg_AllocHeapItem(DCIData, sizeof(PatType), (PUSHORT *)&lpPD) != OK)
    {
        usResult = ERROR_ZERO;
        goto EXIT_FUNC;
    }

    /**************************************************************************/
    /*  Store this in the DCIData                                             */
    /**************************************************************************/
    DCIData->lpPD       = (PBYTE)lpPD;
    DCIData->usPatValid = FALSE;

    /**************************************************************************/
    /*  Reset this memory                                                     */
    /**************************************************************************/
    prdu_memset((PBYTE)lpPD, 0x00, sizeof(PatType));

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/

EXIT_FUNC:

    return(usResult);
}

/******************************************************************************/
/*  FUNCTION:  prdu_FreePatternMemory                                         */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpDCI  DCIData;                                                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function allocates polyscanline memory for the device                */
/*  context.                                                                  */
/******************************************************************************/
/*CON3201
USHORT prdu_FreePatternMemory(DCIData)

lpDCI  DCIData;  */
USHORT prdu_FreePatternMemory(lpDCI DCIData)
{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT    usResult;                /* Result of function                  */
    pPatType  lpPD;                    /* pointer to the scan line info       */

    /**************************************************************************/
    /*  Set up local variables first of all.                                  */
    /**************************************************************************/
    usResult = OK;

    /**************************************************************************/
    /*  Only free up memory if there is some there.                           */
    /**************************************************************************/
    if (DCIData->lpPD)
    {

        /**********************************************************************/
        /*  Remember that the DCIData field is simply a PBYTE so need to      */
        /*  essentially cast the value.  Also reset the pointer in the        */
        /*  DCIData.                                                          */
        /**********************************************************************/
        lpPD = (pPatType)DCIData->lpPD;

        /**********************************************************************/
        /*  Reset the pointer and the valid flag.                             */
        /**********************************************************************/
        DCIData->lpPD       = FNULL;
        DCIData->usPatValid = FALSE;

        /**********************************************************************/
        /*  Free any memory already attached to this pattern information      */
        /**********************************************************************/
        if (lpPD->usPatFlags & PAT_MEM_ALLOCATED)
        {
            (VOID)prdg_FreeHeapItem(DCIData, lpPD->usPatMemSize,
                                 /* (PUSHORT)&lpPD->lpPatBM); */
                                    (PUSHORT)lpPD->lpPatBM);       /* CON3201 */
        }

        /**********************************************************************/
        /*  Free the pattern memory                                           */
        /**********************************************************************/
        (VOID)prdg_FreeHeapItem(DCIData, sizeof(PatType),
                             /* (PUSHORT)&lpPD); */
                                (PUSHORT)lpPD);                    /* CON3201 */
    }

    /**************************************************************************/
    /*  and finished                                                          */
    /**************************************************************************/
    return(usResult);
}
