/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Lexmark Corporation, 1989                                   */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = PRDBBLT
 *
 * DESCRIPTIVE NAME =
 *
 *
 * VERSION
 *
 * DATE
 *
 * DESCRIPTION
 *
 * FUNCTIONS   prdb_GetByteOffset
 *             prdb_GetBitOffset
 *             prdb_GetDwordBitOffset
 *             prdb_StartPosAndDir
 *             prdb_GetMasks
 *             prdb_GetIndices
 *             prdb_CachePattern
 *             prdb_GetPattern
 *             prdb_Pat11BitAligned
 *             prdb_Pat41BitAligned
 *             prdb_GetPatternRow
 *             prdb_GetSourceRow
 *             prdb_PerformRop
 *             prdb_RopOneDword
 *             prdb_RopTwoDwords
 *             prdb_RopManyDwords
 *             prdb_RopDwords
 *             prdb_DoQuickRop
 *             prdb_DoQuickBlt
 *             prdb_QuickBltPossible
 *             prdb_DoBitBlt
 *             prdb_BitBlt
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#define INCL_32                         /* Convert to C/SET2    CON3201       */
#define INCL_DOSPROCESS                 /* Convert to C/SET2    CON3201       */
#define INCL_DOSSEMAPHORES
#define INCL_GPIPRIMITIVES
#define INCL_WINHEAP
#define INCL_GPIBITMAPS
#define INCL_GPIERRORS
#define INCL_GPILOGCOLORTABLE
#include <os2.h>
#undef INCL_DOSPROCESS                  /* Convert to C/SET2    CON3201       */
#undef INCL_DOSSEMAPHORES
#undef INCL_GPIPRIMITIVES
#undef INCL_WINHEAP
#undef INCL_GPIBITMAPS
#undef INCL_GPIERRORS
#undef INCL_GPILOGCOLORTABLE

#define INCL_DDIBUNDLES
#define INCL_DDIFONTSTRUCS
#define INCL_DDIDEFS
#define INCL_DDIMISC
#define INCL_DDICOMFLAGS
#include <pmddi.h>
#undef INCL_DDIBUNDLES
#undef INCL_DDIFONTSTRUCS
#undef INCL_DDIDEFS
#undef INCL_DDIMISC
#undef INCL_DDICOMFLAGS

#define INCL_WINP_SELECTIVE
#define INCL_WINP_SEI
#include <pmwinx.h>
#undef INCL_WINP_SELECTIVE
#undef INCL_WINP_SEI
#undef INCL_32                          /* Convert to C/SET2    CON3201       */

#include <prdbcone.h>
#include <prdccone.h>
#include <prdconse.h>
#include <prddcone.h>
#include <prdncone.h>
#include <prdtcone.h>

#define BMCON_INCL
#define COCON_INCL
#define NO_CONSTANT_INCL
#include <prdinclt.h>
#undef NO_CONSTANT_INCL
#undef BMCON_INCL
#undef COCON_INCL

#include <prdbtyp1.h>
#include <prdbextf.h>
#include <prdbext1.h>
#include <prdbsize.h>
#include <prdcextf.h>
#include <prdgextf.h>
#include <prdiextf.h>
#include <prdmacro.h>
#include <prdncone.h>
#include <prdnextf.h>
#include <prdtcone.h>
#include <prdtextf.h>
#include <prduextf.h>
#include <prdyextf.h>

/******************************************************************************/
/*  Set up access to external data                                            */
/******************************************************************************/
/*extern USHORT           prdd_HugeInc;   CON3201 */
/*extern HHEAP            HeapHandle;     CON3201 */
extern PBYTE            pbGlobalHeap;     /* Global Heap CON3201 */
extern USHORT           RopTable[];
extern ParseStringType  ParseTable[];
extern PatternInfoType  PatternInfoTable[];
extern USHORT           PatternTabSize;
extern PFNL             da_BitBlt;
/*extern PFSRSEM          pBMLockSem;      CON3201 */
extern HMTX             hmtxBMLockSem;
/*extern PFSRSEM          lpGlobalHeapSem; CON3201 */
extern HMTX             hmtxGlobalHeapSem;

/******************************************************************************/
/*  Construction of rops.                                                     */
/*                                                                            */
/*  The rops used are constructed by accessing tables to produce a string of  */
/*  operations (binary, unary and 'stack' operations) which when applied to   */
/*  the destination (and possibly the source and pattern if these are         */
/*  specified) will result in the correct output for the specified rop.  The  */
/*  rop number gives an index to a table (in prdbdata.c) which converts it to */
/*  a 16 bit number from which the sequence of operations can be determined.  */
/*  The format of this key is as follows:                                     */
/*                                                                            */
/*   ii ii ii ii ii p sss oo - 16 bit number                                  */
/*   |  |  |  |  |  | |   |                                                   */
/*   |  |  |  | 1st | |   The amount to rotate left the string from           */
/*   |  |  | 2nd |  | |   ParseTable.                                         */
/*   |  | 3rd |  |  | The string number from the ParseTable to use.           */
/*   | 4th |  |  |  Parity bit determines whether to invert the               */
/*  5th |  |  |  |  result.                                                   */
/*   |  |  |  |  |                                                            */
/*   |~~~~~~~~~~~~                                                            */
/*                                                                            */
/*  Instruction codes to apply to the string                                  */
/*                                                                            */
/*  The rop number corresponds to a mapping from the pattern, source and      */
/*  destination to the resulting value written to the destination (See "OS/2  */
/*  Technical Reference: I/O Subsytems and Device Drivers" description of     */
/*  GreBitBlt).  So, to take a couple of examples...                          */
/*                                                                            */
/*  Rop = A7.                                                                 */
/*                                                                            */
/*  Pattern  Source  Destination  Result                                      */
/*    0        0       0            1                                         */
/*    0        0       1            1                                         */
/*    0        1       0            1                                         */
/*    0        1       1            0   A7 = 10100111                         */
/*    1        0       0            0                                         */
/*    1        0       1            1                                         */
/*    1        1       0            0                                         */
/*    1        1       1            1                                         */
/*                                                                            */
/*  This can be represented in boolean logic as..                             */
/*                                                                            */
/*  NOT (((Pattern OR Source) AND Destination) XOR Pattern)                   */
/*                                                                            */
/*  and in reverse polish notation (using C operators) as..                   */
/*                                                                            */
/*  P D S P | & ^ !  ( ie !(((P | S) & D) ^ P) )                              */
/*                                                                            */
/*  so this is encoded as..                                                   */
/*                                                                            */
/*  (07A5) 00 00 01 11 10 1 001 01 which means...                             */
/*                                                                            */
/*  Use string 1 ( SPDSPDSP )                                                 */
/*                                                                            */
/*  rotate it by 1 ( to give PDSPDSPS )                                       */
/*                                                                            */
/*  and apply the operations OR (10), AND (11) and XOR (01) to the string.    */
/*  The two trailing NOTs (00) are ignored as they cancel each other out and  */
/*  the NOT is applied because the parity bit is set.  The start point in the */
/*  string is determined by the number of binary operations (3 in this case) +*/
/*  1 so we start from the 4th entry in the string (P) and work backwards so  */
/*  we do a P OR S then AND D then finally XOR P then NOT the result.         */
/*                                                                            */
/*  Rop = 2B.                                                                 */
/*                                                                            */
/*  Pattern  Source  Destination  Result                                      */
/*    0        0       0            1                                         */
/*    0        0       1            1                                         */
/*    0        1       0            0                                         */
/*    0        1       1            1   A7 = 00101011                         */
/*    1        0       0            0                                         */
/*    1        0       1            1                                         */
/*    1        1       0            0                                         */
/*    1        1       1            0                                         */
/*                                                                            */
/*  This can be represented in boolean logic as..                             */
/*                                                                            */
/*  NOT (((Pattern XOR Destination) AND (Source XOR Pattern)) XOR Source)     */
/*                                                                            */
/*  and in reverse polish notation (using C operators) as..                   */
/*                                                                            */
/*  S S P ^ P D ^ & ^ !  (ie !(((P ^ D) & (S ^ P)) ^ S) )                     */
/*                                                                            */
/*  so this is encoded as..                                                   */
/*                                                                            */
/*  (1D58) 00 01 11 01 01 0 110 00 which means..                              */
/*                                                                            */
/*  Use string 6 (S-SP+PDS where + is a 'push' and - is a 'pop')              */
/*                                                                            */
/*  don't rotate it                                                           */
/*                                                                            */
/*  and apply the operations XOR (01), XOR, AND (11), XOR and NOT (00) to it. */
/*  The start point is the number of binary operations (4) plus 1, plus 2 for */
/*  the push and the pop.  So we start from the 7th entry (D) so we do a D XOR*/
/*  P, then 'push' the result and do a S XOR P, then we 'pop' the result of   */
/*  the D XOR P and AND this with the result of the S XOR P, then XOR this    */
/*  with the S and NOT the result.  The parity bit is reset so we don't NOT   */
/*  the result again.                                                         */
/******************************************************************************/

/******************************************************************************/
/*  FUNCTION: prdb_GetFastFlags                                               */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;                                                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function is called once for each line.  It sets up flags to indicate */
/*  that certain fast paths can be taken in the code.                         */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
USHORT prdb_GetFastFlags(lpBltParms BB)

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT  usFastFlags = 0;

    /**************************************************************************/
    /*  First check if we will go over a segment boundary while processing    */
    /*  this target row.                                                      */
    /**************************************************************************/

/* CON3201 ********************************************************************/
/* Since we are not using segmented memory we can go ahead and set the        */
/* fast flags.  But we do need to check for NO_SOURCE.                        */
/* This should work since the call to prdb_GetSourceRow is protected from     */
/* being called when there is no Source                                       */
/*  if (HIUSHORT(BB->TrgIndex) == HIUSHORT(BB->TrgEndIndex))                  */
/*  {                                                                         */
/*      usFastFlags |= TRG_ROW_IN_SEG;                                        */
/*  }                                                                         */
/*  if (BB->Source != NO_SOURCE)                                              */
/*  {                                                                         */
/*      if (HIUSHORT(BB->SrcIndex) == HIUSHORT(BB->SrcEndIndex))              */
/*      {                                                                     */
/*          usFastFlags |= SRC_ROW_IN_SEG;                                    */
/*      }                                                                     */
/*  }                                                                         */
/******************************************************************************/

    usFastFlags |= TRG_ROW_IN_SEG;            /*  CON3201 */
                                              /*  CON3201 */
    if (BB->Source != NO_SOURCE)              /*  CON3201 */
    {                                         /*  CON3201 */
        usFastFlags |= SRC_ROW_IN_SEG;        /*  CON3201 */
    }                                         /*  CON3201 */

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return(usFastFlags);
}

/******************************************************************************/
/*  FUNCTION: prdb_GetByteOffset                                              */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  pBMListEntry  ListEntry;  Pointer to list entry                           */
/*  USHORT        XCoord;     X Coord within bitmap                           */
/*  USHORT        YCoord;     Y Coord within bitmap                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function returns the byte offset of the DWORD containing the pel     */
/*  described by the X and Y coord.  Thus the result is always a multiple of  */
/*  four.                                                                     */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
ULONG prdb_GetByteOffset( pBMListEntry  ListEntry,
                          USHORT        usXCoord,
                          USHORT        usYCoord )

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    ULONG  ulByteOffset;

    /**************************************************************************/
    /*  The XCoord and YCoord must be adjusted to take account of the current */
    /*  band in the bitmap's associated DC.                                   */
    /**************************************************************************/
    usYCoord += ListEntry->YOrigin;
    usXCoord += ListEntry->XOrigin;

    /**************************************************************************/
    /*  Determine the bit offset for (1,1) and (4,1) bitmaps.                 */
    /**************************************************************************/
    if (ListEntry->Parms.Bitcount == 1)
    {

        /**********************************************************************/
        /*  First get the number of bytes to the start of this row.  Then get */
        /*  the number of bytes on this row (add 7 then divide by 8)          */
        /**********************************************************************/
        ulByteOffset  = ListEntry->Parms.BytesPerRow * usYCoord;
        ulByteOffset += usXCoord >> 3;
    }
    else
    {

        /**********************************************************************/
        /*  Almost the same, except when we are processing the xCoord include */
        /*  a factor of 4 to get the number of bits in the row which          */
        /*  represents XCoord pels.                                           */
        /**********************************************************************/
        ulByteOffset  = ListEntry->Parms.BytesPerRow * usYCoord;
        ulByteOffset += (usXCoord << 2) >> 3;
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return(ulByteOffset);
}

/******************************************************************************/
/*  FUNCTION: prdb_GetBitOffset                                               */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  pBMListEntry  ListEntry;  Pointer to list entry                           */
/*  USHORT        XCoord;     X Coord within bitmap                           */
/*  USHORT        YCoord;     Y Coord within bitmap                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function returns the bit offset of the pel described by the X and Y  */
/*  coord.  The bit offset is an offset within the DWORD conatining that pel, */
/*  thus is in the range zero to 31.                                          */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
SHORT prdb_GetBitOffset( pBMListEntry  ListEntry,
                         USHORT        usXCoord )

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    SHORT  sBitOffset;

    /**************************************************************************/
    /*  The XCoord and YCoord must be adjusted to take account of the current */
    /*  band in the bitmap's associated DC.                                   */
    /**************************************************************************/
    usXCoord += ListEntry->XOrigin;

    /**************************************************************************/
    /*  Determine the bit offset for (1,1) and (4,1) bitmaps.                 */
    /**************************************************************************/
    if ( ListEntry->Parms.Bitcount == 1)
    {

        /**********************************************************************/
        /*  Number of bits is given by the top three bits                     */
        /**********************************************************************/
        sBitOffset = usXCoord & 7;
    }
    else
    {

        /**********************************************************************/
        /*  Number of bits is given by the top three bits of the transformed  */
        /*  Xcoord.                                                           */
        /**********************************************************************/
        sBitOffset = ((usXCoord << 2) & 7);
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return(sBitOffset);
}

/******************************************************************************/
/*  FUNCTION: prdb_GetDwordBitOffset                                          */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  pBMListEntry  ListEntry;  Pointer to list entry                           */
/*  USHORT        XCoord;     X Coord within bitmap                           */
/*  USHORT        YCoord;     Y Coord within bitmap                           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function returns the bit offset of the pel described by the X and Y  */
/*  coord.  The bit offset is an offset within the DWORD conatining that pel, */
/*  thus is in the range zero to 31.                                          */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
SHORT prdb_GetDwordBitOffset( pBMListEntry  ListEntry,
                              USHORT        usXCoord )

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    SHORT  sBitOffset;

    /**************************************************************************/
    /*  The XCoord and YCoord must be adjusted to take account of the current */
    /*  band in the bitmap's associated DC.                                   */
    /**************************************************************************/
    usXCoord += ListEntry->XOrigin;

    /**************************************************************************/
    /*  Determine the bit offset for (1,1) and (4,1) bitmaps.                 */
    /**************************************************************************/
    if (ListEntry->Parms.Bitcount == 1)
    {

        /**********************************************************************/
        /*  Number of bits is given by the top five bits                      */
        /**********************************************************************/
        sBitOffset = usXCoord & 31;
    }
    else
    {

        /**********************************************************************/
        /*  Number of bits is given by the top five bits of the transformed   */
        /*  Xcoord.                                                           */
        /**********************************************************************/
        sBitOffset = ((usXCoord << 2) & 31);
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return(sBitOffset);
}

/******************************************************************************/
/*  FUNCTION: prdb_StartPosAndDir                                             */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function determines the start position and the direction for         */
/*  processing the blt.  The default (when there is no overlap between source */
/*  and destination) is start at bottom right and process left and up.  If    */
/*  there is an overlap the directions are set to ensure any part of the      */
/*  source which will be overwritten is blt'ed from before it is blt'ed to.   */
/*                                                                            */
/*  BBNEXT : This function should be split in two.                            */
/*                                                                            */
/*  - prdb_SetUpDir() to set up HorzDir and VertDir only.  These do not need  */
/*  to vary with the current Src/TrgRects and so this would be called outside */
/*  the clip loop.                                                            */
/*                                                                            */
/*  - prdb_SetUpStartPos() to set up the StartPos coords.  These vary with the*/
/*  current Src/TrgRects and so must be called within the clip loop.  This    */
/*  function would then be very simple since the X coords vary with HorzDir   */
/*  only; the Y coords with VertDir only.                                     */
/*                                                                            */
/*  BBDIRECT : Default direction changed from right-left to left-right - check*/
/*  prdb_GetSrcDword for reasons.                                             */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_StartPosAndDir( lpBltParms BB )

{
#define TFUNC "prdb_StPosAndDr"

    /**************************************************************************/
    /*  Fast path with no source.                                             */
    /**************************************************************************/
    if ((BB->Source == NO_SOURCE) || (BB->SourceDCI != BB->TargetDCI))
    {

        /**********************************************************************/
        /*  There is no source or the target and source are not the same so   */
        /*  there is no need to worry about an overlap.                       */
        /**********************************************************************/
        BB->VertDir       = UP;
        BB->TrgStartPos.X = (BB->TrgRect)[0].X;
        BB->TrgStartPos.Y = (BB->TrgRect)[0].Y;
        BB->SrcStartPos.X = (BB->SrcRect)[0].X;
        BB->SrcStartPos.Y = (BB->SrcRect)[0].Y;
        return;
    }

    /**************************************************************************/
    /*  The source and target are from the same the same bitmap (ie both      */
    /*  selected into the same DC) so there is a possibility that the areas   */
    /*  may overlap and we must ensure that any overlap is blt'ed from before */
    /*  it is blt'ed to.                                                      */
    /**************************************************************************/

    /**************************************************************************/
    /*  Test for the target bitmap's bottom left corner lying inside the      */
    /*  source bitmap:                                                        */
    /*        _____________                                                   */
    /*        |           |                                                   */
    /*        |           |                                                   */
    /*    ____|________   |                                                   */
    /*    |   |       |   |                                                   */
    /*    |   |       |   |                                                   */
    /*    |   |       |   |Target                                             */
    /*    |   ~~~~~~~~|~~~~                                                   */
    /*    |           |                                                       */
    /*    |           |Source                                                 */
    /*    ~~~~~~~~~~~~~                                                       */
    /**************************************************************************/
    if ((((BB->TrgRect)[0].X > (BB->SrcRect)[0].X) &&
         ((BB->TrgRect)[0].Y >= (BB->SrcRect)[0].Y)) &&
        (((BB->TrgRect)[0].X <= (BB->SrcRect)[1].X) &&
         ((BB->TrgRect)[0].Y < (BB->SrcRect)[1].Y)))
    {

        /**********************************************************************/
        /*  In this case we will do the blt from the top right hand corner so */
        /*  set the start positions and directions accordingly.               */
        /**********************************************************************/
        BB->SrcStartPos.X = (BB->SrcRect)[0].X;
        BB->SrcStartPos.Y = (BB->SrcRect)[1].Y;
        BB->TrgStartPos.X = (BB->TrgRect)[0].X;
        BB->TrgStartPos.Y = (BB->TrgRect)[1].Y;
        BB->VertDir       = DOWN;
        return;
    }

    /**************************************************************************/
    /*  Test for the target bitmap's top left corner lying inside the source  */
    /*  bitmap:                                                               */
    /*   _____________                                                        */
    /*   |           |Source                                                  */
    /*   |           |                                                        */
    /*   |   ________|____                                                    */
    /*   |   |       |   |Target                                              */
    /*   |   |       |   |                                                    */
    /*   |   |       |   |                                                    */
    /*   ~~~~|~~~~~~~~   |                                                    */
    /*       |           |                                                    */
    /*       |           |                                                    */
    /*       ~~~~~~~~~~~~~                                                    */
    /**************************************************************************/
    if ((((BB->TrgRect)[0].X > (BB->SrcRect)[0].X) &&
         ((BB->TrgRect)[1].Y > (BB->SrcRect)[0].Y)) &&
        (((BB->TrgRect)[0].X < (BB->SrcRect)[1].X) &&
         ((BB->TrgRect)[1].Y < (BB->SrcRect)[1].Y)))
    {

        /**********************************************************************/
        /*  In this case we will do the blt from the bottom right hand corner */
        /*  - this is now the default so just return.                         */
        /**********************************************************************/
        BB->SrcStartPos.X = (BB->SrcRect)[0].X;
        BB->SrcStartPos.Y = (BB->SrcRect)[1].Y;
        BB->TrgStartPos.X = (BB->TrgRect)[0].X;
        BB->TrgStartPos.Y = (BB->TrgRect)[1].Y;
        BB->VertDir       = UP;
        return;
    }

    /**************************************************************************/
    /*  Test for the source bitmap's bottom left corner lying inside the      */
    /*  target bitmap:                                                        */
    /*       _____________                                                    */
    /*       |           |                                                    */
    /*       |           |                                                    */
    /*   ____|________   |                                                    */
    /*   |   |       |   |                                                    */
    /*   |   |       |   |                                                    */
    /*   |   |       |   |Source                                              */
    /*   |   ~~~~~~~~|~~~~                                                    */
    /*   |           |                                                        */
    /*   |           |Target                                                  */
    /*   ~~~~~~~~~~~~~                                                        */
    /**************************************************************************/
    if ((((BB->TrgRect)[0].X <= (BB->SrcRect)[0].X) &&
         ((BB->TrgRect)[0].Y < (BB->SrcRect)[0].Y)) &&
        (((BB->TrgRect)[1].X > (BB->SrcRect)[0].X) &&
         ((BB->TrgRect)[1].Y > (BB->SrcRect)[0].Y)))
    {

        /**********************************************************************/
        /*  In this case we will do the blt from the bottom left hand corner  */
        /*  so set the start positions and directions accordingly.            */
        /**********************************************************************/
        BB->SrcStartPos.X = (BB->SrcRect)[0].X;
        BB->SrcStartPos.Y = (BB->SrcRect)[0].Y;
        BB->TrgStartPos.X = (BB->TrgRect)[0].X;
        BB->TrgStartPos.Y = (BB->TrgRect)[0].Y;
        BB->VertDir       = UP;
        return;
    }

    /**************************************************************************/
    /*  Test for the source bitmap's top left corner lying inside the target  */
    /*  bitmap:                                                               */
    /*   _____________                                                        */
    /*   |           |Target                                                  */
    /*   |           |                                                        */
    /*   |   ________|____                                                    */
    /*   |   |       |   |Source                                              */
    /*   |   |       |   |                                                    */
    /*   |   |       |   |                                                    */
    /*   ~~~~|~~~~~~~~   |                                                    */
    /*       |           |                                                    */
    /*       |           |                                                    */
    /*       ~~~~~~~~~~~~~                                                    */
    /**************************************************************************/
    if ((((BB->TrgRect)[0].X <= (BB->SrcRect)[0].X) &&
         ((BB->TrgRect)[0].Y >= (BB->SrcRect)[0].Y)) &&
        (((BB->TrgRect)[1].X > (BB->SrcRect)[0].X) &&
         ((BB->TrgRect)[0].Y < (BB->SrcRect)[1].Y)))
    {

        /**********************************************************************/
        /*  In this case we will do the blt from the top left hand corner so  */
        /*  set the start positions and directions accordingly.               */
        /**********************************************************************/
        BB->SrcStartPos.X = (BB->SrcRect)[0].X;
        BB->SrcStartPos.Y = (BB->SrcRect)[1].Y;
        BB->TrgStartPos.X = (BB->TrgRect)[0].X;
        BB->TrgStartPos.Y = (BB->TrgRect)[1].Y;
        BB->VertDir       = DOWN;
        return;
    }
}
#undef TFUNC

/******************************************************************************/
/*  FUNCTION: prdb_GetMasks                                                   */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function determines the masks required for the start and end DWORDS  */
/*  of each row of the bitblt.                                                */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_GetMasks( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    SHORT  sBitOffset;
    ULONG  ulLeftMask;
    ULONG  ulRightMask;
    SHORT  sCount;
    BYTE   cBits;
    PBYTE  lpMask;

    /**************************************************************************/
    /*  Call prdb_GetBitOffsets with the bottom left corner.                  */
    /**************************************************************************/
    sBitOffset = prdb_GetDwordBitOffset(BB->TrgLE, BB->TrgRect[0].X);

    /**************************************************************************/
    /*  Special case when there is no offset                                  */
    /**************************************************************************/
    if (!sBitOffset)
    {

        /**********************************************************************/
        /*  All of the left DWORD is to be updated.                           */
        /**********************************************************************/
        BB->StartMask = 0xFFFFFFFF;
    }
    else
    {

        /**********************************************************************/
        /*  Set the LeftMask.  But when we use DWORDs the bytes are picked up */
        /*  in the reverse order to that expected, so the mask needs to be    */
        /*  reversed, and as a performance enhancement try to limit the       */
        /*  shifts.                                                           */
        /**********************************************************************/

        /**********************************************************************/
        /*  Watch carefully:                                                  */
        /*                                                                    */
        /*  BitOfset is in the range 0 to 31.  Consider some examples:        */
        /*                                                                    */
        /*  BitOffset = 00000 ( 0) => Mask = FF FF FF FF                      */
        /*  BitOffset = 00001 ( 1) => Mask = 7F FF FF FF                      */
        /*  BitOffset = 00010 ( 2) => Mask = 3F FF FF FF                      */
        /*  BitOffset = 00111 ( 7) => Mask = 01 FF FF FF                      */
        /*  BitOffset = 01000 ( 8) => Mask = 00 FF FF FF                      */
        /*  BitOffset = 01001 ( 9) => Mask = 00 7F FF FF                      */
        /*  BitOffset = 01010 (10) => Mask = 00 3F FF FF                      */
        /*  BitOffset = 01111 (15) => Mask = 00 01 FF FF                      */
        /*  BitOffset = 10000 (16) => Mask = 00 00 FF FF                      */
        /*  BitOffset = 10001 (17) => Mask = 00 00 7F FF                      */
        /*                                                                    */
        /*  Looking carefully we can see that the bottom three bits can be    */
        /*  used to give us the partially filled byte (just shift 0xFF right  */
        /*  by those three bits).                                             */
        /*                                                                    */
        /*  Then the next two bits indicate the number of bytes with no bits  */
        /*  set at the start of the mask.                                     */
        /*                                                                    */
        /*  Then the rest of the bytes have all their bits set.               */
        /*                                                                    */
        /*  So all we have to do is to assign each byte in the mask to either */
        /*  0x00, some partial value or 0xFF.                                 */
        /**********************************************************************/
        sCount = (sBitOffset & 24) >> 3;
        cBits  = (BYTE)0xFF >> (sBitOffset & 7);
        lpMask = (PBYTE)&(BB->StartMask);
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0x00 : (BYTE)0xFF ;
        sCount--;
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0x00 : (BYTE)0xFF ;
        sCount--;
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0x00 : (BYTE)0xFF ;
        sCount--;
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0x00 : (BYTE)0xFF ;
    }

    /**************************************************************************/
    /*  Call prdb_GetDwordOffsets with the top right corner.                  */
    /**************************************************************************/
    sBitOffset = prdb_GetDwordBitOffset(BB->TrgLE, BB->TrgRect[1].X);

    /**************************************************************************/
    /*  PD00486 : If the BitCount is 4, we need to add 3 to the sBitOffset to */
    /*  get to the end of the nibble.                                         */
    /**************************************************************************/
    if (LOUSHORT(BB->TrgLE->Parms.Bitcount) == 4)
        sBitOffset += 3;

    if (sBitOffset == 31)
    {

        /**********************************************************************/
        /*  Blt ended on the last bit in a dword, so all bits are to be set,  */
        /**********************************************************************/
        BB->EndMask = 0xFFFFFFFF;
    }
    else
    {

        /**********************************************************************/
        /*  Set the RightMask.  But when we use DWORDs the bytes are picked up*/
        /*  in the reverse order to that expected, so the mask needs to be    */
        /*  reversed.                                                         */
        /**********************************************************************/

        /**********************************************************************/
        /*  Watch carefully again...                                          */
        /*                                                                    */
        /*  BitOfset is in the range 0 to 31.  Consider some examples:        */
        /*                                                                    */
        /*  BitOffset = 00000 ( 0) => Mask = 80 00 00 00                      */
        /*  BitOffset = 00001 ( 1) => Mask = C0 00 00 00                      */
        /*  BitOffset = 00010 ( 2) => Mask = E0 00 00 00                      */
        /*  BitOffset = 00111 ( 7) => Mask = FF 00 00 00                      */
        /*  BitOffset = 01000 ( 8) => Mask = FF 80 00 00                      */
        /*  BitOffset = 01001 ( 9) => Mask = FF 80 00 00                      */
        /*  BitOffset = 01010 (10) => Mask = FF C0 00 00                      */
        /*  BitOffset = 01111 (15) => Mask = FF FF 00 00                      */
        /*  BitOffset = 10000 (16) => Mask = FF FF 80 00                      */
        /*  BitOffset = 10001 (17) => Mask = FF FF C0 00                      */
        /*  BitOffset = 11111 (31) => Mask = FF FF FF FF                      */
        /*                                                                    */
        /*  Looking carefully we can see that the bottom three bits can be    */
        /*  used to give us the partially filled byte (by shifting 0xFF left  */
        /*  by some number of bytes.                                          */
        /*                                                                    */
        /*  Then the next two bits indicate the number of bytes with all bits */
        /*  set at the start of the mask.                                     */
        /*                                                                    */
        /*  Then the rest of the bytes have none of their bits set.           */
        /*                                                                    */
        /*  So all we have to do is to assign each byte in the mask to either */
        /*  0xFF, some partial value or 0x00.                                 */
        /**********************************************************************/
        sCount = (sBitOffset & 24) >> 3;
        cBits  = (BYTE)0xFF << (7 - sBitOffset & 7);
        lpMask = (PBYTE)&(BB->EndMask);
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0xFF : (BYTE)0x00 ;
        sCount--;
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0xFF : (BYTE)0x00 ;
        sCount--;
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0xFF : (BYTE)0x00 ;
        sCount--;
        *(lpMask++) = (sCount == 0) ? cBits :
                      (sCount >  0) ? (BYTE)0xFF : (BYTE)0x00 ;
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_GetIndices                                                 */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function determines the indexes to the start and finish of the first */
/*  row of the source in the source bitmap and of the target in the target    */
/*  bitmap.  These need only be calculated for the first row, after this they */
/*  can simply be incremented by the source and target bitmap widths (in      */
/*  bytes).  Note the use of SrcStartPos and TrgStartPos to allow for the     */
/*  position where we will start the blt from and the direction we will do it */
/*  in.                                                                       */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_GetIndices( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    SHORT   sSrcOffset;
    SHORT   sTrgOffset;
    USHORT  usTrgBytes;
    USHORT  usSrcBytes;

    /**************************************************************************/
    /*  Work out the index to the first byte of the first row we will do the  */
    /*  blt on.  This is converted into a DWORD offset later on               */
    /**************************************************************************/
    BB->TrgIndex = prdb_GetByteOffset(BB->TrgLE, BB->TrgStartPos.X,
                                      BB->TrgStartPos.Y);

    /**************************************************************************/
    /*  Work out the index to the last byte on the first row.  We will use the*/
    /*  end mask on this byte.  This si converted into a DWORD offset later   */
    /*  on.                                                                   */
    /**************************************************************************/
    BB->TrgEndIndex = prdb_GetByteOffset(BB->TrgLE, BB->TrgRect[1].X,
                                         BB->TrgStartPos.Y);

    /**************************************************************************/
    /*  If there is a source then the plan is to copy the source row into some*/
    /*  spare storage (in prdb_GetSourceRow) so need to set up some useful    */
    /*  values to make this process as painless as possible.                  */
    /**************************************************************************/
    if (BB->Source != NO_SOURCE)
    {

        /**********************************************************************/
        /*  Get the index of the first byte used in the source.               */
        /**********************************************************************/
        BB->SrcIndex = prdb_GetByteOffset(BB->SrcLE, BB->SrcStartPos.X,
                                          BB->SrcStartPos.Y);

        /**********************************************************************/
        /*  Get the index of the last byte used in the source.                */
        /**********************************************************************/
        BB->SrcEndIndex = prdb_GetByteOffset(BB->SrcLE, BB->SrcRect[1].X,
                                             BB->SrcStartPos.Y);

        /**********************************************************************/
        /*  Get the number of target bytes used.                              */
        /**********************************************************************/
        usTrgBytes = LOUSHORT(BB->TrgEndIndex - BB->TrgIndex) + 1;
        usSrcBytes = LOUSHORT(BB->SrcEndIndex - BB->SrcIndex) + 1;

        /**********************************************************************/
        /*  Now set the number of bytes to copy and whether the overlap is at */
        /*  the start of the beginning.                                       */
        /**********************************************************************/
        if (usTrgBytes > usSrcBytes)
        {

            /******************************************************************/
            /*  The last byte produces a carry that must be used              */
            /******************************************************************/
            BB->usSrcBytes  = usSrcBytes;
            BB->usBltFlags |= SRC_END_CARRY;
        }
        else
        {
            if (usSrcBytes > usTrgBytes)
            {

                /**************************************************************/
                /*  The first byte must be used to initialize the carry       */
                /**************************************************************/
                BB->usSrcBytes  = usTrgBytes;
                BB->usBltFlags |= SRC_START_CARRY;
            }
            else
            {

                /**************************************************************/
                /*  Correct number of bytes.                                  */
                /**************************************************************/
                BB->usSrcBytes = usTrgBytes;
            }
        }

        /**********************************************************************/
        /*  Get the bit offsets of the first target pel and the first source  */
        /*  pel.                                                              */
        /**********************************************************************/
        sSrcOffset = prdb_GetBitOffset(BB->SrcLE, BB->SrcRect[0].X);
        sTrgOffset = prdb_GetBitOffset(BB->TrgLE, BB->TrgRect[0].X);

        /**********************************************************************/
        /*  For a positive phase we will process the source row left to right,*/
        /*  otherwise right to left.                                          */
        /**********************************************************************/
        BB->sDirection = sTrgOffset - sSrcOffset;
        BB->cShift     = (BYTE)abs(BB->sDirection);
        BB->cInvShift  = (BYTE)8 - BB->cShift;

        /**********************************************************************/
        /*  Ultimately the new source row should be DWORD aligned with the    */
        /*  target.  But as we are going to copy bytes we need to find the    */
        /*  byte offset within the DWORD for the target.                      */
        /**********************************************************************/
        BB->usSrcAdjust = LOUSHORT(BB->TrgIndex) & 3;
    }                                  /* if (BB->Source != NO_SOURCE)        */

    /**************************************************************************/
    /*  Having got hold of the byte offsets of the target need to convert     */
    /*  these to DWORD offsets.  Just drop the top two bits.                  */
    /**************************************************************************/
    /*LOUSHORT(BB->TrgIndex   )&= 0xFFFC; CON3201 */
    BB->TrgIndex   &= 0xFFFFFFFC; /* CON3201 can't cast on left side so just  */
                                  /* and the whole DWORD                      */
/*  LOUSHORT(BB->TrgEndIndex) &= 0xFFFC;  CON3201 */
    BB->TrgEndIndex &= 0xFFFFFFFC; /* CON3201 can't cast on left side so just */
                                   /* and the whole DWORD                     */
    /**************************************************************************/
    /*  Store the number of DWORDs to be blted on each row.                   */
    /**************************************************************************/
    BB->usTrgDwords = 1 + (BB->TrgEndIndex - BB->TrgIndex) / 4;

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_CachePattern                                               */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function sets up a new copy of the pattern which has been expanded to*/
/*  a DWORD in length and shifted the required amount so that it can be used  */
/*  immediately.                                                              */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_CachePattern( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    USHORT  ii;                        /* loop variable                       */

    /**************************************************************************/
    /*  Initialize the start position                                         */
    /**************************************************************************/
    BB->ptsPatStart.x  = (BB->TrgRect[0].X & ~31) - BB->PatOrigin.X;
    BB->ptsPatStart.x %= BB->PatWidth;

    /**************************************************************************/
    /*  If the start position is zero and the pattern width is 32 then the    */
    /*  bitmap is exactly right as it is, so do nothing.  Otherwise the bitmap*/
    /*  needs expanding to DWORDS and shifting.                               */
    /**************************************************************************/
    if ((BB->ptsPatStart.x != 0) || (BB->PatWidth != 32))
    {

        /**********************************************************************/
        /*  Initialize pointers                                               */
        /**********************************************************************/
        BB->abPatRow = BB->CacheBM;
        BB->lpPatRow = BB->PatBM;

        /**********************************************************************/
        /*  Loop for all the rows, doing the transformation                   */
        /**********************************************************************/
        for (ii = 0; ii < BB->PatHeight; ii++)
        {

            /******************************************************************/
            /*  This function uses lpPatRow as it's input and writes the data */
            /*  to abPatRow.                                                  */
            /******************************************************************/
            (VOID)prdb_Pat11BitAligned(BB);

            /******************************************************************/
            /*  Move to the next position in both bitmaps.                    */
            /******************************************************************/
            BB->abPatRow += 4;
            BB->lpPatRow += BB->PatBytesPerRow;
        }

        /**********************************************************************/
        /*  Now adjust the pattern parameters.                                */
        /**********************************************************************/
        BB->PatBM          = BB->CacheBM;
        BB->PatBytesPerRow = 4;

        /**********************************************************************/
        /*  If the pattern width was eight and the rop is pattern copy then we*/
        /*  can use memset in the blt, so keep the pattern width as eight.    */
        /*  Otherwise the pattern has now been expanded to a full DWORD.      */
        /**********************************************************************/
        if ((BB->PatWidth == 8) && (BB->Mix == ROP_PATCOPY))
        {
            BB->PatWidth = 8;
        }
        else
        {
            BB->PatWidth = 32;
        }

        /**********************************************************************/
        /*  Reset these pointers as they no longer point to anything sensible */
        /**********************************************************************/
        BB->abPatRow = FNULL;
        BB->lpPatRow = FNULL;
    }

    /**************************************************************************/
    /*  Finished                                                              */
    /**************************************************************************/

EXIT_FUNCTION:

    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_GetPattern                                                 */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function sets up the pattern information fot the bitblt.  The pattern*/
/*  is color expanded (to the target bitmap format) using either the given    */
/*  ArgAttrs colors or the pattern bundle colors.                             */
/*                                                                            */
/*  BUNNIES : Code which initializes method of picking up the pattern bytes   */
/*  ripped out and replaced.                                                  */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
USHORT prdb_GetPattern( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    lpDCI              DCIData;        /* current DCIData                     */
    PDAREABUNDLE       AreaAts;        /* current area attributes             */
    USHORT             usResult;       /* OK, ERROR_ZERO or GS_FLAG           */
    lpDDTType          pDDT;           /* DDT Pointer                         */

    /**************************************************************************/
    /*  These used for default pattern arm                                    */
    /**************************************************************************/
    lpPatternInfoType  lpPatternInfo;
    USHORT             usPatIndex;     /* index into pattern info table       */
    USHORT             usPatNumber;    /* index into pattern table            */
    BOOL               fGreyScale;     /* true is greyscaling                 */
    ULONG              ulGSColor;      /* color to greyscale                  */
    USHORT             usPatType;      /* greyscale flags                     */

    /**************************************************************************/
    /*  These used for engine image font arm                                  */
    /**************************************************************************/
    SHORT              i, j;           /* Loop control variables              */
    FMFFileStrucType   FMFData;        /* FMF for current font                */
    FontDataType       FontData;       /* Font information                    */
    CHARDEFS           SaveCharDefs;   /* Temp character attributes           */
    PCHARDEFS          pCurTxtDefs;    /* Pointer to current txt defs         */
    PBYTE              CodePtBuffer;   /* buffer for codepoint defn           */
    USHORT             usPatSize;
    USHORT             PatBytesPerRow; /* no. of bytes in a pattern row       */
    USHORT             CPtBytesPerRow; /* no. of bytes in a codepoint row     */
    USHORT             CPtHeight;      /* code point height                   */
    PBYTE              CPtDefn;        /* pointer to codepoint defn           */

    /**************************************************************************/
    /*  These used for user defined bitmap arm                                */
    /**************************************************************************/
    pBMListEntry       PatLE;          /* The pattern list entry              */

    /**************************************************************************/
    /*  Initialize a few variables...                                         */
    /*                                                                        */
    /*  PD00549 : Moved init of pDDT to common code instead of only default   */
    /*  patterns since we now need to check the number of colors for the      */
    /*  active printer irregardless of whose patterns we're dealing with.     */
    /**************************************************************************/
    DCIData = BB->TargetDCI;
    AreaAts = BB->TargetDCI->DCICurPtnAts;
    pDDT    = &DCIData->DCIPdbInstance->DDT;

    /**************************************************************************/
    /*  Assume that everything works out fine.                                */
    /**************************************************************************/
    usResult = OK;

    /**************************************************************************/
    /*  If the pattern set is '0' then use our default pattern set.           */
    /**************************************************************************/
    if (AreaAts->adef.defSet == 0)
    {

        /**********************************************************************/
        /*  Get the pattern number                                            */
        /**********************************************************************/
        usPatNumber = AreaAts->abnd.usSymbol;

        /**********************************************************************/
        /*  Later on we wish to use usPatNumber as an index into our pattern  */
        /*  table.  Unfortunately the DDI interface does not use contiguous   */
        /*  indentifiers.  So correct for that here.  Also, if the pattern    */
        /*  fill is all ones or all zeroes then store that information for    */
        /*  later use.                                                        */
        /*                                                                    */
        /*  PD00498 : Note that the following values for usBltFlags for all   */
        /*  cases below have been changed.  Here's why:  If we're not going to*/
        /*  greyscale (PATSYM_ specified is not SOLID or DEFAULT), we need to */
        /*  pass in the correct pattern.  Since the pattern will be inverted  */
        /*  as necessary by the bit blit code, we need to represent the       */
        /*  patterns correctly here.  What happens essentially is that we     */
        /*  represent the patterns in an inverted fashion and in the blit code*/
        /*  later on we invert again, leaving us with the correct values at   */
        /*  the printer.  So, to get PATSYM_NOSHADE, we request PAT_ALL_ZEROS */
        /*  (used to be PAT_ALL_ONES), knowing that we'll not greyscale and   */
        /*  that the blit code will invert the pattern to PAT_ALL_ONES later. */
        /*                                                                    */
        /*  Now for the tricky part.  When we're talking about BLANK or       */
        /*  NOSHADE, we'll never greyscale, so there is no chance of the      */
        /*  usBltFlags ever being reset by the code anywhere else.  For the   */
        /*  DEFAULT and SOLID or default processing path however, we might    */
        /*  just greyscale the pattern as well so things get a bit trickier.  */
        /*  We need to make sure that we again have the inverted pattern in   */
        /*  the usBitFlags and if we then greyscale then that will be handled */
        /*  later by prdb_GetGreyScaleInfo().  If we don't greyscale here     */
        /*  though, we still need to pass in the inverted pattern.            */
        /*                                                                    */
        /*  One final note, remember that the patterns that we are assigning  */
        /*  are the inverse patterns that will be blitted - this is not very  */
        /*  consistent with OS/2 and the rest of the code since one would     */
        /*  expect to see PATSYM_SOLID be represented with PAT_ALL_ZEROS      */
        /*  rather than PAT_ALL_ONES as the zero represents a pel set on at   */
        /*  the device within OS/2.                                           */
        /**********************************************************************/
        switch (usPatNumber)
        {
            case PATSYM_DEFAULT  :
                usPatNumber     = PATSYM_SOLID;
                BB->usBltFlags |= PAT_ALL_ONES;
                break;
            case PATSYM_BLANK    :
                usPatNumber     = PATSYM_NOSHADE;
                BB->usBltFlags |= PAT_ALL_ZEROS;
                break;
            case PATSYM_NOSHADE  :
                usPatNumber     = PATSYM_NOSHADE;
                BB->usBltFlags |= PAT_ALL_ZEROS;
                break;
            case PATSYM_SOLID    :
                usPatNumber     = PATSYM_SOLID;
                BB->usBltFlags |= PAT_ALL_ONES;
                break;
            case PATSYM_HALFTONE :     /* PD00599...                          */
                usPatNumber     = PATSYM_DENSE4;
                break;
            default              :
                if (usPatNumber > PatternTabSize)
                {
                    usPatNumber = PATSYM_SOLID;
                    BB->usBltFlags |= PAT_ALL_ZEROS;
                }
        }

        /**********************************************************************/
        /*  Now see if the bit blit has specified any colors to be used during*/
        /*  the blit.  If so, we need to use them.                            */
        /*                                                                    */
        /*  PD00462 : Have to use background color for PATSYM_NOSHADE.  Moved */
        /*  the color assignment to here so we can test for RGB_BLACK - code  */
        /*  was in the following fGreyScale block just before the call to get */
        /*  the greyscale index.                                              */
        /**********************************************************************/
        if (BB->ArgStyle & BLTMODE_ATTRS_PRES)
        {
            if (usPatNumber == PATSYM_NOSHADE)
                ulGSColor = BB->ArgAttrs.lBackColor;
            else
                ulGSColor = BB->ArgAttrs.lColor;
        }
        else
        {
            if (usPatNumber == PATSYM_NOSHADE)
                ulGSColor = AreaAts->abnd.lBackColor;
            else
                ulGSColor = AreaAts->abnd.lColor;
        }

        /**********************************************************************/
        /*  PD00462 : We'll greyscale if we meet the conditions specified in  */
        /*  the prdb_GetGreyScaleInfo() function, the pattern number is       */
        /*  PATSYM_SOLID or PATSYM_NOSHADE and there is not a source bitmap.  */
        /*                                                                    */
        /*  GREY:  Changed prdb_GetGreyScaleInfo() call.                      */
        /**********************************************************************/
        fGreyScale = (BB->Source == NO_SOURCE) &&
                     ((usPatNumber == PATSYM_SOLID) ||
                      (usPatNumber == PATSYM_NOSHADE)) &&
                     (prdb_GetGreyScaleInfo(ulGSColor, &usPatType,
                                            &(BB->GS_Pattern), DCIData));

        /**********************************************************************/
        /*  Different set up for greyscaling                                  */
        /**********************************************************************/
        if (fGreyScale)
        {

            /******************************************************************/
            /*  When we have finished we will return the greyscale flag.      */
            /*  Switch off the pattern description flags.  The correct one is */
            /*  set on GetGreyScaleInfo.                                      */
            /******************************************************************/
            usResult        = GS_FLAG;
            BB->usBltFlags &= ~PAT_ALL_ZEROS;
            BB->usBltFlags &= ~PAT_ALL_ONES;

            /******************************************************************/
            /*  Now fill in the BB structure with the relevant greyscale info */
            /******************************************************************/
            BB->usBltFlags     |= usPatType;
            BB->PatBM           = (PBYTE)BB->GS_Pattern;
            BB->PatWidth        = GS_PATTERN_WIDTH;
            BB->PatHeight       = GS_PATTERN_HEIGHT;
            BB->PatBytesPerRow  = GS_PATTERN_BYTES_PER_ROW;

            /******************************************************************/
            /*  This flag is not set in the prdb_GetGreyScaleInfo but is      */
            /*  required for the pattern bitblt work.                         */
            /******************************************************************/
            BB->usPatType = BLT_PAT_11_BITMAP;
        }
        else
        {

            /******************************************************************/
            /*  Default patterns differ depending on orientation              */
            /******************************************************************/
            if (DCIData->DCIPdbInstance->Orientation == PORTRAIT)
            {
                usPatIndex = pDDT->DDTRasterMode->PortPattIndex;
            }
            else
            {
                usPatIndex = pDDT->DDTRasterMode->LandPattIndex;
            }

            /******************************************************************/
            /*  This gives us the pattern structure                           */
            /******************************************************************/
            lpPatternInfo = &PatternInfoTable[usPatIndex];

            /******************************************************************/
            /*  Pattern number enumerates from zero in our table but one in   */
            /*  the DDI interface.  So correct for that.                      */
            /******************************************************************/
            usPatNumber--;

            /******************************************************************/
            /*  Fill in the pattern dimensions                                */
            /******************************************************************/
            BB->usPatType      = BLT_PAT_11_BITMAP;
            BB->PatBM          = lpPatternInfo->PatternTable[usPatNumber];
            BB->PatWidth       = LOUSHORT(lpPatternInfo->PatternWidth);
            BB->PatHeight      = LOUSHORT(lpPatternInfo->PatternHeight);
            BB->PatBytesPerRow = LOUSHORT(lpPatternInfo->PatternBytesPerRow);
        }
    }
    else                               /* if (AreaAts->adef.defSet == 0)      */
    {

        /**********************************************************************/
        /*  The pattern is not one of our default ones so we must get its     */
        /*  bitmap handle and parameters.  If the generic flag is set then the*/
        /*  pattern is an engine font, otherwise it is a bitmap.              */
        /**********************************************************************/
        if (AreaAts->adef.fFlags & CDEF_GENERIC)
        {

            /******************************************************************/
            /*  Local copy of pointer to text attributes.                     */
            /******************************************************************/
            pCurTxtDefs = &DCIData->DCICurTxtAts->cdef;

            /******************************************************************/
            /*  Save the current text attribute bundle.                       */
            /******************************************************************/
            prdu_memcpy((PBYTE)&SaveCharDefs, (PBYTE)pCurTxtDefs,
                        sizeof(CHARDEFS));

            /******************************************************************/
            /*  Copy area attributes to text bundle.                          */
            /******************************************************************/
            prdu_memcpy((PBYTE)pCurTxtDefs, (PBYTE)&AreaAts->adef,
                        sizeof(AREADEFS));

            /******************************************************************/
            /*  Set up information about the font in our local copy of the    */
            /*  font data structure.                                          */
            /******************************************************************/
            FontData.Info.pFont = &FMFData;
            if (prdt_LocateFont(&FontData, DCIData) != OK)
            {
                usResult = ERROR_ZERO;
                goto EXIT_ARM;
            }

            /******************************************************************/
            /*  If the font is not an engine image font then it cannot be used*/
            /*  as a pattern.                                                 */
            /******************************************************************/
            if ((FontData.Info.Type != FT_ENGINE) ||
                (FontData.pMetrics->fsDefn & FM_DEFN_OUTLINE))
            {
                usResult = ERROR_ZERO;
                goto EXIT_ARM;
            }

            /******************************************************************/
            /*  Get the codepoint information                                 */
            /******************************************************************/
            prdt_NextCodePoint(AreaAts->abnd.usSymbol, &FontData, DCIData);

            /******************************************************************/
            /*  Work out the amount of memory needed for the code point       */
            /*  definition in bitmap format.                                  */
            /******************************************************************/
            CPtHeight      = FontData.CPtInfo.Height;
            CPtBytesPerRow = (FontData.CPtInfo.Width + 7) / 8;
            PatBytesPerRow = ((CPtBytesPerRow + 3) / 4) * 4;
            usPatSize      = PatBytesPerRow * CPtHeight;
            if (prdg_AllocHeapItem(DCIData, usPatSize,
                                   (PUSHORT *)&CodePtBuffer) != OK)
            {
                usResult = ERROR_ZERO;
                goto EXIT_ARM;
            }

            /******************************************************************/
            /*  Save a copy of this pointer so that it can be freed at the end*/
            /*  of the function.                                              */
            /******************************************************************/
            BB->FontBM     = CodePtBuffer;
            BB->FontBMSize = usPatSize;

            /******************************************************************/
            /*  Copy the codepoint definition from the font format to the     */
            /*  bitmap format, ie swap rows and columns                       */
            /******************************************************************/
            CPtDefn = FontData.CPtInfo.Defn;
            for (i = CPtHeight - 1; i >= 0; i--)
            {
                for (j = 0; j < CPtBytesPerRow; j++)
                    CodePtBuffer[j] = CPtDefn[i + CPtHeight * j];
            }

            /******************************************************************/
            /*  Now set up the required fields.                               */
            /******************************************************************/
            BB->usPatType      = BLT_PAT_11_BITMAP;
            BB->PatBM          = CodePtBuffer;
            BB->PatHeight      = CPtHeight;
            BB->PatWidth       = FontData.CPtInfo.Width;
            BB->PatBytesPerRow = PatBytesPerRow;

EXIT_ARM:

            /******************************************************************/
            /*  Restore the current text attribute bundle values.             */
            /******************************************************************/
            prdu_memcpy((PBYTE)pCurTxtDefs, (PBYTE)&SaveCharDefs,
                        sizeof(CHARDEFS));

            /******************************************************************/
            /*  If there was an error then return it now.                     */
            /******************************************************************/
            if (usResult == ERROR_ZERO)
            {
                return(ERROR_ZERO);
            }
        }
        else
        {

            /******************************************************************/
            /*  This is a bitmap.  This is the easy case.                     */
            /******************************************************************/
            PatLE              = (pBMListEntry)AreaAts->adef.defSet;
            BB->PatBM          = PatLE->Bitmap;
            BB->PatHeight      = LOUSHORT(PatLE->Parms.Height);
            BB->PatWidth       = LOUSHORT(PatLE->Parms.Width);
            BB->PatBytesPerRow = LOUSHORT(PatLE->Parms.BytesPerRow);

            /******************************************************************/
            /*  Need a flag to indicate the type of pattern                   */
            /******************************************************************/
            if (PatLE->Parms.Bitcount == 1)
            {

                /**************************************************************/
                /*  Simple (1,1) bitmap.  Pattern needs processing one bit at */
                /*  a time.                                                   */
                /**************************************************************/
                BB->usPatType = BLT_PAT_11_BITMAP;
            }
            else
            {

                /**************************************************************/
                /*  It's a (4,1) pattern.  Pattern needs processing one nibble*/
                /*  at a time.                                                */
                /**************************************************************/
                BB->usPatType = BLT_PAT_41_BITMAP;
            }

            /******************************************************************/
            /*  Store the information about whether it was supposed to be a   */
            /*  (1,1) bitmap.                                                 */
            /******************************************************************/
            if (PatLE->Parms.MonBmapReq)
            {
                BB->usPatType |= BLT_PAT_MONO_REQ;
            }
        }
    }

    /**************************************************************************/
    /*  Get a local copy of the pattern origin and adjust for DCOrigin        */
    /*  (BANDING)                                                             */
    /**************************************************************************/
    BB->PatOrigin.X = DCIData->DCIPattOriDev.X + BB->TrgLE->XOrigin;
    BB->PatOrigin.Y = DCIData->DCIPattOriDev.Y + BB->TrgLE->YOrigin;

    /**************************************************************************/
    /*  Adjust so that the coordinates are negative - since we do calculations*/
    /*  in ULONGs later, we want to avoid getting a negative answer when      */
    /*  subtracting the pattern origin from a point in the target bitmap.     */
    /**************************************************************************/
    BB->PatOrigin.X %= BB->PatWidth;
    if (BB->PatOrigin.X > 0)
    {
        BB->PatOrigin.X -= BB->PatWidth;
    }
    BB->PatOrigin.Y %= BB->PatHeight;
    if (BB->PatOrigin.Y > 0)
    {
        BB->PatOrigin.Y -= BB->PatHeight;
    }

    /**************************************************************************/
    /*  The pattern is essentially a (1,1) bitmap so determine what actual    */
    /*  colors a set pel and a non-set pel map to.                            */
    /**************************************************************************/
    if (usResult == GS_FLAG)
    {

        /**********************************************************************/
        /*  If we're grey scaling, just set the foreground and back ground    */
        /*  colors to black and white respectively.                           */
        /**********************************************************************/
        BB->ConvPat0 = PRD_FOREGROUND;
        BB->ConvPat1 = PRD_BACKGROUND;
    }
    else
    {
        if (BB->ArgStyle & BLTMODE_ATTRS_PRES)
        {

            /******************************************************************/
            /*  Use the BitBltAttrs passed in to BitBlt.                      */
            /******************************************************************/
            BB->ConvPat0 = prdc_ColorToPelBits(BB->ArgAttrs.lBackColor,
                                               BB->TargetDCI,
                                               BB->TrgLE->DCTPtr);

            BB->ConvPat1 = prdc_ColorToPelBits(BB->ArgAttrs.lColor,
                                               BB->TargetDCI,
                                               BB->TrgLE->DCTPtr);
        }
        else
        {

            /******************************************************************/
            /*  Use current pattern attributes                                */
            /******************************************************************/
            BB->ConvPat0 = prdc_ColorToPelBits(BB->TargetDCI->DCICurPtnAts->
                                               abnd.lBackColor, BB->TargetDCI,
                                               BB->TrgLE->DCTPtr);

            BB->ConvPat1 = prdc_ColorToPelBits(BB->TargetDCI->DCICurPtnAts->
                                               abnd.lColor, BB->TargetDCI,
                                               BB->TrgLE->DCTPtr);
        }

        /**********************************************************************/
        /*  PD00462 : If both foreground and background colors for the pattern*/
        /*  go to black, we'll let the foreground color go to black and we'll */
        /*  take the background to white.                                     */
        /**********************************************************************/
        if ((BB->ConvPat0 == PRD_FOREGROUND) &&
            (BB->ConvPat1 == PRD_FOREGROUND))
            BB->ConvPat0 = PRD_BACKGROUND;
    }

    /**************************************************************************/
    /*  Save PatCopyMode for speed later on.                                  */
    /*                                                                        */
    /*  PD00486 : We're expecting to see PatCopyMode in terms of 0's and 1's  */
    /*  as this is a flag that we will use later when we rop the pattern.  And*/
    /*  since the 4224 color will create a value of 0 for ConvPat1 and 7 for  */
    /*  ConvPat0 resulting in PatCopyMode of 07, we need to do something a bit*/
    /*  more sensible here so that we still get 0's and 1's.  Furthermore, we */
    /*  now have an additional complication.  If the app has A)requested one  */
    /*  of our default patterns, B)has requested a 4,1 bitmap, C)we've given  */
    /*  them a 4,1 bitmap, and D)they've set colors in the attribute bundles  */
    /*  for both foreground and background colors, we can get into a real     */
    /*  mess, for this can yield ConvPat1 = 4 and ConvPat0 = 7.  In this case */
    /*  we still need the PatCopyMode to come out as SRC_COPY_BITS, which is  */
    /*  a value of 02.                                                        */
    /*                                                                        */
    /*  PD00529 : We need to catch the situation where we have a 4,1 bitmap   */
    /*  and we need to ensure that we aren't greyscaling either.  If these    */
    /*  conditions pass, then set the PatCopyMode to SRC_COPY_BITS.           */
    /**************************************************************************/
    if ((!fGreyScale) && (pDDT->DDTMaxColors == DDT_EIGHT_COLORS))
    {

        /**********************************************************************/
        /*  PD00567 : If we're working with a source bitmap greyscale-capable */
        /*  driver and we're working with a 1,1 target bitmap then we'll      */
        /*  incorrectly set the pattern copy mode to SRC_COPY_BITS when we    */
        /*  really need to set it to SRC_INVERT_BITS.  We need to catch this  */
        /*  condition and handle it properly...                               */
        /**********************************************************************/
        if (BB->TrgLE->Parms.Bitcount == 1)
        {
            BB->PatCopyMode = SRC_INVERT_BITS;
        }
        else
        {
            BB->PatCopyMode = SRC_COPY_BITS;
        }
    }
    else
    {
        BB->PatCopyMode = ((BB->ConvPat1 != 0) ? 1 : 0) << 1 |
                          ((BB->ConvPat0 != 0) ? 1 : 0);
    }

    /**************************************************************************/
    /*  Should we cache the pattern.  Only do this for the (1,1) case There is*/
    /*  no reason why it cannot be done for the (4,1) case too, I leave this  */
    /*  as an E.E.F.T.R.                                                      */
    /**************************************************************************/
    if ((LOUSHORT(BB->TrgLE->Parms.Bitcount) == 1) &&
        ((BB->PatWidth ==  8) || (BB->PatWidth == 16) || (BB->PatWidth == 32)))
    {

        /**********************************************************************/
        /*  Yes, good idea.  Allocate the memory for the cached pattern.      */
        /**********************************************************************/
        BB->CacheBMSize = 4 * BB->PatWidth;
        if (prdg_AllocHeapItem(BB->TargetDCI, BB->CacheBMSize,
                               (PUSHORT *)&BB->CacheBM) != OK)
        {
            usResult = ERROR_ZERO;
            goto EXIT_FUNCTION;
        }

        /**********************************************************************/
        /*  Initialize it to zeros                                            */
        /**********************************************************************/
        prdu_memset((PBYTE)BB->CacheBM, 0x00, BB->CacheBMSize);

        /**********************************************************************/
        /*  Now fill it with the rows that are required.                      */
        /**********************************************************************/
        (VOID)prdb_CachePattern(BB);
    }
    else
    {

        /**********************************************************************/
        /*  The pattern row must be as large as the target row just in case   */
        /*  the full line is being blted.                                     */
        /**********************************************************************/
        BB->PatRowBMSize = (USHORT) ((BB->TrgLE)->Parms.BytesPerRow);
        if (prdg_AllocHeapItem(BB->TargetDCI, BB->PatRowBMSize,
                               (PUSHORT *)&BB->PatRowBM) != OK)
        {
            usResult = ERROR_ZERO;
            goto EXIT_FUNCTION;
        }

        /**********************************************************************/
        /*  Initialize it to zeros                                            */
        /**********************************************************************/
        prdu_memset((PBYTE)BB->PatRowBM, 0x00, BB->PatRowBMSize);

        /**********************************************************************/
        /*  PD00508 : Set the pointer to the original pattern row to the newly*/
        /*  allocated pattern row memory.                                     */
        /**********************************************************************/
        BB->abPatRow = BB->PatRowBM;

        /**********************************************************************/
        /*  In this arm a whole pattern row is to be created at one time, but */
        /*  it is to be done on the fly.                                      */
        /**********************************************************************/
        BB->usBltFlags |= BLT_PAT_ROW_REQD;
    }

EXIT_FUNCTION:

    /**************************************************************************/
    /*  Return the greyscaling flag or OK.                                    */
    /**************************************************************************/
    return(usResult);
}

/******************************************************************************/
/*  FUNCTION: prdb_Pat11BitAligned                                            */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This gets a pattern row adjusting it so that it is DWORD aligned with the */
/*  target.                                                                   */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_Pat11BitAligned( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    USHORT  usNumDwords;               /* number of pattern dwords            */
    USHORT  usByteIndex;               /* current byte index into row         */
    USHORT  usBitIndex;                /* current bit  index into byte        */
    BYTE    cPatByte;                  /* current pattern byte                */
    USHORT  ii,jj;                     /* loop variables                      */
    USHORT  usPatBit= 0;               /* current pattern                     */
    USHORT  usStart;                   /* starting x co-ordinate              */
    PBYTE   lpPattern;                 /* pattern row                         */
    PBYTE   lpRow;                     /* pointer to pattern row              */
    ULONG   ulPattern;                 /* DWORD to put into pattern row       */

    /**************************************************************************/
    /*  Get the old pattern (lpPattern) and the new pattern pointer           */
    /*  (abPatRow).                                                           */
    /**************************************************************************/
    lpPattern = BB->lpPatRow;
    lpRow     = BB->abPatRow;

    /**************************************************************************/
    /*  May need to generate the full row or just one DWORD.                  */
    /**************************************************************************/
    if (BB->usBltFlags & BLT_PAT_ROW_REQD)
    {
        usNumDwords = BB->usTrgDwords;
    }
    else
    {
        usNumDwords = 1;
    }

    /**************************************************************************/
    /*  This is the bit index into the pattern where we start.                */
    /**************************************************************************/
    usStart = BB->ptsPatStart.x;

    /**************************************************************************/
    /*  Fastpath an offset of zero (ie no bit shift) and pattern width of a   */
    /*  multiple of 8.                                                        */
    /**************************************************************************/
    if ((usStart == 0) && (BB->PatWidth & 7 == 0))
    {

        /**********************************************************************/
        /*  Get the first byte                                                */
        /**********************************************************************/
        cPatByte = lpPattern[0];

        /**********************************************************************/
        /*  Here we can just copy bytes.                                      */
        /**********************************************************************/
        for (ii = 0; ii < usNumDwords; ii++)
        {
            for (jj = 3; jj >= 0; jj--)
            {

                /**************************************************************/
                /*  Get the byte and the address of the next byte.            */
                /**************************************************************/
                lpRow[jj]   = cPatByte;
                usBitIndex += 8;

                /**************************************************************/
                /*  End of the pattern, go back to the beginning              */
                /**************************************************************/
                if (usBitIndex == BB->PatWidth)
                {
                    cPatByte    = *lpPattern;
                    usByteIndex = 0;
                }
                else
                {

                    /**********************************************************/
                    /*  Just get the next byte                                */
                    /**********************************************************/
                    cPatByte = lpPattern[++usByteIndex];
                }

                /**************************************************************/
                /*  Move to the next position in the target                   */
                /**************************************************************/
                lpRow += 4;
            }
        }

        /**********************************************************************/
        /*  and finished now                                                  */
        /**********************************************************************/
        goto EXIT_FUNCTION;
    }

    /**************************************************************************/
    /*  This is the more complicated case, when the bits need shifting around.*/
    /**************************************************************************/
    ulPattern   = 0;
    usByteIndex = usStart >> 3;
    cPatByte    = lpPattern[usByteIndex];
    usBitIndex  = usStart & 7;
    cPatByte  <<= usBitIndex;

    /**************************************************************************/
    /*  Start bit index at the correct place, taking into account any full    */
    /*  bytes that we have skipped.                                           */
    /**************************************************************************/
    usBitIndex += usByteIndex << 3;

    /**************************************************************************/
    /*  Loop through the number of DWORDs required for this row.              */
    /**************************************************************************/
    for (ii = 0; ii < usNumDwords; ii++)
    {

        /**********************************************************************/
        /*  Loop through the number of bits in this DWORD                     */
        /**********************************************************************/
        for (jj = 0, ulPattern = 0; jj < 32; jj++)
        {

            /******************************************************************/
            /*  Shift pattern byte and extract the lowest bit                 */
            /******************************************************************/
            usPatBit   = (cPatByte & 0x80) >> 7;
            cPatByte <<= 1;

            /******************************************************************/
            /*  Rotate target left to make room for the new bit and then      */
            /*  include that new bit.                                         */
            /******************************************************************/
/*          ulPattern            = _lrotl(ulPattern, 1);              CON3201 */
/*          LOUSHORT(ulPattern) |= usPatBit;                          CON3201 */
            ulPattern = prdu_lrotl(ulPattern, 1) | (ULONG)usPatBit;/* CON3201 */

            /******************************************************************/
            /*  Get the address of the next bit                               */
            /******************************************************************/
            usBitIndex++;

            /******************************************************************/
            /*  If we have reached the end of the row then start again        */
            /******************************************************************/
            if (usBitIndex == BB->PatWidth)
            {
                cPatByte    = *lpPattern;
                usBitIndex  = 0;
                usByteIndex = 0;
            }
            else
            {

                /**************************************************************/
                /*  If we have reached the end of a byte then get the next    */
                /*  byte                                                      */
                /**************************************************************/
                if (!(usBitIndex & 7))
                {
                    cPatByte = lpPattern[++usByteIndex];
                }
            }
        }

        /**********************************************************************/
        /*  Dword is now full.  Insert it into the pattern row with the bytes */
        /*  in the correct place.                                             */
        /**********************************************************************/
        *lpRow++ = ((PBYTE)&ulPattern)[3];
        *lpRow++ = ((PBYTE)&ulPattern)[2];
        *lpRow++ = ((PBYTE)&ulPattern)[1];
        *lpRow++ = ((PBYTE)&ulPattern)[0];
    }

EXIT_FUNCTION:

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_Pat41BitAligned                                            */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This gets a pattern row adjusting it so that it is DWORD aligned with the */
/*  target.                                                                   */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_Pat41BitAligned( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    USHORT  usByteIndex;               /* current byte index into row         */
    USHORT  usBitIndex;                /* current bit index into byte         */
    BYTE    cPatByte;                  /* current pattern byte                */
    USHORT  ii,jj;                     /* loop variables                      */
    USHORT  usPatBit = 0;              /* current pattern                     */
    USHORT  usStart;                   /* starting x co-ordinate              */
    PBYTE   lpPattern;                 /* pattern row                         */
    PBYTE   lpRow;                     /* pointer to pattern row              */
    ULONG   ulPattern;                 /* DWORD to put into pattern row       */
    BYTE    cForeColor;                /* Foreground color and...             */
    BYTE    cBackColor;                /* Background color                    */
    USHORT  usPatWidth;                /* Pattern width in bits (not pels)    */

    /**************************************************************************/
    /*  Get the pattern row                                                   */
    /**************************************************************************/
    lpRow      = BB->abPatRow;
    lpPattern  = BB->lpPatRow;
    usStart    = BB->ptsPatStart.x;
    usPatWidth = BB->PatWidth;         /* PD00486...                          */

    /**************************************************************************/
    /*  PD00486 : Need to init the colors.  Note that we're working with a 4,1*/
    /*  pattern so we need to set the pattern to the colors associated with   */
    /*  the colors set up for foreground and background.                      */
    /**************************************************************************/
    cForeColor = BB->ConvPat1;
    cBackColor = BB->ConvPat0;

    /**************************************************************************/
    /*  Problem here is that the pattern could be:                            */
    /*                                                                        */
    /*  1. (1,1) from one of our pre-defined patterns                         */
    /*  2. (4,1) but (1,1) bitmap requested                                   */
    /*  3. (4,1) truly.                                                       */
    /*                                                                        */
    /*  All of these are different cases.                                     */
    /**************************************************************************/
    if (BB->usPatType & BLT_PAT_41_BITMAP)
    {

        /**********************************************************************/
        /*  Here we step through the bitmap one nibble at a time.  There are  */
        /*  two pels per byte, so shifting right by one gives an index to the */
        /*  actual byte.  ie 0 -> 0, 1 -> 0, 2 -> 1, etc                      */
        /**********************************************************************/
        usByteIndex = usStart >> 1;

        /**********************************************************************/
        /*  Bit index should either be 0 or 4 depending on whether it is the  */
        /*  first pel or the second pel in the byte.                          */
        /**********************************************************************/
        usBitIndex = usStart & 1 << 2;

        /**********************************************************************/
        /*  PD00486 : Pattern width in bits for the 4,1 bitmap case requires  */
        /*  a small tweak here...                                             */
        /**********************************************************************/
        usPatWidth <<= 2;
    }
    else
    {

        /**********************************************************************/
        /*  Here we step through the bitmap one bit at a time, expanding each */
        /*  bit to a nibble.                                                  */
        /**********************************************************************/
        usByteIndex = usStart >> 3;
        usBitIndex  = usStart & 7;

        /**********************************************************************/
        /*  PD00486 : Need to init the colors.  Note that we're working with a*/
        /*  1,1 pattern so we need to set the pattern to either 0's or 7's if */
        /*  we have been asked for a mono bitmap.                             */
        /**********************************************************************/
        if (BB->usPatType & BLT_PAT_MONO_REQ)
        {
            cForeColor = 7;
            cBackColor = 0;
        }
    }

    /**************************************************************************/
    /*  Get the byte and align it correctly.                                  */
    /**************************************************************************/
    cPatByte = lpPattern[usByteIndex] << usBitIndex;

    /**************************************************************************/
    /*  Start bit index at the correct place, taking into account any full    */
    /*  bytes that we have skipped.                                           */
    /**************************************************************************/
    usBitIndex += usByteIndex << 3;

    /**************************************************************************/
    /*  Loop through the number of DWORDs required for this row.              */
    /**************************************************************************/
    for (ii = 0; ii < BB->usTrgDwords; ii++)
    {

        /**********************************************************************/
        /*  Loop through the number of pels in this DWORD                     */
        /**********************************************************************/
        for (jj = 0, ulPattern = 0; jj < 8; jj++)
        {

            /******************************************************************/
            /*  Shift pattern byte and extract the lowest bit                 */
            /******************************************************************/
            if (BB->usPatType & BLT_PAT_41_BITMAP)
            {

                /**************************************************************/
                /*  (4,1) case.  Get nibble.                                  */
                /**************************************************************/
                usPatBit    = (cPatByte & 0xF0) >> 4;
                cPatByte  <<= 4;
                usBitIndex += 4;
            }
            else
            {

                /**************************************************************/
                /*  (1,1) case.  Get bit                                      */
                /**************************************************************/
                usPatBit    = (cPatByte & 0x80) >> 7;
                cPatByte  <<= 1;
                usBitIndex += 1;
            }

            /******************************************************************/
            /*  Convert this bit into a nibble.                               */
            /*                                                                */
            /*  PD00486 : Changed following to use actual colors rather than  */
            /*  assuming that we want a 7...                                  */
            /*                                                                */
            /*  PD00529 : Moved this out of the else block above as this is   */
            /*  necessary for both the 1,1 and 4,1 cases.                     */
            /******************************************************************/
            if (usPatBit)
            {
                usPatBit = cForeColor;
            }
            else
            {
                usPatBit = cBackColor;
            }

            /******************************************************************/
            /*  Rotate target left to make room for the new nibble and then   */
            /*  include that new bit.                                         */
            /******************************************************************/
/*          ulPattern            = _lrotl(ulPattern, 4);              CON3201 */
/*          LOUSHORT(ulPattern) |= usPatBit;                          CON3201 */
            ulPattern = prdu_lrotl(ulPattern, 4) | (ULONG)usPatBit; /*CON3201 */

            /******************************************************************/
            /*  If we have reached the end of the row then start again        */
            /******************************************************************/
            if (usBitIndex == usPatWidth)
            {
                cPatByte    = *lpPattern;
                usBitIndex  = 0;
                usByteIndex = 0;
            }
            else
            {

                /**************************************************************/
                /*  If we have reached the end of a byte then get the next    */
                /*  byte                                                      */
                /**************************************************************/
                if (!(usBitIndex & 7))
                {
                    cPatByte = lpPattern[++usByteIndex];
                }
            }
        }

        /**********************************************************************/
        /*  Dword is now full.  Insert it into the pattern row with the bytes */
        /*  in the correct place.                                             */
        /**********************************************************************/
        *lpRow++ = ((PBYTE)&ulPattern)[3];
        *lpRow++ = ((PBYTE)&ulPattern)[2];
        *lpRow++ = ((PBYTE)&ulPattern)[1];
        *lpRow++ = ((PBYTE)&ulPattern)[0];
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_GetPatternRow                                              */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This gets a pattern row adjusting it so that it is DWORD aligned with the */
/*  target.                                                                   */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_GetPatternRow( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    USHORT  usPatIndex;                /* index into pattern row              */
    USHORT  usPatWidth;                /* index into pattern row              */
    USHORT  ii, jj;
    ULONG   ulPattern;                 /* pattern DWORD                       */
    USHORT  usPatBit;                  /* new bit for pattern DWORD           */
    USHORT  usByte;                    /* index of first byte in row          */
    USHORT  usBit;                     /* index of first bit  in byte         */
    BYTE    cPatByte;                  /* current pattern byte                */
    USHORT  usByteIndex;               /* index of current byte               */
    USHORT  usBitIndex;                /* index of current bit                */
    USHORT  usStart;                   /* y index of row                      */
    PULONG  lpRow;                     /* pointer to row                      */

    /**************************************************************************/
    /*  Two arms because the processing is different in each case             */
    /**************************************************************************/
    if (LOUSHORT(BB->TrgLE->Parms.Bitcount) == 1)
    {

        /**********************************************************************/
        /*  Co-ordinates of start of DWORD row.  Note that x is constant      */
        /*  throughout this function so could take it out.                    */
        /**********************************************************************/
        usStart  = BB->TrgStartPos.Y - BB->PatOrigin.Y;
        usStart %= BB->PatHeight;

        /**********************************************************************/
        /*  If the full row is required then get it.  Otherwise we're done    */
        /*  because the pattern has been cached in PatBM.                     */
        /**********************************************************************/
        if (BB->usBltFlags & BLT_PAT_ROW_REQD)
        {

            /******************************************************************/
            /*  Get a pointer to the current row                              */
            /******************************************************************/
            BB->lpPatRow = BB->PatBM + BB->PatBytesPerRow * usStart;

            /******************************************************************/
            /*  Set up the x co-ordinate of the start of the DWORD.           */
            /******************************************************************/
            BB->ptsPatStart.x = (BB->TrgRect[0].X & ~31) - BB->PatOrigin.X;
            BB->ptsPatStart.x %= BB->PatWidth;

            /******************************************************************/
            /*  This function takes lpPatRow and expands it to abPatRow.      */
            /******************************************************************/
            prdb_Pat11BitAligned(BB);
        }
        else
        {

            /******************************************************************/
            /*  Pattern has been cached so just point at the current row.     */
            /******************************************************************/
            BB->abPatRow = BB->PatBM + BB->PatBytesPerRow * usStart;
        }
    }
    else
    {

        /**********************************************************************/
        /*  Target BitCount = 4.  TrgStartPos.X & ~7 gives the x-coordinate in*/
        /*  the bitmap of the start of the DWORD in which TrgStartPos.X       */
        /*  occurs.                                                           */
        /**********************************************************************/
        BB->ptsPatStart.y = BB->TrgStartPos.Y - BB->PatOrigin.Y;
        BB->ptsPatStart.x = (BB->TrgRect[0].X & ~7) - BB->PatOrigin.X;

        /**********************************************************************/
        /*  Map these to starting co-ordinates in the bitmap                  */
        /**********************************************************************/
        BB->ptsPatStart.y %= BB->PatHeight;
        BB->ptsPatStart.x %= BB->PatWidth;

        /**********************************************************************/
        /*  Get the offset of the start of this pattern row                   */
        /**********************************************************************/
        BB->lpPatRow = BB->PatBM + BB->PatBytesPerRow * BB->ptsPatStart.y;

        /**********************************************************************/
        /*  Mark that the full row has been produced                          */
        /**********************************************************************/
        BB->usBltFlags |= BLT_PAT_ROW_REQD;

        /**********************************************************************/
        /*  Do complicated case.  Room for speeding this arm up               */
        /**********************************************************************/
        prdb_Pat41BitAligned(BB);
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_GetSourceRow                                               */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function gets the row for the source bitmap, shifting it the required*/
/*  amount.                                                                   */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_GetSourceRow( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    PBYTE   lpReadIx;                  /* Pointer to source buffer            */
    PBYTE   lpWritIx;                  /* Pointer to source row storage       */
    USHORT  usNumBytes;                /* Number of bytes to copy             */
    BYTE    cShift;                    /* shift count to get source byte      */
    BYTE    cInvShift;                 /* shift count to get carry byte       */
    BYTE    cCarry;                    /* carry byte                          */
    USHORT  ii;                        /* loop variable                       */

    /**************************************************************************/
    /*  Get pointers to the source bitmap and the storage for the row         */
    /**************************************************************************/
    lpReadIx  = (PBYTE)BB->SrcIndex;
    lpWritIx  = (PBYTE)BB->abSrcRow + BB->usSrcAdjust;
    cInvShift = BB->cInvShift;
    cShift    = BB->cShift;
    cCarry    = 0;

    /**************************************************************************/
    /*  Code handles all three cases although there is a performance          */
    /*  enhancement to be made which avoids copying the data when the phase is*/
    /*  zero and the source row is all in one segment.  But may not be worth  */
    /*  it because memcpy is very fast.                                       */
    /**************************************************************************/
    if (BB->sDirection == 0)
    {
        if (BB->usFastFlags & SRC_ROW_IN_SEG)
        {

            /******************************************************************/
            /*  All of the source row is in one segment.                      */
            /******************************************************************/
            prdu_memcpy(lpWritIx, lpReadIx, BB->usSrcBytes);
        }
        /* CON3201
        ** Should never go into this ELSE since usFastFlags will be set if
        ** there is a Source.
        */
        else
        {
// CON3201
// CON3201  /******************************************************************/
// CON3201  /*  Not all of the bytes are in one segment.  First copy those    */
// CON3201  /*  into the lowest segment.                                      */
// CON3201  /******************************************************************/
// CON3201  usNumBytes = 0xFFFF - LOUSHORT(BB->SrcIndex);
// CON3201  usNumBytes++;
// CON3201  prdu_memcpy(lpWritIx, lpReadIx, usNumBytes);
// CON3201
// CON3201  /******************************************************************/
// CON3201  /*  Now the other segment.  Get the new addresses.  The selector  */
// CON3201  /*  of the lpWritIx does not change.                              */
// CON3201  /******************************************************************/
// CON3201  OFFSETOF(lpWritIx)   += usNumBytes;
// CON3201  SELECTOROF(lpReadIx) += LOUSHORT(prdd_HugeInc);
// CON3201  OFFSETOF(lpReadIx)    = 0;
// CON3201
// CON3201  /******************************************************************/
// CON3201  /*  And do the memcpy                                             */
// CON3201  /******************************************************************/
// CON3201  usNumBytes = LOUSHORT(BB->SrcEndIndex);
// CON3201  usNumBytes++;
// CON3201  prdu_memcpy(lpWritIx, lpReadIx, usNumBytes);
        }
    }
    else
    {
        if (BB->sDirection > 0)
        {

            /******************************************************************/
            /*  Phase is greater than zero.  This is this case:               */
            /*                                                                */
            /*           sssssssssssssssssss                                  */
            /*               tttttttttttttttttttt                             */
            /*                                                                */
            /*  Here we work from left to right, shifting the source by the   */
            /*  actual phase difference and storing the carry for use in the  */
            /*  next byte.                                                    */
            /*                                                                */
            /*  We may need to use the first source byte to provide the       */
            /*  initial carry.                                                */
            /******************************************************************/
            if (BB->usBltFlags & SRC_START_CARRY)
            {
                cCarry |= *lpReadIx << cInvShift;
                lpReadIx++;
            }
            if (BB->usFastFlags & SRC_ROW_IN_SEG)
            {

                /**************************************************************/
                /*  Adjust the starting positions for bytes that will not be  */
                /*  referenced.                                               */
                /**************************************************************/
                for (ii = 0; ii < BB->usSrcBytes; ii++, lpWritIx++, lpReadIx++)
                {
                    *lpWritIx = (*lpReadIx >> cShift) | cCarry;
                    cCarry    = (*lpReadIx << cInvShift);
                }
            }
            /* CON3201
            ** Should never go into this ELSE since usFastFlags will be set if
            ** there is a Source.
            */
            else
            {
// CON3201
// CON3201      /**************************************************************/
// CON3201      /*  Adjust the starting position and the number of bytes      */
// CON3201      /**************************************************************/
// CON3201      usNumBytes = 0xFFFF - OFFSETOF(lpReadIx);
// CON3201
// CON3201      /**************************************************************/
// CON3201      /*  Now copy the bytes in the first segment.                  */
// CON3201      /**************************************************************/
// CON3201      for (ii = 0; ii <= usNumBytes; ii++, lpWritIx++, lpReadIx++)
// CON3201      {
// CON3201          *lpWritIx = (*lpReadIx >> cShift) | cCarry;
// CON3201          cCarry    = (*lpReadIx << cInvShift);
// CON3201      }
// CON3201
// CON3201      /**************************************************************/
// CON3201      /*  Now the other segment.  Get the new read address          */
// CON3201      /**************************************************************/
// CON3201      SELECTOROF(lpReadIx) += LOUSHORT(prdd_HugeInc);
// CON3201      OFFSETOF(lpReadIx)    = 0;
// CON3201
// CON3201      /**************************************************************/
// CON3201      /*  Get the number of bytes                                   */
// CON3201      /**************************************************************/
// CON3201       usNumBytes = OFFSETOF(BB->SrcEndIndex);
// CON3201       for (ii = 0; ii <= usNumBytes; ii++, lpWritIx++, lpReadIx++)
// CON3201       {
// CON3201           *lpWritIx = (*lpReadIx >> cShift) | cCarry;
// CON3201           cCarry    = (*lpReadIx << cInvShift);
// CON3201       }
            }

            /******************************************************************/
            /*  May need to use the final carry in the final byte             */
            /******************************************************************/
            if (BB->usBltFlags & SRC_END_CARRY)
            {
                *lpWritIx = cCarry;
            }
        }
        else
        {

            /******************************************************************/
            /*  Phase is less than zero.  This is this case:                  */
            /*                                                                */
            /*                    ssssssssssssssssssss                        */
            /*               tttttttttttttttttttt                             */
            /*                                                                */
            /*  Here we work from right to left, shifting the source by the   */
            /*  actual phase difference and storing the carry for use in the  */
            /*  next byte.                                                    */
            /******************************************************************/
            if (BB->usFastFlags & SRC_ROW_IN_SEG)
            {

                /**************************************************************/
                /*  Get to the first byte (at the end of the row).  If there  */
                /*  is n bytes then move the pointer (n-1) places.  The read  */
                /*  pointer is adjusted when the carry is taken out.          */
                /**************************************************************/
                lpReadIx += BB->usSrcBytes - 1;
                lpWritIx += BB->usSrcBytes - 1;

                /**************************************************************/
                /*  We may need to use the first source byte to provide the   */
                /*  initial carry.                                            */
                /**************************************************************/
                if (BB->usBltFlags & SRC_START_CARRY)
                {
                    cCarry |= *(lpReadIx + 1) >> cInvShift;
                }

                /**************************************************************/
                /*  Now copy the bytes.                                       */
                /**************************************************************/
                for (ii = 0; ii < BB->usSrcBytes; ii++, lpWritIx--, lpReadIx--)
                {
                    *lpWritIx = (*lpReadIx << cShift) | cCarry;
                    cCarry    = (*lpReadIx >> cInvShift);
                }
            }
            /* CON3201
            ** Should never go into this ELSE since usFastFlags will be set if
            ** there is a Source.
            */
            else
            {

// CON3201      /**************************************************************/
// CON3201      /*  Adjust the starting positions for bytes that will not be  */
// CON3201      /*  referenced.                                               */
// CON3201      /**************************************************************/
// CON3201      SELECTOROF(lpReadIx) += LOUSHORT(prdd_HugeInc);
// CON3201      OFFSETOF(lpReadIx)    = LOUSHORT(BB->SrcEndIndex);
// CON3201      OFFSETOF(lpWritIx)   += BB->usSrcBytes - 1;
// CON3201
// CON3201      /**************************************************************/
// CON3201      /*  We may need to use the first source byte to provide the   */
// CON3201      /*  initial carry.                                            */
// CON3201      /**************************************************************/
// CON3201      if (BB->usBltFlags & SRC_START_CARRY)
// CON3201      {
// CON3201          cCarry |= *lpReadIx >> cInvShift;
// CON3201          lpReadIx--;
// CON3201       }
// CON3201
// CON3201       /**************************************************************/
// CON3201       /*  set the number of bytes and copy them                     */
// CON3201       /**************************************************************/
// CON3201       usNumBytes = LOUSHORT(lpReadIx);
// CON3201       for (ii = 0; ii <= usNumBytes; ii++, lpWritIx--, lpReadIx--)
// CON3201       {
// CON3201           *lpWritIx = (*lpReadIx << cShift) | cCarry;
// CON3201           cCarry    = (*lpReadIx >> cInvShift);
// CON3201       }
// CON3201
// CON3201       /**************************************************************/
// CON3201       /*  Now the second segment                                    */
// CON3201       /**************************************************************/
// CON3201       SELECTOROF(lpReadIx) -= LOUSHORT(prdd_HugeInc);
// CON3201       OFFSETOF(lpReadIx)    = 0xFFFF;
// CON3201
// CON3201       /**************************************************************/
// CON3201       /*  set the number of bytes and copy them                     */
// CON3201       /**************************************************************/
// CON3201       usNumBytes = 0xFFFF - LOUSHORT(BB->SrcIndex);
// CON3201       for (ii = 0; ii <= usNumBytes; ii++, lpWritIx--, lpReadIx--)
// CON3201       {
// CON3201           *lpWritIx = (*lpReadIx << cShift) | cCarry;
// CON3201           cCarry    = (*lpReadIx >> cInvShift);
// CON3201       }
            }

            /******************************************************************/
            /*  We may need to use the first source byte to provide the       */
            /*  initial carry.                                                */
            /******************************************************************/
            if (BB->usBltFlags & SRC_END_CARRY)
            {
                *lpWritIx = cCarry;
            }
        }
    }

    /**************************************************************************/
    /*  And finished.                                                         */
    /**************************************************************************/
    return;
}
#undef TFUNC

/******************************************************************************/
/*  FUNCTION: prdb_PerformRop                                                 */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;        Pointer to Bitblt working storage                  */
/*  PULONG      OpsArray;  Pointer to array of source[0], target[1]           */
/*                         and pattern[2] dwords to perform rop on.           */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function performs the rop specified by the ParseStr(ing) and the     */
/*  StrIndex as set up in prdb_GetAndCheckRop.  It returns the resulting      */
/*  dword.                                                                    */
/*                                                                            */
/*  BBDWORD : Now works in dwords not bytes.                                  */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
ULONG prdb_PerformRop( PULONG      OpsArray,
                       lpBltParms  BB )

{

#define TFUNC "prdb_PerformRop"

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    SHORT    Index;                    /* Index into the operand string       */
    ULONG    FirstOp,                  /* The 1st and ...                     */
             SecondOp;                 /* 2nd operand for binary operations   */
    ULONG    Result;                   /* Accumulator to build result of rop  */
    USHORT   i;                        /* Loop control variable               */
    ULONG    Stack;                    /* 1 element stack for pushes and pops */
    USHORT   Operation;                /* The operation we are doing          */
    PUSHORT  ParseStr;


    /**************************************************************************/
    /*  Optimise some common rops                                             */
    /**************************************************************************/
    switch ((USHORT)BB->Mix)
    {
        case ROP_SRCCOPY    :

           /*******************************************************************/
           /*  Source copy: just return the source byte                       */
           /*******************************************************************/
           return(OpsArray[0]);
           break;
        case ROP_PATCOPY    :

           /*******************************************************************/
           /*  Pattern copy: just return the pattern byte                     */
          /********************************************************************/
           return(OpsArray[2]);
           break;
        case 0xFA:

           /*******************************************************************/
           /*  Pattern OR Destination: default for PolyScanLines              */
           /*******************************************************************/
           return(OpsArray[2] | OpsArray[1]);
           break;

        case ROP_NOTSRCCOPY :

           /*******************************************************************/
           /*  Not Source copy: just return the source byte inverted          */
           /*******************************************************************/
           return(~OpsArray[0]);
           break;

        case ROP_DSTINVERT  :

           /*******************************************************************/
           /*  Destination invert: return the target byte inverted            */
           /*******************************************************************/
           return(~OpsArray[1]);
           break;

        case ROP_ZERO       :

           /*******************************************************************/
           /*  All zeroes                                                     */
           /*******************************************************************/
           return(0L);
           break;

        case ROP_ONE        :

           /*******************************************************************/
           /*  All ones                                                       */
           /*******************************************************************/
           return(-1L);
           break;

        default             :

           /*******************************************************************/
           /*  All other rops: calculate return value.  Set first and second  */
           /*  operands from the OpsArray.                                    */
           /*******************************************************************/
           Index = (SHORT)(BB->StrIndex);

           /*******************************************************************/
           /*  Get the two operands for the first operation.                  */
           /*******************************************************************/
           ParseStr = (PUSHORT)BB->ParseStr;
           FirstOp  = OpsArray[ParseStr[Index--]];

           /*******************************************************************/
           /*  Be careful as it is possible that StrIndex may be only 1 and we*/
           /*  don't want to go accessing the -1'th element of any arrays.    */
           /*******************************************************************/
           if (Index != -1)
               SecondOp = OpsArray [ParseStr[Index--]];

           /*******************************************************************/
           /*  Go through each operator from the rop - which we have          */
           /*  translated into OpString and perform the required operation    */
           /*******************************************************************/
           for (i = 0; i < MAX_OPERANDS; i++)
           {
               Operation = (BB->OpString)[i];
               switch (Operation)
               {
                   case BB_NOT :
                       Result = ~FirstOp;
                       break;
                   case BB_XOR :
                       Result = FirstOp ^ SecondOp;
                       break;
                   case BB_OR  :
                       Result = FirstOp | SecondOp;
                       break;
                   case BB_AND :
                       Result = FirstOp & SecondOp;
                       break;
               }

               /***************************************************************/
               /*  Set up operands for next operation.  The 1st operand for   */
               /*  the next operation is the result of the previous operation.*/
               /*  The second operand only needs to be worked out if the last */
               /*  operation was binary (ie not a NOT).                       */
               /***************************************************************/
               FirstOp = Result;
               if (Operation != BB_NOT)
               {
                   if (ParseStr[Index] == PUSH)
                   {

                       /*******************************************************/
                       /*  The next operand is a PUSH so save the current     */
                       /*  accumulated value to the 'Stack' and fetch a new   */
                       /*  first and second operand.                          */
                       /*******************************************************/
                       Stack = Result;
                       --Index;
                       FirstOp  = OpsArray[ParseStr[Index--]];
                       SecondOp = OpsArray[ParseStr[Index--]];
                   }
                   else
                   {
                       if (ParseStr[Index] == POP)
                       {

                           /***************************************************/
                           /*  The next operand is a POP so restore the value */
                           /*  currently held on the stack as the second      */
                           /*  operand.                                       */
                           /***************************************************/
                           --Index;
                           SecondOp = Stack;
                       }
                       else
                       {

                           /***************************************************/
                           /*  The next operand is simply P,S or D so get the */
                           /*  appropriate byte from OpsArray.                */
                           /***************************************************/
                           SecondOp = OpsArray[ParseStr[Index--]];
                       }
                   }
               }
           }

           /*******************************************************************/
           /*  If parity specified in Rop then NOT result before returning it.*/
           /*******************************************************************/
           if ((BB->Rop & 0x0020) != 0)
           {
               return(~Result);
           }
           else
           {
               return(Result);
           }

    }
}
#undef TFUNC

/******************************************************************************/
/*  FUNCTION: prdb_RopOneDword                                                */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function performs the rop for a single DWORD in the row.             */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_RopOneDword( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    ULONG   OpsArray[3];               /* src, trg and pat DWORDs             */
    ULONG   NewTrgDword;               /* target DWORD asfter the rop         */
 /* ULONG   TrgIndex; */               /* index to target DWORD               */
    PULONG  TrgIndex;                  /* index to target DWORD      CON3201  */
    ULONG   ulMask;                    /* mask for target DWORD               */
    PULONG  lpSource;                  /* pointer to source dwords            */
    PULONG  lpPattern;                 /* pointer to pattern dwords           */

    /**************************************************************************/
    /*  Local variable for speed.                                             */
    /**************************************************************************/
 /* TrgIndex = BB->TrgIndex; */
    TrgIndex = (PULONG)(BB->TrgIndex);                             /* CON3201 */

    /**************************************************************************/
    /*  Initiate the mask.  There is only one DWORD to blt so combine both    */
    /*  masks.                                                                */
    /*                                                                        */
    /*  PD00496 : AND not OR the masks together.                              */
    /**************************************************************************/
    ulMask = BB->StartMask & BB->EndMask;
    if (BB->Source != NO_SOURCE)
    {

        /**********************************************************************/
        /*  May need to invert the source DWORD based on the colors being set.*/
        /**********************************************************************/
        lpSource = (PULONG)BB->abSrcRow;
        prdm_GetDword(BB->SrcCopyMode, *lpSource, OpsArray[0]);
    }
    if (BB->PatternReq)
    {

        /**********************************************************************/
        /*  May need to invert the pattern DWORD based on the colors being    */
        /*  set.                                                              */
        /**********************************************************************/
        lpPattern = (PULONG)BB->abPatRow;
        prdm_GetDword(BB->PatCopyMode, *lpPattern, OpsArray[2]);
    }

    /**************************************************************************/
    /*  Get the next target DWORD.                                            */
    /**************************************************************************/
/*  OpsArray[1] = *((PULONG)TrgIndex);                                CON3201 */
    OpsArray[1] = *TrgIndex;

    /**************************************************************************/
    /*  Do the Rop for the current DWORD in the target bitmap.                */
    /**************************************************************************/
    NewTrgDword = prdb_PerformRop(OpsArray, BB);

    /**************************************************************************/
    /*  Store the new target DWORD.                                           */
    /**************************************************************************/
/* CON3201
    *((PULONG)TrgIndex) = (NewTrgDword & ulMask) | (OpsArray[1] & ~ulMask);
*/
    *TrgIndex = (NewTrgDword & ulMask) | (OpsArray[1] & ~ulMask);

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_RopTwoDwords                                               */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function performs the rop for two DWORDs in the row.                 */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_RopTwoDwords( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    ULONG   OpsArray[3];               /* src, trg and pat DWORDs             */
    ULONG   NewTrgDword;               /* target DWORD asfter the rop         */
 /* ULONG   TrgIndex; */               /* index to target DWORD               */
    PULONG  TrgIndex;                  /* index to target DWORD      CON3201  */
    ULONG   TrgDword;                  /* index to target DWORD               */
    ULONG   ulMask;                    /* mask for target DWORD               */
    PULONG  lpSource;                  /* pointer to source dwords            */
    PULONG  lpPattern;                 /* pointer to pattern dwords           */

    /**************************************************************************/
    /*  Local variable for speed.                                             */
    /**************************************************************************/
 /* TrgIndex = BB->TrgIndex; */
    TrgIndex = (PULONG)(BB->TrgIndex);                             /* CON3201 */
    ulMask   = BB->StartMask;

    if (BB->Source != NO_SOURCE)
    {

        /**********************************************************************/
        /*  May need to invert the source DWORD based on the colors being set.*/
        /**********************************************************************/
        lpSource = (PULONG)BB->abSrcRow;
        prdm_GetDword(BB->SrcCopyMode, *lpSource, OpsArray[0]);
    }
    if (BB->PatternReq)
    {

        /**********************************************************************/
        /*  Get pointer as DWORD pointer                                      */
        /**********************************************************************/
        lpPattern = (PULONG)BB->abPatRow;
        prdm_GetDword(BB->PatCopyMode, *lpPattern, OpsArray[2]);
    }

    /**************************************************************************/
    /*  Get the next target DWORD.                                            */
    /**************************************************************************/
/*  OpsArray[1] = TrgDword = *((PULONG)TrgIndex);                     CON3201 */
    OpsArray[1] = TrgDword = *TrgIndex;

    /**************************************************************************/
    /*  Do the Rop for the current DWORD in the target bitmap.                */
    /**************************************************************************/
    NewTrgDword = prdb_PerformRop(OpsArray, BB);

    /**************************************************************************/
    /*  Store the new target DWORD.                                           */
    /**************************************************************************/
/* CON3201
    *((PULONG)TrgIndex) = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);
*/
    *TrgIndex = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);

    /**************************************************************************/
    /*  Move to the next, being careful about end of segments.                */
    /**************************************************************************/
 /* CON3201 ***********************************************/
 /* Don't worry about segments                            */
 /* if (!(LOUSHORT(TrgIndex) += 4))                       */
 /* {                                                     */
 /*     SELECTOROF(TrgIndex) += LOUSHORT(prdd_HugeInc);   */
 /* }                                                     */
 /*********************************************************/

    TrgIndex ++;             /* CON3201 - increment by four bytes            */
    if (BB->Source != NO_SOURCE)
    {

        /**********************************************************************/
        /*  Move to the next source DWORD (always in one segment)             */
        /**********************************************************************/
        lpSource++;
        prdm_GetDword(BB->SrcCopyMode, *lpSource, OpsArray[0]);
    }
    if (BB->PatternReq)
    {

        /**********************************************************************/
        /*  Move onto the next pattern if required.  If the full pattern row  */
        /*  is set up then the next pattern DWORD is required.  Otherwise we  */
        /*  already have the pattern DWORD.                                   */
        /**********************************************************************/
        if (BB->usBltFlags & BLT_PAT_ROW_REQD)
        {

            /******************************************************************/
            /*  Patterns also in one segment.                                 */
            /******************************************************************/
            lpPattern++;
            prdm_GetDword(BB->PatCopyMode, *lpPattern, OpsArray[2]);
        }
    }

    /**************************************************************************/
    /*  Get the next target DWORD.                                            */
    /**************************************************************************/
    OpsArray[1] = TrgDword = *((PULONG)TrgIndex);

    /**************************************************************************/
    /*  Do the Rop for the current DWORD in the target bitmap.                */
    /**************************************************************************/
    NewTrgDword = prdb_PerformRop(OpsArray, BB);

    /**************************************************************************/
    /*  Get the mask for the last DWORD                                       */
    /**************************************************************************/
    ulMask = BB->EndMask;

    /**************************************************************************/
    /*  Store the new target DWORD.                                           */
    /**************************************************************************/
/* CON3201
    *((PULONG)TrgIndex) = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);
*/
    *TrgIndex = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_RopManyDwords                                              */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function performs the rop for greater than one DWORD.                */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_RopManyDwords( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    USHORT  usCount;                   /* number of dwords to blt             */
 /* ULONG   ulTrgIndex; */             /* index to next target dword          */
    PULONG  ulTrgIndex;                /* index to next target dword  CON3201 */
    ULONG   TrgDword;                  /* current target dword                */
    ULONG   NewTrgDword;               /* new target dword after the rop      */
    ULONG   OpsArray[3];               /* src, trg and pattern dwords         */
    BOOL    fSrcReq;                   /* source required                     */
    BOOL    fPatReq;                   /* pattern required                    */
    PULONG  lpSource;                  /* pointer to source dwords            */
    PULONG  lpPattern;                 /* pointer to pattern dwords           */
    ULONG   ulMask;                    /* start and end mask pointers         */
    USHORT  usBltFlags;                /* useful flags                        */
    USHORT  usFastFlags;               /* useful flags                        */

    /**************************************************************************/
    /*  BUNNIES + BBDWORD : Add in extra local variables for speed.           */
    /**************************************************************************/
    fSrcReq     = (BOOL)BB->Source;
    fPatReq     = BB->PatternReq;
    usBltFlags  = BB->usBltFlags;
    usFastFlags = BB->usFastFlags;
 /* ulTrgIndex  = BB->TrgIndex;  */
    ulTrgIndex  = (PULONG)BB->TrgIndex;                            /* CON3201 */
    usCount     = BB->usTrgDwords;
    if (fSrcReq)
    {

        /**********************************************************************/
        /*  Get the source pointer and adjust it according to the copy mode.  */
        /**********************************************************************/
        lpSource = (PULONG)BB->abSrcRow;
        prdm_GetDword(BB->SrcCopyMode, *lpSource, OpsArray[0]);
    }
    if (fPatReq)
    {

        /**********************************************************************/
        /*  Get the pattern pointer and adjust it according to the copy mode. */
        /**********************************************************************/
        lpPattern = (PULONG)BB->abPatRow;
        prdm_GetDword(BB->PatCopyMode, *lpPattern, OpsArray[2]);

        /**********************************************************************/
        /*  If there is only one pattern DWORD then the pattern is essentially*/
        /*  not required again.                                               */
        /**********************************************************************/
        if (!(usBltFlags & BLT_PAT_ROW_REQD))
        {
           fPatReq = FALSE;
        }
    }

    /**************************************************************************/
    /*  Rop the first DWORD.  Get the First target DWORD.                     */
    /**************************************************************************/
    OpsArray[1] = TrgDword = *((PULONG)ulTrgIndex);

    /**************************************************************************/
    /*  Do the Rop for the current DWORD in the target bitmap.                */
    /**************************************************************************/
    NewTrgDword = prdb_PerformRop(OpsArray, BB);

    /**************************************************************************/
    /*  Initiate the mask.                                                    */
    /**************************************************************************/
    ulMask = BB->StartMask;

    /**************************************************************************/
    /*  Store the new target DWORD.                                           */
    /**************************************************************************/
/* CON3201
    *((PULONG)ulTrgIndex) = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);
*/
    *ulTrgIndex = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);

    /**************************************************************************/
    /*  There are N DWORDs to do, and N is greater than 2.  We've done one,   */
    /*  and one is the last one, so the number to do in the loop is ...       */
    /**************************************************************************/
    usCount -=2;

    /**************************************************************************/
    /*  There is no mask, just N DWORDS to write.  This can be fast- pathed.  */
    /*  Currently only fast path PATCOPY when there is one DWORD to be blted. */
    /**************************************************************************/
    if ((BB->Mix == ROP_PATCOPY) && (BB->usFastFlags & TRG_ROW_IN_SEG) &&
        (!(BB->usBltFlags & BLT_PAT_ROW_REQD)))
    {

        /**********************************************************************/
        /*  Set the DWORD into the target.                                    */
        /**********************************************************************/
     /* (VOID)prdu_dwordset((PBYTE)(ulTrgIndex + 4), (PBYTE)&(OpsArray[2]), */
                                                                   /* CON3201 */
        (VOID)prdu_dwordset((PBYTE)(ulTrgIndex) + 4, (PBYTE)&(OpsArray[2]),
                            usCount);

        /**********************************************************************/
        /*  Move trgIndex to the last DWORD                                   */
        /**********************************************************************/
/* CON3201
        LOUSHORT(ulTrgIndex) = LOUSHORT(BB->TrgEndIndex);
*/
        ulTrgIndex = (PULONG)(BB->TrgBM + BB->TrgEndIndex);
        goto ROP_LAST_DWORD;
    }

    /**************************************************************************/
    /*  Long path for combining ROPs.  Still much quicker than the old method.*/
    /**************************************************************************/
    while (TRUE)
    {

        /**********************************************************************/
        /*  Move to the next DWORD                                            */
        /**********************************************************************/
/*      LOUSHORT(ulTrgIndex) += 4;                                    CON3201 */
        ulTrgIndex ++;

        /**********************************************************************/
        /*  If this index is now zero then we wrapped to the next segment, so */
        /*  increment the segment pointer.                                    */
        /**********************************************************************/
     /* CON3201 **************************************************/
     /* Don't worry about segments                               */
     /* if (!LOUSHORT(ulTrgIndex))                               */
     /* {                                                        */
     /*     SELECTOROF(ulTrgIndex) += LOUSHORT(prdd_HugeInc);    */
     /* }                                                        */
     /************************************************************/
        if (usCount--)
        {

            /******************************************************************/
            /*  Ok, get the DWORDs and do the blt.                            */
            /******************************************************************/
            if (fSrcReq)
            {

                /**************************************************************/
                /*  Get the source pointer                                    */
                /**************************************************************/
                lpSource++;
                prdm_GetDword(BB->SrcCopyMode, *lpSource, OpsArray[0]);
            }
            if (fPatReq)
            {

                /**************************************************************/
                /*  Get the pattern pointer                                   */
                /**************************************************************/
                lpPattern++;
                prdm_GetDword(BB->PatCopyMode, *lpPattern, OpsArray[2]);
            }

            /******************************************************************/
            /*  Rop the first DWORD.  Get the First target DWORD.             */
            /******************************************************************/
            OpsArray[1] = *((PULONG)ulTrgIndex);

            /******************************************************************/
            /*  Store the new target DWORD.                                   */
            /******************************************************************/
/*          *((PULONG)ulTrgIndex) = prdb_PerformRop(OpsArray, BB);    CON3201 */
            *ulTrgIndex = prdb_PerformRop(OpsArray, BB);
        }
        else
        {

            /******************************************************************/
            /*  We've finished, just exit from the loop                       */
            /******************************************************************/
            break;
        }
    }

ROP_LAST_DWORD:

    /**************************************************************************/
    /*  Do the last DWORD.                                                    */
    /**************************************************************************/
    if (fSrcReq)
    {

        /**********************************************************************/
        /*  Get the source pointer                                            */
        /**********************************************************************/
        lpSource++;
        prdm_GetDword(BB->SrcCopyMode, *lpSource, OpsArray[0]);
    }
    if (fPatReq)
    {

        /**********************************************************************/
        /*  Get the pattern pointer                                           */
        /**********************************************************************/
        lpPattern++;
        prdm_GetDword(BB->PatCopyMode, *lpPattern, OpsArray[2]);
    }

    /**************************************************************************/
    /*  Rop the first DWORD.  Get the First target DWORD.                     */
    /**************************************************************************/
    OpsArray[1] = TrgDword = *((PULONG)ulTrgIndex);

    /**************************************************************************/
    /*  Do the Rop for the current DWORD in the target bitmap.                */
    /**************************************************************************/
    NewTrgDword = prdb_PerformRop(OpsArray,BB);

    /**************************************************************************/
    /*  Initiate the mask.                                                    */
    /**************************************************************************/
    ulMask = BB->EndMask;

    /**************************************************************************/
    /*  Store the new target DWORD.                                           */
    /**************************************************************************/
/* CON3201
    *((PULONG)ulTrgIndex) = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);
*/
    *ulTrgIndex = (NewTrgDword & ulMask) | (TrgDword & ~ulMask);

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_DoQuickRop                                                 */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;     Pointer to Bitblt working storage                     */
/*  USHORT      usRop;                                                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function implements some ROPS very fast.  In particular:             */
/*                                                                            */
/*  - ROP_ZEROS                                                               */
/*  - ROP_ONES                                                                */
/*  - ROP_PATCOPY (with pattern all zeros or all ones)                        */
/*                                                                            */
/*  PD00406: Created.                                                         */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_DoQuickRop( ULONG       ulNewTrgDword,
                      lpBltParms  BB )

{

    /**************************************************************************/
    /*  Local variables.                                                      */
    /**************************************************************************/
    USHORT  usCount;                   /* number of DWORDs or BYTES           */
 /* ULONG   ulTrgIndex; */             /* index of first DWORD                */
    PULONG  ulTrgIndex;                /* index of first DWORD       CON3201  */
    ULONG   ulTrgDword;                /* current target DWORD                */
    ULONG   ulMask;                    /* mask to put DWORDs into target      */

    /**************************************************************************/
    /*  Initialize                                                            */
    /**************************************************************************/
 /* ulTrgIndex = BB->TrgIndex; */
    ulTrgIndex = (PULONG)(BB->TrgIndex);                           /* CON3201 */
    usCount    = BB->usTrgDwords;

    /**************************************************************************/
    /*  Do the first DWORD                                                    */
    /**************************************************************************/
    ulMask     = BB->StartMask;
    ulTrgDword = *(PULONG)ulTrgIndex;
/* CON3201
    *((PULONG)ulTrgIndex) = (ulNewTrgDword & ulMask) | (ulTrgDword & ~ulMask);
*/
    *ulTrgIndex = (ulNewTrgDword & ulMask) | (ulTrgDword & ~ulMask);

    /**************************************************************************/
    /*  Do the middle DWORDs.  Two cases depending on whether the row crosses */
    /*  a segment boundary.                                                   */
    /**************************************************************************/
/*  if (BB->usFastFlags & TRG_ROW_IN_SEG)                  kran               */
    {

        /**********************************************************************/
        /*  Use a memset for these.                                           */
        /**********************************************************************/
        usCount  -= 2;
        usCount <<= 2;
     /* OFFSETOF(ulTrgIndex) += 4; */
        ulTrgIndex++;                                              /* CON3201 */
        prdu_memset((PBYTE)ulTrgIndex, LOBYTE(LOUSHORT(ulNewTrgDword)),
                    usCount);
     /* OFFSETOF(ulTrgIndex) += usCount; */
        ulTrgIndex = (PULONG)((PBYTE)ulTrgIndex + usCount);        /* CON3201 */
    }
    /* CON3201
    ** Should never go into this ELSE since usFastFlags will be set
    */
/*  else                                                     kran             */
    {

// CON3201     /**********************************************************************/
// CON3201     /*  Loop around writing the new DWORDs, but getting ready to move to  */
// CON3201     /*  the new segment.                                                  */
// CON3201     /**********************************************************************/
// CON3201     usCount -= 2;
// CON3201     while (TRUE)
// CON3201     {
// CON3201
// CON3201         /******************************************************************/
// CON3201         /*  Move to the next DWORD                                        */
// CON3201         /******************************************************************/
// CON3201         OFFSETOF(ulTrgIndex) += 4;
// CON3201         if (!OFFSETOF(ulTrgIndex))
// CON3201         {
// CON3201
// CON3201             /**************************************************************/
// CON3201             /*  This has wrapped, so move to the next segment             */
// CON3201             /**************************************************************/
// CON3201             SELECTOROF(ulTrgIndex) += prdd_HugeInc;
// CON3201         }
// CON3201
// CON3201         /******************************************************************/
// CON3201         /*  Now check if we have finished yet                             */
// CON3201         /******************************************************************/
// CON3201         if (usCount--)
// CON3201         {
// CON3201
// CON3201             /**************************************************************/
// CON3201             /*  This is a middle DWORD, so we have no need to worry about */
// CON3201             /*  the mask at all                                           */
// CON3201             /**************************************************************/
// CON3201             *((PULONG)ulTrgIndex) = ulNewTrgDword;
// CON3201         }
// CON3201         else
// CON3201         {
// CON3201
// CON3201             /**************************************************************/
// CON3201             /*  Have finished here, so just do the last DWORD and we're   */
// CON3201             /*  done.                                                     */
// CON3201             /**************************************************************/
// CON3201             break;
// CON3201  }
//      }
    }

EXIT_FUNCTION:

    /**************************************************************************/
    /*  The DWORD is the last on this row so use the End mask then            */
    /**************************************************************************/
    ulMask = BB->EndMask;
    ulTrgDword = *(PULONG)ulTrgIndex;
/* CON3201
    *((PULONG)ulTrgIndex) = (ulNewTrgDword & ulMask) | (ulTrgDword & ~ulMask);
*/
    *ulTrgIndex = (ulNewTrgDword & ulMask) | (ulTrgDword & ~ulMask);

    /**************************************************************************/
    /*  and finished                                                          */
    /**************************************************************************/
    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_DoQuickBlt                                                 */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;     Pointer to Bitblt working storage                     */
/*  USHORT      usRop;                                                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function implements some ROPS very fast.  In particular:             */
/*                                                                            */
/*  - ROP_ZEROS                                                               */
/*  - ROP_ONES                                                                */
/*  - ROP_PATCOPY (with pattern all zeros or all ones)                        */
/*  - ROP_DSTINVERT                                                           */
/*                                                                            */
/*  If the ROP is not one of these then it return FALSE, otherwise it returns */
/*  TRUE and has done the ROP.                                                */
/*                                                                            */
/*  PD00406: Created.                                                         */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_DoQuickBlt( BYTE        cByte,
                      BYTE        cByteValid,
                      lpBltParms  BB )

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT  usTrgBytes;                /* number of bytes per row             */
    USHORT  usNumRows;                 /* number of rows                      */
    USHORT  usStart;                   /* Y index of current row in pat       */
    ULONG   ulSaveTrgIndex;            /* address of start of row             */
    ULONG   ulMask;                    /* mask for first and last DWORDS      */
    ULONG   ulTrgDword;                /* current target DWORD                */
    ULONG   ulNewTrgDword;             /* new target DWORD                    */
    BYTE    abNewTrgDword[4];          /* array for target DWORD              */
    PULONG  pulTrgIndex;               /* need a PULONG for address  CON3201  */

    /**************************************************************************/
    /*  Local variables for speed.                                            */
    /**************************************************************************/
    usTrgBytes = (USHORT)((BB->TrgLE)->Parms.BytesPerRow);
    usNumRows  = (BB->TrgRect)[1].Y - (BB->TrgRect)[0].Y + 1;
    if (cByteValid == BLT_USE_BYTE)
    {

        /**********************************************************************/
        /*  Expand the byte supplied into a DWORD.                            */
        /**********************************************************************/
        prdu_memset((PBYTE)&ulNewTrgDword, cByte, 4);
    }

    /**************************************************************************/
    /*  Repeat for each row in the blt rectangle...                           */
    /**************************************************************************/
    while (usNumRows--)
    {

        /**********************************************************************/
        /*  If the pattern is 8 bytes wide, but differs for each row then we  */
        /*  need to get hold of the correct DWORD                             */
        /**********************************************************************/
        if (cByteValid != BLT_USE_BYTE)
        {

            /******************************************************************/
            /*  Get the index into the pattern.                               */
            /******************************************************************/
            usStart  = BB->TrgStartPos.Y - BB->PatOrigin.Y;
            usStart %= BB->PatHeight;

            /******************************************************************/
            /*  Byte is either used as is or is inverted.                     */
            /******************************************************************/
            if (cByteValid == BLT_USE_PAT)
            {
                cByte = *(BB->PatBM + BB->PatBytesPerRow * usStart);
            }
            else
            {
                cByte = ~*(BB->PatBM + BB->PatBytesPerRow * usStart);
            }

            /******************************************************************/
            /*  construct the dword.                                          */
            /******************************************************************/
            prdu_memset((PBYTE)&ulNewTrgDword, cByte, 4);
        }

        /**********************************************************************/
        /*  TrgIndex is byte indices at this point - save their current values*/
        /*  so that we can "freshen up" SrcIndex and TrgIndex with byte       */
        /*  indices at the end of the loop.                                   */
        /**********************************************************************/
        ulSaveTrgIndex = BB->TrgIndex;

        /**********************************************************************/
        /*  Get some flags to speed up certain paths later on.                */
        /**********************************************************************/
        BB->usFastFlags = prdb_GetFastFlags(BB);

        /**********************************************************************/
        /*  Processing as for prdb_DoBitBlt.                                  */
        /**********************************************************************/
    /* CON3201 *******************************************************/
    /*  SELECTOROF(BB->TrgIndex) = SELECTOROF(BB->TrgBM) +           */
    /*                             (SELECTOROF(BB->TrgIndex) *       */
    /*                              LOUSHORT(prdd_HugeInc));         */
    /*****************************************************************/

        BB->TrgIndex = (ULONG)BB->TrgBM + BB->TrgIndex; /* CON3201 */

        /**********************************************************************/
        /*  Do the ROP.                                                       */
        /**********************************************************************/
        if (BB->usTrgDwords == 1)
        {

            /******************************************************************/
            /*  Just one DWORD to work with so do it here.                    */
            /******************************************************************/
            ulMask     = BB->StartMask & BB->EndMask;
            ulTrgDword = *(PULONG)BB->TrgIndex;
/*          *((PULONG)BB->TrgIndex) = (ulNewTrgDword & ulMask) |      CON3201 */
/*                                    (ulTrgDword & ~ulMask);         CON3201 */
            pulTrgIndex = (PULONG)(BB->TrgIndex);                  /* CON3201 */
            *pulTrgIndex = (ulNewTrgDword & ulMask) |              /* CON3201 */
                           (ulTrgDword & ~ulMask);
        }
        else
        {
            if (BB->usTrgDwords == 2)
            {

                /**************************************************************/
                /*  Just two DWORDs to work with so do then here.             */
                /**************************************************************/
                ulMask     = BB->StartMask;
                ulTrgDword = *(PULONG)BB->TrgIndex;
              /* CON3201 *********************************************/
              /**((PULONG)BB->TrgIndex) = (ulNewTrgDword & ulMask) | */
              /*                          (ulTrgDword & ~ulMask);    */
              /*******************************************************/
                pulTrgIndex = (PULONG)(BB->TrgIndex);              /* CON3201 */
                *pulTrgIndex = (ulNewTrgDword & ulMask) |          /* CON3201 */
                                     (ulTrgDword & ~ulMask);

                /**************************************************************/
                /*  Move to the next                                          */
                /**************************************************************/
/*              if (BB->usFastFlags & TRG_ROW_IN_SEG)           kran         */
                {
/*                  OFFSETOF(BB->TrgIndex) += 4;                     CON3201 */
                    BB->TrgIndex += 4;
/*                  BB->TrgIndex++;    I think Kranthi screwed up            */
                }
                /* CON3201
                ** Should never go into this ELSE since usFastFlags will be set
                */
/*              else                                              kran       */
                {
// CON3201          SELECTOROF(BB->TrgIndex) += LOUSHORT(prdd_HugeInc);
// CON3201          OFFSETOF(BB->TrgIndex) = 0;
                }

                /**************************************************************/
                /*  Do the second (and last) DWORD                            */
                /**************************************************************/
                ulMask     = BB->EndMask;
                ulTrgDword = *(PULONG)BB->TrgIndex;

// CON3201      *((PULONG)BB->TrgIndex) = (ulNewTrgDword & ulMask) |
// CON3201                                (ulTrgDword & ~ulMask);

                pulTrgIndex = (PULONG)(BB->TrgIndex);              /* CON3201 */
                *pulTrgIndex = (ulNewTrgDword & ulMask) |          /* CON3201 */
                               (ulTrgDword & ~ulMask);             /* CON3201 */
            }
            else
            {

                /**************************************************************/
                /*  Do this in another function.                              */
                /**************************************************************/
                (VOID)prdb_DoQuickRop(ulNewTrgDword, BB);
            }
        }

        /**********************************************************************/
        /*  Update start positions for the target row.                        */
        /**********************************************************************/
        if (BB->VertDir == DOWN)
        {

            /******************************************************************/
            /*  Move to the next target position                              */
            /******************************************************************/
            BB->TrgStartPos.Y --;
            BB->TrgIndex     = ulSaveTrgIndex - usTrgBytes;
            BB->TrgEndIndex -= usTrgBytes;
        }
        else
        {

            /******************************************************************/
            /*  Move to the next target position                              */
            /******************************************************************/
            BB->TrgStartPos.Y ++;
            BB->TrgIndex     = ulSaveTrgIndex + usTrgBytes;
            BB->TrgEndIndex += usTrgBytes;
        }
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/

EXIT_FUNCTION:

    return;
}

/******************************************************************************/
/*  FUNCTION: prdb_QuickBltPossible                                           */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function descides if the ROPS can be done very fast.  The following  */
/*  ROPS may be implemented fast.                                             */
/*                                                                            */
/*  - ROP_ZEROS                                                               */
/*  - ROP_ONES                                                                */
/*  - ROP_PATCOPY (with pattern all zeros or all ones)                        */
/*                                                                            */
/*  If the ROP is not one of these then it return FALSE, otherwise it returns */
/*  TRUE and has done the ROP.                                                */
/*                                                                            */
/*  PD00406: Created.                                                         */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
USHORT prdb_QuickBltPossible( lpBltParms BB )

{

    /**************************************************************************/
    /*  Local variables                                                       */
    /**************************************************************************/
    USHORT  usResult;                  /* function return                     */
    USHORT  usArgMix;                  /* bitblt ROP                          */
    USHORT  usBltFlags;                /* copy of blt flags                   */
    USHORT  usPatCopyMode;             /* copy of pattern copy mode           */
    USHORT  usStart;                   /* y co-ordinate of current row        */
    BYTE    cPatByte;                  /* byte to go in the pattern           */

    /**************************************************************************/
    /*  set up variables for speed                                            */
    /**************************************************************************/
    usResult = FALSE;
    usArgMix = LOUSHORT(BB->Mix);

    /**************************************************************************/
    /*  PD00486 : First ensure that we're dealing with a 1,1 bitmap.  If not, */
    /*  bail out.                                                             */
    /**************************************************************************/
    if (LOUSHORT(BB->TrgLE->Parms.Bitcount) != 1)
        goto EXIT_FUNCTION;

    /**************************************************************************/
    /*  Try and return as quick as possible.                                  */
    /**************************************************************************/
    if ((usArgMix != ROP_ZERO) && (usArgMix != ROP_ONE) &&
        (usArgMix != ROP_PATCOPY))
    {

        /**********************************************************************/
        /*  Definitely can't do anything                                      */
        /**********************************************************************/
        goto EXIT_FUNCTION;
    }

    /**************************************************************************/
    /*  Pattern copies can be fast if the pattern is all ones or all zeroes,  */
    /*  or the pattern colors map to all ones or all zeros.                   */
    /**************************************************************************/
    if (usArgMix == ROP_PATCOPY)
    {

        /**********************************************************************/
        /*  Some local variables                                              */
        /**********************************************************************/
        usBltFlags    = BB->usBltFlags;
        usPatCopyMode = BB->PatCopyMode;

        /**********************************************************************/
        /*  May be able to blt ones                                           */
        /**********************************************************************/
        if ((usBltFlags & PAT_ALL_ONES) || (usPatCopyMode == PAT_ONES))
        {

            /******************************************************************/
            /*  Just blt ones (or zeros if this is an inversion).             */
            /******************************************************************/
            if (usPatCopyMode == PAT_INVERT_BITS)
            {
                (VOID)prdb_DoQuickBlt(ROP_ZERO, BLT_USE_BYTE, BB);
            }
            else
            {
                (VOID)prdb_DoQuickBlt(ROP_ONE, BLT_USE_BYTE, BB);
            }
            usResult = TRUE;
            goto EXIT_FUNCTION;
        }

        /**********************************************************************/
        /*  May be able to blt zeros                                          */
        /**********************************************************************/
        if ((usBltFlags & PAT_ALL_ZEROS) || (usPatCopyMode == PAT_ZEROS))
        {

            /******************************************************************/
            /*  Just blt zeroes (or ones if this is an inversion).            */
            /******************************************************************/
            if (usPatCopyMode == PAT_INVERT_BITS)
            {
                (VOID)prdb_DoQuickBlt(ROP_ONE, BLT_USE_BYTE, BB);
            }
            else
            {
                (VOID)prdb_DoQuickBlt(ROP_ZERO, BLT_USE_BYTE, BB);
            }
            usResult = TRUE;
            goto EXIT_FUNCTION;
        }
        if (BB->PatWidth == 8)
        {

            /******************************************************************/
            /*  Eight bit pattern (ie probably greyscaling), can memset just  */
            /*  the pattern byte.  Need to get the actual byte required       */
            /*  though.                                                       */
            /******************************************************************/
            usStart  = BB->TrgStartPos.Y - BB->PatOrigin.Y;
            usStart %= BB->PatHeight;
            cPatByte = *(BB->PatBM + BB->PatBytesPerRow * usStart);

            /******************************************************************/
            /*  Now do the blt                                                */
            /******************************************************************/
            if (usPatCopyMode == PAT_INVERT_BITS)
            {
                (VOID)prdb_DoQuickBlt(0, BLT_USE_INV_PAT, BB);
            }
            else
            {
                (VOID)prdb_DoQuickBlt(0, BLT_USE_PAT, BB);
            }

            /******************************************************************/
            /*  Now exit                                                      */
            /******************************************************************/
            usResult = TRUE;
            goto EXIT_FUNCTION;
        }
    }

    /**************************************************************************/
    /*  Other options are ROP_ZERO or ROP_ONE                                 */
    /**************************************************************************/
    if ((usArgMix == ROP_ZERO) || (usArgMix == ROP_ONE))
    {
        usResult = TRUE;
        (VOID)prdb_DoQuickBlt(LOBYTE(usArgMix), BLT_USE_BYTE, BB);
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/

EXIT_FUNCTION:

    return(usResult);
}

/******************************************************************************/
/*  FUNCTION: prdb_DoBitBlt                                                   */
/*                                                                            */
/*  PARAMETERS:                                                               */
/*                                                                            */
/*  lpBltParms  BB;  Pointer to Bitblt working storage                        */
/*                                                                            */
/*  DESCRIPTION:                                                              */
/*                                                                            */
/*  This function performs the blt (only after high level checks have been    */
/*  done).  The blt is done by a series of calls to the sub-functions above   */
/*  with updating of the source and target positions after each line.         */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
VOID prdb_DoBitBlt( lpBltParms  BB,
                    lpDCI       DCIData )

{

#define TFUNC "prdb_DoBitBlt"

    /**************************************************************************/
    /*  Local Variables                                                       */
    /**************************************************************************/
    SHORT   NoOfRows;                  /* Loop control variable               */
    USHORT  SrcBytesPerRow;            /* Source row width in bytes           */
    USHORT  TrgBytesPerRow;            /* Target row width in bytes           */
    ULONG   SaveSrcIndex;              /* Save area for start index           */
    ULONG   SaveTrgIndex;              /* of source and target rows           */
    USHORT  usNumDwords;
    pBMListEntry  NewEntry;            /* PD00567 : Pointer to new list entry */
    BitmapParms   NewParms;            /* PD00567 : Bitmap info header        */
    BOOL          fGrey;               /* PD00567 : Greyscale flag            */

    /**************************************************************************/
    /*  Init a few things...                                                  */
    /**************************************************************************/
    fGrey = FALSE;                     /* Memory management flag...           */

    /**************************************************************************/
    /*  PD00567 : We need to convert the IBM4019 4,1 bitmap to 1,1 format here*/
    /*  before we try to do much else.  This will involve the creation of a   */
    /*  1,1 internal bitmap and conversion to 1,1 internal.  This is so that  */
    /*  all of the subsequent BitBlt code will work as originally intended in */
    /*  terms of the 1,1 bitmap instead of the 4,1 bitmap.  We'll do the      */
    /*  following:                                                            */
    /*                                                                        */
    /*    - call prdb_CreateBitmap()                                          */
    /*    - call prdb_Convert_Int4_to_Int1()                                  */
    /*    - switch pointers for the bitmaps                                   */
    /*    - cross our fingers for good luck and let 'er rip                   */
    /*                                                                        */
    /*  With any luck, we'll get a good 1,1 bitmap and we can then fall       */
    /*  through the remainder of the processing as we normally would.         */
    /*                                                                        */
    /*  Note that if the target bitmap is a 4,1 bitmap, this implies that the */
    /*  app has not created a target bitmap and is intending to blit directly */
    /*  to the printer's page band.  In the process, the blit has been        */
    /*  returned to the engine for a simulation and the engine has created a  */
    /*  'target' bitmap as well as an intermediate stretch bitmap.  The engine*/
    /*  has then called us to blit the source to the target and that's where  */
    /*  we are now.  We need to let the blit go on unchanged as the engine    */
    /*  will call us again once the bitmap has been stretched.  At that point */
    /*  we will have the stretched 4,1 bitmap as the source and we will have  */
    /*  our printer's page band bitmap, a 1,1 bitmap, as the target.  Then we */
    /*  get to greyscale the source and finally blit to the 1,1 target.       */
    /**************************************************************************/
    if ((BB->Source != NO_SOURCE) && (BB->SrcLE->ColorDetected) &&
        (BB->TrgLE->Parms.Bitcount == 1))
    {

        /**********************************************************************/
        /*  Need to fill in some pertinent information concerning the new     */
        /*  source bitmap...                                                  */
        /**********************************************************************/
        NewParms.Width       = BB->SrcLE->Parms.Width;
        NewParms.Height      = BB->SrcLE->Parms.Height;
        NewParms.Planes      = 1;
        NewParms.Bitcount    = 1;
        NewParms.MonBmapReq  = FALSE;
        NewParms.BytesPerRow = BB->SrcLE->Parms.BytesPerRow / 4;

        /**********************************************************************/
        /*  We'll call prdb_CreateBitmap() directly since we know that the    */
        /*  parameters are OK (no need to call prdb_DeviceCreateBitmap() and  */
        /*  have all the parameters checked...).                              */
        /**********************************************************************/
        if (!(NewEntry = prdb_CreateBitmap(DCIData, FALSE, &NewParms)))
        {

            /******************************************************************/
            /*  We have a real problem... what do we do here?                 */
            /******************************************************************/
            TRACE4(TFUNC, "CreateBitmap failed!", FNULL, 0);
            goto EXIT_FUNCTION;
        }
        else
        {

            /******************************************************************/
            /*  Everything's OK so far.  We need to fill in a few more fields */
            /*  in NewEntry and set the fGrey flag so we can deallocate the   */
            /*  new ListEntry and the bitmap at the end of the blit.          */
            /******************************************************************/
            fGrey                   = TRUE;
            NewEntry->XOrigin       = BB->SrcLE->XOrigin;
            NewEntry->YOrigin       = BB->SrcLE->YOrigin;
            NewEntry->ColBmapReq    = BB->SrcLE->ColBmapReq;
            NewEntry->ColorDetected = TRUE;

            /******************************************************************/
            /*  OK, now for the fun.  We know that we have a 4,1 bitmap that  */
            /*  is described in full by BB->SrcLE.  We also know we should    */
            /*  have a sufficiently large 1,1 bitmap described in full by     */
            /*  NewEntry.  We now need to convert this 4,1 bitmap to 1,1      */
            /*  format and replace the references to the 4,1 ListEntry and the*/
            /*  4,1 bitmap with the new 1,1 ListEntry and 1,1 bitmap.  During */
            /*  the conversion process, we'll greyscale any colors encountered*/
            /*  as we're converting the 4,1 pels.                             */
            /******************************************************************/
            prdb_GreyScaleSourceBitmap(BB, NewEntry, DCIData);
        }
    }

    /**************************************************************************/
    /*  Get starting position and processing direction.                       */
    /**************************************************************************/
    (VOID)prdb_StartPosAndDir(BB);

    /**************************************************************************/
    /*  Set up any masks needed.                                              */
    /**************************************************************************/
    (VOID)prdb_GetMasks(BB);

    /**************************************************************************/
    /*  Get start and end indices for source and target rows.                 */
    /**************************************************************************/
    (VOID)prdb_GetIndices(BB);

    /**************************************************************************/
    /*  There may be some quick blts that we can do.  If so this function will*/
    /*  have done them and returned TRUE, so just exit.                       */
    /*                                                                        */
    /*  Should be able to do something with the 4,1 case... need to think it  */
    /*  through though... (currently we bail out immediately if we're in a    */
    /*  4,1 bitmap)                                                           */
    /**************************************************************************/
    if (prdb_QuickBltPossible(BB))
    {
        goto EXIT_FUNCTION;
    }

    /**************************************************************************/
    /*  Local variables for speed.                                            */
    /**************************************************************************/
    TrgBytesPerRow = (USHORT)((BB->TrgLE)->Parms.BytesPerRow);
    NoOfRows       = (BB->TrgRect)[1].Y - (BB->TrgRect)[0].Y + 1;
    usNumDwords    = BB->usTrgDwords;
    if (BB->Source != NO_SOURCE)
    {
        SrcBytesPerRow = (USHORT)((BB->SrcLE)->Parms.BytesPerRow);
    }

    /**************************************************************************/
    /*  Repeat for each row in the blt rectangle...                           */
    /**************************************************************************/
    while (NoOfRows--)
    {

        /**********************************************************************/
        /*  SrcIndex and TrgIndex are byte indices at this point - save their */
        /*  current values so that we can "freshen up" SrcIndex and TrgIndex  */
        /*  with byte indices at the end of the loop.                         */
        /**********************************************************************/
        SaveSrcIndex = BB->SrcIndex;
        SaveTrgIndex = BB->TrgIndex;

        /**********************************************************************/
        /*  Get some flags to speed up certain paths in rop dwords.           */
        /**********************************************************************/
        BB->usFastFlags = prdb_GetFastFlags(BB);

        /**********************************************************************/
        /*  BBDWORD : Always store pointer in (Src/Trg)Index.                 */
        /*                                                                    */
        /*  Store the segment address in the high word of the index (SrcIndex */
        /*  or TrgIndex) - the index can then be used directly as a pointer.  */
        /*  It is adjusted as a pointer (i.e.  allowing for crossing segment  */
        /*  boundaries) within prdb_RopDwords and prdb_GetSrcDword.           */
        /*                                                                    */
        /*  After processing the row (Src/Trg)Index will be restored back to a*/
        /*  byte index with (Src/Trg)SaveIndex.                               */
        /**********************************************************************/
    /* CON3201 ****************************************************************/
    /*  SELECTOROF(BB->TrgIndex) = SELECTOROF(BB->TrgBM) +                    */
    /*                             (SELECTOROF(BB->TrgIndex) *                */
    /*                              LOUSHORT(prdd_HugeInc));                  */
    /**************************************************************************/
        BB->TrgIndex = (ULONG)BB->TrgBM + BB->TrgIndex;
        if (BB->Source != NO_SOURCE)
        {

            /******************************************************************/
            /*  Get the starting source address.                              */
            /******************************************************************/
        /* CON3201 ************************************************************/
        /*  SELECTOROF(BB->SrcIndex) = SELECTOROF(BB->SrcBM) +                */
        /*                             (SELECTOROF(BB->SrcIndex) *            */
        /*                              LOUSHORT(prdd_HugeInc));              */
        /**********************************************************************/
            BB->SrcIndex = (ULONG)BB->SrcBM + BB->SrcIndex; /* CON3201 */

            /******************************************************************/
            /*  Get the source row shifted to the correct position.           */
            /******************************************************************/
            (VOID)prdb_GetSourceRow(BB);
        }

        /**********************************************************************/
        /*  BUNNIES: Deal with the pattern: work out which row in the pattern */
        /*  we are using and get the index of the first pattern pel in that   */
        /*  row to use.  Get a pointer to the start of the correct row in the */
        /*  pattern.  PD00404 : Only do this work if a pattern is required.   */
        /**********************************************************************/
        if (BB->PatternReq)
        {
            (VOID)prdb_GetPatternRow(BB);
        }

        /**********************************************************************/
        /*  Special case one, two and many, allowing the many to go as fast as*/
        /*  possible.                                                         */
        /**********************************************************************/
        if (usNumDwords > 2)
        {
            (VOID)prdb_RopManyDwords(BB);
        }
        else
        {
            if (usNumDwords == 2)
            {
                (VOID)prdb_RopTwoDwords(BB);
            }
            else
            {
                (VOID)prdb_RopOneDword(BB);
            }
        }

        /**********************************************************************/
        /*  Update start positions and indexes for the source and target      */
        /*  bitmaps depending on which direction we are doing the blt in.     */
        /**********************************************************************/
        if (BB->VertDir == DOWN)
        {

            /******************************************************************/
            /*  Move to the next target position                              */
            /******************************************************************/
            BB->TrgStartPos.Y --;
            BB->TrgIndex     = SaveTrgIndex - TrgBytesPerRow;
            BB->TrgEndIndex -= TrgBytesPerRow;

            /******************************************************************/
            /*  Move to the next source position                              */
            /******************************************************************/
            if (BB->Source != NO_SOURCE)
            {
                BB->SrcStartPos.Y --;
                BB->SrcIndex     = SaveSrcIndex - SrcBytesPerRow;
                BB->SrcEndIndex -= SrcBytesPerRow;
            }
        }
        else
        {

            /******************************************************************/
            /*  Move to the next target position                              */
            /******************************************************************/
            BB->TrgStartPos.Y ++;
            BB->TrgIndex     = SaveTrgIndex + TrgBytesPerRow;
            BB->TrgEndIndex += TrgBytesPerRow;

            /******************************************************************/
            /*  Move to the next source position                              */
            /******************************************************************/
            if (BB->Source != NO_SOURCE)
            {
                BB->SrcStartPos.Y ++;
                BB->SrcIndex     = SaveSrcIndex + SrcBytesPerRow;
                BB->SrcEndIndex += SrcBytesPerRow;
            }
        }
    }

    /**************************************************************************/
    /*  And finished                                                          */
    /**************************************************************************/

EXIT_FUNCTION:

    /**************************************************************************/
    /*  PD00567 : Well, if we've created a new bitmap to handle the source    */
    /*  bitmap greyscaling, we need to get rid of both the bitmap and the     */
    /*  ListEntry.  As the functions that require the BB structure and it's   */
    /*  contents are done, we can simply deallocate both the bitmap and the   */
    /*  SrcLE.                                                                */
    /**************************************************************************/
    if (fGrey)
    {
        (VOID)prdb_FreeBitmap((PBYTE)BB->SrcLE->Bitmap);
        (VOID)prdb_DeleteBitmap(BB->SrcLE);
    }

    return;
}
#undef TFUNC

/******************************************************************************/
/*                                                                            */
/*   FUNCTION: prdb_BitBlt                                                    */
/*                                                                            */
/*   PARAMETERS:                                                              */
/*                                                                            */
/*   See "OS/2 Technical Reference: I/O Subsytems and Device Drivers"         */
/*                                                                            */
/*   hanDC                DcH;                                                */
/*   PBYTE                ArgSource;                                          */
/*   ULONG                ArgCount;                                           */
/*   PULONG               ArgParm;                                            */
/*   ULONG                ArgMix;                                             */
/*   ULONG                ArgStyle;                                           */
/*   BitBltAttrsType far *ArgAttrs;                                           */
/*   lpDCI                DCIData;                                            */
/*   ULONG                FunN;                                               */
/*                                                                            */
/*   DESCRIPTION:                                                             */
/*                                                                            */
/*   This function is the hooked function from the dispatch table.            */
/*   It checks the source (if present) and destination for validity.          */
/*   It sets up some of the fields in the Bitblt working storage              */
/*   which relate to parameters passed in to prdb_Bitblt.  It                 */
/*   determines the source and target rectangles and from these it            */
/*   determines whether a simulation is required in which case it             */
/*   passes the call to the engine.  It handles bounds accumulation           */
/*   and clipping on the target (it adjusts the source to correspond          */
/*   to the clipped section of the target after clipping the target)          */
/*   and calls prdb_DoBitBlt to do the blt.                                   */
/*                                                                            */
/*   NOTE                                                                     */
/* The ArgAttrs parameter is used only in getting the pattern byte.           */
/* It should also be used when converting 1,1 source to color -               */
/* see GetNextSourceByte. Also when transfering from color source             */
/* to mono target - documentation is a little unclear.                        */
/******************************************************************************/
/*   CON3201  Convert to C/SET2                                               */
ULONG prdb_BitBlt ( hanDC            DcH,
                    PBYTE            ArgSource,
                    ULONG            ArgCount,
                    PULONG           ArgParm,
                    ULONG            ArgMix,
                    ULONG            ArgStyle,
                    BitBltAttrsType *ArgAttrs,
                    lpDCI            DCIData,
                    ULONG            FunN )

{
#define TFUNC "prdb_BitBlt"

    /**************************************************************************/
    /* Local variables                                                        */
    /**************************************************************************/
    ULONG       Command;      /* Command part of FunN                         */
    USHORT      rc;           /* return code                                  */
    BOOL        fSrcReq;      /* true if source needed                        */
    BltParms    BB;
    lpBltParms  lpBBParms;    /* Structure for passing                        */
                              /* parameters between functions                 */
                              /* in this file                                 */
    SHORT       Result;       /* Function call return values                  */
    DevRect     lcClipReg;    /* Current clip rectangle                       */
    DevRect     Bounds;       /* Bounding rectangle                           */
    ULONG       dSwap;        /* Temporary storage for                        */
    USHORT      Swap;         /* swapping values                              */
    SHORT       ii,jj;        /* Used to read in the parse string             */
    SHORT       Rotations;    /* The number of rotations to do to the         */
                              /* parse string                                 */
    SHORT       Ops;          /* Used to hold the operation part of           */
                              /* the rop.                                     */
    USHORT      StrWork;      /* Used as a 'carry' when rotating the          */
                              /* parse string                                 */
                              /*                                              */
    USHORT      ImageForeColor;
    USHORT      ImageBackColor;
    USHORT      TrgBytesPerRow;
    RECTL       BoundsRectL;  /* Bounds for clipping processing               */

    /**************************************************************************/
    /* If the rop we are asked to do the bitblt for is 0xAA then we           */
    /* can simply exit as this is a destination copy or leavealone            */
    /**************************************************************************/
    if (ArgMix ==0xaa)
        return (OK);

    /**************************************************************************/
    /* Check that this is a valid call.                                       */
    /**************************************************************************/
    Command = FunN & DCIData->CommandMask;
    prdm_ValidatePath;
    prdm_ValidateArea;

    /**************************************************************************/
    /* Lock the DCI data to this thread                                       */
    /**************************************************************************/
#ifdef PRD_TIMING
    DEKHOOK0(A,7,44)
#endif
    prdm_EnterDriver(DCIData);

    /**************************************************************************/
    /* Set the device text only flag to FALSE since we're about to            */
    /* draw into the band.                                                    */
    /**************************************************************************/
    DCIData->TextOnlyDC = FALSE;

    /**************************************************************************/
    /* Initialize the BB structure to zeros.                                  */
    /**************************************************************************/
    prdu_memset ( (PBYTE) &BB, 0x00, sizeof(BltParms) );

    /**************************************************************************/
    /* Save the parameters in our bitblt structure. The only thing            */
    /* not saved here is ArgSource which needs some transformation            */
    /* first.                                                                 */
    /**************************************************************************/
    BB.TargetDCI = DCIData;
    BB.ArgAttrs  = (ArgAttrs) ? *ArgAttrs: BB.ArgAttrs;
    BB.ArgStyle  = ArgStyle;
    BB.Mix       = LOUSHORT(ArgMix);
    BB.Parm      = ArgParm;
    BB.Count     = ArgCount;

    /**************************************************************************/
    /* Parameter validation.                                                  */
    /**************************************************************************/
    if (ArgStyle & BLTMODE_ATTRS_PRES)
    {
        if (!ArgAttrs)
        {
            /******************************************************************/
            /* Error: style flag indicates attrs passed in, but               */
            /* pointer is NULL.                                               */
            /******************************************************************/
            rc = PMERR_INV_BITBLT_STYLE;
            goto BITBLT_LOGERR;
        }
    }

    /**************************************************************************/
    /* Process the target bitmap. If this is invalid then we cannot           */
    /* proceed any further.                                                   */
    /**************************************************************************/
    BB.TrgLE = (pBMListEntry)BB.TargetDCI->DCISelBitmap;

    if ( !BB.TrgLE )
    {
        /**********************************************************************/
        /*  PD00257 : Changed PMERR_INV_HBITMAP to PMERR_BITMAP_NOT_SELECTED  */
        /**********************************************************************/
        rc = PMERR_BITMAP_NOT_SELECTED;
        goto BITBLT_LOGERR;
    }

    /**************************************************************************/
    /* Store the address of the target bitmap                                 */
    /**************************************************************************/
    BB.TrgBM = BB.TrgLE->Bitmap;

    /**************************************************************************/
    /* Get the information about the source bitmap.  First decide if          */
    /* there is one.  if bits 0 and 1 are the same as bits 2 and 3            */
    /* AND bits 4 and 5 are the same as bits 6 and 7 then no source           */
    /* is used in the rop                                                     */
    /**************************************************************************/
    fSrcReq = ~( ((ArgMix & 0x03) == ((ArgMix & 0x0c)>>2)) &&
                 ((ArgMix & 0x30) == ((ArgMix & 0xc0)>>2)) );

    /**************************************************************************/
    /* if the high four bits of the rop are the same as the low four          */
    /* bits then no pattern is required                                       */
    /**************************************************************************/
    BB.PatternReq = ( ((ArgMix & 0xf0) >> 4) != (ArgMix & 0x0f) );

    /**************************************************************************/
    /* Check the ArgSource parameter which will tell us where to get          */
    /* the source bitmap (if there is one) and validate this                  */
    /* selection as far as is possible at this stage.                         */
    /**************************************************************************/
    /* Allowed values for BB.Source:                                          */
    /*  NO_SOURCE         no source supplied                                  */
    /*  SELECTED_BITMAP   source is a bitmap, selected into a DC              */
    /*  UNSELECTED_BITMAP source is a bitmap, not selected into a DC          */
    /*  SOURCE_DC         source is a DC                                      */
    /**************************************************************************/
    if ( ~fSrcReq )
    {
        /**********************************************************************/
        /* No source required by the mix mode                                 */
        /**********************************************************************/
        BB.Source = NO_SOURCE;
    }
    else
    {
        /**********************************************************************/
        /* A source is required by the mix mode. Is it present and            */
        /* valid?                                                             */
        /**********************************************************************/
        if (ArgSource == FNULL)
        {
            /******************************************************************/
            /* This is the easy condition - no source handle supplied         */
            /* so  set BB.Source to NO_SOURCE and prepare an error            */
            /* code to useif a source is required for the rop.                */
            /******************************************************************/
            rc = PMERR_INV_HDC;
            goto BITBLT_LOGERR;
        }

        else if ((ArgCount < 3) || (ArgCount > 4))
        {
            /******************************************************************/
            /* They've given a source but have not specified any              */
            /* source co-ordinates so bounce them for that.                   */
            /* PD00257 : ArgCount can have the value of 3 or 4 anything       */
            /* else is invalid.                                               */
            /******************************************************************/
            rc = PMERR_INV_LENGTH_OR_COUNT;
            goto BITBLT_LOGERR;
        }

        else if (ArgStyle & BLTMODE_SRC_BITMAP)
        {
            /******************************************************************/
            /* The ArgSource parameter is an engine handle to a               */
            /* bitmap so we call GetDriverInfo for the ListEntry and          */
            /* the Bitmap and the DCIData. The relationship between           */
            /* these entities must not change while we are doing this         */
            /* so grab a semaphore.                                           */
            /******************************************************************/
/*          FSRSemEnter(hmtxBMLockSem);                               CON3201 */
            DosRequestMutexSem(hmtxBMLockSem, SEM_INDEFINITE_WAIT);

            BB.SrcLE = (pBMListEntry)GetDriverInfo((ULONG)ArgSource, /*CON3201*/
                                                   DI_HBITMAP, /* added ULONG */
                                                   (HDC)DcH);

            if ( (LONG)(BB.SrcLE) == GPI_ALTERROR )
            {
                /**************************************************************/
                /* A -1 indicates that ArgSource is not a valid               */
                /* bitmap for our device so set BB.Source to                  */
                /* NO_SOURCE and wait till we find out if we need a           */
                /* source for the rop.                                        */
                /**************************************************************/
                rc = PMERR_INV_HBITMAP;
/*              FSRSemLeave(hmtxBMLockSem);                           CON3201 */
                DosReleaseMutexSem(hmtxBMLockSem);
                goto BITBLT_LOGERR;
            }
            else if ( (BB.SrcBM = BB.SrcLE->Bitmap) == FNULL)
            {
                /**************************************************************/
                /* The source has been validated by the engine but we         */
                /* can't find it in our bitmap list so don't let the          */
                /* rop use it.                                                */
                /**************************************************************/
                rc = PMERR_INV_HBITMAP;
/*              FSRSemLeave(hmtxBMLockSem);                           CON3201 */
                DosReleaseMutexSem(hmtxBMLockSem);
                goto BITBLT_LOGERR;
            }

            else if ( (BB.SourceDCI = BB.SrcLE->DCHandle) == FNULL)
            {
                /**************************************************************/
                /* We've found the bitmap in our bitmap list but it           */
                /* isn't selected into a DC so set BB.Source to               */
                /* UNSELECTED_BIT- MAP and lock the bitmap                    */
                /**************************************************************/
                BB.Source = UNSELECTED_BITMAP;
                BB.SrcLE->BMLocked = TRUE;
            }

            else
            {
                /**************************************************************/
                /* We've got the handle for the DC which the source           */
                /* bitmapis selected into so do some checks here.             */
                /**************************************************************/
                if ((BB.SourceDCI->DCIDCType == OD_MEMORY) &&
                    (BB.SourceDCI->DCISelBitmap == FNULL))
                {
                    /**********************************************************/
                    /* The DC we obtained from the handle of the              */
                    /* source bitmap is a memory DC and doesn't think         */
                    /* it has a selected bitmap so we won't let the           */
                    /* blt use the bitmap in a rop.  (This is                 */
                    /* news as it means we have a bitmap which thinks         */
                    /* its selected into a DC which thinks it doesn't         */
                    /* have a selected bitmap!)                               */
                    /**********************************************************/
                    rc = PMERR_BITMAP_NOT_SELECTED;
/*                  FSRSemLeave(hmtxBMLockSem);                       CON3201 */
                    DosReleaseMutexSem(hmtxBMLockSem);
                    goto BITBLT_LOGERR;
                }

                /**************************************************************/
                /* If we are using banding to output to the printer           */
                /* and there are at least two bands per page and th           */
                /* source and target DCs are the same direct DC the           */
                /* we can't usea source in the rop.                           */
                /**************************************************************/
                if ((BB.TargetDCI->DCIPdbInstance->DDT.DDTBandsPerPage > 1)
                    && (BB.SourceDCI == BB.TargetDCI)
                    && (BB.SourceDCI->DCIDCType == OD_DIRECT))
                {
                    rc = PMERR_INV_FOR_THIS_DC_TYPE;
/*                  FSRSemLeave(hmtxBMLockSem);                       CON3201 */
                    DosReleaseMutexSem(hmtxBMLockSem);
                    goto BITBLT_LOGERR;
                }

                /**************************************************************/
                /* The source bitmap has survived all these tests so          */
                /* we'llreward it with a SELECTED_BITMAP status.              */
                /**************************************************************/
                BB.Source = SELECTED_BITMAP;

            }

            /******************************************************************/
            /* Everything worked fine, so free the semaphore.                 */
            /******************************************************************/
/*                  FSRSemLeave(hmtxBMLockSem);                       CON3201 */
                    DosReleaseMutexSem(hmtxBMLockSem);

        }
        else
        {
            /******************************************************************/
            /* ArgSource is a DC handle so validate it by calling             */
            /* GetDriverInfo.                                                 */
            /******************************************************************/
            BB.SourceDCI = (lpDCI)GetDriverInfo((ULONG)ArgSource, /* CON3201 */
                                                DI_HDC,    /* added ULONG    */
                                                (HDC)DcH);

            if ( (LONG)(BB.SourceDCI) == GPI_ALTERROR )
            {
                /**************************************************************/
                /* A -1 indicates that ArgSource is not a valid DC            */
                /* handle for our driver so set BB.Source to NO_SOURC         */
                /* and wait  till we find out if we need a source for         */
                /* the rop.                                                   */
                /**************************************************************/
                rc = PMERR_INV_HDC;
                goto BITBLT_LOGERR;
            }
            else
            {
                /**************************************************************/
                /* validate the source bitmap handle                          */
                /**************************************************************/
/*              if ( !((PBYTE)BB.SrcLE = BB.SourceDCI->DCISelBitmap) ) CON3210*/
                if ( !(BB.SrcLE = (pBMListEntry)BB.SourceDCI->DCISelBitmap) )
                {
                    /**********************************************************/
                    /* We can't find an entry in our bitmap list for          */
                    /* the bitmap handle we got from the DC so we             */
                    /* won't let the rop use it.  (This situation is          */
                    /*        dire anyway as it means we have a               */
                    /* selected bitmap which doesn't exist!)                  */
                    /**********************************************************/
                    rc = PMERR_NO_BITMAP_SELECTED;
                    goto BITBLT_LOGERR;
                }
                else
                {
                   /***********************************************************/
                   /* Get the bitmap handle from the source DC                */
                   /***********************************************************/
                   BB.SrcBM = BB.SrcLE->Bitmap;

                   /***********************************************************/
                   /* If we are using banding to output to the                */
                   /* printer and there are at least two bands per            */
                   /* page and the source and target DCs are the same         */
                   /* direct DC then we can't use a source in the             */
                   /* rop.                                                    */
                   /***********************************************************/
                   if ((BB.SourceDCI->DCIPdbInstance->DDT.DDTBandsPerPage > 1)
                       && (BB.SourceDCI == BB.TargetDCI)
                       && (BB.SourceDCI->DCIDCType == OD_DIRECT))
                   {
                       rc = PMERR_INV_FOR_THIS_DC_TYPE;
                       goto BITBLT_LOGERR;
                   }

                   /***********************************************************/
                   /* The source DC and its bitmap have survived all          */
                   /* the validation so give it SOURCE_DC status.             */
                   /***********************************************************/
                   BB.Source = SOURCE_DC;
                }
            }
        } /*.. else ! (ArgStyle & BLTMODE_SRC_BITMAP) ........................*/
    } /*.. else ! ( No Source ) ..............................................*/

    /**************************************************************************/
    /* Do the new framework drawing function - sets DRAWN_INTO flag           */
    /* if COM_DRAW is set. Returns error if not in a suitable state.          */
    /**************************************************************************/
    if ( (Result = prdn_CheckDrawingState( (PULONG)&Command,
                                           DCIData ) ) != OK    )
    {
        rc = (Result==ERR_PURGE) ? OK : ERROR_ZERO ; /* CON3203 */
        goto BITBLT_EXIT;
    }

    /**************************************************************************/
    /* First-level checks have been completed. If neither the Draw            */
    /* Bit or Bounds Bits are on, then simply exit OK.                        */
    /**************************************************************************/
    if ((Command & (COM_DRAW | COM_BOUND)) == 0)
    {
        rc = OK;
        goto BITBLT_EXIT;
    }

    /**************************************************************************/
    /* Use mix value as index into Rop table to get Rop                       */
    /**************************************************************************/
    BB.Rop = RopTable [BB.Mix];

    /**************************************************************************/
    /* BBNEXT : The following code sets up BB.ParseStr, BB.StrIndex           */
    /* and BB.OpString - these are not used for the common rops (see          */
    /* prdb_PerformRop for list of these).  Hence this code can be            */
    /* skipped for the common rops.                                           */
    /**************************************************************************/

    /**************************************************************************/
    /* Get the parse string using the index in the Rop.                       */
    /**************************************************************************/
    for (ii = 0; ii < PARSE_STRING_SIZE; ii++)
    {
        (BB.ParseStr)[ii] = ParseTable [(BB.Rop & 0x001C) >> 2] [ii];
    }

    /**************************************************************************/
    /* Rotate the parse string the required number of places to the           */
    /* left.                                                                  */
    /**************************************************************************/
    Rotations = BB.Rop & 0x0003;

    for (ii = 0; ii < Rotations; ii++)
    {
        /**********************************************************************/
        /* For each place the parse string must be rotated get the            */
        /* first entry, shift the remaining entries and put the first         */
        /* entry at the end of the string.                                    */
        /* ??? Index2 < PARSE_STRING_SIZE-1 ???                               */
        /**********************************************************************/
        StrWork = (BB.ParseStr)[0];

        for (jj = 0; jj < PARSE_STRING_SIZE; jj++)
        {
            (BB.ParseStr)[jj] = (BB.ParseStr) [jj+1];
        }

        (BB.ParseStr) [PARSE_STRING_SIZE - 1] = StrWork;
    }

    /**************************************************************************/
    /* Now work out the index into the parse string at which to start.        */
    /* This is the number of binary operations + 1. At the same time          */
    /* set up the operand string.  The two least significant bits of          */
    /* Ops are the next operand.                                              */
    /**************************************************************************/
    BB.StrIndex = 0;
    Ops         = BB.Rop >> 6;

    for (ii = 0; ii < MAX_OPERANDS; ii++)
    {
        /**********************************************************************/
        /* For each two bit field which may be an operand.                    */
        /**********************************************************************/
        if ((Ops & 0x0003) != BB_NOT)
        {
            /******************************************************************/
            /* The operator is not a NOT so add one to the StrIndex.          */
            /******************************************************************/
            BB.StrIndex = BB.StrIndex + 1;
        }

        /**********************************************************************/
        /* Make up the operand string and move to the next operand.           */
        /**********************************************************************/
        (BB.OpString)[ii] = (Ops & 0x0003);
        Ops = Ops >> 2;
    }

    /**************************************************************************/
    /* Make allowance for pushes and pops by adding 2 to the StrIndex         */
    /* if we find a PUSH operation (it will have a matching POP and           */
    /* there will be at most one so we can break on finding it).              */
    /**************************************************************************/
    for (ii = 0; ii < PARSE_STRING_SIZE; ii++)
    {
        if ( (BB.ParseStr) [ii] == PUSH )
        {
            BB.StrIndex = BB.StrIndex + 2;
            break;
        }
    }

    /**************************************************************************/
    /* BBNEXT : End of set up of BB.ParseStr, BB.StrIndex and                 */
    /* BB.OpString.                                                           */
    /**************************************************************************/

    /**************************************************************************/
    /* Set up the target rectangle.                                           */
    /**************************************************************************/
    if (ArgStyle & BBO_TARGWORLD)
    {
        /**********************************************************************/
        /* The target coordinates must be transformed                         */
        /**********************************************************************/
        if ( prdg_Convert ( ArgParm,
                            (ULONG *) BB.SaveTrgRect,              /* CON3201 */
                            COORD_WORLD,
                            COORD_DEVICE_WORD,
                            2L,
                            DcH,
                            Command | COM_TRANSFORM ) != OK )
        {
            rc = ERROR_ZERO;
            goto BITBLT_EXIT;
        }

        /**********************************************************************/
        /* We need to determine the coordinates of the other two              */
        /* corners of the target bitmap to work out if the target             */
        /* bitmap has been rotated.                                           */
        /**********************************************************************/

        /**********************************************************************/
        /* Swap the y coordinates to get the other corners.                   */
        /**********************************************************************/
        dSwap      = ArgParm[3];
        ArgParm[3] = ArgParm[1];
        ArgParm[1] = dSwap;

        /**********************************************************************/
        /* Transform these corners and put the result (as a temporary         */
        /* measure) into TrgRect.                                             */
        /**********************************************************************/
        if ( prdg_Convert ( ArgParm,
                            (PULONG)BB.TrgRect,
                            COORD_WORLD,
                            COORD_DEVICE_WORD,
                            2L,
                            DcH,
                            Command | COM_TRANSFORM ) != OK )
        {
            rc = ERROR_ZERO;
            goto BITBLT_EXIT;
        }

        /**********************************************************************/
        /* Swap the y coordinates back - just for style as its                */
        /* probably not necessary.                                            */
        /**********************************************************************/
        dSwap      = ArgParm[3];
        ArgParm[3] = ArgParm[1];
        ArgParm[1] = dSwap;

        /**********************************************************************/
        /* We need to know the direction from the first coordinate            */
        /* given to the second one given for later use.  Use HorizDir         */
        /* just now as it won't be needed for its real use until              */
        /* we've finished with it.                                            */
        /**********************************************************************/
        if (ArgParm[0] < ArgParm[2])
            BB.HorzDir = RIGHT;
        else
            BB.HorzDir = LEFT;

        /**********************************************************************/
        /* Now all four corners are known in device coordinates.              */
        /* Remember that in world coordinate terms the rectangle was          */
        /* square-on before we converted it and we swapped the y              */
        /* coordinates so...                                                  */
        /*                                                                    */
        /* If we number the points 1,2 (the points we got in ArgParm          */
        /* originally) and 3,4 (the points we got by swapping the y           */
        /* coordinates of points 1 and 2) then we will see that for a         */
        /* conversion which does not rotate the bitmap we would               */
        /* expect...                                                          */
        /*                                                                    */
        /* Point 1s x coord to equal point 3s x coord                         */
        /* Point 2s x coord to equal point 4s x coord                         */
        /* Point 1s y coord to equal point 4s y coord                         */
        /* Point 2s y coord to equal point 3s y coord                         */
        /*                                                                    */
        /* So this is what we test for.  Note that this will allow a          */
        /* target passed as other than bottom left and top right              */
        /* coordinates - ie specifing an inversion of some kind - to          */
        /* pass these conditions so the coordinates will not be               */
        /* altered and this blt will be passed to the engine (if              */
        /* there is a source).                                                */
        /*                                                                    */
        /* However, before we use this test we will use a simpler             */
        /* test which determines whether the x coordinates of the             */
        /* points which specified the target rectangle have reversed          */
        /* their order.  If they have then the target bitmap must             */
        /* have been rotated so we need to do the re-ordering.                */
        /*                                                                    */
        /* The reason we need this test is that the test described            */
        /* above will be TRUE when a transformation has caused a 180          */
        /* degree rotation and we don't want this to happen - we want         */
        /* to re-order the points so no inversion is done.                    */
        /*                                                                    */
        /* So the first four lines evaluate TRUE for some rotations           */
        /* including 180 degrees and the second four lines (with the          */
        /* !) evaluate TRUE for every rotation other than 0 and 180           */
        /* degrees so in total the if statement is executed for any           */
        /* rotation.                                                          */
        /**********************************************************************/
        if ( ( ( (BB.HorzDir == RIGHT) &&
                 (BB.SaveTrgRect[0].X > BB.SaveTrgRect[1].X) ) ||
               ( (BB.HorzDir == LEFT ) &&
                 (BB.SaveTrgRect[0].X < BB.SaveTrgRect[1].X) ) ) ||
             (!(BB.SaveTrgRect[0].X == BB.TrgRect[0].X) &&
               (BB.SaveTrgRect[1].X == BB.TrgRect[1].X) &&
               (BB.SaveTrgRect[0].Y == BB.TrgRect[1].Y) &&
               (BB.SaveTrgRect[1].Y == BB.TrgRect[0].Y) ) )
        {
            /******************************************************************/
            /* We need to re-order the target coordinates so that             */
            /* they are bottom left and top right.  SaveTrgRect and           */
            /* TrgRect hold the four vertices of a square which has           */
            /* been rotated through a number of degrees so we will            */
            /* make the target rectangle the smallest rectangle               */
            /* square to the target bitmap which contains all these           */
            /* vertices.                                                      */
            /******************************************************************/

            /******************************************************************/
            /* First make sure we have the smaller of the x coords            */
            /* ordered in SaveTrgRect as this is where the final              */
            /* coords go.                                                     */
            /******************************************************************/
            if (BB.SaveTrgRect[0].X > BB.SaveTrgRect[1].X)
            {
                Swap = BB.SaveTrgRect[1].X;
                BB.SaveTrgRect[1].X = BB.SaveTrgRect[0].X;
                BB.SaveTrgRect[0].X = Swap;
            }

            /******************************************************************/
            /* Then we can see if either of the x coords in TrgRect           */
            /* is smaller than the smallest in SaveSrcRect or larger          */
            /* than the largest in SaveTrgRect and adopt new values           */
            /* if necessary.                                                  */
            /******************************************************************/
            BB.SaveTrgRect[0].X = min(BB.SaveTrgRect[0].X, BB.TrgRect[0].X);
            BB.SaveTrgRect[0].X = min(BB.SaveTrgRect[0].X, BB.TrgRect[1].X);
            BB.SaveTrgRect[1].X = max(BB.SaveTrgRect[1].X, BB.TrgRect[0].X);
            BB.SaveTrgRect[1].X = max(BB.SaveTrgRect[1].X, BB.TrgRect[1].X);

            /******************************************************************/
            /* The y coordinates are handled in the same way as the           */
            /* x ones.                                                        */
            /******************************************************************/
            if (BB.SaveTrgRect[0].Y > BB.SaveTrgRect[1].Y)
            {
                Swap = BB.SaveTrgRect[0].Y;
                BB.SaveTrgRect[0].Y = BB.SaveTrgRect[1].Y;
                BB.SaveTrgRect[1].Y = Swap;
            }

            BB.SaveTrgRect[0].Y = min(BB.SaveTrgRect[0].Y, BB.TrgRect[0].Y);
            BB.SaveTrgRect[0].Y = min(BB.SaveTrgRect[0].Y, BB.TrgRect[1].Y);
            BB.SaveTrgRect[1].Y = max(BB.SaveTrgRect[1].Y, BB.TrgRect[0].Y);
            BB.SaveTrgRect[1].Y = max(BB.SaveTrgRect[1].Y, BB.TrgRect[1].Y);
        }
        /*.. if ( (BB.SaveTrgRect[0].X ...target was rotated..................*/

    }
    /*.. if (ArgStyle & BBO_TARGWORLD) .......................................*/

    else
    {
        /**********************************************************************/
        /* No need to transform target coordinates given in device            */
        /* coordinates.                                                       */
        /**********************************************************************/
        BB.SaveTrgRect[0].X = (USHORT) ArgParm[0];
        BB.SaveTrgRect[0].Y = (USHORT) ArgParm[1];
        BB.SaveTrgRect[1].X = (USHORT) ArgParm[2];
        BB.SaveTrgRect[1].Y = (USHORT) ArgParm[3];
    }

    /**************************************************************************/
    /* Set up the source rectangle                                            */
    /**************************************************************************/
    if (BB.Source == NO_SOURCE)
    {
        /**********************************************************************/
        /* There is no source so nothing to do, but this implies that         */
        /* inversions are not supported so the target rectangle must          */
        /* be ordered and may not be at this point so do it.                  */
        /**********************************************************************/
        if (BB.SaveTrgRect[0].X > BB.SaveTrgRect[1].X)
        {
            Swap = BB.SaveTrgRect[1].X;
            BB.SaveTrgRect[1].X = BB.SaveTrgRect[0].X;
            BB.SaveTrgRect[0].X = Swap;
        }

        if (BB.SaveTrgRect[0].Y > BB.SaveTrgRect[1].Y)
        {
            Swap = BB.SaveTrgRect[0].Y;
            BB.SaveTrgRect[0].Y = BB.SaveTrgRect[1].Y;
            BB.SaveTrgRect[1].Y = Swap;
        }

    }
    else
    {
        /**********************************************************************/
        /* There is a source, get the rectangle coordinates                   */
        /**********************************************************************/
        BB.SaveSrcRect[0].X = (USHORT) ArgParm[4];
        BB.SaveSrcRect[0].Y = (USHORT) ArgParm[5];

        if (BB.Count == 4)
        {
            /******************************************************************/
            /* Source rectangle is completely specified                       */
            /******************************************************************/
            BB.SaveSrcRect[1].X = (USHORT) ArgParm[6];
            BB.SaveSrcRect[1].Y = (USHORT) ArgParm[7];
        }
        else if (BB.Count == 3)
        {
            /******************************************************************/
            /* Source rectangle origin only is specified, size must           */
            /* be inferred from target rectangle size                         */
            /******************************************************************/
            BB.SaveSrcRect[1].X = BB.SaveSrcRect[0].X +
                                  BB.SaveTrgRect[1].X -
                                  BB.SaveTrgRect[0].X;

            BB.SaveSrcRect[1].Y = BB.SaveSrcRect[0].Y +
                                  BB.SaveTrgRect[1].Y -
                                  BB.SaveTrgRect[0].Y;
        }

        /**********************************************************************/
        /* Call simulation if                                                 */
        /*     - compress or stretch is required                              */
        /*     - inversion is required (only if source supplied)              */
        /**********************************************************************/
        if (
            !(Command & COM_DEVICE) &&
              (
              (BB.SaveSrcRect[1].X - BB.SaveSrcRect[0].X) !=
                             (BB.SaveTrgRect[1].X - BB.SaveTrgRect[0].X)
              || (BB.SaveSrcRect[1].Y - BB.SaveSrcRect[0].Y) !=
                             (BB.SaveTrgRect[1].Y - BB.SaveTrgRect[0].Y)

              || (BB.SaveTrgRect[0].X > BB.SaveTrgRect[1].X)
              || (BB.SaveTrgRect[0].Y > BB.SaveTrgRect[1].Y)
              )
           )
        {
            /******************************************************************/
            /* Return for simulation.                                         */
            /******************************************************************/
            if (BB.Source == UNSELECTED_BITMAP)
            {
                BB.SrcLE->BMLocked = FALSE;
            }

            prdm_LeaveDriver(DCIData);
#ifdef PRD_TIMING
    DEKHOOK0(A,7,C4)
#endif
            /******************************************************************/
            /*  PD00567 : We need to or in a flag to let us know we've        */
            /*  returned for an engine simulation...                          */
            /******************************************************************/
            DCIData->DCIStateFlags |= BLIT_SIML_IN_PROCESS;
            rc = (USHORT)da_BitBlt(DcH, ArgSource, ArgCount, ArgParm, ArgMix,
                                   ArgStyle, ArgAttrs, DCIData, FunN);
            DCIData->DCIStateFlags &= ~BLIT_SIML_IN_PROCESS;
            return(rc);
        }

        /**********************************************************************/
        /*  Now calculate whether this is a memory->printer bitblt.  If so, we*/
        /*  need to convert ("expand") the source bits using the image        */
        /*  foreground and background colors.  We do this by calculating here */
        /*  what sort of source conversion will be required.  This can take   */
        /*  the values:                                                       */
        /*                                                                    */
        /*  SRC_COPY_BITS.....Copy bits unchanged                             */
        /*  SRC_INVERT_BITS...Invert source bits                              */
        /*  SRC_ZEROS.........Supply zeros as the source                      */
        /*  SRC_ONES..........Supply ones as the source                       */
        /*                                                                    */
        /*  First assume that no conversion will take place...                */
        /**********************************************************************/
        BB.SrcCopyMode = SRC_COPY_BITS;

        if ( BB.TargetDCI->DCIDCType != OD_MEMORY )
        {
            /******************************************************************/
            /*  The target surface is the printer's band.                     */
            /******************************************************************/
            if ( (BB.Source == UNSELECTED_BITMAP) ||
                 ( ( (BB.Source == SOURCE_DC) ||
                     (BB.Source == SELECTED_BITMAP) ) &&
                   (BB.SourceDCI->DCIDCType == OD_MEMORY) ) )
            {

                /**************************************************************/
                /*  Don't expand the source bits if the the driver created a  */
                /*  mono bitmap when asked for a color format bitmap.  This is*/
                /*  done in an attempt to avoid the problems caused by the    */
                /*  apps not setting the image attributes properly for a mono */
                /*  bitmap (they dont since they think they have created a    */
                /*  color bitmap).  A good example of this effect is          */
                /*  Describe's Generic Fish test case.  ( SMP 18/9/90 )       */
                /*                                                            */
                /*  PD00486 : Added a check to ensure that we're dealing with */
                /*  a 1,1 bitmap.                                             */
                /**************************************************************/
                if (LOUSHORT(BB.TrgLE->Parms.Bitcount) == 1)
                {
                    if (!BB.SrcLE->ColBmapReq)
                    {

                        /******************************************************/
                        /*  The source is a memory bitmap and we need to use  */
                        /*  the image attributes.                             */
                        /******************************************************/
                        ImageBackColor = prdc_ColorToPelBits(
                                    BB.TargetDCI->DCICurImgAts->ibnd.lBackColor,
                                    BB.TargetDCI, BB.TrgLE->DCTPtr);

                        ImageForeColor = prdc_ColorToPelBits(
                                    BB.TargetDCI->DCICurImgAts->ibnd.lColor,
                                    BB.TargetDCI, BB.TrgLE->DCTPtr);

                        /******************************************************/
                        /*  Set up the SrcCopyMode.                           */
                        /*                                                    */
                        /*  The image BackColor gives the value that an       */
                        /*  internal 0 should be converted to, and the image  */
                        /*  ForeColor gives the value that an internal 1      */
                        /*  should be converted to.                           */
                        /*                                                    */
                        /*  Therefore we can create a code using bit 0 as the */
                        /*  image background color, and bit 1 as the image    */
                        /*  foreground color:                                 */
                        /*                                                    */
                        /*  0 (00000000b) : SRC_ZEROS                         */
                        /*  1 (00000001b) : SRC_INVERT_BITS                   */
                        /*  2 (00000010b) : SRC_COPY_BITS                     */
                        /*  3 (00000011b) : SRC_ONES                          */
                        /******************************************************/
                        BB.SrcCopyMode =
                           (((ImageForeColor == 7) ? 1 : ImageForeColor) << 1) |
                            ((ImageBackColor == 7) ? 1 : ImageBackColor);
                    }
                }
                else
                {
                    if (BB.SrcLE->Parms.MonBmapReq)
                    {

                        /******************************************************/
                        /*  Set up the SrcCopyMode.                           */
                        /*                                                    */
                        /*  The image BackColor gives the value that an       */
                        /*  internal 0 should be converted to, and the image  */
                        /*  ForeColor gives the value that an internal 1      */
                        /*  should be converted to.                           */
                        /*                                                    */
                        /*  Therefore we can create a code using bit 0 as the */
                        /*  image background color, and bit 1 as the image    */
                        /*  foreground color:                                 */
                        /*                                                    */
                        /*  0 (00000000b) : SRC_ZEROS                         */
                        /*  1 (00000001b) : SRC_INVERT_BITS                   */
                        /*  2 (00000010b) : SRC_COPY_BITS                     */
                        /*  3 (00000011b) : SRC_ONES                          */
                        /******************************************************/
                        BB.SrcCopyMode = SRC_INVERT_BITS;
                    }
                }
            }
        }
        /* .... if ( BB.TargetDCI->DCIDCType != OD_MEMORY ) .................. */
    }
    /*.. else ! (BB.Source == NO_SOURCE) ......................................*/

    /**************************************************************************/
    /*  If bounds accumulation required then call prdg_AddBounds with the     */
    /*  SaveTrgRect.  Note that by this time this will definitely have the    */
    /*  bottom left and top right coordinates as they will either have been   */
    /*  ordered in the if arm of the preceding if statement or they will have */
    /*  been passed to the engine in the else arm if they are out of order.   */
    /**************************************************************************/
    if ((Command & COM_BOUND) != 0)
    {
        Bounds[0].X = BB.SaveTrgRect[0].X;
        Bounds[0].Y = BB.SaveTrgRect[0].Y;
        Bounds[1].X = BB.SaveTrgRect[1].X - 1;
        Bounds[1].Y = BB.SaveTrgRect[1].Y - 1;

     /* (VOID) prdg_AddBounds((DevRect far *)Bounds, DCIData); */
        (VOID) prdg_AddBounds((DevRect *)Bounds, DCIData);         /* CON3201 */
    }

    /**************************************************************************/
    /*  First-level checks and set-up have been completed, and Bounds have    */
    /*  been calculated (if required).  So if the Draw Bit is not on we can   */
    /*  exit OK.                                                              */
    /**************************************************************************/
    if ( !(Command & COM_DRAW) )
    {
        rc = OK;
        goto BITBLT_EXIT;
    }

    /**************************************************************************/
    /*  Make space on the heap for our bitblt structure.                      */
    /**************************************************************************/
    if ( prdg_AllocHeapItem( BB.TargetDCI,
                             (USHORT)sizeof(BltParms),
                             (PUSHORT *)&lpBBParms) != OK )
    {
        rc = ERROR_ZERO;
        goto BITBLT_EXIT;
    }

    /**************************************************************************/
    /*  Initialize it to zeros                                                */
    /**************************************************************************/
    prdu_memset ( (PBYTE) lpBBParms, 0x00, sizeof(BltParms) );

    /**************************************************************************/
    /*  If there is a pattern then get the information required for it.       */
    /**************************************************************************/
    if ( BB.PatternReq )
    {
        Result = prdb_GetPattern( (lpBltParms)&BB );

        if ((Result != OK) && (Result != GS_FLAG))
        {
            rc = ERROR_ZERO;
            goto BITBLT_MEMORY_EXIT;
        }
    }

    /**************************************************************************/
    /*  If there is a source then get some memory for the source row          */
    /**************************************************************************/
    if (BB.Source != NO_SOURCE)
    {
        /**********************************************************************/
        /*  The source row must be as large as the target row just in case the*/
        /*  full line is being blted.                                         */
        /**********************************************************************/
        TrgBytesPerRow = (USHORT) ((BB.TrgLE)->Parms.BytesPerRow);

        if ( prdg_AllocHeapItem( BB.TargetDCI,
                                 TrgBytesPerRow,
                                 (PUSHORT *)&BB.abSrcRow) != OK )
        {
            rc = ERROR_ZERO;
            goto BITBLT_EXIT;
        }

        /**********************************************************************/
        /*  Initialize it to zeros                                            */
        /**********************************************************************/
        prdu_memset ( (PBYTE) BB.abSrcRow, 0x00, TrgBytesPerRow );

    }

    /**************************************************************************/
    /*  PD00267 : Pick up the bounding rectangle for the image and pass it to */
    /*  prdi_GetNextClip().                                                   */
    /**************************************************************************/
    BoundsRectL.xLeft   = (ULONG)BB.SaveTrgRect[0].X;
    BoundsRectL.yBottom = (ULONG)BB.SaveTrgRect[0].Y;
    BoundsRectL.xRight  = (ULONG)BB.SaveTrgRect[1].X + 2; /* PD00308...       */
    BoundsRectL.yTop    = (ULONG)BB.SaveTrgRect[1].Y + 2; /* PD00308...       */

    /**************************************************************************/
    /*  For every valid clip rectangle returned by prdi_GetNextClip.          */
    /**************************************************************************/
    for (ii = 1; prdi_GetNextClip(BB.TargetDCI,
                                  ii,
                               /* (DevRect far*)lcClipReg, */
                                  (DevRect *)lcClipReg,            /* CON3201 */
                                  &BoundsRectL) == OK; ii++)
    {
        /**********************************************************************/
        /*  Make sure we clip the original coordinates.                       */
        /**********************************************************************/
        BB.SrcRect[0] = BB.SaveSrcRect[0];
        BB.SrcRect[1] = BB.SaveSrcRect[1];
        BB.TrgRect[0] = BB.SaveTrgRect[0];
        BB.TrgRect[1] = BB.SaveTrgRect[1];

        /**********************************************************************/
        /*  Clip the target rectangle against the clip rectangle.             */
        /**********************************************************************/
     /* if ( prdi_ClipIntSect((DevRect far*) lcClipReg,          */
     /*                       (DevRect far*) BB.TrgRect) != OK ) */
        if ( prdi_ClipIntSect((DevRect *) lcClipReg,               /* CON3201 */
                              (DevRect *) BB.TrgRect) != OK )      /* CON3201 */
        {
            /******************************************************************/
            /*  No intersection so continue to the next clip rectangle        */
            /******************************************************************/
            continue;
        }

        /**********************************************************************/
        /*  There is an intersect - TrgRect has been updated with the         */
        /*  coordinates of the clipped target.  Now work out that bit of the  */
        /*  source which corresponds to the clipped target.  The bitblt will  */
        /*  only be done for that bit of the target which lies in the clip    */
        /*  region.                                                           */
        /**********************************************************************/
        if ( BB.Source != NO_SOURCE )
        {
            BB.SrcRect[0].X += (BB.TrgRect[0].X - BB.SaveTrgRect[0].X);
            BB.SrcRect[0].Y += (BB.TrgRect[0].Y - BB.SaveTrgRect[0].Y);

            /******************************************************************/
            /*  In the next two lines we will add a number which is <= 0 to   */
            /*  get the SrcRect top right corner.                             */
            /******************************************************************/
            BB.SrcRect[1].X += (BB.TrgRect[1].X - BB.SaveTrgRect[1].X);
            BB.SrcRect[1].Y += (BB.TrgRect[1].Y - BB.SaveTrgRect[1].Y);

            /******************************************************************/
            /*  BBNEXT : The code below which checks that the source rectangle*/
            /*  is within its bitmap (and clips it, and the target rectangle, */
            /*  if necessary) would be better done just once before we get    */
            /*  into the clip loop.                                           */
            /******************************************************************/
            if ( (BB.SrcRect[0].X >= (int)BB.SrcLE->Parms.Width) ||
                 (BB.SrcRect[0].Y >= (int)BB.SrcLE->Parms.Height) )
            {
                rc = OK;
                goto BITBLT_MEMORY_EXIT;
            }

            if (BB.SrcLE->Parms.Width < BB.SrcRect[1].X)
            {
                BB.TrgRect[1].X -= (BB.SrcRect[1].X -
                                          (SHORT)BB.SrcLE->Parms.Width);
                BB.SrcRect[1].X = (SHORT)BB.SrcLE->Parms.Width;
            }

            if (BB.SrcLE->Parms.Height < BB.SrcRect[1].Y)
            {
                BB.TrgRect[1].Y -= (BB.SrcRect[1].Y -
                                         (SHORT)BB.SrcLE->Parms.Height);
                BB.SrcRect[1].Y = (SHORT)BB.SrcLE->Parms.Height;
            }

            if (BB.SrcRect[0].X < 0)
            {
                BB.TrgRect[0].X -= BB.SrcRect[0].X;
                BB.SrcRect[0].X = 0;
            }

            if (BB.SrcRect[0].Y < 0)
            {
                BB.TrgRect[0].Y -= BB.SrcRect[0].Y;
                BB.SrcRect[0].Y = 0;
            }
        }

        /***********************************************************************/
        /* Ensure that after the clip there is something to blt.  The          */
        /* clip will have returned a rectangle which is non inclusive          */
        /* at the top right corner so we have to decrement the                 */
        /* coordinates for the top right corners of both source and            */
        /* target.  Allow for this in the test.  Note that SaveTrgRect         */
        /* and SaveSrcRect are non-inclusive whereas TrgRect and               */
        /* SrcRect are inclusive so even if the rectangles are not             */
        /* clipped at the top and/or right we should still decrement           */
        /* the coordinates.                                                    */
        /*                                                                     */
        /* BBNEXT : Since at this point the size of the source and             */
        /* target rects are the same we only need the following test:          */
        /*                                                                     */
        /*     if ( (BB.TrgRect[1].X > BB.TrgRect[0].X) &&                     */
        /*          (BB.TrgRect[1].Y > BB.TrgRect[0].Y) )                      */
        /*                                                                     */
        /***********************************************************************/
        if ((BB.TrgRect[1].X > BB.TrgRect[0].X)
            && (BB.TrgRect[1].Y > BB.TrgRect[0].Y)
            && ((BB.Source == NO_SOURCE)
                || ((BB.SrcRect[1].X > BB.SrcRect[0].X)
                     && (BB.SrcRect[1].Y > BB.SrcRect[0].Y))))

        {
            /******************************************************************/
            /*  Decrement the top right corners of the target and source.     */
            /******************************************************************/
            BB.SrcRect[1].X--;
            BB.SrcRect[1].Y--;
            BB.TrgRect[1].X--;
            BB.TrgRect[1].Y--;

            /******************************************************************/
            /*  Do the blt for this clip rectangle.                           */
            /******************************************************************/
            *lpBBParms = BB;
            (VOID)prdb_DoBitBlt(lpBBParms, DCIData);
        }
        /*.. if ((BB.TrgRect[1].X > BB.TrgRect[0].X) ..an intersect...         */

    }
    /*.. for (i = 1; prdi_GetNextClip(BB.TargetDCI, ..clip rectangles.         */

    /**************************************************************************/
    /*  If we have reached here then we are doing OK                          */
    /**************************************************************************/
    rc = OK;

BITBLT_MEMORY_EXIT:
    /**************************************************************************/
    /*  free heap used for bitblt structure and font pattern                  */
    /**************************************************************************/
    if ( BB.PatternReq )
    {
        /**********************************************************************/
        /*  Pattern uses all kinds of memory.  Tidy this up                   */
        /**********************************************************************/
        if (BB.PatRowBM)
        {
            /******************************************************************/
            /*  Free the memory allocated for the pattern row                 */
            /******************************************************************/
            (VOID) prdg_FreeHeapItem( BB.TargetDCI,
                                      BB.PatRowBMSize,
                                      (PUSHORT)BB.PatRowBM);
        }

        if (BB.CacheBM)
        {
            /******************************************************************/
            /*  Free the memory allocated for the pattern cache               */
            /******************************************************************/
            (VOID) prdg_FreeHeapItem( BB.TargetDCI,
                                      BB.CacheBMSize,
                                      (PUSHORT)BB.CacheBM);
        }

        if ( BB.FontBM )
        {
            /******************************************************************/
            /*  Free the memory allocated for the font bitmap.                */
            /******************************************************************/
            (VOID) prdg_FreeHeapItem( BB.TargetDCI,
                                      BB.FontBMSize,
                                      (PUSHORT)BB.FontBM);
        }
    }

    if (BB.Source != NO_SOURCE)
    {
        /**********************************************************************/
        /*  Free the source row memory                                        */
        /**********************************************************************/
        (VOID) prdg_FreeHeapItem( BB.TargetDCI,
                                  TrgBytesPerRow,
                                  (PUSHORT)BB.abSrcRow);
    }

    /**************************************************************************/
    /*  Finally free the BltParms memory.                                     */
    /**************************************************************************/
    (VOID) prdg_FreeHeapItem( BB.TargetDCI,
                              sizeof(BltParms),
                              (PUSHORT)lpBBParms);

BITBLT_EXIT:
    /**************************************************************************/
    /*  Normal exit                                                           */
    /**************************************************************************/
    if (BB.Source == UNSELECTED_BITMAP)
        BB.SrcLE->BMLocked = FALSE;

    prdm_LeaveDriver(DCIData);
#ifdef PRD_TIMING
    DEKHOOK0(A,7,C4)
#endif
    return ((ULONG)rc);

BITBLT_LOGERR:

    /**************************************************************************/
    /*  Log error exit                                                        */
    /**************************************************************************/
    LOGERR(TFUNC, "Error", FNULL, 0, rc);
    rc = ERROR_ZERO;
    goto BITBLT_EXIT;
}
#undef TFUNC
