/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Microsoft Corporation, 1989                                 */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = QPRAW.C
 *
 * DESCRIPTIVE NAME = PM Spooler processor
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION
 *
 * This module contains the code for printing a PM_Q_RAW format spooler file
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#define INCL_DEV

#include "pmprint.h"

/*
** Constants section:
**
** Raw data buffer size to read from spool file
*/
#define BUFLEN 4096

/****************************************************************************
 *
 * FUNCTION NAME = SplQpRawPrintFile
 *
 * DESCRIPTION   = Print a spool file in "PM_Q_RAW" format( raw data )
 *
 * INPUT         = pQProc      - QProc instance pointer for print job
 *                 pszFileName - Path to raw file to print
 *
 * OUTPUT        = TRUE   - Job printed successfully to our knowledge
 *                 FALSE  - Unable to print job due to Dev failure
 *                          WinGetLastError() gets failure code.
 *                          Message displayed to user.
 *
 * RETURN-NORMAL = TRUE
 *
 *
 * RETURN-ERROR  = FALSE
 *
 *
 ****************************************************************************/

BOOL SplQpRawPrintFile(PQPROCINST pQProc, PSZ pszFileName )
{
   BOOL   result ;
   ULONG  rc;
   ULONG  cb;
   ULONG  cbRead;
   ULONG  cbTotal;


   result = FALSE ;
   pQProc->pRawBuf = NULL;

   /*
   ** Open spool file and OD_DIRECT DC(without associating to PS)
   ** If this fails, OpenQPxxx call issues message to user
   */
   if (OpenQPInputFile( pQProc, pszFileName, TRUE ) &&
       OpenQPOutputDC(pQProc, NOASSOC))
   {
       /*
       ** Allocate max buf size of 4K to read spool file
       */
       if ((cbTotal = pQProc->cbFile) > BUFLEN)
           cb = BUFLEN;
       else
           cb = cbTotal;

       rc = DosAllocMem( &pQProc->pRawBuf, cb, PAG_COMMIT|PAG_READ|PAG_WRITE);
       if (rc == 0)
       {
          result = TRUE;

          while (cbTotal && !(pQProc->fsStatus & QP_ABORTED))
          {
             if (cbTotal > BUFLEN)
                cb = BUFLEN;
             else
                cb = cbTotal;

             cbRead = 0;
             rc = DosRead( pQProc->hFile, pQProc->pRawBuf, cb, &cbRead );
             if ( rc || (cb != cbRead) )
             {
                /*
                ** Read of spool file failed
                ** Give a message to user and fail print job
                ** Force read loop to terminate by zeroing cbTotal
                */
                cbTotal = 0 ;
                pQProc->fsStatus &= ~ QP_PAUSED;
                Panic("QP: DosRead failed for %0s",
                      (PSZ)pQProc->szFileName, rc);
                SplQpMessage(pQProc->pszPortName, SPL_ID_QP_FILE_NOT_FOUND,
                             PMERR_SPL_FILE_NOT_FOUND);

             } else if ( !(pQProc->fsStatus & QP_ABORTED)) {

                /*
                ** Job not aborted, so send job data to printer
                */
                if (DevEscape( pQProc->hdc,
                               DEVESC_RAWDATA,
                               (LONG)cbRead,
                               pQProc->pRawBuf,
                               (PLONG)NULL, (PBYTE)NULL ) == DEVESC_ERROR)
                {
                   /*
                   ** Send of raw data failed
                   **
                   ** This should not happen, since drivers should
                   **   always accept raw data.
                   **   (FYI: if printer is offline and user selects
                   **           to Abort the job,
                   **         then the printer driver should still
                   **           not fail the DevEscape(RAWDATA)
                   **
                   ** For very old print drivers which may fail
                   **   the DevEscape(RAWDATA) when the printer is
                   **   offline, we should not display an error message
                   **   at this time to the user(simulate an Abort).
                   */
                   pQProc->fsStatus |= QP_ABORTED;
                   pQProc->fsStatus &= ~ QP_PAUSED;
                   Panic("QP: DevEscape(RAWDATA) failed for %0s",
                         (PSZ)pQProc->szFileName, rc);

                } else
                    cbTotal -= cbRead;
             }
             /*
             ** If user paused the print job
             **   wait till user releases the job
             */
             if (pQProc->fsStatus & QP_PAUSED)
                 DosWaitEventSem(pQProc->semPaused, SEM_INDEFINITE_WAIT);

          }                                 /* end while reading spool file */

          if (pQProc->fsStatus & QP_ABORTED)
          {
              DevEscape( pQProc->hdc, DEVESC_ABORTDOC,
                         (LONG)0, (PBYTE)NULL,
                         (PLONG)NULL, (PBYTE)NULL );
              result = TRUE;

          } else if (!cbTotal) {

              /*
              ** All spool file data was printed
              */
              result = TRUE;
          }
       } else {
          /*
          ** Failed to allocate buffer to read file into
          */
          result = FALSE ;
          SplQpMessage(pQProc->pszPortName, SPL_ID_QP_MEM_ERROR,
                       PMERR_SPL_NO_MEMORY);
       }
   }

   if (pQProc->pRawBuf)
   {
       DosFreeMem( pQProc->pRawBuf );
       pQProc->pRawBuf = NULL;
   }

   /*
   ** if ENDDOC fails(called by CloseQPOutputDC)
   **    return FALSE
   */
   if ( (CloseQPOutputDC( pQProc, result ) == FALSE) &&
         !(pQProc->fsStatus & QP_ABORTED))
     result = FALSE ;

   CloseQPInputFile( pQProc );

   if (!result)
       Panic( "SplQpRawPrintFile failed for %0s", pszFileName, 0 );

   return( result );
}
