/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = ipmimain.c
 *
 * DESCRIPTIVE NAME = Chip specific PMI functions
 *
 *
 * VERSION = V2.1
 *
 * DATE
 *
 * DESCRIPTION PMI-file handler - Exported entry points
 *
 * FUNCTIONS
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 * Organization of the imported PMI binary objects:
 * PMI binary objects are a shared libary (DLL) which contains worker routines
 * for all of the imported PMI functions, as specified in the complementing PMI
 * file. The PMI file contains the #includecode "Y" with the name of the
 * imported binary object and call pfnFunction references for all of the imported
 * PMI functions. The pair (X.PMI and Y.DLL) are treated as the adapters
 * SetMode (and other base video functions) handler. The suggested delegation of
 * services between the two is:
 * X.PMI file  sections
 * Hardware (may be generic and corrected by the IdentifyAdapter)
 * SetBank
 * GetBank
 * UnLock
 * Lock
 * Cleanup
 * ModeInfo
 * MonitorModeInfo
 * SetMode (optionally in Y.DLL)
 * SetMonitorTimings (optionally in Y.DLL)
 * Declarations
 * IdentifyAdapter (optionally in Y.DLL)
 *
 * IBMGPMI is the imported PMI library which handles the following chips/adapters:
 * ATI Mach32 and Mach64, S3 805/928 planar implementations, S3 864 planar implementations,
 * Cirrus 543x planar implementations and Diamond Speedstar 24(x) and Stealth W32
 * as well as Number 9 s3 9GXE and STB S3 and Cirrus implementations.
 * The library exports three functions:
 * IdentifyAdapter, SetMonitorTimings and SetMode.
 * IdentifyAdapter is the function which ensures that the hardware is
 * recognized. Release level of IBMGPMI internally delegates the identification of the
 * hardware to screendd.sys physical driver for the already supported adapters.
 * This driver is shipped only
 * as part of a release and therefore, any device support asynchrounous to the
 * release should include device identification in the IdentifyAdapter
 * function (this includes drivers released by both IBM and OEM) rather than
 * delegating it to the screendd.
 ********************************************************************************
 * NOTE to OEM developers:
 *
 * DO NOT MODIFY AND SHIP IBMGPMI.DLL!
 *
 * Create a DLL of your own with a unique name which
 * exports all of the required functions. Format your PMI files to include your
 * own DLL rather than IBMGPMI. Use this code as an example only!
 ********************************************************************************
 * NOTE:        Diamond Stealth Pro code was too difficult to convert to 32 bit,
 *              so I temporarely disabled it. The clock function is still built into
 *              the BVHSVGA.                    @senja
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#pragma langlvl(extended)
#include "ipmitype.h"
#include <string.h>
#include <conio.h>

/*           
 * IOPL segment cannot exceed 64 K.
 * We need to divide code into several segments, which is less than 64KB.
 */
#pragma alloc_text (CODE_TIMING, pfnPMISetMonitorTimings)

/*****************************************************************************
 *
 *  FUNCTION NAME:      _DLL_InitTerm()
 *
 *  DESCRIPTIVE NAME: Initialize the DLL
 *
 *  FUNCTION:         Perform the adapter identification and initialize all of
 *                    global data for the adapter
 *
 *
 *  EXIT:             APIRET - return code 0 if identified, ERROR_ADAPTER_NOT_SUPPORTED otherwise.
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
APIRET _System _DLL_InitTerm(ULONG hModule, ULONG ulFlags)
{
  switch(ulFlags)
  {
     case 0:            //initialization
       if (_CRT_init() == -1)
         return 0UL;      //failed to initialize
       /* fail if hardware not supported or initialization failed */
       if (!Identify(&SVGAHardware,&OEMHardware))
         return 1;      //success
       else
         return 0;      //failed to initialize
     case 1:            //termination
       _CRT_term();     //terminate C run-time
       return 1;      //success
     default:
       return 0;      //failed to initialize
  }
}
/*****************************************************************************
 *
 *  FUNCTION NAME:      pfnIdentifyAdapter()
 *
 *  DESCRIPTIVE NAME: Identify the adapter and confirm that the OEMString
 *                    DACString passed conform to the hardware.
 *
 *  FUNCTION:
 *
 *  INPUT:            PVIDEO_ADAPTER - Pointer to current state of the adapter/mode
 *                    PREGS- Pointer to current register state
 *
 *  EXIT:             APIRET - return code
 *
 *  NOTES:            PMI files contains  [Hardware] header which is
 *                    formatted at the time the PMI file is created.
 *                    The header is as specific as the source providing it
 *                    wished to make it. The headers description can be
 *                    updated by the IdentifyAdapter, if the OEM wishes to
 *                    provide a provisional header and use the IdentifyAdapter
 *                    as a vehicle to setup the device specific support at the
 *                    time the PMI file is being loaded rather than at the time
 *                    the PMI file is created. The update is made thru the VIDEO_ADAPTER
 *                    which does not cause changes to the PMI file itself, but
 *                    lets all of the PMI clients have the exact description.
 *                    An adapter is described using
 *                    OEMString = "CHIPNAME ADAPTERNAME, MANUFACTURER NAME",
 *                    Version string and DACString.
 *                    GENERIC can be used for the ADAPTERNAME and
 *                    DACString. When used, the IdentifyAdapter should only confirm that
 *                    chip identified matches the CHIPNAME. If they are not set to
 *                    GENERIC, confirmation of the identity has to include ADAPTERNAME
 *                    and/or DACString.
 *
 *                    In IBM case, this function ensures both that the hardware
 *                    is identified and that a correct PMI file has made a reference
 *                    into this DLL.
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
APIRET EXPENTRY pfnPMIIdentifyAdapter(PVIDEO_ADAPTER pAdapter,PREGS pRegs)
{
  APIRET rc = NO_ERROR;                       /* default no error                  */
  UCHAR szOEMString[256];
  PCHAR pszChipName, pszAdapterName,pChip;
  UCHAR szDACString[256];                                         //@V3.0ET001
  PCHAR pszDACName;                                               //@V3.0ET001
  ULONG i,j;

  /*
  ** Tokenize the DACString to get DAC name                       //@V3.0ET001
  */                                                              //@V3.0ET001
  strcpy(szDACString,pAdapter->Adapter.szDACString);              //@V3.0ET001
//            for(i=0;i<256 && szDACString[i] != '_';i++);                    //@V3.0ET001
  for(i=0;i<256 && szDACString[i] != ' ';i++);                    //          
  szDACString[i] = 0;                                             //@V3.0ET001
  pszDACName = &szDACString[0];                                   //@V3.0ET001

  pChip = ChipsetName[SVGAHardware.AdapterType][SVGAHardware.ChipType-1];
  /*
  ** Tokenize the OEMString and verify that we have matched OEMString
  ** chip name. Set the level of support for this adapter
  */
  strcpy(szOEMString,pAdapter->Adapter.szOEMString);
  for(i=0;i<256 && szOEMString[i] != ' ';i++);
  szOEMString[i] = 0;
  pszChipName = &szOEMString[0];
  while(szOEMString[++i] == ' ');    //skip all the blanks
  for(j=i;i<256 && (szOEMString[i] != ' ') && (szOEMString[i] != ',');i++);
  szOEMString[i] = 0;
  pszAdapterName = &szOEMString[j];
  /*
  ** Verify that the chipname corresponds to the identified chip.
  */
  if (strcmp(pszChipName,pChip))
     rc = ERROR_ADAPTER_NOT_SUPPORTED;          //fail the pfnPMIIdentifyAdapter
  else
  {
    /*
    ** Verify that the adapter string corresponds to the identified
    ** hardware and or what we can handle. 
    ** This function should examine the chiptype, the dac type and the
    ** revision string (manufacturer ID) from the PMI file/screendd and
    ** determine the internal key to be used in the clock support function.
    ** The clock support is keyed off the Manufacturer ID and additional
    ** Manufacturer Data. If there is no Manuf. ID for the chip/adapter in
    ** question, establish a ManufacturerData key off the manufacturer ID
    ** for the chip. For example, 864 has two implementations depending on the
    ** DAC, so use common S3_MANUFACTURER and ICD2061_CLOCK or SDAC_CLOCK manuf.
    ** data.                                                        
    */
    switch(SVGAHardware.AdapterType)
    {
      case TSENG_ADAPTER      :
        if ((SVGAHardware.ChipType == TSENG_ET4000_CHIP) &&
             (OEMHardware.Manufacturer == DIAMOND_MANUFACTURER))
             flAdapterSupport = strcmp(pszAdapterName, "SPEEDSTAR") ?
                 ADAPTER_NOT_SUPPORTED : ADAPTER_CLOCK_SUPPORTED;
        else if ((SVGAHardware.ChipType > TSENG_ET4000_CHIP) && /*            */
                  !strcmp(pszAdapterName, "STEALTH"))
             {
                OEMHardware.Manufacturer = DIAMOND_MANUFACTURER;
                StealthOldScheme = TRUE;        /*            */
                if (!strcmp(pszDACName,SGS1702_NAME) &&
                  (pAdapter->Adapter.ulTotalMemory >= 0x200000))     //JWK22
                  StealthOldScheme = FALSE;        /*            */

                flAdapterSupport = ADAPTER_CLOCK_SUPPORTED;
             }
             else                                               /*            */
               flAdapterSupport = ADAPTER_NOT_SUPPORTED;
        break;
      case WESTERNDIG_ADAPTER :
        if (((SVGAHardware.ChipType == WESTERNDIG_WD9030_CHIP) ||
             (SVGAHardware.ChipType == WESTERNDIG_WD9031_CHIP)) &&
             (OEMHardware.Manufacturer == DIAMOND_MANUFACTURER))
             flAdapterSupport = strcmp(pszAdapterName, "SPEEDSTAR") ?
                 ADAPTER_NOT_SUPPORTED : ADAPTER_CLOCK_SUPPORTED;
        else
             flAdapterSupport = ADAPTER_NOT_SUPPORTED;
        break;
      case S3_ADAPTER :
        switch(SVGAHardware.ChipType)
        {
           case S3_86C805_CHIP:
           case S3_86C928_CHIP:
              if (!strcmp(pszAdapterName,"STB_CARD"))
              {
                OEMHardware.Manufacturer = STB_MANUFACTURER; //late addition
                flAdapterSupport = ADAPTER_CLOCK_SUPPORTED;
                break;
              }
              switch(OEMHardware.Manufacturer)
              {
                case DIAMOND_MANUFACTURER:
//@senja                    flAdapterSupport = strcmp(pszAdapterName, "STEALTH") ?
//@senja                       ADAPTER_NOT_SUPPORTED : ADAPTER_CLOCK_SUPPORTED;
                    flAdapterSupport = ADAPTER_NOT_SUPPORTED;
                    break;
                case NUMBER9_MANUFACTURER:
                    flAdapterSupport = strcmp(pszAdapterName, "9GXE") ?
                       ADAPTER_NOT_SUPPORTED : ADAPTER_CLOCK_SUPPORTED;
                    break;
                case LACUNA_MANUFACTURER:                        /*@V3.0YEE04*/
                    flAdapterSupport = ADAPTER_CLOCK_SUPPORTED;  /*@V3.0YEE04*/
                    break;                                       /*@V3.0YEE04*/
                default:
                   flAdapterSupport = ADAPTER_NOT_SUPPORTED;
                   if (!strcmp(pszAdapterName, "GENERIC") )
                   {
                     OEMHardware.Manufacturer = S3_MANUFACTURER;
                     flAdapterSupport =ADAPTER_CLOCK_SUPPORTED;
                     OEMHardware.ManufacturerData = GENERIC_S3_CLOCK;     /*          */
                   }
              }
              break;
           case S3_86C864_CHIP:
              if ((!strcmp(pszAdapterName,"ICD2061")) ||         /*@V3.0YEE01*/
                  (!strcmp(pszDACName,S3SDAC_NAME)))                /*          */
              {                                                  /*@V3.0YEE01*/
                OEMHardware.Manufacturer = S3_MANUFACTURER;      /*@V3.0YEE01*/
                flAdapterSupport = ADAPTER_CLOCK_SUPPORTED;      /*@V3.0YEE01*/
                if (!strcmp(pszAdapterName,"ICD2061"))           /*@V3.0ET001*/
                  OEMHardware.ManufacturerData = ICD2061_CLOCK;  /*@V3.0YEE01*/
                else                                             /*@V3.0ET001*/
                  OEMHardware.ManufacturerData = SDAC_CLOCK;     /*@V3.0ET001*/
              }
              else
                flAdapterSupport = ADAPTER_NOT_SUPPORTED;
              break;
           default:
              flAdapterSupport = ADAPTER_NOT_SUPPORTED;
        }
        break;
      case ATI_ADAPTER:
        OEMHardware.Manufacturer = ATI_MANUFACTURER;      /*          */
        switch(SVGAHardware.ChipType)
        {
           case ATI_68800_CHIP:
              flAdapterSupport = strcmp(pszAdapterName, "GENERIC") ?
                 ADAPTER_NOT_SUPPORTED : ADAPTER_CLOCK_SUPPORTED;
              break;
           case ATI_88800_CHIP:
              flAdapterSupport = strcmp(pszAdapterName, "GENERIC") ?
                 ADAPTER_NOT_SUPPORTED : ADAPTER_MODE_SUPPORTED;
              break;
           default:
              flAdapterSupport = ADAPTER_NOT_SUPPORTED;
        }
        break;
      case CIRRUS_ADAPTER:                      /* @V3.0JAO01 start */
        switch(SVGAHardware.ChipType)
        {
           case CIRRUS_543X_CHIP:
           case CIRRUS_5434_CHIP:
              if (!strcmp(pszAdapterName,"STB_CARD"))
              {
                OEMHardware.Manufacturer = STB_MANUFACTURER;
                flAdapterSupport = ADAPTER_CLOCK_SUPPORTED;
                break;
              }
              else if (!strcmp(pszAdapterName,"GENERIC"))
              {
                OEMHardware.Manufacturer = CIRRUS_MANUFACTURER; //late addition
                flAdapterSupport = ADAPTER_CLOCK_SUPPORTED;
                break;
              }
              else
                flAdapterSupport = ADAPTER_NOT_SUPPORTED;
           default:
           case CIRRUS_5426_CHIP:
           case CIRRUS_5428_CHIP:
           case CIRRUS_5429_CHIP:
              flAdapterSupport = ADAPTER_NOT_SUPPORTED;
        }
        break;                                  /* @V3.0JAO01 end   */
      case WEITEK_ADAPTER:
        if(OEMHardware.Manufacturer == DIAMOND_MANUFACTURER)
             flAdapterSupport = strcmp(pszAdapterName, "VIPER") ?
                 ADAPTER_NOT_SUPPORTED : ADAPTER_CLOCK_SUPPORTED;
        break;
      default:
        flAdapterSupport = ADAPTER_NOT_SUPPORTED;
    }
  }

  return rc;

}
/*****************************************************************************
 *
 *  FUNCTION NAME:      pfnSetMode()
 *
 *  DESCRIPTIVE NAME:   Set the mode according to the input parameter.
 *
 *  FUNCTION:           This is the worker routine for the [SetMode] PMI
 *                      sections which elect to delegate the set mode functionality
 *                      to the imported PMI binary object. Such PMI section would
 *                      contain just " call pfnPMISetMode" reference. It is also possible
 *                      to mix PMI sequence and call the pfnSetMode within a SetMode
 *                      section (the assumption is that both PMI file and this
 *                      code are written by the same person who knows what is responsability
 *                      of the pfnSetMode and what is left to the PMI file). Note that the name
 *                      chosen is arbitrary, there is no limitation on the number or names
 *                      of the exported PMI workers.
 *
 *  INPUT:              PVIDEO_ADAPTER - Pointer to current state of the adapter/mode
 *                      PREGS- Pointer to current register state
 *
 *  EXIT:               APIRET - return code
 *
 *  NOTES:              If called, it has to internally call pfnSetMonitorTimings
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
APIRET EXPENTRY pfnPMISetMode(PVIDEO_ADAPTER pAdapter,PREGS pRegs)
{
  if (flAdapterSupport & ADAPTER_MODE_SUPPORTED)//proceed
                                                // S3 864 and ATI64 to be added
    return NO_ERROR;
  else
    return ERROR_ADAPTER_NOT_SUPPORTED;
}
/*****************************************************************************
 *
 *  FUNCTION NAME:      pfnSetMonitorTimings()
 *
 *  DESCRIPTIVE NAME:   Program the clock related registers based on the input data.
 *
 *  FUNCTION:
 *
 *  INPUT:              PVIDEO_ADAPTER - Pointer to current state of the adapter/mode
 *                      PREGS- Pointer to current register state
 *
 *  EXIT:               APIRET - return code
 *
 *  NOTES:              It is not necessary to verify the manufacturer if this
 *                      code handles only one manufacturer per chip. In that case,
 *                      the identify adapter would fail to identify manufacturers which
 *                      are not handled thru the ibmgpmi.dll so no futher calls into this
 *                      dll for services would be expected. However, if there are
 *                      multiple manufacturers (adapter) handled, then this function has
 *                      to specifically identify which one is being handled.
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 * Tables obtained from various BIOS' source code.
 * Diamond Speedstar 24:
 *ET4000         Bios < 6   Bios > = 6
 * 640 60         a23d15     823d15
 * 640 72         b2c422     d2c422
 * 800 56         f2842a     b2842a
 * 800 60         8a040e     8a040e
 * 800 72         a2bc35     82dc35
 * 1024 60        b2c422     d2c422
 * 1024 70        f21435     f21435
 * 1024 72        f23427     8a441e
 * 1024 int       c25410     cc2910
 * 40,80          923506     a23506
 * 132            8a040e     8a040e                     Tested bvhsvga
 *WD             Bios < 2   Bios >= 2        Bios < 2   Bios >= 2
 * 640 60         a23d15     86ac1e                      86ac1e
 * 640 72         d2ad33     92ad33                      ?
 * 800 56         f2852a     b2852a                      b2852a
 * 800 60         a2bc2a     ee153d                      ee153d
 * 800 72         a2bc35     82bc35                      ?
 * 1024 int       c25410     aa2910                      aa2910
 * 1024 60        b2c422     d2c422  ?
 * 1024 70        f21435     f21435  ?
 * 1024 72        f29423     8a9423  ?
 * 40,80          923506     86ac1e                      86ac1e
 * 132            c25410     aa2910                      aa2910
 *
 *Number 9 GXE:
 *
 *  640 @ 70Hz:  08FA170h
 *  640 @ 76Hz:  093B16Ah
 *  640 @ 60Hz:  08AC158h
 *  640 @ 72Hz:  08F896Ah
 *  800 @ 70Hz:  0AB5942h
 *  800 @ 76Hz:  0AF8940h
 *  800 @ 56Hz:  09AA942h
 *  800 @ 72Hz:  0AEF138h
 * 1024 @ 70Hz:  0A2F846h
 * 1024 @ 76Hz:  0A7804Ch
 * 1024 @ 60Hz:  0968846h
 * 1024 @ 72Hz:  0A26036h
 * 1280 @ 70Hz:  093D86Eh
 * 1280 @ 76Hz:  09B785Ah
 * 1280 @ 60Hz:  08B486Eh
 * 1280 @ 72Hz:  0937862h
 * 1600 @ 60Hz:  0AAE03Ch
 * 1600 @ 60Hz:  0AAE03Ch
 * 1600 @ 60Hz:  0AAE03Ch
 * 1600 @ 60Hz:  0AAE03Ch
 * 132 columns:  0A2E140h
 *  80 columns:  0835178h
 *
 *Viper VLB:
 * 640 @ 60:    0x045a8bc
 * 640 @ 72:    0x04bd8b5;
 * 800 @ 56:    0x04f54a1;
 * 800 @ 72:    0x045ac3d;
 * 1024 @ 60:   0x04d4423;
 * 1024 @ 70:   0x04FAC28;
 * 1024 @ 72:   0x04F7821;
 * 1024 @ 76:   0x04F8021;
 * 1280 @ 60:   0x05BF81E;
 * 1280 @ 74:   0x05B8013;
 *
 * STB values for IDC2061a              V2.2SENJA fixing few entries
 * Res        Serial    MHz
 * 640 @ 60   0x01A8BC  25.1723
 * 640 @ 76   0x4B44A3  32.5061
 * 720 @ 60   0x2560AC  28.3251
 * 800 @ 56   0x4FACA8  37.5000  ?
 * 800 @ 60   0x51A8A5  40.0174 640 x 480 @ 90            
 * 800 @ 70   0x55C4A3  44.999  1024 @ 60                 
 * 800 @ 72   0x56D88F  48.0080                           
 * 800 @ 75   0x41AC3D  50.0000 640x480x16bpp @ 60            
 * 1024 @ 44i 0x55C4A3  44.8892
 * 1024 @ 60  0x4B4423  65.0123            
 * 1024 @ 70  0x4FAC28  75.0000 640x480x16bpp @ 75, 800x600x16bpp @ 56            
 * 1024 @ 75  0x538021  81.0000 1280 @ 46i, 800x600x16bpp @ 60                    
 *  0x55F829  85.9091  1024x768x16bpp @ 43i, 1280x1024 @ 46i
 *  0x571814  95.0206  640x480x16bpp at 72                                        
 *  0x57F823  99.8403  800x600x16bpp at 70
 *  0x591C13 100.9091  800x600x16bpp @ 72
 *  0x594014 108.0372
 *  0x59BC1C 108.8182
 *  0x59CC1D 109.0029
 * 1280 @ 60  0x59F81F 111.9421
 *  0x5BD81C 115.5000
 *  0x5B5413 120.0000
 *
 *S3864 with ICD2061:
 *  640 @ 75Hz:  049D8B5h   8bpp;16bpp
 *  640 @ 72Hz:  049D8B5h
 *  640 @ 60Hz:  041A8BCh
 *
 *  640 @ 75Hz:  04B4423h   32bpp
 *  640 @ 72Hz:  04B4423h
 *  640 @ 60Hz:  041AC3Dh
 *
 *  800 @ 75Hz:  041AC3Dh   4bpp;8bpp;16bpp
 *  800 @ 72Hz:  041AC3Dh
 *  800 @ 60Hz:  05170A0h
 *  800 @ 56Hz:  05170A0h
 *
 *  800 @ 75Hz:  057801Ch   32bpp
 *  800 @ 72Hz:  057801Ch
 *  800 @ 60Hz:  051D82Bh
 *  800 @ 56Hz:  051D82Bh
 *
 * 1024 @ 75Hz:  05170A0h   8bpp
 * 1024 @ 70Hz:  04FACA8h
 * 1024 @ 60Hz:  04B44A3h
 * 1024 @ 43Hz:  0550894h
 *
 * 1024 @ 75Hz:  0517020h   4bpp;16bpp
 * 1024 @ 70Hz:  04FAC28h
 * 1024 @ 60Hz:  04B4423h
 * 1024 @ 43Hz:  0550894h
 *
 * 1280 @ 75Hz:  04D8028h
 * 1280 @ 72Hz:  04B4423h
 * 1280 @ 60Hz:  045D83Dh
 * 1152 @ 60Hz:  0517020h
 * 1600 @ 60Hz:  04FAC28h
 * 132 columns:  05170A0h
 * Diamond Stealth : see DIA_ClockTable in ipmidata.c
 *
 * This is the strategy for Stealth 16bpp and 24 bpp
 * 16bpp require ICD to program 8bpp pixel clock and DAC to use command value=3
 * which will double the pixel rate. (ATC 16[5:4]=10)
 * 24bpp on 1700 (1 & 2MB), 1702 (1MB), requires ICD to setup 3*8bpp pixel clock
 * DAC command register value = 4F and ATC 16[5:4]=00.  (this is called
 * StealthOldScheme). In this scheme, pixel clocks are limited to 85Mz, so that
 * only 60Hz refresh is supported for 16/24 bpp.
 * 24bpp on 1702 (2MB), requires ICD to setup 1.5*8bpp pixel clock and DAC
 * command register value = 9 and  ATC 16[5:4]=10.
 * pfnPMIFixupClock should be called from the PMI before ProgramDAC
 * with the register values already set for the DAC mode and the function will fix it up
 * and will reprogram ATC 16. ProgramDAC can then propagate changed r0 into the command
 * register.
 * Due to the fact that old scheme allows only upto 60Hz refresh for 24bpp,
 * we are going to limit our support to the same even with the 1702/2MB as to
 * make this piece of code more maintainable.
 *   hres yres refr bpp index into DIA_ClockTable
 *   640, 480, 60, 4,0x00
 *   640, 480, 60, 8,0x00
 *   640, 480, 60,16,0x00
 *   640, 480, 60,24,0x07       if StealthOldScheme
 *   640, 480, 60,24,0x17
 *   640, 480, 72, 4,0x02
 *   640, 480, 72, 8,0x02
 *   640, 480, 72,16,0x02
 *   640, 480, 72,24,0x13       not supported
 *   640, 480, 75, 4,0x02
 *   640, 480, 75, 8,0x02
 *   640, 480, 75,16,0x02
 *   640, 480, 75,24,0x18       not supported
 *   640, 480, 75,24,0x18
 *   640, 480, 90, 4,0x04
 *   640, 480, 90, 8,0x04
 *   640, 480, 90,16,0x04
 *   640, 480, 90,24,0x14        not supported
 *   800, 600, 56, 4,0x03
 *   800, 600, 56, 8,0x03
 *   800, 600, 56,16,0x03
 *   800, 600, 56,24,0x16       not supported if StealthOldScheme
 *   800, 600, 60, 4,0x04
 *   800, 600, 60, 8,0x04
 *   800, 600, 60,16,0x04
 *   800, 600, 60,24,0x14       not supported if StealthOldScheme
 *   800, 600, 72, 4,0x10
 *   800, 600, 72, 8,0x10
 *   800, 600, 72,16,0x10
 *   800, 600, 72,24,0x07       not supported if StealthOldScheme
 *   800, 600, 75, 4,0x0d
 *   800, 600, 75, 8,0x0d
 *   800, 600, 75,16,0x0d
 *   800, 600, 75,24,0x15       not supported if StealthOldScheme
 *   800, 600, 90, 4,0x09
 *   800, 600, 90, 8,0x09
 *   800, 600, 90,16,0x09
 *  1024, 768, 43, 4,0x05
 *  1024, 768, 43, 8,0x05
 *  1024, 768, 43,16,0x05
 *  1024, 768, 60, 4,0x08
 *  1024, 768, 60, 8,0x08
 *  1024, 768, 60,16,0x08
 *  1024, 768, 70, 4,0x07
 *  1024, 768, 70, 8,0x07
 *  1024, 768, 70,16,0x07
 *  1024, 768, 72, 4,0x0b
 *  1024, 768, 72, 8,0x0b
 *  1024, 768, 72,16,0x0b
 *  1024, 768, 75, 4,0x0e
 *  1024, 768, 75, 8,0x0e
 *  1024, 768, 75,16,0x0e
 *  1280,1024, 43, 4,0x12
 *  1280,1024, 43, 8,0x12
 *  1280,1024, 60, 8,0x12
 *  1280,1024, 72, 8,0x12
 *  1280,1024, 75, 8,0x12
 ****************************************************************************/
APIRET EXPENTRY  pfnPMISetMonitorTimings(PVIDEO_ADAPTER pAdapter,PREGS pRegs)
{
   ULONG ClockSerialData;
   ULONG rc = NO_ERROR;
   USHORT BiosMajorRev;
   USHORT crtcport = (_inp(0x3cc) & 0x01) ? 0x3D4 : 0x3B4;       /*@V3.0YEE01*/
   BYTE r0, r1, r2, r3, r4;                                      /*@V3.0ET001*/
   USHORT i, j;                                                  /*@V3.0YEE01*/

   if (!(flAdapterSupport & ADAPTER_CLOCK_SUPPORTED))                //proceed
      return ERROR_ADAPTER_NOT_SUPPORTED;

   if (OEMHardware.Manufacturer == CIRRUS_MANUFACTURER)          /*@V3.0JAO01*/
   {                                                             /*@V3.0JAO01*/
     rc = SetCirrusClock((PVIDEO_ADAPTER) pAdapter);             /*@V3.0JAO01*/
     return rc;                                                  /*@V3.0JAO01*/
   }                                                             /*@V3.0JAO01*/

   /*
   ** STB ICD2061A based cards:
   ** Cirrus and S3 928
   */
   if(OEMHardware.Manufacturer == STB_MANUFACTURER)
   {
      if (!(pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS))
          ClockSerialData =0x2560AC;
      else
      {
        switch(pAdapter->ModeInfo.usXResolution)
        {
          case 640:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                case 90:                        /*            */
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                  {
                    ClockSerialData =0x051a8a5;
                    break;
                  }
                case 76:                        /*            */
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                    ClockSerialData =0x04B44A3;
                  else
                    ClockSerialData =0x04FAC28;
                  break;
                default:
                case 60:                        /*            */
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                    ClockSerialData =0x01A8BC;
                  else
                    ClockSerialData =0x41AC3D;
                  break;
             }
             break;
          case 800:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                case 76:
                case 75:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                  {
                    ClockSerialData =0x041AC3D; /*            */
                    break;
                  }
                case 72:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                    ClockSerialData =0x56D88F;  /*            */
                  else
                    ClockSerialData =0x0591c13;
                  break;
                case 70:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                    ClockSerialData =0x055C4A3;
                  else
                    ClockSerialData =0x057F823;
                  break;
                case 60:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                    ClockSerialData =0x051A8A5;
                  else
                    ClockSerialData =0x0538021;
                  break;
                default:
                case 56:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                    ClockSerialData =0x04FACA8;
                  else
                    ClockSerialData =0x04FAC28;
                  break;
             }
             break;
          case 1024:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                case 75:
                case 74:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                  {
                    ClockSerialData =0x538021;
                    break;
                  }
                case 70:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                  {
                    ClockSerialData =0x4FAC28;
                    break;
                  }
                case 60:
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                  {
                    ClockSerialData =0x04B4423;
                    break;
                  }
                default:
                case 43:                /*            */
                  if (pAdapter->ModeInfo.bBitsPerPixel <= 8)
                    ClockSerialData =0x055C4A3;
                  else
                    ClockSerialData =0x055F829;
                  break;
             }
             break;
          case 1280:
             if (pAdapter->ModeInfo.bVrtRefresh == 60)
                ClockSerialData =0x059F81F;     //60
             else if (pAdapter->ModeInfo.bVrtRefresh <= 46)
                ClockSerialData =0x055F829;     //46i
             else
                rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
          default:
             ClockSerialData =0x01A8BC;
        }
      }
      if (!rc)
        SetSTBClock(ClockSerialData);
      return(rc);
   }

   if (SVGAHardware.AdapterType == S3_ADAPTER)
   {
      if  (SVGAHardware.ChipType <= S3_86C928_CHIP)
      {
        if (OEMHardware.ManufacturerData == GENERIC_S3_CLOCK)
        {
          /*
          ** Generic on-chip clock for S3
          */
          if (pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS)
          {
            SetS3Clock((ULONG) pAdapter->ModeInfo.usXResolution,
                     (ULONG) pAdapter->ModeInfo.bBitsPerPixel,
                     (ULONG) pAdapter->ModeInfo.bVrtRefresh);
            return rc;
          }
          else
            return(ERROR_MODE_NOT_SUPPORTED);
        }
        /*
        ** Number 9 GXE VLB/ISA
        ** All Number 9 clock values shifted right 1 due to        @V3.0YEE01
        ** change in SETNUMBER9CLK                                 @V3.0YEE01
        */
        if  (OEMHardware.Manufacturer == NUMBER9_MANUFACTURER)
        {
          if (!(pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS))
          {
            if (pAdapter->ModeInfo.usBytesPerScanLine == 132)
/*            ClockSerialData = 0x0A2E140;                         @V3.0YEE01*/
              ClockSerialData = 0x05170A0;                       /*@V3.0YEE01*/
            else
/*            ClockSerialData =0x0835178;                          @V3.0YEE01*/
              ClockSerialData =0x041A8BC;                        /*@V3.0YEE01*/
          }
          else if (pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS)
          {
            switch(pAdapter->ModeInfo.usXResolution)
            {
              case 640:
                 switch(pAdapter->ModeInfo.bVrtRefresh)
                 {
                    case 70:
/*                    ClockSerialData =0x08FA170;              @V3.0YEE01*/
                      ClockSerialData =0x047D0B8;            /*@V3.0YEE01*/
                      break;
                    case 76:
/*                    ClockSerialData =0x093B16A;              @V3.0YEE01*/
                      ClockSerialData =0x049D8B5;            /*@V3.0YEE01*/
                      break;
                    case 60:
/*                    ClockSerialData =0x08AC158;              @V3.0YEE01*/
                      ClockSerialData =0x04560AC;            /*@V3.0YEE01*/
                      break;
                    case 72:
/*                    ClockSerialData =0x08F896A;              @V3.0YEE01*/
                      ClockSerialData =0x047C4B5;            /*@V3.0YEE01*/
                      break;
                    default:
/*                    ClockSerialData =0x08AC158;              @V3.0YEE01*/
                      ClockSerialData =0x04560AC;            /*@V3.0YEE01*/
                      break;
                 }
                 break;
              case 800:
                 switch(pAdapter->ModeInfo.bVrtRefresh)
                 {
                    case 70:
/*                    ClockSerialData =0x0AB5942;                  @V3.0YEE01*/
                      ClockSerialData =0x055ACA1;                /*@V3.0YEE01*/
                      break;
                    case 76:
/*                    ClockSerialData =0x0AF8940;                  @V3.0YEE01*/
                      ClockSerialData =0x057C4A0;                /*@V3.0YEE01*/
                      break;
                    case 56:
/*                    ClockSerialData =0x09AA942;                  @V3.0YEE01*/
                      ClockSerialData =0x04D54A1;                /*@V3.0YEE01*/
                      break;
                    case 72:
/*                    ClockSerialData =0x0AEF138;                  @V3.0YEE01*/
                      ClockSerialData =0x057789C;                /*@V3.0YEE01*/
                      break;
                    default:
/*                    ClockSerialData =0x09AA942;                  @V3.0YEE01*/
                      ClockSerialData =0x04D54A1;                /*@V3.0YEE01*/
                      break;
                 }
                 break;
              case 1024:
                 switch(pAdapter->ModeInfo.bVrtRefresh)
                 {
                    case 70:
/*                    ClockSerialData =0x0A2F846;                  @V3.0YEE01*/
                      ClockSerialData =0x0517C23;                /*@V3.0YEE01*/
                      break;
                    case 76:
/*                    ClockSerialData =0x0A7804C;                  @V3.0YEE01*/
                      ClockSerialData =0x053C026;                /*@V3.0YEE01*/
                      break;
                    case 60:
/*                    ClockSerialData =0x0968846;                  @V3.0YEE01*/
                      ClockSerialData =0x04B4423;                /*@V3.0YEE01*/
                      break;
                    case 72:
/*                    ClockSerialData =0x0A26036;                  @V3.0YEE01*/
                      ClockSerialData =0x051301B;                /*@V3.0YEE01*/
                      break;
                    default:
/*                    ClockSerialData =0x0968846;                  @V3.0YEE01*/
                      ClockSerialData =0x04B4423;                /*@V3.0YEE01*/
                      break;
                 }
                 break;
              case 1280:
                 switch(pAdapter->ModeInfo.bVrtRefresh)
                 {
                    case 70:
/*                    ClockSerialData =0x093D86E;                  @V3.0YEE01*/
                      ClockSerialData =0x049EC37;                /*@V3.0YEE01*/
                      break;
                    case 76:
/*                    ClockSerialData =0x09B785A;                  @V3.0YEE01*/
                      ClockSerialData =0x04DBC2D;                /*@V3.0YEE01*/
                      break;
                    case 60:
/*                    ClockSerialData =0x08B486E;                  @V3.0YEE01*/
                      ClockSerialData =0x045A437;                /*@V3.0YEE01*/
                      break;
                    case 72:
/*                    ClockSerialData =0x0937862;                  @V3.0YEE01*/
                      ClockSerialData =0x049BC31;                /*@V3.0YEE01*/
                      break;
                    default:
/*                    ClockSerialData =0x08B486E;                  @V3.0YEE01*/
                      ClockSerialData =0x045A437;                /*@V3.0YEE01*/
                      break;
                 }
                 break;
              case 1600:
                 if (pAdapter->ModeInfo.bVrtRefresh == 60)
/*                  ClockSerialData = 0x0AAE03C;                   @V3.0YEE01*/
                    ClockSerialData = 0x055701E;                 /*@V3.0YEE01*/
                 else
                    rc = ERROR_REFRESH_NOT_SUPPORTED;
                 break;
              default:
/*               ClockSerialData =0x0835178;                       @V3.0YEE01*/
                 ClockSerialData =0x041A8BC;                     /*@V3.0YEE01*/
            }
          }             /* end MODE_FLAG_GRAPHICS test   */
          if (!rc)
               SETNUMBER9CLK(ClockSerialData);
          return rc;
        }               /* end NUMBER9_MANUFACTURER test */

        if (OEMHardware.Manufacturer == LACUNA_MANUFACTURER) /*@V3.0YEE04*/
        {                                                         /*@V3.0YEE04*/
            pfnSetS3Lacuna(pAdapter,pRegs);                       /*@V3.0YEE04*/
            return rc;                                            /*@V3.0YEE04*/
         }              /* end LACUNA_MANUFACTURER test */        /*@V3.0YEE04*/

        /*
        ** Diamond Stealth 24 and Pro (VRAM not supported)
        ** Monitor selection and configuration possible only thru a DOS utility.
        */
        if (OEMHardware.Manufacturer == DIAMOND_MANUFACTURER)
        {
//@senja    SETDIAMONDCLK_S3((ULONG) pAdapter->ModeInfo.usXResolution,
//                       (ULONG) pAdapter->ModeInfo.bBitsPerPixel);
            return rc;
        }              /* end DIAMOND_MANUFACTURER test */
        return rc;
      }                 /* end <= S3_86C928_CHIP test    */

      /*
      ** S3 864 with ICD 2061 external clock chip             start @V3.0YEE01
      */
      if ((OEMHardware.Manufacturer == S3_MANUFACTURER) &&
          (OEMHardware.ManufacturerData == ICD2061_CLOCK))
      {
        r0 = _inp(0x3cc);
        if ((r0 & 0x0c) != 0x0c)
          return (ERROR_REFRESH_NOT_SUPPORTED);

        pfnSetS3864ICD(pAdapter,pRegs);                       /*@V3.0YEE03*/
        return rc;
      }                 /* end S3_86C864_CHIP & ICD2061     end @V3.0YEE01*/

      /*
      ** S3 864 with S3DAC                                      @V3.0ET001
      */

      if ((OEMHardware.Manufacturer == S3_MANUFACTURER) &&
          (OEMHardware.ManufacturerData == SDAC_CLOCK))
      {
        if (!(pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS))
        {
          if (pAdapter->ModeInfo.usBytesPerScanLine == 132)
          {
            r0 = 0x5d;
            r1 = 0x2f;     /* 132 cols*/
            r2 = 0x55;
            r3 = 0x29;
            r4 = 0x00;
          }
          else
          {
            r0 = 0x6b;
            r1 = 0x24;     /* 40 and 80 cols*/
            r2 = 0x51;
            r3 = 0x29;
            r4 = 0x00;
          }
        }
        else              /* graphics mode */
        {
          switch(pAdapter->ModeInfo.usXResolution)
          {
            case 640:
              switch(pAdapter->ModeInfo.bBitsPerPixel)
              {
                case 4:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 85:        /*do 75*/
                    case 75:
                    case 72:

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0x6400);
                      _outpw(crtcport, 0x4f02);
                      _outpw(crtcport, 0x8003);
                      _outpw(crtcport, 0x5204);
                      _outpw(crtcport, 0x1a05);
                      _outpw(crtcport, 0xf206);
                      _outpw(crtcport, 0x1f07);
                      _outpw(crtcport, 0xff0e);

                      _outpw(crtcport, 0xe010);
                      _outpw(crtcport, 0x8311);
                      _outpw(crtcport, 0xdf15);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0xf55b);
                      _outpw(crtcport, 0x025c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x5f;
                      r1 = 0x49;
                      r2 = 0x51;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    default: /*60*/

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0x5f00);
                      _outpw(crtcport, 0x5002);
                      _outpw(crtcport, 0x8203);
                      _outpw(crtcport, 0x5404);
                      _outpw(crtcport, 0x8005);
                      _outpw(crtcport, 0x0b06);
                      _outpw(crtcport, 0x3e07);
                      _outpw(crtcport, 0x000e);

                      _outpw(crtcport, 0xea10);
                      _outpw(crtcport, 0x8c11);
                      _outpw(crtcport, 0xe715);
                      _outpw(crtcport, 0x0416);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x005c);

                      _outpw(crtcport, 0x0042);

                      r0 = 0x60;
                      r1 = 0x4c;
                      r2 = 0x51;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

                case 8:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 85:

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0x5f00);
                      _outpw(crtcport, 0x4f02);
                      _outpw(crtcport, 0x8003);
                      _outpw(crtcport, 0x5204);
                      _outpw(crtcport, 0x1e05);
                      _outpw(crtcport, 0x0d06);
                      _outpw(crtcport, 0x3e07);

                      _outpw(crtcport, 0xeb10);
                      _outpw(crtcport, 0x9d11);
                      _outpw(crtcport, 0xe015);
                      _outpw(crtcport, 0x0d16);

                      _outpw(crtcport, 0x4052);
                      _outpw(crtcport, 0xf65b);

                      _outpw(crtcport, 0x583b);
                      _outpw(crtcport, 0x2f3c);

                      r0 = 0x77;
                      r1 = 0x4a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 75:           /*do 75*/
                    case 72:

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0x6400);
                      _outpw(crtcport, 0x4f02);
                      _outpw(crtcport, 0x8003);
                      _outpw(crtcport, 0x5204);
                      _outpw(crtcport, 0x1a05);
                      _outpw(crtcport, 0xf206);
                      _outpw(crtcport, 0x1f07);

                      _outpw(crtcport, 0xe010);
                      _outpw(crtcport, 0x8311);
                      _outpw(crtcport, 0xdf15);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0xf55b);

                      _outpw(crtcport, 0x5d3b);
                      _outpw(crtcport, 0x323c);

                      r0 = 0x5f;
                      r1 = 0x49;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    default: /*60*/

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0x5f00);
                      _outpw(crtcport, 0x5002);
                      _outpw(crtcport, 0x8203);
                      _outpw(crtcport, 0x5204);
                      _outpw(crtcport, 0x9e05);
                      _outpw(crtcport, 0x0b06);
                      _outpw(crtcport, 0x3e07);

                      _outpw(crtcport, 0xe910);
                      _outpw(crtcport, 0x8b11);
                      _outpw(crtcport, 0xe715);
                      _outpw(crtcport, 0x0416);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);

                      _outpw(crtcport, 0x583b);
                      _outpw(crtcport, 0x2f3c);

                      r0 = 0x60;
                      r1 = 0x4c;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

                case 16:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 85:           /*do 75*/
                    case 75:
                    case 72:

                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xcf00);
                      _outpw(crtcport, 0x9203);
                      _outpw(crtcport, 0xa504);
                      _outpw(crtcport, 0x1505);
                      _outpw(crtcport, 0xf106);
                      _outpw(crtcport, 0x1f07);

                      _outpw(crtcport, 0xe010);
                      _outpw(crtcport, 0x8311);
                      _outpw(crtcport, 0xdf15);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0xf55b);
                      _outpw(crtcport, 0x5067);

                      _outpw(crtcport, 0xc83b);
                      _outpw(crtcport, 0x673c);

                      r0 = 0x5f;
                      r1 = 0x49;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;

                    default: /*60*/

                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xc200);
                      _outpw(crtcport, 0x8403);
                      _outpw(crtcport, 0xa304);
                      _outpw(crtcport, 0x1b05);
                      _outpw(crtcport, 0x0c06);
                      _outpw(crtcport, 0x3e07);

                      _outpw(crtcport, 0xe910);
                      _outpw(crtcport, 0x8b11);
                      _outpw(crtcport, 0xe715);
                      _outpw(crtcport, 0x0416);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x5067);

                      _outpw(crtcport, 0xbb3b);
                      _outpw(crtcport, 0x613c);

                      r0 = 0x60;
                      r1 = 0x4c;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

                case 24:
                case 32:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 85:
                    case 75:
                    case 72:

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0xad00);
                      _outpw(crtcport, 0x8703);
                      _outpw(crtcport, 0x4904);
                      _outpw(crtcport, 0x8a05);
                      _outpw(crtcport, 0xf306);
                      _outpw(crtcport, 0x1f07);

                      _outpw(crtcport, 0xe110);
                      _outpw(crtcport, 0x8411);
                      _outpw(crtcport, 0xe015);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0xf55b);
                      _outpw(crtcport, 0x7f5d);

                      _outpw(crtcport, 0xa63b);
                      _outpw(crtcport, 0xd63c);

                      r0 = 0x6b;
                      r1 = 0x44;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x70;
                      break;

                    default: /*60*/

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0x8a00);
                      _outpw(crtcport, 0x8803);
                      _outpw(crtcport, 0x4504);
                      _outpw(crtcport, 0x1505);
                      _outpw(crtcport, 0x0a06);
                      _outpw(crtcport, 0x3e07);

                      _outpw(crtcport, 0xe910);
                      _outpw(crtcport, 0x8b11);
                      _outpw(crtcport, 0xe715);
                      _outpw(crtcport, 0x0416);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x7f5d);

                      _outpw(crtcport, 0x833b);
                      _outpw(crtcport, 0xc53c);

                      r0 = 0x60;
                      r1 = 0x2c;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x70;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

              }                        /*end bpp switch*/
              break;                   /*from xresolution*/

            case 800:
              switch(pAdapter->ModeInfo.bBitsPerPixel)
              {
                case 4:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 85:
                    case 75:

                      _outpw(crtcport, 0x0e11);
                      _outpw(crtcport, 0x7f00);
                      _outpw(crtcport, 0x6302);
                      _outpw(crtcport, 0x6804);
                      _outpw(crtcport, 0x1205);
                      _outpw(crtcport, 0x6f06);
                      _outpw(crtcport, 0xe007);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8b11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0xf55b);
                      _outpw(crtcport, 0x085d);

                      _outpw(crtcport, 0x783b);
                      _outpw(crtcport, 0x3f3c);

                      r0 = 0x51;
                      r1 = 0x2a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 72:

                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7d00);
                      _outpw(crtcport, 0x6302);
                      _outpw(crtcport, 0x6d04);
                      _outpw(crtcport, 0x1c05);
                      _outpw(crtcport, 0x9906);
                      _outpw(crtcport, 0xf007);

                      _outpw(crtcport, 0x7c10);
                      _outpw(crtcport, 0xa211);
                      _outpw(crtcport, 0x9916);

                      _outpw(crtcport, 0x1052);
                      _outpw(crtcport, 0xd15b);
                      _outpw(crtcport, 0x085d);

                      _outpw(crtcport, 0x763b);
                      _outpw(crtcport, 0x3e3c);

                      r0 = 0x60;
                      r1 = 0x2c;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 56:

                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7b00);
                      _outpw(crtcport, 0x6402);
                      _outpw(crtcport, 0x6a04);
                      _outpw(crtcport, 0x1305);
                      _outpw(crtcport, 0x7006);
                      _outpw(crtcport, 0xf007);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8a11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0x0052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x005d);

                      _outpw(crtcport, 0x743b);
                      _outpw(crtcport, 0x3d3c);

                      r0 = 0x77;
                      r1 = 0x4a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    default: /*60*/

                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7f00);
                      _outpw(crtcport, 0x6302);
                      _outpw(crtcport, 0x6b04);
                      _outpw(crtcport, 0x1b05);
                      _outpw(crtcport, 0x7206);
                      _outpw(crtcport, 0xf007);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8c11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x085d);

                      _outpw(crtcport, 0x783b);
                      _outpw(crtcport, 0x3f3c);

                      r0 = 0x5d;
                      r1 = 0x2f;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

                case 8:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 85:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7f00);
                      _outpw(crtcport, 0x6302);
                      _outpw(crtcport, 0x6904);
                      _outpw(crtcport, 0x0f05);
                      _outpw(crtcport, 0x7506);
                      _outpw(crtcport, 0xe007);

                      _outpw(crtcport, 0x5710);
                      _outpw(crtcport, 0x8a11);
                      _outpw(crtcport, 0x7416);

                      _outpw(crtcport, 0x4052);
                      _outpw(crtcport, 0xf65b);
                      _outpw(crtcport, 0x085d);

                      _outpw(crtcport, 0x783b);
                      _outpw(crtcport, 0x3f3c);

                      r0 = 0x55;
                      r1 = 0x29;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 75:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7f00);
                      _outpw(crtcport, 0x6302);
                      _outpw(crtcport, 0x6604);
                      _outpw(crtcport, 0x1005);
                      _outpw(crtcport, 0x6f06);
                      _outpw(crtcport, 0xe007);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8b11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0xf55b);
                      _outpw(crtcport, 0x085d);

                      _outpw(crtcport, 0x783b);
                      _outpw(crtcport, 0x3f3c);

                      r0 = 0x51;
                      r1 = 0x2a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 72:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7d00);
                      _outpw(crtcport, 0x6302);
                      _outpw(crtcport, 0x6b04);
                      _outpw(crtcport, 0x1a05);
                      _outpw(crtcport, 0x9906);
                      _outpw(crtcport, 0xf007);

                      _outpw(crtcport, 0x7c10);
                      _outpw(crtcport, 0xa211);
                      _outpw(crtcport, 0x9916);

                      _outpw(crtcport, 0x1052);
                      _outpw(crtcport, 0xd15b);
                      _outpw(crtcport, 0x085d);

                      _outpw(crtcport, 0x763b);
                      _outpw(crtcport, 0x3e3c);

                      r0 = 0x60;
                      r1 = 0x2c;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 56:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7b00);
                      _outpw(crtcport, 0x6402);
                      _outpw(crtcport, 0x6804);
                      _outpw(crtcport, 0x1105);
                      _outpw(crtcport, 0x7006);
                      _outpw(crtcport, 0xf007);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8a11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0x0052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x005d);

                      _outpw(crtcport, 0x743b);
                      _outpw(crtcport, 0x3d3c);

                      r0 = 0x77;
                      r1 = 0x4a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;


                    default: /*60*/
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x7f00);
                      _outpw(crtcport, 0x6302);
                      _outpw(crtcport, 0x6904);
                      _outpw(crtcport, 0x1905);
                      _outpw(crtcport, 0x7206);
                      _outpw(crtcport, 0xf007);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8c11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x085d);

                      _outpw(crtcport, 0x783b);
                      _outpw(crtcport, 0x3f3c);

                      r0 = 0x5d;
                      r1 = 0x2f;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

                case 16:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 85:
                    case 75:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x0400);
                      _outpw(crtcport, 0x8703);
                      _outpw(crtcport, 0xcd04);
                      _outpw(crtcport, 0x0105);
                      _outpw(crtcport, 0x6f06);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8b11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0x9054);
                      _outpw(crtcport, 0xf55b);

                      _outpw(crtcport, 0xfd3b);
                      _outpw(crtcport, 0x823c);

                      r0 = 0x51;
                      r1 = 0x2a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;

                    case 72:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x0000);
                      _outpw(crtcport, 0x8303);
                      _outpw(crtcport, 0xd704);
                      _outpw(crtcport, 0x1505);
                      _outpw(crtcport, 0x9906);

                      _outpw(crtcport, 0x7c10);
                      _outpw(crtcport, 0x8211);
                      _outpw(crtcport, 0x9916);

                      _outpw(crtcport, 0x1052);
                      _outpw(crtcport, 0x9054);
                      _outpw(crtcport, 0xd15b);

                      _outpw(crtcport, 0xf93b);
                      _outpw(crtcport, 0x803c);

                      r0 = 0x60;
                      r1 = 0x2c;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;

                    default:           /*56,60*/
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x0300);
                      _outpw(crtcport, 0x8603);
                      _outpw(crtcport, 0xd304);
                      _outpw(crtcport, 0x1305);
                      _outpw(crtcport, 0x7206);

                      _outpw(crtcport, 0x5810);
                      _outpw(crtcport, 0x8c11);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xb854);
                      _outpw(crtcport, 0xa85b);

                      _outpw(crtcport, 0xfc3b);
                      _outpw(crtcport, 0x813c);

                      r0 = 0x5d;
                      r1 = 0x2f;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

              }                        /*end bpp switch*/
              break;                   /*from xresolution*/

            case 1024:
              switch(pAdapter->ModeInfo.bBitsPerPixel)
              {
                case 4:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 75:
                    case 72:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xa200);
                      _outpw(crtcport, 0x8303);
                      _outpw(crtcport, 0x8404);
                      _outpw(crtcport, 0x9005);
                      _outpw(crtcport, 0x1f06);
                      _outpw(crtcport, 0xfd07);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0110);
                      _outpw(crtcport, 0x8411);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0x0015);
                      _outpw(crtcport, 0x0016);
                      _outpw(crtcport, 0xeb17);


                      _outpw(crtcport, 0x0634);
                      _outpw(crtcport, 0x9b3b);
                      _outpw(crtcport, 0x513c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x41;
                      r1 = 0x0a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 70:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xa200);
                      _outpw(crtcport, 0x8503);
                      _outpw(crtcport, 0x8504);
                      _outpw(crtcport, 0x9605);
                      _outpw(crtcport, 0x1f06);
                      _outpw(crtcport, 0xf507);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0210);
                      _outpw(crtcport, 0x8811);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0xff15);
                      _outpw(crtcport, 0x1f16);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xac5b);
                      _outpw(crtcport, 0x005d);
                      _outpw(crtcport, 0x0167);

                      _outpw(crtcport, 0x0234);
                      _outpw(crtcport, 0x9b3b);
                      _outpw(crtcport, 0x513c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x28;
                      r1 = 0x22;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 43:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x9a00);
                      _outpw(crtcport, 0x9d03);
                      _outpw(crtcport, 0x8304);
                      _outpw(crtcport, 0x1905);
                      _outpw(crtcport, 0x9706);
                      _outpw(crtcport, 0x1f07);
                      _outpw(crtcport, 0x4009);

                      _outpw(crtcport, 0x8010);
                      _outpw(crtcport, 0x8411);
                      _outpw(crtcport, 0x7f12);
                      _outpw(crtcport, 0x8015);
                      _outpw(crtcport, 0x0016);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa05b);
                      _outpw(crtcport, 0x005d);
                      _outpw(crtcport, 0x0067);

                      _outpw(crtcport, 0x0634);
                      _outpw(crtcport, 0x933b);
                      _outpw(crtcport, 0x4d3c);

                      _outpw(crtcport, 0x2242);

                      r0 = 0x56;
                      r1 = 0x45;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    default:           /*60*/
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xa400);
                      _outpw(crtcport, 0x8703);
                      _outpw(crtcport, 0x8504);
                      _outpw(crtcport, 0x9605);
                      _outpw(crtcport, 0x2006);
                      _outpw(crtcport, 0xf507);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0210);
                      _outpw(crtcport, 0x8811);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0xff15);
                      _outpw(crtcport, 0x2016);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x005d);
                      _outpw(crtcport, 0x0067);

                      _outpw(crtcport, 0x0234);
                      _outpw(crtcport, 0x9d3b);
                      _outpw(crtcport, 0x523c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x6b;
                      r1 = 0x44;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

                case 8:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 75:
                    case 72:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xa200);
                      _outpw(crtcport, 0x8303);
                      _outpw(crtcport, 0x8204);
                      _outpw(crtcport, 0x8e05);
                      _outpw(crtcport, 0x1f06);
                      _outpw(crtcport, 0xfd07);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0110);
                      _outpw(crtcport, 0x8411);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0x0015);
                      _outpw(crtcport, 0x0016);
                      _outpw(crtcport, 0xeb17);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0xf55b);
                      _outpw(crtcport, 0x005d);
                      _outpw(crtcport, 0x0167);

                      _outpw(crtcport, 0x1434);
                      _outpw(crtcport, 0x9b3b);
                      _outpw(crtcport, 0x513c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x41;
                      r1 = 0x0a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 70:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xa200);
                      _outpw(crtcport, 0x8503);
                      _outpw(crtcport, 0x8304);
                      _outpw(crtcport, 0x9405);
                      _outpw(crtcport, 0x1f06);
                      _outpw(crtcport, 0xf507);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0210);
                      _outpw(crtcport, 0x8811);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0xff15);
                      _outpw(crtcport, 0x1f16);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xac5b);
                      _outpw(crtcport, 0x005d);
                      _outpw(crtcport, 0x0167);

                      _outpw(crtcport, 0x1034);
                      _outpw(crtcport, 0x9b3b);
                      _outpw(crtcport, 0x513c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x28;
                      r1 = 0x22;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    case 43:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x9a00);
                      _outpw(crtcport, 0x9d03);
                      _outpw(crtcport, 0x8104);
                      _outpw(crtcport, 0x1705);
                      _outpw(crtcport, 0x9706);
                      _outpw(crtcport, 0x1f07);
                      _outpw(crtcport, 0x4009);

                      _outpw(crtcport, 0x8010);
                      _outpw(crtcport, 0x8411);
                      _outpw(crtcport, 0x7f12);
                      _outpw(crtcport, 0x8015);
                      _outpw(crtcport, 0x0016);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa05b);
                      _outpw(crtcport, 0x005d);
                      _outpw(crtcport, 0x0067);

                      _outpw(crtcport, 0x1434);
                      _outpw(crtcport, 0x933b);
                      _outpw(crtcport, 0x4d3c);

                      _outpw(crtcport, 0x2242);

                      r0 = 0x56;
                      r1 = 0x45;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                    default:           /*60*/
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xa400);
                      _outpw(crtcport, 0x8703);
                      _outpw(crtcport, 0x8304);
                      _outpw(crtcport, 0x9405);
                      _outpw(crtcport, 0x2006);
                      _outpw(crtcport, 0xf507);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0210);
                      _outpw(crtcport, 0x8811);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0xff15);
                      _outpw(crtcport, 0x2016);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x005d);
                      _outpw(crtcport, 0x0067);

                      _outpw(crtcport, 0x1034);
                      _outpw(crtcport, 0x9d3b);
                      _outpw(crtcport, 0x523c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x6b;
                      r1 = 0x44;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

                case 16:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 75:
                    case 72:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x4800);
                      _outpw(crtcport, 0x8803);
                      _outpw(crtcport, 0x0504);
                      _outpw(crtcport, 0x1d05);
                      _outpw(crtcport, 0x1f06);
                      _outpw(crtcport, 0xfd07);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0110);
                      _outpw(crtcport, 0x8411);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0x0015);
                      _outpw(crtcport, 0x0016);
                      _outpw(crtcport, 0xeb17);

                      _outpw(crtcport, 0xa052);
                      _outpw(crtcport, 0x2854);
                      _outpw(crtcport, 0xf55b);
                      _outpw(crtcport, 0x5d5d);
                      _outpw(crtcport, 0x5167);

                      _outpw(crtcport, 0x1434);
                      _outpw(crtcport, 0x413b);
                      _outpw(crtcport, 0xa43c);

                      _outpw(crtcport, 0x0242);
                      r0 = 0x41;
                      r1 = 0x0a;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;

                    case 70:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x4900);
                      _outpw(crtcport, 0x8903);
                      _outpw(crtcport, 0x0704);
                      _outpw(crtcport, 0x0905);
                      _outpw(crtcport, 0x1f06);
                      _outpw(crtcport, 0xf507);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0210);
                      _outpw(crtcport, 0x8811);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0xff15);
                      _outpw(crtcport, 0x1f16);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0x2854);
                      _outpw(crtcport, 0xac5b);
                      _outpw(crtcport, 0x7d5d);
                      _outpw(crtcport, 0x5167);

                      _outpw(crtcport, 0x1034);
                      _outpw(crtcport, 0x423b);
                      _outpw(crtcport, 0xa43c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x28;
                      r1 = 0x22;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;

                    case 43:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x3800);
                      _outpw(crtcport, 0x8003);
                      _outpw(crtcport, 0x0304);
                      _outpw(crtcport, 0x0f05);
                      _outpw(crtcport, 0x9706);
                      _outpw(crtcport, 0x1f07);
                      _outpw(crtcport, 0x4009);

                      _outpw(crtcport, 0x8010);
                      _outpw(crtcport, 0x8411);
                      _outpw(crtcport, 0x7f12);
                      _outpw(crtcport, 0x8015);
                      _outpw(crtcport, 0x0016);
                      _outpw(crtcport, 0xa317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0x7854);
                      _outpw(crtcport, 0xa05b);
                      _outpw(crtcport, 0x755d);
                      _outpw(crtcport, 0x5067);

                      _outpw(crtcport, 0x1434);
                      _outpw(crtcport, 0x313b);
                      _outpw(crtcport, 0x9c3c);

                      _outpw(crtcport, 0x2242);

                      r0 = 0x56;
                      r1 = 0x45;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;

                    default:           /*60*/
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0x4d00);
                      _outpw(crtcport, 0x8d03);
                      _outpw(crtcport, 0x0704);
                      _outpw(crtcport, 0x0905);
                      _outpw(crtcport, 0x2506);
                      _outpw(crtcport, 0xf507);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0210);
                      _outpw(crtcport, 0x8811);
                      _outpw(crtcport, 0xff12);
                      _outpw(crtcport, 0xff15);
                      _outpw(crtcport, 0x2516);
                      _outpw(crtcport, 0xe317);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0x4854);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x7d5d);
                      _outpw(crtcport, 0x5067);

                      _outpw(crtcport, 0x1034);
                      _outpw(crtcport, 0x463b);
                      _outpw(crtcport, 0xa63c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x6b;
                      r1 = 0x44;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x50;
                      break;           /*from vrtrefresh*/
                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

              }                        /*end bpp switch*/
              break;                   /*from xresolution*/

            case 1280:
              switch(pAdapter->ModeInfo.bBitsPerPixel)
              {
                case 8:
                  switch(pAdapter->ModeInfo.bVrtRefresh)
                  {
                    case 75:       /*just do 60*/
                    case 72:
                    case 60:
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xd900);
                      _outpw(crtcport, 0xa002);
                      _outpw(crtcport, 0x9c03);
                      _outpw(crtcport, 0xa504);
                      _outpw(crtcport, 0x1d05);
                      _outpw(crtcport, 0x1e06);
                      _outpw(crtcport, 0x5207);
                      _outpw(crtcport, 0x4009);

                      _outpw(crtcport, 0x0310);
                      _outpw(crtcport, 0x8611);
                      _outpw(crtcport, 0x0016);

                      _outpw(crtcport, 0x8052);
                      _outpw(crtcport, 0x5054);
                      _outpw(crtcport, 0xa85b);
                      _outpw(crtcport, 0x555e);
                      _outpw(crtcport, 0x1067);

                      _outpw(crtcport, 0x1434);
                      _outpw(crtcport, 0xd23b);
                      _outpw(crtcport, 0x6c3c);

                      _outpw(crtcport, 0x0242);

                      r0 = 0x7b;
                      r1 = 0x26;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x10;
                      break;           /*from vrtrefresh*/

                    default:                     /*45 int*/
                      _outpw(crtcport, 0x0e11);

                      _outpw(crtcport, 0xc000);
                      _outpw(crtcport, 0x9f02);
                      _outpw(crtcport, 0x8303);
                      _outpw(crtcport, 0xa404);
                      _outpw(crtcport, 0x1f05);
                      _outpw(crtcport, 0x1806);
                      _outpw(crtcport, 0xb207);
                      _outpw(crtcport, 0x6009);

                      _outpw(crtcport, 0x0110);
                      _outpw(crtcport, 0x8511);
                      _outpw(crtcport, 0x1816);

                      _outpw(crtcport, 0x4052);
                      _outpw(crtcport, 0xa854);
                      _outpw(crtcport, 0x165b);
                      _outpw(crtcport, 0x005e);
                      _outpw(crtcport, 0x0167);

                      _outpw(crtcport, 0x1034);
                      _outpw(crtcport, 0xb93b);
                      _outpw(crtcport, 0x603c);

                      _outpw(crtcport, 0x2242);

                      r0 = 0x28;
                      r1 = 0x22;
                      r2 = 0x55;
                      r3 = 0x29;
                      r4 = 0x00;
                      break;

                  }                    /*end vrtrefresh*/
                  break;               /*from bpp*/

              }                        /*end bpp switch*/
              break;                   /*from xresolution*/
          }                     /*end xres switches*/

        }                       /*end graphics register setup for s3sdac*/

        SetS3SDACClock(r0, r1, r2, r3, r4);
        return rc;
      }      /*end endif for s3 & s3sdac                          @V3.0ET001*/
   }                    /* end S3_ADAPTER test           */

   if (OEMHardware.Manufacturer == DIAMOND_MANUFACTURER)
   {
     /*
     ** Diamond Stealth W32 adapter
     */
     if ((SVGAHardware.AdapterType == TSENG_ADAPTER) &&
        (SVGAHardware.ChipType > TSENG_ET4000_CHIP))
     {
        switch(pAdapter->ModeInfo.usXResolution)
        {
          case 640:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                case 90:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x14];
                    break;
                  }     /* don't allow this refresh for >8 bpp */
                case 75:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x02];
                    break;
                  }     /* don't allow this refresh for >8 bpp */
                case 72:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x2];
                    break;
                  }
                default:
                case 60:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x0];
                  }
                  else
                  {
                    if (StealthOldScheme)
                      ClockSerialData = DIA_ClockTable[0x07];   //3*8bpp
                    else
                      ClockSerialData = DIA_ClockTable[0x17];   //1.5*8bpp
                  }                                                                 break;
             }
             break;
          case 800:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                case 90:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x9];
                    break;
                  }     /* don't allow this refresh for >8 bpp */
                case 75:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x0d];
                    break;
                  }     /* don't allow this refresh for >8 bpp */
                case 72:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x10];
                    break;
                  }
                case 60:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x04];
                    break;
                  }
                  else
                  {
                    /*
                    ** 800x600x64K is not supported unless 1072 w 2MB
                    */
                    if (!StealthOldScheme)
                      ClockSerialData = DIA_ClockTable[0x14];
                    else
                      rc = ERROR_REFRESH_NOT_SUPPORTED;
                  }
                  break;
                default:
                case 56:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x03];
                    break;
                  }
                  else          /* 244 bpp */
                  {
                    /*
                    ** 800x600x64K is not supported unless 1072 w 2MB
                    */
                    if (!StealthOldScheme)
                      ClockSerialData = DIA_ClockTable[0x16];
                    else
                      rc = ERROR_REFRESH_NOT_SUPPORTED;
                  }
                  break;
             }
             break;
          case 1024:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                case 75:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x0e];
                    break;
                  }     /* don't allow this refresh for >8 bpp */
                case 72:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x0b];
                    break;
                  }
                  else          /* 24 bpp */
                  {
                    /*
                    ** 1024x768x16M is not supported unless 1072 w 2MB
                    */
                    if (!StealthOldScheme)
                      ClockSerialData = DIA_ClockTable[0x15];
                    else
                      rc = ERROR_REFRESH_NOT_SUPPORTED;
                    break;
                  }
                case 70:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x07];
                    break;
                  }
                  else  /* 24 bpp */
                  {
                    /*
                    ** 1024x768x16M is not supported unless 1072 w 2MB
                    */
                    if (!StealthOldScheme)
                      ClockSerialData = DIA_ClockTable[0x15];
                    else
                      rc = ERROR_REFRESH_NOT_SUPPORTED;
                    break;
                  }
                case 60:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x08];
                    break;
                  }
                  else  /* 24 bpp */
                  {
                    /*
                    ** 1024x768x16M is not supported unless 1072 w 2MB
                    */
                    if (!StealthOldScheme)
                      ClockSerialData = DIA_ClockTable[0x15];
                    else
                      rc = ERROR_REFRESH_NOT_SUPPORTED;
                    break;
                  }
                default:
                case 43:
                  if (pAdapter->ModeInfo.bBitsPerPixel == 8 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 16 ||
                      pAdapter->ModeInfo.bBitsPerPixel == 4)
                  {
                    ClockSerialData = DIA_ClockTable[0x05];
                    break;
                  }
                  else  /* 24 bpp */
                  {
                    /*
                    ** 1024x768x16M is not supported unless 1072 w 2MB
                    */
                    if (!StealthOldScheme)
                      ClockSerialData = DIA_ClockTable[0x15];
                    else
                      rc = ERROR_REFRESH_NOT_SUPPORTED;
                    break;
                  }
             }
             break;
          case 1280:
             if (pAdapter->ModeInfo.bBitsPerPixel == 8)
                ClockSerialData = DIA_ClockTable[0x12];  /* 46i supported only */
             else
                rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
          default:
           if (!(pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS) &&     /*            */
                (pAdapter->ModeInfo.usBytesPerScanLine == 132))
                ClockSerialData = DIA_ClockTable[0x4];  
           else
             rc = ERROR_REFRESH_NOT_SUPPORTED;
        }
        if (!rc)
           SetDiaIDCClk(ClockSerialData);
        return (rc);                    /*            end */
     }
     /*
     ** Diamond Speedstar 24 BIOS revs 5, 6 up
     */
     BiosMajorRev = (USHORT) (OEMHardware.ManufacturerData & 0x000F);   /*            */
     if (SVGAHardware.AdapterType == TSENG_ADAPTER)
     {
       if (!(pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS))
       {
         if (pAdapter->ModeInfo.usBytesPerScanLine == 132)
             ClockSerialData = 0x8a040e;
         else if (BiosMajorRev < 6)
             ClockSerialData = 0x923506;
         else ClockSerialData = 0xa23506;
       }
       else
       {
         switch(pAdapter->ModeInfo.usXResolution)
         {
             case 640:
               switch(pAdapter->ModeInfo.bVrtRefresh)
               {
                  default:
                  case 60:
                     if (BiosMajorRev < 6)
                       ClockSerialData = 0xa23d15;
                     else ClockSerialData = 0x823d15;
                        break;
                  case 72:
                     if (BiosMajorRev < 6)
                        ClockSerialData = 0xb2c422;
                     else ClockSerialData = 0xd2c422;
                        break;
                }
                break;
             case 800:
                switch(pAdapter->ModeInfo.bVrtRefresh)
                {
                  default:
                  case 56:
                     if (BiosMajorRev < 6)
                          ClockSerialData = 0xf2842a;
                     else ClockSerialData = 0xb2842a;
                     break;
                  case 60:
                     ClockSerialData = 0x8a040e;
                     break;
                  case 72:
                     if (BiosMajorRev < 6)
                          ClockSerialData = 0xa2bc35;
                     else ClockSerialData = 0x82dc35;
                     break;
                }
                break;
             case 1024:
                switch(pAdapter->ModeInfo.bVrtRefresh)
                {
                   default:
                   case 44:             //interlaced
                      if (BiosMajorRev < 6)
                           ClockSerialData = 0xc25410;
                      else ClockSerialData = 0xcc2910;
                      break;
                   case 60:
                      if (BiosMajorRev < 6)
                           ClockSerialData = 0xb2c422;
                      else ClockSerialData = 0xd2c422;
                      break;
                   case 70:
                       ClockSerialData =0x0f21435;
                       break;
                   case 72:
                       if (BiosMajorRev < 6)
                           ClockSerialData = 0xf23427;
                       else ClockSerialData = 0x8a441e;
                       break;
                }
                break;
             default:
                rc = ERROR_REFRESH_NOT_SUPPORTED;
           }
       }
       if (!rc)
          SETDIAMONDCLK(ClockSerialData);
       return rc;
     }
     /*
     ** Diamond Speedstar24X, BIOS revision 1, 2 up
     */
     else if (SVGAHardware.AdapterType == WESTERNDIG_ADAPTER)
     {
        if (!(pAdapter->ModeInfo.usType & MODE_FLAG_GRAPHICS))
        {
           if (pAdapter->ModeInfo.usBytesPerScanLine == 132)
             if (BiosMajorRev < 2)
               ClockSerialData = 0xc25410;
             else ClockSerialData = 0xaa2910;
           else if (BiosMajorRev < 2)
               ClockSerialData = 0x923506;
           else ClockSerialData = 0x86ac1e;
        }
        else
        {
          switch(pAdapter->ModeInfo.usXResolution)
          {
             case 640:
                switch(pAdapter->ModeInfo.bVrtRefresh)
                {
                    default:
                    case 60:
                        if (BiosMajorRev < 2)
                           ClockSerialData = 0xa23d15;
                        else ClockSerialData = 0x86ac1e;
                        break;
                    case 72:
                        if (BiosMajorRev < 2)
                           ClockSerialData = 0xd2ad33;
                        else ClockSerialData = 0x92ad33;
                        break;
                }
                break;
             case 800:
                switch(pAdapter->ModeInfo.bVrtRefresh)
                {
                    default:
                    case 56:
                        if (BiosMajorRev < 2)
                           ClockSerialData = 0xf2852a;
                        else ClockSerialData = 0xb2852a;
                        break;
                    case 60:
                        if (BiosMajorRev < 2)
                           ClockSerialData = 0xa2bc2a;
                        else ClockSerialData = 0xee153d;
                        break;
                    case 72:
                        if (BiosMajorRev < 2)
                           ClockSerialData = 0xa2bc35;
                        else ClockSerialData = 0x82dc35;
                        break;
                }
                break;
             case 1024:
                switch(pAdapter->ModeInfo.bVrtRefresh)
                {
                    default:
                    case 44:             //interlaced
                        if (BiosMajorRev < 2)
                           ClockSerialData = 0xc25410;
                        else ClockSerialData = 0xaa2910;
                        break;
                    case 60:
                        if (BiosMajorRev < 2)
                           ClockSerialData =0xb2c422;
                        else ClockSerialData = 0xd2c422;
                        break;
                    case 70:
                        ClockSerialData =0x0f21435;
                        break;
                    case 72:
                        if (BiosMajorRev < 2)
                           ClockSerialData = 0xf29423;
                        else ClockSerialData = 0x8a9423;
                        break;
                }
                break;
             default:
                rc = ERROR_REFRESH_NOT_SUPPORTED;
          }
        }
        if (!rc)
        {
             SETDIAMONDCLK(ClockSerialData);
             //2nd pass for 256 color graphics modes on Bios 2
             if ((BiosMajorRev == 2) && (pAdapter->ModeInfo.bBitsPerPixel == 8))
                SETDIAMONDCLK(0x0e6f409);
        }
        return rc;
     }
     /*
     ** Diamond Viper VLB. Viper function will also set registers to
     ** contain the r31 = hrzt, r32 = hrzs, r32 = vrtt, r33 = vrtsr.
     ** It does not program them, since we don't want to muck with
     ** getting the virtual address for it. I leave that to the calling
     ** PMI file. The PMI file also provides active display tuning function
     ** and takes care of clock doubling thru the DAC if needed.
     */
     else if (SVGAHardware.AdapterType == WEITEK_ADAPTER)
     {
        // set mem speed
        SetDiaIDCClk(0x00679c35);
        switch(pAdapter->ModeInfo.usXResolution)
        {
          case 640:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                default:
                case 60:
                  ClockSerialData =0x045a8bc;
                  pRegs[31] = 0xC7;             //set hrzt
                  pRegs[32] = 0x17;             //set hrzsr
                  pRegs[33] = 0x20d;            //set vrtt
                  pRegs[34] = 0x01;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x21;             //set hrzbr:Screen left edge
                  pRegs[36] = 0xc1;             //set hrzbf:Screen right edge
                  pRegs[37] = 0x19;            //set vrtbr:Screen top edge
                  pRegs[38] = 0x1f9;             //set vrtbf: Screen bottom edge
                  break;
                case 72:
                  ClockSerialData =0x04bd8b5;
                  pRegs[31] = 0xCF;             //set hrzt
                  pRegs[32] = 0x05;             //set hrzsr
                  pRegs[33] = 0x208;            //set vrtt
                  pRegs[34] = 0x03;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x25;             //set hrzbr:Screen left edge
                  pRegs[36] = 0xc5;             //set hrzbf:Screen right edge
                  pRegs[37] = 0x1f;            //set vrtbr:Screen top edge
                  pRegs[38] = 0x1ff;             //set vrtbf: Screen bottom edge
                  break;
             }
             break;
          case 800:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                default:
                case 56:
                  ClockSerialData =0x04f54a1;
                  pRegs[31] = 0xFD;             //set hrzt
                  pRegs[32] = 0x1D;             //set hrzsr
                  pRegs[33] = 0x278;            //set vrtt
                  pRegs[34] = 0x04;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x33;             //set hrzbr:Screen left edge
                  pRegs[36] = 0xfb;             //set hrzbf:Screen right edge
                  pRegs[37] = 0x1b;            //set vrtbr:Screen top edge
                  pRegs[38] = 0x273;             //set vrtbf: Screen bottom edge
                  break;
                case 72:
                  ClockSerialData =0x045ac3d;
                  pRegs[31] = 0x103;            //set hrzt
                  pRegs[32] = 0x1B;             //set hrzsr
                  pRegs[33] = 0x29A;            //set vrtt
                  pRegs[34] = 0x06;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x2b;             //set hrzbr:Screen left edge
                  pRegs[36] = 0xf3;             //set hrzbf:Screen right edge
                  pRegs[37] = 0x1d;             //set vrtbr:Screen top edge
                  pRegs[38] = 0x275;            //set vrtbf: Screen bottom edge
                  break;
             }
             break;
          case 1024:
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                default:
                case 60:
                  ClockSerialData =0x04d4423;
                  pRegs[31] = 0x14F;            //set hrzt
                  pRegs[32] = 0x1D;             //set hrzsr
                  pRegs[33] = 0x326;            //set vrtt
                  pRegs[34] = 0x06;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x45;             //set hrzbr:Screen left edge
                  pRegs[36] = 0x145;             //set hrzbf:Screen right edge
                  pRegs[37] = 0x23;             //set vrtbr:Screen top edge
                  pRegs[38] = 0x323;             //set vrtbf: Screen bottom edge
                  break;
                case 70:
                  ClockSerialData =0x04FAC28;
                  pRegs[31] = 0x14B;            //set hrzt
                  pRegs[32] = 0x21;             //set hrzsr
                  pRegs[33] = 0x326;            //set vrtt
                  pRegs[34] = 0x06;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x45;             //set hrzbr:Screen left edge
                  pRegs[36] = 0x145;             //set hrzbf:Screen right edge
                  pRegs[37] = 0x23;             //set vrtbr:Screen top edge
                  pRegs[38] = 0x323;             //set vrtbf: Screen bottom edge
                  break;
                case 72:
                  ClockSerialData =0x04F7821;
                  pRegs[31] = 0x14B;            //set hrzt
                  pRegs[32] = 0x21;             //set hrzsr
                  pRegs[33] = 0x326;            //set vrtt
                  pRegs[34] = 0x06;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x41;             //set hrzbr:Screen left edge
                  pRegs[36] = 0x141;             //set hrzbf:Screen right edge
                  pRegs[37] = 0x21;             //set vrtbr:Screen top edge
                  pRegs[38] = 0x321;             //set vrtbf: Screen bottom edge
                  break;
                case 76:
                  ClockSerialData =0x04F8021;
                  pRegs[31] = 0x14A;            //set hrzt
                  pRegs[32] = 0x21;             //set hrzsr
                  pRegs[33] = 0x327;            //set vrtt
                  pRegs[34] = 0x06;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x3f;             //set hrzbr:Screen left edge
                  pRegs[36] = 0x13f;            //set hrzbf:Screen right edge
                  pRegs[37] = 0x1f;             //set vrtbr:Screen top edge
                  pRegs[38] = 0x31f;            //set vrtbf: Screen bottom edge
                  break;
             }
             break;
          case 1280:
             if (pAdapter->ModeInfo.bVrtRefresh >= 68)
                pRegs[255] = 1;              /* clock doubling needed */
             switch(pAdapter->ModeInfo.bVrtRefresh)
             {
                default:
                case 60:
                  ClockSerialData =0x05BF81E;
                  pRegs[31] = 0x1BF;            //set hrzt
                  pRegs[32] = 0x37;             //set hrzsr
                  pRegs[33] = 0x42E;            //set vrtt
                  pRegs[34] = 0x03;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x77;             //set hrzbr:Screen left edge
                  pRegs[36] = 0x1b7;            //set hrzbf:Screen right edge
                  pRegs[37] = 0x2d;             //set vrtbr:Screen top edge
                  pRegs[38] = 0x42d;            //set vrtbf: Screen bottom edge
                  break;
                case 74:
                  ClockSerialData =0x05B8013;
                  pRegs[31] = 0x1AB;            //set hrzt
                  pRegs[32] = 0x23;             //set hrzsr
                  pRegs[33] = 0x42A;            //set vrtt
                  pRegs[34] = 0x03;             //set vrtsr
               /* blanking signals: should evenutally be delegated to pfnPMITuneDisplay */
                  pRegs[35] = 0x63;             //set hrzbr:Screen left edge
                  pRegs[36] = 0x1a3;            //set hrzbf:Screen right edge
                  pRegs[37] = 0x29;             //set vrtbr:Screen top edge
                  pRegs[38] = 0x429;            //set vrtbf: Screen bottom edge
                  break;
             }
             break;
          default:
             ClockSerialData =0x045a8bc;
        }
        SetDiaIDCClk(ClockSerialData);
        /*
        ** Set Polarity. If it wasn't set, default to negative
        */
        if (pAdapter->ModeInfo.bVrtPolPos > 1)
           pAdapter->ModeInfo.bVrtPolPos = 0;
        if (pAdapter->ModeInfo.bHrtPolPos > 1)
           pAdapter->ModeInfo.bHrtPolPos = 0;

        _outp(0x3c4,0x12);
        r0 = _inp(0x3c5);
        r0 &= 0x3f;
        r0 |= ((BYTE) (pAdapter->ModeInfo.bHrtPolPos << 6) |
              (BYTE) (pAdapter->ModeInfo.bVrtPolPos << 7));
        _outp(0x3c5,r0);
        /*
        ** Set horizontal count for high/true color. The above values
        ** were in pixel count, so convert into dot count.
        */
        if (pAdapter->ModeInfo.bBitsPerPixel >= 16)
        {
           pRegs[31] <<= 1;
           pRegs[31]++;
           pRegs[32] <<= 1;
           pRegs[32]++;
           pRegs[35] <<= 1;
           pRegs[35]++;
           pRegs[36] <<= 1;
           pRegs[36]++;
        }
        if (pAdapter->ModeInfo.bBitsPerPixel == 24)
        {
           pRegs[31] <<= 1;
           pRegs[31]++;
           pRegs[32] <<= 1;
           pRegs[32]++;
           pRegs[35] <<= 1;
           pRegs[35]++;
           pRegs[36] <<= 1;
           pRegs[36]++;
        }
        return rc;
     }
     /*
     ** ATI Mach 32 is handled here.
     */
     if (SVGAHardware.AdapterType == ATI_ADAPTER)
     {
        pfnSetATI32(pAdapter,pRegs);
        return rc;
     }
   }
   return(ERROR_ADAPTER_NOT_SUPPORTED);

}
/*****************************************************************************
 *
 *  FUNCTION NAME:    OpenScreenDD()
 *
 *
 *  DESCRIPTIVE NAME: Open SCREEN$ dd.
 *
 *  FUNCTION:
 *
 *  ENTRY POINT:
 *
 *  INPUT: (Passed on stack)
 *
 *  EXIT-NORMAL: 0
 *
 *  EXIT-ERROR: return code
 *
 *  EFFECTS:
 *
 *  NOTES:
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
APIRET _System OpenScreenDD(VOID)
{
  APIRET rc = FALSE;
  ULONG  ulOpenAction;

  rc = DosOpen(SCREENDD_NAME,
               &hScreenDD,
               &ulOpenAction,
               0L,
               FILE_NORMAL,
               OPEN_ACTION_OPEN_IF_EXISTS,
               OPEN_ACCESS_READWRITE |
               OPEN_FLAGS_NOINHERIT |
               OPEN_SHARE_DENYNONE,
               NULL);
  return rc;
}
/*****************************************************************************
 *
 *  FUNCTION NAME:    VIDEOIOCTL()
 *
 *
 *  DESCRIPTIVE NAME: Open SCREEN$ dd.
 *
 *  FUNCTION:
 *
 *  ENTRY POINT:
 *
 *  INPUT: (Passed on stack)
 *
 *  EXIT-NORMAL: 0
 *
 *  EXIT-ERROR: return code
 *
 *  EFFECTS:
 *
 *  NOTES:
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
APIRET _System videoIoctl(ULONG ulFunction,
                         VOID *pParam,
                         ULONG ulParamLen)
{
  APIRET rc = FALSE;
  ULONG ulParmlenMax = ulParamLen;
  if (!hScreenDD)
    rc = OpenScreenDD();
  if (!rc)
  {             //screendd_svga_id & oem uses data packet, others use parameter packet
    if (ulFunction <= SCREENDD_SVGA_OEM)
      rc = DosDevIOCtl (hScreenDD,
                           SCREENDD_CATEGORY,
                           ulFunction,
                           NULL, 0L, NULL,
                           pParam,
                           ulParamLen,
                           &ulParmlenMax);
    else
      rc = DosDevIOCtl (hScreenDD,
                           SCREENDD_CATEGORY,
                           ulFunction,
                           pParam,
                           ulParamLen,
                           &ulParmlenMax,
                           NULL, 0L, NULL);
  }

  return rc;
}
/*****************************************************************************
 *
 *  FUNCTION NAME:      Identify()
 *
 *  DESCRIPTIVE NAME: Identify the chip and manufacturer.
 *
 *  FUNCTION:
 *
 *  OUTPUT:            SVGAINFO  AdapterTYpe, ChipType, TotalMemory
 *                    OEMINFO  Manufacturer name and data
 *
 *  EXIT:             APIRET - return code 0 if identified, ERROR_ADAPTER_NOT_SUPPORTED otherwise.
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 * Some PCI ID's:
 *      S3 Vision864 8880h
 *      S3 Vision866 8890h
 *      S3 Vision964 88B0h
 ****************************************************************************/
APIRET _System Identify(OEMSVGAINFO *SVGAHardware,OEMINFO *OEMHardware)
{
  ULONG rc = ERROR_ADAPTER_NOT_SUPPORTED;
  /*
  **  Identify the new chips: S3 864, PCI and non-PCI, generic support
  **  ATI 88800 - generic support
  */
  if (rc || SVGAHardware->AdapterType == DEFAULT_ADAPTER)
  {
    /*
     * get adapter type from screenDD
     */
    videoIoctl(SCREENDD_SVGA_ID, (VOID *)SVGAHardware, sizeof(OEMSVGAINFO));

    if (SVGAHardware->AdapterType)      /* if we have SVGA installed         */
      rc = videoIoctl(SCREENDD_SVGA_OEM,(VOID *)OEMHardware,sizeof(OEMINFO));
  }
  /*
  ** Perform any adapter initialization
  */

  if (SVGAHardware->AdapterType)
    InitializeAdapterData(SVGAHardware,OEMHardware);
  return rc;
}
/*****************************************************************************
 *
 *  FUNCTION NAME:      InitializeAdapterData()
 *
 *  DESCRIPTIVE NAME:  Initialize all global data for this adapter
 *
 *  FUNCTION:
 *
 *  INPUT:            SVGAINFO  AdapterTYpe, ChipType, TotalMemory
 *                    OEMINFO  Manufacturer name and data
 *
 *  EXIT:             APIRET - return code 0 if identified, ERROR_ADAPTER_NOT_SUPPORTED otherwise.
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
APIRET _System InitializeAdapterData(OEMSVGAINFO *SVGAHardware,OEMINFO *OEMHardware)
{
  /*
  **  Identify the new chips: S3 864, PCI and non-PCI, generic support
  **  ATI 88800 - generic support
  */
  return NO_ERROR;
}
/*****************************************************************************
 *
 *  FUNCTION NAME:      SetDiaIDCClk()
 *
 *  DESCRIPTIVE NAME:  Yet another version of the IDC2061.
 *
 *  FUNCTION:         This is a rough translation of the PMI function.
 *                    SETDIAMONDCLK function would hose the bus, so I've
 *                    included the PMI version of the IDC2061 for now.
 *
 *  INPUT:            ULONG serial clock
 *
 *  EXIT:             NONE
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
VOID _System SetDiaIDCClk(ULONG ClockData)
{
  ULONG r1,r2,r3,r4,r5,r6,r16;
  r1 = _inp(0x3cc);
  r1   &= 0x000000f3;
  r2   = r1   |  0x00000008;
  _outp(0x3c2, r2);
  r3   = r1   |  0x0000000C;
  for(r16=0;r16 < 0x05;r16++)
  {
    _outp(0x3c2, r2);
    _outp(0x3c2, r3);
  }
  r3   = r1   |  0x00000004;
  _outp(0x3c2, r1);
  _outp(0x3c2, r3);
  _outp(0x3c2, r1);
  _outp(0x3c2, r3);
  for(r16= 0;r16 < 0x18;r16++)
  {
    r4   = ClockData   << 0x00000003;
    r5   =~r4   &  0x00000008;
    r6   = r5   |  0x00000004;
    r5   |= r1  ;
    r6   |= r1  ;
    _outp(0x3c2, r6);
    _outp(0x3c2, r5);
    r5   = r4   &  0x00000008;
    r6   = r5   |  0x00000004;
    r5   |= r1  ;
    r6   |= r1  ;
    _outp(0x3c2, r5);
    _outp(0x3c2, r6);
    ClockData   >>= 0x00000001;
  }
  r2   = r1   |  0x00000008;
  r3   = r1   |  0x0000000C;
  _outp(0x3c2, r3);
  _outp(0x3c2, r2);
  _outp(0x3c2, r3);
  _outp(0x3c2, r3);
}

/*****************************************************************************
 *
 *  FUNCTION NAME:      SetSTBClk()
 *
 *  DESCRIPTIVE NAME:  Yet another version of the IDC2061.
 *
 *  FUNCTION:         Program IDC for Cirrus and S3.
 *
 *  INPUT:            ULONG serial clock
 *
 *  EXIT:             NONE
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
#define WrtClkBit(clk_bit_port,value)  \
        _outp(clk_bit_port,value);
VOID _System SetSTBClock(ULONG ClockData)
{
   UCHAR nclk[2], clk[2];
   USHORT restore42;
   USHORT oldclk;
   USHORT bitval;
   USHORT clk_bit_port;
   INT i;
   USHORT crtcport = (_inp(0x3cc) & 0x01) ? 0x3D4 : 0x3B4;

   if (SVGAHardware.AdapterType == CIRRUS_ADAPTER)
   {
      _outp(0x3C4, 0x08);
      oldclk = _inp(0x3C5);
      clk_bit_port = 0x3C5;
   }
   else
   {
      oldclk = _inp(0x3CC);
      clk_bit_port = 0x3C2;
      _outp(crtcport, 0x58);            //          
      i = (_inp(crtcport+1) & 0xfe) | 0x01;
      _outp(crtcport+1, i);

      _outp(crtcport, 0x42);
      restore42 = (_inp(crtcport+1) << 8) & 0x3F00;
      restore42 |= 0x42;

      _outpw(0x3C4, 0x0100);

      _outp(0x3C4, 1);
      _outp(0x3C5, 0x20 | _inp(0x3C5));

      _outp(crtcport, 0x42);
//                _outp(crtcport+1, 0x03);
      _outp(crtcport+1, 0x02);

      _outpw(0x3C4, 0x0300);

      _outp(crtcport, 0x42);
      i = _inpw(crtcport);

      _outpw(0x3C4, 0x0100);
   }

   nclk[0] = oldclk & 0xF3;
   nclk[1] = nclk[0] | 0x08;
   clk[0] = nclk[0] | 0x04;
   clk[1] = nclk[0] | 0x0C;

   WrtClkBit(clk_bit_port,oldclk | 0x08);
   WrtClkBit(clk_bit_port,oldclk | 0x0C);
   for (i=0; i<5; i++)
   {
      WrtClkBit(clk_bit_port,nclk[1]);
      WrtClkBit(clk_bit_port,clk[1]);
   }
   WrtClkBit(clk_bit_port,nclk[1]);
   WrtClkBit(clk_bit_port,nclk[0]);
   WrtClkBit(clk_bit_port,clk[0]);
   WrtClkBit(clk_bit_port,nclk[0]);
   WrtClkBit(clk_bit_port,clk[0]);
   for (i=0; i<24; i++)
   {
      bitval = ClockData & 0x01;
      ClockData >>= 1;
      WrtClkBit(clk_bit_port,clk[1-bitval]);
      WrtClkBit(clk_bit_port,nclk[1-bitval]);
      WrtClkBit(clk_bit_port,nclk[bitval]);
      WrtClkBit(clk_bit_port,clk[bitval]);
   }
   WrtClkBit(clk_bit_port,clk[1]);
   WrtClkBit(clk_bit_port,nclk[1]);
   WrtClkBit(clk_bit_port,clk[1]);

   if (SVGAHardware.AdapterType == CIRRUS_ADAPTER)
   {
      _outp(clk_bit_port, oldclk);
   }
   else
   {
      _outp(0x3C4, 1);
      _outp(0x3C5, 0xDF & _inp(0x3C5));

      _outpw(crtcport, restore42);

      _outp(clk_bit_port, oldclk);

      _outpw(0x3C4, 0x0300);
   }
}
/*****************************************************************************
 *
 *  FUNCTION NAME:    SetS3Clk()
 *
 *  DESCRIPTIVE NAME: Program on-chip clock for S3.
 *
 *  FUNCTION:         All "generic" S3 cards run this code. Note that due
 *                    to the fact that this is a generic support, chances are
 *                    that startup configuration is not known, therefore the
 *                    default values will be treated as invalid. It is better to
 *                    leave it to the PMI file to set the clock, until the
 *                    customer has selected a monitor from the list and we
 *                    are given values we expect.           
 *
 *  INPUT:            ULONG serial clock
 *
 *  EXIT:             NONE
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
//#pragma optimize("',off)
VOID _System  SetS3Clock(ULONG usXResolution, ULONG bBitsPerPixel, ULONG bVrtRefresh)
{
   BYTE r1,r2;
   USHORT crtcport = (_inp(0x3cc) & 0x01) ? 0x3D4 : 0x3B4;

   r1 = _inp(0x3cc);                 //read misc output reg
   switch(usXResolution)
   {
     case 640:
        switch(bVrtRefresh)
        {
           case 72:
             if (bBitsPerPixel <= 8)            //          
             {
                _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cr42
                _outp(crtcport,0x42);
                r2 = _inp(crtcport+1);
                r2 &= 0xD0;
                _outp(crtcport+1,r2 | 0x0B);
                break;
             }           //for 16 and 24 bpp let it fall to 60Hz.
           case 60:
             _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cr42
             _outp(crtcport,0x42);
             r2 = _inp(crtcport+1);
             r2 &= 0xD0;
             if (bBitsPerPixel == 16)
               _outp(crtcport+1,r2 | 0x04);
             else if (bBitsPerPixel == 24)
               _outp(crtcport+1,r2 | 0x0e);
             else
               _outp(crtcport+1,r2);
             break;
        }
        break;
     case 800:
        switch(bVrtRefresh)
        {
           case 72:
             if (bBitsPerPixel <= 8)            //          
             {
                _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cr42
                _outp(crtcport,0x42);
                r2 = _inp(crtcport+1);
                r2 &= 0xD0;
                _outp(crtcport+1,r2 | 0x04);
                break;
             }                                  //           60 only for high color
           case 56:
             if (bBitsPerPixel <= 8)            //          
             {
                _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cr42
                _outp(crtcport,0x42);
                r2 = _inp(crtcport+1);
                r2 &= 0xD0;
                _outp(crtcport+1,r2 | 0x06);
                break;
             }
           case 60:
             _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cr42
             _outp(crtcport,0x42);
             r2 = _inp(crtcport+1);
             r2 &= 0xD0;
             if (bBitsPerPixel == 16)
              _outp(crtcport+1,r2 | 0x0a);              //          
             else
              _outp(crtcport+1,r2 | 0x02);
             break;
        }
        break;
     case 1024:
       {
        USHORT clock;
        _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cr42
        _outp(crtcport,0x42);
        r2 = _inp(crtcport+1);
        clock = r2 & 0xD0;

        switch(bVrtRefresh)
        {
           case 70:
             if (bBitsPerPixel <= 8)
             {
                clock |= 0xe;
                break;
             }          //let it fall thru to 43Hz for 16bpp
           case 72:
             if (bBitsPerPixel <= 8)            //          
             {
                clock |= 0x05;
                break;
             }
           case 60:
             if (bBitsPerPixel <= 8)            //          
             {
                clock |= 0x0d;
                break;
             }
           case 43:
             if (bBitsPerPixel == 8)
                clock |= 0x27;    //set the clock.
             else
               clock |= 0x2f;     //empirical value: Actix
             break;
           default:     //make no guesses, return.
             return;
        }
        /*
        ** If changing from a non-interlaced into the interlaced and vice versa
        ** we have to adjust the vertical crts and the clock.
        */
        if ((bBitsPerPixel == 8) && (r2 & 0x20) && (bVrtRefresh > 44))
        {       //from interlaced into a non-interlaced
           ULONG vtotal, vrstart, vrend, vend, vbstart, vbend;
           ULONG ovflw, new_ovflw, ovflw_1, new_ovflw_1, rlock;
           /*
           ** Divide the clock
           ** Change all vertical crts (*2)
           */
           _outp(crtcport,0x11);   //lock register
           rlock = _inp(crtcport+1) & 0x7f;
           _outp(crtcport+1,rlock); //unlock crtc0-7

           _outp(crtcport,0x7);    //overflow register 1
           ovflw = _inp(crtcport+1);  //overflow bits
           new_ovflw = 0;         //overflow value to be set

           _outp(crtcport,0x9);    //overflow register 2
           ovflw_1 = _inp(crtcport+1);  //overflow bits
           new_ovflw_1 = ovflw_1 & 0xdf;

           _outp(crtcport,0x6);         //vertical total
           vtotal = _inp(crtcport+1);
           vtotal += ((ovflw & 0x1) << 8) + (((ovflw & 0x20) >> 5) << 9);
           vtotal <<= 1;   //multiply by 2
           _outp(crtcport+1,vtotal & 0xff);
           new_ovflw |= ((vtotal & 0x100) >> 8) | (((vtotal & 0x200) >> 9) << 5);

           _outp(crtcport,0x10);        //vertical retrace start
           vrstart = _inp(crtcport+1);
           vrstart += (((ovflw & 0x4) >> 2) << 8) + (((ovflw & 0x80) >> 7) << 9);
           vrstart <<= 1;   //multiply by 2
           _outp(crtcport+1,vrstart & 0xff);
           new_ovflw |= (((vrstart & 0x100) >> 8) << 2) | (((vrstart & 0x200) >> 9) << 7);

           _outp(crtcport,0x12);
           vend = _inp(crtcport+1);        //vertical display end
           vend += (((ovflw & 0x2) >> 1) << 8) + (((ovflw & 0x40) >> 6) << 9);
           vend <<= 1;   //multiply by 2
           _outp(crtcport+1,vend & 0xff);
           new_ovflw |= (((vend & 0x100) >> 8) << 1) | (((vend & 0x200) >> 9) << 6);

           _outp(crtcport,0x15);           //vertical blank start
           vbstart = _inp(crtcport+1);
           vbstart += (((ovflw & 0x8) >> 3) << 8) + (((ovflw_1 & 0x20) >> 5) << 9);
           vbstart <<= 1;   //multiply by 2
           _outp(crtcport+1,vbstart & 0xff);

           _outp(crtcport,0x16);           //vertical blank end
           vbend = _inp(crtcport+1);
           vbend += (((ovflw & 0x8) >> 3) << 8) + (((ovflw_1 & 0x20) >> 5) << 9);
           vbend <<= 1;   //multiply by 2
           _outp(crtcport+1,vbend & 0xff);

           new_ovflw |= ((vbstart & 0x100) >> 8) << 3;
           new_ovflw_1 |= ((vbstart & 0x200) >> 9) << 5;
           _outp(crtcport,0x9);            //set the new overflow
           _outp(crtcport+1,new_ovflw_1);

           _outp(crtcport,0x7);            //set the new overflow
           _outp(crtcport+1,new_ovflw);

           _outp(crtcport,0x11);           //vertical retrace end
           rlock = _inp(crtcport+1);
           vrend = rlock & 0xf;
           vrend <<= 1;   //multiply by 2
           rlock |= (vrend & 0xf) | 0x80;       //lock crt0-7
           _outp(crtcport+1,rlock & 0xff);

           _outp(crtcport,0x14);
           ovflw_1 = (_inp(crtcport+1) & 0x9f) | 0x60;
           _outp(crtcport+1,ovflw_1);           //turn the double word on

           _outp(crtcport,0x17);
           ovflw_1 = (_inp(crtcport+1) & 0xf7) | 0x8;
           _outp(crtcport+1,ovflw_1);           //turn the count by 2 off

           _outp(crtcport,0x42);
           //the clock will be set accordingly.
        }
        else if((bBitsPerPixel == 8) && !(r2 & 0x20) && (bVrtRefresh <= 44))
        {               //from non-interlaced into interlaced
           ULONG vtotal, vrstart, vrend, vend, vbstart, vbend;
           ULONG ovflw, new_ovflw, ovflw_1, new_ovflw_1, rlock;
           /*
           ** Make sure clock is not divided
           ** Change all vertical crts (/2)
           */
           _outp(crtcport,0x11);   //lock register
           rlock = _inp(crtcport+1) & 0x7f;
           _outp(crtcport+1,rlock); //unlock crtc0-7

           _outp(crtcport,0x7);    //overflow register 1
           ovflw = _inp(crtcport+1);  //overflow bits
           new_ovflw = 0;         //overflow value to be set

           _outp(crtcport,0x9);    //overflow register 2
           ovflw_1 = _inp(crtcport+1);  //overflow bits
           new_ovflw_1 = ovflw_1 & 0xdf;

           _outp(crtcport,0x6);         //vertical total
           vtotal = _inp(crtcport+1);
           vtotal += ((ovflw & 0x1) << 8) + (((ovflw & 0x20) >> 5) << 9);
           vtotal >>= 1;   //divide by 2
           _outp(crtcport+1,vtotal & 0xff);
           new_ovflw |= ((vtotal & 0x100) >> 8) | (((vtotal & 0x200) >> 9) << 5);

           _outp(crtcport,0x10);        //vertical retrace start
           vrstart = _inp(crtcport+1);
           vrstart += (((ovflw & 0x4) >> 2) << 8) + (((ovflw & 0x80) >> 7) << 9);
           vrstart >>= 1;   //divide by 2
           _outp(crtcport+1,vrstart & 0xff);
           new_ovflw |= (((vrstart & 0x100) >> 8) << 2) | (((vrstart & 0x200) >> 9) << 7);

           _outp(crtcport,0x12);
           vend = _inp(crtcport+1);        //vertical display end
           vend += (((ovflw & 0x2) >> 1) << 8) + (((ovflw & 0x40) >> 6) << 9);
           vend >>= 1;   //divide by 2
           _outp(crtcport+1,vend & 0xff);
           new_ovflw |= (((vend & 0x100) >> 8) << 1) | (((vend & 0x200) >> 9) << 6);

           _outp(crtcport,0x15);           //vertical blank start
           vbstart = _inp(crtcport+1);
           vbstart += (((ovflw & 0x8) >> 3) << 8) + (((ovflw_1 & 0x20) >> 5) << 9);
           vbstart >>= 1;   //divide by 2
           _outp(crtcport+1,vbstart & 0xff);

           _outp(crtcport,0x16);           //vertical blank end
           vbend = _inp(crtcport+1);
           vbend += (((ovflw & 0x8) >> 3) << 8) + (((ovflw_1 & 0x20) >> 5) << 9);
           vbend >>= 1;   //divide by 2
           _outp(crtcport+1,vbend & 0xff);

           new_ovflw |= ((vbstart & 0x100) >> 8) << 3;
           _outp(crtcport,0x7);            //set the new overflow
           _outp(crtcport+1,new_ovflw);

           new_ovflw_1 |= ((vbstart & 0x200) >> 9) << 5;
           _outp(crtcport,0x9);            //set the new overflow
           _outp(crtcport+1,new_ovflw_1);

           _outp(crtcport,0x11);           //vertical retrace end
           rlock = _inp(crtcport+1);
           vrend = rlock & 0xf;
           vrend >>= 1;   //divide by 2
           rlock |= (vrend & 0xf) | 0x80;       //lock crt0-7
           _outp(crtcport+1,rlock & 0xff);

           _outp(crtcport,0x14);
           ovflw_1 = _inp(crtcport+1) & 0x9f;
           _outp(crtcport+1,ovflw_1);           //turn the double word off

           _outp(crtcport,0x17);
           ovflw_1 = _inp(crtcport+1) & 0xf7;
           _outp(crtcport+1,ovflw_1);           //turn the count by 2
           _outp(crtcport,0x42);
           //the clock will be set accordingly.
        }
        _outp(crtcport+1,clock);                //set the clock
        break;
       }
     case 1280:
        /*
        ** Changing from a non-interlaced into interlaced (vc vs) may require adjusting
        ** the vertical crt. Will do so after testing.
        */
        switch(bVrtRefresh)
        {
           case 60:
             if (SVGAHardware.ChipType == S3_86C928_CHIP)
             {
                _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cnew_ovflw2
                _outp(crtcport,0x42);
                r2 = _inp(crtcport+1);
                r2 &= 0xD0;
                _outp(crtcport+1,r2 | 0x0C);
                break;
             }  //805 doesn't support 60Hz. default to 46.
           case 46:
             _outp(0x3c2,r1 | 0x0c);                   //enable clock sel bits in cnew_ovflw2
             _outp(crtcport,0x42);
             r2 = _inp(crtcport+1);
             r2 &= 0xD0;
             _outp(crtcport+1,r2 | 0x2A);
             break;
        }
        break;
   }
}
/*****************************************************************************
 *
 *  FUNCTION NAME:    SetS3SDACClock()
 *
 *  DESCRIPTIVE NAME: Program the SDAC clock for S3 864 chip.
 *
 *  FUNCTION:         This function programs the SDAC based of the information
 *                    extracted from the PMI.
 *
 *  INPUT:            r0 thru r4
 *
 *  EXIT:             NONE
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
//#pragma optimize("',off)

VOID _System  SetS3SDACClock(BYTE r0, BYTE r1, BYTE r2, BYTE r3, BYTE r4)
{
   BYTE  r5, r6;
   USHORT crtcport = (_inp(0x3cc) & 0x01) ? 0x3D4 : 0x3B4;

//inb(r5, 0x3c8);
  r5 = _inp(0x3c8);

//inb(r5, 0x3d4);
  r5 = _inp(0x3d4);

//rmwbi(0x3d4, 0x3d5, 0x55, 0xfc, 0x01);
  _outp(crtcport, 0x55);
  r6  = _inp(crtcport+1);
  r6 &= 0xfc;
  r6 |= 0x01;
  _outp(crtcport+1, r6);

//outb(0x3c6, r4);
  _outp(0x3c6, r4);

//outb(0x3c8, 0x02);
  _outp(0x3c8, 0x02);

//outb(0x3c9, r0);
  _outp(0x3c9, r0);

//outb(0x3c9, r1);
  _outp(0x3c9, r1);

//outb(0x3c8, 0x0a);
  _outp(0x3c8, 0x0a);

//outb(0x3c9, r2);
  _outp(0x3c9, r2);

//outb(0x3c9, r3);
  _outp(0x3c9, r3);

//rmwbi(0x3d4, 0x3d5, 0x55, 0xfc, 0x00);
  _outp(crtcport, 0x55);
  r6 = _inp(crtcport+1);
  r6 &= 0xfc;
  r6 |= 0x00;
  _outp(crtcport+1, r6);

//outb(0x3c6, 0xff);
  _outp(0x3c6, 0xff);

//outb(0x3d4, r5);
  _outp(0x3d4, r5);

//inb(r5, 0x3c8);
  r5 = _inp(0x3c8);
}
/******************************************************************************
*  Fixups for going from interlaced into a non-interlaced mode.
******************************************************************************/

VOID _System FixupS3CRTToNONInterlaced(VOID)
{

  USHORT crtcport = (_inp(0x3cc) & 0x01) ? 0x3D4 : 0x3B4;

  ULONG vtotal, vrstart, vrend, vend, vbstart, vbend;
  ULONG ovflw, new_ovflw, ovflw_1, new_ovflw_1, rlock;

  /*
  ** Divide the clock
  ** Change all vertical crts (*2)
  */

  _outp(crtcport,0x11);   //lock register
  rlock = _inp(crtcport+1) & 0x7f;
  _outp(crtcport+1,rlock); //unlock crtc0-7


  _outp(crtcport,0x7);    //overflow register 1
  ovflw = _inp(crtcport+1);  //overflow bits
  new_ovflw = 0;         //overflow value to be set


  _outp(crtcport,0x9);    //overflow register 2
  ovflw_1 = _inp(crtcport+1);  //overflow bits
  new_ovflw_1 = ovflw_1 & 0xdf;


  _outp(crtcport,0x6);         //vertical total
  vtotal = _inp(crtcport+1);
  vtotal += ((ovflw & 0x1) << 8) + (((ovflw & 0x20) >> 5) << 9);
  vtotal <<= 1;   //multiply by 2
  _outp(crtcport+1,vtotal & 0xff);
  new_ovflw |= ((vtotal & 0x100) >> 8) | (((vtotal & 0x200) >> 9) << 5);


  _outp(crtcport,0x10);        //vertical retrace start
  vrstart = _inp(crtcport+1);
  vrstart += (((ovflw & 0x4) >> 2) << 8) + (((ovflw & 0x80) >> 7) << 9);
  vrstart <<= 1;   //multiply by 2
  _outp(crtcport+1,vrstart & 0xff);
  new_ovflw |= (((vrstart & 0x100) >> 8) << 2) | (((vrstart & 0x200) >> 9) << 7);


  _outp(crtcport,0x12);
  vend = _inp(crtcport+1);        //vertical display end
  vend += (((ovflw & 0x2) >> 1) << 8) + (((ovflw & 0x40) >> 6) << 9);
  vend <<= 1;   //multiply by 2
  _outp(crtcport+1,vend & 0xff);
  new_ovflw |= (((vend & 0x100) >> 8) << 1) | (((vend & 0x200) >> 9) << 6);


  _outp(crtcport,0x15);           //vertical blank start
  vbstart = _inp(crtcport+1);
  vbstart += (((ovflw & 0x8) >> 3) << 8) + (((ovflw_1 & 0x20) >> 5) << 9);
  vbstart <<= 1;   //multiply by 2
  _outp(crtcport+1,vbstart & 0xff);


  _outp(crtcport,0x16);           //vertical blank end
  vbend = _inp(crtcport+1);
  vbend <<= 1;   //multiply by 2
  _outp(crtcport+1,vbend & 0xff);


  new_ovflw |= ((vbstart & 0x100) >> 8) << 3;
  new_ovflw_1 |= ((vbstart & 0x200) >> 9) << 5;
  _outp(crtcport,0x9);            //set the new overflow
  _outp(crtcport+1,new_ovflw_1);


  _outp(crtcport,0x7);            //set the new overflow
  _outp(crtcport+1,new_ovflw);


  _outp(crtcport,0x11);           //vertical retrace end
  rlock = _inp(crtcport+1);
  vrend = rlock & 0xf;
  vrend <<= 1;   //multiply by 2
  rlock |= (vrend & 0xf) | 0x80;       //lock crt0-7
  _outp(crtcport+1,rlock & 0xff);


  _outp(crtcport,0x14);
  ovflw_1 = (_inp(crtcport+1) & 0x9f) | 0x60;
  _outp(crtcport+1,ovflw_1);           //turn the double word on


  _outp(crtcport,0x17);
  ovflw_1 = (_inp(crtcport+1) & 0xf7) | 0x8;
  _outp(crtcport+1,ovflw_1);           //turn the count by 2 off

}
/***************************************************************************
*  Fixups for going from a non-interlaced mode into interlaced mode.
****************************************************************************/

VOID _System FixupS3CRTToInterlaced(VOID)
{

  USHORT crtcport = (_inp(0x3cc) & 0x01) ? 0x3D4 : 0x3B4;

  ULONG vtotal, vrstart, vrend, vend, vbstart, vbend;
  ULONG ovflw, new_ovflw, ovflw_1, new_ovflw_1, rlock, ovflw_2, new_ovflw_2;

  /*
  ** Make sure clock is not divided
  ** Change all vertical crts (/2)
  */

  _outp(crtcport,0x11);   //lock register
  rlock = _inp(crtcport+1) & 0x7f;
  _outp(crtcport+1,rlock); //unlock crtc0-7


  _outp(crtcport,0x7);    //overflow register 1
  ovflw = _inp(crtcport+1);  //overflow bits
  new_ovflw = 0;         //overflow value to be set


  _outp(crtcport,0x9);    //overflow register 2
  ovflw_1 = _inp(crtcport+1);  //overflow bits
  new_ovflw_1 = ovflw_1 & 0xdf;


  _outp(crtcport,0x5E);    //overflow register 3 for vertical
  ovflw_2 = _inp(crtcport+1);  //overflow bits
  new_ovflw_2 = ovflw_2 & 0x1f;


  _outp(crtcport,0x6);         //vertical total
  vtotal = _inp(crtcport+1);
  vtotal += ((ovflw & 0x1) << 8) + (((ovflw & 0x20) >> 5) << 9);
  vtotal += ((ovflw_2  & 0x1) << 10);
  vtotal >>= 1;   //divide by 2
  _outp(crtcport+1,vtotal & 0xff);
  new_ovflw |= ((vtotal & 0x100) >> 8) | (((vtotal & 0x200) >> 9) << 5);
  new_ovflw_2 |= ((vtotal & 0x400) >> 10);


  _outp(crtcport,0x10);        //vertical retrace start
  vrstart = _inp(crtcport+1);
  vrstart += (((ovflw & 0x4) >> 2) << 8) + (((ovflw & 0x80) >> 7) << 9);
  vrstart += (((new_ovflw_2 & 0x10) >> 4) << 10);
  vrstart >>= 1;   //divide by 2
  _outp(crtcport+1,vrstart & 0xff);
  new_ovflw |= (((vrstart & 0x100) >> 8) << 2) | (((vrstart & 0x200) >> 9) << 7);
  new_ovflw_2 |= (((vrstart & 0x400) >> 10) << 4);


  _outp(crtcport,0x12);
  vend = _inp(crtcport+1);        //vertical display end
  vend += (((ovflw & 0x2) >> 1) << 8) + (((ovflw & 0x40) >> 6) << 9);
  vend += (((ovflw_2 & 0x2) >> 1) << 10);
  vend >>= 1;   //divide by 2
  _outp(crtcport+1,vend & 0xff);
  new_ovflw |= (((vend & 0x100) >> 8) << 1) | (((vend & 0x200) >> 9) << 6);
  new_ovflw_2 |= (((vend & 0x400) >> 10) << 1);


  _outp(crtcport,0x15);           //vertical blank start
  vbstart = _inp(crtcport+1);
  vbstart += (((ovflw & 0x8) >> 3) << 8) + (((ovflw_1 & 0x20) >> 5) << 9);
  vbstart += (((ovflw_2 & 0x4) >> 2) << 10);
  vbstart >>= 1;   //divide by 2
  _outp(crtcport+1,vbstart & 0xff);


  _outp(crtcport,0x16);           //vertical blank end
  vbend = _inp(crtcport+1);
  vbend >>= 1;   //divide by 2
  _outp(crtcport+1,vbend & 0xff);


  new_ovflw |= ((vbstart & 0x100) >> 8) << 3;
  _outp(crtcport,0x7);            //set the new overflow
  _outp(crtcport+1,new_ovflw);


  new_ovflw_1 |= ((vbstart & 0x200) >> 9) << 5;
  _outp(crtcport,0x9);            //set the new overflow
  _outp(crtcport+1,new_ovflw_1);


  _outp(crtcport,0x11);           //vertical retrace end
  rlock = _inp(crtcport+1);
  vrend = rlock & 0xf;
  vrend >>= 1;   //divide by 2
  rlock |= (vrend & 0xf) | 0x80;       //lock crt0-7
  _outp(crtcport+1,rlock & 0xff);


  _outp(crtcport,0x14);
  ovflw_1 = _inp(crtcport+1) & 0x9f;
  _outp(crtcport+1,ovflw_1);           //turn the double word off


  _outp(crtcport,0x17);
  ovflw_1 = _inp(crtcport+1) & 0xf7;
  _outp(crtcport+1,ovflw_1);           //turn the count by 2


  new_ovflw_2 |= (((vbstart & 0x400) >> 10) << 2);
  _outp(crtcport,0x5e);
  _outp(crtcport+1,new_ovflw_2);           //turn the count by 2

}
/**********************************************************************
 *
 * FUNCTION NAME:    SetCirrusClock()                   @V3.0JAO01
 *
 * DESCRIPTIVE NAME: Program on-chip clock for Cirrus.
 *
 * FUNCTION:         All "generic" Cirrus cards run this code.
 *
 * INPUT:            Pointer to structure pAdapter of type
 *                      PVIDEO_ADAPTER.
 *
 * EXIT:             Pass/fail indication of type APIRET with values:
 *                      NO_ERROR
 *                      ERROR_MODE_NOT_SUPPORTED
 *                      ERROR_REFRESH_NOT_SUPPORTED
 *
 * INTERNAL REFERENCES: NONE
 *   ROUTINES:
 *
 * EXTERNAL REFERENCES: NONE
 *   ROUTINES:
 *
 **********************************************************************/
APIRET _System SetCirrusClock( PVIDEO_ADAPTER pAdapter )
{

APIRET rc = NO_ERROR;
USHORT volatile miscport = 0x3c2;
USHORT volatile xseqport = 0x3c4;
USHORT volatile hdacport = 0x3c6;
USHORT volatile crtcport = (_inp(0x3cc) & 0x01) ? 0x3d4 : 0x3b4;
enum { CLGD5430 , CLGD5434 , OTHER }
                chiptype = (USHORT) OTHER;
enum { BELOW2M , TWOMEG , ABOVE2M }
                TotalMemory = (USHORT) BELOW2M;

ULONG XResolution  = pAdapter->ModeInfo.usXResolution;
ULONG BitsPerPixel = pAdapter->ModeInfo.bBitsPerPixel;
ULONG VrtRefresh   = pAdapter->ModeInfo.bVrtRefresh;

typedef struct { BYTE cr0,  cr3,  cr4,  cr5,  cr6,
                      cr7,  cr9,  cr10, cr12, cr15,
                      cr16, cr11, cr1c, cr19, cr1a,
                      cr1,  cr2,  cr17; } CRFX;
                 BYTE *cri, *fxi;
        struct { CRFX crx,  fx0,  fx1,  fx2,  fx3,  fx4,
                      fx5,  fx6,  fx7,  fx8,  fx9,  fxA,
                      fxB,  fxC,  fxD,  fxE,  fxF,  fx10,
                      fx11, fx12, fx13, fx14; } rf = {
{0x00,0x03,0x04,0x05,0x06,0x07,0x09,0x10,0x12,0x15,0x16,0x11,0x1c, /*i*/
 0x19,0x1a,0x01,0x02,0x17},
{0x64,0x88,0x53,0x9b,0xf2,0x1f,0x40,0xe0,0xdf,0xdf,0xf3,0x03,0x02},/*0*/
{0x64,0x87,0x54,0x9c,0xf2,0x1f,0x40,0xe1,0xdf,0xe7,0xeb,0x04,0x06},/*1*/
{0x5f,0x82,0x54,0x9f,0x0b,0x3e,0x40,0xea,0xdf,0xe7,0x04,0x0c,0x00},/*2*/
{0x7b,0x9e,0x68,0x91,0x6f,0xf0,0x60,0x58,0x57,0x58,0x6f,0x0a,0x00},/*3*/
{0x5f,0x82,0x53,0x9f,0x0b,0x3e,0x40,0xea,0xdf,0xe7,0x04,0x0c,0x00},/*4*/
{0x7f,0x82,0x6b,0x1b,0x72,0xf0,0x60,0x58,0x57,0x58,0x72,0x0c,0x04},/*5*/
{0x7d,0x80,0x6d,0x1c,0x98,0xf0,0x60,0x7c,0x57,0x5f,0x91,0x02,0x00},/*6*/
{0x7f,0x82,0x68,0x12,0x6f,0xf0,0x60,0x58,0x57,0x57,0x6f,0x0b,0x00},/*7*/
{0xa3,0x86,0x85,0x96,0x24,0xfd,0x60,0x02,0xff,0x00,0x24,0x08,0x00},/*8*/
{0xa1,0x84,0x85,0x96,0x24,0xfd,0x60,0x02,0xff,0x00,0x24,0x08,0x00},/*9*/
{0xa1,0x84,0x85,0x93,0x2a,0xfd,0x60,0x12,0xff,0x00,0x2a,0x09,0x00},/*A*/
{0x9f,0x82,0x84,0x90,0x1e,0xf5,0x60,0x00,0xff,0xff,0x1e,0x13,0x00},/*B*/
{0x99,0x9c,0x84,0x1a,0x96,0x1f,0x40,0x80,0x7f,0x80,0x96,0x04,0x00, /*C*/
 0x4a,0x01},
{0xbd,0x80,0xa5,0x1a,0x2a,0xb2,0x60,0x0b,0xff,0x00,0x2a,0x00,0x00, /*D*/
 0x60,0x01,0x9f,0xa0,0xe3},
{0xd2,0x95,0xa2,0x81,0x14,0xb2,0x60,0x05,0xff,0x00,0x12,0x0b,0x00},/*E*/
{0x63,0x9a,0x53,0x1e,0x14,0xb2,0x60,0x03,0xff,0x00,0x12,0x07,0x04},/*F*/
{0x63,0x9a,0x52,0x1e,0x14,0xb2,0x60,0x03,0xff,0x00,0x12,0x07,0x01},/*1*/
                                                                   /*0*/
{0x64,0x9a,0x52,0x1b,0x15,0xb2,0x60,0x00,0xff,0x00,0x12,0x01,0x07},/*1*/
                                                                   /*1*/
{0x63,0x86,0x54,0x99,0x06,0x3e,0x40,0xe9,0xdf,0xe8,0xff,0x0c,0x03},/*1*/
                                                                   /*2*/
{0x7f,0x82,0x6a,0x1a,0x72,0xf0,0x60,0x58,0x57,0x58,0x72,0x0c,0x00},/*1*/
                                                                   /*3*/
{0x5f,0x82,0x53,0x9f,0x0b,0x3e,0x40,0xea,0xdf,0xe7,0x04,0x0c,0x04}}/*1*/
                                                                  ;/*4*/

_outp(xseqport, 0x06); _outp(xseqport+1, 0x12);
_outp(crtcport, 0x11); _outp(crtcport+1, (_inp(crtcport+1) & 0x7f) );

_outp(crtcport,0x27);
if ( (_inp(crtcport+1) & 0xfc) == 0xa0) chiptype = CLGD5430;
if ( (_inp(crtcport+1) & 0xfc) == 0xa8) chiptype = CLGD5434;

_outp(xseqport,0x15);
if ( (_inp(xseqport+1) & 0x0f) <  0x03) TotalMemory = BELOW2M;
if ( (_inp(xseqport+1) & 0x0f) == 0x03) TotalMemory = TWOMEG;
if ( (_inp(xseqport+1) & 0x0f) >  0x03) TotalMemory = ABOVE2M;

if (chiptype != OTHER)
switch(XResolution)
{
 default:
      rc = ERROR_MODE_NOT_SUPPORTED;
 break;
 case 1280:
      switch(BitsPerPixel)
      {
       default:
            rc = ERROR_MODE_NOT_SUPPORTED;
       break;
       case 16:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 43:
             case 44:
                 if (chiptype != CLGD5434)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 11a*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x6e);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x2a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x07);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxD.cr0; fxi<=&rf.fxD.cr17;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
       break;
       case 8:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 43:
             case 44:
                 /*vesa mode 107*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x6e);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x2a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  _inp(hdacport);_inp(hdacport);_inp(hdacport);_inp(hdacport);
                  _outp(hdacport, 0x00);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxD.cr0; fxi<=&rf.fxD.cr17;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
      }
 break;
 case 1024:
      switch(BitsPerPixel)
      {
       default:
            rc = ERROR_MODE_NOT_SUPPORTED;
       break;
       case 24:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 43:
             case 44:
                 if (chiptype != CLGD5434)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 118*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x55);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x36);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x09);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxC.cr0; fxi<=&rf.fxC.cr1a;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
       break;
       case 16:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 75:
                 if (chiptype != CLGD5434)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 117*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x2c);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x10);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x07);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxB.cr0; fxi<=&rf.fxB.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 70:
                 if (chiptype != CLGD5434)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 117*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x6e);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x2a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x07);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx9.cr0; fxi<=&rf.fx9.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 60:
                 if (chiptype != CLGD5434)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 117*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x3b);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x1a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x07);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx8.cr0; fxi<=&rf.fx8.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 43:
             case 44:
                 /*vesa mode 117*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x55);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x36);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x07);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxC.cr0; fxi<=&rf.fxC.cr1a;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
       break;
       case 8:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 75:
                 /*vesa mode 105*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x2c);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x10);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxB.cr0; fxi<=&rf.fxB.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 72:
                 if (chiptype != CLGD5430)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 105*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x61);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x24);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxA.cr0; fxi<=&rf.fxA.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 70:
                 /*vesa mode 105*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x6e);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x2a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx9.cr0; fxi<=&rf.fx9.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 60:
                 /*vesa mode 105*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x3b);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x1a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx8.cr0; fxi<=&rf.fx8.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 43:
             case 44:
                 /*vesa mode 105*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x55);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x36);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fxC.cr0; fxi<=&rf.fxC.cr1a;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
      }
 break;
 case 800:
      switch(BitsPerPixel)
      {
       default:
            rc = ERROR_MODE_NOT_SUPPORTED;
       break;
       case 24:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 60:
                 if (chiptype != CLGD5434)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 115*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x51);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x3a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x09);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx5.cr0; fxi<=&rf.fx5.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 56:
                 if (chiptype != CLGD5434)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 115*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x7e);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x33);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x09);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx3.cr0; fxi<=&rf.fx3.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
       break;
       case 16:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 75:
                 if ((chiptype != CLGD5434) || (TotalMemory == BELOW2M))
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 114*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x53);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x30);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x07);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx7.cr0; fxi<=&rf.fx7.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 72:
                 if (TotalMemory == BELOW2M)
                 {
                  rc = ERROR_REFRESH_NOT_SUPPORTED; break;
                 }
                 /*vesa mode 114*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x64);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x3a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x07);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx6.cr0; fxi<=&rf.fx6.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 60:
                 /*vesa mode 114*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x5f);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x22);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x03);
                  for
                  (cri=&rf.crx.cr0,fxi=&rf.fx13.cr0;fxi<=&rf.fx13.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 56:
                 /*vesa mode 114*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x7e);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x32);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x03);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx3.cr0; fxi<=&rf.fx3.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
       break;
       case 8:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 75:
                 /*vesa mode 103*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x53);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x30);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx7.cr0; fxi<=&rf.fx7.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 72:
                 /*vesa mode 103*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x64);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x3a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx6.cr0; fxi<=&rf.fx6.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 60:
                 /*vesa mode 103*/
                  _outp(miscport, 0x2f);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x51);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x3a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx5.cr0; fxi<=&rf.fx5.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 56:
                 /*vesa mode 103*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x7e);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x33);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx3.cr0; fxi<=&rf.fx3.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
      }
 break;
 case 640:
      switch(BitsPerPixel)
      {
       default:
            rc = ERROR_MODE_NOT_SUPPORTED;
       break;
       case 24:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 60:
                 /*vesa mode 112*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x3a);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x16);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x05);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx4.cr0; fxi<=&rf.fx4.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
       break;
       case 16:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 75:
                 /*vesa mode 111*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x65);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x2e);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x03);
                  _outp(crtcport, 0x1b); _outp(crtcport+1, 0x22);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx0.cr0; fxi<=&rf.fx0.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 72:
                 /*vesa mode 111*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x65);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x2e);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x03);
                  _outp(crtcport, 0x1b); _outp(crtcport+1, 0x22);
                  for
                  (cri=&rf.crx.cr0,fxi=&rf.fx12.cr0;fxi<=&rf.fx12.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 60:
                 /*vesa mode 111*/
                  _outp(miscport, 0xef);
                  _outp(xseqport, 0x0e); _outp(xseqport+1, 0x65);
                  _outp(xseqport, 0x1e); _outp(xseqport+1, 0x3a);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x03);
                  _outp(crtcport, 0x1b); _outp(crtcport+1, 0x02);
                  for
                  (cri=&rf.crx.cr0,fxi=&rf.fx14.cr0;fxi<=&rf.fx14.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
       break;
       case 8:
            switch(VrtRefresh)
            {
             default:
                 rc = ERROR_REFRESH_NOT_SUPPORTED;
             break;
             case 75:
                 /*vesa mode 101*/
                  _outp(miscport, 0xeb);
                  _outp(xseqport, 0x0b); _outp(xseqport+1, 0x4a);
                  _outp(xseqport, 0x1b); _outp(xseqport+1, 0x2b);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0, fxi=&rf.fx1.cr0; fxi<=&rf.fx1.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 72:
                 /*vesa mode 101*/
                  _outp(miscport, 0xeb);
                  _outp(xseqport, 0x0b); _outp(xseqport+1, 0x4a);
                  _outp(xseqport, 0x1b); _outp(xseqport+1, 0x2b);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0,fxi=&rf.fx12.cr0;fxi<=&rf.fx12.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
             break;
             case 60:
                 /*vesa mode 101*/
                  _outp(miscport, 0xe3);
                  _outp(xseqport, 0x0b); _outp(xseqport+1, 0x4a);
                  _outp(xseqport, 0x1b); _outp(xseqport+1, 0x2b);
                  _outp(xseqport, 0x07); _outp(xseqport+1, 0x01);
                  for
                  (cri=&rf.crx.cr0,fxi=&rf.fx4.cr0;fxi<=&rf.fx4.cr1c;)
                  {
                   _outp(crtcport,*cri++); _outp(crtcport+1,*fxi++);
                  }
            }
      }
}
/* _outp(crtcport, 0x11); _outp(crtcport+1, (_inp(crtcport+1) | 0x80) ); */

return(rc);
}                                                     /* @V3.0JAO01 end */
/*****************************************************************************
 *
 *  FUNCTION NAME:      pfnPMIFixupClock()
 *
 *  DESCRIPTIVE NAME:   Fixup the clock related registers, usage is per adapter.
 *
 *  FUNCTION:           This routine works in conjunction with pfnPMISetMonitorTimings
 *                      and does post-mode fixups which can't be done in pfnPMISetMonitorTimings
 *                      due to some timing or order constraints on adapter basis
 *
 *  INPUT:              PVIDEO_ADAPTER - Pointer to current state of the adapter/mode
 *                      PREGS- Pointer to current register state
 *
 *  EXIT:               APIRET - return code
 *
 *  NOTES:              pfnPMISetMonitorTimings can be assumed as already called
 *                      either internally (in the PMI) or externally (from IBMGPMI).
 *                      Since PMI interface specifies that r0 will be set to reflect
 *                      the return from the function and we doing the fixup to r0 here,
 *                      make sure that we return the new value of r0. Very convoluted, but
 *                      so is the hardware too and svga.exe is already bloated with
 *                      special casing, so I'm trying to retain all of the special casing
 *                      for the clock in this module.
 *
 *  INTERNAL REFERENCES:
 *    ROUTINES:
 *
 *  EXTERNAL REFERENCES:
 *    ROUTINES:
 *
 ****************************************************************************/
APIRET EXPENTRY  pfnPMIFixupClock(PVIDEO_ADAPTER pAdapter,PREGS pRegs)  /*            */
{
  BYTE ClockByte;
  BYTE bData;                                                      //JWK22
  USHORT volatile crtcport = (_inp(0x3cc) & 0x01) ? 0x3d4 : 0x3b4; //JWK22

  if (!(flAdapterSupport & ADAPTER_CLOCK_SUPPORTED)                //proceed
     || (!((SVGAHardware.ChipType > TSENG_ET4000_CHIP) &&
           OEMHardware.Manufacturer == DIAMOND_MANUFACTURER))
     || (pAdapter->ModeInfo.bBitsPerPixel < 16))
    return (pRegs[0]);

  switch(pAdapter->ModeInfo.bBitsPerPixel)
  {
    case 16:
      /*
      ** Force the DAC to double the pixel rate. Make sure ATC 16(5:4) = 10
      ** r0 is command reg value, r1 is the primary pixel mode and r2 is the
      ** secondary pixel mode. Change r1 & r2 only.
      */
      pRegs[1] = 3;
      pRegs[2] = 3;
      _inp(0x3da);_outp(0x3c0,0x16);      //JWK22
      ClockByte = _inp(0x3c1);
      _inp(0x3da);_outp(0x3c0,0x16);      //JWK22
      _outp(0x3c1,ClockByte | 0x20);

//      if (800) fixup crt 5,6,16,17
//JWK22 if (800) fixup crt 0,1,2,3,4,5,3f
      if (pAdapter->ModeInfo.usXResolution == 800)
      {
          _outp(crtcport, 0x00);       /* select crtc reg 0 7:0 */
          _outp(crtcport+1, 0x7f);

          _outp(crtcport, 0x01);       /* select crtc reg 1 7:0 */
          _outp(crtcport+1, 0x63);

          _outp(crtcport, 0x02);       /* select crtc reg 2 7:0 */
          _outp(crtcport+1, 0x64);


          _outp(crtcport, 0x03);       /* select crtc reg 3 4:0 */
          bData = _inp(crtcport+1) & ~0x1f;
//                    bData |= 0x02;
          bData |= 0x1d;                //          
          _outp(crtcport+1, bData);

          _outp(crtcport, 0x04);       /* select crtc reg 4 7:0 */
          _outp(crtcport+1, 0x6a);

          _outp(crtcport, 0x05);       /* select crtc reg 5 4:0 */
          bData = _inp(crtcport+1) & ~0x1f;
//                    bData |= 0x1d;
          bData |= 0x9a;                //          
          _outp(crtcport+1, bData);

          _outp(crtcport, 0x3f);       /* select crtc reg x3f bits 0,2,4 */
          bData = _inp(crtcport+1) & ~0x15;
          bData |= 0x00;
          _outp(crtcport+1, bData);
      }
      break;


    case 24:
      if (StealthOldScheme)
      {
        pRegs[0] = 0xF0;
        pRegs[1] = 4;
        pRegs[2] = 4;
        _inp(0x3da);_outp(0x3c0,0x16);
        ClockByte = _inp(0x3c1);
        _inp(0x3da);_outp(0x3c0,0x16);
        _outp(0x3c1,ClockByte & 0xDF);
      }
      else
      {
        /*
        ** This scheme allows higher resolutions and refreshes thanks
        ** to the fact that DAC is doubling the clock produced by ICD
        ** which allows us to escape the 90MHz limit of the old scheme.
        */
//      if (800) fixup crt 5,6,16,17
//JWK22 if (800) fixup crt 0,1,2,3,4,5,3f
        if (pAdapter->ModeInfo.usXResolution == 800)
        {
            _outp(crtcport, 0x00);       /* select crtc reg 0 7:0 */
            _outp(crtcport+1, 0xc1);

            _outp(crtcport, 0x01);       /* select crtc reg 1 7:0 */
            _outp(crtcport+1, 0x95);

            _outp(crtcport, 0x02);       /* select crtc reg 2 7:0 */
            _outp(crtcport+1, 0x95);


            _outp(crtcport, 0x03);       /* select crtc reg 3 4:0 */
            bData = _inp(crtcport+1) & ~0x1f;
            bData |= 0x05;
            _outp(crtcport+1, bData);

            _outp(crtcport, 0x04);       /* select crtc reg 4 7:0 */
            _outp(crtcport+1, 0x9e);

            _outp(crtcport, 0x05);       /* select crtc reg 5 4:0 */
            bData = _inp(crtcport+1) & ~0x1f;
            bData |= 0x13;
            _outp(crtcport+1, bData);

            _outp(crtcport, 0x3f);       /* select crtc reg x3f bits 0,2,4 */
            bData = _inp(crtcport+1) & ~0x15;
            bData |= 0x00;
            _outp(crtcport+1, bData);
        }

        pRegs[1] = 9;
        pRegs[2] = 9;
        _inp(0x3da);_outp(0x3c0,0x16);
        ClockByte = _inp(0x3c1);
        _inp(0x3da);_outp(0x3c0,0x16);
        _outp(0x3c1,ClockByte | 0x20);
      }
      break;
  }
  return (pRegs[0]);
}
//#pragma optimize("',on)
