/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Microsoft Corporation, 1989                                 */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/*      Page  (58,132)
**      Title   KBDSTRA - Kbd Device Driver Strategy Routine
**      Name    Kbdstra
*/

/*****************************************************************************/
/*                                                                           */
/* SOURCE FILE NAME = KBDSTRA.C                                              */
/*                                                                           */
/* DESCRIPTIVE NAME = Physical Keyboard Device Driver Strategy Routine       */
/*                                                                           */
/*                                                                           */
/* VERSION      V2.2                                                         */
/*                                                                           */
/* DATE         01/11/94                                                     */
/*                                                                           */
/* DESCRIPTION  Entry Point from Device Independent Driver and Kernel.       */
/*                                                                           */
/* FUNCTIONS   void Block_Thread(void)                                       */
/*             void Run_Thread(void)                                         */
/*           USHORT Query_Open(void)                                         */
/*           USHORT Query_Close(void)                                        */
/*           USHORT Query_Capabilites(USHORT,USHORT)                         */
/*           USHORT Query_Typematic(void)                                    */
/*           USHORT Query_LEDs(void)                                         */
/*           USHORT Query_ID(USHORT,USHORT)                                  */
/*           USHORT Query_Disabled(void)                                     */
/*           USHORT Disable_Keyboard(void)                                   */
/*           USHORT Enable_Keyboard(void)                                    */
/*           USHORT CheckKBD(void)                                           */
/*           USHORT Reset_Hardware(void)                                     */
/*           USHORT Set_Typematic(BYTE)                                      */
/*           USHORT Set_LEDs(BYTE)                                           */
/*           USHORT Send_Generic(USHORT,USHORT)                              */
/*           USHORT Query_Kbd_Rdy(void)                                      */
/*           USHORT Flush_Partial(void)                                      */
/*           USHORT Save_State(void)                                         */
/*           USHORT Restore_State(void)                                      */
/*           USHORT IDC_Entry_Point(USHORT,USHORT,USHORT)                    */
/*             void Strategy(void)                                           */
/*                                                                           */
/* NOTES        DEPENDENCIES:  Controller or keyboard must be set to the     */
/*                             PC compatible scan code set.                  */
/*              RESTRICTIONS:  Machine must be a 386 or compatible with      */
/*                             a 386.                                        */
/*                                                                           */
/*                                                                           */
/* STRUCTURES     CapStruc                                                   */
/*                IDstruc                                                    */
/*                CmdStruc                                                   */
/*                                                                           */
/* EXTERNAL FUNCTIONS                                                        */
/*              KbdStatusWait                                                */
/*              SendData                                                     */
/*              DevHelp_ProcBlock                                            */
/*              DevHelp_ProcRun                                              */
/*              DevHelp_Yield                                                */
/*                                                                           */
/* ??CHANGE ACTIVITY =                                                       */
/* ??  DATE      FLAG        APAR   CHANGE DESCRIPTION                       */
/* ??  --------  ----------  -----  -----------------------------------------*/
/* ??  mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx                                  */
/*                                                                           */
/*****************************************************************************/

/*-----------------------------*/
/*     Include files           */
/*-----------------------------*/

 #include "os2.h"
 #include "conio.h"
 #include "devcmd.h"
 #include "strat2.h"
 #include "reqpkt.h"
 #include "dhcalls.h"
 #include "kbd.h"
 #include "kbddd.h"

/*-----------------------------*/
/*   Define Constants          */
/*-----------------------------*/

#define MK_FP(seg,off) (void FAR *)(((ULONG)(((ULONG)(seg)<<16)+(ULONG)(off))))

/*-----------------------------*/
/*     External Variables      */
/*-----------------------------*/

 extern  PRPH  pRPH;

 extern  USHORT KData;
 extern  USHORT KbdCS;
 extern  USHORT Keycount;

 extern  BYTE  Current_State;
 extern  BYTE  LED_State;
 extern  BYTE  Type_Rate;
 extern  BYTE  DataByte1;
 extern  BYTE  DataByte2;
 extern  BYTE  CMD_In_Progress;
 extern  BYTE  InterruptTime;
 extern  BYTE  KbdError;
 extern  BYTE  DisableFlag;
 extern  BYTE  Save_LED;
 extern  BYTE  Save_Type;
 extern  BYTE  Next_Block;
 extern  BYTE  Next_Run;
 extern  BYTE  KbdResend;
 extern  BYTE  Enabler;

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Block_Thread                                             */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Blocks the current thread.  Checks for normal wake-up.   */
/*                                                                           */
/*****************************************************************************/

void Block_Thread(void)
{
  Next_Block++;                             /* Increment pointer.            */
/*                                                                           */
/* Block the current thread and test if it is a natural wakeup.              */
/*                                                                           */
  if (DevHelp_ProcBlock((ULONG)MK_FP(KbdCS,Next_Block),(ULONG)-1,(USHORT)0) != 0) {
    Next_Block--;                       /* If unnatural then remove pointer. */
  }
  CLI;
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Run_Thread                                               */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Starts up the next blocked thread.                       */
/*                                                                           */
/*****************************************************************************/

void Run_Thread(void)
{
  USHORT *awake;

  Next_Run++;                               /* Increment pointer to threads. */
                                            /* Run blocked thread.           */
  DevHelp_ProcRun((ULONG)MK_FP(KbdCS,Next_Run), (PUSHORT) &awake);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Query_Open                                               */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* RETURN-NORMAL  : AX=0  : There is no command in progress.                 */
/*                  AX=-1 : There is currently a command in progress.        */
/*                                                                           */
/*****************************************************************************/

USHORT Query_Open(void)
{
  return(CMD_In_Progress ? -1 : 0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Query_Close                                              */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : This function has no function.                           */
/*                                                                           */
/*****************************************************************************/

USHORT Query_Close(void)
{
  return (0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Query_Capabilities                                       */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* INPUT          : Offset and data segment to structure that needs to be    */
/*                  filled in with keyboard capabilities                     */
/*                                                                           */
/*****************************************************************************/

USHORT Query_Capabilites(USHORT CMD_Offset, USHORT CMD_Segment)
{
  CapStruc FAR *ptr;

  ptr=MK_FP(CMD_Segment,CMD_Offset);        /* Initialize pointer.           */
  ptr->DevFlags = 0x33;                     /* Three LEDs, HotPlug, Keyboard.*/
  ptr->KeyCount = Keycount;                 /* Number of keys.               */
  ptr->MaxTypa = 31;                        /* Max type rate = 30 codes sec. */
  ptr->MinTypa = 0;                         /* Min type rate = 2  codes sec. */
  ptr->MaxDelay = 3;                        /* Max delay = 1 second.         */
  ptr->MinDelay = 0;                        /* Min delay = 250 milliseconds. */
  return(0);                                /* Return finished with command. */
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Query_Typematic                                          */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* RETURN-NORMAL  : Current type rate and delay                              */
/*                                                                           */
/*****************************************************************************/

USHORT Query_Typematic(void)
{
  return(Type_Rate);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Query_LEDs                                               */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* RETURN-NORMAL  : Current LED state                                        */
/*                                                                           */
/*****************************************************************************/

USHORT Query_LEDs(void)
{
  return(LED_State);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Query_ID                                                 */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* INPUT          : Offset and data segment to structure that needs to be    */
/*                  filled in with keyboard identification.                  */
/*                                                                           */
/* RETURN-NORMAL  : Number of keys on the keyboard.                          */
/*                                                                           */
/*****************************************************************************/

USHORT Query_ID(USHORT CMD_Offset, USHORT CMD_Segment)
{
  IDstruc FAR *ptr;

  ptr = MK_FP(CMD_Segment,CMD_Offset);      /* Initialize pointer.           */
  ptr->idlen = 2;                           /* Enter two ID byte length.     */
  ptr->idbytes[0] = DataByte1;              /* Store first byte.             */
  ptr->idbytes[1] = DataByte2;              /* Store second byte.            */
  return(Keycount);                         /* Return number of keys.        */
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Disable_Keyboard                                         */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* RETURN-NORMAL  : AX=0  : Keyboard is enabled.                             */
/*                  AX=-1 : Keyboard is disabled.                            */
/*                                                                           */
/*****************************************************************************/

USHORT Query_Disabled(void)
{
  return(DisableFlag ? -1 : 0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Disable_Keyboard                                         */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Disable the keyboard.                                    */
/*                                                                           */
/* RETURN-NORMAL  : AX=0 : Did not disable keyboard.                         */
/*                  AX=1 : Disabled keyboard.                                */
/*                                                                           */
/*****************************************************************************/

USHORT Disable_Keyboard(void)
{
  if (DisableFlag == TURNOFF) {             /* keyboard not already disabled */
    CLI;
    if (KbdStatusWait()) {                  /* Make sure keyboard is ready.  */
      outp(PORT_CMD, CMD_DISABLEKBD);       /* Disable Keyboard.             */
      DisableFlag = TURNON;                 /* Show that it is disabled.     */
      STI;
      return(1);                            /* Return that keyboard disabled.*/
    }
    STI;
  }
  return(0);                             /* Kbd not disabled by this routine.*/
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Enable_Keyboard                                          */
/*                  "IBM Unique Code"                                        */
/*                                                                           */
/* DESCRIPTION    : Enable the keyboard.                                     */
/*                                                                           */
/* RETURN-NORMAL  : AX=0 : Enabled keyboard.                                 */
/*                                                                           */
/*****************************************************************************/

USHORT Enable_Keyboard(void)
{
  if (DisableFlag == TURNON) {              /* If disabled.                  */
    CLI;
    if (KbdStatusWait()) {                  /* Make sure keyboard is ready.  */
      outp(PORT_CMD, CMD_ENABLEKBD);        /* Enable Keyboard.              */
      DisableFlag = TURNOFF;                /* Show that it is enabled.      */
      Enabler = Enable_OFF;                 /* Reset Enabler.                */
    }
    STI;
  }
  return(0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : CheckKBD                                                 */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Find out if a command is in progress already.  If there  */
/*                  is, and it's not interrupt time then block this thread.  */
/*                  If it is interrupt time then we can not run this routine */
/*                  at this time.  If there isn't a command in progress then */
/*                  continue to process this command.                        */
/*                                                                           */
/*****************************************************************************/

USHORT CheckKBD(void)
{
  CLI;
  while (CMD_In_Progress) {                 /* If someone not useing kbd.    */
    if (!(InterruptTime)) {                 /* If not interrupt time         */
      Block_Thread();                       /*  then block thread.           */
    }
    else {
      KeyERROR();
      return(0);                            /* Cannot process at this time.  */
    }
  }
  STI;
  CMD_In_Progress = TURNON;                 /* Flag that we are doing a CMD. */
  return(Disable_Keyboard());               /* Disable keyboard.             */
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Reset_Hardware                                           */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Send a reset to the keyboard hardware.                   */
/*                                                                           */
/*****************************************************************************/

USHORT Reset_Hardware(void)
{
  if (CheckKBD()) {
    Enabler = HW_Enable;                    /* Reset_HW  disabled keyboard.  */
  }
  if (!KbdError) {
    do {
      Current_State = S_WAIT_ACK;           /* Change current state.         */
      SendData(KBD_RESET);                  /* Reset keyboard.               */

      while (Current_State != S_NOCMDIPG) { /* Wait until CMD finished.      */
        if (!(InterruptTime)) {             /* If not int & not init time    */
          DevHelp_Yield();                  /*  then Yield processor.        */
        }
      }
    }  while (KbdResend);
  }
  CMD_In_Progress = TURNOFF;                /* Clear CMD in Progress flag.   */
  if (Enabler == HW_Enable) {               /* If we disabled keyboard       */
    Enable_Keyboard();                      /*  then enable it.              */
  }
  else {
    DisableFlag = TURNOFF;
    Disable_Keyboard();                     /* Make sure kbd is disabled.    */
  }
  if (Next_Block != Next_Run) {             /* Check if blocked thread.      */
    Run_Thread();                           /* If yes then run it.           */
  }
  return(0);                                /* Return done.                  */
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Set_Typematic                                            */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Set the type rate and delay of the keyboard.             */
/*                                                                           */
/*****************************************************************************/

USHORT Set_Typematic(BYTE New_Type_Rate)
{
  if (New_Type_Rate != Type_Rate) {         /* If type rate is not the same. */
    if (CheckKBD()) {
      Enabler = TYPE_Enable;                /* Type Rate disabled keyboard.  */
    }
    if (!KbdError) {
      Type_Rate = New_Type_Rate;            /* Get new type rate.            */
      Current_State = S_SENTTYPC;           /* Set new current state.        */
      SendData(KBD_SETREPEAT);              /* Send data to keyboard.        */
    }
  }
  return(0);                                /* Return that we are finished.  */
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Set_LEDs                                                 */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Changes the state of the keyboard LEDs.                  */
/*                                                                           */
/* INPUT          : New state of LEDs                                        */
/*                                                                           */
/*****************************************************************************/

USHORT Set_LEDs(BYTE New_LED_State)
{
  if (New_LED_State != LED_State) {         /* Do we need to change the LEDs?*/
    if (CheckKBD()) {
      Enabler = LEDS_Enable;                /* LEDs disabled keyboard.       */
    }
    if (!KbdError) {
      LED_State = New_LED_State;            /* Save new LED state.           */
      Current_State = S_SENTLEDC;           /* Change current state.         */
      SendData(KBD_SETLEDS);                /* Send data to keyboard.        */
    }
  }
  return(0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Send_Generic                                             */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : This function will send a command or set of commands     */
/*                  providing that the data structure is set up right and    */
/*                  the wait necessary are consistent.                       */
/*                                                                           */
/* INPUT          : Data stucture offset and data segment.                   */
/*                                                                           */
/*****************************************************************************/

USHORT Send_Generic(USHORT CMD_Offset, USHORT CMD_Segment)
{
  USHORT c;
  CmdStruc FAR *ptr;

  ptr=MK_FP(CMD_Segment,CMD_Offset);        /* Initialize pointer.           */
  if (CheckKBD()) {
    Enabler = GEN_Enable;                   /* Send_Gen disabled keyboard.   */
  }
  if (!KbdError) {
    if (ptr->wait_time == TURNOFF) {
      for (c=0; c < ptr->strlength; c++) {   /* Loop through commands.       */
        if (KbdStatusWait())  {              /* Wait for controller ready.   */
          outp(PORT_BUFFER,ptr->strbytes[c]);/* Send to keyboard.            */
          IODelay();                         /* Delay to allow keyboard setup*/
        }
      }
    }
    else {
      for (c=0; c < ptr->strlength; c++) {  /* Loop through commands.        */
        do  {
          Current_State = S_WAIT_ACK;       /* Change current state.         */
          SendData(ptr->strbytes[c]);       /* Reset keyboard.               */

          while (Current_State != S_NOCMDIPG) { /* Wait until CMD finished.  */
            if (!(InterruptTime)) {             /* If not interrupt time.    */
              DevHelp_Yield();                  /*  then Yield processor.    */
            }
          }
        }  while (KbdResend);
      }
    }
  }
  CMD_In_Progress = TURNOFF;                /* Clear CMD in Progress flag.   */
  if (Enabler == GEN_Enable) {              /* If we disabled keyboard       */
    Enable_Keyboard();                      /*  then reenable it.            */
  }
  else {
    DisableFlag = TURNOFF;
    Disable_Keyboard();                     /* Make sure kbd is disabled.    */
  }
  if (Next_Block != Next_Run) {             /* Check if blocked thread.      */
    Run_Thread();                           /* If yes then run it.           */
  }
  return(0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Query_Kbd_Rdy                                            */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* RETURN-NORMAL  : AX=0  : The keyboard is in the process of a command.     */
/*                  AX=-1 : The keyboard is ready for a new command.         */
/*                                                                           */
/*****************************************************************************/

USHORT Query_Kbd_Rdy(void)
{
  return(CMD_In_Progress ? 0 : -1);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Flush_Partial                                            */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Resets the state machine.                                */
/*                                                                           */
/*****************************************************************************/

USHORT Flush_Partial(void)
{
  Current_State = S_NOCMDIPG;               /* Reset the state machine.      */
  return(0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Save_Satae                                               */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Saves the current state of the keyboard hardware.        */
/*                                                                           */
/*****************************************************************************/

USHORT Save_State(void)
{
  Save_LED  = LED_State;                    /* Save the LED state.           */
  Save_Type = Type_Rate;                    /* Save type rate.               */
  return(0);
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Restore_State                                            */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Restores saved state of keyboard.                        */
/*                                                                           */
/*****************************************************************************/

USHORT Restore_State(void)
{
  Set_Typematic(Save_Type);                 /* Restore type rate.            */
  Set_LEDs(Save_LED);                       /* Restore LED state.            */
  return(0);
}
/*****************************************************************************/
/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : IDC_Entry_Point                                          */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Entry point for other device drivers, especially the     */
/*                  device independent keyboard.  Routes function to         */
/*                  correct procedure.                                       */
/*                                                                           */
/* INPUT          : Function and possibility of two other variables.         */
/*                                                                           */
/* RETURN-NORMAL  : AX=0 or value.                                           */
/*                                                                           */
/* RETURN-ERROR   : AX = -1.                                                 */
/*                                                                           */
/*****************************************************************************/

USHORT FAR _loadds IDC_Entry_Point(USHORT Req, USHORT Req_Off, USHORT Req_Seg)
{
  USHORT Ret_Req = 0;
  BYTE   Set_INFO = (BYTE)Req_Off;
  BYTE   IntCheck = 0;

  if (InterruptTime == TURNOFF) {           /* If not currently int time.    */
    InterruptTime = (BYTE)(Req & 0x8000 ? TURNON : TURNOFF);
    IntCheck = InterruptTime;               /* Flag that we turned on int.   */
  }
  switch(Req & 0x7FFF) {                    /* Case which function requested.*/
    case Q_Open     : Ret_Req = Query_Open();                         break;
    case Q_Close    : Ret_Req = Query_Close();                        break;
    case Q_Cap      : Ret_Req = Query_Capabilites(Req_Off,Req_Seg);   break;
    case Q_Type     : Ret_Req = Query_Typematic();                    break;
    case Q_LEDs     : Ret_Req = Query_LEDs();                         break;
    case Q_ID       : Ret_Req = Query_ID(Req_Off,Req_Seg);            break;
    case Q_Disabled : Ret_Req = Query_Disabled();                     break;
    case Disable_K  : Ret_Req = Disable_Keyboard();                   break;
    case Enable_K   : Ret_Req = Enable_Keyboard();                    break;
    case Reset_Hw   : Ret_Req = Reset_Hardware();                     break;
    case S_Type     : Ret_Req = Set_Typematic(Set_INFO);              break;
    case S_LED      : Ret_Req = Set_LEDs(Set_INFO);                   break;
    case Send_Gen   : Ret_Req = Send_Generic(Req_Off,Req_Seg);        break;
    case Q_Kbd_Rdy  : Ret_Req = Query_Kbd_Rdy();                      break;
    case Flush_Pt   : Ret_Req = Flush_Partial();                      break;
    case Save_St    : Ret_Req = Save_State();                         break;
    case Restore_St : Ret_Req = Restore_State();                      break;
    default         : Ret_Req = -1;                                   break;
  }
  if (IntCheck) {                           /* If we turned on the int flag. */
    InterruptTime = TURNOFF;                /*  then turn it back off.       */
  }
  if (KbdError) {
    Ret_Req = -1;                           /* Return error to caller.       */
    KbdError = TURNOFF;
  }
  return(Ret_Req);                          /* Return value to requester.    */
}

/*****************************************************************************/
/*                                                                           */
/* FUNCTION NAME  : Strategy                                                 */
/*                 "IBM Unique Code"                                         */
/*                                                                           */
/* DESCRIPTION    : Strategy routine entry point.                            */
/*                                                                           */
/* NOTES          : Currently only initialization is supported through the   */
/*                  Stategy routine.                                         */
/*                                                                           */
/* INPUT          : Initialization Packet from Kernel                        */
/*                                                                           */
/* RETURN-NORMAL  : Timer interrupt routine will occur in two clock cycles.  */
/*                                                                           */
/* RETURN-ERROR   : Do not install code.                                     */
/*                                                                           */
/* INTERNAL REFERENCES:  Init()                                              */
/*                                                                           */
/*****************************************************************************/

void FAR Strategy(void)
{
  _asm {
     mov word ptr pRPH[0], bx   /* pRPH (Request Packet Header) initialized. */
     mov word ptr pRPH[2], es   /* To ES:BX passed from the kernel.          */
  }
  switch (pRPH->Cmd) {
    case CMDInitBase:                                 /* 0x00                */
      Init();
      STI;
      break;                    /* Init is the only one supported right now. */
    case CMDDeInstall:
    case CMDOpen:                                     /* 0x0d                */
    case CMDClose:                                    /* 0x0e                */
      pRPH->Status = STATUS_DONE;
      break;
    case CMDGenIOCTL:                                 /* 0x10                */
    default:
      pRPH->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD;
      break;                                          /* end switch          */
  }
}



