/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME =  RMHELP.C
 *
 * DESCRIPTIVE NAME =  Helpers to interface RM subsystem to AD1848 code.
 *
 *
 * VERSION = V0.1
 *
 * DATE  09/02/94
 *
 * DESCRIPTION : (see above)
 *
 * Purpose: (see above)
 *
 *
 *
 * FUNCTIONS  :
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 *
 *
 * EXTERNAL FUNCTIONS
 *
*/
#define INCL_NOPMAPI
#define INCL_DOSMISC                    /* DosGetMsg and DosPutMsg */
#include <os2.h>
#include <os2medef.h>
#include <ssm.h>
#include <audio.h>

#define DRV_16
#include "os2mixer.h"

#include "mvprodd.h"
#include "cdevhlp.h"
#include "pasdef.h"
#include "findpas.h"
#include "proto.h"
#include "rmhelp.h"  // brings in rmcalls.h
#include "types.h"   // brings in WORD

/*--------------------------------------------------------*/
/* # D E F I N E S                                        */
/*--------------------------------------------------------*/

typedef struct _rmportdesc
{
      HRESOURCE   hResPort;
      USHORT      baseport;
      USHORT      RangeOffset;
}RMPORTDESC, *PRMPORTDESC;


#define MAX_RM_ERR_COUNT   20

typedef struct _rmerrors
{
      USHORT restype;
      USHORT resval;
      USHORT rc;
}RMERRORS, *PRMERRORS;

/*--------------------------------------------------------*/
/* E X T E R N S                                          */
/*--------------------------------------------------------*/
extern WORD FPwTranslateCode; // findpas.c global that gets xlate code

/*--------------------------------------------------------*/
/* Set the DRIVERSTRUCT and ADAPTERSTRUCT data areas      */
/*--------------------------------------------------------*/

/*----------------------------------------------*/
/* Driver Description   (rmbase.h)              */
/*----------------------------------------------*/
DRIVERSTRUCT DriverStruct =
{
   NULL,                                     /* DrvrName                */
   "Pro Audio 16 Spectrum Sound Card Driver", /* DrvrDescript            */
   "Pro Audio Spectrum",                     /* VendorName              */
   CMVERSION_MAJOR,                          /* MajorVer                */
   CMVERSION_MINOR,                          /* MinorVer                */
   1994,9,02,                                /* Date                    */
   DRF_STATIC,                               /* DrvrFlags               */
   DRT_AUDIO,                                /* DrvrType                */
   0,                                        /* DrvrSubType             */
   NULL                                      /* DrvrCallback            */
};

/*----------------------------------------------*/
/* Adapter Description  (rmbase.h)              */
/*----------------------------------------------*/
ADAPTERSTRUCT AdapterStruct =
{
  "Pro Audio 16 Sound Card",         /* AdaptDescriptName; */
  AS_16MB_ADDRESS_LIMIT,             /* AdaptFlags;        */
  AS_BASE_MMEDIA,                    /* BaseType;          */
  AS_SUB_MM_AUDIO,                   /* SubType;           */
  AS_INTF_GENERIC,                   /* InterfaceType;     */
  AS_HOSTBUS_ISA,                    /* HostBusType;       */
  AS_BUSWIDTH_16BIT,                 /* HostBusWidth;      */
  NULL                               /* pAdjunctList;      */
};

/*----------------------------------------------*/
/* Device Description  (rmbase.h)               */
/*----------------------------------------------*/
DEVICESTRUCT DeviceStruct =
{
  "PAS16 AUDIO Device",             /* DevDescriptName   */
  DS_FIXED_LOGICALNAME,             /* DevFlags;         */
  DS_TYPE_AUDIO,                    /* DevType;          */
  NULL                              /* pAdjunctList;     */
};

/*----------------------------------------------*/
/* GLOBAL VARS FOR RM                           */
/*                                              */
/* RM.LIB needs these declared                  */
/*----------------------------------------------*/
PFN             RM_Help               = 0L;  /*VPNP*/
ULONG           RMFlags               = 0L;  /*VPNP*/
PFN             RM_Help0              = 0L;  /*VPNP*/
PFN             RM_Help3              = 0L;  /*VPNP*/

/*----------------------------------------------*/
/* GLOBAL HANDLE VARIABLES                      */
/*                                              */
/* These variables get the handles for          */
/* drivers, adapter (only 1), and resources.    */
/* The RMHELP wrappers are written so that most */
/* drivers using RMHELP as a template will not  */
/* have to reference these directly.            */
/*                                              */
/*----------------------------------------------*/

HDRIVER   hDriver = NULL;     // global handle to driver
HADAPTER  hAdapter = NULL;    // global handle to adapter

USHORT ResPortsCnt = 0;       // Count of res port handles
USHORT PortsMarker = 0;       // Gets set to ResPortsCnt by SetPortsMarker
RMPORTDESC rmPorts[ MAX_PORT_HANDLES];    // global handles to port resources
                                          // and the port range itself

HRESOURCE hResIRQ = NULL;    // global handle to IRQs resource
HRESOURCE hResDMA = NULL;    // global handle to DMA resource

BOOL bLastAllocOverlapped;    // Flag that says if port request overlapped
                              // previous granted request


USHORT RMErrCnt = 0;                    // Count of RM errors
RMERRORS RMErrors[ MAX_RM_ERR_COUNT];   // array to hold RM ERRORs

/*--------------------------------------------------------*/
/* RMHELP_CreateDriver - registers to RM susbsystem       */
/*                       (which initializes the DD to     */
/*                        RM connection)                  */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* PSZ DrvrName - ptr to driver name (taken from command  */
/*                line)                                   */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* This function is a thin wrapper for the RMCreateDriver */
/* call. Modifications for specific drivers might include */
/* adding more parameters to be set into the              */
/* DriverStruct and actually returning the hDriver        */
/* to the caller (instead of maintaining it in a global   */
/* var). In any case, modifications to the DriverStruct   */
/* decalred above will have to made.                      */
/*--------------------------------------------------------*/
USHORT RMHELP_CreateDriver( PSZ DrvrName)
{
   USHORT rc;

   DriverStruct.DrvrName = DrvrName;   // set the driver name

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   if( (rc = RMCreateDriver( &DriverStruct, &hDriver)) )
   {
      RMHELP_PutMessage( "CreateDriver", 0, 0, rc);
   }

   return( rc);
}

/*--------------------------------------------------------*/
/* RMHELP_DestroyDriver - deregisters the driver from     */
/*                        RM sub-system and releases all  */
/*                        claimed resources               */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_DestroyDriver( VOID)
{
   RMDestroyDriver( hDriver);
}

/*--------------------------------------------------------*/
/* RMHELP_CreateAdapter - tells RM to associate adapter   */
/*                        to driver and resources to      */
/*                        adapater                        */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* This function builds up the AHRESOURCE array for the   */
/* the call to RMCreateAdapter. To modify this for a      */
/* specfic DD the handles to resource variables set into  */
/* the array will have to be touched along with the lines */
/* that set the NumResource. Also, AdapterStruct (see     */
/* above) will have to be set up apropriately.            */
/*                                                        */
/*--------------------------------------------------------*/
USHORT RMHELP_CreateAdapter( VOID)
{
   UCHAR ResourceBuf[ MAX_RES_BUF_SIZE];
   PAHRESOURCE pResourceList = (PAHRESOURCE)ResourceBuf;
   USHORT ResNdx = 0;
   USHORT i;
   USHORT rc;
   HDEVICE hDevice;

   for ( i = 0; i < ResPortsCnt; i++)
   {
      pResourceList->hResource[ ResNdx] = rmPorts[ i].hResPort;
      ++ResNdx;
   }

   if ( hResIRQ )
   {
      pResourceList->hResource[ ResNdx] = hResIRQ;
      ++ResNdx;
   }

   if ( hResDMA )
   {
      pResourceList->hResource[ ResNdx] = hResDMA;
      ++ResNdx;
   }

   pResourceList->NumResource = ResNdx;

   if ( (rc = RMCreateAdapter( hDriver,
                               &hAdapter,
                               &AdapterStruct,
                               NULL,
                               pResourceList)) != RMRC_SUCCESS )
   {
      RMHELP_PutMessage( "CreateAdapter", 0, 0, rc);
      return( rc);
   }

   if ( (rc = RMCreateDevice( hDriver,
                              &hDevice,
                              &DeviceStruct,
                              hAdapter,
                              NULL))         )
   {
      RMHELP_PutMessage( "CreateDevice", 0, 0, rc);
   }

   return( rc);
}

/*--------------------------------------------------------*/
/* RMHELP_GrabPort - special for PAS16, this call         */
/*                   xlates and allocs 1-port range       */
/*                   EXCLUSIVE and returns status.        */
/* ARGS:                                                  */
/*                                                        */
/* USHORT p - un-translated port                          */
/*--------------------------------------------------------*/
USHORT RMHELP_GrabPort( USHORT p)
{
   USHORT ShareType = RS_IO_EXCLUSIVE;

   // Special case to get along with TMV1SCSI.add
   // Ports in the 388 range need to be RS_IO_SHARED

   if (p >= 0x388 && p <= 0x38B)
      {
      ShareType = RS_IO_SHARED;
      }

   return( RMHELP_AllocPorts( p ^ FPwTranslateCode, 1, ShareType));
}

/*--------------------------------------------------------*/
/* RMHELP_AllocPorts - attempts to claim a port resource  */
/*                     from RM subsystem. If port range   */
/*                     requested has already been claimed */
/*                     by PAS16 then this function simply */
/*                     returns RMRC_SUCCESS               */
/* ARGS:                                                  */
/*                                                        */
/* USHORT baseport - base IO port.                        */
/* USHORT nPorts   - number of prts inclusive of base port*/
/* USHORT ShareType - RS_IO_EXCLUSIVE or RS_IO_SHARED     */
/*                                                        */
/*--------------------------------------------------------*/
USHORT RMHELP_AllocPorts( USHORT baseport, USHORT nPorts, USHORT ShareType)
{
   RESOURCESTRUCT Resource;
   USHORT rc;
   USHORT i;
   USHORT RangeOffset;
   USHORT over;

   //*******************************************************************
   // FIRST, CHECK TO SEE IF PAS16 DRIVER HAS ALREADY ALLOCATED A PORT
   // RANGE THAT OVERLAPS INCLUSIVE THE REQUESTED PORT RANGE, IF SO JUST
   // RETURN SUCCESS.
   // IF ONLY PART OF THE REQUESTED RANGE OVERLAPS ADJUST baseport & nPorts
   // REQUEST TO INCLUDE ONLY PORTS THAT HAVE NOT BEEN PREVIOUSLY ALLOCATED
   //*******************************************************************

   bLastAllocOverlapped = FALSE;    // INIT FLAG

   if ( ResPortsCnt)
   {
      RangeOffset = nPorts-1;

      for ( i = 0; i < ResPortsCnt; i++)
      {
         // EXACT MATCH
         if ( (baseport == rmPorts[ i].baseport) &&
              (RangeOffset == rmPorts[ i].RangeOffset) )
         {
            return( RMRC_SUCCESS);
         }

         if ( baseport > (rmPorts[ i].baseport + rmPorts[ i].RangeOffset) )
         {
            continue;
         }

         if ( (baseport + RangeOffset) < rmPorts[ i].baseport )
         {
            continue;
         }

         // OVERLAP, THIS IS THE HARD PART
         if ( baseport >= rmPorts[ i].baseport )
         {
            over = rmPorts[ i].baseport + rmPorts[ i].RangeOffset -
                   baseport + 1;

            if ( RangeOffset < over )
            {
               // ENTIRELY IN RANGE
               bLastAllocOverlapped = TRUE;
               return( RMRC_SUCCESS);
            }

            // ADJUST BASEPORT & RANGEOFFSET AND TRY AGAIN
            baseport += over;
            RangeOffset -= over;

            i = 0;   // START OVER WITH NEW BASEPORT & RANGE
            continue;
         }
         else // OVERLAP IS THAT RANGE REACHES INTO rmPorts.baseport RANGE
         {
            // ADJUST RANGEOFFSET AND TRY AGAIN
            RangeOffset = rmPorts[ i].baseport - baseport - 1;
            i = 0;
            continue;
         }
      }

      nPorts = RangeOffset + 1;
   }

   if ( ResPortsCnt >= MAX_PORT_HANDLES)
   {
      return( RMHELP_ERR_TOOMANYRESPORTS);
   }

   Resource.ResourceType              = RS_TYPE_IO;
   Resource.IOResource.BaseIOPort     = baseport;
   Resource.IOResource.NumIOPorts     = nPorts;
   Resource.IOResource.IOAddressLines = DECODE_WIDTH;
   Resource.IOResource.IOFlags        = ShareType;

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   rc = RMAllocResource( hDriver,
                         &rmPorts[ ResPortsCnt].hResPort,
                         &Resource);

   if ( rc == RMRC_SUCCESS )
   {
      rmPorts[ ResPortsCnt].baseport = baseport;
      rmPorts[ ResPortsCnt].RangeOffset = nPorts - 1;
      ++ResPortsCnt;
   }
   else
   {
      RMHELP_PostRMError( RS_TYPE_IO, baseport, rc);
   }

   return( rc);

}

/*--------------------------------------------------------*/
/* RMHELP_DeAllocLastPort - deallocates the last port     */
/*                       acquired from RMHELP_AllocPorts  */
/*                                                        */
/* NOTE: Alloc/Dealloc calls cannot be nested if the      */
/*       Alloc overlaps, i.e., This may not work:         */
/*                                                        */
/*       RMCALLS_AllocPort                                */
/*             RMCALLS_AllocPort                          */
/*             RMCALLS_DeallocLatsPort                    */
/*       RMCALLS_DeallocLatsPort                          */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* VOID                                                   */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_DeAllocLastPort( VOID)
{
   // IF LAST ALLOC REQUEST OVERLAPPED COMPLETEY A PREVIOUS PORT RANGE
   // ALLOC THEN WE DON'T WANT TO DEALLOC A PORT RANGE
   // !!! NOTE THAT WE CANNOT NEST ALLOC/DEALLOC PAIRS

   if ( bLastAllocOverlapped)
   {
      return;
   }

   if ( ResPortsCnt)
   {
      --ResPortsCnt;
      RMDeallocResource( hDriver, rmPorts[ ResPortsCnt].hResPort);
   }
}

/*--------------------------------------------------------*/
/* RMHELP_SetPortsMarker - saves the ResPortsCount so     */
/*                   that a subsequent call to            */
/*                   RMHELP_DeAllocPortsToMarker will     */
/*                   cause all port ranges alloc'd after  */
/*                   SetPortsMarker will be released      */
/*                   NOTE: CANNOT BE NESTED.              */
/*--------------------------------------------------------*/
VOID RMHELP_SetPortsMarker( VOID)
{
   PortsMarker = ResPortsCnt;
}

/*--------------------------------------------------------*/
/* RMHELP_DeAllocsPortsToMarker                           */
/*                   Releases all port ranges alloc's     */
/*                   until ResPortCnt = PortsMarker.      */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_DeAllocPortsToMarker( VOID)
{

   bLastAllocOverlapped = FALSE; // RESET THE OVERLAP FLAG

   while( ResPortsCnt > PortsMarker)
   {
      RMHELP_DeAllocLastPort();
   }
}

/*--------------------------------------------------------*/
/* RMHELP_AllocIRQ   - attempts to claim a IRQ resource   */
/*                       from RM subsystem                */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* USHORT IRQlevel - IRQ level to claim                   */
/*                                                        */
/*--------------------------------------------------------*/
USHORT RMHELP_AllocIRQ( USHORT IRQlevel)
{
   RESOURCESTRUCT Resource;
   USHORT rc;

   /*------------------------------------------------*/
   /* Request IRQ Resource  from RM......            */
   /*------------------------------------------------*/
   Resource.ResourceType          = RS_TYPE_IRQ;
   Resource.IRQResource.PCIIrqPin = RS_PCI_INT_NONE;
   Resource.IRQResource.IRQFlags  = RS_IRQ_MULTIPLEXED;

   Resource.IRQResource.IRQLevel = IRQlevel;

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   if ( (rc = RMAllocResource( hDriver, &hResIRQ, &Resource)) )
   {
      RMHELP_PostRMError( RS_TYPE_IRQ, IRQlevel, rc);
   }

   return( rc);
}


/*--------------------------------------------------------*/
/* RMHELP_AllocDMA   - attempts to claim a DMA resource   */
/*                       from RM subsystem                */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* USHORT DMAchannel   - DMA channel to claim             */
/*                                                        */
/*--------------------------------------------------------*/
USHORT RMHELP_AllocDMA( USHORT DMAChannel)
{
   USHORT rc;
   RESOURCESTRUCT Resource;

   /*------------------------------------------------*/
   /* Request DMA Resource  from RM......            */
   /*------------------------------------------------*/

   Resource.ResourceType         = RS_TYPE_DMA;
   Resource.DMAResource.DMAChannel = DMAChannel;
   Resource.DMAResource.DMAFlags = RS_DMA_EXCLUSIVE;

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   if ( (rc = RMAllocResource( hDriver, &hResDMA, &Resource)) )
   {
      RMHELP_PostRMError( RS_TYPE_DMA, DMAChannel, rc);
   }

   return( rc);
}

/*--------------------------------------------------------*/
/* RMHELP_DeAllocIRQ - deallocs a IRQ resource            */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_DeAllocIRQ( VOID)
{
   RMDeallocResource( hDriver, hResIRQ);
   hResIRQ = 0;
}

/*--------------------------------------------------------*/
/* RMHELP_DeAllocDMA - deallocs a DMA resource            */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_DeAllocDMA( VOID)
{
   RMDeallocResource( hDriver, hResDMA);
   hResDMA = 0;
}

/*--------------------------------------------------------*/
/* RMHELP_PostRMError - if an error is returned by RM     */
/*                      for any resource allocs then      */
/*                      this stores error and resource    */
/*                      requested into an array.          */
/* ARGS:                                                  */
/*                                                        */
/* USHORT restype  - (RS_TYPE_*  from rmbase.h)           */
/* USHORT resval   - base 'value' of res (port, IRQ level)*/
/* USHORT rc       - return code from RMAlloc call        */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_PostRMError( USHORT restype, USHORT resval, USHORT rc)
{
   if ( RMErrCnt < MAX_RM_ERR_COUNT)
   {
         RMErrors[ RMErrCnt].restype = restype;
         RMErrors[ RMErrCnt].resval = resval;
         RMErrors[ RMErrCnt].rc     = rc;

        ++RMErrCnt;
   }
}

/*--------------------------------------------------------*/
/* RMHELP_DumpRMErrors - displays all errors logged with  */
/*                       PostRMError to stdout            */
/*--------------------------------------------------------*/
VOID RMHELP_DumpRMErrors( VOID)
{
   PSZ    pszRes;
   USHORT fieldwidth;

   while( RMErrCnt)
   {
      --RMErrCnt;          // TURNS COUNT INTO ARRAY INDEX

      fieldwidth = 2;      // ASSUME RES VAL CAN BE DISPLAYED WITH 2 DIGITS

      switch( RMErrors[ RMErrCnt].restype)
      {

         case RS_TYPE_IO:
         pszRes = "PORT";
         fieldwidth = 4;
         break;

         case RS_TYPE_IRQ:
         pszRes = "IRQ";
         break;

         case RS_TYPE_DMA:
         pszRes = "DMA";
         break;

         default:
         continue;
      };

      RMHELP_PutMessage( pszRes,
                         RMErrors[ RMErrCnt].resval,
                         fieldwidth,
                         RMErrors[ RMErrCnt].rc);
   }
}


/*--------------------------------------------------------*/
/* RMHELP_PutMessage - displays a message to stdout       */
/*                     converts binary to hex ascii       */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* PSZ pszResource - string describing resource           */
/* USHORT value    - binary value representing resource   */
/* USHORT fieldwidth - max chars in conversion result     */
/* USHORT rc - err code returned by RM call               */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_PutMessage( PSZ pszResource,
                        USHORT value,
                        USHORT fieldwidth,
                        USHORT rc)
{
      SHORT  i;
      USHORT digit;
      UCHAR  cvt2ascii[ 6];
      UCHAR  rm_msg[ 55];   // !!! BE CAREFUL THAT THIS IS LARGE ENOUGH


      if ( fieldwidth )    // 0 FIELDWIDTH INDICATES DONT DISPLAY VALUE
      {
         // CONVERT RESOURCE 'VALUE' TO HEX ASCII

         if( fieldwidth > 4 )
         {
            fieldwidth = 4;
         }

         cvt2ascii[ fieldwidth] = 0;

         for ( i = ((SHORT)fieldwidth-1); i >= 0; i--)
         {
            digit = value & 0x000F; // MASK OFF DIGIT

            // CONVERT DIGIT TO HEX ASCII
            cvt2ascii[ i] = (UCHAR)((digit < 10) ? (digit + '0') : (digit + '7'));

            value >>= 4;            // MOVE NEXT DIGIT TO LOW NIBBLE
         }
      }

      // BUILD STRING

      lstrcpy( (LPSTR)rm_msg, (LPSTR)"\r\n** RM:");
      lstrcat( (LPSTR)rm_msg, (LPSTR)pszResource);

      if ( fieldwidth )    // 0 FIELDWIDTH INDICATES DONT DISPLAY VALUE
      {
         lstrcat( (LPSTR)rm_msg, (LPSTR)" ");
         lstrcat( (LPSTR)rm_msg, (LPSTR)cvt2ascii);
      }

      lstrcat( (LPSTR)rm_msg, (LPSTR)", rc = ");


      // CONVERT RC TO DECIMAL ASCII (ALLOW FOR FIVE DIGITS)

      cvt2ascii[ 5] = 0;   // SET STRING TERMINATOR

      for ( i = 4; i >= 0; i--)
      {
         if ( value)
         {
            digit = value % 10;
            cvt2ascii[ i] = (UCHAR)(digit + '0');
            value /= 10;
         }
         else
         {
            cvt2ascii[ i] = ' ';
         }
      }

      lstrcat( (LPSTR)rm_msg, (LPSTR)cvt2ascii);      // APPEND RC VALUE AS DECIMAL ASCII
      lstrcat( (LPSTR)rm_msg, (LPSTR)" **\r\n");      // APPEND END OF MESSAGE STRING

      DosPutMessage( 1, lstrlen( (LPSTR)rm_msg), rm_msg); // DISPLAY STRING
}
