/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************
*                 Pro AudioSpectrum16 Physical Device Driver
*                     Production code and toolkit sample
*
*
* DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
* sample code created by IBM Corporation and Media Vision Corporation.
* It is provided to you solely for the purpose of assisting you in the
* development of your applications.
* The code is provided "AS IS", without warranty of any kind.
* IBM and Media Vision shall not be liable for any damages arising out of
* your use of the sample code, even if they have been advised of the
* possibility of such damages.
*
*******************************************************************************
*
* mmddcmds.c - Multimedia Device Driver Commands - Stream handler communicaton
*
* DESCRIPTION:
* Audio device driver IDC routines for MMPM/2
* Process IOCTLS received from the Amp Mixer and stream data to and
* from the stream handler.
* The Amp Mixer sent IOCTLS arrive via the device driver strategy routine.
* The stream handler communication is via InterDevice Communication (IDC).
*
*/

#define  INCL_NOPMAPI
#define  INCL_BASE
#define  INCL_DOS
#define  INCL_DOSDEVICES
#define  INCL_ERRORS
#include <os2.h>

#include <os2medef.h>
#include <meerror.h>     // Multimedia error return codes
#include <ssm.h>         // Sync Stream Manager
#include <shdd.h>        // IDC interface
#include <audio.h>

#define  DRV_16
#include "os2mixer.h"
#include "pasdef.h"
#include "findpas.h"
#include "mvprodd.h"
#include "patch.h"
#include "globals.h"
#include "cdevhlp.h"
#include "proto.h"
#include "debug.h"
#include "commdbg.h"



/*
** Function prototypes for IDC routines - called from stream handler
** The IDC call from the stream hanlder lands briefly in assembler
** code (startup.asm) where it is routed to the "internal" entry point
** defined below.
** Based on the function, the internal entrypoint routine routes the
** call the the appropriate DDCmdxxx routine.
*/

VOID ShdReportInt (ULONG);
RC FAR DDCMDInternalEntryPoint (PDDCMDCOMMON     pCommon);
RC     DDCmdSetup              (PDDCMDSETUP      pSetup);
RC     DDCmdRead               (PDDCMDREADWRITE  pRead);
RC     DDCmdWrite              (PDDCMDREADWRITE  pWrite);
RC     DDCmdStatus             (PDDCMDSTATUS     pStatus);
RC     DDCmdControl            (PDDCMDCONTROL    pControl);
RC     DDCmdRegister           (PDDCMDREGISTER   pRegister);
RC     DDCmdDeRegister         (PDDCMDDEREGISTER pDeRegister);
RC     GetStreamEntry          (PSTREAM far      *ppStream, HSTREAM hStream);


/*
** EXTERNS
*/

extern ULONG dwDMAPhysAddr;
extern PROTOCOLTABLE ProtocolTable8[];
extern PROTOCOLTABLE ProtocolTable16[];

/*
** Locally defined data.  Note this data is always
** accessible to the device driver.
** The function table defined below is referenced at IDC
** entry to route the incoming call to the appropriate
** function via indirect call.
*/

ULONG (*DDCMDFuncs[])(PVOID pCommon) =
      {
      DDCmdSetup,       // 0
      DDCmdRead,        // 1
      DDCmdWrite,       // 2
      DDCmdStatus,      // 3
      DDCmdControl,     // 4
      DDCmdRegister,    // 5
      DDCmdDeRegister   // 6
      };

USHORT MaxDDCMDFuncs = sizeof(DDCMDFuncs)/sizeof(USHORT);
USHORT usPauseDMAAddr=0;


/*
** C O D E
*/


/*                                      ------------- DDCMDInternalEntryPoint -
** This routine is the IDC entry point as called from the MMPM/2
** stream handler.
** (execution arrives here after parameter massage in assembler code)
**
** This function is entered every time the stream handler calls
** this device driver.  It is the responsibility of this routine to
** route the incoming message to an appropriate function.
** In this respect, this function behaves much like a PDD strategy routine.
**
** INPUT:        Pointer to command-dependent parameter structure.
** EXIT-NORMAL:  NO_ERROR
** EXIT_ERROR:   ERROR_INVALID_FUNCTION
**               ERROR_INVALID_BLOCK
** EFFECTS:
** INTERNAL REFERENCES: DDCMD functions
** EXTERNAL REFERENCES: None
*/
RC far DDCMDInternalEntryPoint (PDDCMDCOMMON pCommon)
{
   if (pCommon==NULL)
      return (ERROR_INVALID_BLOCK);

   /*
   ** Verify function code does not go beyond the
   ** limits of our table.  That is, insure function
   ** code is valid.
   */
   if (pCommon->ulFunction > (ULONG)MaxDDCMDFuncs)
      return (ERROR_INVALID_FUNCTION);

   /*
   ** Call appropriate function based on command code.
   ** Return the RC of the called function as this functions RC.
   */
   return (DDCMDFuncs[pCommon->ulFunction](pCommon));
}


/*                                      -------------------------- DDCmdSetup -
** The stream handler is calling the PDD asking it to prepare to
** either consume or produce data.
** Preparing a stream implies that a stream already exists.  If
** the stream handle the stream handler passes doesn't exist,
** then return an error to the stream handler.
**
** Likewise, the Amp Mixer and audio (or midi) MCD must have already
** opened the device and sent down an "init" packet.
** If the operation commanded from the stream handler contradicts
** commands already received from the ring-3 modules, tell the
** stream handler that someone is confused (return error)
**
** Assuming the stream setup is doable based on the current state,
** do what the stream handler asked and return success.
** Note that no real action takes place in this function.
** The device driver changes state for this stream only and prepares
** for later messages that will actually command data consumption/production.
**
** EXIT-NORMAL:  NO_ERROR
** EXIT_ERROR:   ERROR_INVALID_STREAM
**               ERROR_INVALID_REQUEST
**               ERROR_INVALID_FUNCTION
**
** INPUTS:       ulFunction              DDCMD_SETUP
**               hStream                 Stream handle to set up
**               pSetupParm              Not used
**               ulSetupParmSize         Not used
*/
RC DDCmdSetup (PDDCMDSETUP pSetup)
{
   PSTREAM pStream;    // pointer to Stream table
   RC      rc;

   #ifdef   PROGRESS_MONITOR
   StringOut ("DDCmdSetup");
   #endif

   /*
   ** Search stream table for matching stream handle
   */
   pStream = GlobalTable.paStream;
   if (rc = GetStreamEntry(&pStream, pSetup->hStream))
      {
      #ifdef DEBUG_CHK
        StringOut( "DDCmdSetup: can't find Stream Entry");
      #endif
        return(rc);       // can't find stream in table
      }

   /*
   ** Ensure that this stream operation matches the
   ** operation that the PDD is set to perform.
   ** Put another way.  If we were setup for an operation
   ** different than that which the stream handler is
   ** asking us to perform, then we are not prepared.
   ** In this case, return error to stream handler.
   **
   ** For example, if setting up to start a stream, make sure the
   ** stream we are about to start is the same stream that the
   ** amp mixer most recently inited.
   */
   if ((pStream->ulSysFileNum != ulActiveSysFileNum.pcm) &&
       (pStream->ulSysFileNum != ulActiveSysFileNum.midi) &&
       (pStream->ulSysFileNum != ulActiveSysFileNum.synth))
      {
      #ifdef DEBUG_CHK
      StringOut("DDCmdSetup: Invalid sysfilenum");
      #endif
      return(ERROR_INVALID_REQUEST);
      }

   switch (operation.pcm)
      {
      case OPERATION_PLAY:
         if (pStream->ulOperation != STREAM_OPERATION_CONSUME)
            {
            #ifdef DEBUG_CHK
            StringOut("DDCmdSetup: Not Consume");
            #endif
            return (ERROR_INVALID_REQUEST);
            }
         break;

      case OPERATION_RECORD:
         if (pStream->ulOperation != STREAM_OPERATION_PRODUCE)
            {
            #ifdef DEBUG_CHK
            StringOut("DDCmdSetup: Not Produce");
            return (ERROR_INVALID_REQUEST) ;
            #endif
            }
         break;

      case PLAY_AND_RECORD:
         return (ERROR_INVALID_FUNCTION) ;       /* can't */
         break;


      default:
         #ifdef DEBUG_CHK
         StringOut( "DDCmdSetup: Invalid Function");
         #endif
         return (ERROR_INVALID_FUNCTION) ;       /* Stream not initialized */
      } // end switch
   StringOut( "DDCmdSetup6");

   #ifdef DEBUG_CHK
   StringOut("DDCmdSetup: Setting Stream Time");
   #endif

   /*
   ** Stream handler needs to adjust CumTime when doing a setup
   */
   SetStreamTime (pStream, *((unsigned long far *)pSetup->pSetupParm));

   #ifdef PROGRESS_MONITOR
   StringOut("DDCmdSetup: Exit OK");
   #endif

   /*
   ** Make sure a stopped stream is "CLEAN" before
   ** starting again.
   */
   if (pStream->ulFlags & STREAM_STOPPED)
      {
      iEmptyPCMBuffers           = 2;
      iNextBuffer                = 0;
      pStream->ulPCMBufferCount  = 0;
      pStream->usCurrIOBuffIndex = 0;
      pStream->usNextIOBuffIndex = 0;
      pStream->usDoneIOBuffIndex = 0;
      }

   return (NO_ERROR);
}



/*                                      --------------------------- DDCmdRead -
** Read from the audio device (Recording Mode)
**
** Allows the Audio Stream Handler to read buffers from the audio device.
**
** This routine chains the incoming buffers from the stream
** handler into Iobuf packets and updates the next index buffer
** packet.  The current index is only updated during
** interrupt time in the get_next_rbuf routine.
**
** This routine is called at interrupt and non-interrupt time.
**
** INPUT:        Parm pointer
** EXIT-NORMAL:  NO_ERROR
** EXIT_ERROR:   ERROR_INVALID_STREAM
**               ERROR_INVALID_BLOCK
**               ERROR_STREAM_NOT_ACTIVE
** INPUTS:       ulFunction              DDCMD_READ
**               hStream                 Stream handle
**               pBuffer                 Address of buffer to record data to
**               ulBufferSize            Size of pBuffer
*/
RC DDCmdRead (PDDCMDREADWRITE pRead)
{
   LPMCI_AUDIO_IOBUFFER lpNextBuff;
   USHORT               Current, Next, Previous;
   PSTREAM              pStream;
   RC                   rc;

   #ifdef PROGRESS_MONITOR
   PrintfOut( "DDCmdRead: ulBufferSize=%0ld", pRead->ulBufferSize);
   #endif

   /*
   ** Stream handler should never send a NULL pointer.  If it did,
   ** something is strange.
   */
   if (pRead->pBuffer==NULL)
      return (ERROR_INVALID_BLOCK);

   pStream = GlobalTable.paStream;
   if (rc = GetStreamEntry(&pStream, pRead->hStream))
      return(rc);

   #ifdef BUF_MONITOR
   PrintfOut ("DDCmdRead Entry: Done: %d   Curr: %d   Next: %d\r\n",
              pStream->usDoneIOBuffIndex,
              pStream->usCurrIOBuffIndex,
              pStream->usNextIOBuffIndex);
   #endif


   /*
   ** Set indexes that point to IOBuff packets
   */
   Current = pStream->usCurrIOBuffIndex;
   Next    = pStream->usNextIOBuffIndex;

   lpNextBuff=&pStream->IOBuff[Next];

   /*
   ** Copy buffer pointer to "next" and
   ** Fill IOBuff packet with data from pRead - size of data,
   ** address, usRunFlags, and place it in the queue.
   ** If there's a previous buffer in the queue, the
   ** new packet will be filled in with the previous
   ** buffer's values for position and delay.
   ** Otherwise these values will be set to zero.
   **
   ** pBufPhys - Physical address
   **
   ** Buffer   - Physical address to virtual
   ** Head     - beginning of buffer   (VIRTUAL)
   ** Tail     - end of buffer         (VIRTUAL)
   */

   lpNextBuff->lSize   = pRead->ulBufferSize;
   lpNextBuff->lCount  = 0;
   lpNextBuff->pBuffer = pRead->pBuffer;
   lpNextBuff->pHead   = pRead->pBuffer;
   lpNextBuff->pTail   = lpNextBuff->pHead + pRead->ulBufferSize;

   lpNextBuff->usRunFlags =
   pStream->IOBuff[Current].usRunFlags & (USHORT)~IOB_OVERRUN;

   if (Current!=Next)      /* There is a previous buffer */
      {
      if (Next == 0)
         Previous = MAXIOBUFFS - 1;
      else
         Previous = Next - 1 ;

      lpNextBuff->ulPosition = pStream->IOBuff[Previous].ulPosition
                               + pStream->IOBuff[Previous].lSize;
      lpNextBuff->lDelay = pStream->IOBuff[Previous].lDelay;
      pStream->ulMaxPosition = lpNextBuff->ulPosition + lpNextBuff->lSize;
      }
   else
      {
      lpNextBuff->ulPosition = 0;
      lpNextBuff->lDelay     = 0;
      pStream->ulMaxPosition = lpNextBuff->lSize;

      // Adjust buffer size for Software Motion Video support

      gdCurrentDMABufferSize = DMA_BUFFER_MAX;

      wDMABuffLength=(WORD) gdCurrentDMABufferSize-1;
      dwDMAUnitSize=gdCurrentDMABufferSize>>1;
      }

   /*
   ** Increment next buffer index and check for wrap
   */
   if ((++Next == MAXIOBUFFS))
      Next = 0;
   pStream->usNextIOBuffIndex = Next;

   return (NO_ERROR);
}


/*                                      -------------------------- DDCmdWrite -
** Write data to the audio device
**
** Allows the Audio Stream Handler to write buffers to
** the audio device by passing buffers (IOBuffs) to
** the physical hardware device.
**
** StreamNumber is set at the IDC entry point
** This routine chains the incoming buffers from the stream handler
** into Iobuf packets and updates the next index buffer
** packet.  The current index is only updated during
** interrupt time in the get_next_xbuf routine.
**
** This routine is called at interrupt and non-interrupt time.
**
** EXIT-NORMAL:  NO_ERROR
** EXIT_ERROR:   ERROR_INVALID_STREAM
**               ERROR_INVALID_BLOCK
** INPUTS:       ulFunction              DDCMD_WRITE
**               hStream                 Stream handle
**               pBuffer                 Address of buffer to play data from
**               ulBufferSize            Size of pBuffer
*/


RC DDCmdWrite (PDDCMDREADWRITE pWrite)
{
   ULONG   ulBufCount;
   ULONG   ulBufferSize;
   USHORT  Current, Next, Previous;
   PSTREAM pStream;
   RC      rc;
   LPMCI_AUDIO_IOBUFFER lpNextBuff;

   #ifdef PROGRESS_MONITOR
   PrintfOut ("DDCmdWrite - %lx %lx", pWrite->pBuffer, pWrite->ulBufferSize);
   #endif

   /*
   ** Validate data buffer pointer
   ** Stream handler shouldn't send a NULL address.
   ** If did, something is strange.
   */
   if (pWrite->pBuffer==NULL)
      return (ERROR_INVALID_BLOCK);

   /*
   ** Search stream table for matching stream handle
   */
   pStream = GlobalTable.paStream;
   if (rc = GetStreamEntry(&pStream, pWrite->hStream))
      return(rc);

   #ifdef BUF_MONITOR
   PrintfOut ("DDCmdWrite Entry: Done: %d   Curr: %d   Next: %d\r\n",
              pStream->usDoneIOBuffIndex,
              pStream->usCurrIOBuffIndex,
              pStream->usNextIOBuffIndex
              );
   #endif

   /*
   ** Set indices
   */
   Current = pStream->usCurrIOBuffIndex;
   Next    = pStream->usNextIOBuffIndex;

   lpNextBuff=&pStream->IOBuff[Next];

   /*
   ** Copy buffer pointer to "next"
   **
   ** pBufPhys - Physical address is sent from stream handler
   **
   ** Buffer   - pointer to start of buffer
   ** Head     - current position in buffer (VIRTUAL)
   ** Tail     - end of buffer              (VIRTUAL)
   */

   /*
   ** Fill buffer with data from pWrite - size of data,
   ** address, usRunFlags, and place it in the queue.
   ** If there's a previous buffer in the queue, the
   ** new buffer will be filled in with the previous
   ** buffer's values for ulposition and lDelay. Other-
   ** wise ulPosition will be set to zero, lDelay to 1.
   **
   ** Note: I interpret the ulPosition field to
   ** mean the position of the buffer within the stream.
   ** For example, the first byte of the second of
   ** two 16K buffers would have the position 16,384.
   */

   ulBufferSize=pWrite->ulBufferSize;
   lpNextBuff->lCount=ulBufferSize;
   lpNextBuff->lSize=ulBufferSize;

   #ifdef BUF_SIZE
   PrintfOut( "BSize: %0ld",ulBufferSize);
   #endif

   ///pStream->IOBuff[Next].pBuffer = pWrite->pBuffer;
   ///pStream->IOBuff[Next].pHead   = pStream->IOBuff[Next].pBuffer;

   #ifdef BUF_IN_TIME
      PrintfOut( "In Time: %0ld",pStream->current_time);
   #endif



   lpNextBuff->pHead      = lpNextBuff->pBuffer = pWrite->pBuffer;
   lpNextBuff->pTail      = lpNextBuff->pHead + ulBufferSize;
   lpNextBuff->usRunFlags = pStream->IOBuff[Current].usRunFlags &

                               (USHORT)~IOB_UNDERRUN;

   // who sets the first buffer's run flags?

   if (pStream->usMode==PCM)
      ulBufCount=pStream->ulPCMBufferCount;
   else if (pStream->usMode==MIDI)
      ulBufCount=ulMIDI_BufferCount;

   if (ulBufCount)
      {
      if (Next == 0)
         Previous = MAXIOBUFFS - 1;
      else
         Previous = Next - 1;

      lpNextBuff->ulPosition = pStream->IOBuff[Previous].ulPosition +
                               pStream->IOBuff[Previous].lSize;
      lpNextBuff->lDelay = 0;

      // total number of bytes in stream
      pStream->ulMaxPosition = lpNextBuff->ulPosition + lpNextBuff->lSize;

      pStream->ulPCMBufferCount++;
      }
   else
      { /* Set ulPosition, lDelay to 0 */
      #ifdef BUF_MONITOR
      StringOut("First buffer of stream");
      #endif
      if (pStream->usMode==PCM)
         {
         lpNextBuff->ulPosition = ulStreamPosition.pcm;

         // total number of bytes in stream
         pStream->ulMaxPosition = lpNextBuff->ulPosition + lpNextBuff->lSize;

         /*
         ** This is the first buffer of the currently inited PCM stream.
         ** Set the DMA buffer pointer and size variables such that
         ** half the total DMA buffer (DMAUnitSize) is never larger than
         ** the buffer size SSM has just passed.  (Note: SSM always
         ** sends buffers of the same size except the final buffer.)
         **
         ** Adjusting the DMAUnitSize to the size of the incoming buffers
         ** is necessary because Ultimotion/Matinee sends small buffers
         ** and cannot tolerate the interrupt reporting rates that
         ** would result from playing small buffers in a large DMA buffer.
         **
         ** Software Motion Video requires ability to adjust DMA buffer size
         */

         if (pStream->usMode == PCM) {
           if ((ulBufferSize >= 0x4000) && fStereoMode &&
               (gdwSampleRate == 44100) && (bBitsSample == 16))
             gdCurrentDMABufferSize = DMA_BUFFER_MAX;
           else if (fStereoMode) {
             if (ulBufferSize < (DMA_BUFFER_MAX / 2))
               gdCurrentDMABufferSize = (2 * ulBufferSize);
             else
               gdCurrentDMABufferSize = DMA_BUFFER_MAX;
           }
           else if (ulBufferSize >= 0x400)
             gdCurrentDMABufferSize = 0x800;
           else if (ulBufferSize <= 0x100)
             gdCurrentDMABufferSize = 0x100;
           else
             gdCurrentDMABufferSize = ((2 * ulBufferSize) & 0xff00);
         }
         else {
           if (ulBufferSize < DMA_BUFFER_MAX/2)
             gdCurrentDMABufferSize = 2 * ulBufferSize;
           else
             gdCurrentDMABufferSize=DMA_BUFFER_MAX;
         }

         wDMABuffLength=(WORD) gdCurrentDMABufferSize-1;
         dwDMAUnitSize=gdCurrentDMABufferSize>>1;
         pStream->ulDMABuffSize=gdCurrentDMABufferSize;
         }
      else if (pStream->usMode==MIDI)
         lpNextBuff->ulPosition = ulStreamPosition.synth;

      lpNextBuff->lDelay = 0;
      pStream->ulPCMBufferCount++;
      } // end else

   if (pStream->usMode==PCM)
      if (pStream->ulDMABuffSize!=gdCurrentDMABufferSize)
         {
           if ((ulBufferSize >= 0x4000) && fStereoMode &&
               (gdwSampleRate == 44100) && (bBitsSample == 16))
             gdCurrentDMABufferSize = DMA_BUFFER_MAX;
           else if (fStereoMode) {
             if (ulBufferSize < (DMA_BUFFER_MAX / 2))
               gdCurrentDMABufferSize = (2 * ulBufferSize);
             else
               gdCurrentDMABufferSize = DMA_BUFFER_MAX;
           }
           else if (ulBufferSize >= 0x400)
             gdCurrentDMABufferSize = 0x800;
           else if (ulBufferSize <= 0x100)
             gdCurrentDMABufferSize = 0x100;
           else
             gdCurrentDMABufferSize = ((2 * ulBufferSize) & 0xff00);

           wDMABuffLength = (WORD)gdCurrentDMABufferSize-1;
           dwDMAUnitSize=gdCurrentDMABufferSize>>1;
         }

   #ifdef BUF_MONITOR
   PrintfOut("WRITE: Setting New Buffer Position to %lx",
             lpNextBuff->ulPosition);
   #endif

   //***********************************************
   // Increment next buffer index and check for wrap
   //***********************************************

   if ((pStream->usCurrIOBuffIndex == pStream->usNextIOBuffIndex) &&
       (pStream->usDoneIOBuffIndex == pStream->usCurrIOBuffIndex))
      {
      iNextBuffer=0;
      iEmptyPCMBuffers=2;
      }

   if ((++Next == MAXIOBUFFS))
      Next = 0 ;
   pStream->usNextIOBuffIndex = Next;

   if (pStream->usMode==PCM)  // PCM requires prefilling DMA before starting
      if (iEmptyPCMBuffers)
         if (!(pStream->IOBuff[Current].usRunFlags & IOB_STARTED))
             InitialBufferLoad(pStream);

   #ifdef BUF_MONITOR
   PrintfOut ("DDCmdWrite Exit: Done: %d   Curr: %d   Next: %d\r\n",
              pStream->usDoneIOBuffIndex,
              pStream->usCurrIOBuffIndex,
              pStream->usNextIOBuffIndex
              );
   #endif

   return( NO_ERROR);
}


/*                                      ------------------------- DDCmdStatus -
** Query current ulPosition (stream time)
**
** Allows the Audio stream handler to get the current
** stream time ulPosition value in milliseconds.
**
** This routine will return the current real-time
** "stream-time" for the stream.
**
** This routine is called at non-interrupt time.
**
** EXIT-NORMAL:  NO_ERROR
** EXIT_ERROR:   ERROR_INVALID_STREAM
** INPUTS:       ulFunction              DDCMD_STATUS
**               hStream                 Stream handle
*/
RC DDCmdStatus (PDDCMDSTATUS pStatus)
{
   PSTREAM pStream;
   RC      rc;

   #ifdef PROGRESS_MONITOR
   StringOut( "DDCmdStatus");
   #endif
   pStream = GlobalTable.paStream;
   if (rc = GetStreamEntry(&pStream, pStatus->hStream))
       return(rc);

   /*
   ** Return far pointer containing real-time position to stream handler
   */
   pStatus->ulStatusSize = sizeof(recio.ulPosition);
   pStatus->pStatus = GetStreamTime(pStream);  // return time in ms.
   return (NO_ERROR);
}


/*                                      ------------------------ DDCmdControl -
** Maps the AUDIO_CONTROL IDC commands to their IOCTL equilivent.
**
** Since the entire routine uses the AUDIO_HPI calls,
** Just set the current buffer's usRunFlags to the right command
** Command from stream handler comes in pControl->ulCmd.
** This routine sets flags, IOBuff and pStream structures.
** These will be referenced at interrupt time for command execution.
** For some operations, we will call stream handler directly.
**
** EXIT-NORMAL:  NO_ERROR
** EXIT_ERROR:   ERROR_INVALID_FUNCTION
**               ERROR_STREAM_NOT_STARTED
**               ERROR_INVALID_SEQUENCE
**               ERROR_INVALID_STREAM
**               ERROR_TOO_MANY_EVENTS
** INPUTS:       ulFunction              DDCMD_CONTROL
**               hStream                 Stream handle
**               ulCmd                   Specific control command
**               pParm                   Not used
**               ulParmSize              Not used
*/
RC DDCmdControl (PDDCMDCONTROL pControl)
{
   USHORT  Current, i;
   PSTREAM pStream;
   RC      rc;
   ULONG   ulCuePoint;

   pStream = GlobalTable.paStream;
   if (rc = GetStreamEntry(&pStream, pControl->hStream))
      return(rc);
   Current = pStream->usCurrIOBuffIndex;

   /*
   ** Perform specific DDCMD_CONTROL function
   */
   switch (pControl->ulCmd)
      {
      case DDCMD_START:
         #ifdef PROGRESS_MONITOR
         StringOut ("DDCmdControl: Start");
         #endif
         //*********************************************
         // ... PAUSE - START ... is an invalid sequence
         //*********************************************
         if (pStream->IOBuff[Current].usRunFlags & IOB_PAUSED)
            return (ERROR_INVALID_SEQUENCE);

         // 2/26/93
         // added in case PCM is started and only one buffer was prefilled!
         // PCM requires prefilling DMA before starting
         if (pStream->usMode==PCM)
            if (operation.pcm==OPERATION_PLAY)
               if (iEmptyPCMBuffers) // PCM Requires prefilling buffers
                  if (!(pStream->IOBuff[Current].usRunFlags & IOB_STARTED))
                     WriteDataToCard(pStream);

         iNextBuffer=0;   // next buffer to be filled

         /*
         ** stream handler told us to start and we have enough
         ** full buffers, so send this buffer to the
         ** audio card.  If operation is PLAY, the
         ** routine will output the contents of the
         ** buffer to the card.  If operation is RECORD
         ** the routine will fill the buffer with data
         ** from the card's input jack.
         */

         if (pStream->usMode==PCM)
            {
            switch (operation.pcm)
               {
               case OPERATION_PLAY:
                  PCMPlay();
               break;

               case OPERATION_RECORD:
                  PCMRecord();
               break;
               } // end switch
            } // end if

         else if (pStream->usMode==MIDI)
            {
            switch (operation.synth)
               {
            case OPERATION_PLAY:
                  if ((BYTE)PASX_IN(FILTER_REGISTER) == 0) {
                    SetFilter(1);
                  }
                  Enable_MIDI_IO();
                  MIDI_Output_Start();

                  pStream->usPPQN=24;        // default pulses per quarter note
                  pStream->ulTempo=1200;     // default tempo
                  ///pStream->ulCumTime      =0;
                  ///pStream->ulCumTimeX10   =0;
                  ///pStream->ulMIDIStreamTimeX10=0;
                   StringOut( "RESETTING MIDI TIMING");

                  StopMIDIInterrupts(); // make sure no MIDI interrupts pending
                  StartMIDIInterrupts(pStream);
                  ulStreamPosition.synth=0;     // count of ticks.
                  ulTimingDelay=0;
                  break;

               case OPERATION_RECORD:  // MIDI record not supported
                  break;
               } // end switch
            }

         pStream->ulFlags |= STREAM_STREAMING;
         pStream->ulFlags &= ~STREAM_STOPPED;
         //Audio_IOCTL_Hpi (pControl->pParm);

         /*
         ** Set all IOBuffs to START
         */
         for (i=0; i<MAXIOBUFFS; i++)
            pStream->IOBuff[i].usRunFlags |= IOB_STARTED;
         break;

      case DDCMD_STOP:
         /*
         ** because we always write to the "next" buffer and play
         ** from the "current" buffer, reset the indexes to point
         ** at the same packet. So when a re-start comes in we will
         ** not play a null buffer.
         */

         #ifdef PROGRESS_MONITOR
         StringOut( "DDCmdControl: Stop");
         #endif

         if (pStream->usMode==PCM)
            {
            StopPCM();
            }
         else if (pStream-> usMode==MIDI)
            {
            Disable_MIDI_IO();
            MIDI_Output_Stop();
            StopMIDIInterrupts();
            shut_off_all();                                                             /* shut off any notes which are playing */
            reset_all();
            MIDI_All_Notes_Off();

            #ifdef PROGRESS_MONITOR
            StringOut("--Shutting OFF MIDI notes--");
            #endif

            #ifdef PROGRESS_MONITOR
            StringOut ("MIDI received STOP command");
            #endif
            }
         pStream->ulFlags |= STREAM_STOPPED;
         pStream->ulFlags &= ~(STREAM_STREAMING|STREAM_PAUSED);


         /*
         ** Send all buffers back to stream handler
         */
         while (pStream->usDoneIOBuffIndex != pStream->usCurrIOBuffIndex)
            {
            if (pStream->usMode==PCM)
               {
               if (operation.pcm == OPERATION_PLAY)
                  DisposeOfPlayedBuffer(pStream);
               else
                  DisposeOfFilledBuffer(pStream);
               }
            else // MIDI
               {
               DisposeOfPlayedBuffer(pStream);
               }
            } // while

         /*
         ** Do this so that PDD does not return an old buffer
         ** to handler after handler received a STOP-DISCARD.
         */
         Current                    =
         pStream->usCurrIOBuffIndex =
         pStream->usNextIOBuffIndex = 0;

         for (i=0; i<MAXIOBUFFS; i++)
            {
            pStream->IOBuff[i].usRunFlags &= ~(IOB_RUNNING |
                                               IOB_STARTED |
                                               IOB_PAUSED |
                                               IOB_PAUSECORRUPTED);

            pStream->IOBuff[i].usRunFlags |= IOB_STOPPED;
            pStream->IOBuff[i].lSize      = 0;
            pStream->IOBuff[i].pBuffer   = NULL;
            pStream->IOBuff[i].lCount    = 0;
            pStream->IOBuff[i].ulPosition  = 0;
            pStream->IOBuff[i].lDelay     = 0;
            pStream->IOBuff[i].pBuffer    = NULL;
            }

         // number of buffers sent for inited stream
         pStream->ulPCMBufferCount=0;

         // Always return stream time when
         // stopping or pausing device.
         pControl->pParm = GetStreamTime(pStream);

         pControl->ulParmSize = sizeof(recio.ulPosition);
         iEmptyPCMBuffers = 2;
         iNextBuffer = 0;      // Need this for repeated plays
         pStream->usCurrIOBuffIndex = 0;
         pStream->usNextIOBuffIndex = 0;
         pStream->usDoneIOBuffIndex = 0;
         break;

      case DDCMD_PAUSE:
         #ifdef PROGRESS_MONITOR
         StringOut ("DDCmdControl: Pause");
         #endif

         /*
         ** Trying to PAUSE a non-started stream is an error
         */
         if (!(pStream->IOBuff[Current].usRunFlags & IOB_STARTED))
            return (ERROR_STREAM_NOT_STARTED);

         if (pStream-> usMode==PCM)
            {
            PausePCM();                      // MV36  testing
            usPauseDMAAddr=ReadDMA();
            #ifdef PROGRESS_MONITOR
            PrintfOut("Pause at DMA pointer %0d ", (WORD) usPauseDMAAddr);
            PrintfOut("Pause at DMA pointer %0x ", (WORD) usPauseDMAAddr);
            #endif
            }
         else if (pStream-> usMode==MIDI)
            {
            shut_off_all();    /* shut off any notes which are playing */
            MIDI_All_Notes_Off();
            #ifdef PROGRESS_MONITOR
            StringOut("MIDI paused");
            #endif
            }

         #ifdef PROGRESS_MONITOR
         StringOut("Marking Stream buffers Paused");
         #endif
         for (i=0; i<MAXIOBUFFS; i++ )
            pStream->IOBuff[i].usRunFlags |= IOB_PAUSED;

         pStream->ulFlags |= STREAM_PAUSED;
         pStream->ulFlags &= ~STREAM_STREAMING;

         // Always return stream time when
         // stopping or pausing device.
         pControl->pParm = GetStreamTime(pStream);
         pControl->ulParmSize = sizeof(recio.ulPosition);
         #ifdef PROGRESS_MONITOR
         StringOut("Pause OK");
         #endif
         break ;

      case DDCMD_RESUME:
         #ifdef PROGRESS_MONITOR
         StringOut ("DDCmdControl: Resume");
         #endif

         /*
         ** Trying to RESUME a stopped/non-paused stream is an error
         */
         if (!(pStream->IOBuff[Current].usRunFlags & IOB_STARTED))
            {
            #ifdef DEBUG_CHK
                           StringOut( "ERROR: Can't resume buffer, not started!");
            #endif
            return (ERROR_INVALID_SEQUENCE);
            }

         // stream must be unpaused before restarting PCM
         // or the buffer will loop on unrefreshed data (NO STREAM)

         for (i=0; i<MAXIOBUFFS; i++ )
            pStream->IOBuff[i].usRunFlags &= ~IOB_PAUSED;

         pStream->ulFlags |= STREAM_STREAMING;
         pStream->ulFlags &= ~STREAM_PAUSED;
         fPCMStarved = FALSE;
         fPCMUnderran = FALSE;

         if (pStream->IOBuff[Current].usRunFlags & IOB_PAUSECORRUPTED)
            {
            pStream->IOBuff[Current].usRunFlags &= ~IOB_PAUSECORRUPTED;
            WriteDataToCard (pStream);
            WriteDataToCard (pStream);
            if (pStream-> usMode==PCM)
               {
               if (pStream->usMode==PCM)
                  if (pStream->ulDMABuffSize!=gdCurrentDMABufferSize)
                     {
                     gdCurrentDMABufferSize = pStream->ulDMABuffSize;
                     wDMABuffLength = (WORD) gdCurrentDMABufferSize-1;
                     dwDMAUnitSize  =        gdCurrentDMABufferSize>>1;
                     }
               PCMPlay();
               ResumePCM();
               TogglePCMIntEnable();
               }
            #ifdef PROGRESS_MONITOR
            StringOut("Resuming CORRUPTED, paused stream");
            #endif
            }
         else
            {
            if (pStream-> usMode==PCM)
               {
               ResumePCM();
               #ifdef PROGRESS_MONITOR
               StringOut ("resuming PCM stream");
               #endif
               TogglePCMIntEnable();
               }
            else if (pStream-> usMode==MIDI)
               {
               if ((BYTE)PASX_IN(FILTER_REGISTER) == 0) {
                  SetFilter(1);
               }
               #ifdef PROGRESS_MONITOR
               StringOut ("resuming MIDI stream");
               #endif
               ToggleMIDIIntEnable();
               }
            #ifdef BUF_MONITOR
            StringOut("Resuming uncorrupted, paused stream");
            #endif
            } // end else
         break;


      case DDCMD_ENABLE_EVENT:
         // Create an event with specific cuepoint.
         // This PDD will detect the cuepoint and report
         // its occurance to the stream handler via SHD_REPORT_EVENT

         #ifdef PROGRESS_MONITOR
         StringOut ("DDCmdControl: Enable Event");
         #endif
         ulCuePoint = *((unsigned long far *)pControl->pParm);
         rc = Enqueue_Event (pControl->hEvent, ulCuePoint);
         return (rc);

      case DDCMD_DISABLE_EVENT:
         #ifdef   PROGRESS_MONITOR
         StringOut ("DDCmdControl: Disable Event");
         #endif
         rc = Dequeue_Event (pControl->hEvent);
         return (rc);

      default:
         #ifdef PROGRESS_MONITOR
         StringOut ("DDCmdControl: Invalid Function");
         #endif
         return (ERROR_INVALID_FUNCTION) ;
      } // end switch

   return (NO_ERROR);
}


/*                                      ----------------------- DDCmdRegister -
** Register stream with this PDD
** Initialize stream.
**
** Will receive the handler ID, handle of stream instance,
** address of entry point from stream handler.
** Set waiting flag (waiting for data)
** Set usRunFlags = 0
** Set No_circular_buffer flag (0x80)
** Set protocol info for stream handler
** Set Address type to Physical
**
** EXIT-NORMAL:  NO_ERROR
** EXIT-ERROR:   ERROR_INVALID_STREAM
**               ERROR_HNDLR_REGISTERED
**               ERROR_INVALID_FUNCTION
**               ERROR_INVALID_SPCBKEY
** INPUTS:       ulFunction              DDCMD_REG_STREAM
**               hStream                 Stream handle
**               ulSysFileNum            Device handle so PDD can map device
**                                       instance to hStream
**               pSHDEntryPoint          Stream handler entry point
**               ulStreamOperation       Record (PRODUCE) or Play (CONSUME)
**               spcbkey                 Protocol info which determines outputs
** OUTPUTS:      ulBufSize               Buffer size in bytes for SPCB
**               ulNumBufs               # of buffers for SPCB
**               ulAddressType           Address type of data buffer
**               ulBytesPerUnit          Bytes per unit
**               mmtimePerUnit           MMTIME per unit
*/
RC DDCmdRegister (PDDCMDREGISTER pRegister)
{
   WORD          wNumProtocols;
   USHORT        i;
   PSTREAM       pStream;
   RC            rc;
   PROTOCOLTABLE *pProtocolTable;

   #ifdef PROGRESS_MONITOR
   StringOut( "DDCmdRegister");
   #endif
   if (pRegister->hStream == -1)
      return (ERROR_INVALID_STREAM) ;    /* Stream handle invalid */

   if (pRegister->ulSysFileNum == DATATYPE_NULL)
      return (ERROR_INITIALIZATION);

   if ((pRegister->ulStreamOperation != STREAM_OPERATION_CONSUME) &&
       (pRegister->ulStreamOperation != STREAM_OPERATION_PRODUCE))
      {
      #ifdef DEBUG_CHK
      StringOut ("DDCmdRegister:error invalid function");
      #endif
      return (ERROR_INVALID_FUNCTION);
      }

   /*
   ** Verify requested stream characteristics
   ** match our capabilities.
   ** Do protocol table lookup, using DataType,
   ** DataSubType as keys and return ulBufSize,
   ** ulNumBufs, ulAddressType, mmtimePerUnit;
   */

   if (pf.ProCard[gwBoardIndex].Caps.CapsBits.DAC16)
      {
      #ifdef PROGRESS_MONITOR
      StringOut ("DDCmdRegister: 16-bit Protocol table");
      #endif
      wNumProtocols=N16PROTOCOLS;
      pProtocolTable=ProtocolTable16;
      }
   else
      {
      #ifdef PROGRESS_MONITOR
      StringOut ("DDCmdRegister: 8-bit Protocol table");
      #endif
      wNumProtocols=N8PROTOCOLS;
      pProtocolTable=ProtocolTable8;
      }

   for (i=0; i<wNumProtocols; i++, pProtocolTable++)
      {
      if (pProtocolTable->ulDataType == pRegister->spcbkey.ulDataType)
         {
         if (pProtocolTable->ulDataSubType == pRegister->spcbkey.ulDataSubType)
            {
            // We found a match for data type, data sub type
            break ;
            }
         }
      } // end for

   if (i==wNumProtocols)                           // No match found
      return (ERROR_INVALID_SPCBKEY) ;

   /*
   ** Match found: Pass back protocol ("stream characteristics")
   **              to stream handler.
   */
   pRegister->ulBufSize       = pProtocolTable->ulBufSize ;
   pRegister->ulNumBufs       = pProtocolTable->ulNumBufs ;
   pRegister->mmtimePerUnit   = 1; // Chinatown MMTIME unit: 1/3000th/sec

   /*
   ** Bytes per unit =
   **
   **  Samples     Channels     Bits      Byte     Sec
   **  -------  *           *  ------  *  ---- *  ------
   **    Sec                   Sample     Bits    MMTIME
   */

   if (pRegister->spcbkey.ulDataType==DATATYPE_WAVEFORM)
      {
      #ifdef PROGRESS_MONITOR
      StringOut( "DDCmdRegister: notify setting bytes per unit");
      #endif
      pRegister->ulBytesPerUnit = gdCurrentDMABufferSize/2;

      // Has this been Init'ed?
      if (ulActiveSysFileNum.pcm !=pRegister->ulSysFileNum)
         {
         #ifdef DEBUG_CHK
         StringOut ("DDCmdRegister:error invalid stream");
         #endif
         return(ERROR_INVALID_STREAM);
         }
      }

   if (pRegister->spcbkey.ulDataType==DATATYPE_MIDI)
      {
      pRegister->ulBytesPerUnit  = 512;

      // Has this been Init'ed?
      if (ulActiveSysFileNum.midi !=pRegister->ulSysFileNum)
         return (ERROR_INVALID_STREAM);
      }

   /*
   ** Tell stream handler what type of data buffer pointer to reference
   ** change to virtual or linear
   */
   pRegister->ulAddressType = ADDRESS_TYPE_VIRTUAL;

   /*
   ** Initialize pStream to point at the first entry for track trk
   */
   pStream = GlobalTable.paStream;
   if (!(rc = GetStreamEntry(&pStream, pRegister->hStream)))
      return(ERROR_HNDLR_REGISTERED);
   pStream = GlobalTable.paStream;      // no match, so create stream
   i=0;

   while(pStream->hStream != -1)
      { // find an empty stream entry
      if (++i >= GlobalTable.usMaxNumStreams)
         return(ERROR_STREAM_CREATION);
      pStream++;
      }

   //**************************************
   // Found empty stream entry, so use it
   // Fill Stream structure
   //**************************************
   for (i=0; i<MAXIOBUFFS; i++)
      {
      pStream->IOBuff[i].lSize        =  0;
      pStream->IOBuff[i].pHead        =  NULL;
      pStream->IOBuff[i].pTail        =  NULL;
      pStream->IOBuff[i].lCount       =  0;
      pStream->IOBuff[i].ulPosition   =  0;
      pStream->IOBuff[i].lDelay       =  0;
      pStream->IOBuff[i].usRunFlags   =  IOB_CHAIN_BUFFERS;
      pStream->IOBuff[i].pBuffer      =  NULL;
      }

   /*
   ** Save register info
   */
   pStream->hStream             = pRegister->hStream;
   pStream->ulFlags             = STREAM_REGISTERED;
   pStream->ulOperation         = pRegister->ulStreamOperation;
   pStream->ulCumTime           = 0;
   pStream->current_time        = 0;
   pStream->ulCumTimeX10        = 0;
   pStream->ulMaxPosition       = 0;
   pStream->ulTempo             = 1200;
   pStream->ulMIDIStreamTimeX10 = 0;
   pStream->usPPQN              = 24;   // MIDI parts per quarter note

   pStream->usCurrIOBuffIndex = 0;
   pStream->usNextIOBuffIndex = 0;
   pStream->usDoneIOBuffIndex = 0;
   (PSHDFN)pStream->ADSHEntry = pRegister->pSHDEntryPoint;

   if (pRegister->spcbkey.ulDataType==DATATYPE_WAVEFORM)
      {
      #ifdef DEBUG_CHK
      StringOut ("DDCmdRegister: notify usMode=PCM");
      #endif
      pStream->usMode = PCM;
      pStream->ulSysFileNum   = ulActiveSysFileNum.pcm;
      }

   if (pRegister->spcbkey.ulDataType==DATATYPE_MIDI)
      {
      pStream->usMode = MIDI;
      pStream->ulSysFileNum   = ulActiveSysFileNum.midi;
      }

   switch (operation.pcm)
      {
      case OPERATION_PLAY:
         if (pStream->usMode == PCM)
            {
            #ifdef DEBUG_CHK
            StringOut ("DDCmdRegister: notify OPERATION_PLAY PCM selected");
            #endif
            ulStreamPosition.pcm = ulPCM_PosSemiBufA = ulPCM_PosSemiBufB = 0;
            }

         if (pStream->usMode ==MIDI)
            {
            ulStreamPosition.midi=0;
            ulStreamPosition.synth=0;
            }
         break;

      case OPERATION_RECORD:
         if (pStream->usMode ==PCM)
            {
            ulStreamPosition.pcm = ulPCM_PosSemiBufA = 0;
            ulPCM_PosSemiBufB = (gdCurrentDMABufferSize/2);
            }

         if (pStream->usMode ==MIDI)
            {
            ulStreamPosition.midi = 0;
            }
         break;

      default:
         #ifdef DEBUG_CHK
         StringOut ("DDCmdRegister: error invalid operation");
         #endif
         ;
      } // end switch

   pStream->ulPCMBufferCount = 0;
   iNextBuffer = 0;
   return (NO_ERROR);
}


/*                                      --------------------- DDCmdDeRegister -
** Deregister this stream
** Removes stream instance from this PDD
**
** Done by setting table handle to value -1
**
** EXIT-NORMAL:  NO_ERROR
** EXIT_ERROR:   ERROR_INVALID_STREAM
** INPUTS:       ulFunction              DDCMD_DEREG_STREAM
**               hStream                 Stream handle
*/
RC DDCmdDeRegister (PDDCMDDEREGISTER pDeRegister)
{
   USHORT  i;
   PSTREAM pStream;
   RC      rc;

   #ifdef PROGRESS_MONITOR
   StringOut ("DDCmdDeRegister");
   #endif

   /*
   ** Check table to see if stream is registered and a valid handle
   */
   pStream = GlobalTable.paStream;

   if (rc = GetStreamEntry (&pStream, pDeRegister->hStream))
      {
      #ifdef DEBUG_CHK
      StringOut ("Whoops... couldn't find stream to deregister");
      #endif
      return (rc);
      }

   pStream->usCurrIOBuffIndex = 0;
   pStream->usNextIOBuffIndex = 0;
   pStream->usDoneIOBuffIndex = 0;

   /*
   ** De-activate this stream
   ** and clear all flags
   */
   pStream->hStream = -1;       // make stream available
   pStream->ulFlags = 0;        // clear flags

   for (i=0; i<MAXIOBUFFS; i++)
      pStream->IOBuff[i].usRunFlags = 0;

   return( NO_ERROR);
}


/*                                      ---------------------- GetStreamEntry -
** Get the stream table entry.
** Search the stream table finding a match with the given parm.
**
** Notes: This routine is called internally.
**
** INPUT: pointer to stream table, stream handle to find
** Returns ERROR_INVALID_STREAM if stream not found in table
*/
RC GetStreamEntry(PSTREAM far *ppStream, HSTREAM hStream)
{
   USHORT  i;
   PSTREAM pStream;

   i = 0;
   pStream = *ppStream;
   while(pStream->hStream != hStream)            // find stream entry
      {
      if (++i >= GlobalTable.usMaxNumStreams)
         return(ERROR_INVALID_STREAM);
      pStream++;
      };
   *ppStream = pStream;
   return (NO_ERROR);
}
