/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************
*                 Pro AudioSpectrum16 Physical Device Driver
*                     Production code and toolkit sample
*
*
* DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
* sample code created by IBM Corporation and Media Vision Corporation.
* It is provided to you solely for the purpose of assisting you in the
* development of your applications.
* The code is provided "AS IS", without warranty of any kind.
* IBM and Media Vision shall not be liable for any damages arising out of
* your use of the sample code, even if they have been advised of the
* possibility of such damages.
*
*******************************************************************************
*
* findpas.c - Find and initialize PAS-16 adapters
*
* DESCRIPTION:
* This module contains code for locating and initializing a
* Pro AudioSpectrum 16 card or other device with MV101-based hardware.
*
* The Pro AudioSpectrum Plus, Pro AudioSpectrum 16 and CDPC
* all support relocatable I/O.   PAS hardware must occupy four
* contiguous I/O locations.  All I/O accesses use those four addresses
* and their 'harmonic' addresses.  Harmonic addresses are those
* addresses that under normal circumstance would decode to the same
* contiguous four I/O locations.  For example, harmonic addresses of
* port 388h would include 788h, 1388h, 1788h, 1f88h, etc.
*
* Most hardware will not conflict with the default address range of
* 388h-38Bh.  This address range is based on the I/O locations
* chosen by Adlib and this range is supposed to be reserved for
* SDLC or 2nd Bisynchronous communications hardware.  It has been
* reported that an ISDN card decodes to this range also.  More
* importantly, if a customer wants to have multiple sound cards
* co-resident in his system, we must provide a means of relocating
* the I/O range.
*
* Media Vision's I/O relocation capability is software selectable to
* any 4 byte location in the I/O space.  We originally scanned for
* the presence of PAS hardware by reading the values at 64 quadport
* locations.  This caused havoc with ATI Grahpics Ultra and some
* network cards that don't appreciate having their ports read by
* other software.  We've settled on scanning seven of the more
* likely locations and this seems to be acceptable.
*
*        0     2     4     6     8     A     C     E
*
*  200h  |<------------- game control------------------->|
*
*  210h  |<-- expansion unit -->|            |           |            |
*
*  220h  |<---- Sound Blaster decode #1 ---------------->|
*
*  230h  |<---- Sound Blaster decode #2 ---------------->|
*
*  240h  |<---- Sound Blaster decode #3 ---------------->|
*
*  250h  |<------------- not scanned ------------------->|
*
*  260h  |<------------- not scanned ------------------->|
*
*  270h  |<---- not scanned ---|<--3rd Parallel Printer->|
*
*  280h  |<---PAS--->|<---PAS--->|<---PAS--->|<---PAS--->|
*
*  290h  |<------------- not scanned ------------------->|
*
*  2A0h  |<------------- not scanned ------------------->|
*
*  2B0h  |<------------- not scanned ------------------->|
*
*  2C0h  |<------------- not scanned ------------------->|
*
*  2D0h  |<-----------    3270    PC     --------------->|
*
*  2E0h  |<------------- not scanned ------------------->|
*
*  2F0h  |<----- reserved ------>|<-Async comm. 2 ------.|
*
*        0     2     4     6     8     A     C     E
*
*  300h  |<401---------- Prototype card ---------------->|
*
*  310h  |<401---------- Prototype card ---------------->|
*
*  320h  |<401----------    hard disk   ---------------->|
*
*  330h  |<401---XT/3270-------->|<----- not scanned --->|
*
*  340h  |<401|<----------------- not scanned ---------->|
*
*  350h  |<401|<----------------- not scanned ---------->|
*
*  360h  |<401|<----------------- not scanned ---------->|
*
*  370h  |<401|<-not scanned ->|<--2nd Parallel Printer->|
*
*  380h  |<401, ---->|<---PAS--->|<---PAS--->|<---PAS--->|
*
*  390h  |<401|<----------------- not scanned ---------->|
*
*  3A0h  |<401|<----------------- not scanned ---------->|
*
*  3B0h  |<401, Monochrome Display and Parallel 1  ----->|
*
*  3C0h  |<401, ------------ VGA   --------------------->|
*
*  3D0h  |<401,-------    3270    PC     --------------->|
*
*  3E0h  |<401,  reserved ------>|<---- COM 3 serial---->|
*
*  3F0h  |<401,   Disk    ------>|<-- Asych comm. 1 ---->|
*
*
*
*  Note: 401 indicates one of the possible locations of the
*        MPU-401 MIDI interface.  It decodes at 3x0 - 3x1 where
*        x is an element of {0,1,2,3,4,5,6,7,8,9,a,b,c d,e,f}
*
*        SoundBlaster decode range is 2x0 thru 2xf where x is
*        an element of {2,3,4}
*
*  It requires very little effort to support the relocation of
*  the hardware if the routines provided here are used.  Once
*  the Pro AudioSpectrum chip has been located, any I/O address
*  can be relocated by XOR-ing the address (usually contained in)
*  the DX register) with a 16-bit variable.
*
*  TO SUPPORT THE PRO AUDIOSPECTRUM PROPERLY YOU MUST SUPPORT
*  RELOCATABILITY!!  Adherence to this rule will allow multiple
*  sound cards to be present in a system.  It is also advisable
*  to support multiple Pro AudioSpectra in the same system.
*
*  It is well known that PC hardware installation and configuration
*  is one of the biggest challenges that faces the average user.
*  One of the provided routines will attempts to find a suitable
*  I/O address range for the card to reside at.
*
*/

#include <os2.h>

#include <os2medef.h>
#include "types.h"
#include "findpas.h"
#include "pasdef.h"
#include "debug.h"
#include "commdbg.h"
#include "rmhelp.h"


//--------------========================---------------------------
//---------====< DEFINES AND MACROS >====-------------------------
//--------------========================---------------------------


#define NUM_POSSIBLE_LOCATIONS   7

//--------------========================---------------------------
//---------====< EXTERN GLOBAL DATA SECTION >====------------------
//--------------========================---------------------------

//--------------========================---------------------------
//---------====< GLOBAL DATA SECTION >====-------------------------
//--------------========================---------------------------

// SEARCH LOCATIONS LISTED IN ORDER OF SEARCH

WORD search_locations[NUM_POSSIBLE_LOCATIONS]=
        {
        0x0388,0x0384,0x038C,
        0x0288,0x0280,0x0284,0x028C
        };


// The board signature is the first value in the PAS 16 wakeup sequence
// BC is the factory default.  A board jumpered to recognize the BD signature
// will not respond to a BC init command.

BYTE SignatureTable[4]={0xBC,0xBD,0xBE,0xBF};

// The IRQ table is used to convert an IRQ index to the IRQ
// channel number that the hardware has been told to use

BYTE IRQTable[16]={0,2,3,4,5,7,10,11,12,14,15,0,0,0,0,0};

// The DMA table is used to convert a DMA index value to the DMA
// channel number that the hardware has been told to use

BYTE DMATable[8]={0xff,1,2,3,0,5,6,7};

unsigned int NumRequests=0;
WORD request_locations[MAXCARDS];


//--------------========================---------------------------
//---------====< FUNCTION PROTOTYPES >====-------------------------
//--------------========================---------------------------

WORD GetProTable( pPROFILE pPf );
WORD GetRequestedLocations(pPROFILE pPf);
BOOL WakeUpAtAddress(WORD wPort, FOUNDINFO _far *pFoundInfo );
BOOL VerifyProHardware(WORD port, FOUNDINFO _far *pFI);
BOOL VerifyNothingThere(WORD port);
void ClearProfileTable(pPROFILE pPf);

//--------------==============---------------------------
//---------====< CODE SECTION >====----------------------
//--------------==============---------------------------

WORD FPwTranslateCode;

                                        //---------------------------- PAS_IN -
BYTE PAS_IN (WORD port)
{
   BYTE bRetval;

   port ^= FPwTranslateCode;

   _asm {
        mov     dx,port
        in      al,dx
        mov     bRetval,al
        }
   return bRetval;
}

                                        //--------------------------- PAS_OUT -
void PAS_OUT (WORD port, BYTE data)
{
   port ^= FPwTranslateCode;

   _asm {
        mov     dx,port
        mov     al,data
        out     dx,al
        }
}


/*                                      ------------------------- GetProTable -
** Detects which version of the Pro AudioSpectrum is installed
**
** Entry Conditions:
** Pointer to Profile Structure.  If the caller wants to specify
** the preferred base address for cards not yet init'd, they
** are passed in this structure.  The NumFound field indicates
** the number of location requests and the board address elements
** indicate the locations.
**
** Exit Conditions:
** Returns number of cards found
** ProFile structure has been updated.
*/
WORD GetProTable (pPROFILE pPf)
{
   WORD           i;
   FOUNDINFO _far *pFoundInfo;

   #ifdef INIT_MONITOR
   StringOut( "GetProTable");
   #endif

   GetRequestedLocations(pPf);     // Caller passes in location requests
   ClearProfileTable(pPf); // Initialize ProFile Table to zero

   //
   //  First, locate all cards that have already been initialized.
   //
   for (i=0; i<NUM_POSSIBLE_LOCATIONS; i++)
      {
      #ifdef INIT_MONITOR
      PrintfOut ("Scan for Preexisting card at %0x", search_locations[i]);
      #endif
      pFoundInfo=&pPf->ProCard[pPf->wNumFound];
      if (VerifyProHardware(search_locations[i],pFoundInfo))
         {

         pPf->wNumFound++;
         break;
         }

      if (pPf->wNumFound >= MAXCARDS)
         goto FoundEmAll;
      } // end for

   //
   //  Now, search through list of requested locations and try to wake
   //  up any relocatables that haven't yet been addressed
   //
   if (NumRequests==0)
      {
      #ifdef INIT_MONITOR
      StringOut( "No requested locations");
      #endif
      }
   else
      for (i=0; i<NumRequests; i++)
         {
         // For each requested location:
         //   1) see if a card has already been found there.
         //      If so, skip location.
         //   2) if no other hardware appears to be there
         //     try waking up hardware until you find it there
         //      otherwise, choose another place to locate card

         pFoundInfo=&pPf->ProCard[pPf->wNumFound];
         #ifdef INIT_MONITOR
         PrintfOut ("Verifying vacant request location: %0x",
                    request_locations[i]);
         #endif
         if (VerifyLegalAddress (request_locations[i]) &&
             VerifyNothingThere (request_locations[i]))
            {
            #ifdef INIT_MONITOR
            PrintfOut ("Attempting wake up at request location: %0x",
                       request_locations[i] );
            #endif
            if (WakeUpAtAddress (request_locations[i], pFoundInfo))
               {
               pPf->wNumFound++;
               goto FoundEmAll;     // Got what we wanted!
               }
            else
               goto FoundEmAll;
            }

         if (pPf->wNumFound >= MAXCARDS)
            goto FoundEmAll;
         } // end for

   //
   // Finally, we attempt to wake up hardware at default locations
   //

   for (i=0; i<NUM_POSSIBLE_LOCATIONS; i++)
      {
      pFoundInfo=&pPf->ProCard[pPf->wNumFound];
      if (VerifyNothingThere (search_locations[i]))
         {
         #ifdef INIT_MONITOR
         StringOut ("Attempting Wake Up....");
         #endif
         if (WakeUpAtAddress (search_locations[i] ,pFoundInfo))
            pPf->wNumFound++;
         else
            goto FoundEmAll;
         }

      if (pPf->wNumFound >= MAXCARDS)
         goto FoundEmAll;
      } // end for

   FoundEmAll:
   #ifdef INIT_MONITOR
   StringOut( "FoundEmAll");
   #endif
   return pPf->wNumFound;
}


                                        //------------- GetRequestedLocations -
WORD GetRequestedLocations (pPROFILE pPf)
{
   WORD i;

   if (!pPf->wNumFound)
      {
      NumRequests=0;
      }
   else
      {
      // NumFound contains number of locations request
      for (i=0; i<pPf->wNumFound ; i++)
         {
         if ( i > MAXCARDS)
            break;
         request_locations[i]=pPf->ProCard[i].ProPort;
         #ifdef INIT_MONITOR
         PrintfOut("Getting requested location %0x",pPf->ProCard[i].ProPort);
         #endif
         }
      NumRequests=pPf->wNumFound;
      }

   return(pPf->wNumFound);
}


                                        //----------------- ClearProfileTable -
void ClearProfileTable (pPROFILE pPf)
{
   WORD i=sizeof(PROFILE);
   char _far *p1=(char _far *) pPf;

   pPf->wNumFound=0;

   while (i--)
      *p1=0;
}


/*                                      ------------------- VerifyProHardware -
** Detects which version of the Pro AudioSpectrum is installed
**
** Entry Conditions:
**    wParam1= Base I/O address to check
**
** Exit Conditions:
**     Returns AX=PORT ADDRESS      if Pro AudioSpectrum found
**     Returns AX=NO_PAS_INSTALLED  if Pro AudioSpectrum not found
**
** WARNING:  THIS IS PASCAL TYPE, IT POPS ITS PARAMETERS
*/
BOOL VerifyProHardware (WORD port, FOUNDINFO _far *pFI)
{
   BYTE bData;
   BYTE bTemp;

   FPwTranslateCode = port ^ DEFAULT_BASE;

   RMHELP_SetPortsMarker();   // PORTS ALLOC'D FROM WILL BE RELEASED IF
                              // RMHELP_DeAllocPortsToMarker CALLED

   if ( RMHELP_GrabPort( INTERRUPT_CTRL_REG) != RMRC_SUCCESS )
   {
      goto VerifyFailed;   // SOMEONE ALREADY HAS CLAIMED THIS PORT EXCLUSIVE
   }

   bData = PAS_IN (INTERRUPT_CTRL_REG);

   if (bData==0xFF)             // 0xFF usually means nothing there
      {
      goto VerifyFailed;
      }

   pFI->wBoardRev= (bData >>5); // board rev is 3 topmost bits

   switch (pFI->wBoardRev)
      {
      case PAS_VERSION_1:
      //case PAS_PLUS:           // same boardrev as PAS_SIXTEEN
      case PAS_SIXTEEN:
      case PAS_STUDIO:           // ProAudio Studio 16
		case PAS_CDPC_LC:				// aka Memphis
		case PAS_BASIC:				// PAS Basic w/508-B mixer
      case PAS_CDPC:
         break;

      default:
         #ifdef DEBUG_CHK
         StringOut ("bad board id");
         #endif
         goto VerifyFailed;     // unknown hardware type
      } // end switch

   /*
   ** try changing version bits
   ** they should be read only
   */
   PAS_OUT( INTERRUPT_CTRL_REG, (BYTE) (bData ^ (BYTE)0xE0));
   bTemp=PAS_IN (INTERRUPT_CTRL_REG);

   if ((bTemp & (D7+D6+D5)) != (bData & (D7+D6+D5)))
      {
      PAS_OUT (INTERRUPT_CTRL_REG, bData);     // Excuse me, stranger.
      #ifdef INIT_MONITOR
      StringOut( "writeable version");
      #endif
      goto VerifyFailed;
      }

   if (pFI->wBoardRev==PAS_VERSION_1)
      {
      pFI->Caps.CapsBits.CDInterfaceType=SCSI_TYPE;

      // test for Enhancecd SCSI mod (U48)

      if ( RMHELP_GrabPort( ENHANCED_SCSI_DETECT_REG) != RMRC_SUCCESS )
      {
         // SOMEONE ALREADY HAS CLAIMED THIS PORT EXCLUSIVE, SO CAN'T VERIFY
         goto VerifyFailed;
      }

      PAS_OUT ( ENHANCED_SCSI_DETECT_REG, 0);    // try changing version bits
      // KeStallExecutionProcessor(10); // wait 10 us
      bTemp = PAS_IN (ENHANCED_SCSI_DETECT_REG);  // they should be read only

      switch (bTemp & 1)                        // bit0==1 means old SCSI PAL
         {
         case 0:
            pFI->Caps.CapsBits.EnhancedSCSI = TRUE;
            // allow to fall thru

         case 1:
            goto ProVerified;
         } // end switch
      }

   else
      {
      // if PAS hardware installed, the reset bit can never be on

      if ( (RMHELP_GrabPort( SYSTEM_CONFIG_1) != RMRC_SUCCESS)
                                  ||
           (RMHELP_GrabPort( SLAVE_MODE_READ) != RMRC_SUCCESS)   )
      {
         // SOMEONE ALREADY HAS CLAIMED THIS PORT EXCLUSIVE, SO CAN'T VERIFY
         goto VerifyFailed;
      }


      bTemp=PAS_IN (SYSTEM_CONFIG_1);           // get PAS config register
      if (bTemp & D7)                           // D7 is reset bit
         goto VerifyFailed;

      bTemp = PAS_IN (SLAVE_MODE_READ);

      if (bTemp & SLAVE_MODE_OPL3)
         pFI->Caps.CapsBits.OPL_3 = TRUE;

      if (bTemp & SLAVE_MODE_16)
         {
         pFI->Caps.CapsBits.DAC16   = TRUE;
         pFI->Caps.CapsBits.DualDAC = TRUE;

         // if 16-bit DAC, and not a CDPC, it has a 508 chip.
         // Note: PAS 16 w/ VGA will have Mixer 508 also.

         if (pFI->wBoardRev != PAS_CDPC)
            pFI->Caps.CapsBits.Mixer_508 = TRUE;
         }

      pFI->Caps.CapsBits.CDInterfaceType=(bTemp & (D1+D0));

      if (pFI->Caps.CapsBits.CDInterfaceType==SCSI_TYPE)
         pFI->Caps.CapsBits.SCSI_IO_16=TRUE;

      pFI->Caps.CapsBits.Slot16       = TRUE;
      pFI->Caps.CapsBits.SoundBlaster = TRUE;


      if ( RMHELP_GrabPort( MASTER_MODE_READ) != RMRC_SUCCESS )
      {
         // SOMEONE ALREADY HAS CLAIMED THIS PORT EXCLUSIVE, SO CAN'T VERIFY
         goto VerifyFailed;
      }

      bTemp=PAS_IN (MASTER_MODE_READ);          // get slave bits
      if ((bTemp & D0)==0)
         pFI->Caps.CapsBits.MCA=TRUE;

      if (bTemp & D2)
          pFI->Caps.CapsBits.CDPC = TRUE;

      if ( RMHELP_GrabPort( CHIP_REV) != RMRC_SUCCESS )
      {
         // SOMEONE ALREADY HAS CLAIMED THIS PORT EXCLUSIVE, SO CAN'T VERIFY
         goto VerifyFailed;
      }
      pFI->wChipRev = PAS_IN (CHIP_REV);

      if (pFI->wChipRev >= CHIP_REV_D)
         {

         if ( (RMHELP_GrabPort( EMULATION_ADDRESS_POINTER) != RMRC_SUCCESS)
                                          ||
              (RMHELP_GrabPort( COMPATIBLE_REGISTER_ENABLE) != RMRC_SUCCESS)
                                          ||
              (RMHELP_GrabPort( IO_PORT_CONFIG_3) != RMRC_SUCCESS)
                                          ||
              (RMHELP_GrabPort( IO_PORT_CONFIG_2) != RMRC_SUCCESS) )
         {
            // SOMEONE ALREADY HAS CLAIMED ONE OR MORE OF THESE PORT EXCLUSIVE,
            // SO CAN'T VERIFY
            goto VerifyFailed;
         }

         bData = PAS_IN (EMULATION_ADDRESS_POINTER);
         bTemp = PAS_IN (COMPATIBLE_REGISTER_ENABLE);

         if (bTemp & MPU_ENABLE_BIT)            // MPU emulation Enabled?
            pFI->MPUPort=0x300 + (bData & 0xf0);

         if (bTemp & SB_ENABLE_BIT_REV_D)       // MPU emulation Enabled?
            {
            pFI->SBPort=0x200 + ((bData & 0x0f)<<4);
            }

         // Report back IRQ usage of PAS DAC and CD
         bData=PAS_IN (IO_PORT_CONFIG_3);

         pFI->ProIRQ = IRQTable[bData & 0x0f];    // convert to IRQ value
         pFI->CDIRQ = IRQTable[bData >> 4];       // convert to IRQ value

         // Report back DMA usage of PAS
         bData = PAS_IN (IO_PORT_CONFIG_2);
         pFI->ProDMA = DMATable[bData & (D2+D1+D0)];

         // Note: Rev D doesn't allow readback of SB IRQ/DMA pointers
         //       nor the MPU IRQ.  The "Set and forget" feature, we
         //       call it.
         }
      } // end else

   ProVerified:

   pFI->ProPort=port;                           // found at this port
   pFI->wTranslateCode=(port ^ DEFAULT_BASE);
   #ifdef INIT_MONITOR
   PrintfOut ("Verified Pro Hardware at %0x",port);
   #endif
   return TRUE;

   VerifyFailed:
   RMHELP_DeAllocPortsToMarker();         // RELEASE ALL PORTS
   pFI->wBoardRev   = 0;                        // found at this port
   pFI->Caps.dwCaps = 0;                        // No Board, No Caps
   return FALSE;
}

/*                                      ------------------ VerifyNothingThere -
** Tries to determine whether a quadport is vacant
**
** Entry Conditions:
**    port = Base I/O address to check
**
** Exit Conditions:
**    Returns TRUE  if nothing found
**    Returns FALSE if it looks like hardware is there
*/
BOOL VerifyNothingThere (WORD port)
{
   BYTE    b0,b1,b2,b3;

   FPwTranslateCode=0;     // PAS_IN macro needs this to be in scope


   if ( RMHELP_AllocPorts( port, 4, RS_IO_EXCLUSIVE) != RMRC_SUCCESS )
   {
      return TRUE;  // SOMEONE ALREADY HAS CLAIMED PORTS
   }

   b0=PAS_IN(port+0);
   b1=PAS_IN(port+1);
   b2=PAS_IN(port+2);
   b3=PAS_IN(port+3);

   if (b0==b1 && b1==b2 && b2==b3)
   {
      RMHELP_DeAllocLastPort();  // RELEASE ACCESS TO THESE PORTS
      return TRUE;
   }
   else
      return FALSE;
}


/*                                      --------------------- WakeUpAtAddress -
** Tries to wake up sleeping relocatable hardware at a specified
** address.  Does not check for hardware already in that location
** If it does wake up a card, it does the minimum amount of
** initialization to enable the hardware.
**
** Entry Conditions:
**     wPort= Base I/O address to wake card up at.
**
** Exit Conditions:
**     Returns TRUE if wake up successful, FALSE otherwise
*/
BOOL WakeUpAtAddress (WORD wPort, FOUNDINFO _far *pFoundInfo)
{
   WORD    i,j;
   FPwTranslateCode = 0;   // PAS_IN macro needs this to be in scope


   if ( RMHELP_GrabPort( PAS_2_WAKE_UP_REG) != RMRC_SUCCESS )
   {
         // SOMEONE ALREADY HAS CLAIMED THIS PORT EXCLUSIVE, SO CAN'T VERIFY
      return( FALSE);
   }

   for (i=0; i<MAXCARDS-1; i++)
      {
      #ifdef INIT_MONITOR
      PrintfOut ("Board ID #%d wakeup attempt...", i, wPort);
      #endif

      for (j=0; j<20; j++)
         {
         PAS_OUT(PAS_2_WAKE_UP_REG, SignatureTable[i]);
         PAS_OUT(PAS_2_WAKE_UP_REG, (BYTE) ((wPort >> 2)& 0xFF));
         }

      if (VerifyProHardware(wPort,pFoundInfo))
      goto FoundAtAddress;
      } // end for


   RMHELP_DeAllocLastPort();  // RELEASE ACCESS TO THESE PORTS
   return (FALSE);         // not found

   FoundAtAddress:
   #ifdef INIT_MONITOR
   PrintfOut("Board ID #%d woken up at address %0x!!!", i, wPort);
   #endif

   FPwTranslateCode = pFoundInfo->wTranslateCode;
   pFoundInfo->Caps.CapsBits.Did_HW_Init = TRUE;

   // don't turn on rev C SB enable or FM enable.  Done in INITC.C
   if (pFoundInfo->wBoardRev > PAS_VERSION_1)
   {
      if ( RMHELP_GrabPort( FEATURE_ENABLE) != RMRC_SUCCESS )
      {
            // SOMEONE ALREADY HAS CLAIMED THIS PORT EXCLUSIVE, SO CAN'T VERIFY
         return( FALSE);
      }

      PAS_OUT (FEATURE_ENABLE, PCM_FEATURE_ENABLE | MIXER_FEATURE_ENABLE);
   }

   return (TRUE);
}

/*                                      ------------------ VerifyLegalAddress -
** Tests a caller-nominated base port address for being a legal
** place for a relocatable PAS to reside.
**
** Entry Conditions:
**     wPort= Base I/O address to check
**
** Exit Conditions:
**     Returns AX= TRUE if the address is legal
**     Returns AX= FALSE otherwise
*/
BOOL VerifyLegalAddress (WORD wPort)
{
   if ((wPort <0x240) || (wPort > 0x3c0) || (wPort & 0x3))
      return FALSE;
   else
      return TRUE;
}
