/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************
* util.c - String and other utility functions for 16 and 32 bit device drivers
*
*
* The following IBM OS/2 source code is provided to you solely for the
* the purpose of assisting you in your development of OS/2 device drivers.
* You may use this code in accordance with the IBM License Agreement
* provided in the IBM Device Driver Source Kit for OS/2.
*******************************************************************************
*
* DEPENENCIES:
*    _VDD_ or _PDD_ need to be defined to pre-processor.
*    Use -d_VDD_ or -d_PDD_ on options to compiler.
*
******************************************************************************/

#include <stdlib.h>
#include <string.h>

#include <os2.h>        // type definitions
#include "ddstring.h"

//
// Define functions that are often in the C runtime.
// Issolating them here insulates the using C code from
// specifics of a particular C compiler.
// Code changes for moving code between intrinsic implementations
// or resorting to coding a runtime routine can be made here, once,
// to insulate the device driver code from compiler specifics.
//

                                        //------------------------- dd_memcpy -
void *dd_memcpy (void *__s1, const void *__s2, size_t __n)
{
   #pragma inline (memcpy);
   return (memcpy (__s1, __s2, __n));
}

                                        //------------------------ dd_fmemcpy -
void __far *dd_fmemcpy (void __far *__s1, const void __far *__s2, size_t __n)
{
   #pragma inline (_fmemcpy);
   return (_fmemcpy (__s1, __s2, __n));
}

                                        //------------------------- dd_memset -
void *dd_memset (void *__s, int __c, size_t __n)
{
   #ifdef _PDD_
   #pragma inline (memset);
   return (memset (__s, __c, __n));  // Use intrinsic version (16-bit compiler)
   #else
   // Could be done more efficiently.
   // For this VDD, below is adequate.
   int I;
   char *s = __s;
   for (I = 0; I < __n; I++, s++)
      *s = __c;
   return (__s);
   #endif
}

                                        //------------------------- dd_memchr -
void *dd_memchr (const void *__s, int __c, size_t __n)
{
   #ifdef _PDD_
   return (memchr (__s, __c, __n));  // Use intrinsic version (16-bit compiler)
   #else
   // search for the character - return position if found
   int I;
   char *s = __s;
   I = 0;
   while (I < __n)
      {
      if (*s == __c)
         return (s);
      s++;
      I++;
      }

   // character not found
   return (NULL);
   #endif
}

                                        //------------------------ dd_toupper -
#pragma intrinsic (dd_toupper);
int dd_toupper (int ch)
{
   if (ch >= 'a' && ch <= 'z')
      ch &= 0xDF;  // ASCII dependent
   return (ch);
}

                                        //------------------------- dd_strcpy -
char *dd_strcpy (char *str1P, char *str2P)
{
   return (strcpy(str1P, str2P));
}

                                        //------------------------ dd_strncpy -
char *dd_strncpy (char *__s1, const char *__s2, size_t __n)
{
   int i;
   char *RetVal = __s1;

   for (i=0; i < __n; __s1++, __s2++, i++)
      {
      *__s1 = *__s2;
      if (*__s1 == '\0')
         break;
      }
   for (; i < __n; __s1++, i++)                 // Pad with nulls
      *__s1 = '\0';

   return (RetVal);
}

                                        //------------------------- dd_strcat -
char *dd_strcat (char *__s1, const char *__s2)
{
   #pragma inline (strcat);
   return (strcat (__s1, __s2));  // Use intrinsic version (16-bit compiler)
}

                                        //------------------------- dd_strcmp -
int dd_strcmp (const char *__s1, const char *__s2)
{
   #ifdef _PDD_
   #pragma inline (strcmp);
   return (strcmp (__s1, __s2));  // Use intrinsic version (16-bit compiler)
   #else
   for (; *__s1 == *__s2; __s1++, __s2++)
      if (*__s1 == '\0')
         return (0);
   return (*__s1 - *__s2);
   #endif
}


                                        //------------------------ dd_stricmp -
int dd_stricmp (const char *__s1, const char *__s2)
{
   for (; dd_toupper(*__s1) == dd_toupper(*__s2); __s1++, __s2++)
      if (*__s1 == '\0')
         return (0);
   return (dd_toupper(*__s1) - dd_toupper(*__s2));
}

                                        //------------------------ dd_strncmp -
int dd_strncmp (const char *__s1, const char *__s2, size_t __n )
{
   int i;

   for (i=0; (i < __n && *__s1 == *__s2); __s1++, __s2++, i++)
      if (*__s1 == '\0')
         return (-1);   // String S1 must be at least n characters

   if (i = __n)
      return (0);
   return (*__s1 - *__s2);
}

                                        //----------------------- dd_strincmp -
int dd_strincmp (const char *__s1, const char *__s2, size_t __n)
{
   int i;

   for (i=0;
        i < __n && (dd_toupper(*__s1) == dd_toupper(*__s2)); __s1++, __s2++,
        i++)
      {
      if (*__s1 == '\0')
         return (-1);   // String S1 must be at least n characters
      }

   if (i = __n)
      return (0);
   return (dd_toupper(*__s1) - dd_toupper(*__s2));
}


//                                      -------------------------- SplitToken -
// Given token which is first element of an ASCII-Z string,
// parse the token to separate the parts left and right of the colon.
// This can be useful for parsing config.sys command line options.
// Numeric parameters are converted as HEX.
//
void SplitToken (char *pszLine, PPARSEDTOKEN pParsedToken)
{
   char   Ch;
   INT    I;
   USHORT usRC;

   dd_memset (pParsedToken, '\0', sizeof (PARSEDTOKEN));

   // Copy all characters before the colon
   Ch = dd_toupper (*pszLine);
   I = 0;
   while (Ch > ' ' && Ch != ':' && I < MAXTOKENSIZE)
      {
      pParsedToken->szKeyword[I] = Ch;          // Copy to output string
      pszLine++;
      Ch = dd_toupper (*pszLine);
      I++;
      }

   if (*pszLine == ':')
      {
      pParsedToken->fColonFound = TRUE;
      pszLine++;
      }

   usRC = ascii_to_ushort (pszLine, &pParsedToken->usValue, 16);
   pParsedToken->fNumericFound = (usRC == 0);

   // Copy the characters right of the colon
   Ch = dd_toupper (*pszLine);
   I = 0;
   while (Ch > ' ' && I < MAXTOKENSIZE)
      {
      pParsedToken->szValue[I] = Ch;        // Copy to output string
      pszLine++;
      Ch = dd_toupper (*pszLine);
      I++;
      }

   return;
}


/*                                      --------------------------- NextToken -
** Adjust callers variable to skip to the beginning
** of the next token of the string being parsed.
*/
void NextToken (char **ppSt)
{
   char *pSt;

   pSt = *ppSt;                                 // Dereference pointer
   while (*pSt > ' ')                           // Skip rest of current token
      pSt++;

   while (*pSt > '\0' && *pSt <= ' ')           // Skip spaces
      pSt++;

   *ppSt = pSt;                                 // Adjust callers variable
}

//                                      --------------------- ascii_to_ushort -
// This function converts an ascii string representing a
// number of the given base to an unsigned short integer.
// String token must be only digits of the indicated base and
// must be terminated by space character or nul.
//
// The parameter 'srcP' is a pointer to the ascii string.
// The function returns the converted value in the parameter 'numP'.
//
// RETURNS
//    0 - Successful conversion
//   -1 - String does not represent a number
//
static char Digits[] = {"0123456789ABCDEF"};
USHORT ascii_to_ushort (char *srcP, USHORT *numP, USHORT usBase)
{
   USHORT x;
   char *chrP;

   *numP = 0;
   if (usBase > 16)
      return (-1);

   // skip leading spaces
   while (*srcP != '\0' && *srcP <= ' ')
      srcP++;

   // skip leading zeros
   while (*srcP == '0')
      srcP++;

   // accumulate digits - return error if unexpected character encountered
   x = 0;
   while (*srcP > ' ')
      {
      if ((chrP = dd_memchr(Digits, *srcP, usBase)) == NULL)
         return (-1);
      x = (x * usBase) + (chrP - Digits);
      srcP++;
      }
   *numP = x;

   // return successful
   return (0);
}
