/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************
* audioreq.c - Request hardware access (also, release)
*
*
* The following IBM OS/2 source code is provided to you solely for the
* the purpose of assisting you in your development of OS/2 device drivers.
* You may use this code in accordance with the IBM License Agreement
* provided in the IBM Device Driver Source Kit for OS/2.
*******************************************************************************
*
* This file contains routines to coordinate access to the shared hardware.
* Contention is managed with the Physical Device Driver (OS/2 sessions)
* and DOS sessions.
* When hardware is not available, a PopUp is displayed informing the
* user of the conflict.  Different actions are taken based on user direction.
*
******************************************************************************/

#pragma data_seg ("_DATA");

#include <mvdm.h>
#include <utilmid.h>                            // for MSG_MM_DEVICE_IN_USE
#include <idc_vdd.h>
#include "audiovdd.h"
#include "data.h"
#include "ddstring.h"
#include "audioreq.h"


//                                      ------------------------------- PopUp -
// Display PopUp window
// This routine is called when a hardware access is denied
// Function asks user for desired action regarding session life.
//
// User has a few choices:
//    Abort  - Terminate this DOS session
//    Retry  - Suspend this DOS session until hardware becomes available
//    Ignore - Tell DOS session that the hardware doesn't exist
//
// RETURNS (eax)
//    VDHP_TERMINATE_SESSION - Terminate session has been ordered
//    VDHP_RETRY             - Session should block till hardware is available
//    VDHP_IGNORE            - Flags were set to virtualize hardware as
//                             not present
ULONG PopUp (void)
{
   ULONG ulRC;
   char  *pValue;

   ulRC = 0;

   pValue = (char *) VDHQueryProperty (szPropNameAudio);
   if (pValue)
      {
      if (dd_strcmp (pValue, szEnumIfAvail) == 0) // If DOS property is set to
         {                                      // "use, if free", then ignore
         ulRC = VDHP_IGNORE;                    // hardware. Otherwise ask user
         }                                      // for direction via popup.
      VDHFreeMem (pValue);
      }

   if (ulRC != VDHP_IGNORE)
      {
      VDHPopup (szAudioDDName,
                1,
                MSG_MM_DEVICE_IN_USE,
                &ulTemp,                        // Variable in data segment
                VDHP_TERMINATE_SESSION + VDHP_RETRY + VDHP_IGNORE,
                0);
      ulRC = ulTemp;
      }

   if (ulRC == VDHP_TERMINATE_SESSION)          // Result of pop-up
      {
      VDHKillVDM (HVDM_Current);
      }

   return (ulRC);
}

//                                      -------------------- MuxtexSemIsOwned -
// Returns TRUE if another DOS session already owns the hardware.
//
BOOL MuxtexSemIsOwned (void)
{
   VDHQuerySem (hsemMutex, &SemState);
   return (SemState.vss_fWaiter || SemState.vss_fOwned); // Any others waiting?
}

//                                      ----------------- GainOwnershipForVDD -
// Returns true if the VDD succeeds in gaining hardware ownership.
// On entry, it is assumed that the VDD does not own the hardware.
// This routine will not block
//
BOOL GainOwnershipForVDD (void)
{
   BOOL fRC;
   fRC = pAudioDD (PDDFUNC_OPENDEVICE, 0, 0);  // RC 0==deny  1==honor
   return (fRC);
}


//                                      ------------------------- GrantAccess -
// Routine called after DOS session has gained access to hardware.
// Disable further I/O port traps.
//
void GrantAccess (void)
{
   pWakeupDD  (IDCSETMODE, MODESOUNDBLASTER, 0);

   hvdm_Owning = HVDM_Current;
   pdb_Owning = PDB_Current;
   VDHSetIOHookState (HVDM_Current,
                      pWakeupSpecs->usSBBase,
                      pWakeupSpecs->usSBRange,
                      &iohHookInfo,
                      FALSE);
}


//                                      ----------------------- ReleaseAccess -
// Routine called when a DOS session/application that
// owns hardware terminates.
// That is, the DOS session relinquishes the adapter.
//
void ReleaseAccess (void)
{
   hvdm_Owning = 0;                             // Temporarily, no DOS session
   pdb_Owning = 0;                              // owns the hardware

   //
   // The Sound Blaster using app is terminating.
   // Insure that the FM chips are quiet
   //
   pWakeupDD  (IDCOPL3ALLNOTESOFF, 0, 0);

   // Re-enable port trapping
   VDHSetIOHookState (HVDM_Current,
                      pWakeupSpecs->usSBBase,
                      pWakeupSpecs->usSBRange,
                      &iohHookInfo,
                      TRUE);
   VDHReleaseMutexSem (hsemMutex);

   // We have released the mutex semaphore.  Any other DOS sessions
   // waiting on the device will now compete for the mutex sem.

   if (MuxtexSemIsOwned())                      // Any others waiting?
      return;                                   // Yes, were done

   //
   // No other waiters - return hardware to the PDD
   //
   pWakeupDD  (IDCSETMODE, MODENATIVE, 0);
   pAudioDD (PDDFUNC_CLOSEDEVICE, 0, 0);
}



//                                      ----------------------- RequestAccess -
// This routine is called the first time a DOS session touches
// the shared hardware.  This is taken as a request for ownership.
// This routine coordinates ownership with the PDD and other DOS sessions.
// When granted, subsequent traps are disabled giving the session
// direct access to the hardware.
// When denied, a pop-up is displayed giving the user notification of
// the conflict.  The session is then either terminated, suspended, or
// a flag is set to act like the hardware doesn't exist.
//
//
// Two things have to happen before the VDD can give ownership to a VDM.
// First, the VDD itself must gain ownership from the PDD.
// Second, no other VDM can already have the device.
//
// Rather than maintain lists, two semaphores are used to maintain access.
// The event semaphore is used to block all waiting DOS sessions
// until the event of the VDD gaining ownership from the PDD occurs.
// When the VDD has ownership, the DOS sessions compete for ownership
// of the mutex semaphore.  The one that owns the semaphore uses the device.
// The others block until they own the semaphore.
// The kernel semaphore rules govern which VDM next gets the hardware.
// Experimentation shows it to be first come, first served.
//
// RETURNS
//    TRUE  - Access granted
//    FALSE - Denied
//
BOOL RequestAccess (void)
{
   BOOL  fVDDOwnsHardware = FALSE;
   BOOL  fPopupDisplayed  = FALSE;

   //
   // Step one: Insure VDD owns the device
   //
   fVDDOwnsHardware = MuxtexSemIsOwned();       // Any other VDM own the H/W?
   if (! fVDDOwnsHardware)
      fVDDOwnsHardware = GainOwnershipForVDD(); // Get ownership from PDD

   if (! fVDDOwnsHardware)
      {
      if (PopUp() != VDHP_RETRY)                // Display PopUp
         return (FALSE);
      fPopupDisplayed = TRUE;
      fVDDOwnsHardware = MuxtexSemIsOwned();    // Others may die during popup?
      }

   if (! fVDDOwnsHardware)
      {
      //
      // Suspend session until VDD gains hardware ownership
      // Note, while suspended, user or sytem may kill the session.
      // If this happens, control will return here with an unusual
      // return code on the VDH call - we return immediately to give
      // the MVDM kernel a chance to kill the session.
      //
      if (VDHWaitEventSem (hsemEvent, SEM_INDEFINITE_WAIT) == FALSE)
         return (FALSE); // Semaphore error, session being terminated
      }

   //
   // Step two: VDD gives ownership to only one DOS session
   //
   if (MuxtexSemIsOwned() && !fPopupDisplayed)  // Hardware busy?
      {
      if (PopUp() != VDHP_RETRY)                // Display PopUp
         return (FALSE);
      }

   //
   // Compete with other DOS sessions to gain/ ownership of the device.
   //
   // Like the Event semaphore case, the user or system may decide to
   // kill the session that is blocked on the semaphore.  Unlike the
   // event semaphore and while the specs say it can, the OS never fails
   // the Mutex event semaphore call!  Below is coded per spec anyway.
   //
   if (VDHRequestMutexSem (hsemMutex, SEM_INDEFINITE_WAIT) == FALSE)
      return (FALSE); // Semaphore error, session being terminated

   //
   // Step three: Prevent further I/O traps
   //
   pWakeupDD  (IDCSETMODE, MODESOUNDBLASTER, 0);
   GrantAccess ();
   return (TRUE);
}

//                                      ---------------- IDCEntryFromAudioPDD -
// Inter-Device Driver communication entry point.
// Routine is called from PDD to coordinate shared hardware
// access and for interrupt virtualization.
//
#pragma off (unreferenced);
BOOL pascal IDCEntryFromAudioPDD_c (ULONG    ulFunc,
                                    F16PVOID f16p1,
                                    F16PVOID f16p2)
{
   if (ulFunc != VDDFUNC_CLOSE)                 // Support only the close call
      return (FALSE);

   // The PDD no longer needs the hardware and is giving us a chance
   // to restart DOS sessions that were blocked pending availability.

   VDHQuerySem (hsemEvent, &SemState);
   if (! SemState.vss_fWaiter)
      return (TRUE);

   // At least one DOS session is waiting for
   // the VDD to gain access to the hardware.

   if (GainOwnershipForVDD() == FALSE)
      {
      // This is unlikely, but another OS/2 session got H/W
      return (TRUE);
      }

   //
   // VDD gained ownership of the device.  Post the event semaphore.
   // Waiting VDMs compete for the mutex
   //
   VDHPostEventSem (hsemEvent);
   return (TRUE);
}
#pragma on (unreferenced);


//                                      --------------- IDCEntryFromWakeupPDD -
// Entrypoint for IDC calls from the wakeup device driver.
// There is no reason the wakeup driver would call the VDD.
// The VDD initiates all operation.
// This function is however required to setup the IDC interface.
//
#pragma off (unreferenced);
BOOL pascal IDCEntryFromWakeupPDD_c (ULONG ulFunc,
                                     F16PVOID f16p1,
                                     F16PVOID f16p2)
{
   return (FALSE);
}
#pragma on (unreferenced);
