/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************
*     WSS.c - Code to program AD1848/CS4231         Sound System device
*
*******************************************************************************
*
* Contains code to parse configuration string and program the
*         Sound System architecture
*
*         Sound System (Analog Devices AD1848, Crystal Semiconductor CS4231)
*
* Code/data in this file is discarded after initialization
*
******************************************************************************/

#define  INCL_16
#define  INCL_DOSINFOSEG
#include <os2.h>

#include <conio.h>              // Watcom intrinsic port i/o routines
#include <dos.h>                // Watcom intrinsic enable/disable interrupts

#include "idc_vdd.h"
#include "data.h"
#include "util.h"
#include "wss.h"
#include "ddprintf.h"
#include <dhcalls.h>            // 'C' callable kernel device helper functions

BOOL    ACALSet = FALSE;

//                                      ---------------------------- Delay_ms -
// Delay at least n milliseconds.
// 31.25 ms tick granular
//
//ULONG ulEvent;
//void Delay_ms (USHORT usMilliseconds)
//{
//   USHORT usRC = WAIT_INTERRUPTED;
//
//   _disable ();
//   while (usRC != WAIT_TIMED_OUT)
//      {
//      usRC = DevHelp_ProcBlock ((ULONG)&ulEvent, usMilliseconds, 1);
//      _disable ();
//      }
//   _enable ();
//
//   return;
//}

BOOL WSS_IsValidBaseAddr (USHORT usBaseAddr)
{
   return (usBaseAddr == 0x530 ||       // Valid values per MAD 16 capabilities
           usBaseAddr == 0xE80 ||
           usBaseAddr == 0xF40 ||
           usBaseAddr == 0x604);
}


//                                      ------------------- WSS_WaitForResync -
// When clearing the mode change enable bit and
// when writing the clock and data format register,
// the part requires some time to recover.
//
// Wait for msb of Index register to read as a zero
//
USHORT WSS_WaitForResync (VOID)
{
   INT    I;
   BYTE   bIn;

   //
   // Wait for msb of Index register to read as a zero
   //
   for (I = 0; I < 3000; I++)
      {
      bIn = inp (Specs.usWSSIndex);
      if ((bIn & 0x80) == 0x00) // High bit off?
         return (0);            // Yes, done waiting
      }

   return (1);                  // Timed out
}



//                                      ------------ WSS_WaitForAutoCalibrate -
// After clearing MCE, the part begins calibration.
// This routine watches the device to wait for calibration
// to complete.
//
// The Autocalibrate-In-Progress bit will transition
// from LO to HIGH within 5 clock periods and will
// transition back LO after approximately 384 periods.
//
// This operation can take a long time.  For the worst case:
// (5+384) / (5.5125*1000) = 70.57 milliseconds
//      5  / (5.5125*1000) =   .001 milliseconds = 1.00 microsecond
//
USHORT WSS_WaitForAutoCalibrate (VOID)
{
   USHORT usRC = 1; // Assume timeout
   INT    I;
   BYTE   bIn;

   //
   // In the first few clocks, the ACI bit will
   // transition from LO to HIGH.  Then it will again go low.
   // To make the code debuggable, delay a bit then wait
   // for the line to go low.
   //

   usRC = 1;

   for (I = 0; I < 3; I++)
      bIn = inp (Specs.usWSSIndex);     // Around 3 microseconds each

   for (I = 0; I < 100; I++)
      {
      bIn = WSS_ReadIndirect (WSS_TEST_AND_INITIALIZATION);
      if ((bIn & 0x20) == 0x00)
         {
         usRC = 0;
         break;
         }
      }

   return (usRC);
}


                                        //------------------------ WSS_SetMCE -
VOID WSS_SetMCE (VOID)
{
   outp (Specs.usWSSIndex, 0x40);         // Set mode change enable bit (bit-6)
   IOdelay ();
}


USHORT WSS_ClearMCE (VOID)
{
   USHORT usRC;

   outp (Specs.usWSSIndex, 0x00);         // Clear mode change enable bit (bit-6)
   usRC = WSS_WaitForResync();
   if (usRC != 0)
      return (usRC);

   if (ACALSet)    // if ACAL bit is on in the interface config register
      usRC = WSS_WaitForAutoCalibrate();

   return (usRC);
}

//                                      ----------- WSS_SetClockAndDataFormat -
// Set the WSS clock and data format register.
// ClockRate parm must be either 11025 or 44100.
//
USHORT WSS_SetClockAndDataFormat (USHORT usClockRate)
{
   BYTE bTemp;
   USHORT usRC;

   if (usClockRate == 11025)    // OPTi part requires XTAL2 (bit-0 = 1)
      bTemp = 0x03;
   else
      bTemp = 0x0B; // 44100

   WSS_SetMCE ();
   WSS_WriteIndirect (WSS_CLOCK_AND_DATA_FORMAT, bTemp);
   usRC = WSS_WaitForResync();
   if (usRC != 0)
      {
      WSS_ClearMCE();
      return (usRC);
      }
   usRC = WSS_ClearMCE();
   return (usRC);
}


//                                      -------------- WSS_SetInterfaceConfig -
// Set the WSS interface configuration register
//
USHORT WSS_SetInterfaceConfig (BYTE Value)
{
   USHORT usRC = 0;

   WSS_SetMCE ();
   WSS_WriteIndirect (WSS_INTERFACE_CONFIGURATION, Value);
   ACALSet = ((Value & 0x40) != 0);
   usRC = WSS_WaitForResync();
   if (usRC != 0)
      {
      WSS_ClearMCE();
      return (usRC);
      }
   usRC = WSS_ClearMCE();

   return (usRC);
}


//                                      ------------------- WSS_QueryPartType -
// Query WSS chip grade and reset part
//
USHORT WSS_QueryPartType (VOID)
{
   BYTE   bTemp;
   USHORT usRC;

   usRC = WSS_WaitForResync ();                 // Make sure part is ready
   if (usRC != 0)
      return (usRC);

   // Query type of CODEC (CS or AD and grade)
   bTemp = WSS_ReadIndirect (WSS_MISCELLANEOUS_INFO);

   // Query J-GRADE or K-GRADE
   if ((bTemp & 0x0F) == 0x09)
      Specs.usWSSGrade = 'J';
   else
      Specs.usWSSGrade = 'K';

   // If bit-7 is on, CS otherwise AD
   if (bTemp & 0x80)
      Specs.usWSSCodecType = 'C';
   else
      Specs.usWSSCodecType = 'A';

   if (Specs.usWSSCodecType == 'C')
      {
      WSS_WriteIndirect (WSS_MISCELLANEOUS_INFO, 0xCA); // CS4231 Set mode 2
      WSS_WriteIndirect (0x1A, 0xC0);                   // mono in/out mute
      WSS_WriteIndirect (WSS_MISCELLANEOUS_INFO, 0x8A); // Turn off mode 2
      }
   return (0);
}

//                                      ---------------- WSS_EnableInterrupts -
// Program WSS part to enable interrupt generation
//
VOID WSS_EnableInterrupts (VOID)
{
   BYTE bTemp;

   //
   // Read/Modify/Write on Pin Control register
   // Force TTL logic low (pin outs)
   // Interrupt enable
   //
   bTemp = WSS_ReadIndirect (WSS_PIN_CONTROL);
   bTemp &= 0x3F;  // TTL Logic low (bits 7..6)
   bTemp |= 0x02;  // Interrupt enable
   WSS_WriteIndirect (WSS_PIN_CONTROL, bTemp);
}


//                                      ------------------ WSS_MuteEverything -
// Program WSS part to mute all outputs
// Also set attenuation levels for unmute
//
VOID WSS_MuteEverything (VOID)
{
   BYTE bTemp;

   //ddprintf ("MUTING");

   bTemp = WSS_ReadIndirect (WSS_LEFT_OUTPUT_CONTROL) | 0x80;
   WSS_WriteIndirect (WSS_LEFT_OUTPUT_CONTROL, bTemp);
   //ddprintf ("LOC = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_RIGHT_OUTPUT_CONTROL) | 0x80;
   WSS_WriteIndirect (WSS_RIGHT_OUTPUT_CONTROL, bTemp);
   //ddprintf ("ROC = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_LEFT_AUX1_INPUT_CONTROL) | 0x80;
   WSS_WriteIndirect (WSS_LEFT_AUX1_INPUT_CONTROL,  bTemp);
   //ddprintf ("LA1 = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_RIGHT_AUX1_INPUT_CONTROL) | 0x80;
   WSS_WriteIndirect (WSS_RIGHT_AUX1_INPUT_CONTROL, bTemp);
   //ddprintf ("RA1 = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_LEFT_AUX2_INPUT_CONTROL)  | 0x80;
   WSS_WriteIndirect (WSS_LEFT_AUX2_INPUT_CONTROL,  bTemp);
   //ddprintf ("LA2 = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_RIGHT_AUX2_INPUT_CONTROL) | 0x80;
   WSS_WriteIndirect (WSS_RIGHT_AUX2_INPUT_CONTROL, bTemp);
   //ddprintf ("RA2 = %0x", bTemp);
}

//                                      ---------------- WSS_UnMuteEverything -
// Program WSS part to UnMute all outputs and inputs
//
VOID WSS_UnMuteEverything (VOID)
{
   BYTE bTemp;

   bTemp = 0x99;
   //ddprintf ("UNMUTING");

   bTemp = WSS_ReadIndirect (WSS_LEFT_OUTPUT_CONTROL);
   bTemp &= 0x7F;
   WSS_WriteIndirect (WSS_LEFT_OUTPUT_CONTROL, bTemp);
   //ddprintf ("LOC = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_RIGHT_OUTPUT_CONTROL);
   bTemp &= 0x7F;
   WSS_WriteIndirect (WSS_RIGHT_OUTPUT_CONTROL, bTemp);
   //ddprintf ("ROC = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_LEFT_AUX1_INPUT_CONTROL);
   bTemp &= 0x7F;
   WSS_WriteIndirect (WSS_LEFT_AUX1_INPUT_CONTROL, bTemp);
   //ddprintf ("LA1 = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_RIGHT_AUX1_INPUT_CONTROL);
   bTemp &= 0x7F;
   WSS_WriteIndirect (WSS_RIGHT_AUX1_INPUT_CONTROL, bTemp);
   //ddprintf ("RA1 = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_LEFT_AUX2_INPUT_CONTROL);
   bTemp &= 0x7F;
   WSS_WriteIndirect (WSS_LEFT_AUX2_INPUT_CONTROL, bTemp);
   //ddprintf ("LA2 = %0x", bTemp);

   bTemp = WSS_ReadIndirect (WSS_RIGHT_AUX2_INPUT_CONTROL);
   bTemp &= 0x7F;
   WSS_WriteIndirect (WSS_RIGHT_AUX2_INPUT_CONTROL, bTemp);
   //ddprintf ("RA2 = %0x", bTemp);
}


//                                      ---------------- WSSSetConfigRegister -
// On 82C929, the WSS Configuration register needs to be
// written to clear a     state that prevents the CS4231
// audio driver from loading on warm boots.
// Note, this will clear any previous setting for DMA and IRQ.
// We can safely clear this reg at initialization as the audio
// device driver will set the register during its load.
// WSS Configuration Register is write-only
//
VOID WSSClearConfigRegister (VOID)
{
   outp (Specs.usWSSBase, 0);
}

//VOID WSSSetConfigRegister (VOID)
//{
//   BYTE bVal = 0;
//
//   switch (Specs.usWSSIRQ)              // Set IRQ value via bits 5..3
//      {
//      case  7: bVal = 0x08; break;      // 001
//      case  9: bVal = 0x10; break;      // 010
//      case 10: bVal = 0x18; break;      // 011
//      case 11: bVal = 0x20; break;      // 100
//      }
//   switch (Specs.usWSSPlayDMA)          // Set DMA value via bits 2..0
//      {
//      case  0: bVal |= 1; break;
//      case  1: bVal |= 2; break;
//      case  3: bVal |= 3; break;
//      }
//
//   outp (Specs.usWSSBase, bVal);        // Set MS Sound system config reg
//}

//                                      --------------------------- WSS_Setup -
// Place WSS device in known idle state
//
USHORT WSS_Setup (VOID)
{
   USHORT usRC;

   usRC = WSS_WaitForResync ();                 // Make sure part is ready
   if (usRC != 0)
      return (usRC);

   if (Specs.usMad16PartNum != 928)
      {
      WSSClearConfigRegister();
      }

   WSS_SetMixerState (&WSSMixerState);
   WSS_MuteEverything ();

   usRC = WSS_SetInterfaceConfig (0x08);        // Enable auto-calibrate
   if (usRC != 0)
      return (usRC);

   usRC = WSS_SetClockAndDataFormat ((USHORT)44100);
   if (usRC != 0)
      return (usRC);

   usRC = WSS_SetInterfaceConfig (0x00);        // Disable future auto-calibrat
   if (usRC != 0)                               // Helps avoid clicks when
      return (usRC);                            // games change sample rate
   WSS_EnableInterrupts ();

   WSS_UnMuteEverything ();

   return (0);
}
