/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************
* RMHELP.c - OS/2 Resource Manager communication routines
*
*
* The following IBM OS/2 source code is provided to you solely for the
* the purpose of assisting you in your development of OS/2 device drivers.
* You may use this code in accordance with the IBM License Agreement
* provided in the IBM Device Driver Source Kit for OS/2.
*******************************************************************************
*
* Contains code to communicate with OS/2 resource manager.
*
* It queries the resource manager to attain ownership of
* I/O address space, DMA and interrupt levels.
* After getting ownership of the hardware resource, the device driver
* is clear to wakeup the on-board hardware at the indicated configuration.
*
* The resource manager helps avoid conflicts between device drivers where
* driver may compete for ownership of the same resources (such as I/O space).
* Drivers that load later can query if resources they want are free.
*
* This driver queries resources before waking up the hardware (e.g CD-ROM).
* Should an resource not be free, we can select an alternate address, IRQ
* or resource.
* The CD-ROM device drivers look for their hardware at multiple possible
* base addresses.  With this, if a first choice location is occupied, we
* can wakeup the device at an alternate address - where the CD-ROM device
* driver will find it via auto-scan.  This driver should issue a warning
* message to inform the user of the changed location.
*
* In practice, this device driver loads *very early* in system boot and
* will rarely be denied access to any resource it wants.
* Indeed, this driver must load before the CD-ROM device drivers or there
* will be no functional CD-ROM controller when the CD-ROM driver looks.
* This greatly reduces the chance of collision from the perspective of
* this wakeup device driver.
*
******************************************************************************/

#define INCL_NOPMAPI
#include <os2.h>

#include "rmcalls.h"
#include "rmhelp.h"
#include "idc_vdd.h"
#include "data.h"

DRIVERSTRUCT DriverStruct =
{
   NULL,                                     /* DrvrName                */
   "OPTi 82C928/82C929 MAD16 Wakeup driver", /* DrvrDescript            */
   "OPTi",                                   /* VendorName              */
   CMVERSION_MAJOR,                          /* MajorVer                */
   CMVERSION_MINOR,                          /* MinorVer                */
   1995,2,24,                                /* Date                    */
   DRF_STATIC,                               /* DrvrFlags               */
   DRT_UNKNOWN,                              /* DrvrType                */
   0,                                        /* DrvrSubType             */
   NULL                                      /* DrvrCallback            */
};

/*----------------------------------------------*/
/* Adapter Description  (rmbase.h)              */
/*----------------------------------------------*/
ADAPTERSTRUCT AdapterStruct =
{
  "Wakeup and mode change device driver", /* AdaptDescriptName; */
  AS_NO16MB_ADDRESS_LIMIT,           /* AdaptFlags;        */
  AS_BASE_MMEDIA,                    /* BaseType;          */
  AS_SUB_MM_AUDIO,                   /* SubType;           */
  AS_INTF_GENERIC,                   /* InterfaceType;     */
  AS_HOSTBUS_ISA,                    /* HostBusType;       */
  AS_BUSWIDTH_16BIT,                 /* HostBusWidth;      */
  NULL                               /* pAdjunctList;      */
};

/*----------------------------------------------*/
/* Device Description  (rmbase.h)               */
/*----------------------------------------------*/
DEVICESTRUCT DeviceStruct =
{
  "MAD16 Integrated Digital Audio Controller", /* DevDescriptName   */
  DS_FIXED_LOGICALNAME,             /* DevFlags;         */
  DS_TYPE_AUDIO,                    /* DevType;          */
  NULL                              /* pAdjunctList;     */
};

/*----------------------------------------------*/
/* GLOBAL VARS FOR RM                           */
/*                                              */
/* RM.LIB needs these declared                  */
/*----------------------------------------------*/
PFN             RM_Help               = 0L;  /*VPNP*/
PFN             RM_Help0              = 0L;  /*VPNP*/
PFN             RM_Help3              = 0L;  /*VPNP*/
ULONG           RMFlags               = 0L;  /*VPNP*/

/*----------------------------------------------*/
/* GLOBAL HANDLE VARIABLES                      */
/*                                              */
/* These variables get the handles for          */
/* drivers, adapter (only 1), and resources.    */
/* The RMHELP wrappers are written so that most */
/* drivers using RMHELP as a template will not  */
/* have to reference these directly.            */
/*                                              */
/*----------------------------------------------*/

HDRIVER   hDriver = NULL;      // global handle to driver
HADAPTER  hAdapter = NULL;     // global handle to adapter

USHORT ResPortsCnt = 0;       // Count of res port handles
HRESOURCE hResPorts[ MAX_PORT_HANDLES];    // global handles to port resources

HRESOURCE hResIRQs = NULL;     // global handle to IRQs resource
HRESOURCE hResDMA1 = NULL;     // global handle to DMA resource 1
HRESOURCE hResDMA2 = NULL;     // global handle to DMA resource 2


//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//                                                          !
// NOTE: FOR THE AUDIO DRIVERS AD1848 and COMAUDIO          !
//       THIS CODE WILL BE SET UP TO BE DISCARDED           !
//       AFTER INIT SINCE THESE CALLS ONLY ARE              !
//       DONE DURING INIT TIME. FOR OTHER DRIVERS           !
//       THE PROGRAMMER WILL HAVE TO DETERMINE              !
//       IF AND HOW TO DISCARD THE CODE.                    !
//                                                          !
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

/*--------------------------------------------------------*/
/* RMHELP_CreateDriver - registers to RM susbsystem       */
/*                       (which initializes the DD to     */
/*                        RM connection)                  */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* PSZ DrvrName - ptr to driver name (taken from command  */
/*                line)                                   */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* This function is a thin wrapper for the RMCreateDriver */
/* call. Modifications for specific drivers might include */
/* adding more parameters to be set into the              */
/* DriverStruct and actually returning the hDriver        */
/* to the caller (instead of maintaining it in a global   */
/* var). In any case, modifications to the DriverStruct   */
/* declared above will have to made.                      */
/*--------------------------------------------------------*/

USHORT RMHELP_CreateDriver (PSZ DrvrName)
{
   DriverStruct.DrvrName = DrvrName;   // set the driver name

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   return (RMCreateDriver (&DriverStruct, &hDriver));

}

/*--------------------------------------------------------*/
/* RMHELP_DestroyDriver - deregisters the driver from     */
/*                        RM sub-system and releases all  */
/*                        claimed resources               */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* This function is a thin wrapper for the RMDestroyDriver*/
/* call. Modifications for specific drivers might include */
/* adding a parameter for hDriver, if RMCreateDriver has  */
/* been modified to return it to the DD layer.            */
/*                                                        */
/*--------------------------------------------------------*/

VOID RMHELP_DestroyDriver (VOID)
{
   RMDestroyDriver (hDriver);
}

/*--------------------------------------------------------*/
/* RMHELP_CreateAdapter - tells RM to associate adapter   */
/*                        to driver and resources to      */
/*                        adapater                        */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* This function builds up the AHRESOURCE array for the   */
/* the call to RMCreateAdapter. To modify this for a      */
/* specfic DD the handles to resource variables set into  */
/* the array will have to be touched along with the lines */
/* that set the NumResource. Also, AdapterStruct (see     */
/* above) will have to be set up apropriately.            */
/*                                                        */
/*--------------------------------------------------------*/

USHORT RMHELP_CreateAdapter (VOID)
{
   UCHAR       ResourceBuf[ MAX_RES_BUF_SIZE];
   PAHRESOURCE pResourceList = (PAHRESOURCE)ResourceBuf;
   USHORT ResNdx = 0;
   USHORT i;
   USHORT rc;
   HDEVICE hDevice;

   for  (i = 0; i < ResPortsCnt; i++)
   {
      pResourceList->hResource[ ResNdx] = hResPorts[ i];
      ++ResNdx;
   }

   pResourceList->hResource[ ResNdx] = hResIRQs;
   ++ResNdx;

   if  (hResDMA1)
   {
      pResourceList->hResource[ ResNdx] = hResDMA1;
      ++ResNdx;
   }

   if  (hResDMA2)
   {
      pResourceList->hResource[ ResNdx] = hResDMA2;
      ++ResNdx;
   }


   pResourceList->NumResource = ResNdx;

   if  ((rc = RMCreateAdapter (hDriver,
                               &hAdapter,
                               &AdapterStruct,
                               NULL,
                               pResourceList)) != RMRC_SUCCESS)
   {
      return (rc);
   }

   return (RMCreateDevice (hDriver,
                           &hDevice,
                           &DeviceStruct,
                           hAdapter,
                           NULL));
}

/*--------------------------------------------------------*/
/* RMHELP_AllocPorts - attempts to claim a port resource  */
/*                       from RM subsystem                */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* USHORT baseport - base IO port.                        */
/* USHORT nPorts   - number of prts inclusive of base port*/
/* USHORT ShareType - RS_IO_EXCLUSIVE or RS_IO_SHARED     */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* Modifying this for a specific DD might consist of:     */
/*                                                        */
/* 1 Setting the res handle to a variable ptr passed in   */
/* 2 Assigning IOResource members proper values (see      */
/*   rmbase.h for all the allowed values)                 */
/*                                                        */
/*--------------------------------------------------------*/
USHORT RMHELP_AllocPorts (USHORT baseport, USHORT nPorts, USHORT ShareType)
{
   RESOURCESTRUCT Resource;
   USHORT rc;

   if (ResPortsCnt >= MAX_PORT_HANDLES)
   {
      return (RMHELP_ERR_TOOMANYRESPORTS);
   }

   Resource.ResourceType              = RS_TYPE_IO;
   Resource.IOResource.BaseIOPort     = baseport;
   Resource.IOResource.NumIOPorts     = nPorts;
   Resource.IOResource.IOAddressLines = Specs.usMad16DecodeWidth;
   Resource.IOResource.IOFlags        = ShareType;

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   rc = RMAllocResource (hDriver, &hResPorts[ ResPortsCnt], &Resource);

   if  (rc == RMRC_SUCCESS)
   {
      ++ResPortsCnt;
   }

   return (rc);
}

/*--------------------------------------------------------*/
/* RMHELP_DeAllocLastPort - deallocates the last port     */
/*                       acquired from RMHELP_AllocPorts  */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* VOID                                                   */
/*                                                        */
/*--------------------------------------------------------*/
VOID RMHELP_DeAllocLastPort (VOID)
{
   if  (ResPortsCnt)
   {
      --ResPortsCnt;
      RMDeallocResource (hDriver, hResPorts[ ResPortsCnt]);
   }
}

/*--------------------------------------------------------*/
/* RMHELP_AllocIRQ   - attempts to claim a IRQ resource   */
/*                       from RM subsystem                */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* USHORT IRQlevel - IRQ level to claim                   */
/*                                                        */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* Modifying this for a specific DD might consist of:     */
/*                                                        */
/* 1 Setting the IRQ handle to a variable ptr passed in   */
/* 2 Assigning IRQResource members proper values (see     */
/*   rmbase.h for all the allowed values)                 */
/*                                                        */
/*--------------------------------------------------------*/
USHORT RMHELP_AllocIRQ (USHORT IRQlevel)
{
   RESOURCESTRUCT Resource;

   if (hResIRQs)
   {
      return (RMHELP_ERR_RESOURCECLAIMED);
   }

   /*------------------------------------------------*/
   /* Request IRQ Resource  from RM......            */
   /*------------------------------------------------*/
   Resource.ResourceType          = RS_TYPE_IRQ;
   Resource.IRQResource.PCIIrqPin = RS_PCI_INT_NONE;
   Resource.IRQResource.IRQFlags  = RS_IRQ_MULTIPLEXED;

   Resource.IRQResource.IRQLevel = IRQlevel;

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   return (RMAllocResource (hDriver, &hResIRQs , &Resource));
}

/*--------------------------------------------------------*/
/* RMHELP_AllocDMA   - attempts to claim a DMA resource   */
/*                       from RM subsystem                */
/*                                                        */
/* ARGS:                                                  */
/*                                                        */
/* USHORT DMAchannel   - DMA channel to claim             */
/*                                                        */
/*                                                        */
/* HINT (Using this generically):                         */
/*                                                        */
/* Modifying this for a specific DD might consist of:     */
/*                                                        */
/* 1 Setting the DMA handle to a variable ptr passed in   */
/* 2 Assigning DMAResource members proper values (see     */
/*   rmbase.h for all the allowed values)                 */
/*                                                        */
/*--------------------------------------------------------*/
USHORT RMHELP_AllocDMA (USHORT DMAChannel)
{

   PHRESOURCE phResDMA;
   RESOURCESTRUCT Resource;

   if (hResDMA2)
   {
      return (RMHELP_ERR_RESOURCECLAIMED);
   }

   phResDMA = &hResDMA1;
   if (hResDMA1)
   {
      phResDMA = &hResDMA2;
   }

   /*------------------------------------------------*/
   /* Request DMA Resource  from RM......            */
   /*------------------------------------------------*/

   Resource.ResourceType         = RS_TYPE_DMA;
   Resource.DMAResource.DMAChannel = DMAChannel;
   Resource.DMAResource.DMAFlags = RS_DMA_EXCLUSIVE;

   /*---------------------------------------------------*/
   /* Allocate Driver Handle                            */
   /*---------------------------------------------------*/
   return (RMAllocResource (hDriver, phResDMA , &Resource));

}
