/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************
* MAD16.c - MAD16 initialization routines - Wakeup the device, CD and Audio
*
*
* The following IBM OS/2 source code is provided to you solely for the
* the purpose of assisting you in your development of OS/2 device drivers.
* You may use this code in accordance with the IBM License Agreement
* provided in the IBM Device Driver Source Kit for OS/2.
*******************************************************************************
*
* Contains code to parse configuration string and program the MAD16
* device to come to life at the indicated configuration.
*
* Routines and data in this file are discarded after initialization
*
* OPTI Inc 82C928 chip --> MAD 16 Pro
* This chip and accompanying CS4231 and OPL-3 make up Media Magic ISP-16.
* OPTI chip - handles ISA bus interface, Sound Blaster wave emulation,
* various CD-ROM emulation and provides bus interface logic.
* When in Sound Blaster mode, the 82C928 chip receives Sound Blaster
* port operations - translates those to equivalent WSS commands and sends
* to the WSS CODEC (who actually does the work).
* With this, it is not possible to do WSS wave audio and Sound Blaster
* wave audio concurrently.
* On the 82C928 revision B, you also cannot do FM and WSS wave concurrently.
* This problem is solved with revision C.
*
* The 82C930 will support dual wave audio DMA.
* For technical docs on the programming, consult the MediaCHIPS
* MAD16 16-bit Audio Controller docs.
* Version used to write this code: 10/23/93, revision C.
*
* The 82C928 has a concept of shadow registers.
* It turns out that knowing how these work is crucial to programming the part.
*
* Early sound blaster games use programmed I/O to move data.  This programming
* is based on the premise that the Sound Blaster will regulate I/O rate.
* The 82C928 has no concept of this time, but needs some concept of clocking
* to help it regulate the rate at which the data is moved to the WSS part.
* For this, the 82C928 maintains an internal register (shadow register) one of
* which is the sampling rate that it will use for programmed I/O operations.
* This register should be set using set of 44.1 KHz sample rate.
*
* The 82C928 also has no ability to perform Sound Blaster mixing functions.
* This problem is fixed in the 82C929.   For the 82C928, all writes to
* the Sound Blaster mixing registers are merely remembered.
* If software tries to read the set values later, values written are returned.
* NOTE: Nothing actually happens to the mix values on writes to SB registers.
* Under DOS, OPTi provides VOLTSR - Volume TSR - that occasionally reads the
* Sound Blaster mixer registers and if changed, makes a corresponding change
* to the WSS mixer registers.
*
* The path between the OPTi and WSS chip is one-way, write only.
* The shadow registers may or may not be updated on c
* When the sampling rate information or mixing values are changed in the WSS,
* the shadow registers may or may not be updated -- depending on bit in MC5.
* When protection is enabled, the internal (82C928) shadow registers are not
* effected by writes to the WSS device.
* When protection is disabled, the shadow register reflect writes to the
* WSS device.
*
* Once initialized, the protection is left enabled to protect the frequency
* information for Sound Blaster programmed I/O.
* The protection is temporarily disabled when mixer values are changed so that
* the updated value can be reflected to the Sound Blaster side.?
*
* Software watches the switch between WSS and SB modes - on switch, remembers
* the states of the WSS registers and restores when mode is switched back.
* This makes the concepts of mixing remain constant across switches.
*
* On 82c928 revision B, mode switch does inadvertant
*
* NOTES:
*    82C928 decodes 10 bits on ISA bus - makes some WSS base addresses invalid
*    82C929 decodes 12 bits on ISA bus
*
******************************************************************************/

#define  INCL_16
#define  INCL_DOSINFOSEG
#include <os2.h>
#include <dhcalls.h>
#include <conio.h>              // Watcom intrinsic port i/o routines
#include <dos.h>                // Watcom intrinsic enable/disable interrupts
#include "idc_vdd.h"
#include "data.h"
#include "util.h"
#include "mad16.h"
#include "wss.h"
#include "sndblast.h"
#include "ddstring.h"
#include "trace.h"
#include "ddprintf.h"

PARSEDTOKEN ParsedToken;

//
// Define strings for comparison to
// user supplied options.
//
CHAR szMC1      [] = "MC1";
CHAR szMC2      [] = "MC2";
CHAR szMC3      [] = "MC3";
CHAR szMC4      [] = "MC4";
CHAR szMODE     [] = "MODE";
CHAR szBASE     [] = "BASE";
CHAR szSBBASE   [] = "SBBASE";
CHAR szWSSBASE  [] = "WSSBASE";
CHAR szJOYSTICK [] = "JOYSTICK";
CHAR szOPL      [] = "OPL";
CHAR szCDTYPE   [] = "CDTYPE";
CHAR szCDBASE   [] = "CDBASE";
CHAR szCDIRQ    [] = "CDIRQ";
CHAR szCDDMA    [] = "CDDMA";
CHAR szSBIRQ    [] = "SBIRQ";
CHAR szSBDMA    [] = "SBDMA";
CHAR szFMAP     [] = "FMAP";
CHAR szSBVER    [] = "SBVER";
CHAR szWSSIRQ   [] = "WSSIRQ";
CHAR szWSSDMA   [] = "WSSDMA";


VOID LogErrorNumericExpected (PSZ St)
{
   ddprintf ("MAD16: Expected numeric to follow %s:", St);
}


//                                      ------------------------ ProcessToken -
// Given a parsed token, process the given information
// to change the selected configuration for the device.
// INPUT:    ParsedToken (referenced globally)
// EFFECTED: MC1, MC2, MC3, MC4
// RETURNS:
//   zero -> Token was valid
//   else -> Invalid token (ignored)

USHORT ProcessToken (VOID)
{
   USHORT usRC = 0;

   if (ParsedToken.szKeyword[0] == '-')
      {
      switch (ParsedToken.szKeyword[1])
         {
         case 'D': Options.Debug = TRUE;
                   int3();              // Debug command line option
                   break;
         case 'V': Options.Verbose = TRUE;
                   break;
         default:
            ddprintf ("MAD16: Unrecognized option: %s", ParsedToken.szKeyword);
         }
      }
                                        //------------------------------- MC1 -
   else if (dd_strcmp (ParsedToken.szKeyword, szMC1) == 0)
      {
      if (ParsedToken.fNumericFound)
         MC1 = (BYTE) ParsedToken.usValue;
      else
         {
         LogErrorNumericExpected (ParsedToken.szKeyword);
         usRC = -1;
         }
      }

                                        //------------------------------- MC2 -
   else if (dd_strcmp (ParsedToken.szKeyword, szMC2) == 0)
      {
      if (ParsedToken.fNumericFound)
         MC2 = (BYTE) ParsedToken.usValue;
      else
         {
         LogErrorNumericExpected (ParsedToken.szKeyword);
         usRC = -1;
         }
      }

                                        //------------------------------- MC3 -
   else if (dd_strcmp (ParsedToken.szKeyword, szMC3) == 0)
      {
      if (ParsedToken.fNumericFound)
         MC3 = (BYTE) ParsedToken.usValue;
      else
         {
         LogErrorNumericExpected (ParsedToken.szKeyword);
         usRC = -1;
         }
      }

                                        //------------------------------- MC4 -
   else if (dd_strcmp (ParsedToken.szKeyword, szMC4) == 0)
      {
      if (ParsedToken.fNumericFound)
         MC4 = (BYTE) ParsedToken.usValue;
      else
         {
         LogErrorNumericExpected (ParsedToken.szKeyword);
         usRC = -1;
         }
      }

                                        //------------------------------ MODE -
   else if (dd_strcmp (ParsedToken.szKeyword, szMODE) == 0)
      {
      switch (ParsedToken.szValue[0])
         {
         case 'W': MC1 &= MC1_MODE_CLEAR;
                   MC1 |= MC1_MODE_WSS;
                   break;
         case 'S': MC1 &= MC1_MODE_CLEAR;
                   break;
         default:
            {
            ddprintf ("MAD16: Expected MODE:wss or MODE:sb");
            usRC = -1;
            }
         }
      } // szMODE


   //                                   -------------------------------- BASE -
   // This parm is defined primarily for use with the 928 chip where
   // both the Sound Blaster and WSS base addresses are set with same bits.
   // This creates a relation between the two sets of base addresses.
   // Additionally, on the 928, two WSS choices are invalid (E80, F40)
   // as the device decodes only 10 address lines and these require 12.
   //
   // For the 929 chip, the WSS base is set in MC1 and SB base is set in MC3.
   // NOTE: For the 929, the set of WSS base addresses is the same as the 928,
   // but the bits values for setting those values are different.
   //
   else if (dd_strcmp (ParsedToken.szKeyword, szBASE) == 0)
      {
      if (Specs.usMad16PartNum == 928)
         {
         switch (ParsedToken.usValue)
            {
            case 0x220: MC1 &= MC1_BASE_CLEAR;  // For Sound Blaster mode
                        MC1 |= MC1_BASE_SB220;  // Implies WSS base of 530
                        break;
            case 0x240: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_SB240;  // Implies WSS base of 604
                        break;
            case 0x530: MC1 &= MC1_BASE_CLEAR;  // For WSS mode
                        MC1 |= MC1_BASE_WSS530; // Implies SB base of 220
                        break;
            case 0x604: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSS604; // Implies SB base of 240
                        break;
            case 0xE80: // 928 only decodes 10-bits
            case 0xF40: // Can't address these values
               ddprintf ("MAD16: BASE setting %d not valid with this level hardware. "
                         "(220, 240, 530, 604)", ParsedToken.usValue);
               usRC = -1;
               break;
            default:
               ddprintf ("MAD16: For BASE, expected 220, 240, 530, 604, E80 or F40");
               usRC = -1;
            }
         }
      else // 82C929
         {
         ddprintf ("MAD16: Consider using SBBASE and WSSBASE parms");
         ddprintf ("MAD16: BASE parm links the two addresses");
         switch (ParsedToken.usValue)
            {
            // Although the 929 has separate fields for WSS and SB base, using
            // this parm (BASE:) implies that the values are set as one unit.
            // 929 user should use separate SBBASE and WSSBASE to bypass this
            // linked behavior.
            case 0x220: MC3 &= MC3_SBBASE_CLEAR;// For Sound Blaster mode
                        MC3 |= MC3_SBBASE_220;
                        MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSS530; // Implement implied WSS base
                        break;
            case 0x240: MC3 &= MC3_SBBASE_CLEAR;
                        MC3 |= MC3_SBBASE_240;
                        MC1 &= MC1_BASE_CLEAR;  // Implement implied WSS base
                        MC1 |= MC1_BASE_WSS604;
                        break;
            case 0x530: MC1 &= MC1_BASE_CLEAR;  // For WSS mode
                        MC1 |= MC1_BASE_WSS530;
                        MC3 &= MC3_SBBASE_CLEAR;// Implement implied SB base
                        MC3 |= MC3_SBBASE_220;
                        break;
            case 0x604: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSS604;
                        MC3 &= MC3_SBBASE_CLEAR;// Implement implied SB base
                        MC3 |= MC3_SBBASE_240;
                        break;
            case 0xE80: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSSE80;
                        MC3 &= MC3_SBBASE_CLEAR;// Implement implied SB base
                        MC3 |= MC3_SBBASE_220;
                        break;
            case 0xF40: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSSF40;
                        MC3 &= MC3_SBBASE_CLEAR;// Implement implied SB base
                        MC3 |= MC3_SBBASE_240;
                        break;
            default:
               ddprintf ("MAD16: For BASE, expected "
                         "220, 240, 530, 604, E80 or F40");
               usRC = -1;
            }
         }
      } // szBASE

   //                                   ------------------------------ SBBASE -
   // This parm is primarily for 929 based systems.
   // Permits setting the Sound Blaster base independently of the WSS base
   //
   else if (dd_strcmp (ParsedToken.szKeyword, szSBBASE) == 0)
      {
      if (Specs.usMad16PartNum == 928)
         {
         switch (ParsedToken.usValue)
            {
            case 0x220: MC1 &= MC1_BASE_CLEAR;  // For Sound Blaster mode
                        MC1 |= MC1_BASE_SB220;  // Implies WSS base of 530
                        break;
            case 0x240: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_SB240;  // Implies WSS base of 604
                        break;
            default:
               ddprintf ("MAD16: SBBASE, expected 220 or 240");
               usRC = -1;
            }
         }
      else // 82C929
         {
         switch (ParsedToken.usValue)
            {
            case 0x220: MC3 &= MC3_SBBASE_CLEAR;// For Sound Blaster mode
                        MC3 |= MC3_SBBASE_220;
                        break;
            case 0x240: MC3 &= MC3_SBBASE_CLEAR;
                        MC3 |= MC3_SBBASE_240;
                        break;
            default:
               ddprintf ("MAD16: SBBASE, expected 220 or 240");
               usRC = -1;
            }
         }
      } // szSBBASE

   //                                   ----------------------------- WSSBASE -
   // This parm is primarily for 929 based systems.
   // Permits setting the WSS base independently of the Sound Blaster base
   //
   else if (dd_strcmp (ParsedToken.szKeyword, szWSSBASE) == 0)
      {
      if (Specs.usMad16PartNum == 928)
         {
         switch (ParsedToken.usValue)
            {
            case 0x530: MC1 &= MC1_BASE_CLEAR;  // For WSS mode
                        MC1 |= MC1_BASE_WSS530; // Implies SB base of 220
                        break;
            case 0x604: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSS604; // Implies SB base of 240
                        break;
            case 0xE80: // 928 only decodes 10-bits
            case 0xF40: // Can't address these values
               ddprintf ("MAD16: WSSBASE setting %d not valid with this level hardware. "
                         "(530, 604)", ParsedToken.usValue);
               usRC = -1;
               break;
            default:
               ddprintf ("MAD16: WSSBASE, expected 530 or 604");
               usRC = -1;
            }
         }
      else // 82C929
         {
         switch (ParsedToken.usValue)
            {
            case 0x530: MC1 &= MC1_BASE_CLEAR;  // For WSS mode
                        MC1 |= MC1_BASE_WSS530;
                        break;
            case 0x604: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSS604;
                        break;
            case 0xE80: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSSE80;
                        break;
            case 0xF40: MC1 &= MC1_BASE_CLEAR;
                        MC1 |= MC1_BASE_WSSF40;
                        break;
            default:
               ddprintf ("MAD16: WSSBASE, expected 530, 604, E80 or F40");
               usRC = -1;
            }
         }
      } // szWSSBASE

                                        //-------------------------- JOYSTICK -
   else if (dd_strcmp (ParsedToken.szKeyword, szJOYSTICK) == 0)
      {
      if (ParsedToken.fNumericFound &&
         (ParsedToken.usValue == 0 || ParsedToken.usValue == 1))
         {
         MC1 &= MC1_GAME_CLEAR;
         if (ParsedToken.usValue == 0)
            MC1 |= MC1_GAME_DISABLE;    // Turn bit on to disable joystick
         }
      else
         {
         ddprintf ("MAD16: For JOYSTICK, expected 0 or 1");
         usRC = -1;
         }
      }

                                        //------------------------------- OPL -
   else if (dd_strcmp (ParsedToken.szKeyword, szOPL) == 0)
      {
      if (ParsedToken.fNumericFound &&
         (ParsedToken.usValue == 3 || ParsedToken.usValue == 4))
         {
         MC2 &= MC2_OPL_CLEAR;          // Turn off OPL-4 bit
         if (ParsedToken.usValue == 4)
            MC2 |= MC2_OPL_4;           // Turn on if OPL:4
         }
      else
         {
         ddprintf ("MAD16: For OPL, expected 3 or 4");
         usRC = -1;
         }
      }

                                        //---------------------------- CDTYPE -
   else if (dd_strcmp (ParsedToken.szKeyword, szCDTYPE) == 0)
      {
      switch (ParsedToken.szValue[0])
         {
         case 'N': MC1 &= MC1_CD_CLEAR;
                   MC1 |= MC1_CD_DISABLE;
                   Specs.usCDRange = 0;
                   break;
         case 'S': MC1 &= MC1_CD_CLEAR;
                   MC1 |= MC1_CD_SONY31A;
                   Specs.usCDRange = 4;
                   break;
         case 'M': MC1 &= MC1_CD_CLEAR;
                   MC1 |= MC1_CD_MITSUMI;
                   Specs.usCDRange = 3;
                   break;
         case 'P': MC1 &= MC1_CD_CLEAR;
                   MC1 |= MC1_CD_PANASONIC;
                   Specs.usCDRange = 4;
                   Specs.usCDIRQ = 0;   // Panasonic is polled device
                   break;
         case 'I': MC1 &= MC1_CD_CLEAR;
                   MC1 |= MC1_CD_IDE;
                   Specs.usCDRange = 8; // Also one port at base + 0x206
                   break;
         default:
            {
            ddprintf ("MAD16: For CDTYPE, expected (N,S,P,M,I)");
            usRC = -1;
            }
         }
         if (usRC == 0)
            Specs.usCDType = ParsedToken.szValue[0];
      } // CDTYPE

                                        //---------------------------- CDBASE -
   else if (dd_strcmp (ParsedToken.szKeyword, szCDBASE) == 0)
      {
      // Panasonic can reside at: 220, 240, 250, 300, 330, 340, 630
      // (actually, 10 greater than each of these base addresses)
      // Our only candidates for configuration via the MAD16 are:
      // Disabled, 320, 330, 340, 360.
      // Limit choices to places where the device will be found
      // Sony31A supports all of the possible choices.

      if (MC1 & ~MC1_CD_CLEAR == MC1_CD_PANASONIC)
         {
         if (ParsedToken.usValue != 0x330 &&
             ParsedToken.usValue != 0x340)
            {
            ddprintf ("MAD16: For CDBASE with Panasonic, expected 330 or 340");
            usRC = -1;
            }
         }
      else if (MC1 & ~MC1_CD_CLEAR == MC1_CD_MITSUMI)
         {
         if (ParsedToken.usValue != 0x320 &&
             ParsedToken.usValue != 0x340)
            {
            ddprintf ("MAD16: For CDBASE with Mitsumi, expected 320 or 340");
            usRC = -1;
            }
         }
      else if (MC1 & ~MC1_CD_CLEAR == MC1_CD_IDE)
         {
         if (ParsedToken.usValue != 0x170)
            {
            ddprintf ("MAD16: For CDBASE with IDE, only valid value is 170");
            usRC = -1;
            }
         }
      else if (MC1 & ~MC1_CD_CLEAR == MC1_CD_SONY31A)
         {
         if (ParsedToken.usValue != 0x320 &&
             ParsedToken.usValue != 0x330 &&
             ParsedToken.usValue != 0x340 &&
             ParsedToken.usValue != 0x360)
            {
            ddprintf ("MAD16: For CDBASE with Sony31A, "
                      "expected 320, 330, 340 or 360");
            usRC = -1;
            }
         }

      if (usRC == 0)
         {
         MC2 &= MC2_CDBASE_CLEAR;
         switch (ParsedToken.usValue)
            {
            case 0x170:                        break; // Bits not used for IDE
            case 0x320: MC2 |= MC2_CDBASE_320; break;
            case 0x330: MC2 |= MC2_CDBASE_330; break;
            case 0x340: MC2 |= MC2_CDBASE_340; break;
            case 0x360: MC2 |= MC2_CDBASE_360; break;
            }
         Specs.usCDBase = ParsedToken.usValue;
         }
      } // CDBASE

                                        //----------------------------- CDIRQ -
   else if (dd_strcmp (ParsedToken.szKeyword, szCDIRQ) == 0)
      {
      // Token parser assumes hexadecimal numbers.
      // Convert those greater than 9 to decimal equivalent

      if (ParsedToken.usValue >= 0xA)
         ParsedToken.usValue -= 6;

      // OPTI chip will generate CD IRQs on None, 3, 5, 7, 9, 10, 11
      // Panasonic doesn't do interrupts.  Prevent invalid configurations.

      if (ParsedToken.szValue[0] == 'N')
         {
         MC2 &= MC2_CDIRQ_CLEAR;
         MC2 |= MC2_CDIRQ_NONE;
         }
      else if ((MC1 & MC1_CD_PANASONIC) == MC1_CD_PANASONIC)
         {
         ddprintf ("MAD16: Panasonic accepts only CDIRQ:None");
         usRC = -1;
         }
      else    // Sony31A, Mitsumi or IDE
         {
         if (Specs.usMad16PartNum == 929 && ParsedToken.usValue == 3)
            {
            ddprintf ("MAD16: CDIRQ cannot be 3 with this hardware");
            return (1);
            }
         if (ParsedToken.usValue != 3 &&
             ParsedToken.usValue != 5 &&
             ParsedToken.usValue != 7 &&
             ParsedToken.usValue != 9 &&
             ParsedToken.usValue != 10 &&
             ParsedToken.usValue != 11)
            {
            ddprintf ("MAD16: For CDIRQ, expected 3,5,7,9,10 or 11");
            return (1);
            }

         MC2 &= MC2_CDIRQ_CLEAR;
         switch (ParsedToken.usValue)
            {
            case  3: MC2 |= MC2_CDIRQ_3 ; break;
            case  5: MC2 |= MC2_CDIRQ_5 ; break;
            case  7: MC2 |= MC2_CDIRQ_7 ; break;
            case  9: MC2 |= MC2_CDIRQ_9 ; break;
            case 10: MC2 |= MC2_CDIRQ_10; break;
            case 11: MC2 |= MC2_CDIRQ_11; break;
            }
         }
         Specs.usCDIRQ = ParsedToken.usValue;
      } // CDIRQ


                                          //--------------------------- CDDMA -
   // OS/2 CD-ROM drivers for all of the CD-ROM interfaces
   // supported by the OPTi hardware do not use DMA.
   // With this, configuring CD-DMA to other than none is not needed.
   //
   // This parm does not need to be documented
   //
   else if (dd_strcmp (ParsedToken.szKeyword, szCDDMA) == 0)
      {
      if (ParsedToken.szValue[0] == 'N')
         {
         MC2 &= MC2_CDDMA_CLEAR;
         MC2 |= MC2_CDDMA_NONE;
         }
      else
         {
         switch (ParsedToken.szValue[0])
            {
            case  0:
               if (Specs.usMad16PartNum == 928)
                  {
                  ddprintf ("MAD16: CDDMA:0 not valid with this hardware");
                  return (1);
                  }
               MC2 &= MC2_CDDMA_CLEAR;
               MC2 |= MC2_CDDMA_0;
               break;
            case  1:
               MC2 &= MC2_CDDMA_CLEAR;
               MC2 |= MC2_CDDMA_1;
               break;
            case  3:
               MC2 &= MC2_CDDMA_CLEAR;
               MC2 |= MC2_CDDMA_3;
               break;
            default:
               {
               ddprintf ("MAD16: For CDDMA, expected 0, 1, 3");
               return (1);
               }
            }
         }
         Specs.usCDDMA = ParsedToken.usValue;
      } // CDDMA

                                        //----------------------------- SBIRQ -
   else if (dd_strcmp (ParsedToken.szKeyword, szSBIRQ) == 0)
      {
      // Token parser assumes hexadecimal numbers.
      // Convert those greater than 9 to decimal equivalent

      if (ParsedToken.usValue >= 0xA)
         ParsedToken.usValue -= 6;

      if (ParsedToken.szValue[0] == 'N')
         {
         MC3 &= MC3_SBIRQ_CLEAR;
         MC3 |= MC3_SBIRQ_NONE;
         }
      else
         {
         if (Specs.usMad16PartNum == 928 &&
             ParsedToken.usValue != 5 &&
             ParsedToken.usValue != 7 &&
             ParsedToken.usValue != 11)
            {
            ddprintf ("MAD16: SBIRQ must be 5, 7 or 11");
            return (1);
            }
         if (Specs.usMad16PartNum != 928 &&
             ParsedToken.usValue != 5 &&
             ParsedToken.usValue != 7 &&
             ParsedToken.usValue != 10)
            {
            ddprintf ("MAD16: SBIRQ must be 5, 7 or 10");
            return (1);
            }

         MC3 &= MC3_SBIRQ_CLEAR;
         switch (ParsedToken.usValue)
            {
            case  5: MC3 |= MC3_SBIRQ_5 ; break;
            case  7: MC3 |= MC3_SBIRQ_7 ; break;
            case 10: MC3 |= MC3_SBIRQ_10; break;
            case 11: MC3 |= MC3_SBIRQ_11; break;
            }
         }
      } // SBIRQ

                                        //----------------------------- SBDMA -
   else if (dd_strcmp (ParsedToken.szKeyword, szSBDMA) == 0)
      {
      if (ParsedToken.szValue[0] == 'N')
         {
         MC3 &= MC3_SBDMA_CLEAR;
         MC3 &= MC3_SBDMA_NONE;
         }
      else
         {
         if (Specs.usMad16PartNum == 928 && ParsedToken.usValue == 0)
            {
            ddprintf ("MAD16: SBDMA must be 1, 3 or None");
            return (1);
            }
         if (Specs.usMad16PartNum != 928 && ParsedToken.usValue == 2)
            {
            ddprintf ("MAD16: SBDMA must be 0, 1, 3 or None");
            return (1);
            }

         switch (ParsedToken.usValue)
            {
            case 0: MC3 &= MC3_SBDMA_CLEAR;
                    MC3 |= MC3_SBDMA_0;
                    break;
            case 1: MC3 &= MC3_SBDMA_CLEAR;
                    MC3 |= MC3_SBDMA_1;
                    break;
            case 2: MC3 &= MC3_SBDMA_CLEAR;
                    MC3 |= MC3_SBDMA_2;
                    break;
            case 3: MC3 &= MC3_SBDMA_CLEAR;
                    MC3 |= MC3_SBDMA_3;
                    break;
            default:
               {
               ddprintf ("MAD16: For SBDMA, expected 1, 2 or 3");
               usRC = -1;
               }
            }
         }
      } // SBDMA

                                        //---------------------------- ENMIDI -
   // The MIDI bit of MC2 is enalbed based on the ability of the
   // installed hardware to support simultaneous WSS wave audio and
   // Sound Blaster external MIDI (programmed via the SB address range)
   // NOTE: This is a MIDI statement, not FM
   // The 82C928 revision B cannot do both simultaneously. Later versions can.
   //
   // This parm does not need to be documented
   //else if (dd_strcmp (ParsedToken.szKeyword, szENMIDI) == 0)
   //   {
   //   if (ParsedToken.fNumericFound &&
   //      (ParsedToken.usValue == 0 || ParsedToken.usValue == 1))
   //      {
   //      MC3 &= MC3_ENMIDI_CLEAR;
   //      if (ParsedToken.usValue == 1)
   //         MC3 |= MC3_ENMIDI;
   //      }
   //   else
   //      {
   //      ddprintf ("MAD16: For ENMIDI, expected 0 or 1");
   //      usRC = -1;
   //      }
   //   } // ENMIDI

                                        //----------------------------- FMMAP -
   else if (dd_strcmp (ParsedToken.szKeyword, szFMAP) == 0)
      {
      if (ParsedToken.fNumericFound &&
         (ParsedToken.usValue == 0 || ParsedToken.usValue == 1))
         {
         MC3 &= MC3_FMAP_CLEAR;
         if (ParsedToken.usValue == 1)
            MC3 |= MC3_FMAP_SNGL;
         }
      else
         {
         ddprintf ("MAD16: For FMAP, expected 0 or 1");
         usRC = -1;
         }
      } // FMAP

                                        //----------------------------- SBVER -
   else if (dd_strcmp (ParsedToken.szKeyword, szSBVER) == 0)
      {
      if (ParsedToken.usValue >= 1 && ParsedToken.usValue <= 4)
         {
         MC4 &= MC4_SBVER_CLEAR;
         switch (ParsedToken.usValue)
            {
            case  1: MC4 |= MC4_SBVER_15; break;
            case  2: MC4 |= MC4_SBVER_20; break;
            case  3: MC4 |= MC4_SBVER_32; break;
            case  4: MC4 |= MC4_SBVER_44; break;
            }
         }
      else
         {
         ddprintf ("MAD16: For SBVER, Expected 1, 2, 3, 4");
         usRC = -1;
         }
      } // SBVER

   // The WSS IRQ and DMA fields are programmed using the
   // Microsoft Sound System defined configuration registers.
   // These values are set by the audio device driver when it loads.
   // By convention, the MAD16 VDD and the Audio PDD communciate with
   // this driver to query these settings.  This driver doesn't itself
   // use the information, it just remembers it to provide to the others.

                                        //---------------------------- WSSIRQ -
   else if (dd_strcmp (ParsedToken.szKeyword, szWSSIRQ) == 0)
      {
      if (ParsedToken.usValue >= 0xA)
         ParsedToken.usValue -= 6;

      if (ParsedToken.usValue == 7  ||
          ParsedToken.usValue == 9  ||
          ParsedToken.usValue == 10 ||
          ParsedToken.usValue == 11)
         {
         Specs.usWSSIRQ = ParsedToken.usValue;
         }
      else
         {
         ddprintf ("MAD16: For WSSIRQ, Expected 7, 9, 10, 11");
         usRC = -1;
         }
      } // WSSIRQ

                                        //---------------------------- WSSDMA -
   else if (dd_strcmp (ParsedToken.szKeyword, szWSSDMA) == 0)
      {
      if (ParsedToken.fNumericFound &&
          ParsedToken.usValue == 0 ||
          ParsedToken.usValue == 1 ||
          ParsedToken.usValue == 3)
         {
         // Fill in both play and record DMA values
         Specs.usWSSPlayDMA   = ParsedToken.usValue;
         Specs.usWSSRecordDMA = ParsedToken.usValue;
         }
      else
         {
         ddprintf ("MAD16: For WSSDMA, Expected 0, 1, 3");
         usRC = -1;
         }
      } // WSSDMA

                                        //---------------------- Unrecognized -
   else  // Unrecognized keyword
      {
      ddprintf ("MAD16: Unrecognized keyword: %s", ParsedToken.szKeyword);
      usRC = -1;
      }

   return (usRC);
} // ProcessToken


                                        //------------ CalcSoundPartBaseAddrs -
// After parsing the options, this routine can be
// called to determine and set the base address for
// the wss and SB parts.
VOID CalcSoundPartBaseAddrs (VOID)
{
   USHORT usTemp;

   switch (MC1 & ~MC1_BASE_CLEAR)
      {
      case MC1_BASE_WSS530: usTemp = 0x530; break;
      case MC1_BASE_WSSE80: usTemp = 0xE80; break;
      case MC1_BASE_WSSF40: usTemp = 0xF40; break;
      case MC1_BASE_WSS604: usTemp = 0x604; break;
      }
   Specs.usWSSBase  = usTemp;
   Specs.usWSSIndex = usTemp + 4;
   Specs.usWSSData  = usTemp + 5;
   Specs.usWSSRange = 8;

   usTemp = 0x220;                              // Assume base is 0x220.
   if (Specs.usMad16PartNum == 928)             // Bit 5 alone gives SB base
      {                                         // Bit 4 is a don't care
      usTemp = 0x220 + (MC1 & 0x20);            // 0==>220, 1==>240
      }
   else // 82C929, SB base comes from MC3 bit 2
      {
      if ((MC3 & ~MC3_SBBASE_CLEAR) == MC3_SBBASE_240)
         usTemp = 0x240;
      }
   Specs.usSBBase = usTemp;

   switch (MC3 & ~MC3_SBIRQ_CLEAR)
      {
      case MC3_SBIRQ_5:
           Specs.usSBIRQ = 5;
           break;
      case MC3_SBIRQ_7:
           Specs.usSBIRQ = 7;
           break;
      case MC3_SBIRQ_10:        // IRQ-10 and 11 have same bit value
      //case MC3_SBIRQ_11:
           if (Specs.usMad16PartNum == 928)
              Specs.usSBIRQ = 11;
           else
              Specs.usSBIRQ = 10;
           break;
      default: Specs.usSBIRQ = 0;
      }

   switch (MC3 & ~MC3_SBDMA_CLEAR)
      {
      case MC3_SBDMA_1:
           Specs.usSBDMA = 1;
           break;
      case MC3_SBDMA_0:
      //case MC3_SBDMA_2:
           if (Specs.usMad16PartNum == 928)
              Specs.usSBDMA = 2;
           else
              Specs.usSBDMA = 0;
           break;
      case MC3_SBDMA_3:
           Specs.usSBDMA = 3;
           break;
      default:
           Specs.usSBDMA = 0xFF;
      }
}

// Mad16_SetReg implemented in permanently resident code

//                                      ----------------------- Mad16_ReadReg -
// Read a MAD16 configuration register
//
BYTE Mad16_ReadReg (INT Reg)
{
   BYTE pw;
   BYTE RetVal;

   pw = 0xE2 + (Specs.usMad16PartNum - 928); // Make pw E2 for 82C928, E3 for 82C929
   if (Reg >= ISPBase && Reg <= MC7_PORT)
      {
      _disable ();
      outp (ISPPW_PORT, pw);            // I/O operations must be back to back
      IOdelay ();
      RetVal = inp (Reg);               // Any other I/O terminates cycle
      IOdelay ();
      _enable ();
      }
   return (RetVal);
}


//                                      -------- DetermineOPTiPartAndRevision -
// Figure out what version of the OPTi/MediaChips MAD16 is installed.
// The 82C928 has password value of E2.  The 82C929 has password E3.
// Both live at the same base address.
// On the 82C928, we need to differentiate between revision 'B' and 'C'.
// By convention, the 3 unused bits of MC4 are used to store the version.
// At power up, these are all zero.  Anyone who figures out the version
// stores that value in MC4 so that it can be read by others.
// The 3 bits have the following meaning:
//    Bit-5    : Set to a 1 if bits 4..3 have been set
//    Bit-4..3 : 00 -> 'B',  01 -> 'C'
//
// The version information stored globally to allow other
// routines to take specific action depending hardware type.
//
USHORT DetermineOPTiPartAndRevision (VOID)
{
   BYTE bTemp;

   Specs.usMad16PartNum = 929;
   Specs.usMad16DecodeWidth = 12;               // 929 decodes 12 address bits

   bTemp = Mad16_ReadReg (MC3_PORT);
   if (bTemp != 0xFF)
      {
      Specs.usMad16Revision = 'C';

      //
      // Modify registers to accommodate 82C929 specific meanings
      //
      MC3 &= MC3_ENMIDI_CLEAR;          // ENMIDI on 928 is same bit position
                                        // as SB Base address bit on 929

      MC3 &= 0xFC;      // On 929, bits 1..0 set to 10 enables internal game
      MC3 |= 0x02;      // port timer.  On 928, it is for reading MC4, MC5.

      MC4 |= 0xA0;      // Turn on SB Pro ADPCM enable (Mortal combat game)
                        // or any other that supports SB ADPCM
      return (0);                               // Success, done
      }

   //
   // Everything from here on is 82C928 only
   //
   Specs.usMad16PartNum = 928;
   Specs.usMad16DecodeWidth = 10;               // 928 decodes 10 address bits
   bTemp = Mad16_ReadReg (MC3_PORT);            // This is not redundent
   if (bTemp == 0xFF)
      {
      Specs.usMad16PartNum = 0;
      Specs.usMad16DecodeWidth = 0;
      return (1);                               // OPTi 82C92x not present
      }

   // Determine if revision 'B' or 'C'
   if ((bTemp & ~MC3_SEL_CLEAR) != MC3_SEL_MC4)
      Mad16_SetReg (MC3_PORT, MC3);             // Assume lower bits are zero

   bTemp = Mad16_ReadReg (MC4_PORT);
   if (bTemp & 0x20)                            // Anyone else figure it out?
      {                                         // Yes, use their conclusion
      bTemp = (bTemp & 0x18) >> 3;              // 0 -> Rev 'B', 1 -> Rev C
      Specs.usMad16Revision = 'B' + bTemp;
      }
   else // Version unknown so far, figure it out
      {
      // Read MC2: If bits 1..0 are set, have revision 'C'
      bTemp = Mad16_ReadReg (MC2_PORT);

      if ((bTemp & 0x03) == 0x03)               // Either bit not '1'?
         Specs.usMad16Revision = 'C';
      else
         Specs.usMad16Revision = 'B';
      }

   // Modify local concept of MC4 to reflect
   // version of OPTi 82C928
   MC4 &= MC4_OPTI_VER_CLR;                     // Clear bits 5..3
   MC4 |= 0x20;                                 // Show we set bits 4..3
   bTemp = (Specs.usMad16Revision - 'B') << 3;  // If 'C', set 01, else zero 2
   MC4 |= bTemp;

   //
   // Modify registers to accommodate chip level requirements
   //
   if (Specs.usMad16Revision == 'B')    // Can't do concurrent MIDI and WSS
      MC3 &= MC3_ENMIDI_CLEAR;          // wave audio on 928 rev B

   return (0);
}

//                                      -------------- MAD16_SetConfiguration -
// Given MC1..MC4, set the MAD16 hardware
// to the desired configuration.
//
USHORT MAD16_SetConfiguration (VOID)
{
   BOOL   fSettingSBMode;
   USHORT usRC;

   //
   // Write MC1 to insure the device is powered up
   // NOTE: On 928 rev B, writing MC1 forces reset of CODEC.
   //
   Mad16_SetReg (MC1_PORT, 0);

   fSettingSBMode = ((MC1 & ~MC1_MODE_CLEAR) == MC1_MODE_SB);

   CalcSoundPartBaseAddrs ();                   // Determine wss & sb base

   //
   // Set target values for MC1..MC4
   //
   Mad16_SetReg (MC1_PORT, MC1);
   Mad16_SetReg (MC2_PORT, MC2);
   Mad16_SetReg (MC3_PORT, MC3);
   Mad16_SetReg (MC4_PORT, MC4);

   //
   // Disable shadow protect and enable access to WSS registers.
   // Program data format into CODEC for 8-bit MONO 44.1 KHz
   // Do this with the shadow registers not-protected so that the OPTi
   // part absorbs the 44.1K clock information
   // The value is written into the 82C928 and used as clock
   // reference for handling programmed I/O Sound Blaster games.
   //
   Mad16_SetReg (MC5_PORT, 0x1A);       // Enable WSS access, shadow unprotect
   usRC = WSS_QueryPartType ();         // Check version of audio CODEC
   if (usRC != 0)
      {
      ddprintf ("MAD16: WSS version check failed, aborting load.");
      return (1);
      }

   if (Options.Verbose)
      {
      ddprintf ("OPTi  82C%d", Specs.usMad16PartNum);
      if (Specs.usWSSCodecType == 'C')
         ddprintf ("CODEC Crystal Semiconductor");
      else
         ddprintf ("CODEC Analog Devices");
      ddprintf ("MC1   %0x", MC1);
      ddprintf ("MC2   %0x", MC2);
      ddprintf ("MC3   %0x", MC3);
      ddprintf ("MC4   %0x", MC4);
      }

   usRC = WSS_Setup ();                 // Set sampling rate 44.1K 8-bit MONO
   Mad16_SetReg (MC5_PORT, 0x2A);       // and initialize the mixer registers
   if (usRC != 0)                       // No WSS access, protect shadow regs
      {
      ddprintf ("MAD16: WSS_Setup failed, aborting load.");
      return (1);
      }

   //
   // If changing to Sound Blaster mode, restore or
   // initialize the Sound Blaster registers
   //
   if (fSettingSBMode)
      {
      usRC = SB_Reset ();                  // Initialize SB shadow registers
      if (usRC != 0)
         {
         ddprintf ("MAD16: Sound Blaster reset failed, aborting load.");
         return (1);
         }

      // Set Sound Blaster mixer registers.
      // NOTE: On 82C928, These writes don't actually do anything.
      // The values can be read back when in SB mode via VOLTSR who
      // actually programs the WSS mixer to equivalent values.
      // On the 82C929, the Sound Blaster mixer values are reflected
      // to the WSS codec.
      //
      SB_MixerWrite (0x04, 0x88);          // Voice volume  L:R 8:8
      SB_MixerWrite (0x22, 0x88);          // Master volume L:R 8:8
      SB_MixerWrite (0x26, 0x88);          // MIDI volume   L:R 8:8
      }

   //
   // In all cases (SB or WSS mode) set the audio codec
   // to 11025 Hz sampling rate with the shadow registers
   // protected.
   //
   // Enable WSS access, shadow protect!
   Mad16_SetReg (MC5_PORT, MC5_SHPASS | MC5_SPACCESS);
   usRC = WSS_SetClockAndDataFormat (11025);
   Mad16_SetReg (MC5_PORT, 0x2A);       // No WSS access, protect shadow regs
   if (usRC != 0)
      {
      ddprintf ("MAD16: WSS sample rate set failed, aborting load");
      return (1);
      }

   Mad16_SetReg (MC5_PORT, 0x1A);       // Enable WSS access, shadow unprotect
   WSS_EnableInterrupts();
   Mad16_SetReg (MC5_PORT, 0x2A);       // No WSS access, protect shadow regs

   //
   // Set final value for MC5, MC6 and MC7
   //
   // The MC5 bit patern depends on type of audio CODEC and
   // the version of the OPTi part.
   //
   MC5 = MC5_SHPASS;                    // Protect shadow registers (all cases)

   if (Specs.usWSSCodecType == 'C')
      {
      if (Specs.usMad16PartNum == 928)
         MC5 |= (MC5_CFIFO_PDMA | MC5_CFIX_PEN);  // 0x2A
      else
         MC5 |= 0x0F;   // For 929                // 0x2F
      }
   else // AD CODEC
      {
      if (Specs.usMad16PartNum != 928)
         MC5 |= 0x05;                             // 0x25
      }

   if (Specs.usMad16PartNum != 928)     // If 929 or beyond
      {
      MC5 |= 0x80;                      // Turn on CD-ROM direct access bit
      Mad16_SetReg (MC6_PORT, MC6);     // Set MPU-401 config register
      if (Specs.usSBBase == 0x240)      // MC7 holds info for VOLTSR to
         MC7 |= 0x80;                   // program mixer state on 929
      Mad16_SetReg (MC7_PORT, MC7);
      }

   Mad16_SetReg (MC5_PORT, MC5);        // Set final value for MC5

   return (0);
}


//                                      ------------------------ Mad16_Wakeup -
// Parse command line to enable the MAD16 at the
// indicated configuration (DMA/IO/IRQ/CDTYPE/SOUNDTYPE).
//
USHORT Mad16_Wakeup (char *pszLine)
{
   USHORT usRC;

   usRC = DetermineOPTiPartAndRevision();
   if (usRC != 0)
      {
      ddprintf ("MAD16: No OPTi 82C928/82C929 hardware in this machine");
      return (1);

      }

   //
   // The driver maintains a data structure of information describing
   // the hardware configuration.  Most of the information is determined
   // based on the set values of the MC registers.
   // Some of the fields are not based on the MC regs and need to be
   // initialized explicitly.
   //
   Specs.usStrucSize        = sizeof (Specs);
   Specs.usMad16Base        = 0xF8C;    // Fixed location
   Specs.usMad16Range       = 8;
   Specs.usWSSIRQ           = 7;
   Specs.usWSSPlayDMA       = 1;
   Specs.usWSSRecordDMA     = 1;
   Specs.usSBRange          = 16;
   Specs.usCDType           = 'N';
   Specs.usCDBase           = 0x340;
   Specs.usOPL3Base         = 0x388;  // Fixed location
   Specs.usOPL3Range        = 4;

   Trace_Save_Message (pszLine);        // Copy command line to trace buffer

   NextToken ((char **)&pszLine);       // Skip device driver .ADD file name

   while (*pszLine != '\0')
      {
      // Each token is of form "KeyWord:Value"
      // Value can be either hex numeric or string.
      // The colon and value are optional.

      SplitToken (pszLine, &ParsedToken);       // Keyword:Value
      ProcessToken ();
      NextToken ((char **)&pszLine);
      }

   usRC = MAD16_SetConfiguration ();

   return (usRC);
}
