;*DDK*************************************************************************/
;
; COPYRIGHT    Copyright (C) 1995 IBM Corporation
;
;    The following IBM OS/2 WARP source code is provided to you solely for
;    the purpose of assisting you in your development of OS/2 WARP device
;    drivers. You may use this code in accordance with the IBM License
;    Agreement provided in the IBM Device Driver Source Kit for OS/2. This
;    Copyright statement may not be removed.;
;*****************************************************************************/
;******************************************************************************
;                 IBM Sample Audio Virtual Device Driver
;
;******************************************************************************
;
; DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
; sample code created by IBM Corporation.
; It is provided to you solely for the purpose of assisting you in the
; development of your applications.
; The code is provided "AS IS", without warranty of any kind.
; IBM shall not be liable for any damages arising out of your use of
; the sample code, even if they have been advised of the possibility
; of such damages.
;
; audiovdd.asm - Catch DOS session events.
;
; Initial Release
;
; TITLE   audiovdd.sys
;
; This Virtual Device Driver demonstrates hardware sharing between
; OS/2 Physical Device Drivers (PDDs) and DOS sessions executing
; under OS/2 2.0 with MMPM/2.
;
; Virtual Device Drivers are ring-0 32-bit device drivers responsible
; for controlling V8086/MVDM sessions (DOS sessions).
; Normal VDD architecture calls for a device driver pair;
; Physical Device Driver and Virtual Device Driver working together to
; serialize and control a common piece of hardware (the shared hardware).
; The Physical Device Driver controls access and controls OS/2 sessions.
; The Virtual Device Driver controls DOS sessions and communicates with
; the PDD to serialize access.  That is, the PDD and VDD communicate to
; insure DOS sessions and OS/2 sessions cannot concurrently access
; the shared hardware.
;
; Without a VDD, DOS sessions have complete access to system hardware.
; Most DOS visible hardware resources are controlled by system VDDs thereby
; preventing DOS sessions from interfering with each other and protecting
; OS/2 applications from conflict with DOS sessions.
;
; Where a PDD exists with no VDD, the hardware device is visible
; to the OS/2 side (through the PDD) and to all DOS sessions (through the
; VDM manager).  This presents an opportunity for DOS sessions to step on
; a hardware resource owned by an OS/2 Physical Device Driver.
; Writing a Virtual Device Driver closes this hole resulting in system wide
; protected access to the hardware resource.
;
; This PDD-VDD pair need usually results in a VDD (like a PDD) being written
; to the specifications of a specific hardware device.
; This VDD supports multiple adapters by using a protocol for PDD-VDD
; communication where the PDD defines the characteristics of its device
; at system boot time.
;
; The key to supporting multiple PDDs is communication at system boot to
; attain a description of the physical characteristics of each adapter.
; Physical Device Drivers are loaded before the MVDM manager which intern
; precedes the load of Virtual Device Drivers.
; When the PDD loads, it registers itself as willing to communicate with
; VDDs providing its IDC name and entry point for this communication.
; When the VDD loads, it calls the dev help VDHOpenPDD to resolve the
; IDC entry point of the PDD.  On this call, the OS calls the PDD to deliver
; the VDD's IDC entry point.
; Once the PDD and VDD know the others IDC entry point, the OS is no longer
; needed for Inter Device Communication.
; For these calls to function, the VDD must know the name by which the
; PDD registered.  In a non-generic implementation, this is not a problem
; as the two device drivers can reference the name from a common include file.
; This "generic" VDD takes the PDD names as parameters on the config.sys
; command line.
; In the case where a single PDD supports multiple devices, it should
; register multiple IDC names / entry points and specify all of these
; names on the VDD's config.sys command line.
;
; During initialization, the VDD calls all of the PDDs to establish
; communication and to query the hardware resources they want protected.
; When a DOS session steps on a PDD owned resource, the VDD traps the
; DOS session pending permission to use the hardware.
; As the PDD controls access, the VDD queries the PDD for permission to use
; the hardware.
; If granted, the VDD disables further traps and allows the DOS session
; to continue uninterrupted.  Traps are disabled to give maximum performance.
; As the VDD does not watch port accesses and it does not know the details
; of the adapters it supports, it is unable to remember the state of the
; adapter.  That is, once a DOS session owns the hardware, ownership cannnot
; be relinquished and restored as the VDD doesn't know how to restore the
; state of the device.  The VDD supports serialization (not virtualization).
;
; If access is denied by the PDD, or if another DOS session already owns the
; hardware, a PopUp message is displayed asking the user for direction.
; The user can give instruction to terminate the session, suspend the
; session or to pretend hardware doesn't exist.  Based on this response,
; the VDD executes VDHxx operations to perform the necessary actions.
; Semaphores are used to block DOS session execution when they are suspended
; awaiting access to hardware.
; When the last DOS session releases the hardware, the VDD returns
; hardware ownership to the PDD.
;
; To follow the VDD execution flow, start with vddinit.asm; progress
; to this file to watch the session hooks and then to audioreq.asm
; for hardware sharing between the VDD and PDD.
;
; Program structure:
;    makefile
;    VDDINIT.ASM   Initialization code
;    AUDIOVDD.ASM  DOS session event hooks
;    AUDIOREQ.ASM  Request/Free hardware (PDD Communication)
;    AUDIOEOI.ASM  End of interrupt notifications
;    GLOBDATA.ASM  Global/Common data declarations
;    INSTDATA.ASM  Per-DOS session data
;
;    AUDIOVDD.INC  function definitions for audiovdd.asm
;    AUDIOREQ.INC  function definitions for audioreq.asm
;    AUDIOEOI.INC  function definitions for audioeoi.asm
;    VDDDATA.INC   EXTERNs for globdata.asm and instdata.asm
;    VDDSTRUC.INC  Structure declarations - for globdata.asm and instdata.asm
;    DEVHELP.INC   EXTERNs for OS/2 Virtual Dev Help calls
;******************************************************************************
        .386P                   ; Enable use of ring-0 80386 instructions

        OPTION  OLDMACROS       ; Needed for OS/2 Toolkit when using MASM 6.
       ;INCL_NONE  EQU     1
        INCL_VDH   EQU     1
        INCLUDE mvdm.inc        ; from OS/2 base toolkit
        INCLUDE basemid.inc     ; from OS/2 base toolkit
        INCLUDE devhelp.inc     ; externs for OS/2 Virtual Dev Helps
        INCLUDE vdddata.inc     ; EXTRNS for instance and global variables
        INCLUDE audioreq.inc    ; EXTRNS for procedures in audioreq.asm

; Primary code segment.  NON-swappable, NON-discardable
_TEXT   SEGMENT DWORD USE32 PUBLIC 'CODE'
        ASSUME  CS:_TEXT
        ASSUME  DS:FLAT, SS:FLAT, ES:FLAT

                                        ;--------------------- AVDDPropVerify -
; Function is called when user modifies our DOS property.
; As our property is an enumeration property, the call is
; only to "let us know" the value has changed.
; The validation function is required for use of DOS properties.
AVDDPropVerify  PROC NEAR
        xor     eax, eax                        ; Return success (zero)
        ret     16                              ; Pop parms and return
AVDDPropVerify  ENDP

                                        ;------------------------- AVDDCreate -
; Routine is called every time a DOS session is created.
; PARMS:
;    [ebp+8] - hvdm
;
AVDDCreate      PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        mov     eax, [ebp+8]                    ; Remember hvdm - store in
        mov     HVDM_CURRENT, eax               ; instance data segment

        ; For each adapter, hook I/O ports
        ; If hook install fails, then the DOS session
        ; create will be rejected.
        ; The VDD will then receive a terminate message

        call    InstallHooks
        or      eax, eax
        jz      Failed

Success:mov     eax, TRUE                       ; Indicate success
        jmp     Done
Failed: xor     eax, eax                        ; zero return, indicate failure
Done:   leave
        ret     4                               ; Is ret 8 in OS/2 toolkit
AVDDCreate      ENDP                            ; Believe ret 4 is correct


                                        ;---------------------- AVDDPDBChange -
; Routine is called every time the current executing DOS
; application changes.
; The parm received points to the program segment prefix for
; the currently executing application.
; The PDB is remembered in instance data segment for reference
; at hardware request.
; The currently executing program is marked as the hardware owning
; application when ownership is attained.
; When the marked program terminates, the hardware is released.
; See AVDDPDBDestroy for description of hardware release.
;
; PARMS:
;    [ebp+C] - hvdm   - Handle to DOS Session
;    [ebp+8] - segPDB - V86 segment of new PDB
;
AVDDPDBChange   PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        mov     eax, [ebp+8]                    ; Remember current program
        mov     PDB_Current, ax                 ; segment (word value)
        mov     eax, TRUE                       ; Indicate success
        leave
        ret     8
AVDDPDBChange   ENDP

                                        ;--------------------- AVDDPDBDestroy -
; Care should be taken to make this routine efficient as
; it is called every time a DOS application terminates.
;
; For each adapter, if this session owns the hardware and this
; application was the active application when the hardware was
; was acquired, then the session is done with the hardware.
;
; PARMS:
;    [ebp+C] - hvdm   - Handle to DOS Session
;    [ebp+8] - segPDB - V86 segment of terminating PDB
;
; RETURNS:
;    nothing
;
; REGISTER USE:
;    ebx - Index into adapters array
;    ecx - Limit number of adapters
;    esi - Holds hvdm for this DOS session
;          Kept in register to speed search
;     dx - Holds V86 segment of terminating PDB
;    edi - Used to index false hardware array
;
AVDDPDBDestroy PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ebx                             ; Save reqs that can't change
        push    esi
        push    edi

        mov     esi, DWORD PTR [ebp+12]         ; hvdm - Keep parms in regs
        mov     dx, WORD PTR [ebp+8]            ; PDB  - to speed search

        xor     ebx, ebx                        ; Point to first adapter
        mov     ecx, MAX_NUM_ADAPTERS           ; Limit number of PDDs

NextAdapter:
        lea     eax, Adapters[ebx].pszPddName   ; If name is null, then
        cmp     byte ptr [eax], 0               ; remainder of table is empty.
        je      Success

        ; If we are virtualizing the hardware (as not installed), and
        ; this application (terminating) initiated the state, then
        ; restore to initial state (not virtualizing hardware).

        mov     edi, Adapters[ebx].FalseHWIndex ; If field is non-zero, then
        mov     ax, FalseHWArray[edi]           ; we are ignoring the hardware.
        cmp     ax, dx                          ; If this PDB != "owning PDB"
        jnz     CheckOwn                        ; then check for H/W ownership

        xor     eax, eax                        ; Otherwise, stop virtualize
        mov     FalseHWArray[edi], ax           ; state.
        jmp     LoopBot

CheckOwn:
        cmp     esi, Adapters[ebx].hvdm_Owning  ; Does this session own adapter
        jne     LoopBot                         ; If not, next adapter

        cmp     dx, Adapters[ebx].pdb_Owning    ; Does this app own adapter?
        jne     LoopBot                         ; If not, next adapter

        call    ReleaseAdapter                  ; DOS session releases hardware

LoopBot:
        add     ebx, sizeof PER_CARD_DATA       ; Bump table index
        loop    NextAdapter

Success:                                        ; Function has no return value
        pop     edi                             ; Restore callers registers
        pop     esi
        pop     ebx
        mov     eax, TRUE                       ; Indicate success
        leave
        ret     8
AVDDPDBDestroy ENDP


                                        ;---------------------- AVDDTerminate -
; Routine is called every time a vdm terminates
; If this DOS session owns any hardware, the
; hardware will be released.
;
; PARMS:
;    [ebp+8] - hvdm - DOS Session being terminated
;
AVDDTerminate   PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ebx                             ; Save reqs
        push    esi

        mov     esi, [ebp+8]                    ; hvdm

        xor     ebx, ebx                        ; Point to first adapter
        mov     ecx, MAX_NUM_ADAPTERS           ; Limit number of PDDs

NextAdapter:
        lea     eax, Adapters[ebx].pszPddName   ; If name is null, then
        cmp     byte ptr [eax], 0               ; remainder of table is empty.
        je      Success

        cmp     esi, Adapters[ebx].hvdm_Owning  ; Does this session own adapter
        jne     LoopBot                         ; If not, next adapter

        call    ReleaseAdapter                  ; DOS session releases adapter

LoopBot:
        add     ebx, sizeof PER_CARD_DATA       ; Bump table index
        loop    NextAdapter

Success:                                        ; Function has no return value
        mov     eax, TRUE                       ; Indicate success anyway

        pop     esi                             ; Restore callers registers
        pop     ebx
        leave
        ret     4
AVDDTerminate   ENDP

                                        ;---------------------- SearchAdapter -
; Routine is called during I/O port trap handling.
; On entry, ebx points to the Adapter of interest.
; Routine searches all port ranges used by that adapter.
; If suspect range is found, function returns TRUE.
; All registers preserved (except eax)
;
; PARMS
;    [ebp+8] - Search key (port number)
;    ebx     - Adapters index
;
; RETURNS (eax)
;    TRUE  - Found
;    FALSE - Not found

SearchAdapter PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ebx
        push    ecx
        push    edx

        mov     eax, [ebp+8]                    ; Port number
        mov     ecx, Adapters[ebx].NumPortRanges
        mov     ebx, Adapters[ebx].PortArrayPtr ; Pointer to port array

        ; Search adapter port ranges for the specified port
        ; REGISTER USE:
        ;    eax - Holds search port number
        ;    edx - Holds value for comparison
        ; To "find" the port, the search port number must
        ; be in range of FirstPort..(FirstPort+NumPorts-1)

Next:
        mov     edx, [ebx].PORTARRAYENTRY.FirstPort
        cmp     eax, edx                        ; If search number is greater
        jl      LoopBot                         ; than start of range, jump

        add     edx, [ebx].PORTARRAYENTRY.NumberPorts
        dec     edx
        cmp     eax, edx                        ; If search number is beyond
        jg      LoopBot                         ; end of range, jump

        jmp     Success                         ; Found it!

LoopBot:
        add     ebx, sizeof PORTARRAYENTRY
        loop    Next

        ; If fall through, we didn't find search port

        jmp     Failed

Success:mov     eax, TRUE                       ; Indicate success
        jmp     Done
Failed: xor     eax, eax                        ; Indicate failure
Done:   pop     edx                             ; Restore callers registers
        pop     ecx
        pop     ebx
        leave
        ret     4
SearchAdapter ENDP

                                        ;------------------- DetermineAdapter -
; Given PORT number, this routine sets EBX register
; to the adapter which owns the port.
; Routine is called during I/O port trap handling.
;
; PARMS
;    [ebp+8] - Port number
;
; RETURNS
;    eax - TRUE   Found
;          FALSE  Not found (should never happen)
;    ebx - index into Adapters table of found adapter

DetermineAdapter PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ecx                             ; Save regs except eax, ebx
        push    edx

        xor     ebx, ebx                        ; Point to start of table
        mov     ecx, MAX_NUM_ADAPTERS           ; Limit number of PDDs

NextAdapter:
        lea     eax, Adapters[ebx].pszPddName   ; If name is null, then
        cmp     byte ptr [eax], 0               ; remainder of table is empty.
        je      Failed                          ; Should never happen

        push    [ebp+8]                         ; Port number
        call    SearchAdapter
        or      eax, eax
        jnz     Found                           ; Found it!

        add     ebx, sizeof PER_CARD_DATA       ; Bump table index
        loop    NextAdapter                     ; Search next adapter

        ; If fall through the above loop, something is     busted.
        ; The VDD received a trap for a port that we don't control.
        ; Should never happen.  If does (in debug), it probably
        ; means the table searching routines are busted (above).
        ; Either way, we're broke.
Failed:
        xor     ebx, ebx
        xor     eax, eax                        ; Indicate failure
        jmp     Done

Found:  mov     eax, TRUE

Done:   pop     edx                             ; Restore regs except eax, ebx
        pop     ecx
        leave
        ret     4
DetermineAdapter ENDP

                                        ;------------------------- AVDDstrcmp -
; Assembler equivalent of C strcmp function
; PARMS:
;    [EBP+12] - Left string
;    [EBP+08] - Right string
;
; RETURNS:
;   0 - equal
;   1 - different

AVDDstrcmp PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ecx
        push    esi
        push    edi

        mov     esi, [EBP+12]                   ; Left string
        mov     edi, [EBP+08]                   ; Right string
        cld                                     ; Compare left to right
        xor     eax, eax

   Next:
        cmpsb                                   ; Compare bytes, index next set
        jnz     NotEqual
        mov     al, [esi]                       ; At end of string?
        or      al, al                          ; If see nul, string are equal
        jne     Next                            ; Not nul, compare next pair

        xor     eax, eax                        ; Indicate strings are equal
        jmp     Done

NotEqual:
        mov     eax, 1                          ; Indicate not equal

Done:
        pop     edi
        pop     esi
        pop     ecx
        leave
        ret     8
AVDDstrcmp ENDP


                                        ;---------------------- CheckProperty -
; Compares string given to current sessions value
; for our DOS property.
;
; PARMS:
;    [EBP+8] - Suspect property value
;
; RETURN:
;    0 - Equal
;    1 - Different

CheckProperty PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ecx
        push    edx

        push    OFFSET FLAT:szPropNameAudio     ; Get pointer to current value
        call    VDHQueryProperty                ; Data resides in kernel space
        mov     edx, eax                        ; Save pointer for later free
        push    eax                             ; Compare current value to
        push    [EBP+8]                         ; caller provided string
        call    AVDDstrcmp
        push    eax                             ; Save compare result

        push    edx                             ; Tell VDM manager that we're
        call    VDHFreeMem                      ; done with its memory.

        pop     eax                             ; eax has result of compare

        pop     ecx
        pop     edx
        leave
        ret     4
CheckProperty ENDP

                                        ;--------------------- MaybeRequestHW -
; Function is called the first time a DOS session steps on
; a protected resource.  It provides a layer above the
; RequestHW routine.  A DOS property is queried.  If set to
; disable access to audio hardware, then the session does not
; compete for hardware access.
; The resulting behavior is the same as would be encountered
; by trying to gain access (being denied), displaying popup
; and have the user respond that they do not need the hardware (ignore).
; Advantage is the user's activity is not interrupted by the popup.

MaybeRequestHW PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ecx
        push    edx
        push    esi

        push    OFFSET FLAT:szEnumPrevent       ; If DOS property has "Prevent"
        call    CheckProperty                   ; state, then don't display
        or      eax, eax                        ; popup and pretend there is
        jz      NoAccess                        ; no installed h/w

        call    RequestHW                       ; Otherwise, compete for access
        cmp     eax, VDHP_IGNORE                ; Gained access/term - jump
        jne     Done                            ; Othwerwise, pretend no h/w

NoAccess:
        ; Index array in instance data segment, find
        ; element for this adapter.
        ; Set flag to force future port accesses to be ignored
        ; and remember current DOS application handle.

        mov     esi, Adapters[ebx].FalseHWIndex ; Set ignore flag.  Remember
        mov     ax, PDB_Current                 ; current PDB. State is removed
        mov     FalseHWArray[esi], ax           ; when this app terminates.
        mov     eax, VDHP_IGNORE                ; Function return code

Done:   pop     esi
        pop     edx
        pop     ecx
        leave
        ret
MaybeRequestHW ENDP


                                        ;---------------------- ByteInputHook -
; Capture DOS session IN instructions that
; attempted to step on the protected hardware.
;
; PARMS:
;    EDX - port number
;    EBX - Client register frame pointer
;
; RETURNS
;     AL - Data read

ByteInputHook PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ebx                             ; Save reqs that can't change
        push    esi
        push    edi

        push    edx                             ; Port number
        call    DetermineAdapter                ; sets ebx register
        or      eax, eax                        ; If don't find port, something
        jz      FakeHW                          ; is busted.  Try to continue

        mov     esi, Adapters[ebx].FalseHWIndex ; If field is non-zero, then
        mov     ax, FalseHWArray[esi]           ; we are ignoring the hardware
        or      ax, ax                          ; based on earlier user
        jnz     FakeHW                          ; instruction.

        ; The DOS session has attempted to touch the hardware.

        call    MaybeRequestHW
        cmp     eax, 0                          ; If didn't get access, treat
        jne     FakeHW                          ; hardware as not present.

        ; Session owns the hardware.
        ; The session has been given direct access to the
        ; protected ports, subsequent port operations
        ; (by this session/application) will not be trapped.

        in      al, dx
        jmp     Done

FakeHW: mov     eax, -1                         ; Pretend there is no hardware

Done:
        pop     edi                             ; Restore callers registers
        pop     esi
        pop     ebx
        leave
        ret
ByteInputHook ENDP

                                        ;--------------------- ByteOutputHook -
; Capture DOS session OUT instructions that
; attempted to step on the protected hardware.
;
; PARMS:
;     AL - Data to write
;    EDX - port number
;    EBX - Client register frame pointer
;
; REGISTER USAGE
;    EBX - Adapters table index
;    ECX - Hold parm received in AL
;    EDX - port number
;
; RETURNS
;    nothing

ByteOutputHook PROC NEAR
        push    ebp                             ; Save a few clocks over
        mov     ebp, esp                        ; enter 0,0 instruction

        push    ebx                             ; Save reqs that can't change
        push    esi
        push    edi

        mov     ecx, eax                        ; Remember port data to write

        push    edx                             ; Port number
        call    DetermineAdapter                ; sets ebx register
        or      eax, eax                        ; If don't find port, something
        jz      FakeHW                          ; is busted.  Try to continue

        mov     esi, Adapters[ebx].FalseHWIndex ; If field is non-zero, then
        mov     ax, FalseHWArray[esi]           ; we are ignoring the hardware
        or      ax, ax                          ; based on earlier user
        jnz     FakeHW                          ; instruction.

        ; The DOS session has attempted to touch the hardware.

        call    MaybeRequestHW
        cmp     eax, 0                          ; If didn't get access, treat
        jne     FakeHW                          ; hardware as not present.

        ; Session owns the hardware.
        ; The session has been given direct access to the
        ; protected ports, subsequent port operations
        ; (by this session/application) will not be trapped.

        mov     eax, ecx                        ; Perform port operation
        out     dx, al                          ; for the DOS application
        jmp     Done                            ; (first access only)

FakeHW: ; Do nothing                            ; Pretend there is no hardware

Done:
        pop     edi                             ; Restore callers registers
        pop     esi
        pop     ebx
        leave
        ret
ByteOutputHook ENDP

_TEXT   ENDS                                    ; End of primary code segment
        END                                     ; End of source module
