/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/************************* START OF SPECIinCAT/IONS ***
*
* SOURCE FILE NAME:  VSDSET.C
*
* DESCRIPTIVE NAME: Allows setting of audio attributes via VSD
*
*
* FUNCTION: This source file processes VSD_SET requests.
*
*
* NOTES: This source file illustrates the handling of the following flags:
*           1. VSD_SETVOLUME (Changing instance and master volume ).
*           2. VSD_SETCONNECTOR (enable/disable connectors
*           4. VSD_SETMIXCONNECTIONS ( change the setting of a particular mixer
*                                      connector setting like treble, bass etc.)
*           5. VSD_SETAUDIOATTRIBUTES (change audio attributes like treble bass).
*           6. VSD_SETMONITOR  (enable - disable monitoring ).
*           7. VSD_SETDATATYPE (changing digital audio attributes -- channels etc.).
*           8. VSD_SETMIXCONTROL (modify the settings of the mixer device ).
*
*        Also calling the mixer interface.
*
*/

#define INCL_NOPMAPI
#define INCL_DOS
#define INCL_ERRORS

#define INCL_AUDIO_VSD
#include <os2.h>
#include <os2me.h>
#include <mcd.h>
#include <audio.h>
#include <stdio.h>
#include <string.h>
#include <hhpheap.h>


#include <vsdcmds.h>
#include <vsdaud.h>
#include <os2mixer.h>
#include <vsdmix.h>








/************************ START OF SPECIFICATIONS **************************
*
* SUBROUTINE NAME: VSDSet
*
* FUNCTION: Sets audio attributes (Sample rate, bits per sample, etc)
*
* INPUT:  pInstance - pointer to VSD Instance structure
*         ulFlags   - flags used in set operation
*
* OUTPUT: returns VSDERR_SUCCESS if successful, otherwise it returns an
*         MCI error code.
*
* OS/2 CALLS:
*
* C CALLS: None.
*
* INTERNAL CALLS: MCI_Error(), ModifyAudioAttributes()

*
*************************** END OF SPECIFICATIONS *************************/

LONG VSDSetCommand( PVSD_INSTANCE    pInstance,
                    ULONG            ulFlags,
                    PVOID            pRequest )

    {
    LONG lError = VSDERR_SUCCESS;


//    ULONG ulResourceDecrease;
//    ULONG ulResourceChangeNeeded = FALSE;
    ULONG ulOldResources = pInstance->ulResourcesUsed;
//    ULONG ulMatch;
    ULONG ulResourceChange = FALSE;
    ULONG ulOldClass = pInstance->ulClass;
    ULONG ulMicType;
    ULONG ulSpeakerType;


    VSD_INSTANCE      TempAmp;

    MCIDRV_CHANGERESOURCE_PARMS    mciChangeResource;

    PVSD_AUDIODATATYPE_PARMS pAudioSettings;


   switch ( ulFlags )
     {

     /*----------------------------------------------------------------
     * Set datatype allows the mixer (and only the mixer) to change
     * the mode that the VSD is in.  The following flags can be
     * in the ulFlags field.
     *    VSD_MODE              Modify the datatype (i.e. PCM, mulaw etc.)
     *    VSD_BITSPERSAMPLE     Modify the BPS (i.e. 4, 8, 16 etc.)
     *    VSD_CHANNELS          Modify the number of channels.
     *    VSD_SAMPLESPERSEC     Modify the sampling rate (11K, 22k, 44k etc.)
     *
     * If the mode is not valid, one of the following errors should
     * be returned.
     *
     *  VSDERR_UNSUPP_FORMAT_TAG
     *  VSDERR_UNSUPP_BITSPERSAMPLE
     *  VSDERR_UNSUPP_CHANNELS
     *  VSDERR_UNSUPP_SAMPLESPERSEC
     *  VSDERR_UNSUPP_FORMAT_MODE
     *---------------------------------------------------------------*/
     case VSD_SETDATATYPE:
       {

       pAudioSettings = (PVSD_AUDIODATATYPE_PARMS) pRequest;

       /* Temporarily copy the VSD instance and check the values */

       memmove (&TempAmp, pInstance, sizeof (VSD_INSTANCE));
       TempAmp.ulOperation     = pAudioSettings->ulOperation;


       /*--------------------------------------------------
       * If the caller is changing the mode (ie PCM, MIDI)
       * then there are some checks we will have to perform.
       *--------------------------------------------------*/

       if (pAudioSettings->ulFlags & VSD_MODE)

         {
         TempAmp.sMode = ( SHORT ) pAudioSettings->ulDataType;

         /*-------------------------------------------
         * Mulaw and Alaw are all 8bit types.  Force
         * the bitspersample to be 8.
         *-------------------------------------------*/

         if ( TempAmp.sMode == DATATYPE_MULAW      ||
              TempAmp.sMode == DATATYPE_ALAW       ||
              TempAmp.sMode == DATATYPE_RIFF_MULAW ||
              TempAmp.sMode == DATATYPE_RIFF_ALAW )

           {
           pAudioSettings->ulFlags |= VSD_BITSPERSAMPLE;
           pAudioSettings->ulBitsPerSample = 8;

           /*----------------------------------------------
           * Perform some mapping to deal with the fact that
           * Microsoft defined the same thing twice.
           *----------------------------------------------*/

           if ( TempAmp.sMode == DATATYPE_RIFF_MULAW )
              {
              pAudioSettings->ulDataType = DATATYPE_MULAW;
              }

           if ( TempAmp.sMode == DATATYPE_RIFF_ALAW )
              {
              pAudioSettings->ulDataType = DATATYPE_ALAW;
              }

           }

         /*-------------------------------------------
         * IBM_ADPCM by definition is a 4bit type.
         * compressed 16 bit.
         *-------------------------------------------*/


         if ( TempAmp.sMode == DATATYPE_ADPCM_AVC )
           {
           pAudioSettings->ulFlags |= MCI_WAVE_SET_BITSPERSAMPLE;
           pAudioSettings->ulBitsPerSample = 16;
           }

         }

       /* Store the requested values */

       if (pAudioSettings->ulFlags & VSD_BITSPERSAMPLE)

         {
         TempAmp.lBitsPerSRate = pAudioSettings->ulBitsPerSample;
         }

       if (pAudioSettings->ulFlags & VSD_SAMPLESPERSEC)

         {
         TempAmp.lSRate = pAudioSettings->ulSamplingRate;
         }

       if (pAudioSettings->ulFlags & VSD_CHANNELS)

         {
         TempAmp.sChannels = pAudioSettings->ulChannels;
         }

       /******************************************************
       * When the VSD was loaded, it brought in a resource file
       * check the resource file if this particular device
       * supports the mode the caller requested.
       ******************************************************/

       lError = GetClassInformation( &TempAmp );

       /******************************************************
       * Fix for varied sample rates, if all flags come in
       * assume that sample rates are valid
       ******************************************************/

       if ( pAudioSettings->ulFlags != (   VSD_BITSPERSAMPLE |
                                           VSD_MODE          |
                                           VSD_CHANNELS      |
                                           VSD_SAMPLESPERSEC ) )
          {

          if ( lError )
             {
             pAudioSettings->ulDataType       = TempAmp.sBestFitMode;
             pAudioSettings->ulBitsPerSample  = TempAmp.ulBestFitBPS;
             pAudioSettings->ulChannels       = TempAmp.ulBestFitChan;
             pAudioSettings->ulSamplingRate   = TempAmp.ulBestFitRate;

             return (lError);
             }

          }

       else
          {
          /*--------------------------------------------------*
          * Since we will have to do a best fit on the file
          * all of the resource management stuff is messed
          * up.  Set a variable to indicate that we will have
          * to inform MDM of a possible resource class/unit
          * change.
          *---------------------------------------------------*/
          if ( lError )
             {
             if ( pInstance->ulDeviceID == MACPA &&
                  lError == VSDERR_UNSUPP_SAMPLESPERSEC )
                {
                lError = 0;
                MapModes( &TempAmp );
                }
             else
                {
                pAudioSettings->ulDataType       = TempAmp.sBestFitMode;
                pAudioSettings->ulBitsPerSample  = TempAmp.ulBestFitBPS;
                pAudioSettings->ulChannels       = TempAmp.ulBestFitChan;
                pAudioSettings->ulSamplingRate = TempAmp.ulBestFitRate;
                return ( lError );

                }
             } /* If we could not find this device */

          lError = 0;
          }

       /*********************************************************
       * Copy Temporary Instance Parameters into Actual Instance
       **********************************************************/

       memcpy ( pInstance, &TempAmp, sizeof (MCI_AMP_INSTANCE));


       /***********************************************************
       * For ACPA, block alignment is simply bits/per sample * channels
       * othercards may have a different algorithm
       **********************************************************/

       pInstance->ulBlockAlignment = ( pInstance->lBitsPerSRate / 8 )
                                         * pInstance->sChannels;

//       if ( ulOldResources > ( SHORT ) pInstance->ulResourcesUsed )
//          {
//          ulResourceDecrease = TRUE;
//          }
//       else
//          {
//          ulResourceDecrease = FALSE;
//          }
       if ( ulOldResources != pInstance->ulResourcesUsed  ||
            ulOldClass     != pInstance->ulClass )
          {
          ulResourceChange = TRUE;
          }


       mciChangeResource.pInstance = ( PVOID ) pInstance;
       mciChangeResource.usResourceUnits = ( USHORT ) pInstance->ulResourcesUsed;
       mciChangeResource.usResourceClass = ( USHORT ) pInstance->ulClass;
       mciChangeResource.usResourcePriority = 0;

       if ( ulResourceChange )
          {

          /*-------------------------------------------
          * Report to MDM the number of resource units
          * this new mode will consume.
          *------------------------------------------*/

           lError = mciSendCommand( pInstance->usDeviceID,
                                    MCIDRV_CHANGERESOURCE,
                                    MCI_WAIT,
                                    ( DWORD ) &mciChangeResource,
                                    0 );

           if ( ( DWORD_LOWD( lError ) != VSDERR_SUCCESS ) )
             {
             pInstance->ulResourcesUsed = ulOldResources;
             return ( lError );
             }
           }

         if ( !pInstance->ulActive )
           {
           /* Update the streaming subtype */

           pAudioSettings->ulDataSubType  =  pInstance->ulSubType;

           return ( VSDERR_SUCCESS );
           }

//         /* Check to see if a best fit was done */
//
//         ulMatch = pInstance->ulMatch;

         /*----------------------------------------
         * Setup the card in the appropriate mode
         * i.e. PCM, MIDI etc.
         *---------------------------------------*/

         lError = DoIOCTLLoad( pInstance, pInstance->ulHardwareMode );

         if ( !lError )
            {
          if ( ulResourceChange )
             {

               /*----------------------------------------*
               * Place this code here since the DoIOCTL
               * May have done a best fit and changed
               * the resource class.
               *----------------------------------------*/

               mciChangeResource.usResourceUnits = ( USHORT ) pInstance->ulResourcesUsed;
               mciChangeResource.usResourceClass = ( USHORT ) pInstance->ulClass;
               mciChangeResource.usResourcePriority = 0;

               lError = mciSendCommand( pInstance->usDeviceID,
                                        MCIDRV_CHANGERESOURCE,
                                        MCI_WAIT,
                                        ( DWORD ) &mciChangeResource,
                                        0 );

               if ( ( DWORD_LOWD( lError ) != VSDERR_SUCCESS ) )
                 {
                 pInstance->ulResourcesUsed = ulOldResources;
                 return ( lError );
                 }
             } /* If resource change */
            } /* if no errors after DOIOCTL */

       /*-----------------------------------------------------------------*
       * Call set driver to restore instance volume, balance, etc...
       *-----------------------------------------------------------------*/

       if ( !lError )
          {
          lError = ModifyAudioAttributes(pInstance, VSD_SET_ALL );
          }

       /*-----------------------------------------------------------------*
       * Return private informaiton to the IBM amp-mixer.
       * Used only by AUDIODD drivers.
       *-----------------------------------------------------------------*/

       pAudioSettings->ulReserved1 = pInstance->ulGlobalFile;
       pAudioSettings->ulDataSubType  =  pInstance->ulSubType;

       }
       break;


    case VSD_SETMONITOR:
       {
//       ULONG    ulOldMon = pInstance->lMonitor;   /* Holds the status of the
//                                                     instance monitor */

       /*----------------------------------------
       * Monitor requires that the paramter be
       * non-zero if on, 0 (FALSE) if not.
       *----------------------------------------*/

       if ( pRequest )
          {
          pInstance->lMonitor = TRUE;
          }
       else
          {
          pInstance->lMonitor = FALSE;
          }

        /*-----------------------------------------------------
        * if we opened a real mixer (i.e no AUDIODD), then the
        * IOCTLs will not be necessary.
        *----------------------------------------------------*/
           lError = ModifyAudioAttributes(pInstance, VSD_SET_ALL );

        } /* VSD_SET_MONITOR */
        break;

    case VSD_SETVOLUME :
       {
       PVSD_VOLUME_PARMS   pVol;
//       MIXERCONTROL        MixerControl;

       pVol = ( PVSD_VOLUME_PARMS ) pRequest;
       ulFlags = VSD_SET_VOLUME;

       if ( pVol->ulFlags & VSD_VOLUME )
          {
          pInstance->lLeftVolume = pVol->ulVolume;
          ulFlags = VSD_SET_VOLUME;

          }

       if ( pVol->ulFlags & VSD_MASTERVOLUME )
          {
          pInstance->ulMasterVolume = pVol->ulMasterAudio;
          ulFlags = VSD_SET_MASTER;
          } /* Mastervolume flag is passed in */
       lError = ModifyAudioAttributes( pInstance, ulFlags );
       }
       break;
    case VSD_SETAUDIOATTRIBUTES:
       {
       PVSD_AUDIOATTRIBUTES_PARMS  pAudioAttr;

       pAudioAttr = ( PVSD_AUDIOATTRIBUTES_PARMS ) pRequest;

       if ( pAudioAttr->ulFlags == VSD_TREBLE )
          {

          pInstance->lTreble = pAudioAttr->ulValue;
          ulFlags = VSD_SET_TREBLE;
          }

       if ( pAudioAttr->ulFlags == VSD_BASS )
          {
          pInstance->lBass = pAudioAttr->ulValue;
          ulFlags = VSD_SET_BASS;
          }

       if ( pAudioAttr->ulFlags == VSD_PITCH )
          {
          pInstance->lPitch = pAudioAttr->ulValue;
          ulFlags = VSD_SET_PITCH;
          }

       if ( pAudioAttr->ulFlags == VSD_BALANCE )
          {
          pInstance->lBalance = pAudioAttr->ulValue;
          ulFlags = VSD_SET_BALANCE | VSD_VOLUME;
          }

       if ( pAudioAttr->ulFlags == VSD_GAIN )
          {
          pInstance->ulGainLevel = pAudioAttr->ulValue;
          ulFlags = VSD_SET_GAIN | VSD_VOLUME;
          }


       lError = ModifyAudioAttributes(pInstance, ulFlags );

       } /* case set audio attributes */

       break;
    case VSD_SETMIXCONNECTIONS:
       if ( pInstance->fHardwareMix )
          {
          lError = mixSetConnections ( pInstance->hMix,
                                       ( PLINECONNECTIONS ) pRequest );
          }
       else
          {
          lError = VSDERR_UNSUPPORTED_COMMAND;
          }
       break;
    case VSD_SETMIXCONTROL:
       if ( pInstance->fHardwareMix )
          {
          lError = mixSetControl ( pInstance->hMix,
                                     ( PMIXERCONTROL ) pRequest );
          }
       else
          {
          lError = VSDERR_UNSUPPORTED_COMMAND;
          }

       break;

    case VSD_SETDISPLAY         :
    case VSD_SETINPUTLEVEL      :
    case VSD_SETDOOR            :
    case VSD_SETCOMMSETTINGS    :
    case VSD_SETRATE            :
    case VSD_SETKEYLOCK         :
    case VSD_SETCUE             :
    case VSD_SETCOUNTER         :
    case VSD_SETPOSITION        :
    case VSD_SETDIRECTION       :
    case VSD_SETVIEWPORTPOSITION:
    case VSD_SETVIEWPORT        :
    case VSD_SETIMAGEFORMAT     :
          lError = VSDERR_UNSUPPORTED_FLAG;
          break;


    /*--------------------------------------------------------------------------------
    * The IBM Amp-mixer device has the notion of logical connections. See below:
    *
    *       Input                                            Output
    *                  -------------------------------------
    *                  |                                  |
    *    Microphone  - |                                  |-Speakers (Internal and External)
    *    Line In     - |       Mixer Device               |-Headphones
    * Internal Audio - |                                  |-
    *                  |                                  |
    * Amp Stream     - |                                  |
    *                  |                                  |
    *                  ------------------------------------
    *
    * However, the logical mixer device performs no checking on the type of connectors
    * so, vendors can easily add support for additional connectors.  The logical mixer
    * device takes the request to enable, disable or query the connector and forwards
    * it directly to the VSD.  The VSD should either perform the action or return
    * one of the following errors:
    *
    *     VSDERR_UNSUPPORTED_CONN_TYPE
    *     VSDERR_CANNOT_MODIFY_CONNECTOR
    *     VSDERR_INVALID_CONNECTOR_TYPE
    *-------------------------------------------------------------------------------*/


    case VSD_SETCONNECTOR:
         {
         PVSD_CONNECTOR_PARMS   pVsdCon;

         pVsdCon = ( PVSD_CONNECTOR_PARMS ) pRequest;

         if ( pVsdCon->ulConn_Type  == MCI_HEADPHONES_CONNECTOR  ||
              pVsdCon->ulConn_Type  == MCI_VIDEO_IN_CONNECTOR    ||
              pVsdCon->ulConn_Type  == MCI_VIDEO_OUT_CONNECTOR   ||
              pVsdCon->ulConn_Type  == MCI_PHONE_SET_CONNECTOR   ||
              pVsdCon->ulConn_Type  == MCI_PHONE_LINE_CONNECTOR  ||
              pVsdCon->ulConn_Type  == MCI_UNIVERSAL_CONNECTOR   ||
              pVsdCon->ulConn_Type  == MCI_MIDI_STREAM_CONNECTOR ||
              pVsdCon->ulConn_Type  == MCI_CD_STREAM_CONNECTOR   ||
              pVsdCon->ulConn_Type  == MCI_WAVE_STREAM_CONNECTOR )
            {
            return ( VSDERR_UNSUPPORTED_CONN_TYPE );
            }

         if ( lError = ValidateIndex( pVsdCon->ulConn_Type, pVsdCon->ulIndex  ) )
            {
            return (lError );
            }

         if ( pVsdCon->ulConn_Type  == MCI_AMP_STREAM_CONNECTOR )
            {
            return ( VSDERR_CANNOT_MODIFY_CONNECTOR );
            }

            /*--------------------------------------------------
            * Does the device support the new mixer IOCTL set?
            * If it does, then enabling the connectors will
            * utilze the new IOCTL set.
            *-------------------------------------------------*/
            if ( pInstance->fHardwareMix )
               {
                LINECONNECTIONS     mixinfo;

                /* The MCI connectors map directly to mixer connectors */

                mixinfo.ulLine       = pVsdCon->ulConn_Type;
                /*--------------------------------------------------
                * The instance variables, ulInputDev and ulOutputDev
                * contain a bitmap of the enabled connectors.
                * Update the state of the connectors due to the
                * request to enable/disable.
                *-------------------------------------------------*/

                if ( pVsdCon->fEnabled == VSD_DISABLE )
                   {
                   switch( pVsdCon->ulConn_Type )
                      {
                      case MCI_LINE_IN_CONNECTOR         :
                      case MCI_AUDIO_IN_CONNECTOR         :
                         pInstance->ulInputDev &= ~SOURCE_LINE;

                         break;
                      case MCI_INTERNAL_AUDIO_CONNECTOR  :
                         pInstance->ulInputDev &= ~SOURCE_INTERNAL_AUDIO;

                         break;
                      case MCI_MICROPHONE_CONNECTOR      :
                         pInstance->ulInputDev &= ~SOURCE_MICROPHONE;

                         break;
                      case MCI_LINE_OUT_CONNECTOR        :
                      case MCI_AUDIO_OUT_CONNECTOR       :
                         pInstance->ulOutputDev &= ~SINK_LINE_OUT;

                         break;
                      case MCI_SPEAKERS_CONNECTOR        :
                         if ( pVsdCon->ulIndex == 1)
                            {
                            pInstance->ulOutputDev &= ~SINK_SPEAKER;
                            }
                         else
                            {
                            pInstance->ulInputDev &= ~SOURCE_PC_SPEAKER;
                            }
                         break;

                      case MCI_HEADPHONES_CONNECTOR      :
                         pInstance->ulOutputDev &= ~SINK_HEADPHONES;
                         break;
                      default :
                         return ( VSDERR_INVALID_CONNECTOR_TYPE );
                      } /* switch the connector type */
                   }
                else
                   {
                   switch( pVsdCon->ulConn_Type )
                      {
                      case MCI_LINE_IN_CONNECTOR         :
                      case MCI_AUDIO_IN_CONNECTOR         :
                         pInstance->ulInputDev |= SOURCE_LINE;

                         break;
                      case MCI_INTERNAL_AUDIO_CONNECTOR  :
                         pInstance->ulInputDev |= SOURCE_INTERNAL_AUDIO;

                         break;
                      case MCI_MICROPHONE_CONNECTOR      :
                         pInstance->ulInputDev |= SOURCE_MICROPHONE;

                         break;
                      case MCI_LINE_OUT_CONNECTOR        :
                      case MCI_AUDIO_OUT_CONNECTOR       :
                         pInstance->ulOutputDev |= SINK_LINE_OUT;

                         break;
                      case MCI_SPEAKERS_CONNECTOR        :
                         if ( pVsdCon->ulIndex == 1)
                            {
                            pInstance->ulOutputDev |= SINK_SPEAKER;
                            }
                         else
                            {
                            pInstance->ulInputDev |= SOURCE_PC_SPEAKER;
                            }
                         break;

                      case MCI_HEADPHONES_CONNECTOR      :
                         pInstance->ulOutputDev |= SINK_HEADPHONES;
                         break;
                      default :
                         return ( VSDERR_INVALID_CONNECTOR_TYPE );
                      } /* switch the connector type */
                   } /* if we should enable a connector */

               mixinfo.ulConnection = pInstance->ulOutputDev;
               mixinfo.ulLine       = pInstance->ulInputDev;
               mixinfo.ulLength     = sizeof ( LINECONNECTIONS );

               /*-------------------------------
               * Enable/Disable the connector.
               *-------------------------------*/

               lError = mixSetConnections ( pInstance->hMix,
                                            &mixinfo );
               }
            else
               {
                /* If the caller wishes to disable a connector */

                if ( pVsdCon->fEnabled == VSD_DISABLE)
                   {
                   switch( pVsdCon->ulConn_Type )
                       {
                       case MCI_LINE_IN_CONNECTOR:
                       case MCI_AUDIO_IN_CONNECTOR   :
                               if ( StatusIOPort( STEREO_LINE_INPUT, pInstance, INPUT_PORT ) == MCI_TRUE )
                                  {
                                  lError = RemoveIOPort( STEREO_LINE_INPUT,
                                                         pInstance,
                                                         INPUT_PORT );
                                  }

                               break;

                       case MCI_MICROPHONE_CONNECTOR:
                               if ( pInstance->ulGainLevel >= 70 )
                                  {
                                  ulMicType = BOOSTED_MIC_INPUT;
                                  }
                               else
                                  {
                                  ulMicType = MIC_INPUT;
                                  }
                               if ( StatusIOPort( ulMicType,
                                                  pInstance,
                                                  INPUT_PORT ) == MCI_TRUE )
                                  {
                                  lError = RemoveIOPort( ulMicType,
                                                         pInstance,
                                                         INPUT_PORT );
                                  }

                               break;

                       case MCI_LINE_OUT_CONNECTOR:
                       case MCI_AUDIO_OUT_CONNECTOR  :
                               if ( StatusIOPort( STEREO_LINE_OUTPUT, pInstance, OUTPUT_PORT ) == MCI_TRUE )
                                  {
                                  lError = RemoveIOPort( STEREO_LINE_OUTPUT,
                                                         pInstance,
                                                         OUTPUT_PORT );
                                  }

                               break;

                       case MCI_SPEAKERS_CONNECTOR:
                               if ( pVsdCon->ulIndex == 2 )
                                  {
                                  ulSpeakerType = SPEAKER_OUTPUT;
                                  }
                               else
                                  {
                                  ulSpeakerType = HEADSET_OUTPUT;
                                  }
                               if ( StatusIOPort( ulSpeakerType, pInstance, OUTPUT_PORT ) == MCI_TRUE )
                                  {
                                  lError = RemoveIOPort( ulSpeakerType,
                                                         pInstance,
                                                         OUTPUT_PORT );
                                  }

                               break;


                       case MCI_HEADPHONES_CONNECTOR:
                       case MCI_VIDEO_IN_CONNECTOR   :
                       case MCI_VIDEO_OUT_CONNECTOR  :
                       case MCI_PHONE_SET_CONNECTOR  :
                       case MCI_PHONE_LINE_CONNECTOR :
                       case MCI_UNIVERSAL_CONNECTOR  :

                               lError = VSDERR_UNSUPPORTED_CONN_TYPE;
                               break;
                       default:
                               lError = VSDERR_INVALID_CONNECTOR_TYPE;
                               break;
                       } /* switch connector type */

                     if ( pInstance->ulActive )
                       {
                       ModifyAudioAttributes(pInstance, VSD_SET_ALL );
                       }

                   }  /* if disabling a connector */

                else
                   {
                   /*-------------------------------------------------------------*
                   * Enable connector
                   *-------------------------------------------------------------*/

                   switch( pVsdCon->ulConn_Type )
                      {
                      case MCI_LINE_IN_CONNECTOR    :
                      case MCI_AUDIO_IN_CONNECTOR   :
                         if (StatusIOPort( STEREO_LINE_INPUT, pInstance, INPUT_PORT ) != MCI_TRUE )
                            {
                            lError = AddIOPort( STEREO_LINE_INPUT,
                                              pInstance,
                                              INPUT_PORT );
                            }
                         break;

                      case MCI_MICROPHONE_CONNECTOR:
                          if ( pInstance->ulGainLevel >= 70 )
                             {
                             ulMicType = BOOSTED_MIC_INPUT;
                             }
                          else
                             {
                             ulMicType = MIC_INPUT;
                             }
                          if ( StatusIOPort( ulMicType, pInstance, INPUT_PORT ) != MCI_TRUE )
                             {
                             lError = AddIOPort( ulMicType,
                                               pInstance,
                                               INPUT_PORT );
                             }
                          break;

                      case MCI_LINE_OUT_CONNECTOR:
                      case MCI_AUDIO_OUT_CONNECTOR  :
                          if (StatusIOPort( STEREO_LINE_OUTPUT, pInstance, OUTPUT_PORT ) != MCI_TRUE)
                             {
                             lError = AddIOPort( STEREO_LINE_OUTPUT,
                                               pInstance,
                                               OUTPUT_PORT );
                             }

                          break;

                      case MCI_SPEAKERS_CONNECTOR:
                          if ( pVsdCon->ulIndex == 2 )
                             {
                             ulSpeakerType = SPEAKER_OUTPUT;
                             }
                          else
                             {
                             ulSpeakerType = HEADSET_OUTPUT;
                             }

                          if (StatusIOPort( ulSpeakerType, pInstance, OUTPUT_PORT ) != MCI_TRUE )
                             {
                             lError = AddIOPort( ulSpeakerType,
                                               pInstance,
                                               OUTPUT_PORT );
                             }
                          break;

                      default:
                               lError = VSDERR_INVALID_CONNECTOR_TYPE;
                              break;
                      } /* switch connector type */

                    /*-----------------------------------------------------
                    * if we opened a real mixer (i.e no AUDIODD), then the
                    * IOCTLs will not be necessary.
                    *----------------------------------------------------*/

                    if ( pInstance->ulActive && !pInstance->fHardwareMix )
                       {
                       ModifyAudioAttributes(pInstance, VSD_SET_ALL );
                       }
                   } /* else enable a connector */
               } /* else no hardware mixing functions */

         } /* set connector case */
         break;
     } /* switch ulFlags */

    return(lError);

    } /* VsdSetCommand */


