/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/************************ START OF SPECIinCATIONS ***
*
* SOURCE FILE NAME:  VSDIOCTL.C
*
* DESCRIPTIVE NAME: AUDIODD interface routines
*
*
* STATUS:  OS/2 Release 2.0
*
* FUNCTION: This source module contains the Operating System dependent
*           calls to access the associated device driver.  In the case
*           of OS/2 2.x, and AUDIODD driver is assumed and the
*           DosDevIOCTL interface is used.  If other hardware or OSes
*           are in use, then these routines must be modified.
*
* NOTES:
*        DEPENDENCIES: ACPA PDD
*        RESTRICTIONS: Runs in 32 bit protect mode (OS/2 2.0)
*
* ENTRY POINT:
*        AUDIOIFDriverEntry
*
* EXTERNAL REFERENCES: Sends IOCTL request to ACPA audio board
*
*/

#define INCL_NOPMAPI
#define INCL_DOS
#define INCL_ERRORS
#define INCL_AUDIO_VSD
#include <os2.h>
#include <os2me.h>
#include <mcd.h>
#include <audio.h>
#include <stdio.h>
#include <string.h>
#include <hhpheap.h>


#include <vsdcmds.h>
#include <vsdaud.h>
#include <os2mixer.h>

ULONG mixSetControl( HMIXER hMixer, PMIXERCONTROL   pMixControl );

/************************** START OF SPECIFICATIONS ************************
*                                                                          *
* SUBROUTINE NAME: DoIOCTLLoad                                             *
*                                                                          *
* DESCRIPTIVE NAME: Status for a port                                      *
*                                                                          *
* FUNCTION: Returns if a particular port is currently active.              *
*                                                                          *
* NOTES:    This will init the DSP and load any necessary code.            *
*                                                                          *
* INPUT:    ulPort                                                         *
*                                                                          *
* OUTPUT:   MCI_TRUE                                                       *
*           MCI_FALSE                                                      *
*                                                                          *
* SIDE EFFECTS:                                                            *
*                                                                          *
*************************** END OF SPECIFICATIONS **************************/

LONG  DoIOCTLLoad( PVSD_INSTANCE  pInstance,
                   SHORT          sMode )



{
extern HMTX        hmtxProcessSemVSD;     // Semaphore global to all processes
                                       // which controls ioctl access

LONG               rc;
LONG               lError = VSDERR_SUCCESS;
ULONG              ulParmLengthInOut = 0L;
ULONG              ulDataLengthInOut = (ULONG) sizeof(MCI_AUDIO_INIT);

MCI_AUDIO_INIT     AudioInit;


   // Perform Audio Init IOCTL

   /*-------------------------------------------------------
   * AUDIODD has a tremendous limitation--the modes don't map
   * with RIFF datatypes.  We must perform mapping here.
   * Drivers that support the new capability IOCTL will indicate
   * that this mapping is not necessary.
   *------------------------------------------------------*/

   if ( !pInstance->ulMatch )
      {
      MapModes( pInstance );
      }


   /*-------------------------------------------------------
   * Because the ACPA device driver is older, it has bugs that
   * must be worked around.  You will see numerous comments
   * like if ( pInstance->ulDeviceID == MACPA ),
   * you can safely remove these from your VSD code.
   *------------------------------------------------------*/


   if ( pInstance->ulDeviceID == MACPA )
      {
      /*-------------------------------------------------------
      * Until the 1.03 M-AUDIO driver drops, we will have to
      * do the BEST-FIT in the VSD.
      *------------------------------------------------------*/

      if ( ( !pInstance->ulMatch &&
             pInstance->sMode == DATATYPE_WAVEFORM ) )

         {
         ACPABestFit( pInstance );
         }  /* if best fit is required */

      } /* If this is an ACPA before the 1.03 drop */


   /* Fill in AUDIO INIT structure */

   AudioInit.lResolution   = 0L;
   AudioInit.lSRate        =  pInstance->lSRate;
   AudioInit.lBitsPerSRate = pInstance->lBitsPerSRate;
   AudioInit.sMode         = sMode;

   AudioInit.ulOperation   = (SHORT) pInstance->ulOperation;
   AudioInit.sChannels     = pInstance->sChannels;

   AudioInit.ulFlags       = pInstance->ulFlags;
   if ( pInstance->fDriverCaps & SUPPORT_RIFF_MODES )
      {
      AudioInit.ulFlags |= RIFF_DATATYPE;
      }


   AudioInit.sDeviceID     = pInstance->ulDeviceID;
   AudioInit.sReturnCode   = 0;
   AudioInit.pvReserved    = NULL;
   AudioInit.ulVersionLevel= CURRENT_VERSION;

   if ( AudioInit.sMode == PCM && AudioInit.lBitsPerSRate == 16)
      AudioInit.ulFlags |= TWOS_COMPLEMENT;
   else
     AudioInit.ulFlags &= ~TWOS_COMPLEMENT;





   /*---------------------------------------------------------*
   * Do Audio Init IOCtl call
   *---------------------------------------------------------*/

   ulParmLengthInOut = 0L;
   ulDataLengthInOut = (ULONG) sizeof(MCI_AUDIO_INIT);

   /*---------------------------------------------------------*
   * Utilize a process wide semaphore to prevent more than
   * one process from accessing the device driver at the same
   * time.
   *
   * This is necessary since the amp has instance based
   * semaphores, but not cross process semaphores.
   *---------------------------------------------------------*/

   DosRequestMutexSem( hmtxProcessSemVSD, -1 );


   rc = DosDevIOCtl( pInstance->hFile,
                     AUDIO_IOCTL_CAT,
                     AUDIO_INIT,
                     NULL,
                     0L,
                     &ulParmLengthInOut,
                     &AudioInit,
                     (LONG) sizeof(MCI_AUDIO_INIT),
                     &ulDataLengthInOut);

   DosReleaseMutexSem( hmtxProcessSemVSD );


   if ( rc || AudioInit.sReturnCode )
     {
     if ( AudioInit.sReturnCode == OVERLOADED )
       {
       return ( VSDERR_RESOURCE_NOT_AVAILABLE );
       }
     else
       {
       return ( VSDERR_HARDWARE );
       }
     }
   else
     {
     pInstance->ulGlobalFile = (ULONG) AudioInit.pvReserved;

     if ( AudioInit.ulFlags & BESTFIT_PROVIDED )
        {
        pInstance->lSRate = AudioInit.lSRate;
        if ( pInstance->lBitsPerSRate != AudioInit.lBitsPerSRate ||
             pInstance->sChannels != AudioInit.sChannels )
           {
           return (  VSDERR_HARDWARE );
           }

        /* Get resource information, class data etc. */

        GetClassInformation( pInstance );

        } /* Best fit flag was set */

     if ( AudioInit.ulFlags & VOLUME )
        {
        pInstance->ulVolControl = TRUE;
        }
     else
        {
        pInstance->ulVolControl = FALSE;
        }
     }

   // if a DSP module has not been loaded, then bring it in

   if (AudioInit.ulFlags & LOAD_CODE)
      {
      lError = LoadDSP(pInstance, &AudioInit);
      }

// 10253

   return ( lError );


} /* DoIOCTLLoad */


/************************ START OF SPECIFICATIONS **************************
*
* SUBROUTINE NAME: LoadDSP()
*
* FUNCTION: Loads DSP module.
*
* INPUT:
*
* OUTPUT:
*
* OS/2 CALLS: None
*
* C CALLS: None
*
*************************** END OF SPECIFICATIONS *************************/


LONG LoadDSP( PVSD_INSTANCE      pInstance,
              LPMCI_AUDIO_INIT   pAudioInit)
        {
        LONG rc;
        LONG lError = VSDERR_SUCCESS;
        HFILE hDSPFile;
        ULONG ulParmLengthInOut;
        ULONG ulDataLengthInOut;
        ULONG ulAction;
        ULONG ulDSPFileSize;
        ULONG ulBytesRead;
//        USHORT usDSPFound = MCI_FALSE;
        SHORT sDSPFileOpen;
        CHAR szDSPFile[261], *pEnv;
        FILESTATUS rFileInfo;
        ULONG *pulDSPBase32;
        ULONG *pulDSPBase16;
        CHAR abParmList[128];
        MCI_AUDIO_LOAD rAudioLoad;

        /*-----------------------------------------------------------------*
        * Find and open DSP file
        *-----------------------------------------------------------------*/

        rc = DosScanEnv("MMBASE", &pEnv);

        strcpy(szDSPFile, pEnv);

        if (strlen(szDSPFile))
                {
                if (szDSPFile[strlen(szDSPFile) - 1] == ';')
                        szDSPFile[strlen(szDSPFile) - 1] = 0;

                if (strlen(szDSPFile))
                        {
                        if (szDSPFile[strlen(szDSPFile) - 1] != '\\')
                                strcat(szDSPFile,"\\DSP\\");
                        }
                }

        strcat(szDSPFile,pAudioInit->abLoadPath);

        rc = DosOpen(szDSPFile,
                                 &hDSPFile,
                                 &ulAction,
                                 0L,
                                 FILE_NORMAL,
                                 FILE_OPEN,
                                 OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE,
                                 0);

        if (rc)
                lError = VSDERR_CANNOT_LOAD_DSP_MOD;

        if (lError)
                {
                sDSPFileOpen = MCI_FALSE;
                }
        else
                sDSPFileOpen = MCI_TRUE;

        /*-----------------------------------------------------------------*
        * Get DSP File Size
        *-----------------------------------------------------------------*/

        if (!lError)
                {
                rc = DosQueryFileInfo(hDSPFile,
                                      1L,
                                      &rFileInfo,
                                      sizeof(FILESTATUS));

                lError = MCI_Error(rc);
                ulDSPFileSize = rFileInfo.cbFile;
                }

        /*-------------------------------------------------*
        * Allocate Buffer to Store DSP Code
        *-------------------------------------------------*/

        if (!lError && ulDSPFileSize > 0)
                {
                rc = DosAllocSharedMem((PVOID)&pulDSPBase32,
                                       NULL,
                                       ulDSPFileSize,
                                       PAG_COMMIT | OBJ_GIVEABLE |
                                       PAG_READ | PAG_WRITE);

                lError = MCI_Error(rc);
                }

        /*-------------------------------------------------*
        * Read DSP File into Buffer
        *-------------------------------------------------*/

        if (!lError && ulDSPFileSize > 0)
                {
                rc = DosRead(hDSPFile,
                             pulDSPBase32,
                             ulDSPFileSize,
                             &ulBytesRead);


                lError = MCI_Error(rc);
                }

        /*-------------------------------------------------*
        * Convert 32 bit address to 16 bit address
        *-------------------------------------------------*/

        /*
        * Cant seem to get MAKEP work reliably under C-Set/2 so
        * wrote my own conversion, note, this may not be the best
        * solution
        */

        pulDSPBase16 = ( PULONG ) MAKEULONG( LOUSHORT( pulDSPBase32  ),
                                               (HIUSHORT( pulDSPBase32 ) << 3) + 7);

        /*-------------------------------------------------*
        * Call AUDIO_LOAD IOCtl
        *-------------------------------------------------*/

        if (!lError && ulDSPFileSize > 0)
                {
                rAudioLoad.pbBuffer = (CHAR FAR *) pulDSPBase16;
                rAudioLoad.ulSize = (USHORT) ulBytesRead;
                rAudioLoad.ulFlags = LOAD_START | LOAD_END;

                ulParmLengthInOut = 0L;
                ulDataLengthInOut = (LONG) sizeof(MCI_AUDIO_LOAD);

                rc = DosDevIOCtl(pInstance->hFile,
                                AUDIO_IOCTL_CAT,
                                AUDIO_LOAD,
                                abParmList,
                                128L,
                                &ulParmLengthInOut,
                                &rAudioLoad,
                                (LONG) sizeof(MCI_AUDIO_LOAD),
                                &ulDataLengthInOut);

                lError = MCI_Error(rc);
                DosFreeMem(pulDSPBase32);
                }

        /*-------------------------------------------------*
        * Close DSP File
        *-------------------------------------------------*/

        if (sDSPFileOpen)
                DosClose(hDSPFile);

        return(lError);

        } /* LoadDSP */



/************************ START OF SPECIFICATIONS **************************
*
* SUBROUTINE NAME: OpenDevice()
*
* FUNCTION: Opens the device driver
*
* INPUT:
*
* OUTPUT:
*
* OS/2 CALLS: None
*
* C CALLS: None
*
*************************** END OF SPECIFICATIONS *************************/

ULONG OpenDevice ( PVSD_INSTANCE      pInstance )

{
   ULONG ulAction;
   ULONG rc;


   /*-----------------------------------------------------------------*
   * Open the device driver--and see if it is there
   * NOTE: It is essential to use the noinherit flag.  If you do not
   * use this flag, child processes spawned from this one will duplicate
   * all open handles causing resource to be placed on the device that
   * will never leave it.
   *-----------------------------------------------------------------*/

   rc = DosOpen( pInstance->szDeviceName,
                 &pInstance->hFile,
                 &ulAction,
                 0,
                 FILE_NORMAL,
                 FILE_OPEN,
                 OPEN_ACCESS_READWRITE    | OPEN_SHARE_DENYNONE |
                 OPEN_FLAGS_FAIL_ON_ERROR | OPEN_FLAGS_NOINHERIT,
                 0);

   if ( !rc )
     {
     /***********************************************
     * Speed improvement, keep the device open--there
     * is no penalty if we do so.
     ************************************************/
     pInstance->ulDosOpened = TRUE;

     }

   else
     {
     /*-------------------------------------------------
     * Single-instance device drivers will
     * reject multiple opens :-(.  Handle as necessary.
     *-------------------------------------------------*/

     if ( rc == ERROR_WRITE_PROTECT )
        {
        return ( VSDERR_RESOURCE_NOT_AVAILABLE );
        }

     return ( VSDERR_NO_DEVICE_DRIVER );
     }

   return ( VSDERR_SUCCESS );

} /* OpenDevice */

/************************** START OF SPECIFICATIONS ************************
*                                                                          *
* SUBROUTINE NAME: CapabilityIOCTL                                         *
*                                                                          *
* DESCRIPTIVE NAME: Determines if the device can perform certain actions   *
*                                                                          *
* FUNCTION: This fucntion calls the new capability IOCTL.  If an error is  *
*           returns, it is assumed that the device supports the older      *
*           resource management version of capabilities (see OS/2 2.1 docs)*
*                                                                          *
* OUTPUT:   VSDERR_SUCESS if there was a match                             *
*           An MCI error otherwise                                         *
*                                                                          *
* SIDE EFFECTS:                                                            *
*                                                                          *
*************************** END OF SPECIFICATIONS **************************/


ULONG CapIOCTL( PVSD_INSTANCE    pInstance )
{

   ULONG              ulParmLengthInOut = 0L;
   ULONG              ulDataLengthInOut = sizeof(MCI_AUDIO_CAPS );
   ULONG              rc;
   MCI_AUDIO_CAPS     audiocap;

   /*---------------------------------------------------------
   * Determine if the device supports the mode that we requested
   *----------------------------------------------------------*/

   audiocap.ulSamplingRate = pInstance->lSRate;
   audiocap.ulChannels     = pInstance->sChannels;
   audiocap.ulBitsPerSample= pInstance->lBitsPerSRate;
   audiocap.ulDataType     = pInstance->sMode;
   audiocap.ulOperation    = pInstance->ulOperation;

   rc = DosDevIOCtl( pInstance->hFile,
                     AUDIO_IOCTL_CAT,
                     AUDIO_CAPABILITY,
                     NULL,
                     0L,
                     &ulParmLengthInOut,
                     &audiocap,
                     (LONG) sizeof(MCI_AUDIO_CAPS),
                     &ulDataLengthInOut);

   /*---------------------------------------------------------
   * If the IOCTL failed, or if the card doesn't support this
   * mode, inform the caller of the problem.
   *----------------------------------------------------------*/
   if ( rc || audiocap.ulSupport )
      {
      if ( rc )
         {
         // 10253 --this error is less than optimal
         pInstance->fDriverCaps = FALSE;
         return (VSDERR_HARDWARE );
         }

      switch ( audiocap.ulSupport )
         {
         case UNSUPPORTED_RATE     :
           rc = VSDERR_UNSUPP_SAMPLESPERSEC;
           break;
         case UNSUPPORTED_CHANNELS :
           rc = VSDERR_UNSUPP_CHANNELS;
           break;
         case UNSUPPORTED_BPS      :
           rc = VSDERR_UNSUPP_BITSPERSAMPLE;
           break;
         case UNSUPPORTED_DATATYPE :
           rc = VSDERR_UNSUPP_FORMAT_TAG;
           break;
         case UNSUPPORTED_OPERATION :
           rc = VSDERR_UNSUPP_FORMAT_MODE;
           break;
         } /* Switch error returned from the device driver */

      /* let calling function know the closest matching data type */

      pInstance->sBestFitMode  =   audiocap.ulDataType;
      pInstance->ulBestFitBPS  =   audiocap.ulBitsPerSample;
      pInstance->ulBestFitChan =   audiocap.ulChannels;
      pInstance->ulBestFitRate =   audiocap.ulSamplingRate;

      return ( rc );
      }

   /*---------------------------------------------------------
   * The device driver will return the following information
   * describing streaming capabilities and resource management
   * information.
   *----------------------------------------------------------*/

   pInstance->ulSubType       = audiocap.ulDataSubType;
   pInstance->ulClass         = audiocap.ulResourceClass;
   pInstance->ulResourcesUsed = audiocap.ulResourceUnits;
   pInstance->ulCanRecord     = audiocap.fCanRecord;
   pInstance->fDriverCaps     = audiocap.ulCapability | SUPPORT_CAP;

   return (VSDERR_SUCCESS );
} /* CapIOCTL */


GAINTABLE GainTable[ NUMGAINENTRIES ] =

      { { 0 },
        { 6 },
        { 12 },
        { 18 },
        { 24 },
        { 30 },
        { 36 },
        { 42 },
        { 48 },
        { 54 },
        { 60 },
        { 66 },
        { 72 },
        { 78 },
        { 84 },
        { 90 },
        { 96 },
        { 100 },
      };


typedef struct
        {
        ULONG   ulGainValue;
        } BOOSTGAINTABLE;

GAINTABLE BoostGainTable[ 7 ] =

      {  { 102 },
         { 104 },
         { 106 },
         { 108 },
         { 112 },
         { 114 },
         { 116 }
      };




/************************ START OF SPECIFICATIONS **************************
*
* SUBROUTINE NAME: ModifyAudioAttributes
*
* FUNCTION: Sends IOCTL request to ACPA PDD to change audio settings
*           Items will only be changed if the appropriate VSD_SET_xxx flag
*           is set (i.e. VSD_SET_VOLUME, VSD_SET_TREBLE etc.).  If a flag
*           is not sent, then the AUDIO_IGNORE flag is sent to the PDD>
*
* INPUT:  pInstance - pointer to VSD instance
*
* SUCCESS:  VSDERR_SUCCESS
*
* FAILURE:  VSD error code.
*
* OS/2 CALLS: DosDevIOCtl()
*
* C CALLS: None
*
* INTERNAL CALLS:
*
*************************** END OF SPECIFICATIONS *************************/

LONG ModifyAudioAttributes( PVSD_INSTANCE    pInstance,
                 ULONG            ulFlags )
{

   LONG                  lError = VSDERR_SUCCESS;
   ULONG                 ulDataLen;
   ULONG                 ulParmLengthInOut = 0L;
   LONG                  rc;
   LONG                  l64kCheck;

   ULONG                 ulVolume;
   ULONG                 ulMicType;
   ULONG                 ulNewBalance;
   ULONG                 ulHoldVolPercentage;
   ULONG                 ulGainPercentage;


   MCI_AUDIO_CONTROL     AudioControl;
   MCI_AUDIO_CHANGE      AudioChange;

   LPMCI_AUDIO_CHANGE    prAudioChange32;
   LPMCI_AUDIO_CHANGE    prAudioChange16;

   MCI_TRACK_INFO        rTrackInfo;

   LPMCI_TRACK_INFO      prTrackInfo16;
   LPMCI_TRACK_INFO      prTrackInfo32;

   BOOL                  fAllocatedChangeMemory = FALSE;
   BOOL                  fAllocatedTrackMemory = FALSE;

  if ( !pInstance->ulActive )
     {
     return ( MCIERR_SUCCESS );
     }

  /*-----------------------------------------------------------------*
  * The audio card either supports the new mixer IOCTLs or it doesn't.
  *-----------------------------------------------------------------*/


  if ( pInstance->fHardwareMix )
     {
       MIXERCONTROL        MixerControl;
       MixerControl.ulLength = sizeof ( MIXERCONTROL );


       if ( ulFlags == VSD_SET_ALL )
          {
          ulFlags = VSD_SET_VOLUME | VSD_SET_BALANCE;
          }

       if ( ulFlags & VSD_SET_MASTER )
          {
          MixerControl.ulControl = MIX_VOLUME;

          // LAD--note, this will have to become a define.

          MixerControl.ulSetting = pInstance->ulMasterVolume * MIXER_MULTIPLIER;
          MixerControl.ulLine = SINK_ALL;

          lError = mixSetControl( pInstance->hMix, &MixerControl );
          }

      if ( ulFlags & VSD_SET_VOLUME )
         {
         MixerControl.ulControl = MIX_VOLUME;

         // LAD--note, this will have to become a define.

         MixerControl.ulSetting = pInstance->lLeftVolume * MIXER_MULTIPLIER;
         MixerControl.ulLine = pInstance->ulLine;

         lError = mixSetControl( pInstance->hMix, &MixerControl );
         }


      if ( ulFlags & VSD_SET_BALANCE )
         {
         MixerControl.ulControl = MIX_BALANCE;

         // LAD--note, this will have to become a define.

         MixerControl.ulSetting = pInstance->lBalance * MIXER_MULTIPLIER;
         MixerControl.ulLine = pInstance->ulLine;

         lError = mixSetControl( pInstance->hMix, &MixerControl );
         }

      // 10253 -- new mixer interface will redefine treble/bass--wait till
      // drivers are ready

      if ( ulFlags & VSD_SET_TREBLE )
         {
         MixerControl.ulControl = MIX_TREBLE;

         // LAD--note, this will have to become a define.

         MixerControl.ulSetting = pInstance->lTreble * MIXER_MULTIPLIER;
         MixerControl.ulLine = pInstance->ulLine;

         lError = mixSetControl( pInstance->hMix, &MixerControl );
         }

      if ( ulFlags & VSD_SET_BASS )
         {
         MixerControl.ulControl = MIX_BASS;

         // LAD--note, this will have to become a define.

         MixerControl.ulSetting = pInstance->lBass * MIXER_MULTIPLIER;
         MixerControl.ulLine = pInstance->ulLine;

         lError = mixSetControl( pInstance->hMix, &MixerControl );
         }

      // 10253 -- this is still to be determined

      if ( ulFlags & VSD_SET_MONITOR )
         {
         MixerControl.ulControl = MIX_MONITOR;

         // LAD--note, this will have to become a define.

         MixerControl.ulSetting = pInstance->lBass * MIXER_MULTIPLIER;
         MixerControl.ulLine = pInstance->ulLine;

         lError = mixSetControl( pInstance->hMix, &MixerControl );
         }

      if ( ulFlags & VSD_SET_PITCH )
         {
         MixerControl.ulControl = MIX_PITCH;

         // LAD--note, this will have to become a define.

         MixerControl.ulSetting =  pInstance->lPitch * MIXER_MULTIPLIER;
         MixerControl.ulLine = pInstance->ulLine;

         lError =mixSetControl( pInstance->hMix, &MixerControl );
         }

     } // if new hardware mixer is active

  else
     {

     memset(&AudioChange, 0, sizeof(MCI_AUDIO_CHANGE)); //@12258

     if ( ulFlags == VSD_SET_MASTER )
        {
        ulFlags = VSD_SET_ALL;
        }

      if ( ulFlags & VSD_SET_VOLUME )
         {
         /*-----------------------------------------------------------------*
         * Place level values in ACPA control structures
         *-----------------------------------------------------------------*/


         ulVolume = pInstance->lLeftVolume;

         /*-----------------------------------------------------------------*
         * If the user has turned off the audio, set the volume to zero
         * so nothing can be heard.
         *-----------------------------------------------------------------*/

         if ( pInstance->fMute )
            {
            ulVolume = 0;
            }

         ulNewBalance = pInstance->lBalance;

         ulVolume *= AUDIOIF_MULTIPLIER;
         ulHoldVolPercentage = ulVolume / 100;

         rTrackInfo.usDitherPct = 0;

         if ( pInstance->ulDeviceID == MACPA )
            {

            // if the line in port is enable, increase recording quality

            if (StatusIOPort( STEREO_LINE_INPUT, pInstance, INPUT_PORT ) == MCI_TRUE )
               {
               // we only want to adjust the volume if we are below 70
               if ( pInstance->ulGainLevel <= 70 )
                  {
                  ulGainPercentage = GainTable[ pInstance->ulGainLevel / 4 ].ulGainValue;
                  ulVolume = ulHoldVolPercentage * ulGainPercentage;
                  rTrackInfo.usMasterVolume = ( USHORT ) (pInstance->ulMasterVolume * AUDIOIF_DIVISOR);
                  }
               else
                  {
                  ulGainPercentage = BoostGainTable[  (pInstance->ulGainLevel - 70) / 5 ].ulGainValue;
                  rTrackInfo.usMasterVolume = ( USHORT ) (ulGainPercentage * AUDIOIF_DIVISOR);
                  } /* else gain > 70 */

               }
            else
               {
               if ( pInstance->ulGainLevel >= 70 )
                  {
                  ulMicType = BOOSTED_MIC_INPUT;
                  }
               else
                  {
                  ulMicType = MIC_INPUT;
                  }
               if ( StatusIOPort( ulMicType, pInstance, INPUT_PORT ) != MCI_TRUE )
                  {
                  lError = AddIOPort( ulMicType,
                                      pInstance,
                                      INPUT_PORT );
                  }

               rTrackInfo.usMasterVolume = ( USHORT ) (pInstance->ulMasterVolume * AUDIOIF_DIVISOR);

               }
            } /* If we are working around acpa gain problems */
         else
            {
            rTrackInfo.usMasterVolume = ( USHORT ) (pInstance->ulMasterVolume * AUDIOIF_DIVISOR);
            AudioChange.lGain = pInstance->ulGainLevel * AUDIOIF_MULTIPLIER;

            } /* The card should be able to handle gain */

         // if the amp stream connector is currently active, then set the volume
         // otherwise ensure that it is 0

         AudioChange.lVolume = (LONG) ulVolume;


         AudioChange.lVolumeDelay = pInstance->lVolumeDelay ;


         } /* if the caller wants to set the volume field */
      else
         {
         AudioChange.lVolume        = AUDIO_IGNORE;
         AudioChange.lVolumeDelay   = AUDIO_IGNORE;

         } /* Caller does not want to set balance */

      if ( ulFlags & VSD_SET_BALANCE )
         {
         // if we exceed the correct amount reduce the quantity
         AudioChange.lBalance = (LONG) ulNewBalance * AUDIOIF_MULTIPLIER;
         rTrackInfo.usMasterBalance = (LONG) ulNewBalance * AUDIOIF_MULTIPLIER;

         if ( !pInstance->lBalanceDelay )
           {
           AudioChange.lBalanceDelay = pInstance->lVolumeDelay;
           }
         else
           {
           AudioChange.lBalanceDelay = pInstance->lBalanceDelay;
           }

         }
      else
         {
         rTrackInfo.usMasterVolume  = AUDIO_IGNORE;
         rTrackInfo.usMasterBalance = AUDIO_IGNORE;
         AudioChange.lBalance       = AUDIO_IGNORE;
         AudioChange.lBalanceDelay  = AUDIO_IGNORE;
         } /* caller did not want to set the treble option */




      if ( ulFlags & VSD_SET_TREBLE )
         {

         AudioChange.lTreble = (LONG) pInstance->lTreble * AUDIOIF_MULTIPLIER;
         }
      else
         {
         AudioChange.lTreble = AUDIO_IGNORE;
         } /* caller did not want to set the treble option */

      if ( ulFlags & VSD_SET_BASS )
         {
         AudioChange.lBass = (LONG) pInstance->lBass * AUDIOIF_MULTIPLIER ;
         }
      else
         {
         AudioChange.lBass = AUDIO_IGNORE;
         }

      if ( ulFlags & VSD_SET_PITCH )
         {

         AudioChange.lPitch = (LONG) pInstance->lPitch * AUDIOIF_MULTIPLIER;
         }
      else
         {
         AudioChange.lPitch = AUDIO_IGNORE;
         } /* Caller wants to set pitch */

      if ( ulFlags & VSD_SET_MONITOR )
         {


         if ( pInstance->lMonitor )
            {
            AudioChange.lMonitor = MONITOR_UNCOMPRESSED;
            }
         else
            {
            AudioChange.lMonitor = MONITOR_OFF;
            }
        } /* caller wants to set monitor flag */
      else
        {
        AudioChange.lMonitor = AUDIO_IGNORE;

        } /* Caller wants to set monitor */

      /*-----------------------------------------------------------------*
      * Convert pointers to 16 bit
      *-----------------------------------------------------------------*/

      prAudioChange32 = &AudioChange;

      /* Check to see if pointer crosses 64k boundary */

      l64kCheck = LOUSHORT(prAudioChange32) ;

      /* If the address crosses a 64k boundary--allocate the memory instead */

      if ( l64kCheck > 65500 )
         {
         rc = DosAllocSharedMem((PVOID)&prAudioChange32,
                                NULL,
                                sizeof( MCI_AUDIO_CHANGE ),
                                PAG_COMMIT | OBJ_GIVEABLE |
                                PAG_READ | PAG_WRITE);

         lError = MCI_Error(rc);
         if ( lError )
            {
            return ( lError );
            }


         fAllocatedChangeMemory = TRUE;
         /* Copy the data into the allocated buffer */

         memmove(prAudioChange32, &AudioChange, sizeof( MCI_AUDIO_CHANGE ) );

         prAudioChange16 = ( PVOID ) MAKEULONG( LOUSHORT(prAudioChange32),
                                                (HIUSHORT(prAudioChange32) << 3) + 7);

         }

      else
         {
         prAudioChange16 = ( PVOID ) MAKEULONG( LOUSHORT(prAudioChange32),
                                                (HIUSHORT(prAudioChange32) << 3) + 7);
         }

      AudioControl.pbRequestInfo = prAudioChange16;

      prTrackInfo32 = &rTrackInfo;

      /* Check to see if pointer crosses 64k boundary */

      l64kCheck = LOUSHORT(prTrackInfo32) ;

      /* If the address crosses a 64k boundary--allocate the memory instead */

      if ( l64kCheck > 65500 )
         {
         rc = DosAllocSharedMem((PVOID)&prTrackInfo32,
                                NULL,
                                sizeof( MCI_TRACK_INFO ),
                                PAG_COMMIT | OBJ_GIVEABLE |
                                PAG_READ | PAG_WRITE);

         lError = MCI_Error(rc);
         if ( lError )
            {
            return ( lError );
            }


         fAllocatedTrackMemory = TRUE;
         /* Copy the data into the allocated buffer */

         memmove(prTrackInfo32, &rTrackInfo, sizeof( MCI_TRACK_INFO ) );

         prTrackInfo16 = ( PVOID ) MAKEULONG ( LOUSHORT(prTrackInfo32),
                                               (HIUSHORT(prTrackInfo32) << 3) + 7);

         }

      else
         {
         prTrackInfo16 = ( PVOID ) MAKEULONG ( LOUSHORT(prTrackInfo32),
                                               (HIUSHORT(prTrackInfo32) << 3) + 7);
         } /* stack variable is safe to use */


      AudioChange.pvDevInfo = prTrackInfo16;

      /*-----------------------------------------------------------------*
      * Informat the AUDIODD which connectors (i.e. speakers, line etc.)
      * will be enabled.
      *-----------------------------------------------------------------*/


      Set_IO_Ports( &AudioChange, pInstance, ulFlags );

      /*-----------------------------------------------------------------*
      * Make the IOCtl call
      *-----------------------------------------------------------------*/

      AudioControl.usIOCtlRequest = AUDIO_CHANGE;
      AudioControl.ulPosition = 0L;
      AudioControl.sReturnCode = 0;

      ulDataLen = sizeof(MCI_AUDIO_CONTROL);

      AudioChange.pvModeInfo = 0;

      rc = DosDevIOCtl(pInstance->hFile,
                       AUDIO_IOCTL_CAT,
                       AUDIO_CONTROL,
                       0,
                       0L,
                       &ulParmLengthInOut,
                       &AudioControl,
                       (ULONG) sizeof(MCI_AUDIO_CONTROL),
                       &ulDataLen);

      lError = MCI_Error(rc);

      pInstance->lBalanceDelay = pInstance->lVolumeDelay = 0;

      } /* we are using the older/less functional AUDIODD interface */


   if ( fAllocatedTrackMemory )
      {
      DosFreeMem(prTrackInfo32);

      }

   if ( fAllocatedChangeMemory )
      {
      DosFreeMem(prAudioChange32);

      }

   return(lError);


} /* ModifyAudioAttributes  */



