/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/************************* START OF SPECIinCATIONS ***
*
* SOURCE FILE NAME:  VSDINI.C
*
* DESCRIPTIVE NAME: Ini file and resource file manipulator
*
*
* STATUS:  OS/2 Release 2.x
*
* FUNCTION: This source module contains routines to retrieve information
*           from the VSD resource file and MMPM2.INI file.
*
*           ISV can manipulate their own ini files here.
*
*   Notes: the following concepts are illustrated.
*
*      1. How to read in a resource dll and retrieve VSD specific info.
*          (SetupResourceTable)
*      2. How to determine the amount of resource a particular device
*           consumes (GetClassInformation)
*
*
*/

#define INCL_NOPMAPI
#define INCL_DOS
#define INCL_ERRORS
#define INCL_AUDIO_VSD
#include <os2.h>
#include <os2me.h>
#include <mcd.h>
#include <audio.h>
#include <stdio.h>
#include <string.h>
#include <hhpheap.h>
#include <stdlib.h>

#include <vsdcmds.h>
#include <vsdaud.h>
#include <mmdsload.h>

RIFFTABLE RiffTable[NUMMODES] =
        {
        {DATATYPE_WAVEFORM,    PCM},
        {DATATYPE_ADPCM_AVC,   ADPCM},
        {DATATYPE_MULAW,       MU_LAW},
        {DATATYPE_ALAW,        A_LAW},
        {DATATYPE_MIDI,        MIDI},
        {DATATYPE_SPV2,        SPV2},
        {DATATYPE_CDXA_AUDIO,  ADPCMXA},
        {DATATYPE_CT_ADPCM,    CT_ADPCM}
        };



/************************** START OF SPECIFICATIONS ************************
*                                                                          *
* SUBROUTINE NAME: SetupResourceTable                                      *
*                                                                          *
* DESCRIPTIVE NAME: Retrives VSD Resource information obtained from ini    *
*                                                                          *
* FUNCTION: This function loads a resource dll (specified in the device    *
*           specific parameters of the amp-mix device).  It then parses    *
*           the resource dll and sets up appropriate instance values.      *
*                                                                          *
*           If you are writing your own VSD, you DO NOT have to utilize    *
*           these ini functions, however, you are welcome to utilize the   *
*           same functionality.                                            *
* INPUT:                                                                   *
*                                                                          *
* OUTPUT:   VSDERR_SUCESS if there was a match                             *
*           An MCI error otherwise                                         *
*                                                                          *
* SIDE EFFECTS:                                                            *
*                                                                          *
*************************** END OF SPECIFICATIONS **************************/



ULONG SetupResourceTable( PVSD_INSTANCE    pInstance )

{
ULONG      rc;
PULONG     pulResourceWalker;     // used to walk through the resource file


HMODULE    hModule;               // module handle of dll where the resource
                                  // stored.

//ULONG      ulOffset =1;

  if (!strcmp(pInstance->pResourceDLL, "AUDIOIF") ||           //@LP01
     !pInstance->pResourceDLL[0])
     rc = MM_DosLoadModule( NULL, 0, "MMPM", &hModule );      //@LP01
  else if (!pInstance->pResourceDLL[0])
     rc = MM_DosLoadModule( NULL, 0, "AUDIOIF", &hModule );    //@LP01
  else
     rc = 1;

  if ( rc )                                                    //@LP01
     rc = MM_DosLoadModule( NULL, 0, pInstance->pResourceDLL, &hModule );

//  rc = DosLoadModule( NULL, 0, pInstance->pResourceDLL, &hModule );

  if ( rc )
     return ( rc );

  /*-----------------------------------
  * It the device specific parameters of
  * the amp, the user specified which
  * card to use by the resource id.  The
  * amp placed this info in the instance.
  --------------------------------------*/

  rc = DosGetResource( hModule,
                       RT_RCDATA,
                       pInstance->ulResourceID,
                       &pInstance->pAudioModeData );

  if ( rc )
     return ( rc );

  /*******************************************************
  * Set up product information pointer--this must be the
  * first item in the resource table.
  ********************************************************/

  pInstance->pProdInfo = (PSZ ) pInstance->pAudioModeData;

  pulResourceWalker = (PULONG ) (pInstance->pProdInfo +
                                  strlen(pInstance->pProdInfo) + 1);

  /*******************************************************
  * Set up device idnformation pointer--this follows right
  * after the product information string.
  ********************************************************/


  pInstance->ulDeviceID = *(pulResourceWalker)++;


  /*******************************************************
  * Determine if the manufacturer has any RIFF->AUDIODD maps
  * in the resource table (this mapping is necessary since
  * RIFF format tags do not match with AUDIODD modes.
  ********************************************************/

  pInstance->ulNumMappingRows = *(pulResourceWalker)++;

  if ( pInstance->ulNumMappingRows )
     {
     pInstance->pMapTable = ( RIFFTABLE * ) pulResourceWalker;
     pulResourceWalker += ( 2 * pInstance->ulNumMappingRows) ;
     }

  /*******************************************************
  * Access the data type table.  This contains a description
  * of the data types this device supports.
  ********************************************************/


  pInstance->ulNumDataTypes = *( pulResourceWalker)++;
  pInstance->pDataTypeTable = ( PVOID ) pulResourceWalker;

  return ( VSDERR_SUCCESS );

} /* SetUpResourceTable */








/************************** START OF SPECIFICATIONS ************************
*                                                                          *
* SUBROUTINE NAME: GetClassInformation                                     *
*                                                                          *
* DESCRIPTIVE NAME: Retrives VSD Resource information obtained from ini    *
*                                                                          *
* FUNCTION: This function examines an instance to see if a match can be    *
*           found between what the device currently supports and what      *
*           the user has requested.  If a match cannot be found, then the  *
*           best guess (best fit) will be returned.                        *
*                                                                          *
*           If you are writing your own VSD, you DO NOT have to utilize    *
*           these ini functions, however, you are welcome to utilize the   *
*           same functionality.                                            *
*                                                                          *
* INPUT:    ulPort                                                         *
*                                                                          *
* OUTPUT:   VSDERR_SUCESS if there was a match                             *
*           An MCI error otherwise                                         *
*                                                                          *
* SIDE EFFECTS:                                                            *
*                                                                          *
*************************** END OF SPECIFICATIONS **************************/




ULONG GetClassInformation( PVSD_INSTANCE    pInstance )

{
ULONG      ulLoop;

//ULONG      rc = VSDERR_SUCCESS;

SHORT      sModeBackup = pInstance->sMode;
BOOL       fContinousRate = FALSE;

// these variables will be used to return an intelligent error

BOOL       fFoundBits = FALSE;      // found bitspersample
//BOOL       fFoundOperation = FALSE; // found operation (either play or record)
BOOL       fFoundSamples = FALSE;   // Found sampling rate
BOOL       fFoundChannels = FALSE;  // Found channels
BOOL       fFoundTag = FALSE;       // Found  format tag

VSD_BEST_FIT vsdBestFit;            /* VSD best fit structure */

CLASS_DATA *pDataTypeTable = ( CLASS_DATA * ) pInstance->pDataTypeTable;



  if ( pInstance->fDriverCaps  )
     {
     return ( CapIOCTL( pInstance) );
     }
  else
     {

     /*---------------------------------------------------------
     * Work around problems created by Microsoft (boy isn't this
     * a common theme).  They redefined mulaw and alaw.  Map
     * these to the defines we know about already.
     *----------------------------------------------------------*/
     if ( pInstance->sMode == DATATYPE_RIFF_MULAW )
        {
        pInstance->sMode = DATATYPE_MULAW;
        }

     if ( pInstance->sMode == DATATYPE_RIFF_ALAW )
        {
        pInstance->sMode = DATATYPE_ALAW;
        }

     memset( &vsdBestFit, '\0', sizeof( VSD_BEST_FIT ) );

     pInstance->ulMatch = FALSE;

     /************************************************************
     * The caller has requested a specific data type, sampling rate,
     * bps etc. We will search through the data type table until we
     * either find a match or run out of table elements to compare.
     **************************************************************/

     for ( ulLoop = 0; ulLoop < pInstance->ulNumDataTypes; ulLoop++ )
       {
       /*---------------------------------------------------------
       * Prepare for best fit just in case a match is not found
       * This routine will give us the closest supported rate
       * that the device supports given the sampling rate,
       * bps and channels.
       *---------------------------------------------------------*/

       BestGuess( pInstance->lBitsPerSRate,
                  ( LONG ) pDataTypeTable->sBitsPerSRate,
                  ( LONG ) pInstance->sChannels,
                  ( LONG ) pDataTypeTable->usChannels,
                  pInstance->lSRate,
                  ( LONG ) pDataTypeTable->sSRate,
                  ( LONG ) pDataTypeTable->ulSamplingDescription,
                  &vsdBestFit );

       if ( pInstance->sMode == pDataTypeTable->ulDataType )
          {
          fFoundTag = TRUE;

          if ( pInstance->lBitsPerSRate == (LONG ) pDataTypeTable->sBitsPerSRate )
             {
             fFoundBits = TRUE;
             if ( pInstance->sChannels == ( SHORT ) pDataTypeTable->usChannels )
                {
                fFoundChannels = TRUE;
                if ( pInstance->lSRate == pDataTypeTable->sSRate ||
                     ( fContinousRate &&
                       pInstance->lSRate < pDataTypeTable->sSRate  ) )
                   {

                   fFoundSamples = TRUE;

                   pInstance->ulDataType = pDataTypeTable->ulDataType;
                   pInstance->ulSubType  = pDataTypeTable->ulSubType;

                   /* Translate RIFF mode to AUDIODD mode */

                   MapModes ( pInstance );

                   /***************************************************
                   * The resource file contains resource management
                   * information which is used by MDM.  The playback
                   * mode may contain different resource management
                   * data than record mode so retrieve from the
                   * right spot.
                   ***************************************************/

                   if ( pInstance->ulOperation == OPERATION_RECORD )
                      {
                      if ( pDataTypeTable->usRecordClass == 0 )
                         {
                         return ( VSDERR_UNSUPP_FORMAT_MODE );
                         }

                      pInstance->ulClass = pDataTypeTable->usRecordClass;
                      pInstance->ulResourcesUsed = pDataTypeTable->usRecordUnits;
                      } /* Get playback resource information */
                   else
                      {

                      if ( pDataTypeTable->usPlayClass == 0 )
                         {
                         pInstance->ulClass = pDataTypeTable->usRecordClass;
                         pInstance->ulResourcesUsed = pDataTypeTable->usRecordUnits;
                         pInstance->ulOperation = OPERATION_RECORD;

                         }
                      else
                         {
                         pInstance->ulClass = pDataTypeTable->usPlayClass;
                         pInstance->ulResourcesUsed = pDataTypeTable->usPlayUnits;
                         }



                      } /* Get record resource information */

                   pInstance->ulCanRecord = ( pDataTypeTable->usRecordClass > 0 ) ? 1 : 0;
                   /* Set flag indicating that this mode is valid */

                   pInstance->ulMatch = TRUE;

                   /*-----------------------------------------------
                   * In our attempt to work around a Microsoft
                   * kludge, we may have overwritten the callers
                   * datatype.  Restore before leaving.
                   *----------------------------------------------*/

                   pInstance->sMode = sModeBackup;


                   return ( VSDERR_SUCCESS );

                   } /* Sampling rate is the same */
                else
                   {
                   if ( pInstance->lSRate > pDataTypeTable->sSRate  &&
                        pDataTypeTable->ulSamplingDescription == BEGIN_CONTINUOUS )
                      {
                      fContinousRate = TRUE;
                      }
                   else
                      {
                      fContinousRate = FALSE;
                      }
                   } /* Sampling rate differs */

                } /* if the channels are the same */
             } /* if the bits/sample are the same */

          } /* if the format tags are the same */

          pDataTypeTable++;
       } /* loop through each data type till we find a match */


       /*-----------------------------------------------
       * In our attempt to work around a Microsoft
       * kludge, we may have overwritten the callers
       * datatype.  Restore before leaving.
       *----------------------------------------------*/

       pInstance->sMode = sModeBackup;

     /* Setup best-fit info to the caller */

     pInstance->sBestFitMode  = DATATYPE_WAVEFORM;
     pInstance->ulBestFitBPS  = vsdBestFit.lBPSGuess;
     pInstance->ulBestFitChan = vsdBestFit.lChanGuess;
     pInstance->ulBestFitRate = vsdBestFit.lSampGuess;


     /* Return non-zero value to indicate failure */


     if ( !fFoundTag )
        {
        return ( VSDERR_UNSUPP_FORMAT_TAG );
        }
     else if ( !fFoundBits )
        {
        return ( VSDERR_UNSUPP_BITSPERSAMPLE );
        }
     else if ( !fFoundChannels )
        {
        return ( VSDERR_UNSUPP_CHANNELS );
        }
     else if ( !fFoundSamples )
        {
        return ( VSDERR_UNSUPP_SAMPLESPERSEC );
        }

      else
        {
        return ( VSDERR_UNSUPP_FORMAT_MODE );
        }
     } /* Device driver does not support capabilities IOCTL */

} /* GetClassInformation */


/************************** START OF SPECIFICATIONS ************************
*                                                                          *
* SUBROUTINE NAME: BestGuess                                               *
*                                                                          *
* DESCRIPTIVE NAME: Retrives VSD Resource information obtained from ini    *
*                                                                          *
* FUNCTION: This function tries to determine the closest mode a device     *
*           supports compared to what the user requested.                  *
*                                                                          *
* INPUT:    pInstance                                                      *
*                                                                          *
* OUTPUT:   None.                                                          *
*                                                                          *
* SIDE EFFECTS:                                                            *
*                                                                          *
*************************** END OF SPECIFICATIONS **************************/

void BestGuess ( LONG                 lBPSRequest,
                 LONG                 lBPSValue,
                 LONG                 lChanRequest,
                 LONG                 lChanValue,
                 LONG                 lSampRequest,
                 LONG                 lSampValue,
                 LONG                 lContinuous,
                 VSD_BEST_FIT         *pvsdBestFit )

{
LONG     lDifference;
LONG     lNewDifference;
BOOL     fContinuous = FALSE;



   /*------------------------------------------------------
   * If we have not guessed at the nearest supported
   * bps value already, then the guess and the max and
   * min supported rates must be the currently supported
   * rate.
   *-----------------------------------------------------*/

   if ( !pvsdBestFit->lBPSGuess )
      {
      pvsdBestFit->lBPSGuess = lBPSValue;
      }
   else
      {

      /* Determine which is closer, our guess or the new # */

      lDifference = abs( lBPSRequest - pvsdBestFit->lBPSGuess );
      lNewDifference = abs( lBPSRequest - lBPSValue );

      /* If our new number is closer, then use that as the guess */

      if ( lNewDifference < lDifference )
         {
         pvsdBestFit->lBPSGuess = lBPSValue;
         } /* New number is closer */

      /* 4-bit actually means some type of ADPCM */

      if ( lBPSRequest == 4 && lBPSValue == 16 )
         {
         pvsdBestFit->lBPSGuess = lBPSValue;
         }
      } /* else there was a best fit guess */

   /*------------------------------------------------------
   * If we have not guessed at the nearest supported
   * bps value already, then the guess and the max and
   * min supported rates must be the currently supported
   * rate.
   *-----------------------------------------------------*/

   if ( !pvsdBestFit->lChanGuess )
      {
      pvsdBestFit->lChanGuess = lChanValue;
      }
   else
      {
      /* Determine which is closer, our guess or the new # */

      lDifference = abs( lChanRequest - pvsdBestFit->lChanGuess );
      lNewDifference = abs( lChanRequest - lChanValue );

      /* If our new number is closer, then use that as the guess */

      if ( lNewDifference < lDifference )
         {
         pvsdBestFit->lChanGuess = lChanValue;
         } /* New number is closer */

      } /* else there was a best fit guess */


   if ( lContinuous == BEGIN_CONTINUOUS ||
        lContinuous == END_CONTINUOUS )
      {
      fContinuous = TRUE;
      }
   /*------------------------------------------------------
   * If we have not guessed at the nearest supported
   * samp/rate value already, then the guess and the max and
   * min supported rates must be the currently supported
   * rate.
   *-----------------------------------------------------*/

   if ( fContinuous )
      {
      pvsdBestFit->lSampGuess = lSampRequest;
      }

   else
      {
      if ( !pvsdBestFit->lSampGuess )
         {
         pvsdBestFit->lSampGuess = lSampValue;
         }
      else
         {
         /* Determine which is closer, our guess or the new # */

         lDifference = abs( lSampRequest - pvsdBestFit->lSampGuess );
         lNewDifference = abs( lSampRequest - lSampValue );

         /* If our new number is closer, then use that as the guess */

         if ( lNewDifference < lDifference )
            {
            pvsdBestFit->lSampGuess = lSampValue;
            } /* New number is closer */

         } /* else there was a best fit guess */
      } /* Not a device with integral sampling rates */

} /* BestGuess */



/************************** START OF SPECIFICATIONS ************************
*                                                                          *
* SUBROUTINE NAME: MapModes                                                *
*                                                                          *
* DESCRIPTIVE NAME: Converts a Riff Mode to an AUDIODD mode.               *
*                                                                          *
* FUNCTION: Due to a very unfortunate choice (i.e. one out of our control) *
*           the AUDIODD architecture defines conflict with RIFF modes.     *
*           This function maps from RIFF->AUDIODD mode                     *
*           Newer drivers can indicate that they support RIFF datatypes    *
*           via the AUDIO capability ioctl.                                *
*                                                                          *
*                                                                          *
* INPUT:    pInstance                                                      *
*                                                                          *
* OUTPUT:   None.                                                          *
*                                                                          *
* SIDE EFFECTS:                                                            *
*                                                                          *
*************************** END OF SPECIFICATIONS **************************/



void MapModes( PVSD_INSTANCE    pInstance )

{
ULONG       ulLoop = 0;

RIFFTABLE   *pMapTable = pInstance->pMapTable;




   if ( !(pInstance->fDriverCaps & SUPPORT_RIFF_MODES) )
      {
      while ( ulLoop < KNOWN_AUDIODD_MAPS )
         {

         if  ( pInstance->sMode == RiffTable[ ulLoop ].ulDataType )
            {
            pInstance->ulHardwareMode = RiffTable[ ulLoop ].ulAUDIODDMode;
            return;
            }
         ulLoop++;
         }

      ulLoop = 0;

      while ( ulLoop < pInstance->ulNumMappingRows )
         {
         if  ( pInstance->sMode == (pMapTable + ulLoop)->ulDataType )
            {
            pInstance->ulHardwareMode = (pMapTable + ulLoop)->ulAUDIODDMode;
            // ensure that we can stream this data.
            pInstance->ulDataType = DATATYPE_WAVEFORM;

            return;
            }
         ulLoop++;
         }
      } /* If the driver does not support RIFF datatypes */

  /* Worst case, assume that they are the same */

  pInstance->ulHardwareMode = pInstance->ulDataType;
  return;

} /* MapModes */


/************************ START OF SPECIFICATIONS **************************
*
* SUBROUTINE NAME: ACPABestFit()
*
* FUNCTION: Since ACPA 1.02 has no best fit, best fit for it
*
* INPUT:  pInstance - pointer to AMP/Mixer instance structure
*
* OUTPUT: returns VSDERR_SUCCESS if successful, otherwise it returns an
*         MCI error code.
*
*                                                                                                                                                  *
*************************** END OF SPECIFICATIONS *************************/
void  ACPABestFit( PVSD_INSTANCE   pInstance )

{
    if ( ( !pInstance->ulMatch                   &&
           pInstance->sMode == DATATYPE_WAVEFORM ) )

       {
       if ( pInstance->lSRate < 9000 )
          {
          pInstance->lSRate = 8000;
          }
       else if ( pInstance->lSRate < 15000 )
          {
          pInstance->lSRate = 11025;
          }
       else if ( pInstance->lSRate < 33400 )
          {
          pInstance->lSRate = 22050;
          }
       else
          {
          pInstance->lSRate = 44100;
          }

       pInstance->ulHardwareMode = PCM;

       GetClassInformation( pInstance );

       }  /* If sampling rate is not directly supported */


} /* ACPABestFit */


