/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SOURCE FILE NAME:  AUDIODD.C         (TEMPLATE SAMPLE)                   */
/*                                                                          */
/* DISCRIPTIVE NAME: Audio device driver strategy and IOCTL routines        */
/*                                                                          */
/* LINKAGE: Called from Startup.asm                                         */
/*                                                                          */
/* DESCRIPTION:                                                             */
/*    This audio device driver is an OS/2 16-bit Physical Device driver     */
/*    designed to demonstrate audio device communication with the           */
/*    MMPM/2 stream handler.                                                */
/*    Information on OS/2 Physical Device Drivers architecture and          */
/*    programming is in the "OS/2 2.0 Physical Device Driver reference"     */
/*                                                                          */
/*    MMPM/2 streaming architecture is outlined in                          */
/*    "MMPM/2 Programmers Reference"                                        */
/*       - Data Streaming & synchronization chapter                         */
/*    Specifix information on interdevice communication is in the           */
/*    same document in the "Stream Handler" Chapter in the section on       */
/*    "Interdevice-driver-communication                                     */
/************************** END OF SPECIFICATIONS ***************************/


#include <string.h>

#define  INCL_DOSINFOSEG
#include <os2.h>

#include <os2medef.h>
#include <ssm.h>
#include <audio.h>
#include "audiodd.h"
#include "audsubs.h"
#include "cdevhlp.h"

#define DMABUFFER       32*1024

extern  PDDEntryPoint();                // located in startup.asm

extern  ULONG   (*IOCTLFuncs[])(PREQPACKET rp);
extern  MaxIOCTLFuncs;
extern  ULONG   (*AudioIOCTLFuncs[])(PVOID pParm);
extern  MaxAudioIOCTLFuncs;
extern  FPVOID  DevHlp;
extern  ADAPTERDATA AdapterInfo;
extern  GLOBAL GlobalTable;
extern  DEVHDR DevHdr[];
extern  usGDT[];
extern  ULONG ulActiveSysFileNum;
extern  USHORT usInUse;
extern  EndOfData;
extern  USHORT usNumHeaders;
extern  ATTACHAREA      AttachArea;
extern  SZ      szDevName[9];
extern  POSLENRB POSLenRB;
extern  POSRB   ABIOSrb;
extern  POSCardID[];

int   mode = MIDI;
unsigned long flags = 0;
long  srate;
long  bits_per_sample;
long  bsize;
int   channels;
unsigned long  operation;
int   position_type;
USHORT   usNumOS2Opens=0;


/*********************************************************************/
/* STRATEGY_C                                                        */
/* DD strategy control after entry from assembler Strategy routine.  */
/*********************************************************************/

ULONG   Strategy_c(PREQPACKET rp)
{
        if (rp->RPcommand > (UCHAR)MaxIOCTLFuncs) // check for valid function
                return(RPDONE | RPERR | RPBADCMD);

        return(IOCTLFuncs[rp->RPcommand](rp));  // call request function
                                                // then return its rc
}


/*******************************************************************************/
/*                         INVALID REQUEST                                     */
/*******************************************************************************/
ULONG   IOCTL_Invalid(PREQPACKET rp)
{
        return(RPDONE | RPERR | RPBADCMD);
}

/*****************************************************************************/
/*                      Hardware communication routines                      */
/* Implementation of these functions is dependant on interfaces and physical */
/* characteristics of the card.                                              */
/* IOCTLs listed below are OS/2 defined IOCTLs.  Additional IOCTLs arrive    */
/* from applications and MMPM/2 through the OS/2 generic IOCTL.              */
/*****************************************************************************/

/*****************************************************************************/
/*                              READ                                         */
/* Stub, read from device.  Implemention dependant on hardware architecture. */
/*****************************************************************************/
ULONG   IOCTL_Read(PREQPACKET rp)
{
        ReadDataFromCard();     // Stub routine
        return(RPDONE);
}

/*****************************************************************************/
/*                        NONDESTRUCTIVE READ                                */
/* Stub, routine performing nondestructive read operation.                   */
/* This read operation reads data from a buffer without removing that data.  */
/*****************************************************************************/
ULONG   IOCTL_NondestructiveRead(PREQPACKET rp)
{
        return(RPDONE);
}

/*****************************************************************************/
/*                           READ STATUS                                     */
/* Stub routine                                                              */
/*****************************************************************************/
ULONG   IOCTL_ReadStatus(PREQPACKET rp)
{
        ReadStatus();
        return(RPDONE);
}

/*****************************************************************************/
/*                            FLUSH INPUT                                    */
/*****************************************************************************/
ULONG   IOCTL_FlushInput(PREQPACKET rp)
{
        FlushInputBuffers();
        return(RPDONE);
}

/*****************************************************************************/
/*                          WRITE                                            */
/* Stub, write to device.                                                    */
/*****************************************************************************/
ULONG   IOCTL_Write(PREQPACKET rp)
{
        WriteDataToCard();
        return(RPDONE);
}

/*****************************************************************************/
/*                         WRITE  STATUS                                     */
/* Stub routine                                                              */
/*****************************************************************************/
ULONG   IOCTL_WriteStatus(PREQPACKET rp)
{
        WriteStatus();
        return (RPDONE);
}

/*****************************************************************************/
/*                        FLUSH OUTPUT                                       */
/*****************************************************************************/
ULONG   IOCTL_FlushOutput(PREQPACKET rp)
{
        FlushOutputBuffers();
        return (RPDONE);
}

/*****************************************************************************/
/*                        OPEN                                               */
/* Prepare device for operations.                                            */
/*****************************************************************************/
ULONG   IOCTL_Open(PREQPACKET rp)
{
        /*****************************
        ** Open device, hook interrupts
        **
        ** For this sample, no interrupts will be
        ** hooked, generated or received as there
        ** is no hardare.
        ** Still, AUDINTR.C contains a sample
        ** interrupt handler demonstrating
        ** communication with the MMPM/2 stream handler.
        ******************************/
        if (usInUse & VDM)                      // Is device being used by a VDM?
                return (RPDONE|RPERR);          // yes, return error   CAD 1-26-93

        if (DevOpen())  {                       // set inuse flag so that a VDM can't use us
                usNumOS2Opens++;
                return (RPDONE);
        } else
                return (RPDONE|RPERR);
}

/*****************************************************************************/
/*                          CLOSE                                            */
/* Opposite of open.  Clean up system resources.  If any activity is in      */
/* process, it needs to be ended.                                            */
/*****************************************************************************/
ULONG   IOCTL_Close(PREQPACKET rp)
{
        //*****************************
        // Destroy streams, turn off
        // any hung notes, close device
        //*****************************
        DestroyStreams();
        usNumOS2Opens--;
        DevClose();
        if (usNumOS2Opens == 0)                 // only call VDD when all OS2 processes are closed
                VDMClose();
        return(RPDONE);
}


/*****************************************************************************/
/*                      IOCTL INPUT/OUTPUT                                   */
/*****************************************************************************/
ULONG   IOCTL_Input(PREQPACKET rp)
{
        return(RPDONE);
}

ULONG   IOCTL_Output(PREQPACKET rp)
{
        return(RPDONE);
}



/*****************************************************************************/
/*                       GENERIC IOCTL                                       */
/*****************************************************************************/
ULONG   IOCTL_GenIOCTL(PREQPACKET rp)                   // GENERAL IOCTL (OS/2)
{
        //*****************************
        // Valid category : 0x80
        // Valid functions: 0x40 - 0x5f
        //*****************************
        if (rp->s.IOCtl.category != 0x80)       // must be proper category
                        return(RPDONE);

        if ((rp->s.IOCtl.function < (UCHAR)0x40) ||
            (rp->s.IOCtl.function > ((UCHAR)0x40+(UCHAR)MaxAudioIOCTLFuncs)))  // Is function invalid?
                return(RPDONE | RPERR | RPBADCMD);


        AudioIOCTLFuncs[rp->s.IOCtl.function-0x40](rp); // call request function
                                                       // using table of funcs
                                                       // set up at compile
                                                       // time.
        return (RPDONE);
}




/*****************************************************************************/
/*                     AUDIO_INIT IOCTL                                      */
/*****************************************************************************/
ULONG   Audio_IOCTL_Init(PREQPACKET pParm)
{
        MCI_AUDIO_INIT FAR *pInit;
        ULONG   ulFlags=0;

        pInit = (MCI_AUDIO_INIT FAR *)pParm->s.IOCtl.buffer;

        //****************************************
        // Copy parameters to our global variables
        // in case any of them have changed.
        //****************************************
        operation = pInit->ulOperation;
        flags = pInit->ulFlags;
        mode = pInit->sMode;
        srate = pInit->lSRate;
        bits_per_sample = pInit->lBitsPerSRate;
        bsize = pInit->lBsize;
        channels = pInit->sChannels;
        //*****************************************************
        // retrieve system file number from request
        // packet and stuff it into audio_init struct.
        // Also, store it so PDD knows who is the active stream.
        //******************************************************
        pInit->pvReserved=(VOID FAR *)((PREQPACKET) pParm)->s.IOCtl.sysfilenum;
        ulActiveSysFileNum=(ULONG)pInit->pvReserved;            // this is now the active instance
        switch (pInit->sMode)
                {
                case PCM:                       /* Pulse Coded Modulation */
                        break;

                case MIDI:                      /* MIDI data      */
                        break;

                case IDLE:
                        // set stream to be stopped   call DevPause()
                        // De-init this stream 
                        break;

                default:
                        pInit->sReturnCode= NO_RECORD_AND_PLAY;
                        return (RPDONE|RPERR);
                }

        if (pInit->ulOperation == PLAY_AND_RECORD)
                return(RPDONE|RPERR|NO_RECORD_AND_PLAY);

        if ((pInit->ulOperation != OPERATION_PLAY ) &&
            (pInit->ulOperation != OPERATION_RECORD ))
                return(RPDONE|RPERR|INVALID_REQUEST );

        /***************************************************/
        /* Check for best fit on channels                  */
        /***************************************************/
        if ( pInit->sChannels < 1 )
                {
                pInit->sChannels = 1;
                ulFlags |= BESTFIT_PROVIDED;
                }
        else
                if (pInit->sChannels >2)
                        {
                        pInit->sChannels = 2;
                        ulFlags |= BESTFIT_PROVIDED;
                        }

        /***************************************************/
        /* Check for best fit on sampling rate             */
        /***************************************************/
        if (pInit->lSRate < 2000)
                {
                pInit->lSRate=2000;
                ulFlags |= BESTFIT_PROVIDED;
                }
        else if (pInit->lSRate >48000)
                {
                pInit->lSRate=48000;
                ulFlags |= BESTFIT_PROVIDED;
                }

        ulFlags |= FIXED;                       /* Fixed length data            */
        ulFlags |= LEFT_ALIGNED;                /* Left align bits on byte bndry */
        if (pInit->lBitsPerSRate==8)
                ulFlags|= TWOS_COMPLEMENT;      /* 2's complement data          */

        ulFlags|= INPUT;                        /* Input select is supported    */
        ulFlags|= OUTPUT;                       /* Output select is supported   */
        ulFlags|= MONITOR;                      /* Monitor is supported         */
        ulFlags|= VOLUME;                       /* Volume control is supported  */
        ulFlags|= VOLUME_DELAY;                 /* Volume delay is supported    */
        ulFlags|= BALANCE;                      /* Balance control is supported */
        ulFlags|= BALANCE_DELAY;                /* Balance delay is supported   */
        ulFlags|= TREBLE;                       /* Treble control is supported  */
        ulFlags|= BASS;                         /* Bass control supported       */
        pInit->ulFlags=ulFlags;
        pInit->sReturnCode= 0;
        return(RPDONE);
}


/*****************************************************************************/
/*                    AUDIO_STATUS IOCTL                                     */
/* Query status of audio device in accordance with stream handler spec.      */
/*****************************************************************************/
ULONG   Audio_IOCTL_Status(PREQPACKET rp)
{
        DevIOCTLstatus();
        return(RPDONE);
}


/*****************************************************************************/
/*                    AUDIO_CONTROL IOCTL                                    */
/*****************************************************************************/
ULONG   Audio_IOCTL_Control(PREQPACKET rp)
{
        switch(rp->RPcommand)
        {
           /****************/
           /* AUDIO_CHANGE */
           /****************/
           case AUDIO_CHANGE:                  /* Change adapter  */
                   DevChange ();               /* characteristics */
                   break;

           /***************/
           /* AUDIO_START */
           /***************/
           case AUDIO_START:                   /* Start new operation        */
                   DevStart();
                   break;

           /**************/
           /* AUDIO_STOP */
           /**************/
           case AUDIO_STOP:                 /* Stop current operation        */
                   DevStop();
                   break;

           /***************/
           /* AUDIO_PAUSE */
           /***************/
           case AUDIO_PAUSE:                /* suspend current operation     */
                   DevPause();
                   break;

           /****************/
           /* AUDIO_RESUME */
           /****************/
           case AUDIO_RESUME:               /* resume a suspended operation  */
                   DevResume();
                   break;

           default:                         /* Unknown control               */
                   return (-1);             /* return an error */
                   break;
        }
        return(RPDONE);
}

/*****************************************************************************/
/*                     AUDIO_BUFFER IOCTL                                    */
/*****************************************************************************/
ULONG   Audio_IOCTL_Buffer(PREQPACKET rp)   /* AUDIO_BUFFER IOCTL */
{
        PVOID   pParm;
        pParm = rp->s.IOCtl.parameters;
        DevAudioBuffer();
        return(RPDONE);
}


/*****************************************************************************/
/*                      AUDIO_LOAD IOCTL                                     */
/*****************************************************************************/
ULONG   Audio_IOCTL_Load(PREQPACKET rp)
{
        PVOID   pParm;
        pParm = rp->s.IOCtl.parameters;
        DevIOCTLload();
        return(RPDONE);
}


/*****************************************************************************/
/*                           AUDIO_WAIT IOCTL                                */
/*****************************************************************************/
ULONG   Audio_IOCTL_Wait(PREQPACKET rp)   /* AUDIO_WAIT */
{
        PVOID   pParm;
        pParm = rp->s.IOCtl.parameters;
        DevIOCTLwait();
        return(RPDONE);
}



/*****************************************************************************/
/*                     HIGH PERFORMANCE INTERFACE IOCTL                      */
/*****************************************************************************/
ULONG   Audio_IOCTL_Hpi(PREQPACKET rp)
{
        PVOID   pParm;
        pParm = rp->s.IOCtl.parameters;

        DevIOCTLhpi ();
        return(RPDONE);
}


/*****************************************************************************/
/*                      PDD INITIALIZATION CODE                              */
/*****************************************************************************/

/*****************************************************************************/
/*                                INIT                                       */
/*****************************************************************************/
ULONG   IOCTL_Init(PREQPACKET rp)
{
        USHORT  i;
        BOOL    bFound;
        PSZ     szNum;

        if (!GlobalTable.fInited)            // don't init both headers
             GlobalTable.fInited = TRUE;
        else {
             rp->s.InitExit.finalCS = ((OFFSET)&IOCTL_Init);
             rp->s.InitExit.finalDS = (OFFSET)&EndOfData;
             return(RPDONE);
        }


        /*
        ** Set varable in data segement to point to the
        ** OS/2 DevHlp entry point.  The address of this
        ** routine is in the DevHdr data structure.
        ** This address will be referenced on later calls
        ** to DevHlp.
        */

        DevHlp = (char far *) rp->s.Init.DevHlp;

        ParseArgs(rp->s.Init.args);       // Get CONFIG.SYS command line

        if (usNumHeaders == 1)            // config.sy s did not have 2nd hdr name
               DevHdr[0].DHnext = (PVOID)0xffffffff;

         /******************************************************************/
         /* Search device header chain to find next available audio header */
         /******************************************************************/
        if (! (DevHlp_AttachDD(DevHdr[0].DHname, &AttachArea))) {
                szNum = &DevHdr[0].DHname[5];
                for (i=2; i<=9; i++) {
                        *szNum = (char)('0'+i);
                        if (DevHlp_AttachDD(DevHdr[0].DHname, &AttachArea))
                              break;
                }  /* end of for */
        }  /* end of if */

        /**********************************************************/
        /* Establish Vdd-Pdd link                                 */
        /* ONLY register the header from MMPM/2 installation      */
        /* Do NOT register the AUDIOn$, because the MMPM/2 VDD    */
        /* is only hooking the header from the config.sys /n:     */
        /**********************************************************/
        if (usNumHeaders == 2) {
                strncpy(szDevName, DevHdr[1].DHname, 8);
                if (DevHlp_RegisterPDD(szDevName, (PVOID)PDDEntryPoint)){
                        /*fails - deinstall driver */
                        rp->s.InitExit.finalCS = (OFFSET) 0;
                        rp->s.InitExit.finalDS = (OFFSET) 0;
                        return (RPDONE | RPERR | RPBADCMD);
                }
        }

        /*******************************/
        /* Autoscan device for settings*/
        /*******************************/
        if (Init_POS() == ISA_BUS) {                                    /* check for MCA and read POS */
                for(AdapterInfo.ulPort=0x300; AdapterInfo.ulPort<0xfff8; AdapterInfo.ulPort+=8) {
                        if((AdapterInfo.ulPort+7) == CARD_ID) {
//                              AdapterInfo.usIRQLevel =  ;                             /* get your IRQ */
//                              AdapterInfo.usDMALevel =  ;                             /* get your DMA */
                                break;
                        }
                }
                AdapterInfo.ulPort -=8;
        } else {              /* end of ISA scan */
                bFound = FALSE;
                for (i=0;  i<=MAX_POS_SLOTS; i++) {
                        if (POSCardID[i] == CARD_ID) {
                                bFound = TRUE;
                                break;
                        }
                } /* endfor */
                if (!bFound) {
                        rp->s.InitExit.finalCS = (OFFSET) 0;
                        rp->s.InitExit.finalDS = (OFFSET) 0;
                        return (RPDONE | RPERR | RPBADCMD);
                }

                /* Now that we know which slot our card is in, go get the device dependent data */
                POSCardID[i];
//              AdapterInfo.usIRQLevel =  ;                             /* get your IRQ */
//              AdapterInfo.usDMALevel =  ;                             /* get your DMA */

        } /* end of MCA scan */

        /*
        ** As return values, tell the operating system the
        ** address (offset) of the end of the code and data segments.
        */
        rp->s.InitExit.finalCS = (OFFSET)&IOCTL_Init;
        rp->s.InitExit.finalDS = (OFFSET)&EndOfData;

        /*
        ** Call routine to get streaming initialized with MMPM/2
        ** and then return to the kernel via our assembler code.
        */
        return(InitStreams());
}

/********************* START OF SPECIFICATIONS *********************
* SUBROUTINE NAME:    Init_POS
* DESCRIPTIVE NAME:   Init POS ID data structure
* FUNCTION:   This routine initializes the POS ID data.  Data is retrieved from  ABIOS calls.
* NOTES:
*   PseudoCode
*
*     Get LID Entry for ABIOS POS
*     Return if not successful. (ISA_BUS)
*     Get Required Length of Request Block.
*     if successful and if sufficent storage
*       for each slot
*         get POS ID from ABIOS
*         if ABIOS call successful
*           assign POS ID
*         else
*           POS ID equals zero
*     Free LID Entry
*     Return(MCA_BUS)
*
* ENTRY POINTS:
*     LINKAGE:  Near from IOCTL_Init()
*      PREQPACKET rp - pointer to Device driver request packet
*
* INTERNAL REFERENCES: none
*
* EXTERNAL REFERENCES: DevHlps
*********************** END OF SPECIFICATIONS **********************/
USHORT  Init_POS(void)
{
        USHORT i,rc;            /* index and return code    */
        USHORT usLID;           /* Logical ID               */
        ULONG  ulPOSData;       /* POS data value (102-105) */

        /* Get LID Entry for POS */
        if (rc = DevHlp_GetLIDEntry(LOGICAL_POS_ID, 0, 1, &usLID))
                return(ISA_BUS);

        /* Get length of RB to use for reading POS data. */
        POSLenRB.rb.RBLen = sizeof(POSLENRB);
        POSLenRB.rb.Func  = 0x01;
        POSLenRB.rb.LID   = usLID;
        POSLenRB.rb.Unit  = 0;
        POSLenRB.rb.Resv1 = 0;
        POSLenRB.rb.Resv2 = 0;
        POSLenRB.Rsv1     = 0;
        POSLenRB.Rsv2     = 0;
        POSLenRB.Rsv3     = 0;
        rc = DevHlp_ABIOSCall( usLID, &POSLenRB, 0);

        if ((rc==0) && (sizeof(POSRB) >= POSLenRB.RBLen)) {
          /* Initialize request block for reading POS data. */
          ABIOSrb.rb.RBLen = POSLenRB.RBLen;       /* request block length        */
          ABIOSrb.rb.Func  = 0x0b;                   /* read stored POS data to mem */
          ABIOSrb.rb.LID   = usLID;                  /* Logical ID                  */
          ABIOSrb.rb.Unit  = 0;
          ABIOSrb.DataBuf  = (ULONG)(PVOID)&ulPOSData;

          /* for each slot, get POS ID */
          for(i=0;i<=MAX_POS_SLOTS;i++) {
            ABIOSrb.Slot = (UCHAR)i;
            rc = DevHlp_ABIOSCall(usLID, &ABIOSrb, 0);
            if((rc==0)&&(ABIOSrb.rb.RetCode==0))
              POSCardID[i] = ABIOSrb.AdapterID;
            else
              POSCardID[i] = 0;
            }
          }

        /* Release LID Entry */
        DevHlp_FreeLIDEntry(usLID);
        return(MCA_BUS);
}

/********************* START OF SPECIFICATIONS *********************
*
* SUBROUTINE NAME:          InitStreams
*
* DESCRIPTIVE NAME:
*
* FUNCTION:         Initializes the stream table at device init time.
*
* NOTES: This routine is removed from the code segment after boot time.
*
*
* ENTRY POINTS:
*     LINKAGE:  Near from IOCTL_Init()
*
* INPUT:
*
* EXIT-NORMAL:  NO_ERROR
*
* EXIT_ERROR:
*
* EFFECTS:
*
* INTERNAL REFERENCES: none
*
* EXTERNAL REFERENCES: DevHlps
*
*********************** END OF SPECIFICATIONS **********************/


ULONG InitStreams(VOID)
{
        USHORT          i, j;
        ULONG           BlkSize;
        PVOID           PhysAddress;
        ULONG           rc;
        BOOL            bErrFlg = FALSE;
        PSTREAM         pStream;

        /**********************************/
        /* alloc memory for stream table, */
        /**********************************/

        BlkSize = sizeof(STREAM) * GlobalTable.usMaxNumStreams;

        // Allocate in high memory first.  If it fails allocate in low mem

        if ( DevHlp_AllocPhys(BlkSize, 0, &PhysAddress) ) // Allocate high
           {
              // If that fails, allocate low
              rc = DevHlp_AllocPhys(BlkSize, 1, &PhysAddress);
              // If that fails, installation fails
              if (rc)
                 return(RPDONE | RPERR);
           }

        /*********************************************************/
        /* allocate GDTs                                         */
        /* The GDT addresses are copied from the local variable  */
        /* into the GlobalTable so they can be used when running */
        /* at ring 0 (after initialization)                      */
        /*********************************************************/
        rc = DevHlp_AllocGDTSelector(NUMGDTS, &usGDT[GDT_PSTREAM]);

        /*********************************************************/
        /* Set up a temporary virtual address.                   */
        /* Note, cannot use GDT at init time as init is ring 3.  */
        /* The GDT is allocated during init, but cannot be used  */
        /* until executing at ring 0.                            */
        /*********************************************************/
        rc = DevHlp_PhysToVirt(PhysAddress,
                               BlkSize,
                               &GlobalTable.paStream);
        if (rc)
                return(RPDONE | RPERR);

        //*********************
        // Initialize stream table
        //*********************
        pStream = GlobalTable.paStream;
        for (i=0; i<GlobalTable.usMaxNumStreams; i++)
        {
                pStream->hStream = -1;
                pStream->ulFlags = 0;

                for (j=0; j<MAXIOBUFFS; j++)
                {
                       pStream->IOBuff[j].usRunFlags = 0;
                       pStream->IOBuff[j].lCount  = 1;
                       pStream->IOBuff[j].pBuffer = NULL;
                }
                pStream++;
        }

        //***********************************************
        // Map to GDT selector to address of stream table
        //***********************************************
        if (rc = DevHlp_PhysToGDTSelector(PhysAddress,  // Physical address
                                      BlkSize,          // Length
                                      usGDT[GDT_PSTREAM]))        // Selector to map
                bErrFlg = TRUE;

        else
                GlobalTable.paStream =
                   MAKEP(usGDT[GDT_PSTREAM],0);  // set to virtual GDT pointer

        if (bErrFlg)
                return(RPERR | RPDONE);
        else {
                AllocDMABuffer();
                return(RPDONE);
        }
}

/********************* START OF SPECIFICATIONS *********************
*
* SUBROUTINE NAME:          AllocDMABuffer
*
* DESCRIPTIVE NAME:   Allocate the DMA buffer
*
* FUNCTION:         Allocate the DMA buffer for data transfer to device
*
* NOTES: This routine is removed from the code segment after boot time.
*        For MMPM/2 Video/Audio streaming the buffers sent to the PDD may
*        be samll and not be on a 4K page boundary.  The PDD MUST have
*        logic to partially fill the DMA buffer, request another
*        buffer from MMPM/2, copy that data to the remaining space
*        in the DMA buffer, then finally kick off the DMA transfer
*        to the device.
*        An alternative method would be to allocate the DMA buffer
*        and when "setting up" a video/audio stream stream downsize
*        the DMA buffer to the exact size of the stream data buffer.
*        This will allow a one-to-one buffer transmit from streaming
*        buffer to DMA buffer to kicking off the DMA transfer.
*
* ENTRY POINTS:
*     LINKAGE:  Near from IOCTL_Init()
*
* INPUT:
*
* EXIT-NORMAL:  NO_ERROR
*
* EXIT_ERROR: Return code from DevHlp that failed
*
* EFFECTS:  Allocates fixed non-swappable memory
*
* INTERNAL REFERENCES: none
*
* EXTERNAL REFERENCES: DevHlps
*
*********************** END OF SPECIFICATIONS **********************/

USHORT AllocDMABuffer(VOID)
{
        USHORT  BlkSize;
        USHORT  rc;
        ULONG   ulAlign;
        ULONG   ulAlignmentOffset;
        ULONG   ulAlignedSpace;
        ULONG   ulDMAPhysAddr;


        /**********************************/
        /* alloc memory for DMA Buffer    */
        /**********************************/

        BlkSize = DMABUFFER;

        rc=DevHlp_AllocPhys(((ULONG)BlkSize)*2, 1, &ulDMAPhysAddr);      // alloc low memory
        if (!rc)
                rc=DevHlp_PhysToGDTSelector((PVOID) ulDMAPhysAddr,(USHORT)((BlkSize*2)-2), usGDT[GDT_DMABUFF]);

        switch (AdapterInfo.usDMALevel)
                {
                case 0:                       /*  align for 8bit DMA - 64K boundary */
                case 1:
                case 2:
                case 3:
                        ulAlign=0x10000L;
                        break;

                case 5:                       /*  align for 16bit DMA - 128K boundary */
                case 6:
                case 7:
                        ulAlign=0x20000L;
                        break;

                default:
                        break;
                }


        ulAlignmentOffset= ulDMAPhysAddr & (ulAlign-1);
        ulAlignedSpace   = ulAlign-ulAlignmentOffset;

        /************************************************/
        /* check for re-alignment on boundary           */
        /************************************************/
        if (ulAlignedSpace< (DWORD) BlkSize)
        {
                ulDMAPhysAddr += ulAlignmentOffset;
                rc=DevHlp_PhysToGDTSelector((PVOID) ulDMAPhysAddr,BlkSize, usGDT[GDT_DMABUFF]);
                if (rc)
                        return(rc);
        }

        GlobalTable.pDMABuffer = MAKEP(usGDT[GDT_DMABUFF],0);  // set to virtual GDT pointer
        return(rc);
}

/********************* START OF SPECIFICATIONS *********************
*
* SUBROUTINE NAME: ParseArgs
*
* DESCRIPTIVE NAME: Parse Command line arguments
*
* FUNCTION: To parse the command line arguments from the config.sys command line.
*
* NOTES: This code will be discarded from the device driver after it is envoked.
*
*        Some other command line options may be:
*               /B:xxx  Board address
*               /M:n    where n=1 for on or 0 for off FM midi capability
*               /J:n    where n=1 for on or 0 for off Joy stick enable
*
* ENTRY POINTS:  ParseArgs()
*     LINKAGE:   CALL NEAR
*
* INPUT: pointer to the request packet.
*
* EXIT-NORMAL: ax=0000h
*
* EXIT_ERROR:
*
* EFFECTS:
*
* INTERNAL REFERENCES: none
*
* EXTERNAL REFERENCES: none
*
*********************** END OF SPECIFICATIONS **********************/
VOID    ParseArgs(PUCHAR args)
{
        extern  DEVHDR  MMDevHdr;
        CHAR    Streams[2];
        USHORT  i;
        PGLOBAL pGlobal;                 // ptr to global struct

        pGlobal = &GlobalTable;

        while (*(args)++ != ' ');                      // skip driver file name

        while (*args != NULL)
        {
                while (*(args)++ != '/')                        // skip non slash
                        if (*args == NULL)
                                break;                    // Already set to DEFAULTSTREAMS in table

                switch (*args) {
                                //******************************
                                // NUMBER OF STREAMS TO SUPPORT
                                //******************************
                        case 'S':                              // number of streams /S:99
                        case 's':
                                args++;                  // skip over 'S'
                                if (*args == ':')              // skip over colon
                                        args++;
                                Streams[0] = *(args)++;          // get 1st digit
                                if (*args >= '0' && *args <= '9')
                                        Streams[1] = *args;          // get 2nd digit
                                else
                                        Streams[1] = NULL;
                                /*****************************************************/
                                /* convert ascii to int. From right to left       */
                                /*****************************************************/
                                if (Streams[1] == NULL) {       // single digit
                                        pGlobal->usMaxNumStreams = Streams[0] & 0x000F;
                                } else {
                                        pGlobal->usMaxNumStreams = Streams[1] & 0x000F;
                                        pGlobal->usMaxNumStreams += (Streams[0] & 0x000F)*10;
                                }
                                //***************************************************
                                // safety check in case zeroes or spaces were entered
                                //***************************************************
                                if (pGlobal->usMaxNumStreams == 0)
                                        pGlobal->usMaxNumStreams = DEFAULTSTREAMS;

                                break;
                                //******************************
                                // GET UNIQUE NAME OF DEV HDR
                                //******************************
                        case 'N':
                        case 'n':
                                args++;                  // skip over 'N'
                                if (*args == ':')              // skip over colon
                                        args++;
                                for (i=0; i<sizeof(DevHdr[1].DHname-1); i++)  // clear default name
                                        DevHdr[1].DHname[i] = ' ';
                                for (i=0; i<sizeof(DevHdr[1].DHname-1); i++) {
                                        if (*args == NULL || *args == 0x0d || *args == 0x0a)
                                                break;
                                        DevHdr[1].DHname[i] = *(args)++;                 // get 1st digit
                                }
                                usNumHeaders = 2;     // so we know how many hdrs to register
                                break;
                                //******************************
                                // GET DMA LEVEL
                                //******************************
                        case 'D':                              // DMA channel
                        case 'd':
                                args++;                  // skip over 'D'
                                if (*args == ':')              // skip over colon
                                        args++;
                                AdapterInfo.usDMALevel =*(args)++ & 0x000f;             // DMA limit is 7
                                break;
                                //******************************
                                // GET IRQ LEVEL
                                //******************************
                        case 'I':                              // number of streams /S:99
                        case 'i':
                                args++;                  // skip over 'I'
                                if (*args == ':')              // skip over colon
                                        args++;
                                AdapterInfo.usIRQLevel =*(args)++ & 0x000f;
                                break;
                }       //end case
        };            //end while
}
