/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
// strmhelp.c
//   This file contains routines that support stream related services.


    // include files
#include "types.h"
#include "strmhelp.h"
#include "strmcont.h"
#include "device.h"
#include "mem.h"
#ifdef DEBUG_FLAG
#include "timer.h"
#include "dma.h"
#include "trace.h"
#endif


    // debug info
#ifdef DEBUG_FLAG
#define static /**/
#endif


    // constant definitions
#define NUM_OF_CHANNELS 2
#define MAX_NUM_BUF_ELEMENTS 10
#define NULL_HANDLE 0xffffffff


    // private global variables
static STREAM *Active_streamP, *First_streamP;
static STREAM *Free_stream_tailP, *Free_stream_headP;
static USHORT Update_stream_time_counter = 1;


    // external global variable
extern SHORT Trace_timer;


    // prototypes for private functions
BUF_ELEMENT *alloc_buf_element_list(USHORT);
VOID free_buf_element_list(BUF_ELEMENT *);
BUF_ELEMENT *get_free_buf_element(STREAM *);
VOID put_free_buf_element(STREAM *, BUF_ELEMENT *);
BUF_ELEMENT *get_active_buf_element_from_que(STREAM *);
VOID del_buf_element_from_que_head(STREAM *);
VOID start_stream_play(VOID);
UCHAR *get_next_buf_data(STREAM *, USHORT *);
VOID start_stream_record(VOID);
VOID del_all_buf_elements_from_que(STREAM *);
VOID do_stream_play_operation(VOID);
VOID do_stream_record_operation(VOID);


VOID do_reg_stream(STREAM *streamP, ULONG handle)
//    This function registers a stream.
//    The parameter 'streamP' is a pointer the the stream. The parameter
// 'handle' is a number unique to the stream.
//    The function does not return a value.
{
        // assign the handle
    streamP->handle = handle;

        // finished
    return;
}


VOID do_dereg_stream(STREAM *streamP)
//    This function de-registers a stream.
//    The parameter 'streamP' is a pointer the the stream.
//    The function does not return a value.

{
        // assign a null handle
    streamP->handle = NULL_HANDLE;

        // delete all the stream buffer elements
    del_all_buf_elements_from_que(streamP);

        // finished
    return;
}


BOOL stream_registered(STREAM *streamP)
//    This function determines if a stream is registered.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns the value 'TRUE' if the function is registered.
// Otherwise, the function returns the value 'FALSE'.
{
    return(streamP->handle != NULL_HANDLE);
}


VOID set_active_stream(STREAM *streamP)
//    This function sets the currently active stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.
{
        // set the active stream
    Active_streamP = streamP;

        // finished
    return;
}


ULONG get_stream_handle(STREAM *streamP)
//    This function gets the handle for a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns the handle.
{
    return(streamP->handle);
}


STREAM *get_active_stream(VOID)
//    This function gets the currently active stream.
//    The function does not have parameters.
//    The function returns a pointer to the active stream.
{
    return(Active_streamP);
}


STREAM *get_stream_from_handle(ULONG handle)
//    This function gets a pointer to a stream from its handle.
//    The parameter 'handle' is a unique number associated with the stream.
//    The function returns the pointer to the stream.
{
   STREAM *streamP;

        // traverse the list - return the pointer if found
    streamP = First_streamP;
    while (1) {
        if (streamP == (STREAM *) 0)
            return((STREAM *) 0);
        if (streamP->handle == handle)
            return(streamP);

        streamP = streamP->nextP;
    }
}


STREAM *get_stream_from_sys_file_num(USHORT sys_file_num)
//    This function gets a pointer to a stream from its system file number.
//    The parameter 'sys_file_num' is an unique number associated with
// the stream.
//    The function returns the pointer to the stream.
{
    STREAM *streamP;

        // traverse the list - return the pointer if found
    streamP = First_streamP;
    while (1) {
        if (streamP == (STREAM *) 0)
            return((STREAM *) 0);
        if (streamP->sys_file_num == sys_file_num)
            return(streamP);

        streamP = streamP->nextP;
    }
}


VOID set_stream_clock_and_format(STREAM *streamP, USHORT sample_rate,
              SHORT data_format, UCHAR bits_per_sample, UCHAR num_of_channels)
//    This function sets the sample rate, bits per sample, data format, and
// the number of channels for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameters
// 'sample_rate', 'data_format', 'bits_per_sample', and 'num_of_channels'
// contain the sample rate and data format information.
//    The function does not return a value.
{
        // set the info
    streamP->sample_rate = sample_rate;
    streamP->data_format = data_format;
    streamP->bits_per_sample = bits_per_sample;
    streamP->num_of_channels = num_of_channels;

        // finished
    return;
}


VOID set_stream_operation(STREAM *streamP, SHORT operation)
//    This function sets the operation for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// 'operation' is the operation for the stream.
//    The function does not return a value.
{
        // set the operation
    streamP->operation = operation;

        // finished
    return;
}


SHORT get_stream_operation(STREAM *streamP)
//    This function gets the operation for a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns the operation.
{
        // return the operation
    return(streamP->operation);
}


SHORT get_stream_state(STREAM *streamP)
//    This function gets the state for a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns the state.

{
        // return the state
    return(streamP->state);
}


ULONG get_stream_time(STREAM *streamP)
//    This function gets the time for a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns the time.
{
    return(streamP->current_time);
}


VOID set_stream_input_device(STREAM *streamP, SHORT type)
//    This function sets the input device for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter 'type'
// is the input device type.
//    The function does not return a value.
{
        // set the input device type
    streamP->input_device = type;

        // finished
    return;
}


VOID set_stream_output_device(STREAM *streamP, SHORT type)
//    This function sets the output device for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter 'type'
// is the output device type.
//    The function does not return a value.

{
        // set the output device type
    streamP->output_device = type;

        // finished
    return;
}


VOID set_stream_size(STREAM *streamP, USHORT buffer_size)
//    This function sets the size of a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// 'buffer_size' is the size of the first buffer sent for this stream.
//    The function does not return a value.
{
        // set the stream buffer size
    streamP->buffer_size = buffer_size;

        // return;
    return;
}


VOID init_strmhelp(VOID)
//    This function initializes stream services.
//    The function does not have parameters.
//    The function does not return a value.
{
        // initialize stream pointers
    Active_streamP = First_streamP = (STREAM *) 0;

        // finished
    return;
}


VOID set_stream_time(STREAM *streamP, ULONG time)
//    This function sets the time for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// 'time' is the time for the stream.
//    The function does not return a value.
{
        // set the stream time
    streamP->current_time = time;

        // finished
    return;
}


static VOID start_stream_play(VOID)
//    This function starts a stream for the play operation.
//    The function does not have parameters.
//    The function does not return a value.
{
    USHORT buffer_size_needed, buffer_size;
    UCHAR *dataP;

    if (get_dev_type() != DEV_REEL_MAGIC) {

            // calulate data size for 2 buffers
        buffer_size_needed = get_dev_buffer_size() * 2;
        buffer_size = buffer_size_needed;

            // init device with 2 buffers
        if ((dataP = get_next_buf_data(Active_streamP,
                                                 &buffer_size)) != (UCHAR *) 0) {
            send_dev_data(dataP, buffer_size);

            buffer_size = buffer_size_needed - buffer_size;

            if (buffer_size != 0) {
                if ((dataP = get_next_buf_data(Active_streamP,
                                                    &buffer_size)) != (UCHAR *) 0)
                    send_dev_data(dataP, buffer_size);
            }
        }

            // initialize the volume
        set_dev_left_volume(Active_streamP->channel[0].volume);
        set_dev_right_volume(Active_streamP->channel[1].volume);

            // set up output source type
        switch (Active_streamP->output_device) {
            case STREAM_OUTPUT_STEREO_LINE:
                start_dev_play();
                break;
            case STREAM_OUTPUT_NULL:
                break;
        }
    }

        // finished
    return;
}


VOID set_stream_state(STREAM *streamP, SHORT state)
//    This function sets the state for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter 'state'
// is the state of the stream.
//    The function does not return a value.
{
    BUF_ELEMENT *cur_buf_elementP;
    USHORT buffer_size, buffer_size_needed;
    UCHAR *bufP, *dataP;

    switch (state) {

        case STREAM_STATE_STOP:

                // set the state
            streamP->state = state;

                // if stream stopped during play - stop the device - return
                // all stream buffers from que
            if (streamP->operation == STREAM_OPERATION_PLAY) {
                stop_dev_operation();

                while (1) {

                        // get the element from the head of the que
                    if ((cur_buf_elementP = streamP->head_buf_elementP)
                                                       == (BUF_ELEMENT *) 0)
                        return;

                        // get the buffer pointer and the buffer size
                    bufP = cur_buf_elementP->bufferP;
                    buffer_size = cur_buf_elementP->buffer_size;

                        // delete the element at the head of the que
                    del_buf_element_from_que_head(streamP);

                        // return the buffer to sender
                    return_stream_buffer(streamP, bufP, buffer_size);
                }
            }


                // if stream stopped during record - get remaining data -
                // stop the device - return all buffers
            else {

                    // get the amount of data needed from the device
                buffer_size_needed = get_dev_num_bytes_produced();
                buffer_size = buffer_size_needed;

                    // stop the device
                stop_dev_operation();

                    // get data from the device 
                if ((dataP = get_next_buf_data(Active_streamP,
                                              &buffer_size)) != (UCHAR *) 0) {
                    get_dev_data(dataP, buffer_size);

                    bufP = Active_streamP->head_buf_elementP->bufferP;

                        // delete the element at the head of the que
                    del_buf_element_from_que_head(Active_streamP);

                        // return the buffer to sender
                    return_stream_buffer(Active_streamP, bufP, buffer_size);

                    buffer_size = buffer_size_needed - buffer_size;

                    if (buffer_size != 0) {
 
                        if ((dataP = get_next_buf_data(Active_streamP,
                                               &buffer_size)) != (UCHAR *) 0) {
                            get_dev_data(dataP, buffer_size);

                            bufP = Active_streamP->head_buf_elementP->bufferP;

                                // delete the element at the head of the que
                            del_buf_element_from_que_head(Active_streamP);

                                // return the buffer to sender
                            return_stream_buffer(Active_streamP, 
                                                            bufP, buffer_size);
                        }
                    }
                }

                while (1) {

                        // get the element from the head of the que
                    if ((cur_buf_elementP = streamP->head_buf_elementP)
                                                       == (BUF_ELEMENT *) 0)
                        return;

                        // get the buffer pointer and the buffer size
                    bufP = cur_buf_elementP->bufferP;
                    buffer_size = 0;

                        // delete the element at the head of the que
                    del_buf_element_from_que_head(streamP);

                        // return the buffer to sender
                    return_stream_buffer(streamP, bufP, buffer_size);
                }

            }


            return;

        case STREAM_STATE_PAUSE:

                // set the state
            streamP->state = state;

                // stop the device
            stop_dev_operation();

                // reset position to head of the que
            streamP->cur_buf_elementP = streamP->head_buf_elementP;

                // reset buffer element info
            cur_buf_elementP = streamP->tail_buf_elementP;
            while (1) {
                if (cur_buf_elementP == (BUF_ELEMENT *) 0)
                    break;
                cur_buf_elementP->next_data_trans_pos = 0;
                cur_buf_elementP->cur_dev_consumed_pos = FALSE;
                cur_buf_elementP = cur_buf_elementP->nextP;
            }

            return;

        case STREAM_STATE_START:

                // set the active stream
            Active_streamP = streamP;

                // set the state for the stream
            streamP->state = state;

                // configure device
            set_dev_config_info(streamP->buffer_size,
               streamP->data_format, streamP->bits_per_sample,
                  streamP->sample_rate, streamP->num_of_channels);

                // start the stream
            if (streamP->operation == STREAM_OPERATION_PLAY)
                start_stream_play();

            else
                start_stream_record();

            return;

        default:
            return;
    }
}


VOID set_stream_volume(STREAM *streamP, ULONG setting)
//    This function sets the volume for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// 'setting' is the volume level.
//    The function does not return a value.
{
        // store the new settings
    streamP->channel[0].volume = setting;
    streamP->channel[1].volume = setting;

        // set the device
    set_dev_left_volume(setting);
    set_dev_right_volume(setting);

        // finished
    return;
}


VOID set_stream_input_level(STREAM *streamP, LONG setting)
//    This function sets the input level for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// 'setting' is the input gain level.
//    The function does not return a value.
{
        // store the new settings
    streamP->channel[0].input_level = setting;
    streamP->channel[1].input_level = setting;

        // set the device
    set_dev_left_input_level(setting);
    set_dev_right_input_level(setting);

        // finished
    return;
}


VOID start_stream_record(VOID)
//    This function starts a stream for a record operation.
//    The function does not have parameters.
//    The function does not return a value.
{
        // set the input gain levels
    set_dev_left_input_level(Active_streamP->channel[0].input_level);
    set_dev_right_input_level(Active_streamP->channel[1].input_level);

        // set the input device type
    switch (Active_streamP->input_device) {
        case STREAM_INPUT_STEREO_LINE:
            start_dev_record(DEV_AUX1);
            break;
        case STREAM_INPUT_MIC:
            start_dev_record(DEV_MICROPHONE);
            break;
        case STREAM_INPUT_BOOSTED_MIC:
            start_dev_record(DEV_BOOSTED_MICROPHONE);
            break;
    }

        // finished
    return;
}


VOID update_stream_times(VOID)
//    This function updates the time for the currentlt active stream.
//    The function does not have parameters.
//    The function does not return a value.
{

        // return if no active stream
    if (Active_streamP == (STREAM *) 0)
        return;


        // increment the time
    if (Active_streamP->state == STREAM_STATE_START) {
        if (Update_stream_time_counter == 4) {
            Update_stream_time_counter = 1;
            Active_streamP->current_time += 32L;
        }
        else {
            Update_stream_time_counter++;
            Active_streamP->current_time += 31L;
        }
    }

        // finished
    return;
}


VOID do_stream_operation(VOID)
//    This function services stream operation requests from the device.
//    The function does not have parameters.
//    The function does not return a value.
{
    BUF_ELEMENT *cur_buf_elementP;
    VOID *bufP;
    UCHAR *dataP;
    USHORT buffer_size_needed, buffer_size;

        // finished if no active stream
    if (Active_streamP == (STREAM *) 0)
       return;

        // get the element from the head of the que
    if (Active_streamP->head_buf_elementP == (BUF_ELEMENT *) 0)
        return;

        // send data to the device if play operation
    if (Active_streamP->operation == STREAM_OPERATION_PLAY) {
        do_stream_play_operation();
    }

        // get data from the device if record operation
    else {
        do_stream_record_operation();
    }

        // finished
    return;
}


static VOID do_stream_play_operation(VOID)
//    This function services play stream operation requests from the device.
//    The function does not have parameters.
//    The function does not return a value.
{
    BUF_ELEMENT *cur_buf_elementP;
    VOID *bufP;
    UCHAR *dataP;
    USHORT buffer_size_needed, buffer_size, num_bytes_consumed;
    USHORT overflow;
    ULONG new_position;


        // get the amount of data needed from the device
    buffer_size_needed = get_dev_buffer_size();
    buffer_size = buffer_size_needed;

        // send data to the device
    dataP = get_next_buf_data(Active_streamP, &buffer_size);

    send_dev_data(dataP, buffer_size);

    buffer_size = buffer_size_needed - buffer_size;
    buffer_size_needed = buffer_size;

    dataP = get_next_buf_data(Active_streamP, &buffer_size);

    send_dev_data(dataP, buffer_size);

    buffer_size = buffer_size_needed - buffer_size;

    clear_dev_data(buffer_size);

        // update the current device used position in the buffer - if we
        // cross the buffer boundary with the produced position update the
        // return the buffer
        // get the number of bytes consumed
    cur_buf_elementP = Active_streamP->head_buf_elementP;

    cur_buf_elementP->cur_dev_consumed_pos += get_dev_num_bytes_consumed();

    if (cur_buf_elementP->cur_dev_consumed_pos >= cur_buf_elementP->buffer_size) {

        overflow = cur_buf_elementP->cur_dev_consumed_pos -
                                                cur_buf_elementP->buffer_size;

            // get the buffer pointer and the buffer size
        bufP = cur_buf_elementP->bufferP;
        buffer_size = cur_buf_elementP->buffer_size;

            // delete the element at the head of the que
        del_buf_element_from_que_head(Active_streamP);

            // return the buffer to sender
        return_stream_buffer(Active_streamP, bufP, buffer_size);

            // get the element from the head of the que
        if ((cur_buf_elementP = Active_streamP->head_buf_elementP)
                                                         != (BUF_ELEMENT *) 0)
            cur_buf_elementP->cur_dev_consumed_pos = overflow;
    }


        // finished
    return;
}


VOID do_stream_record_operation(VOID)
//    This function services record stream operation requests from the device.
//    The function does not have parameters.
//    The function does not return a value.
{
    BUF_ELEMENT *cur_buf_elementP;
    VOID *bufP;
    UCHAR *dataP;
    USHORT buffer_size_needed, buffer_size;

        // call function so we get a good reading on the stop
    get_dev_num_bytes_produced();

        // get the element from the head of the que
    cur_buf_elementP = Active_streamP->head_buf_elementP;

        // get the amount of data needed from the device
    buffer_size_needed = get_dev_buffer_size();
    buffer_size = buffer_size_needed;

        // get data from the device if record operation
    if ((dataP = get_next_buf_data(Active_streamP,
                                              &buffer_size)) != (UCHAR *) 0) {
        get_dev_data(dataP, buffer_size);

        buffer_size = buffer_size_needed - buffer_size;

        if (buffer_size != 0) {
            if ((dataP = get_next_buf_data(Active_streamP,
                                                 &buffer_size)) != (UCHAR *) 0)
                get_dev_data(dataP, buffer_size);
        }
    }

        // return buffer if finished
    if (Active_streamP->head_buf_elementP != Active_streamP->cur_buf_elementP) {

            // get the buffer pointer and the buffer size
        bufP = Active_streamP->head_buf_elementP->bufferP;
        buffer_size = Active_streamP->head_buf_elementP->buffer_size;

            // delete the element at the head of the que
        del_buf_element_from_que_head(Active_streamP);

            // return the buffer to sender
        return_stream_buffer(Active_streamP, bufP, buffer_size);
    }

        // finished
    return;
}


SHORT create_stream(USHORT sys_file_num)
//    This function creates a new stream.
//    The parameter 'sys_file_num' is a number unique to this stream.
//    The function returns the value 0 if successful. Otherwise, the value
// -1 is returned.
{
    STREAM *streamP;
    USHORT i;

        // allocate the stream structure
    if ((streamP = (STREAM *) mem_alloc(sizeof(STREAM))) == (STREAM *) 0)
        return (-1);

        // allocate list of free buf elements
    if ((streamP->first_free_buf_elementP =
         alloc_buf_element_list(MAX_NUM_BUF_ELEMENTS)) == (BUF_ELEMENT *) 0) {
        mem_free((UCHAR *) streamP);
        return (-1);
    }

        // initialize stream element
    streamP->head_buf_elementP = (BUF_ELEMENT *) 0;
    streamP->cur_buf_elementP = (BUF_ELEMENT *) 0;
    streamP->tail_buf_elementP = (BUF_ELEMENT *) 0;
    streamP->sys_file_num = sys_file_num;
    streamP->handle = NULL_HANDLE;
    streamP->state = STREAM_STATE_STOP;

        // initialize input devices
    streamP->input_device = STREAM_INPUT_NULL;

        // initialize output devices
    streamP->output_device = STREAM_OUTPUT_NULL;

        // initialize level controls
    for (i = 0; i <= NUM_OF_CHANNELS - 1; i++) {
        streamP->channel[i].volume = 0L;
        streamP->channel[i].input_level = 0L;
    }

        // install new element at head of list
    if (First_streamP == (STREAM *) 0) {
        streamP->prevP = (STREAM *) 0;
        streamP->nextP = (STREAM *) 0;
        First_streamP = streamP;
    }
    else {
        streamP->prevP = (STREAM *) 0;
        streamP->nextP = First_streamP;
        First_streamP->prevP = streamP;
        First_streamP = streamP;
    }

        // return successful
    return(0);
}


static BUF_ELEMENT *alloc_buf_element_list(USHORT num_of_elements)
//    This function allocate a list of buf elements.
//    The parameter 'num_of_elements' is the number of buffer elements to
// insert into the list.
//    The function returns a pointer the head of the list if successful.
// Otherwise, the function returns a null pointer.
{
    BUF_ELEMENT *new_buf_elementP, *cur_buf_elementP, *first_buf_elementP;
    USHORT i;

        // allocate first element in list - return if error
    if ((cur_buf_elementP =
          (BUF_ELEMENT *) mem_alloc(sizeof(BUF_ELEMENT))) == (BUF_ELEMENT *) 0)
        return ((BUF_ELEMENT *) 0);

    cur_buf_elementP->nextP = (BUF_ELEMENT *) 0;
    first_buf_elementP = cur_buf_elementP;

        // allocate the rest of the list - if error then free partial list
        // and return
    for (i = 1; i <= MAX_NUM_BUF_ELEMENTS - 1; i++) {
        if ((new_buf_elementP =
                   (BUF_ELEMENT *) mem_alloc(sizeof(BUF_ELEMENT)))
                                                        == (BUF_ELEMENT *) 0) {
            free_buf_element_list(first_buf_elementP);
            return ((BUF_ELEMENT *) 0);
        }
        cur_buf_elementP->nextP = new_buf_elementP;
        cur_buf_elementP = cur_buf_elementP->nextP;
        cur_buf_elementP->nextP = (BUF_ELEMENT *) 0;
    }

        // return pointer to head of list
    return(first_buf_elementP);
}


static VOID free_buf_element_list(BUF_ELEMENT *first_buf_elementP)
//    This function frees the memory associated with a list of buffer elements.
//    The parameter 'first_buf_elementP' is a pointer to the head of the list.
//    The function does not return a value.
{
    BUF_ELEMENT *next_buf_elementP;
    USHORT i;

        // traverse the list and free the memory
    while (1) {
        if (first_buf_elementP == (BUF_ELEMENT *) 0)
            return;
        next_buf_elementP = first_buf_elementP->nextP;
        mem_free((UCHAR *) first_buf_elementP);
        first_buf_elementP = next_buf_elementP;
    }

        // finished
    return;
}


static VOID put_free_buf_element(STREAM *streamP, BUF_ELEMENT *buf_elementP)
//    This function adds a buffer element to the free list associated with
// a stream.
//    The parameter 'streamP' is a pointer to the stream and the parameter
// 'curP' is a pointer to the buffer element.
//    The function does not return a value.
{
        // add element to front of the list
    buf_elementP->nextP = streamP->first_free_buf_elementP;
    streamP->first_free_buf_elementP = buf_elementP;

        // finished
    return;
}


static BUF_ELEMENT *get_free_buf_element(STREAM *streamP)
//    This function gets a buffer element from the free list associated with
// a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns a pointer to the head of the list.
{
    BUF_ELEMENT *buf_elementP;

        // get next free element - move the head pointer
    if ((buf_elementP = streamP->first_free_buf_elementP) != (BUF_ELEMENT *) 0)
        streamP->first_free_buf_elementP =
                                   streamP->first_free_buf_elementP->nextP;

        // finished
    return (buf_elementP);
}


VOID destroy_stream(STREAM *streamP)
//    This function removes a stream element from the list and frees the
// element's storage.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value
{
    STREAM *prev_streamP;
    STREAM *next_streamP;

        // return if null stream
    if (streamP == (STREAM *) 0)
        return;

        // adjust the active stream if necessary
    if (Active_streamP == streamP)
        Active_streamP = (STREAM *) 0;

        // free que of used buf elements
    free_buf_element_list(streamP->tail_buf_elementP);

        // free list of unused buf elements
    free_buf_element_list(streamP->first_free_buf_elementP);

        // point to next and prev elements
    prev_streamP = streamP->prevP;
    next_streamP = streamP->nextP;

        // adjust the head pointer if necessary
    if (streamP == First_streamP)
        First_streamP = next_streamP;

        // remove the element from the list
    if (prev_streamP != (STREAM *) 0)
        prev_streamP->nextP = next_streamP;
    if (next_streamP != (STREAM *) 0)
        next_streamP->prevP = prev_streamP;

        // free the memory
    mem_free((UCHAR *) streamP);

        // done
    return;
}


VOID del_all_buf_elements_from_que(STREAM *streamP)
//    This function deletes all buffer elements associated with a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.
{
    BUF_ELEMENT *cur_buf_elementP, *next_buf_elementP;

        // put all buffer elements into the free list
    cur_buf_elementP = next_buf_elementP = streamP->tail_buf_elementP;
    while (1) {
        if (cur_buf_elementP == (BUF_ELEMENT *) 0)
            break;
        next_buf_elementP = cur_buf_elementP->nextP;
        put_free_buf_element(streamP, cur_buf_elementP);
        cur_buf_elementP = next_buf_elementP;
    }

        // reset que pointers
    streamP->head_buf_elementP = (BUF_ELEMENT *) 0;
    streamP->cur_buf_elementP = (BUF_ELEMENT *) 0;
    streamP->tail_buf_elementP = (BUF_ELEMENT *) 0;

        // finished
    return;
}


static VOID del_buf_element_from_que_head(STREAM *streamP)
//    This function deletes a stream buffer element from the head of the que.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.
{
    BUF_ELEMENT *cur_buf_elementP;

        // point to head of the que
    if ((cur_buf_elementP = streamP->head_buf_elementP) == (BUF_ELEMENT *) 0)
        return;

        // move the head pointer
    streamP->head_buf_elementP = cur_buf_elementP->prevP;

        // reset all que pointers if que is empty
    if (streamP->head_buf_elementP == (BUF_ELEMENT *) 0) {
        streamP->cur_buf_elementP = (BUF_ELEMENT *) 0;
        streamP->tail_buf_elementP = (BUF_ELEMENT *) 0;
    }
    else
        streamP->head_buf_elementP->nextP = (BUF_ELEMENT *) 0;

        // put buffer element on free list
    put_free_buf_element(streamP, cur_buf_elementP);

        // finished
    return;
}


SHORT add_buf_element_to_que_tail(STREAM *streamP, VOID *bufP,
                                                           ULONG buffer_size)
//    This function adds a stream buffer element to the tail of the stream's
// que.
//    The parameter 'streamP' is a pointer to the stream. The parameter 'bufP'
// is a pointer to the new buffer. The parameter 'buffer_size' is the size
// of the new buffer.
//    The function returns the value 0 is successful. Otherwise, the function
// returns the value -1.
{
    BUF_ELEMENT *new_buf_elementP;

        // allocate buf element
    if ((new_buf_elementP = get_free_buf_element(streamP)) == (BUF_ELEMENT *) 0)
        return (-1);

        // initialize buffer info
    new_buf_elementP->bufferP = bufP;
    new_buf_elementP->buffer_size = buffer_size;
    new_buf_elementP->next_data_trans_pos = 0;
    new_buf_elementP->cur_dev_consumed_pos = 0L;

        // if que is empty then init que pointers
    if (streamP->head_buf_elementP == (BUF_ELEMENT *) 0) {
        new_buf_elementP->nextP = (BUF_ELEMENT *) 0;
        new_buf_elementP->prevP = (BUF_ELEMENT *) 0;
        streamP->head_buf_elementP = new_buf_elementP;
        streamP->cur_buf_elementP = new_buf_elementP;
        streamP->tail_buf_elementP = new_buf_elementP;
    }

        // insert on tail of que
    else {
        new_buf_elementP->prevP = (BUF_ELEMENT *) 0;
        new_buf_elementP->nextP = streamP->tail_buf_elementP;
        streamP->tail_buf_elementP->prevP = new_buf_elementP;
        streamP->tail_buf_elementP = new_buf_elementP;
    }

        // check current buf element pointer
    if (streamP->cur_buf_elementP == (BUF_ELEMENT *) 0)
        streamP->cur_buf_elementP = streamP->tail_buf_elementP;

        // return successful
    return(0);
}


UCHAR *get_next_buf_data(STREAM *streamP, USHORT *num_bytes_needed)
//    This function gets a specified amount of data from a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// 'num_byte_needed' is the amount of data to get from the stream.
//     The function returns the actual amount of data retrieved in the parameter
// 'num_bytes_needed'. As a return value, the function returns a pointer to
// the data.
{
    UCHAR *curP;
    USHORT num_bytes_left;

        /* return if nothing in the list */
    if (streamP->cur_buf_elementP == (BUF_ELEMENT *) 0) {
        *num_bytes_needed = 0;
        return((UCHAR *) 0);
    }

        // seek to current position in the buffer
    curP = (CHAR *) streamP->cur_buf_elementP->bufferP
                                + streamP->cur_buf_elementP->next_data_trans_pos;

        // calculate data left in buffer 
    num_bytes_left = streamP->cur_buf_elementP->buffer_size
                       - streamP->cur_buf_elementP->next_data_trans_pos;

        // mark buffer finished and move que pointer to next buffer if finished
        // with this buffer - update current buffer position - set up
        // the number bytes retrieved
    if (num_bytes_left <= *num_bytes_needed) {
        streamP->cur_buf_elementP = streamP->cur_buf_elementP->prevP;
        *num_bytes_needed = num_bytes_left;
    }
    else
        streamP->cur_buf_elementP->next_data_trans_pos += *num_bytes_needed;

        // return a pointer to the data
    return(curP);
}
