/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
// strmcont.c
//    This file contains functions that service requests from the audio steam
// handler and report events to the audio stream handler.


    // include files 
#include <os2.h>                         // system includes
#include <os2medef.h>                    // MME typedefs
#include <meerror.h>                     // MME errors
#include <ssm.h>                         // for attaching to SSM
#include <shdd.h>                        // for attaching to audio DD
#include "strmhelp.h"
#include "strmcont.h"
#include "timer.h"
#ifdef DEBUG_FLAG
#include "trace.h"
#endif


    // debugging
#ifdef DEBUG_FLAG
#define static /**/
#endif


    // type definitions
typedef ULONG (far *FFP_UL_PV)(void *);


    // private global variables 
static FFP_UL_PV Shd_idc_entry_funcP;
static ULONG Current_trace_time;
static ULONG Last_trace_time;
static STATUS_PARM Status_aux_parm;
static CONTROL_PARM Control_aux_parm;
static BOOL New_stream;


    // external global variables
extern SHORT Trace_timer;


    // prototypes for private functions
VOID status(STREAM *, ULONG *);
ULONG register_stream(USHORT, ULONG, FFP_UL_PV);
VOID deregister_stream(STREAM *);
VOID setup(STREAM *, ULONG);
ULONG control_resume(STREAM *);
ULONG control_pause(STREAM *, ULONG *);
ULONG control_stop(STREAM *, ULONG *);
VOID control_start(STREAM *);
ULONG write_device(STREAM *, UCHAR *, USHORT);
ULONG read_device(STREAM *, UCHAR *, USHORT);


VOID init_strmcont(VOID)
{
#ifdef TRACE_BUFFER_OUT
    init_trace(TRACE_BUFFER_OUT);
#endif

#ifdef TRACE_BUFFER_IN
    init_trace(TRACE_BUFFER_IN);
#endif

#ifdef TRACE_BUFFER_ARRIVAL_TIME
    init_trace(TRACE_BUFFER_ARRIVAL_TIME);
    Trace_timer = acquire_timer();
#endif

}


ULONG do_stream_command(VOID *parmP)
//    This function services requests from the audio stream handler.
//    The parameter 'parmP' is a pointer to a buffer containing information
// specific to the idc request. 
//    The function does not return a value. 
{
    ULONG *command_numP;
    ULONG ret_code, stream_time;
    DDCMDREGISTER *register_parmP;
    DDCMDCONTROL *control_parmP;
    DDCMDDEREGISTER *deregister_parmP;
    DDCMDREADWRITE *readwrite_parmP;
    DDCMDSTATUS *status_parmP;
    DDCMDSETUP *setup_parmP;
    SETUP_PARM *setup_aux_parmP;
    STREAM *streamP;

        // get function number 
    command_numP = (ULONG *) parmP;

    switch (*command_numP) {

            // register stream request
        case DDCMD_REG_STREAM:

                // get the parameter packet
            register_parmP = (DDCMDREGISTER *) parmP;

                // do the request
            ret_code = register_stream((USHORT) register_parmP->ulSysFileNum,
                                 (ULONG) register_parmP->hStream,
                                          register_parmP->pSHDEntryPoint);

                // set up return info
            register_parmP->ulBufSize = 0x4000;
            register_parmP->ulNumBufs = 0L;
            register_parmP->ulAddressType = ADDRESS_TYPE_VIRTUAL;

                // finished
            return(ret_code);

            // deregister stream request
        case DDCMD_DEREG_STREAM:

                // get the parameter packet
            deregister_parmP = (DDCMDDEREGISTER *) parmP;

                // get the stream pointer from handle
            if ((streamP = get_stream_from_handle(deregister_parmP->hStream))
                                                             == (STREAM *) 0) {
#ifdef TRACE_ALL_EVENTS
                trace(TRACE_ALL_EVENTS, 0xffffffff);
                trace(TRACE_ALL_EVENTS, 0xec000000);
#endif
                return(ERROR_INVALID_STREAM);
            }

                // do the request
            deregister_stream(streamP);

                // finished
            return (NO_ERROR);

            // control command request
        case DDCMD_CONTROL:

                // get the parameter packet
            control_parmP = (DDCMDCONTROL *) parmP;

                // set up an auxilary packet
            control_parmP->pParm = &Control_aux_parm;

                // get stream pointer from handle
            if ((streamP = get_stream_from_handle(control_parmP->hStream))
                                                             == (STREAM *) 0) {
#ifdef TRACE_ALL_EVENTS
                trace(TRACE_ALL_EVENTS, 0xffffffff);
                trace(TRACE_ALL_EVENTS, 0xed000000);
#endif
                return(ERROR_INVALID_STREAM);
            }

 
            switch(control_parmP->ulCmd) {

                    // start command request
                case DDCMD_START:

                        // do the request
                    control_start(streamP);
#ifdef TRACE_ALL_EVENTS
    trace(1, 0xffffffff);
    trace(1, 0xb1000000);
    trace(1, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

                        // finished
                    return (NO_ERROR);
 
                    // stop command request
                case DDCMD_STOP:

                        // do the request
                    ret_code = control_stop(streamP, &stream_time);

                        // set up return info
                    Control_aux_parm.ulTime = stream_time;
                    control_parmP->ulParmSize = (ULONG) sizeof(CONTROL_PARM);

                        // finished
                    return(ret_code);
 
                    // pause command request
                case DDCMD_PAUSE:

                        // do the request
                    ret_code = control_pause(streamP, &stream_time);

                        // set upo return info
                    Control_aux_parm.ulTime = stream_time;
                    control_parmP->ulParmSize = (ULONG) sizeof(CONTROL_PARM);

                        // finished
                    return(ret_code);
 
                    // resume command request
                case DDCMD_RESUME:
                    return(control_resume(streamP));

                    // enable event request
                case DDCMD_ENABLE_EVENT:
                    return((ULONG) ERROR_INVALID_REQUEST);
 
                    // disable event request
                case DDCMD_DISABLE_EVENT:
                    return((ULONG) ERROR_INVALID_REQUEST);
 
                default:
                    return(0);
            }

            // set up request
        case DDCMD_SETUP:

                // get the parameter packet
            setup_parmP = (DDCMDSETUP *) parmP;

                // set up auxilary packet
            setup_aux_parmP = (SETUP_PARM *) setup_parmP->pSetupParm;

                // get stream pointer from handle
            if ((streamP = get_stream_from_handle(setup_parmP->hStream))
                                                             == (STREAM *) 0) {
#ifdef TRACE_ALL_EVENTS
                trace(TRACE_ALL_EVENTS, 0xffffffff);
                trace(TRACE_ALL_EVENTS, 0xee000000);
#endif
                return(ERROR_INVALID_STREAM);
            }

                // get the new stream time
            stream_time = setup_aux_parmP->ulStreamTime;

                // do the request
            setup(streamP, stream_time);

                // set up return info
            setup_parmP->ulSetupParmSize = (ULONG) sizeof(SETUP_PARM);

                // finished
            return(NO_ERROR);

            // get status request
        case DDCMD_STATUS:

                // get the parameter packet
            status_parmP = (DDCMDSTATUS *) parmP;

                // get stream pointer from handle
            if ((streamP = get_stream_from_handle(status_parmP->hStream))
                                                             == (STREAM *) 0) {
#ifdef TRACE_ALL_EVENTS
                trace(TRACE_ALL_EVENTS, 0xffffffff);
                trace(TRACE_ALL_EVENTS, 0xef000000);
#endif
                return(ERROR_INVALID_STREAM);
            }

                // do the request
            status(streamP, &stream_time);

                // set up return info
            Status_aux_parm.ulTime = stream_time;
            status_parmP->pStatus = &Status_aux_parm;
            status_parmP->ulStatusSize = (ULONG) sizeof(STATUS_PARM);

                // finished
            return(NO_ERROR);
                                                                                              
            // write buffer request
        case DDCMD_WRITE:

                // get the parameter packet
            readwrite_parmP = (DDCMDREADWRITE *) parmP;

                // get the stream pointer from handle
            if ((streamP = get_stream_from_handle(readwrite_parmP->hStream))
                                                             == (STREAM *) 0) {
#ifdef TRACE_ALL_EVENTS
                trace(TRACE_ALL_EVENTS, 0xffffffff);
                trace(TRACE_ALL_EVENTS, 0xefa00000);
#endif
                return(ERROR_INVALID_STREAM);
            }

                // do the request
            return(write_device(streamP, (UCHAR *) readwrite_parmP->pBuffer,
                                      (USHORT) readwrite_parmP->ulBufferSize));
            // read buffer request
        case DDCMD_READ:

                // get the parameter packet
            readwrite_parmP = (DDCMDREADWRITE *) parmP;

                // get the stream pointer from the handle
            if ((streamP = get_stream_from_handle(readwrite_parmP->hStream))
                                                             == (STREAM *) 0) {
#ifdef TRACE_ALL_EVENTS
                trace(TRACE_ALL_EVENTS, 0xffffffff);
                trace(TRACE_ALL_EVENTS, 0xefb00000);
#endif
                return(ERROR_INVALID_STREAM);
            }

                // do the request
            return(read_device(streamP, (UCHAR *) readwrite_parmP->pBuffer,
                                      (USHORT) readwrite_parmP->ulBufferSize));

        default:
            return(0);
    }
}


static ULONG register_stream(USHORT sys_file_num, ULONG stream_handle,
                                                    FFP_UL_PV idc_entry_point)
//    This function registers a stream.
//    The parameter 'sys_file_num' is a number related to the creation of the
// stream. The parameter 'stream_handle' is a handle to the stream used for
// future requests related to this stream. The parameter 'idc_entry_point' is
// the address of the callback function in the audio stream handler.
//    The function returns the value 'ERROR_STREAM_CREATION' if a stream
// related to 'sys_file_num' was not found. The value 'ERROR_HNDLR_REGISTERED'
// is returned if the stream is already registered. Otherwise, the value
// 'NO_ERROR' is returned.
{
    STREAM *streamP;

#ifdef TRACE_BUFFER_ARRIVAL_TIME
    reset_timer(Trace_timer);
#endif


#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa1000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
    trace(TRACE_ALL_EVENTS, (ULONG) sys_file_num);
    trace(TRACE_ALL_EVENTS, stream_handle);
#endif

        // set the new stream flag
    New_stream = TRUE;


#ifdef TRACE_BUFFER_DATA
    init_trace(TRACE_BUFFER_DATA);
#endif

        // get the stream pointer from the system file number - return error
        // if stream is not created
    if ((streamP = get_stream_from_sys_file_num(sys_file_num))
                                                             == (STREAM *) 0) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe1a00000);
#endif
        return(ERROR_STREAM_CREATION);
    }

        // return error if stream is already registered
    if (stream_registered(streamP)) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe1b00000);
#endif
        return(ERROR_HNDLR_REGISTERED);
    }

        // register the stream
    do_reg_stream(streamP, stream_handle);

        // store entry point into stream handler device driver 
    Shd_idc_entry_funcP = idc_entry_point;

       // initialize stream time
    set_stream_time(streamP, 0L);

        // return successful 
    return(NO_ERROR);
}


static VOID deregister_stream(STREAM *streamP)
//    This function deregisters a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.

{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa200000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // deregister the stream
    do_dereg_stream(streamP);

        // finished
    return ;
}


static VOID setup(STREAM *streamP, ULONG stream_time)
//    This function sets the time for a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// 'stream_time' is the new time for the stream.
//    The function does not return a value.
{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa3000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // set the time for this stream 
    set_stream_time(streamP, stream_time);

        // set the stream as active
    set_active_stream(streamP);

        // successful return 
    return ;
}


static ULONG control_resume(STREAM *streamP)
//    This function starts a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns the value 'ERROR_INVALID_SEQUENCE' if the stream is
// already started. Otherwise, the value 'NO_ERROR' is returned.
{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa4000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // return error if stream stopped
    if (get_stream_state(streamP) == STREAM_STATE_START) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe4000000);
#endif
        return(ERROR_INVALID_SEQUENCE);
    }

        // set the state for this stream
    set_stream_state(streamP, STREAM_STATE_START);

        // return successful
    return(NO_ERROR);
}


static ULONG control_pause(STREAM *streamP, ULONG *stream_time)
//    This function pauses a stream.
//    The parameter 'streamP' is a pointer to the stream.
//    The function returns the value 'ERROR_STREAM_NOT_STARTED' if the stream 
// is not started. Otherwise, the value 'NO_ERROR' is returned.
{
    CONTROL_PARM *control_parmP;

#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa5000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

    New_stream = TRUE;

        // return error if stream stopped 
    if (get_stream_state(streamP) == STREAM_STATE_STOP) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe4000000);
#endif
        return(ERROR_STREAM_NOT_STARTED);
    }

        // set the state for this stream
    set_stream_state(streamP, STREAM_STATE_PAUSE);

        // set up return info 
    *stream_time = get_stream_time(streamP);

        // successful return
    return(NO_ERROR);
}


static ULONG control_stop(STREAM *streamP, ULONG *stream_time)
//    This function stops a stream.
//    The parameter 'streamP' is a pointer to the stream. The current stream
// time is returned in the parameter 'stream_time".
//    The function returns the value 'ERROR_STREAM_NOT_STARTED' if the stream
// is not started. Otherwise, the value 'NO_ERROR' is returned.
{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa6000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

#ifdef TRACE_BUFFER_ARRIVAL_TIME
    trace(TRACE_BUFFER_ARRIVAL_TIME, 0xffffffff);
    trace(TRACE_BUFFER_ARRIVAL_TIME, 0xffffffff);
    trace(TRACE_BUFFER_ARRIVAL_TIME, 0xffffffff);
    trace(TRACE_BUFFER_ARRIVAL_TIME, 0xffffffff);
    reset_timer(Trace_timer);
#endif


#ifdef TRACE_BUFFER_DATA
    trace(TRACE_BUFFER_DATA, 0xffffffff);
    trace(TRACE_BUFFER_DATA, 0xffffffff);
    trace(TRACE_BUFFER_DATA, 0xffffffff);
    trace(TRACE_BUFFER_DATA, 0xffffffff);
#endif

    New_stream = TRUE;

        // return error if stream already stopped
    if (get_stream_state(streamP) == STREAM_STATE_STOP) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe6000000);
#endif
        return(ERROR_STREAM_NOT_STARTED);
    }

        // set the state for this stream 
    set_stream_state(streamP, STREAM_STATE_STOP);

        // set up return info
    *stream_time = get_stream_time(streamP);

        // successful return 
    return(NO_ERROR);
}


static VOID control_start(STREAM *streamP)
//    This function starts a stream.
//    The parameter 'streamP' is a pointer to the stream. 
//    The function does not return a value.
{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa7000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // return if stream already started
    if (get_stream_state(streamP) == STREAM_STATE_START) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe7000000);
#endif
        return ;
    }

        // set up the stream state
    set_stream_state(streamP, STREAM_STATE_START);

        // return successful
    return ;
}


static ULONG write_device(STREAM *streamP, UCHAR *bufP, USHORT buffer_size)
//    This function adds a new data buffer to a streams que of buffers.
//    The parameter 'streamP' is a pointer to the stream. The parameter 'bufP'
// is a pointer to the new buffer. The parameter 'buffer_size' is the size
// of the new buffer.
//    The function returns the value 'ERROR_TOO_MANY_BUFFERS' if the que for
// the stream is full. Otherwise, the value 'NO_ERROR' is returned.
{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa8000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // set the stream size if this is the first write
    if (New_stream) {
        set_stream_size(streamP, buffer_size);
        New_stream = FALSE;

#ifdef TRACE_BUFFER_IN
        trace(TRACE_BUFFER_IN, 0xffffffff);
        trace(TRACE_BUFFER_IN, 0xffffffff);
        trace(TRACE_BUFFER_IN, 0xffffffff);
        trace(TRACE_BUFFER_IN, 0xffffffff);
#endif

#ifdef TRACE_BUFFER_OUT
        trace(TRACE_BUFFER_OUT, 0xffffffff);
        trace(TRACE_BUFFER_OUT, 0xffffffff);
        trace(TRACE_BUFFER_OUT, 0xffffffff);
        trace(TRACE_BUFFER_OUT, 0xffffffff);
#endif

    }

#ifdef TRACE_BUFFER_ARRIVAL_TIME
    trace(TRACE_BUFFER_ARRIVAL_TIME, get_time(Trace_timer));
    trace(TRACE_BUFFER_ARRIVAL_TIME, (ULONG) bufP);
    trace(TRACE_BUFFER_ARRIVAL_TIME, (ULONG) buffer_size);
    trace(TRACE_BUFFER_ARRIVAL_TIME, (ULONG) *bufP);
    reset_timer(Trace_timer);
#endif

#ifdef TRACE_BUFFER_IN
    trace(TRACE_BUFFER_IN, (ULONG) bufP);
    trace(TRACE_BUFFER_IN, (ULONG) buffer_size);
#endif

#ifdef TRACE_BUFFER_DATA
    trace(TRACE_BUFFER_DATA, (ULONG) *bufP);
    trace(TRACE_BUFFER_DATA, (ULONG) *(bufP + 1));
    trace(TRACE_BUFFER_DATA, (ULONG) *(bufP + 2));
    trace(TRACE_BUFFER_DATA, (ULONG) *(bufP + 3));
#endif

        // add the buffer to the que
    if (add_buf_element_to_que_tail(streamP, bufP, buffer_size) != 0) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe8000000);
#endif
        return(ERROR_TOO_MANY_BUFFERS);
    }

#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, (ULONG) bufP);
    trace(TRACE_ALL_EVENTS, (ULONG) buffer_size);
#endif

        // finished
    return(NO_ERROR);
}


static ULONG read_device(STREAM *streamP, UCHAR *bufP, USHORT buffer_size)
//    This function adds a new data buffer to a streams que of buffers.
//    The parameter 'streamP' is a pointer to the stream. The parameter 'bufP'
// is a pointer to the new buffer. The parameter 'buffer_size' is the size
// of the new buffer.
//    The function returns the value 'ERROR_TOO_MANY_BUFFERS' if the que for
// the stream is full. Otherwise, the value 'NO_ERROR' is returned.
{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xa9000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // set the stream size is this is the first read
    if (New_stream) {
        set_stream_size(streamP, buffer_size);
        New_stream = FALSE;
    }

        // add buffer to que
    if (add_buf_element_to_que_tail(streamP, bufP, buffer_size) != 0) {
#ifdef TRACE_ALL_EVENTS
        trace(TRACE_ALL_EVENTS, 0xe9000000);
#endif
        return(ERROR_TOO_MANY_BUFFERS);
    }

#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, (ULONG) bufP);
    trace(TRACE_ALL_EVENTS, (ULONG) buffer_size);
#endif

#ifdef TRACE_BUFFER_IN
    trace(TRACE_BUFFER_IN, (ULONG) bufP);
    trace(TRACE_BUFFER_IN, (ULONG) buffer_size);
#endif

        // finished
    return(NO_ERROR);
}


static VOID status(STREAM *streamP, ULONG *stream_time)
//    This function gets the current stream time.
//    The parameter 'streamP' is a pointer to the stream. The time is returned
// in the parameter 'stream_time'.
//    The function does not return a value.

{
#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xaa000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // get the stream time
    *stream_time = get_stream_time(streamP);

#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, *stream_time);
#endif

        // finished
    return ;
}


VOID return_stream_buffer(STREAM *streamP, VOID *bufP, USHORT buf_size)
//    This function returns a buffer to the audio stream handler. 
//    The parameter 'streamP' is a pointer to the stream. The parameter 'bufP'
// is a pointer to the buffer. The parameter 'buf_size' is the size of the
// buffer being returned.
{
    SHD_REPORTINT buf;

#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, 0xffffffff);
    trace(TRACE_ALL_EVENTS, 0xab000000);
    trace(TRACE_ALL_EVENTS, get_time(Trace_timer));
    reset_timer(Trace_timer);
#endif

        // set up return info
    buf.ulFunction = SHD_REPORT_INT;
    buf.hStream = get_stream_handle(streamP);

    if (get_stream_operation(streamP) == STREAM_OPERATION_PLAY)
        buf.ulFlag = SHD_WRITE_COMPLETE;
    else
        buf.ulFlag = SHD_READ_COMPLETE;

    buf.pBuffer = bufP;
    buf.ulStatus = buf_size;
    buf.ulStreamTime = get_stream_time(streamP);

#ifdef TRACE_ALL_EVENTS
    trace(TRACE_ALL_EVENTS, (ULONG) buf.ulFlag);
    trace(TRACE_ALL_EVENTS, (ULONG) buf.ulStreamTime);
    trace(TRACE_ALL_EVENTS, (ULONG) buf.pBuffer);
    trace(TRACE_ALL_EVENTS, (ULONG) buf.ulStatus);
#endif

#ifdef TRACE_BUFFER_OUT
    trace(TRACE_BUFFER_OUT, (ULONG) buf.ulFlag);
    trace(TRACE_BUFFER_OUT, (ULONG) buf.ulStreamTime);
    trace(TRACE_BUFFER_OUT, (ULONG) buf.pBuffer);
    trace(TRACE_BUFFER_OUT, (ULONG) buf.ulStatus);
#endif

        // call the audio stream handler idc function
    Shd_idc_entry_funcP(&buf);

        // finished
    return;
}
