/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
// mem.c
//     This file contains functions related to dynamic memory allocation.


    // type definitions
#include "types.h"
#include "mem.h"
#include "util.h"
#include "devhlp.h"


    // debug stuff
#ifdef DEBUG_FLAG
#define static /**/
#endif


    // constant definitions
#define MAX_NUM_PHYS_ADDR 1
#define MAX_NUM_LIN_ADDR 99


    // type definitions
typedef struct {
    USHORT selector;
    ULONG aligned_address;
    ULONG base_address;
} PHYSICAL;

typedef struct {
    USHORT selector;
    ULONG address;
} LINEAR;


    // private global variables
static PHYSICAL Physical[MAX_NUM_PHYS_ADDR];
static LINEAR Linear[MAX_NUM_LIN_ADDR];


SHORT init_mem(VOID)
//    This function initializes memory services.
//    The function does not have parameters.
//    The function does not return a value.

{
    USHORT i, j;
    USHORT selector[MAX_NUM_PHYS_ADDR + MAX_NUM_LIN_ADDR];

        // allocate gdt selectors
    if (alloc_gdt_selector(selector, MAX_NUM_PHYS_ADDR + MAX_NUM_LIN_ADDR) != 0)
        return(-1);                                     

        // initialize physical and linear elements
    j = 0;
    for (i = 0; i <= MAX_NUM_PHYS_ADDR - 1; i++) {
        Physical[i].aligned_address = 0L;
        Physical[i].selector = selector[j++];
    }

    for (i = 0; i <= MAX_NUM_LIN_ADDR - 1; i++) {
        Linear[i].address = 0L;
        Linear[i].selector = selector[j++];
    }

        // successful - done
    return(0);
}


UCHAR *mem_alloc_32k_block_aligned(VOID)
//    This function allocates allocates a 64k block of memory alligned on a
// 64k boundary.
//    The function does not have parameters.
//    The function return the virtual address of the buffer if successful.
// Otherwise, null is returned.
{
    ULONG end_addr;
    USHORT i;

        // get free physical address element
    for (i = 0; i <= MAX_NUM_PHYS_ADDR - 1; i++)
        if (Physical[i].aligned_address == 0)
            break;

        // return null if no free elements
    if (Physical[i].aligned_address != 0)
        return ((UCHAR *) 0);

        // allocate 64k block of physical memory
    if ((Physical[i].base_address = alloc_phys((ULONG) 0x10000)) == (ULONG) 0)
        return ((UCHAR *) 0);

        // find the beginning of a 32k contiguous section
    end_addr = Physical[i].base_address + 0x8000;

    if ((Physical[i].base_address & 0xffff0000) != (end_addr & 0xffff0000))
        Physical[i].aligned_address = end_addr & 0xffff0000;
    else
        Physical[i].aligned_address = Physical[i].base_address;

        // map virtual selector to physical adddress
    if (phys_to_gdt_sel(Physical[i].aligned_address,
                                          Physical[i].selector, 0x8000) != 0)
        return((UCHAR *) 0);

        // return the 16:16 address
    return(make_far_ptr(Physical[i].selector, 0));
}


ULONG mem_virt_to_phys(UCHAR *bufP)
//    This function gets the physical address for a buffer that has been
// previously allocated.
//    The parameter 'bufP' is a pointer to the buffer.
//    The function returns the physical address if successful. Otherwise,
// null is returned.

{
    USHORT i;
    USHORT selector;

        // get the selector from the virtual address
    selector = (USHORT) ((ULONG) bufP >> 16);

        // find the selector
    for (i = 0; i <= MAX_NUM_PHYS_ADDR - 1; i++)
        if (Physical[i].selector == selector)
            break;

        // return null if not found
    if (Physical[i].selector != selector)
        return ((ULONG) 0);

        // return physical address
    return(Physical[i].aligned_address);
}


VOID mem_free(UCHAR *bufP)
//    This function frees a memory buffer that was previously allocated with
// mem_alloc.
//    The parameter 'bufP' is a pointer to the buffer.
//    The function does not return a value.

{
    USHORT i;
    USHORT selector;

        // get selector from the virtual address
    selector = (USHORT) ((ULONG) bufP >> 16);

        // search for selector in physical address storage
    for (i = 0; i <= MAX_NUM_PHYS_ADDR - 1; i++) {
        if (Physical[i].selector == selector) {
            free_phys(Physical[i].base_address);
            Physical[i].aligned_address = 0L;
            return;
        }
    }

        // search for selector in linear address storage
    for (i = 0; i <= MAX_NUM_LIN_ADDR - 1; i++) {
        if (Linear[i].selector == selector) {
            vm_free(Linear[i].address);
            Linear[i].address = 0L;
            return;
        }
    }

        // finished
    return;
}


UCHAR *mem_alloc(ULONG size)
//    This function dynamically allocates a memory buffer.
//    The parameter 'size' is the size of the buffer to allocate.
//    The function returns a pointer to the buffer if successful. Otherwise,
// null is returned.

{
    USHORT i;

        // get free linear address element
    for (i = 0; i <= MAX_NUM_LIN_ADDR - 1; i++)
        if (Linear[i]. address == 0)
            break;

        // return null if no free elements
    if (Linear[i].address != 0)
        return ((UCHAR *) 0);

        // allocate memory
    if ((Linear[i].address = vm_alloc(size)) == 0)
        return((UCHAR *) 0);

        // map virtual selector to physical adddress
    if (lin_to_gdt_selector(Linear[i].address, Linear[i].selector, size) != 0)
        return((UCHAR *) 0);

        // return 16:16 address
    return(make_far_ptr(Linear[i].selector, 0));
}
