/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
// dma.c
//    This file contains routines that support dma services.


    // include files
#include "types.h"
#include "dma.h"
#include "util.h"


    // debug info 
#ifdef DEBUG_FLAG
#define static /**/
#endif


    // constant definitions
#define FLIP_FLOP_PORT 0x0c
#define MASK_PORT 0x0a
#define MODE_PORT 0x0b

#define CHAN0_ADDR_PORT 0x00
#define CHAN1_ADDR_PORT 0x02
#define CHAN3_ADDR_PORT 0x06

#define CHAN0_COUNT_PORT 0x01
#define CHAN1_COUNT_PORT 0x03
#define CHAN3_COUNT_PORT 0x07

#define CHAN0_PAGE_PORT  0x87
#define CHAN1_PAGE_PORT 0x83
#define CHAN3_PAGE_PORT  0x82


    // private global variables
static ULONG Buffer_size;
static UCHAR Channel;


VOID init_dma(UCHAR channel, ULONG phys_addr, ULONG buffer_size,
                                                               SHORT operation)
//    This function initializes dma services.
//    The parameter 'channel' is the number of the dma channel to initialize.
// The parameter 'buffer_size' is the size of the dma buffer. The parameter
// 'operation' specifies read or write operation for the channel.
//    The function does not return a value.
{
    UCHAR buffer_page, buffer_offset_low, buffer_offset_high;
    UCHAR addr_port, count_port, page_port;
    UCHAR buffer_size_low, buffer_size_high;
    USHORT i;

        // store the channel
    Channel = channel;

        // store the buffer size
    Buffer_size = buffer_size;

        // get channel specific ports
    switch (channel) {
        case 0:
            addr_port  = CHAN0_ADDR_PORT;
            count_port = CHAN0_COUNT_PORT;
            page_port  = CHAN0_PAGE_PORT;
            break;
        case 1:
            addr_port  = CHAN1_ADDR_PORT;
            count_port = CHAN1_COUNT_PORT;
            page_port  = CHAN1_PAGE_PORT;
            break;
        case 3:
            addr_port = CHAN3_ADDR_PORT;
            count_port = CHAN3_COUNT_PORT;
            page_port = CHAN3_PAGE_PORT;
            break;
        default:
            return;
    }

        // get the page and offset info for the dma buffer
    buffer_page = (UCHAR)(phys_addr >> 16);
    buffer_offset_high = (UCHAR)(phys_addr >> 8);
    buffer_offset_low = (UCHAR) phys_addr;
    buffer_size_high = (UCHAR) ((buffer_size - 1) >> 8);
    buffer_size_low = (UCHAR) (buffer_size - 1);

        // disable dma controller
    outportb(MASK_PORT, (UCHAR)(channel | 0x04));

        // set the mode info
            // bit 7 - 0
            // bit 6 - 1 single mode
            // bit 5 - 0 address increment
            // bit 4 - 1 auto-initialize mode
            // bit 3 - 0          1
            // bit 2 - 1 write    0 read
            // bit 1 - 0          0          1          1
            // bit 0 - 0 chan 0   1 chan 1   0 chan 2   1 chan 3
            // 0101 1000 = 58
            // 0101 0100 = 54
    if (operation == DMA_READ)
        outportb(MODE_PORT,(UCHAR) (channel | 0x58));
    else
        outportb(MODE_PORT,(UCHAR) (channel | 0x54));

        // set address page of DMA buffer
    outportb(page_port, buffer_page);

        // clear the flip-flop
    outportb(FLIP_FLOP_PORT, 0);

        // set low byte of DMA buffer address offset
    outportb(addr_port, buffer_offset_low);

        // set high byte of DMA buffer address offset
    outportb(addr_port, buffer_offset_high);

        // clear the flip-flop
    outportb(FLIP_FLOP_PORT, 0);

        // output low byte of DMA buffer size
    outportb(count_port, buffer_size_low);

        // output high byte of DMA buffer size
    outportb(count_port, buffer_size_high);

        // enable the dma controller
    outportb(MASK_PORT, channel);

        // finished
    return;
}


USHORT get_dma_pos(VOID)
//    This function gets the current dma position.
//    The function does not have parameters.
//    The function returns the dma buffer.
{
    USHORT count;
    UCHAR count_port, count_low, count_high;

        // store channel specific ports
    switch (Channel) {
        case 0:
            count_port = CHAN0_COUNT_PORT;
            break;
        case 1:
            count_port = CHAN1_COUNT_PORT;
            break;
        case 3:
            count_port = CHAN3_COUNT_PORT;
            break;
    }

        // clear the flip-flop
    outportb(FLIP_FLOP_PORT, 0);

        // get low byte of count
    count_low = inportb(count_port);

        // get high byte of count
    count_high = inportb(count_port);

        // set up count
    count = count_high;
    count <<= 8;
    count |= count_low;

        // return the position
    return(Buffer_size - count);
}


VOID disable_dma(UCHAR channel)
//    This function disables a particular channel in the dma controller.
//    The parameter 'channel' is the channel to disable.
//    The function does not return a value.
{
        // disable the channel
    outportb(MASK_PORT, (UCHAR) (channel | 0x04));

        // finished
    return;
}
