/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
// audcont.c
//    This file contains functions that support audio control commands from
// the DosDevIoctl interface to the device driver.


    // include files
#include "types.h"
#include "audcont.h"
#include "strmhelp.h"
#include "device.h"
#include <audio.h>


    // debug info
#ifdef DEBUG_FLAG
#define static /**/
#endif


    // prototypes for private functions 
VOID audio_init(STREAM *, SHORT, USHORT, SHORT, UCHAR, UCHAR);
VOID audio_status(STREAM *);
VOID audio_change(STREAM *, ULONG, USHORT, ULONG, SHORT, SHORT, ULONG);
VOID audio_start(STREAM *);
VOID audio_stop(STREAM *);
VOID audio_pause(STREAM *);
VOID audio_resume(STREAM *);
VOID foo(VOID);
VOID goo(VOID);


SHORT do_audio_control_command(USHORT function,
                                      VOID *bufP, USHORT sys_file_num)
//    This function services all audio control requests from the ioctl
// interface.
//    The parameter 'function' is number of the request. The parameter 'bufP'
// is a pointer to a buffer containing information specific to the request.
// The parameter 'sys_file_num' contains the system file number unique to
// the session issuing the ioctl.
//    The function returns information specific to the request in the buffer
// pointed to by 'bufP'. Also, the function returns the value 0 if successful.
// Otherwise, the function returns the value -1.

{
    MCI_AUDIO_INIT *audio_init_bufP;
    MCI_AUDIO_STATUS *audio_status_bufP;
    MCI_AUDIO_CONTROL *audio_control_bufP;
    MCI_AUDIO_CHANGE *audio_change_bufP;
    MCI_TRACK_INFO *track_info_bufP;
    USHORT master_volume_level;
    STREAM *streamP;

        // get the stream pointer from the system file number
    if ((streamP = get_stream_from_sys_file_num(sys_file_num)) == (STREAM *) 0)
        return (-1);

    switch (function) {

            // service an audio init request
        case AUDIO_INIT:
            audio_init_bufP = (MCI_AUDIO_INIT *) bufP;

            audio_init_bufP->pvReserved = (VOID *) sys_file_num;

            audio_init_bufP->sReturnCode = 0;

            audio_init(streamP, (SHORT) audio_init_bufP->ulOperation,
                              (USHORT) audio_init_bufP->lSRate,
                                   audio_init_bufP->sMode,
                                       (UCHAR) audio_init_bufP->lBitsPerSRate,
                                           (UCHAR) audio_init_bufP->sChannels);

            return (0);
 
            // service an audio status request
        case AUDIO_STATUS:
            audio_status_bufP = (MCI_AUDIO_STATUS *) bufP;
            audio_status(streamP);
            return (0);

            // service an audio control request
        case AUDIO_CONTROL:
            audio_control_bufP = (MCI_AUDIO_CONTROL *) bufP;

            switch (audio_control_bufP->usIOCtlRequest) {

                case AUDIO_CHANGE:
                    audio_change_bufP = audio_control_bufP->pbRequestInfo;
                    track_info_bufP =
                         (MCI_TRACK_INFO *) audio_change_bufP->pvDevInfo;

                    if (track_info_bufP != (MCI_TRACK_INFO *) 0)
                        master_volume_level = track_info_bufP->usMasterVolume;
                    else
                        master_volume_level = 0xffff;

                    audio_control_bufP->sReturnCode = 0;

                    audio_change(streamP,
                          (ULONG) audio_change_bufP->lVolume,
                             master_volume_level,
                                (ULONG) audio_change_bufP->lMonitor,
                                  (SHORT) audio_change_bufP->
                                                       rInputList[0].ulDevType,
                                     (SHORT) audio_change_bufP->
                                                      rOutputList[0].ulDevType,
                                          (ULONG) audio_change_bufP->lGain);
                    return (0);

                case AUDIO_START:
                    audio_start(streamP);
                    audio_control_bufP->sReturnCode = 0;
                    return (0);

                case AUDIO_STOP:
                    audio_stop(streamP);
                    audio_control_bufP->sReturnCode = 0;
                    return (0);

                case AUDIO_PAUSE:
                    audio_pause(streamP);
                    audio_control_bufP->sReturnCode = 0;
                    return (0);

                case AUDIO_RESUME:
                    audio_resume(streamP);
                    audio_control_bufP->sReturnCode = 0;
                    return (0);

                default:
                    return (-1);
            }

         default:
            return (-1);
    }
}


VOID foo(VOID)
{
#define DEBUGA2
#ifdef DEBUGA2
_asm push ax
_asm int 3
_asm mov ax,0a2h
_asm pop ax
#endif
}


VOID goo(VOID)
{
#define DEBUGA3
#ifdef DEBUGA3
_asm push ax
_asm int 3
_asm mov ax,0a3h
_asm pop ax
#endif
}


SHORT audio_open(USHORT sys_file_num)
//    This function creates an audio stream enviroment for a session performing
// an open of the device driver.
//    The parameter 'sys_file_num' is the system file number unique to the
// session performing the open.
//    The function returns the value 0 if successful. Otherwise, the value -1
// is returned.

{
        // create the stream enviroment
    return(create_stream(sys_file_num));
}


SHORT audio_close(USHORT sys_file_num)
//    This function closes an audio stream enviroment for a session performing
// an close of the device driver.
//    The parameter 'sys_file_num' is the system file number unique to the
// session performing the open.
//    The function returns the value 0 if successful. Otherwise, the value -1
// is returned.

{
    STREAM *streamP;

        // get stream pointer from system file number
    if ((streamP = get_stream_from_sys_file_num(sys_file_num)) == (STREAM *) 0)
        return(-1);

        // destroy the stream
    destroy_stream(streamP);

        // finished
    return(0);
}


static VOID audio_init(STREAM *streamP, SHORT operation, USHORT sample_rate,
            SHORT data_format, UCHAR bits_per_sample, UCHAR num_of_channels)
//    This function set up initialization info for a particular stream.
//    The parameter 'streamP' is a pointer to the stream. The parameter
// ' operation' indicates a play or record operation for the stream. The
// parameter 'sample_rate' is the pcm sample rate for the stream in kHz. The
// parameter '
// 'data_format' indicates either a pcm, a-law, or mu-law format for the
// stream. The parameter 'bits_per_sample' indicates either an 8 or 16 bit
// sample data size. The parameter 'num_of_channels' indicates either mono
// or stereo mode.
//    The function does not return a value.
{
    SHORT stream_data_format;

        // set up operation
    switch (operation) {
        case OPERATION_RECORD:
            set_stream_operation(streamP, STREAM_OPERATION_RECORD);
            break;
        case OPERATION_PLAY:
            set_stream_operation(streamP, STREAM_OPERATION_PLAY);
            break;
        default:
            return;
    }

       // set up the clock and data format 
    switch (data_format) {
        case PCM:
            stream_data_format = STREAM_DATA_FORMAT_PCM;
            break;
        case MU_LAW:
            stream_data_format = STREAM_DATA_FORMAT_MU_LAW;
            break;
        case A_LAW:
            stream_data_format = STREAM_DATA_FORMAT_A_LAW;
            break;
        default: 
            return;
    }

        // set up bits per sample
    if ((bits_per_sample != 8) && (bits_per_sample != 16))
        return;

        // set up number of channels
    if ((num_of_channels != 1) && (num_of_channels != 2))
        return;

        // set the stream info
    set_stream_clock_and_format(streamP, sample_rate, 
                   stream_data_format, bits_per_sample, num_of_channels);

        // finished
    return;
}


static VOID audio_status(STREAM *streamP)
{
    return;
}


static VOID audio_change(STREAM *streamP, ULONG volume_level,
         USHORT master_volume_level, ULONG monitor_level,
              SHORT input_device_type, SHORT output_device_type,
                    ULONG input_gain_level)
//    This function changes the operational characteristics of a stream.
//    The parameter 'streamP' is a pointer to the stream. The parameters
// 'volume_level' and 'master_volume_level' describe the volume output level.
// The parameter 'monitor_level' describes the monitor output level. The
// parameters 'input_device_type' and 'output_device_type' describes the
// input and output paths. The parameter 'input_gain_level' describes the level
// of the input gain.
//    The function does not return a value.
{
        // set master volume
    if (master_volume_level != 0xffff)
        set_dev_master_volume(master_volume_level);

        // set the stream volume
    if (volume_level != 0xffffffff) {
        if (volume_level <= 0x7fffffff)
            set_stream_volume(streamP, volume_level);
    }

        // set the input monitor level
    if (monitor_level != 0xffffffff) {
        if (monitor_level == 0)
            set_dev_monitor(OFF);
        else
            set_dev_monitor(ON);
    }

        /* set list of input devices */
    if (input_device_type != -1) {
        switch (input_device_type) {
            case STEREO_LINE_INPUT:
            case LEFT_LINE_INPUT:
            case RIGHT_LINE_INPUT:
                set_stream_input_device(streamP, STREAM_INPUT_STEREO_LINE);
                break;
            case MIC_INPUT:
                set_stream_input_device(streamP, STREAM_INPUT_MIC);
                break;
            case BOOSTED_MIC_INPUT:
                set_stream_input_device(streamP, STREAM_INPUT_BOOSTED_MIC);
                break;
            case NULL_INPUT:
            case PHONE_LINE_INPUT:
            case HANDSET_INPUT:
            case SYNTH_INPUT:
            case DIGITAL_PHONE_LINE_INPUT:
            case DIGITAL_HANDSET_INPUT:
            case MIDI_IN_PORT:
                set_stream_input_device(streamP, STREAM_INPUT_NULL);
                break;
            default:
                break;
        }
    }

        /* set list of output devices */
    if (output_device_type != -1) {
        switch (output_device_type) {
            case STEREO_LINE_OUTPUT:
            case LEFT_LINE_OUTPUT:
            case RIGHT_LINE_OUTPUT:
            case SPEAKER_OUTPUT:
            case HEADSET_OUTPUT:
                set_stream_output_device(streamP, STREAM_OUTPUT_STEREO_LINE);
                break;
            case NULL_OUTPUT:
            case PHONE_LINE_OUTPUT:
            case HANDSET_OUTPUT:
            case SYNTH_OUTPUT:
            case DIGITAL_PHONE_LINE_OUTPUT:
            case DIGITAL_HANDSET_OUTPUT:
            case MIDI_OUT_PORT:
                set_stream_output_device(streamP, STREAM_OUTPUT_NULL);
                break;
            default:
                break;
        }
    }

        /* set input level control */
    if (input_gain_level != 0xffffffff) {
        if (input_gain_level <= 0x7fffffff)
            set_stream_input_level(streamP, input_gain_level);
    }

        // return successful
    return;
}


static VOID audio_start(STREAM *streamP)
//    This fuction sets the state of a stream to start.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.
{
        // update the current status
    set_stream_state(streamP, STREAM_STATE_START);

        // finished 
    return;
}


static VOID audio_stop(STREAM *streamP)
//    This fuction sets the state of a stream to stop.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.
{
        // update the current status
    set_stream_state(streamP, STREAM_STATE_STOP);

        // finished
    return;
}


static VOID audio_pause(STREAM *streamP)
//    This fuction sets the state of a stream to pause.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.
{
        // update the current status
    set_stream_state(streamP, STREAM_STATE_PAUSE);

        // finished 
    return;

}


static VOID audio_resume(STREAM *streamP)
//    This fuction sets the state of a stream to resume.
//    The parameter 'streamP' is a pointer to the stream.
//    The function does not return a value.
{
        // update the current status
    set_stream_state(streamP, STREAM_STATE_START);

        // finished 
    return;
}
