/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
// ad1848.c
//    This file contains functions that interface with the Analog Device
// stereo codec AD1848.


    // include files 
#include "types.h"
#include "dma.h"
#include "device.h"
#include "util.h"
#include "strmhelp.h"
#include "mem.h"
#include "timer.h"
#include "echo.h"
#ifdef DEBUG_FLAG
#include "trace.h"
#endif


    // debug info 
#ifdef DEBUG_FLAG
#define static /**/
#endif


    // constant definitions 
#define MAX_NUM_IO_RANGES 5
#define FREQ_TABLE_SIZE 14

#define NUM_OF_DMA_CHANNELS 1

#define LEFT_INPUT_CONTROL_REG 0                      // indirect registers 
#define RIGHT_INPUT_CONTROL_REG 1
#define LEFT_AUX1_INPUT_CONTROL_REG 2
#define RIGHT_AUX1_INPUT_CONTROL_REG 3
#define LEFT_AUX2_INPUT_CONTROL_REG 4
#define RIGHT_AUX2_INPUT_CONTROL_REG 5
#define LEFT_DAC_CONTROL_REG 6
#define RIGHT_DAC_CONTROL_REG 7
#define CLOCK_AND_DATA_FORMAT_REG 8
#define INTERFACE_CONFIGURATION_REG 9
#define PIN_CONTROL_REG 10
#define TEST_AND_INITIALIZATION_REG 11
#define MISCELLANEOUS_CONTROL_REG 12
#define DIGITAL_MIX_CONTROL_REG 13
#define UPPER_BASE_COUNT_REG 14
#define LOWER_BASE_COUNT_REG 15

#define J_GRADE_ID_NUM 0x09
#define K_GRADE_ID_NUM 0x0a

#define LEVEL_TABLE_SIZE 100

#define RM_LEVEL_TABLE_SIZE 101

#define TRANSFER_BUF_SIZE 0x08000                    // transfer buffer sizes


    // type definitions 
typedef struct {
    USHORT freq;
    UCHAR clock_source;
    UCHAR divide_select;
} FREQ_TABLE_TYPE;

typedef struct {
    USHORT base_port;
    USHORT num_ports;
} IO_RANGE;


    // prototypes for private functions
UCHAR get_dev_revision_info(VOID);
VOID set_reg(UCHAR, UCHAR);
UCHAR get_reg(UCHAR);
VOID get_clock_info(USHORT, UCHAR *, UCHAR *);
VOID get_format_info(SHORT, USHORT, UCHAR *, UCHAR *);
VOID set_dev_interrupts(BOOL);
VOID set_dev_left_output_mute(BOOL);
VOID set_dev_right_output_mute(BOOL);
VOID set_dev_left_input_source(UCHAR);
VOID set_dev_right_input_source(UCHAR);
VOID set_dev_digital_mix_mute(BOOL);
VOID set_dev_digital_mix_atten(UCHAR);
VOID set_dev_left_output_atten(UCHAR);
VOID set_dev_right_output_atten(USHORT);

VOID set_dev_left_aux1_loop_mute(BOOL);
VOID set_dev_right_aux1_loop_mute(BOOL);

VOID set_dev_left_aux1_loop_atten(UCHAR);
VOID set_dev_right_aux1_loop_atten(UCHAR);

VOID set_dev_left_aux2_loop_mute(BOOL);
VOID set_dev_right_aux2_loop_mute(BOOL);

VOID set_dev_left_aux2_loop_atten(UCHAR);
VOID set_dev_right_aux2_loop_atten(UCHAR);
ULONG scale_by_master_volume(ULONG);


    // private variable declarations 
static FREQ_TABLE_TYPE Freq_table[FREQ_TABLE_SIZE] =
    {5512, 1, 0,
     6615, 1, 7,
     8000, 0, 0,
     9600, 0, 7,
    11025, 1, 1,
    16000, 0, 1,
    18900, 1, 2,
    22050, 1, 3,
    27428, 0, 2,
    32000, 0, 3,
    33075, 1, 6,
    37800, 1, 4,
    44100, 1, 5,
    48000, 0, 6};

static UCHAR Output_atten_table[LEVEL_TABLE_SIZE] = {
                    63, 57, 52, 48, 45, 43, 41, 40, 39, 38,
                    37, 36, 35, 34, 33, 32, 31, 30, 29, 28,
                    27, 26, 25, 24, 23, 22, 21, 21, 20, 20,
                    19, 19, 18, 18, 17, 17, 17, 16, 16, 15,
                    15, 14, 14, 13, 13, 12, 12, 11, 11, 10,
                    10,  9,  9,  8,  8,  7,  7,  7,  6,  6,
                     6,  5,  5,  5,  5,  4,  4,  4,  4,  4,
                     3,  3,  3,  3,  3,  3,  2,  2,  2,  2,
                     2,  2,  2,  1,  1,  1,  1,  1,  1,  1,
                     1,  0,  0,  0,  0,  0,  0,  0,  0,  0};

static UCHAR Generic_input_gain_table[LEVEL_TABLE_SIZE] = {
                     0,  0,  0,  0,  0,  0,  1,  1,  1,  1,
                     1,  1,  1,  2,  2,  2,  2,  2,  2,  3,
                     3,  3,  3,  3,  3,  3,  4,  4,  4,  4,
                     4,  4,  4,  5,  5,  5,  5,  5,  5,  6,
                     6,  6,  6,  6,  6,  7,  7,  7,  7,  7,
                     7,  8,  8,  8,  8,  8,  8,  9,  9,  9,
                     9,  9,  9,  9, 10, 10, 10, 10, 10, 10,
                    11, 11, 11, 11, 11, 11, 12, 12, 12, 12,
                    12, 12, 13, 13, 13, 13, 13, 13, 14, 14,
                    14, 14, 14, 14, 15, 15, 15, 15, 15, 15};

static UCHAR Toshiba_input_gain_table[LEVEL_TABLE_SIZE] = {
                     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
                     0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
                     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
                     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
                     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
                     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
                     3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
                     5,  5,  5,  5,  5,  5,  5,  5,  5,  5,
                     7,  7,  7,  7,  7,  7,  7,  7,  7,  7,
                     7,  7,  7,  7,  7,  7,  7,  7,  7,  7};

static UCHAR Clock_and_data_format_reg_value;
static UCHAR Play_dma_channel, Record_dma_channel;
static USHORT Transfer_buf_pos;
static UCHAR Irq_num;
static UCHAR *Transfer_bufP;
static ULONG Transfer_buf_phys_addr;
static UCHAR Monitor_state = 0;
static USHORT Volume_up_reg, Sound_source_config_reg;
static USHORT Base_io_addr, Index_reg, Index_data_reg, Status_reg;
static USHORT Dev_buffer_size;
static SHORT Device_type;
static UCHAR *Input_gain_table;
static UCHAR Left_gain_level, Right_gain_level;
static ULONG Left_gain_setting, Right_gain_setting;
static UCHAR Master_volume = 0x7f;
static ULONG Left_volume_setting = 0;
static ULONG Right_volume_setting = 0;
static USHORT Old_dma_pos;

static SHORT Last_data_format;
static UCHAR Last_bits_per_sample;
static USHORT Last_sample_rate;
static UCHAR Last_num_of_channels;
static SHORT Last_dev_operation;
static BOOL New_configuration;

static USHORT Num_io_ranges = 0;
static IO_RANGE Io_range[MAX_NUM_IO_RANGES];
static UCHAR Silence_value;


VOID clear_dev_data(USHORT buf_size)
//    This function clears data from the DMA data transfer buffer by inserting
// the silence value.
//    The parameter 'dest_bufP' is a pointer to the input buffer. The parameter
// 'buf_size' is the size of the input buffer.
//    The function returns the number of bytes retrieved from the DMA transfer
// buffer.
{
    USHORT buffer_left;
    UCHAR *dest_bufP;

    if (buf_size == 0)
        return;

        // get the data - wrap if necessary
    if (Transfer_buf_pos == TRANSFER_BUF_SIZE)
        Transfer_buf_pos = 0;

        // point to the dma buffer
    dest_bufP = Transfer_bufP;

        // calculate room left to the end
    buffer_left = TRANSFER_BUF_SIZE - Transfer_buf_pos;

    if (buffer_left < buf_size) {
        memfill(dest_bufP + Transfer_buf_pos, buffer_left, Silence_value);
        buf_size -= buffer_left;
        memfill(dest_bufP, buf_size, Silence_value);
        Transfer_buf_pos = buf_size;
    }
    else {
        memfill(dest_bufP + Transfer_buf_pos, buf_size, Silence_value);
        Transfer_buf_pos += buf_size;
    }

        // finished
    return ;
}


VOID send_dev_data(UCHAR *src_bufP, USHORT buf_size)
//    This function transfers data from an input data buffer to the DMA data
//   transfer buffer.
//      The parameter 'src_bufP' is a pointer to the input data buffer. The
// parameter 'buf_size' is the size of the input buffer.
//      The function does not return a value.
{

    USHORT buffer_left;
    UCHAR *dest_bufP;

    if (buf_size == 0)
        return;

        // reset the position if at the end
    if (Transfer_buf_pos == TRANSFER_BUF_SIZE)
        Transfer_buf_pos = 0;

        // point to the dma buffer
    dest_bufP = Transfer_bufP;

        // calculate room left to the end
    buffer_left = TRANSFER_BUF_SIZE - Transfer_buf_pos;

        // fill the buffer - wrap around to the beginning if necessary
    if (buffer_left < buf_size) {
        memcopy(src_bufP, dest_bufP + Transfer_buf_pos, buffer_left);
        src_bufP += buffer_left;
        buf_size -= buffer_left;
        memcopy(src_bufP, dest_bufP, buf_size);
        Transfer_buf_pos = buf_size;
    }
    else {
        memcopy(src_bufP, dest_bufP + Transfer_buf_pos, buf_size);
        Transfer_buf_pos += buf_size;
    }


        // finished
    return;
}


VOID get_dev_data(UCHAR *dest_bufP, USHORT buf_size)
//    This function transfers data from the DMA data transfer buffer to an
// input buffer.
//    The parameter 'dest_bufP' is a pointer to the input buffer. The parameter
// 'buf_size' is the size of the input buffer.
//    The function returns the number of bytes retrieved from the DMA transfer
// buffer.
{
    USHORT buffer_left;
    UCHAR *src_bufP;

        // get the data - wrap if necessary
    if (Transfer_buf_pos == TRANSFER_BUF_SIZE)
        Transfer_buf_pos = 0;

    src_bufP = Transfer_bufP;

    buffer_left = TRANSFER_BUF_SIZE - Transfer_buf_pos;

    if (buffer_left < buf_size) {
        memcopy(src_bufP + Transfer_buf_pos, dest_bufP, buffer_left);
        dest_bufP += buffer_left;
        buf_size -= buffer_left;
        memcopy(src_bufP, dest_bufP, buf_size);
        Transfer_buf_pos = buf_size;
    }
    else {
        memcopy(src_bufP + Transfer_buf_pos, dest_bufP, buf_size);
        Transfer_buf_pos += buf_size;
    }

        // finished
    return ;
}


USHORT get_dev_ad1848_num_io_ranges(VOID)
//    This function determines the number of i/o address ranges accessed by
// the ad1848 portion of the driver.
//    The function does not have parameters.
//    The function returns the number of ranges.
{
    return (Num_io_ranges);
}


VOID get_dev_ad1848_io_range(USHORT range_num,
                                    USHORT *base_port, USHORT *num_ports)
//    This function gets a particular io range nad the number of ports in that
// range.
//    The parameter 'range_num' is the number of the range.
//    The function returns the first port address and the number of ports in 
// range in the parameters 'base_port' and 'num_ports'.
{
    *base_port = Io_range[range_num - 1].base_port;
    *num_ports = Io_range[range_num -1].num_ports;

    return;
}


SHORT set_dev_base_io_addr(USHORT addr)
//    This function stores the value for the device's base i/o address.
//    The parameter 'addr' is the base i/o address.
//    The function returns the value -1 if the i/o address is not valid
// for the device. Otherwise, the function returns the value 0.
{
    switch (Device_type) {
        case DEV_TOSHIBA_T4700CS:
        case DEV_TOSHIBA_T6600C:
            if ((addr != 0x530) && (addr != 0x604) && (addr != 0xe80))
                return(-1);
            Sound_source_config_reg = addr;
            Index_reg = addr + 4;
            Index_data_reg = addr + 5;
            Status_reg = addr + 6;
            Base_io_addr = addr;
 
            Io_range[Num_io_ranges].base_port = addr;
            Io_range[Num_io_ranges].num_ports = 7;

            Num_io_ranges++;

            return(0);

        case DEV_MS_SOUND:
        case DEV_COMPAQ:
        case DEV_SOUNDWAVE_32:
            if ((addr != 0x530) && (addr != 0x604) &&
                                           (addr != 0xe80) && (addr != 0xf40))
                return(-1);
            Sound_source_config_reg = addr;
            Index_reg = addr + 4;
            Index_data_reg = addr + 5;
            Status_reg = addr + 6;
            Base_io_addr = addr;

            Io_range[Num_io_ranges].base_port = addr;
            Io_range[Num_io_ranges].num_ports = 7;

            Num_io_ranges++;

            return(0);

        case DEV_THINKPAD :
            if ((addr != 0x0030) && (addr != 0x4e30) &&
                                         (addr != 0x8e30) && (addr != 0xce30))
                return(-1);
            Index_reg = addr + 0;
            Index_data_reg = addr + 1;
            Status_reg = addr + 2;
            Volume_up_reg = addr + 0x0f;
            Base_io_addr = addr;

            Io_range[Num_io_ranges].base_port = addr;
            Io_range[Num_io_ranges].num_ports = 3;

            Num_io_ranges++;

            Io_range[Num_io_ranges].base_port = Volume_up_reg;
            Io_range[Num_io_ranges].num_ports = 1;

            Num_io_ranges++;

                // store power management register
            Io_range[Num_io_ranges].base_port = 0x15e8;
            Io_range[Num_io_ranges].num_ports = 2;

            Num_io_ranges++;

            return(0);
    }
}


USHORT get_dev_num_bytes_consumed(VOID)
//    This function gets the number of bytes consumed by the device between
// calls.
//    The function does not have parameters.
//    The function returns the number of bytes consumed.
{
    USHORT bytes_consumed, new_dma_pos;

        // get the current dma pos
    new_dma_pos = get_dma_pos();

#ifdef TRACE_DMA_POS
    trace(TRACE_DMA_POS, (ULONG) new_dma_pos);
#endif

        // calculate the bytes consumed
    if (new_dma_pos < Old_dma_pos) {
        bytes_consumed = TRANSFER_BUF_SIZE - Old_dma_pos;
        bytes_consumed += new_dma_pos;
    }
    else
        bytes_consumed = new_dma_pos - Old_dma_pos;

        // store the dma pos
    Old_dma_pos = new_dma_pos;

        // return the count
    return (bytes_consumed);
}


USHORT get_dev_num_bytes_produced(VOID)
//    This function gets the number of bytes produced by the device between
// calls.
//    The function does not have parameters.
//    The function returns the number of bytes read.
{
    USHORT bytes_produced, new_dma_pos;

        // get the current dma pos
    new_dma_pos = get_dma_pos();

        // calculate the bytes consumed
    if (new_dma_pos < Old_dma_pos) {
        bytes_produced = TRANSFER_BUF_SIZE - Old_dma_pos;
        bytes_produced += new_dma_pos;
    }
    else
        bytes_produced = new_dma_pos - Old_dma_pos;

        // store the dma pos
    Old_dma_pos = new_dma_pos;

        // return the count
    return (bytes_produced);
}


SHORT init_device(VOID)
//    This function performs initialization activities.
//    The function does not have parameters.
//    The function returns the value 0 if successful. Otherwise, the value -1
// is returned.
{
    UCHAR revision_id, reg_value;
    USHORT i;

#ifdef TRACE_DMA_POS
    init_trace(TRACE_DMA_POS);
#endif

        // initialize last configuration info
    Last_data_format = 0;
    Last_bits_per_sample = 0;
    Last_sample_rate = 0;
    Last_num_of_channels = 0;
    Last_dev_operation = DEV_OPERATION_OFF;
    New_configuration = TRUE;

        // enable the echo device if appropriate
    if (Device_type == DEV_SOUNDWAVE_32)
        enable_echo_asic(Base_io_addr);

    switch (Device_type) {

        case DEV_MS_SOUND:
        case DEV_COMPAQ:
        case DEV_TOSHIBA_T6600C:
        case DEV_SOUNDWAVE_32:

                // set up input gain table
            Input_gain_table = Generic_input_gain_table;

                // set configuration register
            reg_value = 0x00;
            switch (Irq_num) {
                case 7:
                    reg_value |= 0x08;
                    break;
                case 9:
                    reg_value |= 0x10;
                    break;
                case 10:
                    reg_value |= 0x18;
                    break;
                case 11:
                    reg_value |= 0x20;
                    break;
                default:
                    break;
             }
             switch (Play_dma_channel) {
                case 0:
                    reg_value |= 0x01;
                    break;
                case 1:
                    reg_value |= 0x02;
                    break;
                case 3:
                    reg_value |= 0x03;
                    break;
             }
             outportb(Sound_source_config_reg, reg_value);

                // wait for initialization to end
            for (i = 1; i <= 10; i++) {
                delay_ms(50L);
                reg_value = inportb(Index_reg);
                if ((reg_value & 0x80) == 0x00)
                    break;
            }
            if ((reg_value & 0x80) != 0x00)
                return(-1);

                // check for card presence by reading hardware revision number
            revision_id = get_dev_revision_info();
            if ((revision_id != J_GRADE_ID_NUM) &&
                                           (revision_id != K_GRADE_ID_NUM))
                return(-1);

            break;

        case DEV_THINKPAD:

                // set up input gain table
            Input_gain_table = Generic_input_gain_table;

                // set power management register
            outportb(0x15e8, 0x1c);
            reg_value = inportb(0x15e9);
            reg_value |= 0x02;
            outportb(0x15e8, 0x1c);
            outportb(0x15e9, reg_value);

                // wait for initialization to end
            for (i = 1; i <= 10; i++) {
                delay_ms(50L);
                reg_value = inportb(Index_reg);
                if ((reg_value & 0x80) == 0x00)
                    break;
            }
            if ((reg_value & 0x80) != 0x00)
                return(-1);

                // initialize volume control register
            outportb(Volume_up_reg, 0x01);

            break;


        case DEV_TOSHIBA_T4700CS:

                // set up input gain table
            Input_gain_table = Toshiba_input_gain_table;

                // set configuration register
            reg_value = 0x00;
            switch (Irq_num) {
                case 7:
                    reg_value |= 0x08;
                    break;
                case 9:
                    reg_value |= 0x10;
                    break;
                case 10:
                    reg_value |= 0x18;
                    break;
                case 11:
                    reg_value |= 0x20;
                    break;
                default:
                    break;
             }
             switch (Play_dma_channel) {
                case 0:
                    reg_value |= 0x01;
                    break;
                case 1:
                    reg_value |= 0x02;
                    break;
                case 3:
                    reg_value |= 0x03;
                    break;
             }
             outportb(Sound_source_config_reg, reg_value);

                // wait for initialization to end
            for (i = 1; i <= 10; i++) {
                delay_ms(50L);
                reg_value = inportb(Index_reg);
                if ((reg_value & 0x80) == 0x00)
                    break;
            }
            if ((reg_value & 0x80) != 0x00)
                return(-1);

                // check for card presence by reading hardware revision number
            revision_id = get_dev_revision_info();
            if ((revision_id != J_GRADE_ID_NUM) &&
                                          (revision_id != K_GRADE_ID_NUM))
                return(-1);

            break;

    }

        // make sure device is shut down
    stop_dev_operation();

        // allocate transfer buffer - get virtual address
    if ((Transfer_bufP = mem_alloc_32k_block_aligned()) == (UCHAR *) 0)
        return(-1);

        // get physical address to buffer
    if ((Transfer_buf_phys_addr =
                         mem_virt_to_phys(Transfer_bufP)) == (ULONG) 0)
        return(-1);

    
    
    //clear the mode change enable bit
    outportb(Index_reg, 0x00);

    set_dev_left_output_mute(OFF);
    set_dev_right_output_mute(OFF);

    set_dev_monitor(ON);

    // return successful
    return(0);
}


VOID set_dev_config_info(USHORT stream_buffer_size, SHORT data_format,
              UCHAR bits_per_sample, USHORT sample_rate, UCHAR num_of_channels)
//    This function sets the data consumption interrupt, data format, bits
// per sample, and number of data channels for the device.
//    The parameter 'stream_buffer_size' is the size of the data buffers
// entering the driver and is a factor in determining the interrupt point.
// The other parameters indicate the data format, bits per sample, sample rate,
// and number of channels.
//    The function does not return a value.
{
    UCHAR clock_source, divide_select, linear_comp_select, format_select;
    USHORT num_samples, i;

        // determine the silence value
    switch (data_format) {
        case STREAM_DATA_FORMAT_PCM:
            if (bits_per_sample == 8)
                Silence_value = 0x80;
            else
                Silence_value = 0x00;
            break;

        case STREAM_DATA_FORMAT_MU_LAW:
            Silence_value = 0x7f;
            break;

        case STREAM_DATA_FORMAT_A_LAW:
            Silence_value = 0x55;
            break;
    }


        // store the data format, sample rate, bits per sample, and number of
        // channels if the info has changed since the last call - indicate
        // that a new configuration has occurred
    if ((data_format != Last_data_format)
                    || (bits_per_sample != Last_bits_per_sample)
                         || (sample_rate != Last_sample_rate)
                               || (num_of_channels != Last_num_of_channels)) {

            // indicate new configuration
        New_configuration = TRUE;

            // store the new configuration
        Last_data_format = data_format;
        Last_bits_per_sample = bits_per_sample;
        Last_sample_rate = sample_rate;
        Last_num_of_channels = num_of_channels;

            // get clock source and divide select from sample rate
        get_clock_info(sample_rate, &clock_source, &divide_select);

            // set up clock source and divide select for sample rate 
        Clock_and_data_format_reg_value = clock_source;
        Clock_and_data_format_reg_value |= divide_select << 1;

            // set up linear/companded select and format select
        get_format_info(data_format, bits_per_sample,
                                              &linear_comp_select, &format_select);
        Clock_and_data_format_reg_value |= linear_comp_select << 5;
        Clock_and_data_format_reg_value |= format_select << 6;

            // set up mono or stereo 
        Clock_and_data_format_reg_value |= (num_of_channels - 1) << 4;

    }

        // configuration has not changed
    else
        New_configuration = FALSE;

        // determine the buffer size for the device
    if ((num_of_channels == 2) && (data_format == STREAM_DATA_FORMAT_PCM) &&
                           (sample_rate == 44100) && (bits_per_sample == 16)
                                               && (Dev_buffer_size >= 0x4000))
        Dev_buffer_size = 0x4000;

    else if (stream_buffer_size >= 0x4000)
        Dev_buffer_size = 0x1000;

    else if (stream_buffer_size >= 0x2000)
        Dev_buffer_size = stream_buffer_size & 0xff00;

    else
        Dev_buffer_size = 0x100;

        // calculate the number of samples for this buffer size
    if (bits_per_sample == 16) {
        if (num_of_channels == 2)
            num_samples = Dev_buffer_size / 4;
        else
            num_samples = Dev_buffer_size / 2;
    }
    else {
        if (num_of_channels == 2)
            num_samples = Dev_buffer_size / 2;
        else
            num_samples = Dev_buffer_size;
    }

        // set the data consumption interrupt in number of samples
    set_reg(LOWER_BASE_COUNT_REG, (UCHAR) (num_samples - 1));
    set_reg(UPPER_BASE_COUNT_REG, (UCHAR) ((num_samples - 1) >> 8));

        // initialize transfer buffer position indicator
    Transfer_buf_pos = 0;

        // finished
    return;
}


VOID start_dev_play(VOID)
//    This function starts the device for a play operation.
//    The function does not have parameters.
//    The function does not return a value.
{
    USHORT i;
    UCHAR reg_value;

        // initialize the dma position marker
    Old_dma_pos = 0;

        // if the clock and data format info or the operation has changed since
        // last start - set the device and autocalibrate
    if ((New_configuration) || (Last_dev_operation != DEV_OPERATION_PLAY)) {

            // mute everything
        set_dev_left_output_mute(ON);
        set_dev_right_output_mute(ON);

        set_dev_digital_mix_mute(ON);

        set_dev_left_aux1_loop_mute(ON);
        set_dev_right_aux1_loop_mute(ON);

        set_dev_left_aux2_loop_mute(ON);
        set_dev_right_aux2_loop_mute(ON);

            // set the mode change enable bit 
        outportb(Index_reg, 0x40);

        switch (Device_type) {

               // set single DMA channel - set autocalibration
               // set DMA transfers - set capture pio enable  10001100 = 0x8c
            case DEV_MS_SOUND:
            case DEV_TOSHIBA_T4700CS:
            case DEV_TOSHIBA_T6600C:
            case DEV_COMPAQ:
            case DEV_SOUNDWAVE_32:
                set_reg(INTERFACE_CONFIGURATION_REG, 0x8c);
                break;

               // set dual DMA channel - set autocalibration
               // set DMA transfers 00001000 = 0x08
            case DEV_THINKPAD:
                set_reg(INTERFACE_CONFIGURATION_REG, 0x08);
                break;
        }

           // set the register
        set_reg(CLOCK_AND_DATA_FORMAT_REG, Clock_and_data_format_reg_value);

           // wait for resychronization to complete 
        for (i = 1; i <= 10; i++) {
            delay_ms(10L);
            reg_value = inportb(Index_reg);
            if (reg_value != 0x80)
               break;
        }

           // clear the mode change enable bit
        outportb(Index_reg, 0x00);

           // wait for the autocalibrate in progress bit to transition
           //   from low to high
        delay_ms(200L);

           // poll for the autocalibrate in progress bit to transition
           // from high to low
        for (i = 1; i <= 10; i++) {
            delay_ms(10L);
            reg_value = get_reg(TEST_AND_INITIALIZATION_REG);
            if ((reg_value & 0x20) == 0)
               break;
        }
    }

        // set the PEN only if clock and data format has not changed
    else {
        reg_value = get_reg(INTERFACE_CONFIGURATION_REG);
        reg_value |= 0x01;
        set_reg(INTERFACE_CONFIGURATION_REG, reg_value);
    }


       // initialize the DMA controller 
    init_dma(Play_dma_channel, Transfer_buf_phys_addr, TRANSFER_BUF_SIZE, DMA_READ);

       // turn on the interrupts
    set_dev_interrupts(ON);

        // turn on the monitor if indicated
    if (Monitor_state == ON) {
        set_dev_left_aux1_loop_mute(OFF);
        set_dev_right_aux1_loop_mute(OFF);
        set_dev_left_aux1_loop_atten(0);
        set_dev_right_aux1_loop_atten(0);
    }

        // turn the output mute off
    set_dev_left_output_mute(OFF);
    set_dev_right_output_mute(OFF);

       // set play enable
    reg_value = get_reg(INTERFACE_CONFIGURATION_REG);
    reg_value |= 0x01;
    set_reg(INTERFACE_CONFIGURATION_REG, reg_value);

        // save this operation
    Last_dev_operation = DEV_OPERATION_PLAY;

        // finished
    return;
}


VOID start_dev_record(SHORT source)
//    This function starts the device for a record operation.
//    The function does not have parameters.
//    The function does not return a value.
{
    USHORT i;
    UCHAR reg_value;

        // initialize the dma position marker
    Old_dma_pos = 0;

        // if the clock and data format info or the operation has changed since
        // last start - set the device and autocalibrate
    if ((New_configuration) || (Last_dev_operation != DEV_OPERATION_RECORD)) {

            // mute everything
        set_dev_left_output_mute(ON);
        set_dev_right_output_mute(ON);

        set_dev_digital_mix_mute(ON);

        set_dev_left_aux1_loop_mute(ON);
        set_dev_right_aux1_loop_mute(ON);
 
        set_dev_left_aux2_loop_mute(ON);
        set_dev_right_aux2_loop_mute(ON);
 
            // set the mode change enable bit 
        outportb(Index_reg, 0x40);
 
        switch (Device_type) {
 
                // set single DMA channel - set autocalibration
                // set - set playback pio enable 01001100 = 0x4c
            case DEV_MS_SOUND:
            case DEV_TOSHIBA_T4700CS:
            case DEV_TOSHIBA_T6600C:
            case DEV_COMPAQ:
            case DEV_SOUNDWAVE_32:
                set_reg(INTERFACE_CONFIGURATION_REG, 0x4c);
                break;
 
               // set dual DMA channel - set autocalibration
               // 00001000 = 0x08
            case DEV_THINKPAD:
                set_reg(INTERFACE_CONFIGURATION_REG, 0x08);
                break;
        }
 
            // set clock and data format register
        set_reg(CLOCK_AND_DATA_FORMAT_REG, Clock_and_data_format_reg_value);
 
            // wait for resychronization to complete
        for (i = 1; i <= 10; i++) {
            delay_ms(10L);
            reg_value = inportb(Index_reg);
            if (reg_value != 0x80)
                break;
        }
 
           // clear the mode change enable bit
        outportb(Index_reg, 0x00);
 
           // wait for the autocalibrate in progress bit to transition
           //   from low to high 
        delay_ms(200L);
 
           // poll for the autocalibrate in progress bit to transition
           // from high to low 
        for (i = 1; i <= 10; i++) {
            delay_ms(10L);
            reg_value = get_reg(TEST_AND_INITIALIZATION_REG);
            if ((reg_value & 0x20) == 0)
                break;
        }
    }

        // set the CEN only if config info has not changed
    else {
        reg_value = get_reg(INTERFACE_CONFIGURATION_REG);
        reg_value |= 0x02;
        set_reg(INTERFACE_CONFIGURATION_REG, reg_value);
    }

        // set the input source
    switch (Device_type) {
        case DEV_MS_SOUND:
        case DEV_THINKPAD:
        case DEV_COMPAQ:
        case DEV_TOSHIBA_T6600C:
        case DEV_SOUNDWAVE_32:
            switch (source) {
                case DEV_MICROPHONE:
                    set_dev_left_input_source(DEV_MICROPHONE);
                    set_dev_right_input_source(DEV_MICROPHONE);
                    break;
                case DEV_BOOSTED_MICROPHONE:
                    set_dev_left_input_source(DEV_BOOSTED_MICROPHONE);
                    set_dev_right_input_source(DEV_BOOSTED_MICROPHONE);
                    break;
                case DEV_AUX1:
                    set_dev_left_input_source(DEV_AUX1);
                    set_dev_right_input_source(DEV_AUX1);
                    break;
            }
            break;
 
        case DEV_TOSHIBA_T4700CS:
            switch (source) {
                case DEV_MICROPHONE:
                    set_dev_left_input_source(DEV_MICROPHONE);
                    set_dev_right_input_source(DEV_MICROPHONE);
                    break;
                case DEV_BOOSTED_MICROPHONE:
                    set_dev_left_input_source(DEV_BOOSTED_MICROPHONE);
                    set_dev_right_input_source(DEV_BOOSTED_MICROPHONE);
                    break;
                case DEV_AUX1:
                    set_dev_left_input_source(DEV_MICROPHONE);
                    set_dev_right_input_source(DEV_MICROPHONE);
                    break;
            }
            break;
    }

        // initialize the DMA controller
    init_dma(Record_dma_channel, Transfer_buf_phys_addr,
                                             TRANSFER_BUF_SIZE, DMA_WRITE);

        // turn on the interrupts 
    set_dev_interrupts(ON);

    if (Monitor_state == ON) {
        set_dev_left_aux1_loop_mute(OFF);
        set_dev_right_aux1_loop_mute(OFF);
        set_dev_left_output_mute(OFF);
        set_dev_right_output_mute(OFF);
        set_dev_left_aux1_loop_atten(0);
        set_dev_right_aux1_loop_atten(0);
    }

        // set capture enable
    reg_value = get_reg(INTERFACE_CONFIGURATION_REG);
    reg_value |= 0x02;
    set_reg(INTERFACE_CONFIGURATION_REG, reg_value);

        // store the operation
    Last_dev_operation = DEV_OPERATION_RECORD;

        // finished 
    return;
}


SHORT dev_auto_detect_dev_type(SHORT *device_type)
//    This function supports hardware auto detection.
//    The function returns the device type in the parameter if found.
//    The function returns the value 0 if a known device was found. Otherwise,
// the value -1 is returned.
{
        // check for Orchid Soundwave 32 - return successful if found
    if (dev_is_soundwave_32()) {
        *device_type = DEV_SOUNDWAVE_32;
        return (0);
    }

        // return unsuccessful
    return (-1);
}


VOID set_dev_master_volume(USHORT new_master)
//    This function sets the master volume level for the device. Master volume
// is not supported in hardware, so we save it in a global and scale stream
// volume by this value.
//    The parameter 'new_master' is a value from 0 to 7fff expressing the
// volume level.
//    The function does not return a value.
{
        // save with 8-bit resolution
    Master_volume = new_master >> 8; 

        // set th device
    set_dev_left_volume(Left_volume_setting);
    set_dev_right_volume(Right_volume_setting);

        // finished
    return;
}


VOID set_dev_left_volume(ULONG setting)
//    This function sets the volume of the left channel.
//    The parameter 'setting' is a value from 0 to 7fffffff expressing the
// volume level.
//    The function does not return a value.
{
    UCHAR i;
    ULONG j;

        // store the value for use with the master volume function
    Left_volume_setting = setting;

        // scale by master volume
    setting = scale_by_master_volume(setting);

        // convert setting range of 0 - 7fffffff to range of 0 - 63 which is 
        // the range of 1.5 decibel increments 
    j = 0L;
    for (i = 0; i <= LEVEL_TABLE_SIZE - 1; i++) {
        if (j >= setting)
            break;
        j += 0x147ae14;
    }

        // set the device attenuation
    set_dev_left_output_atten(Output_atten_table[i]);

        // finished
    return;
}


VOID set_dev_right_volume(ULONG setting)
//    This function sets the volume of the right channel.
//    The parameter 'setting' is a value from 0 to 7fffffff expressing the
// volume level.
//    The function does not return a value.
{
    UCHAR i;
    ULONG j;

        // store the value for use with the master volume function
    Right_volume_setting = setting;

        // scale by master volume 
    setting = scale_by_master_volume(setting);

        // convert setting range of 0 - 7fffffff to range of 0 - 63 which is
        // the range of 1.5 decibel increments 
    j = 0L;
    for (i = 0; i <= LEVEL_TABLE_SIZE - 1; i++) {
        if (j >= setting)
            break;
        j += 0x147ae14;
    }

        // set the device attenuation 
    set_dev_right_output_atten(Output_atten_table[i]);

        // finished
    return;
}


VOID set_dev_left_input_level(ULONG setting)
//    This function sets the input gain level of the left channel.
//    The parameter 'setting' is a value from 0 to 7fffffff expressing the
// gain level.
//    The function does not return a value.
{
    UCHAR i;
    ULONG j;

        // store the value of the left gain setting
    Left_gain_setting = setting;

        // convert setting range of 0 - 7fffffff to range of 0 - 63 which is
        // the range of 1.5 decibel increments 
    j = 0L;
    for (i = 0; i <= LEVEL_TABLE_SIZE - 1; i++) {
        if (j >= setting)
            break;
        j += 0x147ae14;
    }     

        // store the gain level
    Left_gain_level = Input_gain_table[i];

        // finished
    return;
}


VOID set_dev_right_input_level(ULONG setting)
//    This function sets the input gain level of the right channel.
//    The parameter 'setting' is a value from 0 to 7fffffff expressing the
// gain level.
//    The function does not return a value.
{
    UCHAR i;
    ULONG j;

        // store the value of the right gain setting
    Right_gain_setting = setting;

        // convert setting range of 0 - 7fffffff to range of 0 - 63 which is
        // the range of 1.5 decibel increments
    j = 0L;
    for (i = 0; i <= LEVEL_TABLE_SIZE - 1; i++) {
        if (j >= setting)
            break;
        j += 0x147ae14;
    }

        // store the gain level
    Right_gain_level = Input_gain_table[i];

        // finished
    return;
}


static VOID set_dev_left_input_source(UCHAR source_num)
//    This function sets the left input source.
//    The parameter 'source_num' is the number of the source.
//    The function does not return a value.
{
    UCHAR reg_value;

    switch (source_num) {
        case DEV_LINE:
            reg_value = 0x00;
            reg_value |= Left_gain_level;
            break;
        case DEV_AUX1:
            reg_value = 0x40;
            reg_value |= Left_gain_level;
            break;
        case DEV_MICROPHONE:
        case DEV_BOOSTED_MICROPHONE:
            reg_value = 0x80;
            if (Left_gain_setting >= 0x40000000)
                reg_value |= 0x20;
            break;
        case DEV_POST_MIXED_DAC_OUTPUT:
            reg_value = 0xc0;
            break;
    }

        // set the register
    set_reg(LEFT_INPUT_CONTROL_REG, reg_value);

        // finished 
    return;
}


static VOID set_dev_right_input_source(UCHAR source_num)
//    This function sets the right input source.
//    The parameter 'source_num' is the number of the source.
//    The function does not return a value.
{
    UCHAR reg_value;

    switch (source_num) {
        case DEV_LINE:
            reg_value = 0x00;
            reg_value |= Right_gain_level;
            break;
        case DEV_AUX1:
            reg_value = 0x40;
            reg_value |= Right_gain_level;
            break;
        case DEV_MICROPHONE:
        case DEV_BOOSTED_MICROPHONE:
            reg_value = 0x80;
            if (Right_gain_setting >= 0x40000000)
                reg_value |= 0x20;
            break;
        case DEV_POST_MIXED_DAC_OUTPUT:
            reg_value = 0xc0;
            break;
    }

        // set the register
    set_reg(RIGHT_INPUT_CONTROL_REG, reg_value);

        // finished
    return;
}


USHORT get_dev_buffer_size(VOID)
//    This function gets the current data transfer buffer size.
//    The function does not have parameters.
//    The function returns the size.
{
    return (Dev_buffer_size);
}


VOID set_dev_type(SHORT device_type)
//    This function stores the device type.
//    The parameter 'device_type' is the number of the device.
//
{
        // store the device type
    Device_type = device_type;

        // finished
    return ;
}


SHORT get_dev_type(VOID)
//    This function gets the device type.
//    The function does not have parameters.
//    The function returns the device number.
{
        // return the device type
    return (Device_type);
}


SHORT set_dev_irq_number(USHORT num)
//    This function stores the value for the device's data consumption
// interrupt.
//    The parameter 'num' is the number of the interrupt.
//    The function returns the value -1 if the interrupt number is not valid
// for the device. Otherwise, the function returns the value 0.
{
    switch (Device_type) {
        case DEV_MS_SOUND:
        case DEV_TOSHIBA_T4700CS:
        case DEV_TOSHIBA_T6600C:
        case DEV_COMPAQ:
        case DEV_SOUNDWAVE_32:
            if ((num != 7) && (num != 9) && (num != 10) && (num != 11))
                return(-1);
            Irq_num = num;
            return(0);

        case DEV_THINKPAD:
            if ((num != 5) && (num != 10) && (num != 11) && (num != 15))
                return(-1);
            Irq_num = num;
            return(0);

        default:
            return(-1);
    }
}


SHORT set_dev_dma_channels(USHORT play_channel)
//    This function stores the value for the device's dma channel.
//    The parameter 'play_channel' is the number of the dma channel.
//    The function returns the value -1 if the dma channel is not valid
// for the device. Otherwise, the function returns the value 0.
{
    switch (Device_type) {
        case DEV_MS_SOUND:
        case DEV_TOSHIBA_T4700CS:
        case DEV_TOSHIBA_T6600C:
        case DEV_COMPAQ:
        case DEV_SOUNDWAVE_32:
            if ((play_channel != 0) &&
                                (play_channel != 1) && (play_channel != 3))
                return(-1);
            Play_dma_channel = Record_dma_channel = play_channel;
            return(0);

        case DEV_THINKPAD:
            if (play_channel != 0)
                return(-1);
            Play_dma_channel = 0;
            Record_dma_channel = 1;
            return(0);
    }

    return(-1);
}


USHORT get_dev_base_io_address(VOID)
//    This function gets the base i/o address for the device.
//    The function does not have parameters.
//    The function returns the i/o address.
{
    return(Base_io_addr);
}


USHORT get_dev_irq_number(VOID)
//    This function gets the irq number for the device.
//    The function does not have parameters.
//    The function returns the irq number.
{
    return(Irq_num);
}


USHORT get_dev_dma_play_channel(VOID)
//    This function gets the dma channel for the device.
//    The function does not have parameters.
//    The function returns the dma channel.
{
    return(Play_dma_channel);
}


SHORT init_dev_dsp(VOID)
//    This function initalizes the dsp task if it exists.
//    The function does not have parameters.
//    The function returns the value 0 if successful. Otherwise, the function
// returns the value -1.
{
    return(0);
}


static VOID set_dev_interrupts(BOOL state)
//    This function enables or disables data consumption interrupts. When 
// interrupts are enabled, the interrupt pin will go hi when the number of 
// samples programmed in the base count register is reached.
//    The parameter 'state' indicates whether interrupts are on or off.
//    The function does not return a value. 

{
    UCHAR reg_value;

        // initialize value
    reg_value = 0x00;

        // set up enable or disable state for interrupts 
    reg_value |= (state << 1);

        // set the register
    set_reg(PIN_CONTROL_REG, reg_value);

        // finished 
    return;
}


VOID clear_dev_interrupt(VOID)
//    This function clears the interrupt flag on the device thus re-enabling
// interrupts.
//    The function does not have parameters.
//    The function does not return a value.      
{
        // clear the flag in the status register 
    outportb(Status_reg, 0);

        // return
    return;
}


VOID stop_dev_operation(VOID)
//    This function shuts down device operations.
//    The function does not have parameters.
//    The function does not return a value.
{
    UCHAR reg_value;

        // shut off CEN and PEN
    reg_value = get_reg(INTERFACE_CONFIGURATION_REG);
    reg_value &= 0xcc;

    set_reg(INTERFACE_CONFIGURATION_REG, reg_value);

        // turn off the interrupts
    set_dev_interrupts(OFF);

        // shut off the DMA
    disable_dma(Play_dma_channel);
    disable_dma(Record_dma_channel);

        // finished
    return;
}


VOID set_dev_monitor(BOOL state)
//    This function sets the input monitor state.
//    The parameter 'state' indicates monitor off or monitor on.
//    The function does not return a value.
{
       // store the state
    Monitor_state = state;

        // set the device
    if (state == ON) {
       set_dev_left_aux1_loop_mute(OFF);
       set_dev_right_aux1_loop_mute(OFF);
    }
    else {
       set_dev_left_aux1_loop_mute(ON);
       set_dev_right_aux1_loop_mute(ON);
    }

        // finished
    return;
}


static VOID set_dev_digital_mix_atten(UCHAR decibel_1_5_increments)
//    This function sets digital mix attenuation.
//    The parameter 'decibel_1_5_increments' specifies the level of attenuation
// of the analog to digital conversion data in mixing with the digital to
// analog conversion input in 1.5 decibel increments.
//    The function does not return a value.

{
    UCHAR reg_value, atten_steps;

        // get present register value and clear the appropriate bits including
        // reserved bits */
    reg_value = get_reg(DIGITAL_MIX_CONTROL_REG);
    reg_value &= 0x03;

        // set up attenuation 
    reg_value |= decibel_1_5_increments << 2;

        // set the register
    set_reg(DIGITAL_MIX_CONTROL_REG, reg_value);

        // finished 
    return;
}


static VOID set_dev_digital_mix_mute(BOOL state)
//    This enables or disables digital mix mute.
//      The parameter 'state' indicates whether digital mix is on or off.
//      The function does not return a value. 
{
    UCHAR reg_value, atten_steps;

        // get present register value and clear the appropriate bits including 
        // reserved bits
    reg_value = get_reg(DIGITAL_MIX_CONTROL_REG);
    reg_value &= 0xfc;

        // set up enable or disable mix state 
    if (state == OFF)
        reg_value |= 0x01;

        // set the register
    set_reg(DIGITAL_MIX_CONTROL_REG, reg_value);

        // finished 
    return;
}


static UCHAR get_dev_revision_info(VOID)
//    This function retrieves to revision info of the AD1848.
//      The function returns the revision ID.
{
    UCHAR reg_value;

        // get the id 
    reg_value = get_reg(MISCELLANEOUS_CONTROL_REG);

        // return the value
    return(reg_value & 0x0f);
}


static VOID get_clock_info(USHORT freq, UCHAR *source, UCHAR *select)
//    This function gets clock source and divide factor information for a
// particular frequency.
//    The parameter 'freq' is the desired frequency.
//    The function returns the clock source in the parameter 'source' and the
// divide factor in the parameter 'select'. 
{
    USHORT i;

        // search the table for an exact or close fit
    for (i = 0; i <= FREQ_TABLE_SIZE - 1; i++) {
        if (freq <= Freq_table[i].freq) {
            *source = Freq_table[i].clock_source;
            *select = Freq_table[i].divide_select;
            break;
        }
    }

        // finished
    return;
}


static VOID get_format_info(SHORT data_format, USHORT bits_per_sample,
                              UCHAR *linear_comp_select, UCHAR *format_select)
//    This function gets the linear / companded select and the format select
// for a data format and data size.
//    The paramters 'data_format' and 'bits_per_sample' describe the data
// format and data size.
//    The function returns the linear / companded select and the format select
// in the parameters 'linear_comp_select' and 'format_select'.
{
        // get the select info 
    switch (data_format) {

        case STREAM_DATA_FORMAT_PCM:
            if (bits_per_sample == 8) {
                *linear_comp_select = 0;
                *format_select = 0;
            }
            else {
                *linear_comp_select = 0;
                *format_select = 1;
            }
            return;

        case STREAM_DATA_FORMAT_MU_LAW:
            *linear_comp_select = 1;
            *format_select = 0;
            return;

        case STREAM_DATA_FORMAT_A_LAW:
            *linear_comp_select = 1;
            *format_select = 1;
            return;

        default:
            return;
    }
}


static VOID set_reg(UCHAR index, UCHAR value)
//    This function sets an indirect register.
//    The parameter 'index' contains the offset of the register. The parameter
// 'value' contains the value to write to the register.
//    The function does not return a value.
{
    UCHAR reg_value;

        // save the unaffected bits 
    reg_value = inportb(Index_reg);
    reg_value &= 0xf0;

        // set offset of indirect register
    reg_value |= index;
    outportb(Index_reg, reg_value);

        // set data value 
    outportb(Index_data_reg, value);

        // finished
    return;
}


static UCHAR get_reg(UCHAR index)
//    This function gets a value from an indirect register.
//    The parameter 'index' is the offset of the register.
//    The function returns the value stored in the register.     
{
    UCHAR reg_value;

        // save the unaffected bits
    reg_value = inportb(Index_reg);
    reg_value &= 0xf0;

        // set offet of indirect register 
    reg_value |= index;
    outportb(Index_reg, reg_value);

        // return the register value
    return(inportb(Index_data_reg));
}


static VOID set_dev_left_output_mute(BOOL state)
//    This function sets the mute state of the left output.
//    The parameter 'state' contains the value 'ON' to mute the output or
// the value 'OFF' to disable mute.
//    The function does not return a value. 
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including
        // reserved bits
    reg_value = get_reg(LEFT_DAC_CONTROL_REG);
    reg_value &= 0x3f;

        // set up mute state 
    reg_value |= state << 7;

        // set the register
    set_reg(LEFT_DAC_CONTROL_REG, reg_value);

        // finished 
    return;
}


static VOID set_dev_right_output_mute(BOOL state)
//    This function sets the mute state of the right output. 
//    The parameter 'state' contains the value 'ON' to mute the output or
// the value 'OFF' to disable mute.
//    The function does not return a value.
{

    UCHAR reg_value;

        // get present register value and clear the appropriate bits including
        // reserved bits 
    reg_value = get_reg(RIGHT_DAC_CONTROL_REG);
    reg_value &= 0x3f;

        // set up mute state
    reg_value |= state << 7;

        // set the register 
    set_reg(RIGHT_DAC_CONTROL_REG, reg_value);

        // finished
    return;
}


static VOID set_dev_left_output_atten(UCHAR decibel_1_5_decrements)
//    This function selects the left output attenuation.
//    The parameter 'decibel_1_5_decrements' is the change in the level of
// attenuation expressed in 1.5 decibel decrements.
//    The function does not return a value. 
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits
        // including reserved bits
    reg_value = get_reg(LEFT_DAC_CONTROL_REG);
    reg_value &= 0x80;

        // set up mute state 
    reg_value |= decibel_1_5_decrements;

        // set the register
    set_reg(LEFT_DAC_CONTROL_REG, reg_value);

        // finished 
    return;
}


static VOID set_dev_right_output_atten(USHORT decibel_1_5_decrements)
//    This function selects the left output attenuation.
//    The parameter 'decibel_1_5_decrements' is the level of attenuation
// expressed in 1.5 decibel decrements.
//    The function does not return a value. 
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits 
        // including reserved bits
    reg_value = get_reg(RIGHT_DAC_CONTROL_REG);
    reg_value &= 0x80;

        // set up mute state
    reg_value |= decibel_1_5_decrements;

        // set the register 
    set_reg(RIGHT_DAC_CONTROL_REG, reg_value);

        // finished
    return;
}


static VOID set_dev_left_aux1_loop_mute(BOOL state)
//    This function sets the state of the lrft aux1 input loop out mute circuit.
//    The parameter 'state' indicates mute on or mute off.
//    The function does not return a value.
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including
        // reserved bits 
    reg_value = get_reg(LEFT_AUX1_INPUT_CONTROL_REG);
    reg_value &= 0x1f;

        // set up mute state
    reg_value |= state << 7;

        // set the register 
    set_reg(LEFT_AUX1_INPUT_CONTROL_REG, reg_value);

        // finished
    return;
}


static VOID set_dev_right_aux1_loop_mute(BOOL state)
//    This function sets the state of the right aux1 input loop out mute
// circuit.
//    The parameter 'state' indicates mute on or mute off.
//    The function does not return a value.
{
    UCHAR reg_value;

        /* get present register value and clear the appropriate bits including */
        /* reserved bits */
    reg_value = get_reg(RIGHT_AUX1_INPUT_CONTROL_REG);
    reg_value &= 0x1f;

        /* set up mute state */
    reg_value |= state << 7;

        /* set the register */
    set_reg(RIGHT_AUX1_INPUT_CONTROL_REG, reg_value);

        /* finished */
    return;
}


static VOID set_dev_left_aux1_loop_atten(UCHAR decibel_1_5_decrements)
//    This function sets the level of the left aux1 input loop out attenuation.
//    The parameter expresses the level in 1.5 decibel units.
//    The function does not return a value.
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including 
        // reserved bits */
    reg_value = get_reg(LEFT_AUX1_INPUT_CONTROL_REG);
    reg_value &= 0x80;

        // set up gain state 
    reg_value |= decibel_1_5_decrements;

        // set the register
    set_reg(LEFT_AUX1_INPUT_CONTROL_REG, reg_value);

        // finished 
    return;
}


static VOID set_dev_right_aux1_loop_atten(UCHAR decibel_1_5_decrements)
//    This function sets the level of the right aux1 input loop out attenuation.
//    The parameter expresses the level in 1.5 decibel units.
//    The function does not return a value.
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including
        // reserved bits 
    reg_value = get_reg(RIGHT_AUX1_INPUT_CONTROL_REG);
    reg_value &= 0x80;

        // set up gain state
    reg_value |= decibel_1_5_decrements;

        // set the register 
    set_reg(RIGHT_AUX1_INPUT_CONTROL_REG, reg_value);

        // finished
    return;
}


static VOID set_dev_left_aux2_loop_mute(BOOL state)
//    This function sets the state of the left aux2 input loop out mute
// circuit.
//    The parameter 'state' indicates mute on or mute off.
//    The function does not return a value.

{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including
        // reserved bits 
    reg_value = get_reg(LEFT_AUX2_INPUT_CONTROL_REG);
    reg_value &= 0x1f;

        // set up mute state
    reg_value |= state << 7;

        // set the register 
    set_reg(LEFT_AUX2_INPUT_CONTROL_REG, reg_value);

        // finished
    return;
}


static VOID set_dev_right_aux2_loop_mute(BOOL state)
//    This function sets the state of the right aux2 input loop out mute
// circuit.
//    The parameter 'state' indicates mute on or mute off.
//    The function does not return a value.
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including
        // reserved bits 
    reg_value = get_reg(RIGHT_AUX2_INPUT_CONTROL_REG);
    reg_value &= 0x1f;

        // set up mute state
    reg_value |= state << 7;

        // set the register
    set_reg(RIGHT_AUX2_INPUT_CONTROL_REG, reg_value);

        // finished
    return;
}


static VOID set_dev_left_aux2_loop_atten(UCHAR decibel_1_5_decrements)
//    This function sets the level of the left aux2 input loop out attenuation.
//    The parameter expresses the level in 1.5 decibel units.
//    The function does not return a value.
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including
        // reserved bits 
    reg_value = get_reg(LEFT_AUX2_INPUT_CONTROL_REG);
    reg_value &= 0x80;

        // set up gain state
    reg_value |= decibel_1_5_decrements;

        // set the register 
    set_reg(LEFT_AUX2_INPUT_CONTROL_REG, reg_value);

        // finished
    return;
}


static VOID set_dev_right_aux2_loop_atten(UCHAR decibel_1_5_decrements)
//    This function sets the level of the right aux2 input loop out attenuation.
//    The parameter expresses the level in 1.5 decibel units.
//    The function does not return a value.
{
    UCHAR reg_value;

        // get present register value and clear the appropriate bits including 
        // reserved bits 
    reg_value = get_reg(RIGHT_AUX2_INPUT_CONTROL_REG);
    reg_value &= 0x80;

        // set up gain state 
    reg_value |= decibel_1_5_decrements;

        // set the register
    set_reg(RIGHT_AUX2_INPUT_CONTROL_REG, reg_value);

        // finished 
    return;
}


static ULONG scale_by_master_volume(ULONG stream_volume)
//    This function scales the stream volume by the master volume.
//    The parameter expresses the volume with a value from 0 to 0x7fff.
//    The function returns the scaled volume.
{
    union {
        USHORT conv_ushort[2];
        ULONG  conv_ulong;
    } convert;
    UCHAR stream8bit;
    USHORT scale, final;

        // done if no volume
    if ((Master_volume == 0) || (stream_volume == 0))
        return (0);

        // convert stream volume to 8 bit 
    convert.conv_ulong = stream_volume;
    stream8bit = convert.conv_ushort[1] >> 8;

        // the decimal point is moved 2 places right for better resolution
    scale = (127 * 100) / (USHORT)stream8bit;
    final = ((USHORT)Master_volume * 100) / scale;

        // convert final back to a ulong 
    convert.conv_ushort[1] = final << 8;
    convert.conv_ushort[0] = 0;

       // finished
    return (convert.conv_ulong);
}
