/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************/
/*  Module Name: FDAPI.C                                                      */
/*                                                                            */
/*      Application interface of for OS/2 DBCS PM font drivers.               */
/*                                                                            */
/*  Created: 15-Oct-1991                                                      */
/*  Author:  Soh Ohta     [jl09057 @ ymtvm3, IBM Japan]                       */
/*                                                                            */
/*                        Refer to Copyright Inscruction Form No.G120-2083    */
/*                                                                            */
/*  Exported Functions:   FONT_ENABLE                                         */
/*                                                                            */
/*  Public Functions:     none                                                */
/*                                                                            */
/*  Public Data:          none                                                */
/*                                                                            */
/*  General Description:                                                      */
/*                                                                            */
/******************************************************************************/

#define INCL_PM
#define INCL_DOS
#include <os2.h>
#include <os2nls.h>
#include <os2nlsp.h>
#include <os2nlsfd.h>
#include <pmwinx.h>                     // WinQueryProcessCP()
#include "resmap.h"


ULONG   fdsem = 0;                      // semaphore to serialize FD access.
USHORT  cFonts = 0;                     // number of fonts available.
USHORT  openCount = 0;                  // open count.
SIZES   szsCell = {0};                  // cellsize.

extern HMODULE  hmod;                   // FD's module handle.

USHORT  QueryFocaFont (USHORT, PFOCAFONT);
USHORT  QueryCellSize (USHORT, USHORT, SIZES NEAR*);
USHORT  QueryCellImage (USHORT, USHORT, SIZES NEAR*, PBYTE, USHORT);



/******************************************************************************/
/*  FdGetVersion                                                              */
/*                                                                            */
/*  Return the font driver version information.                               */
/*  If this font driver is public font driver, and registered to the system   */
/*  as one of system font driver, then return DBE_FD_DESC_SYSTEM.             */
/*  Be sure that this entry may be called before opened.                      */
/*                                                                            */
/*  Parameter:                                                                */
/*      lpVersion = buffer address to return version information.             */
/*  Return:                                                                   */
/*      success   = DBE_FD_NORMAL                                             */
/*      error     = DBE_FD_ERROR                                              */
/******************************************************************************/

#define CBKEYBUF    256
#define CBRESULT    2

ULONG FdGetVersion (PFONT_DRIVER_VERSION lpVersion)
{
    PFD_DESC    lpDesc;
    CHAR        szKeyName[CBKEYBUF];
    CHAR        szError  [CBRESULT];
    CHAR        szBuffer [CBRESULT];
    PSZ         KeyName (PSZ);

    if (DosGetResource2 (hmod,RT_FDDESC,WinQueryProcessCP(),&lpDesc) == 0) {
        if (DosGetModName (hmod,CBKEYBUF,szKeyName) == 0) {

            lpVersion->version_number = DBE_FD_VERSION_3;
            *lpVersion->fd_desc = *lpDesc;
            DosFreeResource (lpDesc);

            szError[0] = szBuffer[0] = '\0';

            if ((lpVersion->fd_desc->flType == DBE_FD_DESC_PUBLIC) &&
                (PrfQueryProfileString (HINI_USERPROFILE,"PMNLSFNT",KeyName (szKeyName),szError,szBuffer,CBRESULT) != 0) &&
                (szBuffer[0] != '\0')) {

                // this font driver will be used as system font driver.
                lpVersion->fd_desc->flType = DBE_FD_DESC_SYSTEM;
            }

            return DBE_FD_NORMAL;
        }
    }
    return DBE_FD_ERROR;
}


PSZ KeyName (PSZ path)
{
    register USHORT i = 0;
    PSZ             psz;

    while (path[i])         i++;
    while (path[i]!='\\')   i--;
    psz = &path[i+1];
    while (path[i]!='.')    i++;
    path[i] = 0;

    return psz;
}

/******************************************************************************/
/*  FdOpen                                                                    */
/*                                                                            */
/*  Validate font driver access.                                              */
/*                                                                            */
/*  Parameter:                                                                */
/*      nothing                                                               */
/*  Return:                                                                   */
/*      success   = number of fonts available                                 */
/*      error     = 0                                                         */
/******************************************************************************/

ULONG FdOpen ()
{
    if ((openCount == 0) &&
        ((cFonts = LoadResources (WinQueryProcessCP())) == 0))
        return 0;

    openCount++;
    return cFonts;
}

/******************************************************************************/
/*  FdClose                                                                   */
/*                                                                            */
/*  Invalidate font driver access.                                            */
/*                                                                            */
/*  Parameter:                                                                */
/*      nothing                                                               */
/*  Return:                                                                   */
/*      success   = DBE_FD_NORMAL                                             */
/*      error     = DBE_FD_ERROR                                              */
/******************************************************************************/

ULONG FdClose ()
{
    if (openCount == 0)
        return DBE_FD_ERROR;

    if (--openCount == 0)
        FreeResources ();
    return DBE_FD_NORMAL;
}

/******************************************************************************/
/*  FdGetInfo                                                                 */
/*                                                                            */
/*  Return specified font attribute.                                          */
/*                                                                            */
/*  Parameter:                                                                */
/*      lpQuery   = buffer address to return character attribute              */
/*      ires      = FD font resource id                                       */
/*      index     = glyph index                                               */
/*  Return:                                                                   */
/*      success   = DBE_FD_NORMAL or DBE_FD_NO_MORE_FONT                      */
/*      error     = DBE_FD_ERROR                                              */
/******************************************************************************/

ULONG FdGetInfo (PFONT_INFO lpInfo, USHORT ires)
{
    if (openCount == 0)
        return DBE_FD_ERROR;
    if (--ires>=cFonts)
        return DBE_FD_NO_MORE_FONT;

    if (QueryFocaFont (ires,lpInfo->font_info_ptr) != 0)
        return DBE_FD_ERROR;
    lpInfo->option = DBE_FD_FONT_DEFAULT;
    return DBE_FD_NORMAL;
}

/******************************************************************************/
/*  FdQueryChar                                                               */
/*                                                                            */
/*  Return specified character attribute.                                     */
/*                                                                            */
/*  Parameter:                                                                */
/*      lpQuery   = buffer address to return character attribute              */
/*      ires      = FD font resource id                                       */
/*      index     = glyph index                                               */
/*  Return:                                                                   */
/*      success   = DBE_FD_NORMAL                                             */
/*      error     = DBE_FD_ERROR                                              */
/******************************************************************************/

ULONG FdQueryChar (PFONT_QUERY lpQuery, USHORT ires, USHORT index)
{
    if (openCount == 0)
        return DBE_FD_ERROR;
    if (--ires>=cFonts)
        return DBE_FD_ERROR;

    if (QueryCellSize (ires,index,&szsCell) != 0)
        return DBE_FD_ERROR;

    if (lpQuery->QueryID & DBE_FD_QUERY_WIDTHHEIGHT) {
        lpQuery->xCellWidth  = szsCell.cx;
        lpQuery->yCellHeight = szsCell.cy;
    }
    if (lpQuery->QueryID & DBE_FD_QUERY_ABC)
        // abc font not supported yet.
        lpQuery->xCellA = lpQuery->xCellB = lpQuery->xCellC = 0;

    if (lpQuery->QueryID & DBE_FD_QUERY_LENGTH)
        lpQuery->data_length = szsCell.cy*(((szsCell.cx-1)/8)+1);

    return DBE_FD_NORMAL;
}

/******************************************************************************/
/*  FdReadChar                                                                */
/*                                                                            */
/*  Return specified character image from specified FD font resource.         */
/*                                                                            */
/*  Parameter:                                                                */
/*      lpFontMap = buffer address which contains image definition            */
/*      ires      = FD font resource id                                       */
/*      index     = glyph index                                               */
/*  Return:                                                                   */
/*      success   = DBE_FD_NORMAL                                             */
/*      error     = DBE_FD_ERROR                                              */
/******************************************************************************/

ULONG FdReadChar (PFONT_MAP lpFontMap, USHORT ires, USHORT index)
{
    if (openCount == 0)
        return DBE_FD_ERROR;
    if (--ires>=cFonts)
        return DBE_FD_ERROR;

    if (QueryCellImage (ires,index,&szsCell,(PBYTE)lpFontMap->buffer_ptr,lpFontMap->buffer_length) != 0)
        return DBE_FD_ERROR;

    lpFontMap->output_width  = szsCell.cx;
    lpFontMap->output_height = szsCell.cy;
    return DBE_FD_NORMAL;
}

/******************************************************************************/
/*  FdWriteChar                                                               */
/*                                                                            */
/*  Update specified character image into specified FD font resource.         */
/*                                                                            */
/*  Parameter:                                                                */
/*      lpFontMap = buffer address which contains image definition            */
/*      ires      = FD font resource id                                       */
/*      index     = glyph index                                               */
/*  Return:                                                                   */
/*      success   = DBE_FD_NORMAL                                             */
/*      error     = DBE_FD_ERROR                                              */
/******************************************************************************/

ULONG FdWriteChar (PFONT_MAP lpFontMap, USHORT ires, USHORT index)
{
    return DBE_FD_ERROR;    // not supported.
}

/******************************************************************************/
/*  FONT_ENABLE - exported entry                                              */
/*                                                                            */
/*  Exported common entry of OS/2 DBCS PM font driver.                        */
/*  Serialize request and dispatch to the requested function.                 */
/*                                                                            */
/*  Parameter:                                                                */
/*      ulFunction= font driver function request id                           */
/*      ul1       = ULONG parameter 1                                         */
/*      ul2       = ULONG parameter 1                                         */
/*      ul3       = ULONG parameter 1                                         */
/*  Return:                                                                   */
/*      success   = DBE_FD_NORMAL or DBE_FD_NO_MORE_FONT                      */
/*      error     = DBE_FD_ERROR                                              */
/******************************************************************************/

ULONG FAR PASCAL _export _saveregs _loadds
FONT_ENABLE (ULONG ul1, ULONG ul2, ULONG ul3, ULONG ulFunction)
{
    register ULONG  rc;

    DosSemRequest ((HSEM)&fdsem,SEM_INDEFINITE_WAIT);
    switch (ulFunction)
    {
        case DBE_FD_READ:       rc = FdReadChar ((PFONT_MAP)ul2,(USHORT)ul1,(USHORT)ul3); break;
        case DBE_FD_WRITE:      rc = FdWriteChar ((PFONT_MAP)ul2,(USHORT)ul1,(USHORT)ul3); break;
        case DBE_FD_GET_INFO:   rc = FdGetInfo ((PFONT_INFO)ul2,(USHORT)ul1); break;
        case DBE_FD_QUERY:      rc = FdQueryChar ((PFONT_QUERY)ul2,(USHORT)ul1,(USHORT)ul3); break;
        case DBE_FD_GET_VERSION:rc = FdGetVersion ((PFONT_DRIVER_VERSION)ul2); break;
        case DBE_FD_OPEN:       rc = FdOpen (); break;
        case DBE_FD_CLOSE:      rc = FdClose (); break;
        default:                rc = DBE_FD_ERROR; break;
    }
    DosSemClear ((HSEM)&fdsem);
    return rc;
}

