/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Microsoft Corporation, 1989                                 */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************
 *
 * SOURCE FILE NAME = VDHMODE.c
 *
 * DESCRIPTIVE NAME = Base video device handlers - Mode, CLUT, Palette
 *
 *
 * VERSION      V2.0
 *
 * DATE
 *
 * DESCRIPTION  This source file contains VDH entry points
 *              which get/set the video mode.
 *
 *              NOTE: These routines execute as ring 2 conforming
 *
 * FUNCTIONS    GetMode,         SetMode
 *              SetHWMode, SetEnvMode, GetModeIndex
 *
 * NOTES        NONE
 *
 * STRUCTURES   NONE
 *
 * EXTERNAL REFERENCES
 *
 *              NONE
 *
 * EXTERNAL FUNCTIONS  SaveRegs, RestoreRegs
 *                     AccessHardware, AccessCLUT
 *                     PhysToUVirt, FreePhysToUVirt
 *
*/

/*
 +----------------------------------------------------------------------------+
 |  Include files                                                             |
 +----------------------------------------------------------------------------+
*/
#define INCL_BASE               /* ALL of OS/2 Base                */
#include <os2.h>

#include "vdhctl.h"             /* Conditional compilation control */
#include "vdh.h"                /* Type definitions                */
#include "fntcalls.h"           /* FNTCALLS definition         */   //J-TS00

/*
 +----------------------------------------------------------------------------+
 |  Externally defined global variables                                       |
 +----------------------------------------------------------------------------+
*/
extern VIDEOMODE  Modes[];                  /* Supported modes                */
extern MEMORYMAPS MemoryMaps[];             /* Memory map info for each mode  */
extern USHORT IStart;                       /* Mode table start index         */
extern UCHAR READABLE;
extern ULONG PartialSaveSize;   /* Room required to save entire PVB in the popup mode */
extern USHORT VGA_PRESENT;      /* TRUE if VGA VDH has been installed */
extern USHORT (APIENTRY *ChainedCallVectorTable[MaxFn])();           /* @T24 */
extern VIDEOHARDWARE VideoHardware;                                     /*@T39*/
extern USHORT OEMFlags;                                                 //MS27
extern USHORT SVGAPresent;                                              // @drw
extern HW_DEFAULT HW_Default[];            /* Default HW Data Table      J-KKJ*/
extern SEL   CommonSelector;                /* Common Sel to XVIO     J-KK0929*/
extern USHORT CnvPalette[];                                             //J-TS00
/*
 +----------------------------------------------------------------------------+
 |  Parameters to ring 2 routines                                             |
 +----------------------------------------------------------------------------+
*/
extern CLUTDATA far ColorCLUT;
extern CLUTDATA far MonoCLUT;
extern CLUTDATA far SumCLUT;

extern CLUTDATA far *LCLUT;            /* @BB9 */                      /*@S40f*/

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: GetMode                                            */
/*                                                                      */
/*  DESCRIPTIVE NAME: Get current video mode setting                    */
/*                                                                      */
/*  FUNCTION: GetMode is called by BVS to get the current mode setting. */
/*            If the request specifies hardware and the hardware is     */
/*            readable, the actual hardware setting will be read and    */
/*            returned.  Otherwise the returned information will be     */
/*            taken from the environment buffer, if it has been passed. */
/*                                                                      */
/*  ENTRY POINT: GetMode                                                */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 271 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     VIOMODEINFO FAR *ModeDataPTR                     */
/*             ULONG Function ( Call vector table entry = 271 )         */
/*         (Referenced)                                                 */
/*             Modes[] (global data - table of supported video modes )  */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Current mode setting is returned                       */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  EFFECTS: If hardware specified and hardware is readable, the        */
/*           environment buffer is updated, if passed.                  */
/*           If the length of ModeData does not exactly fit a           */
/*           parameter, the length is adjusted and returned.            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: SaveRestoreHW, SetEnvMode                               */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY GetMode( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_MODE far *ParmBlock;
ULONG Function;
{
USHORT rc,
       ModeIndex,
       EnvBufferPassed;
UCHAR i,
      MapMask,
      RegValue;
USHORT NumBitplanes;
REGDATA RegData;
REGADDRESS RegAddress;
ENVIRONMENT far *TempEnv;
SEL Selector;
UCHAR TempChar;                         /* @tb21                            */

VIDEOMODE   far *pCurMode;              /* Reduce level of indirection, @S8 */
VIOMODEINFO far *pReqMode;              /* Reduce level of indirection, @S8 */
unsigned register ReqModeLen;           /* Eliminate segment loading, @S8 */

int SumCLUT_found,ii,j;                 /*@BB9*/

/*SaveRegs();*/ /*@B14*/                      /* Preserve registers and flags */

rc = ERROR_VIO_INVALID_PARMS;                          /* Initialize to error */
EnvBufferPassed = SEG( Environment );                  /* Non-zero = TRUE     */

if ( EnvBufferPassed &&
     !Environment->NATIVE_MODE &&
     VGA_PRESENT ) {                /* VGA Mode */                      //J-TS00
                                                                        //J-TS00
    rc = ChainedVDHGetMode(                                             //J-TS00
                 (ENVIRONMENT far *)&Environment->VGAEnvironment,       //J-TS00
                      ParmBlock, Function              );               //J-TS00
                                                                        //J-TS00
} else {                            /* ATLAS Mode */                    //J-TS00

if ( ( Function == FnGetMode ) &&                      /* Valid function request */
     ( ParmBlock->Length >= sizeof( VDH_MODE ) ) &&    /* Valid packet length */
     ( ParmBlock->Flags  <= 1  )         &&            /* Valid flags         */
     ( EnvBufferPassed || ( ParmBlock->Flags & UPDATE_HARDWARE ) ) ) {

  rc = NO_ERROR;                                       /* Initialize no error */

  ReqModeLen = (pReqMode = ParmBlock->ModeDataPTR)->cb; /*@S8*/

  if ( ReqModeLen > sizeof(VIOMODEINFO) ) {     /*@S25*/
//J-KK1016     ReqModeLen = MinDLen_Mode_Vres;            /*@S25*/
     ReqModeLen = MinDLen_Mode_Attrib;                   //J-KK1016
  }                                             /*@S25*/

  if ( ReqModeLen < Mode_Return_Length ) /* @@A */
     rc = ERROR_VIO_INVALID_LENGTH;      /* @@A */

  else if ( ReqModeLen == Mode_Return_Length ) /* @@A */
     ReqModeLen = sizeof( VIOMODEINFO ); /* @@A */
  else {                                                             /* @@A */

/*
 +----------------------------------------------------------------------------+
 |  If requesting application is running in foreground,                       |
 |    - Return hardware value if hardware can be read                         |
 |    - Otherwise return environment buffer value                             |
 |     ( report error if write-only hardware and no env buffer passed )       |
 |  If requesting application is running in background,                       |
 |    - Return environment buffer value                                       |
 |     ( report error if environment buffer was not passed )                  |
 +----------------------------------------------------------------------------+
*/
/*
 +----------------------------------------------------------------------------+
 |  Requesting application is running in the "background"                     |
 +----------------------------------------------------------------------------+
*/
    if ( EnvBufferPassed )
/*
 +----------------------------------------------------------------------------+
 |  Retrieve mode setting from environment buffer                             |
 +----------------------------------------------------------------------------+
*/
      ModeIndex = Environment->ModeIndex;
    else
      rc = ERROR_VIO_INVALID_PARMS;    /* Not FG OR write-only, no env buffer */

  if ( !rc ) {
    pCurMode = (VIDEOMODE far *)&Modes[ModeIndex].cb; /*@S8*/
/*
 +----------------------------------------------------------------------------+
 |  Mode type: xxxxxxxb b = 0 Monochrome,  b = 1 Other                        |
 |             xxxxxxbx b = 0 Text,        b = 1 Graphics                     |
 |             xxxxxbxx b = 0 Color burst, b = 1 Color burst disabled         |
 |             xxxxbxxx b = 0 Normal mode, b = 1 Advanced function ( native ) |
 +----------------------------------------------------------------------------+
*/


       pReqMode->fbType = Environment->ModeData.fbType;

    if ( ReqModeLen < MinDLen_Mode_Color )
      ReqModeLen = MinDLen_Mode_Type;
    else {
/*
 +----------------------------------------------------------------------------+
 |  Color ( power of 2 )                                                      |
 +----------------------------------------------------------------------------+
*/
      pReqMode->color = pCurMode->color;

      if ( ReqModeLen < MinDLen_Mode_Column )
        ReqModeLen = MinDLen_Mode_Color;
      else {
/*
 +----------------------------------------------------------------------------+
 |  Number of text columns: 40 or 80                                          |
 +----------------------------------------------------------------------------+
*/
        pReqMode->col = pCurMode->col;

        if ( ReqModeLen < MinDLen_Mode_Row )
          ReqModeLen = MinDLen_Mode_Column;
        else {
/*
 +----------------------------------------------------------------------------+
 |  Number of text rows                                                       |
 +----------------------------------------------------------------------------+
*/
          pReqMode->row = pCurMode->row;
          if ( EnvBufferPassed && !(ParmBlock->Flags & UPDATE_HARDWARE) )
             pReqMode->row = Environment->ModeData.row;

          if ( ReqModeLen < MinDLen_Mode_Hres )
            ReqModeLen = MinDLen_Mode_Row;
          else {
/*
 +----------------------------------------------------------------------------+
 |  Horizontal resolution                                                     |
 +----------------------------------------------------------------------------+
*/
            pReqMode->hres = pCurMode->hres;

            if ( ReqModeLen < MinDLen_Mode_Vres )
              ReqModeLen = MinDLen_Mode_Hres;
            else {
/*
 +----------------------------------------------------------------------------+
 |  Vertical resolution                                                       |
 +----------------------------------------------------------------------------+
*/
              pReqMode->vres = pCurMode->vres;

              if ( ReqModeLen < MinDLen_Mode_FormatID )
                ReqModeLen = MinDLen_Mode_Vres;
              else {
/*
 +----------------------------------------------------------------------------+
 |  Attribute format ID                                                       |
 +----------------------------------------------------------------------------+
*/
                if ( EnvBufferPassed )                          /*      @P1 */
                  pReqMode->fmt_ID = Environment->ModeData.fmt_ID; /* @P1 */
                else                                            /*      @P1 */
                  pReqMode->fmt_ID = pCurMode->fmt_ID;

                if ( ReqModeLen < MinDLen_Mode_Attrib )
                  ReqModeLen = MinDLen_Mode_FormatID;
                else {
/*
 +----------------------------------------------------------------------------+
 |  Attribute                                                                 |
 +----------------------------------------------------------------------------+
*/
                  if ( EnvBufferPassed )                        /*      @P1 */
                    pReqMode->attrib = Environment->ModeData.attrib; /* @P1 */
                  else                                          /*      @P1 */
                    pReqMode->attrib = pCurMode->attrib;

                  if ( ReqModeLen < MinDLen_Mode_BufStart )
                    ReqModeLen = MinDLen_Mode_Attrib;
                  else {
/*
 +----------------------------------------------------------------------------+
 |  Physical display buffer start address                                     |
 +----------------------------------------------------------------------------+
*/

                    pReqMode->BufferAddress =
                          MemoryMaps[pCurMode->MemMap].Start.FullAddress;

                    if ( ReqModeLen < MinDLen_Mode_BufLen )
                      ReqModeLen = MinDLen_Mode_BufStart;
                    else {
/*
 +----------------------------------------------------------------------------+
 |  Physical display buffer length                                            |
 +----------------------------------------------------------------------------+
*/

                      pReqMode->BufferLength =
                                               0x10000L;                //J-TS00


                      if ( ReqModeLen < MinDLen_Mode_FullBufSz )
                        ReqModeLen = MinDLen_Mode_BufLen;
                      else {
/*
 +----------------------------------------------------------------------------+
 |  Full physical display buffer size                                         |
 +----------------------------------------------------------------------------+
*/
                        pReqMode->FullBufferSize =
                                pReqMode->fbType & GRAPHICS
                              ? MemoryMaps[pCurMode->MemMap].TotalSize
                              : pReqMode->col
                              * pReqMode->row
                              * (pReqMode->attrib+1);
/*
 +----------------------------------------------------------------------------+
 |  Add size of font buffer only if a text mode                               |
 +----------------------------------------------------------------------------+
*/

                        if (ModeIndex <= ModeIndex_ATLASUS) {         /*J-KKJ2*/
                          pReqMode->FullBufferSize += 0x1000;          /*J-KKJ*/
                        } /* endif */                                  /*J-KKJ*/



                        if ( ReqModeLen < MinDLen_Mode_PartBufSz )
                          ReqModeLen = MinDLen_Mode_FullBufSz;
                        else {
/*
 +----------------------------------------------------------------------------+
 |  Full physical display buffer size of popup mode                           |
 +----------------------------------------------------------------------------+
*/
                          pReqMode->PartialBufferSize = PartialSaveSize;
                          if ( pReqMode->attrib == WorldAttrCount) /* @P1 */
                              pReqMode->PartialBufferSize *= WorldAttrMult;

                          if ( ReqModeLen < MinDLen_Mode_ExtData )
                            ReqModeLen = MinDLen_Mode_PartBufSz;
                          else {
/*
 +----------------------------------------------------------------------------+
 |  Pointer to extended mode data area                                        |
 +----------------------------------------------------------------------------+
*/
                            pReqMode->ExtDataArea = (UCHAR far *)NULL;

                            ReqModeLen = MinDLen_Mode_ExtData;
                            }
                          }  /* else get PartialBufferSize */
                        }    /* else get FullBufferSize */
                      }      /* else get BufferLength */
                    }        /* else get BufferAddress */
                  }          /* else get Attribute */
                }            /* else get Format ID */
              }              /* else get Vres */
            }                /* else get Hres */
          }                  /* else get rows */
        }                    /* else get columns */
      }                      /* else get color */
    }  /* if !rc */
  }    /* else good mode length         @@A */
  }    /* if good parameters */

if (!rc) {                              /*@S8*/
    pReqMode->cb = ReqModeLen;          /*@S8*/
}                                       /*@S8*/

}   /* end of atlas mode */                                             //J-TS00

/*RestoreRegs();*/ /*@B14*/              /* Restore registers and flags */
return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: SetMode                                            */
/*                                                                      */
/*  DESCRIPTIVE NAME: Set video mode                                    */
/*                                                                      */
/*  FUNCTION: SetMode is called by BVS to set the video mode.           */
/*            If the request specifies hardware, the hardware and the   */
/*            environment buffer, if passed, will be updated.           */
/*            Otherwise just the environment buffer, if passed, will    */
/*            be updated.                                               */
/*                                                                      */
/*  ENTRY POINT: SetMode                                                */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 272 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     VIOMODEINFO FAR *ModeDataPTR                     */
/*             ULONG Function ( Call vector table entry = 272 )         */
/*         (Referenced)                                                 */
/*             Modes[] (global data - table of supported video modes )  */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Video mode is set                                      */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: SetHWMode                                               */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY SetMode( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_MODE far *ParmBlock;
ULONG Function;
{
USHORT rc,
       i,
       tempflags,
       MODE_FOUND,
       EnvBufferPassed;

VIDEOMODE far   *pCurMode;              /* Reduce level of indirection, @S8 */
VIOMODEINFO far *pReqMode;              /* Reduce level of indirection, @S8 */
unsigned register ReqModeLen;           /* Eliminate segment loading, @S8 */

#ifdef  XVIO
USHORT       xrc;                                                   /*J-KK0804*/
XVS_SUSPEND  XvioSusp;                                              /*J-KK0804*/
XVS_RESUME   XvioResm;                                              /*J-KK0804*/
XVS_SETMODE  XvioMode;                                              /*J-KK0804*/
XVS_SETSTATE XvioState;                                             /*J-KK0928*/
#endif

/*SaveRegs();*/ /*@B14*/                      /* Preserve registers and flags */

rc = ERROR_VIO_INVALID_PARMS;                          /* Initialize to error */
EnvBufferPassed = SEG( Environment );                  /* Non-zero = TRUE     */

if ( ( Function == FnSetMode )                           && /* Valid function request */
//J-TS0616     ( ParmBlock->Flags  <= 0x07 )                       && /* Valid flags         */
     ( (ParmBlock->Flags & ~0x4000) <= 0x07 )            && /* Valid flags         */   //J-TS0616
//@drw     ( ParmBlock->Flags  <= 1  )                         && /* Valid flags         */
     ( EnvBufferPassed || ( ParmBlock->Flags & UPDATE_HARDWARE ) ) ) {

 ReqModeLen = (pReqMode = ParmBlock->ModeDataPTR)->cb; /*@S8*/

 rc = ERROR_VIO_INVALID_LENGTH;                   /* @@TB1 Initialize to error */
 if (( ParmBlock->Length >= sizeof( VDH_MODE ) ) &&  /* @@TB1  Valid packet length */
     ( ReqModeLen >= MinDLen_Mode_Type )) /* @@TB1 Valid mode length   */
   {

#ifdef  XVIO                                                            //J-TS00
//-----------------------------------------------------------------     //J-TS00
// BXVS Call Suspend                                                    //J-TS00
//-----------------------------------------------------------------     //J-TS00
  XvioSusp.Length  = sizeof(XvioSusp);                                  //J-TS00
  XvioSusp.Reserve = NoOption;                                          //J-TS00
  XvioSusp.Session = (UCHAR)Environment->SessionNum;                    //J-TS00
  xrc = XVIOVDHIF( XvsSuspend, (PUSHORT)&XvioSusp );                    //J-TS00
#endif                                                                  //J-TS00
                                                                        //J-TS00
  if ( ( ( ( Environment->DBCSEvBuff[0] == 0 ) &&     /* SBCS     */    //J-TS00
           ( Environment->DBCSEvBuff[1] == 0 ) ) ||   /* SBCS     */    //J-TS00
         ( pReqMode->fbType & GRAPHICS ) ) &&         /* Graphics */    //J-TS00
       VGA_PRESENT ) {                                /* VGA Exists */  //J-TS00
                                                                        //J-TS00
    rc = ChainedVDHSetMode(                                             //J-TS00
                      (ENVIRONMENT far *)&Environment->VGAEnvironment,  //J-TS00
                      ParmBlock, Function );                            //J-TS00
  }                                                                     //J-TS00
                                                                        //J-TS00
  if (!rc) {                                                            //J-TS00
              // If the request mode is accepted by BVHVGA,             //J-TS00
              // Japanese ModeIndex is changed to the indefinite value. //J-TS00
    Environment->ModeIndex   = 0xFF;                                    //J-TS00
    Environment->NATIVE_MODE = FALSE;       /* To VGA Mode */           //J-TS00
    CursorOff();                            /* set hidden cursor */     //J-TS00
                                                                        //J-TS00
#ifdef  XVIO                                                            //J-TS00
//------------------------------------------------------------------    //J-TS00
// BXVS Call SetMode                                                    //J-TS00
//------------------------------------------------------------------    //J-TS00
    if (( pReqMode->fbType & GRAPHICS ) &&  /* XVIO supported */        //J-TS00
        ( pReqMode->hres == 640) &&         /* graph mode     */        //J-TS00
        ( pReqMode->vres == 480)) {                                     //J-TS00
//-----------------------------------------------------------------     //J-TS00
// BXVS Call SetMode ( Japanese graphics mode)                          //J-TS00
//-----------------------------------------------------------------     //J-TS00
      XvioMode.Length              = sizeof(XvioMode);                  //J-TS00
      XvioMode.Option              = NoOption;                          //J-TS00
      if ( ParmBlock->Flags & UPDATE_HARDWARE )     // Foreground       //J-TS00
        XvioMode.Option           |= FG_Session;                        //J-TS00
      XvioMode.fbType              = pReqMode->fbType;                  //J-TS00
      XvioMode.color               = pReqMode->color;                   //J-TS00
      XvioMode.col                 = 80;                                //J-TS00
      XvioMode.row                 = 25;                                //J-TS00
      XvioMode.hres                = pReqMode->hres;                    //J-TS00
      XvioMode.vres                = pReqMode->vres;                    //J-TS00
      XvioMode.fmt_ID              = pReqMode->fmt_ID;                  //J-TS00
      XvioMode.attrib              = pReqMode->attrib;                  //J-TS00
      XvioMode.BufferAddress       = (UCHAR  far *)0xa0000L;            //J-TS00
      XvioMode.BufferLength        = 0x10000L;                          //J-TS00
      XvioMode.SystemReserve       = SystemReserveLine;                 //J-TS00
      XvioMode.HWMode              = 0;                                 //J-TS00
      XvioMode.Default_CursorColor = 0x0f;                              //J-TS00
      XvioMode.Session             = (UCHAR)Environment->SessionNum;    //J-TS00
      xrc = XVIOVDHIF( XvsSetMode, (PUSHORT)&XvioMode );                //J-TS00
    } else {                                                            //J-TS00
//-----------------------------------------------------------------     //J-TS00
// BXVS Call SetMode ( U.S. text Mode )                                 //J-TS00
//-----------------------------------------------------------------     //J-TS00
      XvioMode.Length    = sizeof(XvioMode);                            //J-TS00
      XvioMode.Option    = NoOption;                                    //J-TS00
      if ( ParmBlock->Flags & UPDATE_HARDWARE )   // Foreground         //J-TS00
        XvioMode.Option |= FG_Session;                                  //J-TS00
      XvioMode.HWMode    = -1;                                          //J-TS00
      XvioMode.Session   = (UCHAR)Environment->SessionNum;              //J-TS00
      xrc = XVIOVDHIF( XvsSetMode, (PUSHORT)&XvioMode );                //J-TS00
    }                                                                   //J-TS00
#endif                                                                  //J-TS00
  }                                                                     //J-TS00
                                                                        //J-TS00
#ifdef  XVIO                                                            //J-TS00
//------------------------------------------------------------------    //J-TS00
// BXVS Call Resume                                                     //J-TS00
//------------------------------------------------------------------    //J-TS00
  XvioResm.Length  = sizeof(XvioResm);                                  //J-TS00
  XvioResm.Reserve = NoOption;                                          //J-TS00
  XvioResm.Session = (UCHAR)Environment->SessionNum;                    //J-TS00
  xrc = XVIOVDHIF( XvsResume, (PUSHORT)&XvioResm );                     //J-TS00
#endif                                                                  //J-TS00
                                                                        //J-TS00
  if (rc) {                                     /* native mode */       //J-TS00
                                                                        //J-TS00
/*
 +----------------------------------------------------------------------------+
 |  Find the mode based upon matching specified parameters with entry in the  |
 |  table of supported modes.  If a partial parameter list is provided, use   |
 |  the highest resolution mode that matches the provided parameters.         |
 +----------------------------------------------------------------------------+
*/

  rc = ERROR_VIO_MODE;                                 /* Initialize to error */
  tempflags = ParmBlock->Flags;
  if ( pReqMode->fbType & NO_CLR_BRST ) {
     tempflags |= NO_CLR_BRST; /* Indicate B/W Mode */
  }


// Check for COMMON Mode                                              /*J-KKJ1*/
 if (( ReqModeLen >= MinDLen_Mode_Attrib ) &&                         /*J-KKJ1*/
     ( pReqMode->fmt_ID == WorldFormat)    &&                         /*J-KKJ1*/
     ( pReqMode->attrib == WorldAttrCount) ) {                        /*J-KKJ1*/
                                                                      /*J-KKJ1*/
    pCurMode = (VIDEOMODE far *)&Modes[ModeIndex_ATLASCOMMON].cb;     /*J-KKJ1*/
    i = ModeIndex_ATLASCOMMON + 1;                                    /*J-KKJ1*/
    MODE_FOUND = TRUE;                                                /*J-KKJ1*/

 }                                                                    /*J-KKJ1*/
 else {   /* Not COMMON Mode */                                       /*J-KKJ1*/

  for (i = IStart, MODE_FOUND = FALSE; (i < NUM_MODES) && !MODE_FOUND; i++) {

//                                                                      J-KK0117
// Must not match with the Common Mode Table.                           J-KK0117
// Because that mode has already been checked.                          J-KK0117
//                                                                      J-KK0117
   if ( i == ModeIndex_ATLASCOMMON ) i++;     // Skip !!!               J-KK0117

    pCurMode = (VIDEOMODE far *)&Modes[i].cb; /*@S8*/

/*
 +----------------------------------------------------------------------------+
 |  Verify that the mode is valid by finding it in the mode table             |
 +----------------------------------------------------------------------------+
*/
    if ( pCurMode->fbType == (pReqMode->fbType & ~NO_CLR_BRST ) ) /* @S6 */
/*
 +----------------------------------------------------------------------------+
 |  Type matches - If color specified, check for match. Otherwise             |
 |                 default color, col, row, hres, vres to this mode           |
 +----------------------------------------------------------------------------+
*/
      if ( ReqModeLen < MinDLen_Mode_Color )
        MODE_FOUND = TRUE;                      /* Found a matching mode */
      else
        if ( pReqMode->color == pCurMode->color )

/*
 +----------------------------------------------------------------------------+
 |  Color matches - If col specified, check for match.  Otherwise             |
 |                  default col, row, hres, vres to this mode                 |
 +----------------------------------------------------------------------------+
*/
          if ( ReqModeLen < MinDLen_Mode_Column )
            MODE_FOUND = TRUE;               /* Found a matching mode */
          else
            if ( (pReqMode->col == pCurMode->col) ||
                  (pReqMode->fbType & GRAPHICS) )

/*
 +----------------------------------------------------------------------------+
 |  col matches - If row specified, check for match. Otherwise                |
 |                default row, hres and vres to this mode                     |
 +----------------------------------------------------------------------------+
*/

              if ( ReqModeLen < MinDLen_Mode_Hres )
                MODE_FOUND = TRUE;              /* Found a matching mode */
              else
                if ( pReqMode->hres == pCurMode->hres )
/*
 +----------------------------------------------------------------------------+
 |  hres matches - If vres specified, check for match.                        |
 |                 Otherwise default vres to this mode                        |
 +----------------------------------------------------------------------------+
*/
                if ( ReqModeLen < MinDLen_Mode_Vres )
                  MODE_FOUND = TRUE;            /* Found a matching mode */
                else
                  if ( pReqMode->vres == pCurMode->vres )
//J-KKJ             MODE_FOUND = TRUE;          /* Found a matching mode */

//+----------------------------------------------------------------------------+
//|  vres matches - If FormatID specified, check for match.                    |
//|                 Otherwise default vres to this mode                        |
//+----------------------------------------------------------------------------+

                if ( ReqModeLen < MinDLen_Mode_FormatID )              /*J-KKJ*/
                  MODE_FOUND = TRUE;        /* Found a matching mode *//*J-KKJ*/
                else                                                   /*J-KKJ*/
                  if ( pReqMode->fmt_ID == pCurMode->fmt_ID )          /*J-KKJ*/

//+----------------------------------------------------------------------------+
//|  FormatID matches - If AttrNum specified, check for match.                 |
//|                 Otherwise default vres to this mode                        |
//+----------------------------------------------------------------------------+

                if ( ReqModeLen < MinDLen_Mode_Attrib )                /*J-KKJ*/
                  MODE_FOUND = TRUE;        /* Found a matching mode *//*J-KKJ*/
                else                                                   /*J-KKJ*/
                  if ( pReqMode->attrib == pCurMode->attrib )          /*J-KKJ*/
                    MODE_FOUND = TRUE;      /* Found a matching mode *//*J-KKJ*/

/*---------------------------------------------------------------------------*/
/* If the everything matches so far, call findfont if the rows don't match   */
/* and not graphics mode.  If no font found, set a temporary error code.     */
/* If no font found and buffer not large enough, reset !MODE_FOUND true.  */
/* @@TB1                                                                     */
/*---------------------------------------------------------------------------*/

  if (  MODE_FOUND   &&  ( ReqModeLen >= MinDLen_Mode_Row ))
   if(pReqMode->row > 0){                                              /*@B22*/
     if (!(pReqMode->fbType & GRAPHICS) &&
        ( pReqMode->row != pCurMode->row))
       {
        MODE_FOUND = FALSE;
       }
   }                                                                   /*@B22*/
   else MODE_FOUND = FALSE;                                            /*@B22*/

  } /* End of For that searches the Mode table */
 } /* Check for COMMON Mode */                                        /*J-KKJ1*/

  if ( MODE_FOUND ) {

// Adjust for Epoch Modes.                                            /*J-KKJ1*/
// For EPOCH modes, allow other two sets of data to be set.           /*J-KKJ1*/
    if (( (i-1) == ModeIndex_EPOCHMONO80A )||                         /*J-KKJ1*/
        ( (i-1) == ModeIndex_EPOCHMONO80B )) {                        /*J-KKJ1*/
         i = ModeIndex_EPOCHMONO80 + 1;                               /*J-KKJ1*/
         pCurMode = (VIDEOMODE far *)&Modes[ModeIndex_EPOCHMONO80].cb;/*J-KKJ1*/
     }                                                                /*J-KKJ1*/
    if (( (i-1) == ModeIndex_EPOCHCOLR80A )||                         /*J-KKJ1*/
        ( (i-1) == ModeIndex_EPOCHCOLR80B )) {                        /*J-KKJ1*/
         i = ModeIndex_EPOCHCOLR80 + 1;                               /*J-KKJ1*/
         pCurMode = (VIDEOMODE far *)&Modes[ModeIndex_EPOCHCOLR80].cb;/*J-KKJ1*/
     }                                                                /*J-KKJ1*/

/*
 +----------------------------------------------------------------------------+
 |  Parameters are valid - Update environment buffer if it was passed         |
 |                       - Update hardware if in foreground                   |
 +----------------------------------------------------------------------------+
*/
    if ( EnvBufferPassed ) {

      Environment->NATIVE_MODE = TRUE;        /* To VGA Mode */             //J-TS00

/*
 +----------------------------------------------------------------------------+
 |  Put all of the caller's parameters into the environment buffer            |
 |  Fill in missing parameters with defaults in the mode table                |
 +----------------------------------------------------------------------------+
*/
      Environment->ModeIndex       = (UCHAR)i-1;                      /*J-KKJ1*/
      Environment->ModeData.cb     = MinDLen_Mode_Attrib;             /*J-KKJ1*/

      Environment->ModeData.fbType = pCurMode->fbType;                /*J-KKJ1*/
      if ( tempflags & NO_CLR_BRST) {                               /*J-KK1005*/
         Environment->ModeData.fbType |= NO_CLR_BRST;               /*J-KK1005*/
      } /* endif */                                                 /*J-KK1005*/

      Environment->ModeData.color  = pCurMode->color;                 /*J-KKJ1*/
      Environment->ModeData.col    = pCurMode->col;                   /*J-KKJ1*/
      Environment->ModeData.row    = pCurMode->row;                   /*J-KKJ1*/
      Environment->ModeData.hres   = pCurMode->hres;                  /*J-KKJ1*/
      Environment->ModeData.vres   = pCurMode->vres;                  /*J-KKJ1*/
      Environment->ModeData.fmt_ID = pCurMode->fmt_ID;                /*J-KKJ1*/
      Environment->ModeData.attrib = pCurMode->attrib;                /*J-KKJ1*/

      Environment->ModeData.BufferAddress =                           /*J-KKJ1*/
                     MemoryMaps[pCurMode->MemMap].Start.FullAddress;  /*J-KKJ1*/

      Environment->ModeData.BufferLength =                            /*J-KKJ1*/
                  MemoryMaps[pCurMode->MemMap].TotalSize;             /*J-KKJ1*/

      Environment->ModeData.ExtDataArea = (UCHAR far *)NULL;
      }

/*
 +----------------------------------------------------------------------------+
 |  If Flags = UPDATE_HARDWARE, SetHWMode will set the hardware registers     |
 |  If environment buffer is passed, SetHWMode will shadow the registers      |
 +----------------------------------------------------------------------------+
*/
    SetHWMode( (UCHAR)i-1, tempflags, Environment );


    rc = NO_ERROR;                     /* Signal success */
    }

   }                                        /* end ChainedVDHSetMode */ //J-TS00
                                               /* end check for invalid length*/
   }                                           /* @@TB1                       */

  }                                            /* end check for invalid parms */
                                               /* @@TB1                       */
  if (rc == NO_ERROR) {                                         /*      @P1 */
    if ( EnvBufferPassed ) {
        Environment->ScrollRect.Right = Environment->ModeData.col - 1; /* @P1 */
//J-KK0904  Scrollable Row Size is not changed by SetMode.
//J-KK0904        Environment->ScrollRect.Bottom = Environment->ModeData.row - 1;/* @P1 */
        if ( Environment->ModeData.fmt_ID == WorldFormat )      /*     @P1 */
            Environment->AttrBufSize = 3;                       /*     @P1 */
        else                                                    /*     @P1 */
            Environment->AttrBufSize = 1;                       /*     @P1 */

      /* To Clear LVB */                                  /*J-KKJ3*/
        if ( !( Environment->ModeData.fbType & GRAPHICS ) &&          //J-KK1117
                Environment->LVB_Selector ) {                         //J-KK1117
         if (!(rc = CLEAR_LVB(  Environment->FormatIndex,             //J-KK1117
                          Environment->LVB_Selector,                  //J-KK1117
                          Environment->ScreenSize                     //J-KK1117
                     )))                                              //J-KK1117
           Environment->FormatChanged = FALSE;                        //J-KK1117
         rc = NO_ERROR;                     /* Signal success */      //J-KK1117
        } else {                                                        //J-TS00
           Environment->FormatChanged = TRUE;                           //J-TS00
        }                                                               //J-TS00
    }                                                           /*     @P1 */
  }                                                             /*     @P1 */
/*RestoreRegs();*/ /*@B14*/              /* Restore registers and flags */
return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: SetHWMode                                          */
/*                                                                      */
/*  DESCRIPTIVE NAME: Set the hardware mode                             */
/*                                                                      */
/*  FUNCTION: SetHWMode is called to set the hardware to the supported  */
/*            video mode. If Flags = UPDATE_HARDWARE, the registers     */
/*            are set.  If environment buffer is passed, the registers  */
/*            are shadowed.                                             */
/*                                                                      */
/*  ENTRY POINT: SetHWMode                                              */
/*    LINKAGE:   CALL FAR                                               */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             UCHAR Mode ( Index in table of supported modes )         */
/*         (Referenced)                                                 */
/*             Modes[] (global data - table of supported video modes )  */
/*             Fonts[] (global data - table of ROM font areas )         */
/*                                                                      */
/*  EXIT-NORMAL: Video mode is set on hardware                          */
/*                                                                      */
/*  EFFECTS: NONE                                                       */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: SetDefaultCLUT, PhysToUVirt, FreePhysToUVirt            */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: AccessHardware, AccessRegister                          */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
void PASCAL near SetHWMode( Mode, Flags, Environment ) /*@B15*/
UCHAR Mode;
USHORT Flags;
ENVIRONMENT far *Environment;
{
USHORT EnvBufferPassed,
       Ind,
       ColorMode,
       VideoEnable,
       VideoOn,                                                         /*@C21*/
       JumpCmd,
       i, j, k, m;
REGADDRESS RegAddress;
REGDATA    RegData;

CLUTDATA far *CLUT;
REGDATA ClutParms;

USHORT Id, rc;                                                          //J-TS00
extern USHORT BlinkOption;                                              //J-TS00

#ifdef  XVIO
VIDEOMODE far  *pCurMode;                                           /*J-KK0815*/
USHORT          xrc;                                                /*J-KK0815*/
XVS_SUSPEND     XvioSusp;                                           /*J-KK0815*/
XVS_RESUME      XvioResm;                                           /*J-KK0815*/
XVS_SETMODE     XvioMode;                                           /*J-KK0815*/
XVS_REDRAW      XvioRedraw;                                         /*J-KK0815*/
COMMONAREA far *PCommon;                                            /*J-KK0929*/
#endif

EnvBufferPassed = SEG( Environment );                  /* Non-zero = TRUE     */

//ͻ
//  SetUp Configuration Data                                            J-KK  
//ͼ
if ( EnvBufferPassed ) {

  if ( Flags & UPDATE_HARDWARE )                                        //J-TS00
    ForceToVGA( Environment );                                          //J-TS00

  if ( Mode <= ModeIndex_ATLASUS ) {
                /* For U.S. Emulation or VGA Graphic Mode */

    Environment->DisplayAdapter  = TYPE_VGA;
    Environment->DisplayType     = VideoHardware.display;               //J-TS00
    Environment->MemorySize      = 256L * 1024L;    /* VGA Memory Size 256K */
    Environment->PartialSaveSize = 80L * 25L * 2L;  /* 80x25 text PVB */
    Environment->MaxFullSaveSize = 38400L * 4L;     /* VGA mode 12 */

  } else {     /* DBCS except DBCS VGA */

    Environment->DisplayAdapter  = TYPE_ATLAS;
    switch (VideoHardware.display) {                                    //J-TS00
    case Color8512_8513:                                                //J-TS00
      Environment->DisplayType   = Color5574;                           //J-TS00
      break;                                                            //J-TS00
                                                                        //J-TS00
    case Mono8503:                                                      //J-TS00
      Environment->DisplayType   = Mono5574;                            //J-TS00
      break;                                                            //J-TS00
                                                                        //J-TS00
    case PlasmaDisplay:                                                 //J-TS00
      Environment->DisplayType   = PlasmaDisplay;                       //J-TS00
      break;                                                            //J-TS00
    }                                                                   //J-TS00
    Environment->MemorySize      = VideoHardware.memory;
    Environment->PartialSaveSize = 80L * 25L * 2L; /* 80x25 text PVB */
    Environment->MaxFullSaveSize =  65536L * 8L;   /* 1024x1024x4 bits */

  }

  Id = Modes[Mode].HW_Index;

/*+-----------------------------------------------------------------+   //J-TS00
  |  SetUp Default Cursor                                           |   //J-TS00
  +-----------------------------------------------------------------+*/ //J-TS00
  Environment->CursorRow       =                                        //J-TS00
  Environment->CursorCol       = 0;                                     //J-TS00
  Environment->CursorStart     = HW_Default[Id].CursorStart;            //J-TS00
  Environment->CursorEnd       = HW_Default[Id].CursorEnd;              //J-TS00
  Environment->CursorWidth     = HW_Default[Id].CursorWidth;            //J-TS00
  Environment->CursorAttribute = HW_Default[Id].CursorControl;          //J-TS00
  Environment->CursorColor     = HW_Default[Id].CursorColor;            //J-TS00

/*+-----------------------------------------------------------------+   //J-TS00
  |  SetUp Default Grid                                             |   //J-TS00
  +-----------------------------------------------------------------+*/ //J-TS00
                                                                        //J-TS00
  if ( Mode == ModeIndex_EPOCHMONO80 )   // Epoch Mono Text Mode        //J-TS00
                                                                        //J-TS00
    if ( Environment->First_Grid_Color )                                //J-TS00
      Environment->GridColor = First_Grid_Color_Value;                  //J-TS00
    else                                                                //J-TS00
      Environment->GridColor = CnvPalette[FG_Color];                    //J-TS00
                                                                        //J-TS00
  else                                                                  //J-TS00
    Environment->GridColor   = HW_Default[Id].GridColor;                //J-TS00
  SetGridDrawColor( Environment->GridColor );                           //J-TS00

#ifdef  XVIO
  if ( Environment->SessionNum != 0 ) {      // Not HardError Session   @KK0622
//ͻ
// Set Default Cursor Control and Length Register Value.            J-KK0927  
// At this time, COMMON buffer to Xvio is setuped.                            
//ͼ
// For CommonArea to Xvio

     if (!(rc = DosGetSeg(CommonSelector))) {                       /*J-KK0929*/
         PCommon = (COMMONAREA far *)MakeFarPTR( CommonSelector,    /*J-KK0929*/
                      Environment->SessionNum * sizeof(COMMONAREA));/*J-KK0929*/
         PCommon->CurLength  = HW_Default[Id].CursorWidth;          /*J-KK0929*/
         PCommon->CurControl = (UCHAR)(HW_Default[Id].CursorControl ?   //J-TS00
                                       0 : 1);                          //J-TS00
     } else {                                                       /*J-KK0929*/
         PCommon->CurLength  =               /* Error !!! */        /*J-KK0929*/
         PCommon->CurControl = 0;                                   /*J-KK0929*/
     } /* endif */                                                  /*J-KK0929*/
  }                     // Not Hard Error Session                       @KK0622
#endif

#ifdef  XVIO
//------------------------------------------------------------------/*J-KK0815*/
// BXVS Call Suspend                                                /*J-KK0815*/
//------------------------------------------------------------------/*J-KK0815*/
  XvioSusp.Length  = sizeof(XvioSusp);                              /*J-KK0815*/
  XvioSusp.Reserve = NoOption;                                      //@XVIO
  XvioSusp.Session = (UCHAR)Environment->SessionNum;                /*J-KK0926*/
  xrc = XVIOVDHIF( XvsSuspend, (PUSHORT)&XvioSusp );                /*J-KK0815*/

//------------------------------------------------------------------/*J-KK0815*/
// BXVS Call SetMode                                                /*J-KK0815*/
//------------------------------------------------------------------/*J-KK0815*/
  pCurMode = (VIDEOMODE far *)&Modes[Mode].cb;                      /*J-KK0815*/
  XvioMode.Length     = sizeof(XvioMode);                           /*J-KK0815*/
  XvioMode.Option     = NoOption;
  if ( Flags & UPDATE_HARDWARE )                // Foreground         J-KK0213
    XvioMode.Option  |= FG_Session;                                 //J-KK0213
  XvioMode.fbType     = pCurMode->fbType;                           /*J-KK0815*/
  if (( XvioMode.fbType & NOT_MONO ) &&  /* B&W or Color */         /*J-KK0927*/
      ( Flags & NO_CLR_BRST )        &&  /* B&W */                  /*J-KK0927*/
      !( XvioMode.fbType & GRAPHICS ) )  {                          /*J-KK0927*/
    XvioMode.fbType |= NO_CLR_BRST;                                 /*J-KK0927*/
  }                                                                 /*J-KK0927*/
  XvioMode.color     = pCurMode->color;                             /*J-KK0815*/
  XvioMode.col       = pCurMode->col;                               /*J-KK0815*/
  XvioMode.row       = pCurMode->row;                               /*J-KK0815*/
  XvioMode.hres      = pCurMode->hres;                              /*J-KK0815*/
  XvioMode.vres      = pCurMode->vres;                              /*J-KK0815*/
  XvioMode.fmt_ID    = pCurMode->fmt_ID;                            /*J-KK0815*/
  XvioMode.attrib    = pCurMode->attrib;                            /*J-KK0815*/
                                                                    /*J-KK0815*/
  XvioMode.BufferAddress =                                          /*J-KK0815*/
              MemoryMaps[pCurMode->MemMap].Start.FullAddress;       /*J-KK0815*/
  XvioMode.BufferLength  =                                          /*J-KK0815*/
              MemoryMaps[pCurMode->MemMap].TotalSize;               /*J-KK0815*/

  if (( XvioMode.col == 82 ) ||     // 82x25 Mode ?                 /*J-KK1017*/
      ( Mode == ModeIndex_Real ))   // Real Mode ?                  /*J-KK1017*/
    XvioMode.SystemReserve = 0;                                     /*J-KK0828*/
  else                                                              /*J-KK0828*/
    XvioMode.SystemReserve = SystemReserveLine;                     /*J-KK0828*/

  if ( XvioMode.fbType & GRAPHICS ) // Graphic Mode                   J-KK1016
    XvioMode.HWMode = 0;    // Means Current is the same as HW mode   J-KK1016
  else {                                                            //J-KK1016
    if ( Mode  < ModeIndex_Real ) XvioMode.HWMode = 1;              /*J-KK0815*/
    else                          XvioMode.HWMode = 0;              /*J-KK0815*/
  }                                                                 //J-KK1016

  XvioMode.Default_CursorColor                                      //J-KK1023
                  = HW_Default[pCurMode->HW_Index].CursorColor;     //J-KK1023

  XvioMode.Session = (UCHAR)Environment->SessionNum;                /*J-KK0926*/
  xrc = XVIOVDHIF( XvsSetMode, (PUSHORT)&XvioMode );                /*J-KK0815*/
#endif

} /* EnvBufferPassed */

//J-TS00 ColorMode = Modes[Mode].fbType & NOT_MONO;  /* Needed by AccessVideoEnable on non-VGA */
ColorMode = 1;  /* All video mode are emulated by color hardware mode J-TS00 */

/*
 +----------------------------------------------------------------------------+
 |  Turn the video signal off to reduce snow                                  |
 +----------------------------------------------------------------------------+
*/
if ( Flags & UPDATE_HARDWARE ) {

   if ( EnvBufferPassed )                                               /*@C21*/
      VideoOn = Environment->VideoEnable;                               /*@C21*/
   else                                                                 /*@C21*/
      AccessVideoEnable( ColorMode, GET, &VideoOn );                    /*@C21*/

   VideoEnable = 0; /* Turn video off */
   AccessVideoEnable( ColorMode, SET, &VideoEnable ); /* ring 2 callgate */
}

for ( i= 0, Ind= Modes[Mode].HW_Index; HW_Default[Ind].ModeRegs[i].Command; i++ ) {

  switch ( HW_Default[Ind].ModeRegs[i].Command ) {

/*
 +----------------------------------------------------------------------------+
 |  Set the attribute registers                                               |
 +----------------------------------------------------------------------------+
*/
    case Attributes_CMD:
                         if ( Flags & UPDATE_HARDWARE ) {
                           RegAddress.AddressPort = AttAddressPort;
                           RegAddress.DataPort    = AttDataWritePort;
                           RegAddress.ColorAdjust = NONE;
                           RegAddress.Flags       = Attributes_CMD;
                         }

/*
 +----------------------------------------------------------------------------+
 |  Enter Attribute registers into environment buffer                         |
 +----------------------------------------------------------------------------+
*/
                         if ( EnvBufferPassed ) {                       //J-TS00
                           for ( m = 0,
                                 k = HW_Default[Ind].ModeRegs[i].RegData.FirstEntry;
                                 m < HW_Default[Ind].ModeRegs[i].RegData.NumEntries;
                                 k++, m++ )
                             Environment->Hardware.Attributes.All[k] =
                                   HW_Default[Ind].ModeRegs[i].RegData.DataArea[k];

                           if ( Mode == ModeIndex_EPOCHMONO80 ) { /* For Epoch Mono Text Mode */        //J-TS00
                                                                                                        //J-TS00
                             Environment->Hardware.Attributes.Regs.Palettes[CnvPalette[FG_Color]]  =    //J-TS00
                                         Environment->Hardware.Fore_Color;                              //J-TS00
                                                                                                        //J-TS00
                             Environment->Hardware.Attributes.Regs.Palettes[CnvPalette[FGH_Color]] =    //J-TS00
                                         Environment->Hardware.Fore_Int;                                //J-TS00
                                                                                                        //J-TS00
                             Environment->Hardware.Attributes.Regs.Palettes[CnvPalette[Cur_Color]] =    //J-TS00
                                         Environment->Hardware.Fore_Color + 0x38;                       //J-TS00
                         //                                                                             //J-TS00
                         // Special Handling for PE2                                                    //J-TS00
                         // Make a cursor visible on a command line in PE2.                             //J-TS00
                         //         02h -> 3Ah                                                          //J-TS00
                           }                                            //J-TS00
                         }                                              //J-TS00
                         break;
/*
 +----------------------------------------------------------------------------+
 |  Set the sequencer registers                                               |
 +----------------------------------------------------------------------------+
*/
    case Sequencers_CMD:
                         if ( Flags & UPDATE_HARDWARE ) {
                           RegAddress.AddressPort = SeqAddressPort;
                           RegAddress.DataPort    = SeqDataPort;
                           RegAddress.ColorAdjust = NONE;
                           RegAddress.Flags       = Sequencers_CMD;
                         }

/*
 +----------------------------------------------------------------------------+
 |  Enter Sequencer registers into environment buffer                         |
 +----------------------------------------------------------------------------+
*/
                         if ( EnvBufferPassed )
                           for ( m = 0,
                                 k = HW_Default[Ind].ModeRegs[i].RegData.FirstEntry;
                                 m < HW_Default[Ind].ModeRegs[i].RegData.NumEntries;
                                 k++, m++ )
                             Environment->Hardware.Sequencers.All[k] =
                                   HW_Default[Ind].ModeRegs[i].RegData.DataArea[k];
                         break;
/*
 +----------------------------------------------------------------------------+
 |  Set the graphics registers                                                |
 +----------------------------------------------------------------------------+
*/
    case Graphics_CMD:
                         if ( Flags & UPDATE_HARDWARE ) {
                           RegAddress.AddressPort = GraphAddressPort;
                           RegAddress.DataPort    = GraphDataPort;
                           RegAddress.ColorAdjust = NONE;
                           RegAddress.Flags       = NONE;
                         }

/*
 +----------------------------------------------------------------------------+
 |  Enter Graphics registers into environment buffer                          |
 +----------------------------------------------------------------------------+
*/
                         if ( EnvBufferPassed )
                           for ( m = 0,
                                 k = HW_Default[Ind].ModeRegs[i].RegData.FirstEntry;
                                 m < HW_Default[Ind].ModeRegs[i].RegData.NumEntries;
                                 k++, m++ )
                             Environment->Hardware.GraphicsRegs[k] =
                                   HW_Default[Ind].ModeRegs[i].RegData.DataArea[k];
                         break;

/*
 +----------------------------------------------------------------------------+
 |  Set the CRT registers                                                     |
 +----------------------------------------------------------------------------+
*/
    case CRTCtlRegs_CMD:
                         if ( Flags & UPDATE_HARDWARE ) {
                           RegAddress.AddressPort = CRTAddressPort;
                           RegAddress.DataPort    = CRTDataPort;
                           RegAddress.ColorAdjust = ColorAdjustment;
                           RegAddress.Flags       = NONE;
                         }

/*
 +----------------------------------------------------------------------------+
 |  Enter CRT registers into environment buffer                               |
 +----------------------------------------------------------------------------+
*/
                         if ( EnvBufferPassed )
                           for ( m = 0,
                                 k = HW_Default[Ind].ModeRegs[i].RegData.FirstEntry;
                                 m < HW_Default[Ind].ModeRegs[i].RegData.NumEntries;
                                 k++, m++ )
                             Environment->Hardware.CRTCtlRegs.All[k] =
                                   HW_Default[Ind].ModeRegs[i].RegData.DataArea[k];
                         break;

/*
 +----------------------------------------------------------------------------+
 |  Set the specified register                                                |
 |  (Port address is in 'FirstEntry' field of RegData                         |
 +----------------------------------------------------------------------------+
*/
    case RegOutput_CMD:
                        if ( Flags & UPDATE_HARDWARE )
                          RegAddress.DataPort    =
                                      HW_Default[Ind].ModeRegs[i].RegData.FirstEntry;

                        if ( EnvBufferPassed )
                          switch( HW_Default[Ind].ModeRegs[i].RegData.FirstEntry ) {

                            case MiscOutputRegWrite:
                                      Environment->Hardware.MiscOutputReg =
                                       *HW_Default[Ind].ModeRegs[i].RegData.DataArea;
                                       break;

                            } /* End of inner Switch structure */

                        break;

    } /* End of outer Switch structure */

  if ( Flags & UPDATE_HARDWARE ) {
    if ( HW_Default[Ind].ModeRegs[i].Command == RegOutput_CMD ) {
        if ( HW_Default[Ind].ModeRegs[i].RegData.NumEntries == SETWORD ) { /*@@B1*/
            AccessRegister(&RegAddress,SETWORD,HW_Default[Ind].ModeRegs[i].RegData.DataArea); /*@@B1*/
        } else { /*@@B1*/
            AccessRegister(&RegAddress,SET,HW_Default[Ind].ModeRegs[i].RegData.DataArea); /*@@B1*/
        } /*@@B1*/

    } else if ( HW_Default[Ind].ModeRegs[i].Command == Attributes_CMD ) {   //J-TS00
      if ( EnvBufferPassed ) {                                              //J-TS00
        REGDATA RegData;                                                    //J-TS00
                                                                            //J-TS00
        RegData.DataArea   = Environment->Hardware.Attributes.All;          //J-TS00
        RegData.FirstEntry = 0;                                             //J-TS00
        RegData.NumEntries = NUM_ATT_REGS;                                  //J-TS00
        AccessHardware( &RegAddress, BYTES, ColorMode, SET, &RegData );     //J-TS00
      } else {                                                              //J-TS00
        AccessHardware( &RegAddress, BYTES, ColorMode,                      //J-TS00
                        SET, &HW_Default[Ind].ModeRegs[i].RegData );        //J-TS00
      }                                                                     //J-TS00
/*                                                                          //J-TS00
 +------------------------------------------------------------------------+ //J-TS00
 |  Setting attribute registers on a non-VGA turns video on again         | //J-TS00
 |  Turn the video off once again to reduce further snow                  | //J-TS00
 +------------------------------------------------------------------------+ //J-TS00
*/                                                                          //J-TS00
      AccessVideoEnable( ColorMode, SET, &VideoEnable ); /* callgate */     //J-TS00
                                                                            //J-TS00
    } else {
      AccessHardware( &RegAddress, BYTES, ColorMode,
                      SET, &HW_Default[Ind].ModeRegs[i].RegData );
      }
    }
  }

  Environment->Option = Blink_State + Always_FNTCall ;                  //J-TS00
  BlinkOption = Environment->Option;                                    //J-TS00

  if ( Modes[Mode].Format_Index == ATLASUS_Format ) {

    if ( Modes[Mode].fbType & NOT_MONO ) {  /* Black&White or Color Mode */

       CLUT = &SumCLUT; /* Preset Summed CLUT as default for Color modes */
       if ( ( !(VideoHardware.fVideoType & MONO_DISPLAYS ) ) /* Not MONO displays *//*@T39,@T73*/
          && !( Flags & NO_CLR_BRST ) ) { /* Not a B/W mode */
          CLUT = &ColorCLUT; /* Use Color CLUT only for true Color modes */
       }
       else { /* set NO_CLR_BRST when using summed color lookup table */
          if (!( Environment->ModeData.fbType & GRAPHICS ) )
             Environment->ModeData.fbType |= NO_CLR_BRST;      /*@BB8 */
       }

    } else {                                /* MONO mode */

       CLUT = (VideoHardware.fVideoType & MONO_DISPLAYS )
                    ? &SumCLUT : &ColorCLUT;

    }
  } else if (( Modes[Mode].Format_Index == COMMON_Format ) ||
             ( Modes[Mode].Format_Index == ATLAS3_Format ) ||
             ( Modes[Mode].Format_Index == EPOCH_Format  )) {

    CLUT = (VideoHardware.fVideoType & MONO_DISPLAYS )
                ? &SumCLUT : &ColorCLUT;

  } /* ATLAS Format ? */

    if ( Flags & UPDATE_HARDWARE ) {
        ClutParms.DataArea = (UCHAR far *)CLUT;
        ClutParms.NumEntries = 0x3F+1;

        for (i=0; i < 0xFF+1; i+=0x3f+1) {
            ClutParms.FirstEntry = i;
            AccessCLUT( SET, (CLUTDATA far * far *)&ClutParms );
        }
    }
/*
 +----------------------------------------------------------------------------+
 |  Enter color lookup table into environment buffer                          |
 +----------------------------------------------------------------------------+
*/
    if ( EnvBufferPassed ) {
      for (i= 0x00; i < 0xFF+1; ) {
        for (j= 0x00; j < 0x3F+1; i++, j++) {
          Environment->LookupTable[i].Red   = CLUT[j].Red;
          Environment->LookupTable[i].Green = CLUT[j].Green;
          Environment->LookupTable[i].Blue  = CLUT[j].Blue;
        }
      }

//ͻ
//  Check if Format Changed or not                       J-KK                 
//ͼ

      if ( Environment->FormatIndex != Modes[Mode].Format_Index )   //J-KK1017
        Environment->FormatChanged = TRUE;                          //J-KK1017

//ͻ
//  Store Format Index needed by BufferUpdate routine.   J-KK                 
//ͼ

      Environment->FormatIndex = Modes[Mode].Format_Index;      /*J-KK1017*/

      Environment->ScreenSize  =                                /*J-KK1017*/
                                 Environment->ModeData.col *
                                 Environment->ModeData.row *
                                 ( Environment->ModeData.attrib + 1 );
    }

   if ( Flags & UPDATE_HARDWARE ) {                                     //J-TS00
/*                                                                      //J-TS00
 +------------------------------------------------------------+         //J-TS00
 |  Clear Shadow buffer & APA                                 |         //J-TS00
 +------------------------------------------------------------+         //J-TS00
*/                                                                      //J-TS00
    if ( EnvBufferPassed )                                              //J-TS00
      Clear_PVB( Environment );                                         //J-TS00

//ͻ
//  Notify FNTCALL of changing Session.                             J-KK0206  
//ͼ
    if ( EnvBufferPassed )                                              //J-TS00
      rc = FNTCHGSG( Environment->SessionNum, FNT_SCREEN_GROUP_OLD );

#ifdef   XVIO
//------------------------------------------------------------------/*J-KK0815*/
// BXVS Call ReDraw ( All )                                         /*J-KK0815*/
//------------------------------------------------------------------/*J-KK0815*/
    if ( !( Modes[Mode].fbType & GRAPHICS ) &&  // Not Graphic Mode   @camel2
          EnvBufferPassed ) {                                       /*J-KK1005*/
      XvioRedraw.Length   = sizeof(XvioRedraw);                     /*J-KK0815*/
      XvioRedraw.Option   = NoOption;
      XvioRedraw.StartRow = 0;                                      /*J-KK0815*/
      XvioRedraw.EndRow   = -1;                                     /*J-KK0815*/
      XvioRedraw.Session  = (UCHAR)Environment->SessionNum;         /*J-KK0926*/
      XvioRedraw.LVBSel   = ( Environment->EnvFlags & ENVFLAG_3xBOX ) ? //J-KK0213
                   0 : Environment->LVB_Selector;                   //J-KK0213
      xrc = XVIOVDHIF( XvsRedraw, (PUSHORT)&XvioRedraw );           /*J-KK0815*/
    }                                                               /*J-KK0821*/
#endif

  }
/*
 +----------------------------------------------------------------------------+
 |  Turn the video signal back on                                             |
 +----------------------------------------------------------------------------+
*/
  if ( (Flags & UPDATE_HARDWARE)  &&  VideoOn ) {                      /*@C21*/
     VideoEnable = 1; /* Turn video on */
     AccessVideoEnable( ColorMode, SET, &VideoEnable ); /* ring 2 callgate */
  }

#ifdef   XVIO
//------------------------------------------------------------------/*J-KK0815*/
// BXVS Call Resume                                                 /*J-KK0815*/
//------------------------------------------------------------------/*J-KK0815*/
  if ( EnvBufferPassed ) {                                          /*J-KK1005*/
     XvioResm.Length  = sizeof(XvioResm);                           /*J-KK0815*/
     XvioResm.Reserve = NoOption;                                   //@XVIO
     XvioResm.Session = (UCHAR)Environment->SessionNum;             /*J-KK0926*/
     xrc = XVIOVDHIF( XvsResume, (PUSHORT)&XvioResm );              /*J-KK0815*/
} /* endif */                                                       /*J-KK0817*/
#endif

}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: SetEnvMode                                         */
/*                                                                      */
/*  DESCRIPTIVE NAME: Set the envrionment mode                          */
/*                                                                      */
/*  FUNCTION: SetEnvMode is called to update the environment buffer     */
/*            with the current video mode.                              */
/*                                                                      */
/*  ENTRY POINT: SetEnvMode                                             */
/*    LINKAGE:   CALL FAR                                               */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             UCHAR Mode ( Index in table of supported modes )         */
/*             FAR * Environment ( far pointer to environment buffer )  */
/*             FAR * HWEnvironment (pointer to hardware state buffer)   */
/*         (Referenced)                                                 */
/*             Modes[] (global data - table of supported video modes )  */
/*                                                                      */
/*  EXIT-NORMAL: Environment buffer is updated to reflect HW state      */
/*                                                                      */
/*  EFFECTS: NONE                                                       */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
void PASCAL near SetEnvMode( Mode, Environment, HWEnvironment, GetMode ) /*@B15,@T70*/
UCHAR Mode;
ENVIRONMENT far *Environment;
ENVIRONMENT far *HWEnvironment;
USHORT GetMode;
{
int i;
UCHAR far *Source;
UCHAR far *Destination;
VIDEOMODE far *pCurMode;                /* Reduce level of indirection, @S8 */

/*
 +----------------------------------------------------------------------------+
 |  Update environment buffer with current hardware state                     |
 +----------------------------------------------------------------------------+
*/
if ( HWEnvironment ) {
   Source      = (UCHAR far *)&HWEnvironment->Hardware;
   Destination = (UCHAR far *)&Environment->Hardware;
   for ( i = 0; i < sizeof( HWREGS ); i++ ) {
      Destination[i] = Source[i];
   }
}

if ( GetMode ) {                                                        /*@T70*/
   pCurMode = (VIDEOMODE far *)&Modes[Environment->ModeIndex = Mode].cb; /*@S8*/

   Environment->ModeData.cb     = pCurMode->cb;
   Environment->ModeData.fbType = pCurMode->fbType;
   Environment->ModeData.color  = pCurMode->color;
   Environment->ModeData.col    = pCurMode->col;
   Environment->ModeData.row    = pCurMode->row;
   Environment->ModeData.hres   = pCurMode->hres;
   Environment->ModeData.vres   = pCurMode->vres;
   Environment->ModeData.fmt_ID = pCurMode->fmt_ID;
   Environment->ModeData.attrib = pCurMode->attrib;
   Environment->ModeData.BufferAddress =
                   MemoryMaps[pCurMode->MemMap].Start.FullAddress;
   Environment->ModeData.BufferLength =
                   MemoryMaps[pCurMode->MemMap].TotalSize;


   Environment->ModeData.FullBufferSize =
                   Environment->ModeData.BufferLength;

   Environment->ModeData.PartialBufferSize = (ULONG)4000;

   Environment->ModeData.ExtDataArea = (UCHAR far *)NULL;

//ͻ
//  Store Format Index needed by BufferUpdate routine.   J-KKJ                
//ͼ

   Environment->FormatIndex = Modes[Mode].Format_Index;             /*J-KK1017*/
   Environment->ScreenSize  =                                       /*J-KK1017*/
                              Environment->ModeData.col *             /*J-KKJ*/
                              Environment->ModeData.row *             /*J-KKJ*/
                              ( Environment->ModeData.attrib + 1 );   /*J-KKJ*/

}                                                                       /*@T70*/
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: GetModeIndex                                       */
/*                                                                      */
/*  DESCRIPTIVE NAME: Determine video mode from registers               */
/*                                                                      */
/*  FUNCTION: GetModeIndex returns the index in the table of modes      */
/*            that corresponds to the current hardware setting.         */
/*                                                                      */
/*  ENTRY POINT: GetModeIndex                                           */
/*    LINKAGE:   CALL FAR                                               */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR * Environment ( Registers that have been read )      */
/*         (Referenced)                                                 */
/*             Modes[] ( table of supported video modes )               */
/*                                                                      */
/*  EXIT-NORMAL: AX = mode table index + 1                              */
/*                                                                      */
/*  EXIT-ERROR:  AX = 0                                                 */
/*                                                                      */
/*  EFFECTS: NONE                                                       */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: AccessCLUT                                              */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT PASCAL near GetModeIndex( Environment ) /*@B15*/
ENVIRONMENT far *Environment;
{
USHORT i,
       MODE_FOUND;
VIDEOMODE far *pCurMode;                /* Reduce level of indirection, @S8 */
FarAddress PhysAddress1;                                                /*@C29*/
VIDEO_BIOS far *BiosData;                                               /*@C29*/

/*
 +----------------------------------------------------------------------------+
 |  An examination of just 4 registers is all that is needed to determine     |
 |  which of the supported mode is currently set.  If an unknown mode is      |
 |  set, an error is returned.  The registers examined are:                   |
 |                  - Miscellaneous Output Register                           |
 |                  - Attribute Mode Control Register                         |
 |                  - Attribute Color Plane Enable Register                   |
 |                  - Sequencer Clocking Mode Register                        |
 +----------------------------------------------------------------------------+
*/
for (i = IStart, MODE_FOUND = FALSE; (i < NUM_MODES) && !MODE_FOUND; i++) {
  pCurMode = (VIDEOMODE far *)&Modes[i].cb; /*@S8*/
}

/*
 +----------------------------------------------------------------------------+
 |  If no mode is found, if Protect mode use modeindex from environment   @C29|
 |  else get mode from BIOS data area                                     @C29|
 +----------------------------------------------------------------------------+
*/
if ( !MODE_FOUND ) {                                            /* @C29 start */
   if ( !( Environment->EnvFlags & ENVFLAG_3xBOX ) )
      i = Environment->ModeIndex;
   else {
      PhysAddress1.part.Selector = 0x0000;
      PhysAddress1.part.Offset   = 0x0400;

      PhysToUVirt( PhysAddress1, &PhysAddress1, sizeof( VIDEO_BIOS ) );

      BiosData = (VIDEO_BIOS far *)PhysAddress1.FullAddress;

      switch ( BiosData->BIOS_crtmode & 0x1F ) {

         case 0x00:
         case 0x01: switch ((USHORT)((BiosData->BIOS_rows+1)*BiosData->BIOS_points)) {
                       case 200: i = ModeIndex_VGA1;    break;
                       case 350: i = ModeIndex_VGE1;    break;
                       case 400: i = (VideoHardware.display==PlasmaDisplay) ?
                                     ModeIndex_VGP1p
                                           :
                                     ModeIndex_VGA1p;   break;
                       default:  i = ModeIndex_VGA1;    break;
                    }
                    break;
         case 0x02:
         case 0x03: switch ((USHORT)((BiosData->BIOS_rows+1)*BiosData->BIOS_points)) {
                       case 200: i = ModeIndex_VGA3;    break;
                       case 350: i = ModeIndex_VGE3;    break;
                       case 400: i = (VideoHardware.display==PlasmaDisplay) ?
                                     ModeIndex_VGP3p
                                           :
                                     ModeIndex_VGA3p;   break;
                       default:  i = ModeIndex_VGA3;    break;
                    }
                    break;
         case 0x04:
         case 0x05: i = ModeIndex_VGA4;    break;
         case 0x06: i = ModeIndex_VGA6;    break;
         case 0x07: if ((USHORT)((BiosData->BIOS_rows+1)*BiosData->BIOS_points) == 350 ) {
                       i = (VideoHardware.display==PlasmaDisplay) ?
                           ModeIndex_VGA7p
                                 :
                           ModeIndex_VGM7;
                    } else {
                        i = (VideoHardware.display==PlasmaDisplay) ?
                           ModeIndex_VGP7p
                                 :
                           ModeIndex_VGM7p;
                    }
                    break;
         case 0x0D: i = ModeIndex_VGAD;    break;
         case 0x0E: i = ModeIndex_VGAE;    break;
         case 0x0F: i = ModeIndex_VGAF;    break;
         case 0x10: i = ModeIndex_VGA10;   break;
         case 0x11: i = ModeIndex_VGA11;   break;
         case 0x12: i = ModeIndex_VGA12;   break;
         case 0x13: i = ModeIndex_VGA13;   break;
         default:   i = ModeIndex_VGA3;    break;

      }
      FreePhysToUVirt( PhysAddress1.part.Selector );
   }
   i++;
}                                                                 /* @C29 end */

if ( ( i == ModeIndex_VGC7p+1 ) &&                                      /*@T71*/
     ( VideoHardware.fVideoType & VGM_BIT ) ) {                         /*@T71*/
   i++;      /* Return mono mode index on a mono plasma display */      /*@T71*/
}                                                                       /*@T71*/

return( i );
}

//*** J-KKJ ************  START OF SPECIFICATIONS  *********** J-KK ******/
//*                                                                      */
//*  SUBROUTINE NAME: Clear_PVB                                          */
//*                                                                      */
//*  DESCRIPTIVE NAME: Clear PVB                                         */
//*                                                                      */
//*  FUNCTION: Clear PVB                                                 */
//*                                                                      */
//*  ENTRY POINT: Clear_PVB                                              */
//*    LINKAGE:   CALL FAR                                               */
//*                                                                      */
//*  INPUT: (Passed on stack)                                            */
//*             UCHAR Mode ( Index in table of supported modes )         */
//*             FAR * Environment ( far pointer to environment buffer )  */
//*                                                                      */
//*  EXIT-NORMAL: PVB is cleared.                                        */
//*                                                                      */
//*  EFFECTS: NONE                                                       */
//*                                                                      */
//*  INTERNAL REFERENCES:                                                */
//*    ROUTINES: NONE                                                    */
//*                                                                      */
//*  EXTERNAL REFERENCES:                                                */
//*    ROUTINES: NONE                                                    */
//*                                                                      */
//***********************  END OF SPECIFICATIONS  ************************/
void PASCAL near Clear_PVB( Environment )          /*J-KK129*/
ENVIRONMENT far *Environment;
{
  FarAddress APA, Phys;
  USHORT length;
  USHORT start, end;
  USHORT far *p, i;

  if ( Environment->NATIVE_MODE == FALSE ) {        /* VGA Mode */
    Phys.FullAddress = (UCHAR far *)0xb8000L;
    length           = 0x8000;
    if ( !PhysToUVirt( Phys, &APA, length ) ) {     /* alloc APA selector */
      p = (USHORT far *)APA.FullAddress;            /* clear APA */
      length /= 2;
      for (i = 0;i < length;i++) {
        *p++ = 0x0720;
      }
      FreePhysToUVirt( APA.part.Selector );         /* free selector */
    }
  } else {                                          /* OS/2V mode */
/*
 +---------------------------------------------------------------+
 |  Clear shadow buffer & APA                                    |
 +---------------------------------------------------------------+
*/
    Phys.FullAddress = (UCHAR far *)0xa0000L;
    length           = 0;
    if ( !PhysToUVirt( Phys, &APA, length ) ) {     /* alloc APA selector */
      ClearShadowAPA( APA.part.Selector );          /* clear buffer */
      FreePhysToUVirt( APA.part.Selector );         /* free selector */
    }
/*
 +---------------------------------------------------------------+
 |  Restore cursor                                               |
 +---------------------------------------------------------------+
*/
    InitCursor( Environment->ModeData.col );    /* initialize cursor */

    InitAPA();                                  /* initialize APA */

    SetCursorPos( Environment->CursorRow,       /* restore cursor pos */
                Environment->CursorCol );

    if ( Environment->CursorStart < 0) {
    start = (17 * (-Environment->CursorStart) + 50) / 100;
    } else {
    start = Environment->CursorStart * 450
            / Environment->ModeData.vres;
    if ( start >= 18 )
    start = 17;
    }
    if ( Environment->CursorEnd   < 0) {
    end   = (17 * (-Environment->CursorEnd  ) + 50) / 100;
    } else {
    end   = Environment->CursorEnd   * 450
            / Environment->ModeData.vres;
    if ( end   >= 18 )
    end   = 17;
    }
    SetCursorType( start, end,                  /* restore cursor type */
                Environment->CursorWidth,
                Environment->CursorAttribute );
    SetCursorColor( Environment->CursorColor );
/*
 +---------------------------------------------------------------+
 |  Restore grid color                                           |
 +---------------------------------------------------------------+
*/
    SetGridDrawColor( Environment->GridColor );
  }
}
