/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Microsoft Corporation, 1989                                 */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/*****************************************************************************
 *
 * SOURCE FILE NAME = VDHINFO.c
 *
 * DESCRIPTIVE NAME = Base video device handlers - Cursor/Variable info
 *
 *
 * VERSION      V2.0
 *
 * DATE
 *
 * DESCRIPTION  This source file contains VDH entry points for setting
 *              and getting cursor information and setting and getting
 *              variable information.
 *
 *              These routines execute as ring 2 conforming
 *
 * FUNCTIONS    GetCursorInfo,   SetCursorInfo
 *              GetVariableInfo, SetVariableInfo
 *              GetDBCSDisplayInfo, GetLVBInfo
 *
 * NOTES        NONE
 *
 * STRUCTURES   NONE
 *
 * EXTERNAL REFERENCES
 *
 *              SaveRegs, RestoreRegs
 *              AccessCursorPos, AccessCursorType
 *              AccessBlink, AccessOverscan, AccessUnderscore
 *              AccessVideoEnable, AccessDisplayMask
 *
 * EXTERNAL FUNCTIONS
 *
 *              NONE
 *
*/


/*----------------------------------------------------------------------*/
/*      Include files                                                   */
/*----------------------------------------------------------------------*/

#define INCL_BASE               /* ALL of OS/2 Base                */
#include <os2.h>

#include "vdhctl.h"             /* Conditional compilation control */
#include "vdh.h"                /* Type definitions                */
#include "fntcalls.h"           /* FNTCALLS definition         */   //J-TS00

/*----------------------------------------------------------------------*/
/*  Externally defined global variables                                 */
/*----------------------------------------------------------------------*/

extern VIDEOMODE  Modes[];                  /* Supported modes                */
extern MEMORYMAPS MemoryMaps[];             /* Memory map info for each mode  */
extern short HugeShift;                     /* Huge shift value               */
extern UCHAR READABLE;
extern USHORT VGA_PRESENT;      /* TRUE if VGA VDH has been installed */
extern USHORT (APIENTRY *ChainedCallVectorTable[MaxFn])();           /* @T24 */
extern USHORT OEMFlags;                                                 //MS27
extern USHORT SVGAPresent;                                              // @drw

extern  USHORT CodePageTable[];          /* Code Page Table  J-KK1002*/
extern  USHORT CodePageZero;             // Code Page Zero   J-KK1024
extern VIDEOHARDWARE VideoHardware;     /* installed video hardware */  //J-TS00


/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: GetCursorInfo                                      */
/*                                                                      */
/*  DESCRIPTIVE NAME: Get cursor position and/or cursor type            */
/*                                                                      */
/*  FUNCTION: GetCursorInfo is called by BVS to return selected         */
/*            information about either the cursor position or the       */
/*            cursor type.  If the request specifies hardware and the   */
/*            hardware is readable, the actual hardware setting will    */
/*            be read and returned.  Otherwise the returned             */
/*            information will be taken from the environment buffer,    */
/*            if it has been passed.                                    */
/*                                                                      */
/*  ENTRY POINT: GetCursorInfo                                          */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 265 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     USHORT Row                                       */
/*                     USHORT Column                                    */
/*                     USHORT TopScanLine                               */
/*                     USHORT BottomScanLine                            */
/*                     USHORT Width                                     */
/*                     USHORT Attribute                                 */
/*             ULONG Function ( Call vector table entry = 265 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Cursor information is returned.                        */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  EFFECTS: If hardware specified and hardware is readable, the        */
/*           environment buffer is updated, if passed.                  */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: AccessCursorPos, AccessCursorType                       */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY GetCursorInfo( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_CURSOR far *ParmBlock;
ULONG Function;
{
USHORT rc,
       ModeColumns,
       ColorMode,
       EnvBufferPassed;

union {
  USHORT Position;
  struct{
    UCHAR Low;
    UCHAR High;
    } Pos;
  } Cursor;
USHORT ReqFlags;                        /*@S11*/

/*SaveRegs();*/ /*@B14*/                      /* Preserve registers and flags */

ReqFlags = ParmBlock->Flags;            /*@S11*/

rc = ERROR_VIO_INVALID_PARMS;                          /* Initialize to error */

EnvBufferPassed = SEG( Environment );                  /* Non-zero = TRUE     */

if ( EnvBufferPassed &&                                                /*J-KKJ*/
     !Environment->NATIVE_MODE &&                                      /*J-KKJ*/
     VGA_PRESENT                   ) {   /* VGA Mode */                /*J-KKJ*/
                                                                       /*J-KKJ*/
    rc = ChainedVDHGetCursorInfo(                                      /*J-KKJ*/
              (ENVIRONMENT far *)&Environment->VGAEnvironment,         /*J-KKJ*/
               ParmBlock, Function );                                  /*J-KKJ*/
                                                                       /*J-KKJ*/
} else {        /* ATLAS Mode */                                       /*J-KKJ*/

if ( ( EnvBufferPassed ) &&
     ( Function == FnGetCursorInfo ) &&                 /* Valid function */
     !(ReqFlags & ~VALID_CURSOR_FLAG) &&        /* Valid flags @@S1*/
     ( ParmBlock->Length >= MinSLen_Cursor_Pos ) ) {    /* Min length   */

/*----------------------------------------------------------------------*/
/*  Flags are valid - Get cursor type if enough room provided           */
/*                    Get cursor position if enough room provided       */
/*----------------------------------------------------------------------*/

  rc = ERROR_VIO_MODE;                                 /* Preset invalid mode */
  if ( !(Environment->ModeData.fbType & GRAPHICS) ) {  /* Text mode only      */

    rc = NO_ERROR;                                     /* Initialize no error */

/*----------------------------------------------------------------------*/
/*  Get cursor position -                                               */
/*----------------------------------------------------------------------*/

    if ( ReqFlags & CURSOR_POSITION) { /*@@S1*/
/*----------------------------------------------------------------------*/
/*  Copy cursor position from env buffer to caller's parameter area     */
/*----------------------------------------------------------------------*/

      ParmBlock->Row    = Environment->CursorRow;                       //J-TS00
      ParmBlock->Column = Environment->CursorCol;                       //J-TS00
    } /*@@S1*/

/*----------------------------------------------------------------------*/
/*  Get cursor type -                                                   */
/*----------------------------------------------------------------------*/

    if ( ParmBlock->Length >= MinSLen_Cursor_Type
       && ReqFlags & CURSOR_TYPE ) { /*@@S1*/

/*----------------------------------------------------------------------*/
/*  Copy cursor type from env buffer to the caller's parameter area     */
/*----------------------------------------------------------------------*/

      ParmBlock->TopScanLine    = Environment->CursorStart;             //J-TS00
      ParmBlock->BottomScanLine = Environment->CursorEnd;               //J-TS00
      ParmBlock->Width          = Environment->CursorWidth;             //J-TS00
      ParmBlock->Attribute      = Environment->CursorAttribute;         //J-TS00
    }
  }
}
}               /* VGA Mode J-KKJ*/

/*RestoreRegs();*/ /*@B14*/              /* Restore registers and flags */
return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: SetCursorInfo                                      */
/*                                                                      */
/*  DESCRIPTIVE NAME: Set cursor position and/or cursor type            */
/*                                                                      */
/*  FUNCTION: SetCursorInfo is called by BVS to set selected            */
/*            information about either the cursor position or the       */
/*            cursor type.  If the request specifies hardware, the      */
/*            hardware and the environment buffer, if passed, will be   */
/*            updated.  Otherwise just the environment buffer, if       */
/*            passed, will be updated.                                  */
/*                                                                      */
/*  ENTRY POINT: SetCursorInfo                                          */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 266 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     USHORT Row                                       */
/*                     USHORT Column                                    */
/*                     USHORT TopScanLine                               */
/*                     USHORT BottomScanLine                            */
/*                     USHORT Width                                     */
/*                     USHORT Attribute                                 */
/*             ULONG Function ( Call vector table entry = 266 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Cursor information is set                              */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: AccessCursorPos, AccessCursorType                       */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY SetCursorInfo( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_CURSOR far *ParmBlock;
ULONG Function;
{
USHORT rc,
       ModeColumns,
       ColorMode,
       EnvBufferPassed;
union {
  USHORT Position;
  struct{
    UCHAR Low;
    UCHAR High;
    } Pos;
  } Cursor;
USHORT register ReqRow,ReqCol;          /*@S11*/
USHORT ReqFlags;                        /*@S11*/
USHORT start, end;                                                      //J-TS00

#ifdef  XVIO
USHORT         xrc;                                                 /*J-KK0804*/
XVS_SUSPEND    XvioSusp;                                            /*J-KK0804*/
XVS_RESUME     XvioResm;                                            /*J-KK0804*/
XVS_SETCURPOS  XvioCurPos;                                          /*J-KK0804*/
XVS_SETCURTYPE XvioCurType;                                         /*J-KK0804*/
#endif

/*SaveRegs();*/ /*@B14*/                      /* Preserve registers and flags */

ReqFlags = ParmBlock->Flags;            /*@S11*/

rc = ERROR_VIO_INVALID_PARMS;                          /* Initialize to error */

EnvBufferPassed = SEG( Environment );                  /* Non-zero = TRUE     */

if ( EnvBufferPassed &&                                                /*J-KKJ*/
     !Environment->NATIVE_MODE &&                                      /*J-KKJ*/
     VGA_PRESENT               ) { /* VGA Mode */                      /*J-KKJ*/
                                                                       /*J-KKJ*/
    rc = ChainedVDHSetCursorInfo(                                      /*J-KKJ*/
               (ENVIRONMENT far *)&Environment->VGAEnvironment,        /*J-KKJ*/
                ParmBlock, Function );                                 /*J-KKJ*/
                                                                       /*J-KKJ*/
} else {        /* ATLAS Mode */                                       /*J-KKJ*/

if ( ( EnvBufferPassed ) &&
     ( Function == FnSetCursorInfo ) &&                /* Valid function request */
     ( ( ReqFlags & ~UPDATE_HARDWARE )   ||    /* An option specified */
       ( ReqFlags &  UPDATE_HARDWARE ) ) &&    /* Foreground          */
    !( ReqFlags & ~VALID_CURSOR_FLAG )   &&    /* No illegal options  */
     ( ParmBlock->Length >= MinSLen_Cursor_Pos ) ) {   /* At least min length */

/*
 +----------------------------------------------------------------------------+
 |  Flags are valid - Set cursor type if requested                            |
 |                    Set cursor position if requested                        |
 +----------------------------------------------------------------------------+
*/

  rc = ERROR_VIO_MODE;                                 /* Preset invalid mode */
  if ( !(Environment->ModeData.fbType & GRAPHICS) ) {  /* Text mode only      */

    rc = NO_ERROR;                                     /* Initialize no error */

/*
 +----------------------------------------------------------------------------+
 |  Parameter checking                                                        |
 +----------------------------------------------------------------------------+
*/
    if ( (ReqFlags & CURSOR_POSITION) ) {               /*    @C12  */
      ReqRow = ParmBlock->Row;        /*@S11*/
      ReqCol = ParmBlock->Column;     /*@S11*/
/*
 +----------------------------------------------------------------------------+
 |  Check set cursor position parameters:                                     |
 |  ----- If Foreground, validate using hardware settings                     |
 |        If Background, validate using env buf mode data ----- deleted       |
 |        Always validate using env buf mode data                             |
 +----------------------------------------------------------------------------+
*/
      if ( ( ReqCol >= Environment->ModeData.col ) ||
           ( ReqCol > Environment->ScrollRect.Right) || /* @P1 */
           ( ReqCol < Environment->ScrollRect.Left) )   /* @P1 */
        rc = ERROR_VIO_COL;

      if ( !rc ) {                                           /* @T40 */
        if ( ( ReqRow >= Environment->ModeData.row ) ||
             ( ReqRow > Environment->ScrollRect.Bottom) ||   /* @P1 */
             ( ReqRow < Environment->ScrollRect.Top) )       /* @P1 */
        rc = ERROR_VIO_ROW;
      }                                                      /* @T40 */
    }                                                             /*    @C12  */

    if ( (ReqFlags & CURSOR_TYPE) && !rc) {     /* No errors encounted,@S11*/
/*
 +----------------------------------------------------------------------------+
 |  Check set cursor type parameters                                          |
 +----------------------------------------------------------------------------+
*/
      if ( ( ParmBlock->Length < MinSLen_Cursor_Type ) || /* Missing parameters */
           ( ParmBlock->TopScanLine >= ( Environment->ModeData.vres /
                                         Environment->ModeData.row ) ) ||
           ( ParmBlock->BottomScanLine > 28 ) )  //J-TS00 /* Invalid parameters */
        rc = ERROR_VIO_INVALID_PARMS;

      if ( !rc && (ParmBlock->Width > 2) )                              //J-TS00
        rc = ERROR_VIO_WIDTH;
    }

    if ( !rc ) {                                    /* No errors encountered */
/*
 +----------------------------------------------------------------------------+
 |  Parameters are valid - Update environment buffer if it was passed         |
 |                       - Update hardware if in foreground                   |
 +----------------------------------------------------------------------------+
*/
      if ( ReqFlags & CURSOR_POSITION ) {
/*
 +----------------------------------------------------------------------------+
 |  Copy cursor position from caller's parmeter area to the env buffer        |
 +----------------------------------------------------------------------------+
*/
        Environment->CursorRow = ReqRow;                                //J-TS00
        Environment->CursorCol = ReqCol;                                //J-TS00

        if ( ReqFlags & UPDATE_HARDWARE ) {
/*
 +----------------------------------------------------------------------------+
 |  Set cursor position directly on the hardware                              |
 +----------------------------------------------------------------------------+
*/
          SetCursorPos( Environment->CursorRow,
                        Environment->CursorCol );
        }

#ifdef  XVIO
//-----------------------------------------------------------------
// BXVS Call Cursor Position
//-----------------------------------------------------------------
        XvioCurPos.Length  = sizeof(XvioCurPos);
        XvioCurPos.Row     = ReqRow;
        XvioCurPos.Col     = ReqCol;
        XvioCurPos.Option  = Suspend+Resume;                        //J-KKB12
        XvioCurPos.Session = (UCHAR)Environment->SessionNum;        /*J-KK0926*/
        xrc = XVIOVDHIF( XvsSetCurPos, (PUSHORT)&XvioCurPos );
#endif

      }

      if ( ReqFlags & CURSOR_TYPE ) {

#ifdef  XVIO
//------------------------------------------------------------------/*J-KK0804*/
// BXVS Call Suspend                                                /*J-KK0804*/
//------------------------------------------------------------------/*J-KK0804*/
  XvioSusp.Length  = sizeof(XvioSusp);                              /*J-KK0804*/
  XvioSusp.Reserve = NoOption;                                      //@XVIO
  XvioSusp.Session = (UCHAR)Environment->SessionNum;                /*J-KK0926*/
  xrc = XVIOVDHIF( XvsSuspend, (PUSHORT)&XvioSusp);                 /*J-KK0804*/
#endif

/*
 +----------------------------------------------------------------------------+
 |  Copy cursor type from caller's parmeter area to the env buffer            |
 +----------------------------------------------------------------------------+
*/
        Environment->CursorStart     = ParmBlock->TopScanLine;          //J-TS00
        Environment->CursorEnd       = ParmBlock->BottomScanLine;       //J-TS00
        Environment->CursorAttribute = ParmBlock->Attribute;            //J-TS00
        Environment->CursorWidth     = ( ParmBlock->Width ) ?           //J-TS00
                                         ParmBlock->Width : 1 ;         //J-TS00
        if ( ReqFlags & UPDATE_HARDWARE ) {
/*
 +----------------------------------------------------------------------------+
 |  Set cursor type directly on the hardware                                  |
 +----------------------------------------------------------------------------+
*/
          if ( Environment->CursorStart < 0) {
            start = (17 * (-Environment->CursorStart) + 50) / 100;
          } else {
            start = Environment->CursorStart * 450
                  / Environment->ModeData.vres;
            if ( start >= 18 )
              start = 17;
          }
          if ( Environment->CursorEnd   < 0) {
            end   = (17 * (-Environment->CursorEnd  ) + 50) / 100;
          } else {
            end   = Environment->CursorEnd   * 450
                  / Environment->ModeData.vres;
            if ( end   >= 18 )
              end   = 17;
          }
          SetCursorType( start, end,
                         Environment->CursorWidth,
                         Environment->CursorAttribute );
        }

#ifdef  XVIO
//-----------------------------------------------------------------
// BXVS Call Cursor Type
//-----------------------------------------------------------------
        XvioCurType.Length    = sizeof(XvioCurType);
        XvioCurType.Start     = Environment->CursorStart;
        XvioCurType.End       = Environment->CursorEnd;
        XvioCurType.Attribute = Environment->CursorAttribute;
        XvioCurType.Width     = Environment->CursorWidth;
        XvioCurType.Option    = Resume;
        XvioCurType.Session   = (UCHAR)Environment->SessionNum;     /*J-KK0926*/
        xrc = XVIOVDHIF( XvsSetCurType, (PUSHORT)&XvioCurType );
#endif

      }
    }
  }
}
}               /* VGA Mode J-KKJ*/

/*RestoreRegs();*/ /*@B14*/              /* Restore registers and flags */
return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: GetVariableInfo                                    */
/*                                                                      */
/*  DESCRIPTIVE NAME: Get selected variable information                 */
/*                                                                      */
/*  FUNCTION: GetVariableInfo is called by BVS to return selected       */
/*            information about video state - blink versus background   */
/*            intensity, overscan color, underscore scan line, video    */
/*            enable, or 8514/A display mask. If the request specifies  */
/*            hardware and the hardware is readable, the actual         */
/*            hardware setting will be read and returned.  Otherwise    */
/*            the returned information will be taken from the           */
/*            environment buffer, if it has been passed.                */
/*                                                                      */
/*  ENTRY POINT: GetVariableInfo                                        */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 269 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     USHORT Blink                                     */
/*                     USHORT Overscan                                  */
/*                     USHORT Underscore                                */
/*                     USHORT VideoEnable                               */
/*                     ULONG  DisplayMask                               */
/*                     USHORT CodePageID                                */
/*                     USHORT ScrollLeft                        @P1     */
/*                     USHORT ScrollTop                         @P1     */
/*                     USHORT ScrollRight                       @P1     */
/*                     USHORT ScrollBottom                      @P1     */
/*                     USHORT ScreenRows                        @P1     */
/*                     USHORT ScreenCols                        @P1     */
/*             ULONG Function ( Call vector table entry = 269 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Selected variable information is returned              */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  EFFECTS: If hardware specified and hardware is readable, the        */
/*           environment buffer is updated, if passed.                  */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: AccessCursorBlink, AccessCursorOverscan                 */
/*              AccessCursorUnderscore, AccessCursorVideoEnable         */
/*              AccessDisplayMask                                       */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY GetVariableInfo( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_VARIABLE far *ParmBlock;
ULONG Function;
{
USHORT rc,
       EnvBufferPassed;
USHORT ReqFlags;                        /*@S11*/

/*SaveRegs();*/ /*@B14*/                      /* Preserve registers and flags */

ReqFlags = ParmBlock->Flags;            /*@S11*/

rc = ERROR_VIO_INVALID_PARMS;                          /* Initialize to error */

EnvBufferPassed = SEG( Environment );                 /* Non-zero = TRUE     */

if ( EnvBufferPassed &&                                                /*J-KKJ*/
     !Environment->NATIVE_MODE &&                                      /*J-KKJ*/
     VGA_PRESENT               ) {     /* VGA Mode */                  /*J-KKJ*/
                                                                       /*J-KKJ*/
    rc = ChainedVDHGetVariableInfo(                                    /*J-KKJ*/
                (ENVIRONMENT far *)&Environment->VGAEnvironment,       /*J-KKJ*/
                 ParmBlock, Function );                                /*J-KKJ*/
                                                                       /*J-KKJ*/
} else {        /* ATLAS Mode */                                       /*J-KKJ*/

if ( ( Function == FnGetVariableInfo ) &&              /* Valid function request */
     ( ReqFlags & VALID_VARIABLE_FLAG ) &&     /* @@B Valid flags     */
     ( EnvBufferPassed ||                              /* Env buffer passed   */
       (ReqFlags & UPDATE_HARDWARE) ) &&       /*   or is foreground  */
     ( ParmBlock->Length >= MinSLen_Var_Blink ) ) {    /* At least min length */

/*
 +----------------------------------------------------------------------------+
 |  Flags are valid - Get blink versus background if enough room provided     |
 |                    Get overscan color if enough room provided              |
 |                    Get underscore scan line if enough room provided        |
 |                    Get video enable if enough room provided                |
 |                    Get display mask if enough room provided                |
 |                    Get code page ID if enough room provided                |
 |                    Get scrollable region of the screen               @P1   |
 |                    Get actual screen height and width                @P1   |
 +----------------------------------------------------------------------------+
*/
  rc = NO_ERROR;                                       /* Initialize no error */

/*
 +----------------------------------------------------------------------------+
 |  Requesting application is running in the background OR                    |
 |  the hardware is write-only                                                |
 +----------------------------------------------------------------------------+
*/
    if ( EnvBufferPassed ) {

/*
 +----------------------------------------------------------------------------+
 |  Copy blink setting from env buffer to caller's parameter area             |
 +----------------------------------------------------------------------------+
*/
      if ( ReqFlags & VARIABLE_BLINK )                    /* @@B */
      {
        if ( Environment->ModeData.fbType & GRAPHICS )            /* @@C */
          rc = ERROR_VIO_MODE;                                    /* @@C */
        else
        {
          if ( Environment->ModeIndex <= ModeIndex_ATLASUS )      /*J-KKJ*/
           ParmBlock->Blink   /* ATLAS Text Mode */               /*J-KKJ*/
            = ( Environment->Option & Blink_State )               //J-KK1017
                      ? 0 : 1;                                    /*J-KKJ*/
          else                                                    /*J-KKJ*/
            rc = ERROR_VIO_INVALID_PARMS;                         /* @@B */
        }
      }

/*
 +----------------------------------------------------------------------------+
 |  Copy overscan from env buffer to the caller's parameter area              |
 +----------------------------------------------------------------------------+
*/
      if ( (ReqFlags & VARIABLE_OVERSCAN) && !rc )      /* @@B,@S11 */
      {
         if (( Environment->ModeIndex > ModeIndex_ATLASUS ) &&      /*J-KK0922*/
             ( Environment->ModeIndex != ModeIndex_ATLASCOLR82 )&&  /*J-KK0906*/
             ( Environment->ModeIndex != ModeIndex_ATLASMONO82 ))   /*J-KK1020*/
                  rc = ERROR_VIO_INVALID_PARMS;
         else
          if ( ParmBlock->Length >= MinSLen_Var_Overscan )
                 ParmBlock->Overscan = Environment->Overscan;       /*J-KKJ*/
          else   rc = ERROR_VIO_INVALID_LENGTH;                   /* @@B */
      }

/*
 +----------------------------------------------------------------------------+
 |  Not Support Get DisplayMask data.                                         |
 +----------------------------------------------------------------------------+
*/
      if (( ReqFlags & VARIABLE_DISPLAYMASK ) && !rc )              /*J-KKJ*/
               rc = ERROR_VIO_INVALID_PARMS;                        /*J-KKJ*/

/*
 +----------------------------------------------------------------------------+
 |  Copy underscore from env buffer to the caller's parameter area            |
 +----------------------------------------------------------------------------+
*/
      if ( (ReqFlags & VARIABLE_UNDERSCORE) && !rc )      /* @@B,@S11 */
      {
               rc = ERROR_VIO_INVALID_PARMS;                        /*J-KKJ*/
      }

/*
 +----------------------------------------------------------------------------+
 |  Copy video enable from env buffer to the caller's parameter area          |
 +----------------------------------------------------------------------------+
*/
      if ( (ReqFlags & VARIABLE_VIDEOENABLE) && !rc )     /* @@B,@S11 */
      {
        if ( ParmBlock->Length >= MinSLen_Var_VideoEnable )

            if ( ParmBlock->Flags & UPDATE_HARDWARE ) {                /*J-KKJ*/
                                                                       /*J-KKJ*/
                 AccessVideoEnable( HardwareColor(), GET,               //J-TS00
                                    &ParmBlock->VideoEnable );     /* ring 2 callgate */
                 Environment->VideoEnable = ParmBlock->VideoEnable;    /*J-KKJ*/
                                                                       /*J-KKJ*/
            } else                                                     /*J-KKJ*/
                 ParmBlock->VideoEnable = Environment->VideoEnable;    /*J-KKJ*/

        else rc = ERROR_VIO_INVALID_LENGTH;                   /* @@B */
      }

    }
    else rc = ERROR_VIO_INVALID_PARMS;           /* Write-only, no env buffer */
  }

/*
 +----------------------------------------------------------------------------+
 |  Copy code page ID                                                         |
 +----------------------------------------------------------------------------+
*/
  if ( (ReqFlags & VARIABLE_CODEPAGE) && !rc )             /* @@B,@S11 */
  {
    if ( ParmBlock->Length >= MinSLen_Var_CodePage )
    {
        if ( Environment->UserFont == USER_FONT_SELECTABLE )        /* @@A */
          rc = ERROR_VIO_USER_FONT;                                 /* @@A */
        else                                                        /* @@A */
          ParmBlock->CodePageID = Environment->CodePageID;          /* @@A */
    }
    else rc = ERROR_VIO_INVALID_LENGTH;                             /* @@B */
  }

/*
 +----------------------------------------------------------------------------+
 |  Scrollable rectangle                                                @P1   |
 +----------------------------------------------------------------------------+
*/
  if ( (ReqFlags & VARIABLE_SCROLLRECT) && !rc ) { /* @P1,@S11 */
    if ( ParmBlock->Length >= MinSLen_Var_ScrollRect )          /*      @P1 */
    {                                                           /*      @P1 */
      ParmBlock->ScrlLeft   = Environment->ScrollRect.Left;     /*      @P1 */
      ParmBlock->ScrlTop    = Environment->ScrollRect.Top;      /*      @P1 */
      ParmBlock->ScrlRight  = Environment->ScrollRect.Right;    /*      @P1 */
      ParmBlock->ScrlBottom = Environment->ScrollRect.Bottom;   /*      @P1 */
    }                                                           /*      @P1 */
    else rc = ERROR_VIO_INVALID_LENGTH;                         /*      @P1 */
  }                                                             /*      @P1 */

/*
 +----------------------------------------------------------------------------+
 |  Actual Screen Dimensions                                            @P1   |
 +----------------------------------------------------------------------------+
*/
  if ( (ReqFlags & VARIABLE_SCREENDIM) && !rc ) { /* @P1,@S11 */
    if ( ParmBlock->Length >= MinSLen_Var_ScreenDim )           /*      @P1 */
    {                                                           /*      @P1 */
      ParmBlock->ScreenRows = Environment->ModeData.row;        /*      @P1 */
      ParmBlock->ScreenCols = Environment->ModeData.col;        /*      @P1 */
    }                                                           /*      @P1 */
    else rc = ERROR_VIO_INVALID_LENGTH;                         /*      @P1 */
  }                                                             /*      @P1 */

}                       /* VGA Mode J-KKJ*/

/*RestoreRegs();*/ /*@B14*/              /* Restore registers and flags */
return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: SetVariableInfo                                    */
/*                                                                      */
/*  DESCRIPTIVE NAME: Set selected variable information                 */
/*                                                                      */
/*  FUNCTION: SetVariableInfo is called by BVS to set selected          */
/*            information about video state - blink versus background   */
/*            intensity, overscan color, underscore scan line, video    */
/*            enable, or 8514/A display mask. If the request specifies  */
/*            hardware the hardware and the environment buffer, if      */
/*            passed, will be updated.  Otherwise just the environment  */
/*            buffer, if passed, will be updated.                       */
/*                                                                      */
/*  ENTRY POINT: SetVariableInfo                                        */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 270 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     USHORT Blink                                     */
/*                     USHORT Overscan                                  */
/*                     USHORT Underscore                                */
/*                     USHORT VideoEnable                               */
/*                     ULONG  DisplayMask                               */
/*                     USHORT CodePageID                                */
/*             ULONG Function ( Call vector table entry = 270 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Selected variable information is set                   */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: AccessCursorBlink, AccessCursorOverscan                 */
/*              AccessCursorUnderscore, AccessCursorVideoEnable         */
/*              AccessDisplayMask                                       */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY SetVariableInfo( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_VARIABLE far *ParmBlock;
ULONG Function;
{
USHORT rc,row,col,                                                      /*@C10*/
       ColorMode,
       EnvBufferPassed;
SHORT  i;                                                               /*@C10*/
REGADDRESS RegAddress;

VDH_CURSOR CurParmBlock;                      /* for set scrollable rect @P1 */
USHORT DoSet;                                                         /* @P1 */
USHORT ReqFlags;                        /*@S11*/
USHORT SaveVideoEnable;
VDH_VARIABLE VarParmBlock;

ENVIRONMENT far *ModeEnvironment;                                   /*J-KKJ*/
VIOMODEINFO far *PMode;                                             /*J-KKJ*/
VDH_MODE    far *ModeParmBlock;                                     /*J-KKJ*/
VDH_MODE         MODE_DATA;                                         /*J-KKJ*/
VIOMODEINFO      MODEINFO_DATA;                                     /*J-KKJ*/
USHORT           OldCodePage;                                       /*J-KKJ2*/
USHORT           ReqCodePage;                                       //J-KK1024
USHORT           EpochMode;                                         /*J-KK0828*/
USHORT           TempFlags;                                         //J-KK0207
VIDEOMODE   far *pCurMode;                                          /*J-KK0828*/
COUNTRYCODE      CCode;          /* Country Code */                 /*J-KK0822*/
extern USHORT   BlinkOption;                                            //J-TS00
VIOMODEINFO ModeData;                                                   //J-TS00
VDH_MODE ParmBlockVGA;                                                  //J-TS00

#ifdef  XVIO
USHORT       xrc, XVIOFlag;
XVS_SUSPEND  XvioSusp;                                              /*J-KK0804*/
XVS_RESUME   XvioResm;                                              /*J-KK0804*/
XVS_SETMODE  XvioMode;                                              /*J-KK0804*/
XVS_SETSTATE XvioState;                                             /*J-KK0804*/
XVS_SETCP    XvioCP;
#endif

/*SaveRegs();*/ /*@B14*/                      /* Preserve registers and flags */

ReqFlags = ParmBlock->Flags;            /*@S11*/

rc = ERROR_VIO_INVALID_PARMS;                          /* Initialize to error */

EnvBufferPassed = SEG( Environment );                /* Non-zero = TRUE     */

if ( ( Function == FnSetVariableInfo ) &&              /* Valid function request */
     ( ReqFlags & ~UPDATE_HARDWARE ) &&        /* An option specified */
     ( EnvBufferPassed ||                              /* Env buffer passed   */
       (ReqFlags & UPDATE_HARDWARE) ) &&       /*   or is foreground  */
    !( ReqFlags & ~VALID_VARIABLE_FLAG ) ) {   /* No illegal options  */

/*
 +----------------------------------------------------------------------------+
 |  Flags are valid - Set blink versus background intensity if requested      |
 |                    Set overscan (border) color if requested                |
 |                    Set scan line for underscore if requested               |
 |                    Set video enable if requested                           |
 |                    Set display mask (8514/A) if requested                  |
 |                    Set codepage if requested                         @P1   |
 |                    Force codepage set (override mode) if requested   @P1   |
 |                    Set scrollable region of the screen               @P1   |
 +----------------------------------------------------------------------------+
*/
  rc = NO_ERROR;                                       /* Initialize no error */

//ͻ
//  First of all, Check if the request is "Change Code Page".          J-KKJ1 
//ͼ
  if (( ReqFlags & VARIABLE_CODEPAGE ) &&              /* Code Page Request */
      ( ParmBlock->Length >= MinSLen_Var_CodePage )&&  /* Parameter Ok      */
      ( EnvBufferPassed ))                             /* Passed EnvBuff    */
  {

    rc = ERROR_VIO_BAD_CP;      /* Code page not prepared */

//ͻ
//  Check if this is the User Font request.                         J-KK1002  
//ͼ

    if ( ( ParmBlock->CodePageID == USER_FONT_SELECTABLE ) ||
         ( ParmBlock->CodePageID == USER_FONT_NOT_SELECTABLE ) ) {
                                            /* User Font Request */
      if ( !Environment->NATIVE_MODE &&                                 //J-KK1024
           VGA_PRESENT               ) {            /* VGA Mode */

            /* Issue the SetVariableInfo */                             //J-KK1024
        rc = ChainedVDHSetVariableInfo(                                 //J-KK1024
                    (ENVIRONMENT far *)&Environment->VGAEnvironment,    //J-KK1024
                    ParmBlock, Function );                              //J-KK1024
                                                                        //J-KK1024
      } else {                                      /* ATLAS Mode */

        if ( Environment->UserFont != 0 ) { /* Is there the User Font ? */
          rc = NO_ERROR;

//
//  If User Font Active (CP=-1) is requested, Call FNTCHGCP with -1.
//  Otherwise, User Font Inactive (CP=-2) is requested, Call with Real CodePage.
//
          if ( ParmBlock->CodePageID == USER_FONT_SELECTABLE )          //J-KK1025
            rc = FNTCHGCP( Environment->SessionNum, ParmBlock->CodePageID );
          else                                                          //J-KK1025
            rc = FNTCHGCP( Environment->SessionNum, Environment->CodePageID); //J-KK1025

          if (!rc)
            Environment->UserFont = ParmBlock->CodePageID;
          else
            rc = ERROR_VIO_BAD_CP;      /* Code page not prepared */

        } /* endif */
      }                                                                 //J-KK1024

    } else {   /* Not User Font Request */

//ͻ
//  Check if the requested codepage is valid.                       J-KK1002  
//ͼ

      ReqCodePage = ( ParmBlock->CodePageID ) ?                     //J-KK1024
                        ParmBlock->CodePageID : CodePageZero;       //J-KK1024

      if ( ( ReqCodePage == CodePageTable[0] ) ||                   //J-KK1024
           ( ReqCodePage == CodePageTable[1] ) ) {                  //J-KK1024

        if ( Environment->UserFont &&       // User Font is there and    //J-KK1024
             ( Environment->NATIVE_MODE ||  // Native (ATLAS) mode       //J-KK1024
               !VGA_PRESENT )            )  // or VGA not present.       //J-KK1024
          Environment->UserFont   = USER_FONT_NOT_SELECTABLE;         /*J-KK1002*/

// Save Old Code Page
        OldCodePage = Environment->CodePageID;

// Change Current Code Page
        Environment->CodePageID = ReqCodePage;                      //J-KK1024

#ifdef  XVIO
//------------------------------------------------------------------/*J-KK0822*/
// BXVS Call SetCP                                                  /*J-KK0822*/
//------------------------------------------------------------------/*J-KK0822*/
        XvioCP.Length   = sizeof(XvioCP);
        XvioCP.Option   = Suspend;
        XvioCP.CodePage = Environment->CodePageID;
        XvioCP.Session  = (UCHAR)Environment->SessionNum;           /*J-KK0926*/
        xrc = XVIOVDHIF( XvsSetCP, (PUSHORT)&XvioCP);
        XVIOFlag = 1;
#endif

// Epoch SBCS Mode Flag : Clear
        EpochMode = FALSE;

// Check DBCS Environment Vector
        CCode.country = 0;                                            /*J-KK0822*/
        CCode.codepage = Environment->CodePageID;                     //J-KK1024
        rc = DosGetDBCSEv( DBCSEvBuffSize,                            /*J-KK0822*/
                          (COUNTRYCODE far *)&CCode,                  /*J-KK0822*/
                          (unsigned char far *)&Environment->DBCSEvBuff[0]);              //J-KK1017

// Requested CodePage is DBCS CodePage ?
        if (( Environment->DBCSEvBuff[0] != 0) ||       //J-KK1017
            ( Environment->DBCSEvBuff[1] != 0)) {       //J-KK1017

          Environment->NATIVE_MODE = TRUE;      /* To ATLAS Mode */

          if (( Environment->ModeIndex == ModeIndex_EPOCHMONO80 ) ||
              ( Environment->ModeIndex == ModeIndex_EPOCHCOLR80 ) ||
              ( Environment->ModeIndex == ModeIndex_ATLASCOLR82 ) ||
              ( Environment->ModeIndex == ModeIndex_ATLASMONO82 ))
            EpochMode = TRUE;                   /* Epoch DBCS */

        } else {   /* SBCS */

          if (( Environment->ModeIndex == ModeIndex_EPOCHMONO80 ) ||
              ( Environment->ModeIndex == ModeIndex_EPOCHCOLR80 ) ||
              ( Environment->ModeIndex == ModeIndex_ATLASCOLR82 ) ||
              ( Environment->ModeIndex == ModeIndex_ATLASMONO82 )) {

            EpochMode = TRUE;                   /* Epoch SBCS */
            Environment->NATIVE_MODE = TRUE;    /* To ATLAS Mode */

          } else {   /* Not Epoch Mode */

            if ( !VGA_PRESENT ) {               /* Device Chaining */         //@KK0907

              EpochMode = TRUE;                 /* ATLAS SBCS */        //@KK0907
              Environment->NATIVE_MODE = TRUE;  /* To ATLAS Mode */     //@KK0907

            } else {                                                     //@KK0907

              Environment->NATIVE_MODE = FALSE; /* To VGA Mode */

              TempFlags = ParmBlock->Flags;                             //J-KK0207
              ParmBlock->Flags |= VARIABLE_VIDEOENABLE; // Video Enable   J-KK0207
              ParmBlock->VideoEnable = 1;   // Enable                     J-KK0207

                            /* Issue the SetVariableInfo */
              rc = ChainedVDHSetVariableInfo(
                        (ENVIRONMENT far *)&Environment->VGAEnvironment,
                        ParmBlock, Function );

              ParmBlock->Flags = TempFlags;                             //J-KK0207

#ifdef   XVIO
//-----------------------------------------------------------------/*J-KK0804*/
// BXVS Call SetMode ( U.S. Video Mode )                           /*J-KK0804*/
//-----------------------------------------------------------------/*J-KK0804*/
#ifdef XVIO_DEBUG
int3x(XvsSetMode,   Resume  );
#endif
       XvioMode.Length = sizeof(XvioMode);                   /*J-KK0804*/
       XvioMode.HWMode = -1;                                 /*J-KK0804*/

       XvioMode.Option   = Resume;
       if ( ReqFlags & UPDATE_HARDWARE )     // Foreground          J-KK0213
               XvioMode.Option  |= FG_Session;                    //J-KK0213

       XvioMode.Session = (UCHAR)Environment->SessionNum;       /*J-KK0926*/
       xrc = XVIOVDHIF( XvsSetMode, (PUSHORT)&XvioMode);
       XVIOFlag = 0;
#endif

            } /* Device Chaining */
          }  /* Not Epoch Mode */
        }


//---------------------------------------------------------------------
// If Code Page is actually Changed, SetMode is issued !!!
//---------------------------------------------------------------------
        ModeParmBlock = &MODE_DATA;
        ModeParmBlock->ModeDataPTR = &MODEINFO_DATA;

// DBCS Code Page ? or Epoch Mode
        if (( Environment->DBCSEvBuff[0] != 0 ) ||       //J-KK1017
            ( Environment->DBCSEvBuff[1] != 0 ) ||       //J-KK1017
            ( EpochMode )) {                            /*J-KK0828*/

// To Atlas
          if (OldCodePage != ParmBlock->CodePageID) {                /* Change CP */

            if ( EpochMode ) {   /* Set Mode to Epoch */

              pCurMode = (VIDEOMODE far *)&Modes[Environment->ModeIndex].cb;
              ModeParmBlock->Length              = 18;
              ModeParmBlock->ModeDataPTR->cb     = MinDLen_Mode_Attrib;
              ModeParmBlock->ModeDataPTR->fbType = pCurMode->fbType  ;
              ModeParmBlock->ModeDataPTR->color  = pCurMode->color   ;
              ModeParmBlock->ModeDataPTR->col    = pCurMode->col     ;
              ModeParmBlock->ModeDataPTR->row    = pCurMode->row     ;
              ModeParmBlock->ModeDataPTR->hres   = pCurMode->hres    ;
              ModeParmBlock->ModeDataPTR->vres   = pCurMode->vres    ;
              ModeParmBlock->ModeDataPTR->fmt_ID = pCurMode->fmt_ID  ;
              ModeParmBlock->ModeDataPTR->attrib = pCurMode->attrib  ;

            } else {

             PMode = NULL;                                              //J-TS00
             if ( SVGAPresent && VGA_PRESENT ) {                        //J-TS00
              ParmBlockVGA.Length      = sizeof( VDH_MODE );            //J-TS00
              ParmBlockVGA.Flags       = 0;                             //J-TS00
              ParmBlockVGA.ModeDataPTR = &ModeData;                     //J-TS00
              ModeData.cb              = sizeof( VIOMODEINFO );         //J-TS00
              if (!ChainedVDHGetMode(                                   //J-TS00
                      (ENVIRONMENT far *)&Environment->VGAEnvironment,  //J-TS00
                      &ParmBlockVGA, FnGetMode ))                       //J-TS00
                PMode = &ModeData;                                      //J-TS00
             }                                                          //J-TS00
                                                                        //J-TS00
             if (PMode == NULL) {                                       //J-TS00

              if ( VGA_PRESENT ) {          /* Device Chaining */       //@KK0907
                ModeEnvironment =
                      (ENVIRONMENT far *)&Environment->VGAEnvironment; //@KK0907
              } else {                                                  //@KK0907
                ModeEnvironment = Environment;                          //@KK0907
              } /* endif */                                             //@KK0907

              PMode = (VIOMODEINFO far *)((UCHAR far *)ModeEnvironment +
                                          ModeEnvironment->ModeDataOFF);
             }                                                          //J-TS00

              ModeParmBlock->Length              = 18;
              ModeParmBlock->ModeDataPTR->cb     = MinDLen_Mode_Attrib;
              ModeParmBlock->ModeDataPTR->fbType = PMode->fbType  ;
              ModeParmBlock->ModeDataPTR->color  = PMode->color   ;
              ModeParmBlock->ModeDataPTR->col    = 80             ;
              ModeParmBlock->ModeDataPTR->row    = 25             ;
              ModeParmBlock->ModeDataPTR->hres   = 720            ;   //J-KK1207
              ModeParmBlock->ModeDataPTR->vres   = 400            ;   //J-KK1207
              ModeParmBlock->ModeDataPTR->fmt_ID = PMode->fmt_ID  ;
              ModeParmBlock->ModeDataPTR->attrib = PMode->attrib  ;   //J-KK1207

            } /* endif */

     /* Foreground or Background */                                  //J-KK0126
            ModeParmBlock->Flags = (ReqFlags & UPDATE_HARDWARE) ?
                                        UPDATE_HARDWARE : 0;

//-----------------------------------------------------------------------------
// Notify FNTCALLS to Change CP.                                    /*J-KK0126*/
//-----------------------------------------------------------------------------
            rc = FNTCHGCP( Environment->SessionNum, Environment->CodePageID );

     /* SetMode */                                                   //J-KK0126
            rc = SetMode( Environment, ModeParmBlock, FnSetMode );

          } /* Changed CP */

        } else {   /* To VGA */

          if ( VGA_PRESENT  &&                             /* Device Chaining */
               ( OldCodePage != ParmBlock->CodePageID ) ) {    /* Change CP */

//---------------------------------------------------------
// Issued SetMode with the Japanese current video mode.
//---------------------------------------------------------

            if (( Environment->ModeData.fmt_ID == WorldFormat)    &&
                ( Environment->ModeData.attrib == WorldAttrCount)) {

              ModeParmBlock->Length              = 18;
              ModeParmBlock->ModeDataPTR->cb     = MinDLen_Mode_Attrib;
              ModeParmBlock->ModeDataPTR->fbType = Environment->ModeData.fbType;
              ModeParmBlock->ModeDataPTR->color  = Environment->ModeData.color ;
              ModeParmBlock->ModeDataPTR->col    = Environment->ModeData.col   ;
              ModeParmBlock->ModeDataPTR->row    = Environment->ModeData.row   ;
              ModeParmBlock->ModeDataPTR->hres =                          
                  ( VideoHardware.display == PlasmaDisplay ) ? 640 : 720 ; 
              ModeParmBlock->ModeDataPTR->vres   = 400                         ;
              ModeParmBlock->ModeDataPTR->fmt_ID = Environment->ModeData.fmt_ID;
              ModeParmBlock->ModeDataPTR->attrib = Environment->ModeData.attrib;

            } else {
              ModeParmBlock->Length              = 18;
              ModeParmBlock->ModeDataPTR->cb     = MinDLen_Mode_Row;
              ModeParmBlock->ModeDataPTR->fbType = Environment->ModeData.fbType;
              ModeParmBlock->ModeDataPTR->color  = Environment->ModeData.color ;
              ModeParmBlock->ModeDataPTR->col    = Environment->ModeData.col   ;
              ModeParmBlock->ModeDataPTR->row    = Environment->ModeData.row   ;
            }

      /* Foreground or Background */                                  //J-KK0126
            ModeParmBlock->Flags = (ReqFlags & UPDATE_HARDWARE) ?
                                        UPDATE_HARDWARE : 0;

//-----------------------------------------------------------------------------
// Notify FNTCALLS to Change CP.                                    /*J-KK0214*/
//-----------------------------------------------------------------------------
            rc = FNTCHGCP( Environment->SessionNum, Environment->CodePageID );

      /* SetMode */                                                   //J-KK0126
/*
 +----------------------------------------------------------------------------+
 |  Turn the video signal off to reduce unexpective view                      |
 +----------------------------------------------------------------------------+
*/
            if ( ReqFlags & UPDATE_HARDWARE ) {
              if ( EnvBufferPassed ) {
                VarParmBlock.Length      = sizeof(VarParmBlock);
                VarParmBlock.Flags       = 0x10;
                GetVariableInfo( Environment,
                                &VarParmBlock,
                                 FnGetVariableInfo );
                SaveVideoEnable          = VarParmBlock.VideoEnable;

                VarParmBlock.Length      = sizeof(VarParmBlock);
                VarParmBlock.Flags       = 0x11;
                VarParmBlock.VideoEnable = 0;
                SetVariableInfo( Environment,
                                &VarParmBlock,
                                 FnSetVariableInfo );
              }
            }

            if (!(rc = SetMode( Environment, ModeParmBlock, FnSetMode )) && //J-KK0126
                 ( ReqFlags & UPDATE_HARDWARE ) ) {    /* Foreground */     //J-KK0126

              Clear_PVB( Environment ); /* To Clear PVB */

            }
/*
 +----------------------------------------------------------------------------+
 |  Turn the video signal back on                                             |
 +----------------------------------------------------------------------------+
*/
            if ( ReqFlags & UPDATE_HARDWARE ) {
              if ( EnvBufferPassed ) {
                VarParmBlock.Length      = sizeof(VarParmBlock);
                VarParmBlock.Flags       = 0x11;
                VarParmBlock.VideoEnable = SaveVideoEnable;
                SetVariableInfo( Environment,
                                &VarParmBlock,
                                 FnSetVariableInfo );
              }
            }

          } /* Changed CP */
        }  /* DBCS or SBCS Code Page */

#ifdef XVIO
//------------------------------------------------------------------/*J-KK0804*/
// BXVS Call Resume                                                 /*J-KK0804*/
//------------------------------------------------------------------/*J-KK0804*/
        if ( XVIOFlag ) {
          XvioResm.Length  = sizeof(XvioResm);                      /*J-KK0804*/
          XvioResm.Reserve = NoOption;                              //@XVIO
          XvioResm.Session = (UCHAR)Environment->SessionNum;        /*J-KK0926*/
          xrc = XVIOVDHIF( XvsResume, (PUSHORT)&XvioResm );         /*J-KK0804*/
        } /* endif */
#endif

      } /* endif - Valid Code Page */                                   /*J-KK1002*/
    } /* endif - Not User Font Request */                             /*J-KK1002*/

  } else { /* Code Page Request */

               /***********************************/
               /****** Not Code Page Request ******/
               /***********************************/

 /* VGA Mode */                                                        /*J-KKJ*/
    if ( !Environment->NATIVE_MODE &&                                     /*J-KKJ*/
         EnvBufferPassed &&                                               /*J-KKJ*/
         VGA_PRESENT               ) {                                    /*J-KKJ*/

      rc = ChainedVDHSetVariableInfo(                                   /*J-KKJ*/
                    (ENVIRONMENT far *)&Environment->VGAEnvironment,       /*J-KKJ*/
                    ParmBlock, Function );                                /*J-KKJ*/

    } else {        /* ATLAS Mode */                                     /*J-KKJ*/

/*
 +----------------------------------------------------------------------------+
 |  Parameter checking for non-8514A adapters                                 |
 |----------------------------------------------------------------------------|
 |  Check set blink versus background parameters                              |
 +----------------------------------------------------------------------------+
*/
  if (ReqFlags & VARIABLE_BLINK) {     /* No errors encountered,@S11 */

    if ( Environment->ModeData.fbType & GRAPHICS )            /* @@C */
      rc = ERROR_VIO_MODE;                                    /* @@C */
        else
    {
         if ( Environment->ModeIndex > ModeIndex_ATLASUS )             /*J-KKJ*/
             rc = ERROR_VIO_INVALID_PARMS;                             /*J-KKJ*/
         else {                                                        /*J-KKJ*/
           if ( ParmBlock->Length < MinSLen_Var_Blink )    /* Missing parameter */
                rc = ERROR_VIO_INVALID_LENGTH;
           if ( !rc && ( ParmBlock->Blink > 1 ) )          /* Invalid parameter */
                rc = ERROR_VIO_INVALID_PARMS;
         }                                                             /*J-KKJ*/
    }
  }

/*
 +----------------------------------------------------------------------------+
 |  Check set overscan (border) color parameters                              |
 +----------------------------------------------------------------------------+
*/

  if ( ( ReqFlags & VARIABLE_OVERSCAN ) && !rc ) { /* No errors,@S11 */

         if (( Environment->ModeIndex > ModeIndex_ATLASUS ) &&      /*J-KK0906*/
             ( Environment->ModeIndex != ModeIndex_ATLASCOLR82 ) && /*J-KK0906*/
             ( Environment->ModeIndex != ModeIndex_ATLASMONO82 ))   /*J-KK1020*/
             rc = ERROR_VIO_INVALID_PARMS;                             /*J-KKJ*/
         else {                                                        /*J-KKJ*/
         if ( ParmBlock->Length < MinSLen_Var_Overscan ) /* Missing parm */
              rc = ERROR_VIO_INVALID_LENGTH;
         if ( !rc && (ParmBlock->Overscan & ~0x00FF ) ) /* Invalid parm */
               rc = ERROR_VIO_INVALID_PARMS;
         }                                                             /*J-KKJ*/
  }

/*
 +----------------------------------------------------------------------------+
 |  Check Display Mask Parameters                                             |
 +----------------------------------------------------------------------------+
*/
  if ( ( ReqFlags & VARIABLE_DISPLAYMASK ) && !rc )
             rc = ERROR_VIO_INVALID_PARMS;                             /*J-KKJ*/

/*
 +----------------------------------------------------------------------------+
 |  Check set scan line for underscore parameters                             |
 +----------------------------------------------------------------------------+
*/

  if ( (ReqFlags & VARIABLE_UNDERSCORE) && !rc ) { /* No errors,@S11 */

             rc = ERROR_VIO_INVALID_PARMS;                              /*J-KKJ*/
  }
/*
 +----------------------------------------------------------------------------+
 |  Check set video enable parameters                                         |
 +----------------------------------------------------------------------------+
*/

  if ( (ReqFlags & VARIABLE_VIDEOENABLE) && !rc ) { /* No errors,@S11 */
    if ( ParmBlock->Length < MinSLen_Var_VideoEnable ) /* Missing parameter */
      rc = ERROR_VIO_INVALID_LENGTH;

    if ( !rc && ( ParmBlock->VideoEnable > 1 ) ) /* Invalid parameter */
      rc = ERROR_VIO_INVALID_PARMS;
  }

/*
 +----------------------------------------------------------------------------+
 |  Scrollable rectangle                                                @P1   |
 +----------------------------------------------------------------------------+
*/
  if ( (ReqFlags & VARIABLE_SCROLLRECT) && !rc ) { /* @P1,@S11 */
    if (( ParmBlock->Length >= MinSLen_Var_ScrollRect ) &&      /*      @P1 */
        (Environment->ModeData.col > ParmBlock->ScrlLeft) &&    /*      @P1 */
        (Environment->ModeData.col > ParmBlock->ScrlRight) &&   /*      @P1 */
        (Environment->ModeData.row > ParmBlock->ScrlTop) &&     /*      @P1 */
        (Environment->ModeData.row > ParmBlock->ScrlBottom) &&  /*      @P1 */
        (ParmBlock->ScrlLeft <= ParmBlock->ScrlRight) &&        /*      @P1 */
        (ParmBlock->ScrlTop  <= ParmBlock->ScrlBottom) ) {      /*      @P1 */

        Environment->ScrollRect.Left   = ParmBlock->ScrlLeft;   /*      @P1 */
        Environment->ScrollRect.Top    = ParmBlock->ScrlTop;    /*      @P1 */
        Environment->ScrollRect.Right  = ParmBlock->ScrlRight;  /*      @P1 */
        Environment->ScrollRect.Bottom = ParmBlock->ScrlBottom; /*      @P1 */
        CurParmBlock.Length = sizeof (VDH_CURSOR);              /*      @P1 */
        CurParmBlock.Flags = (ReqFlags & UPDATE_HARDWARE) | /*  @P1 */
                             CURSOR_POSITION;                   /*      @P1 */

#ifdef  XVIO
//------------------------------------------------------------------/*J-KK0804*/
// BXVS Call SetState                                               /*J-KK0804*/
//------------------------------------------------------------------/*J-KK0804*/
        XvioState.Length    = sizeof(XvioState);
        XvioState.Option    = Suspend+Resume;
        XvioState.BlinkHInt = (Environment->Option & Blink_State) ?
                              0 : 1;                                /*J-KK1017*/
        XvioState.NonScrRow = Environment->ModeData.row -
                              Environment->ScrollRect.Bottom - 1 ;
        XvioState.NonScrCol = Environment->ScrollRect.Right;        /*J-KK0928*/
        XvioState.Session   = (UCHAR)Environment->SessionNum;       /*J-KK0926*/
        xrc = XVIOVDHIF( XvsSetState, (PUSHORT)&XvioState );
#endif

        DoSet = FALSE;                                          /*      @P1 */
        if (!GetCursorInfo(Environment, &CurParmBlock, FnGetCursorInfo)) {
           if ( CurParmBlock.Column < Environment->ScrollRect.Left ) {
               CurParmBlock.Column = Environment->ScrollRect.Left;  /*  @P1 */
               DoSet = TRUE;                                    /*      @P1 */
           }                                                    /*      @P1 */
           if ( CurParmBlock.Column > Environment->ScrollRect.Right ) {
               CurParmBlock.Column = Environment->ScrollRect.Right; /*  @P1 */
               DoSet = TRUE;                                    /*      @P1 */
           }                                                    /*      @P1 */
           if ( CurParmBlock.Row < Environment->ScrollRect.Top ) { /*   @P1 */
               CurParmBlock.Row = Environment->ScrollRect.Top;  /*      @P1 */
               DoSet = TRUE;                                    /*      @P1 */
           }                                                    /*      @P1 */
           if ( CurParmBlock.Row > Environment->ScrollRect.Bottom ) { /*@P1 */
               CurParmBlock.Row = Environment->ScrollRect.Bottom; /*    @P1 */
               DoSet = TRUE;                                    /*      @P1 */
           }                                                    /*      @P1 */
           if (DoSet)                                           /*      @P1 */
               SetCursorInfo(Environment, &CurParmBlock, FnSetCursorInfo);
        }                                                       /*      @P1 */
    }                                                           /*      @P1 */
    else rc = ERROR_VIO_INVALID_LENGTH;                         /*      @P1 */
  }                                                             /*      @P1 */

  if ( !rc ) {                                       /* No errors encountered */
/*
 +----------------------------------------------------------------------------+
 |  Parameters are valid - Update environment buffer if it was passed         |
 |                       - Update hardware if in foreground                   |
 +----------------------------------------------------------------------------+
*/
    if ( EnvBufferPassed ) {

      if ( ReqFlags & VARIABLE_BLINK ) {
/*
 +----------------------------------------------------------------------------+
 |  Copy blink setting from caller's parmeter area to the env buffer          |
 +----------------------------------------------------------------------------+
*/

        if ( ParmBlock->Blink == 1 )    /* High Intensity */           /*J-KKJ*/
           Environment->Option                        /*J-KK1017*/
                          &= (0xFFFF-Blink_State);                     /*J-KKJ*/
        else                            /* Blinking */                 /*J-KKJ*/
           Environment->Option     |= Blink_State;    /*J-KK1017*/
        BlinkOption = Environment->Option;                              //J-TS00

#ifdef  XVIO
//-----------------------------------------------------------------J-KK0911
// BXVS Call SetState
//-----------------------------------------------------------------
           XvioState.Length    = sizeof(XvioState);
           XvioState.Option    = Suspend+Resume;
           XvioState.BlinkHInt = (Environment->Option & Blink_State) ?
                                 0 : 1;                             /*J-KK1017*/
           XvioState.NonScrRow = Environment->ModeData.row -
                                 Environment->ScrollRect.Bottom - 1 ;
           XvioState.NonScrCol = Environment->ScrollRect.Right;     /*J-KK0928*/
           XvioState.Session   = (UCHAR)Environment->SessionNum;    /*J-KK0926*/
           xrc = XVIOVDHIF( XvsSetState,  (PUSHORT)&XvioState);
#endif

        }

      if ( ReqFlags & VARIABLE_OVERSCAN ) {
/*
 +----------------------------------------------------------------------------+
 |  Copy border color from caller's parmeter area to the env buffer           |
 +----------------------------------------------------------------------------+
*/
          if (( Environment->ModeIndex == ModeIndex_ATLASCOLR82 ) ||
              ( Environment->ModeIndex == ModeIndex_ATLASMONO82 )) {
            Environment->Overscan =
                           ParmBlock->Overscan & 0x000F;
            Environment->GridColor =
                          (UCHAR)Environment->Overscan;
            SetGridDrawColor( Environment->GridColor );
          }
          else
            Environment->Overscan = ParmBlock->Overscan;
      }

      if ( ReqFlags & VARIABLE_UNDERSCORE ) {
/*
 +----------------------------------------------------------------------------+
 |  Copy underscore SL from caller's parmeter area to the env buffer          |
 +----------------------------------------------------------------------------+
*/

        Environment->Underscore = ParmBlock->Underscore;               /*J-KKJ*/
        }

      if ( ReqFlags & VARIABLE_VIDEOENABLE ) {
/*
 +----------------------------------------------------------------------------+
 |  Copy video enable from caller's parmeter area to the env buffer           |
 +----------------------------------------------------------------------------+
*/

        Environment->VideoEnable = ParmBlock->VideoEnable;

        }

      }

    if ( ReqFlags & UPDATE_HARDWARE ) {

      ColorMode = HardwareColor(); /* color/mono mode ( hardware ) */

      if ( ReqFlags & VARIABLE_OVERSCAN ) {
/*
 +----------------------------------------------------------------------------+
 |  Set overscan (border) color directly on the hardware                      |
 +----------------------------------------------------------------------------+
*/
        if (( Environment->ModeIndex == ModeIndex_ATLASCOLR82 ) ||  //J-KK1020
            ( Environment->ModeIndex == ModeIndex_ATLASMONO82 ))    //J-KK1020
        ;   /* OS/2V unique code is needed here */
      }


      if ( ReqFlags & VARIABLE_VIDEOENABLE )
/*
 +----------------------------------------------------------------------------+
 |  Set video enable directly on the hardware                                 |
 +----------------------------------------------------------------------------+
*/
        AccessVideoEnable( ColorMode, SET, &ParmBlock->VideoEnable );   /* ring 2 callgate */

      } /* End of foreground service */
    }

    }   /* Atlas mode */                                                //J-TS00
  }     /* Code page request */                                         //J-TS00

  }

/*RestoreRegs();*/ /*@B14*/              /* Restore registers and flags */
return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: GetDBCSDisplayInfo                                 */
/*                                                                      */
/*  DESCRIPTIVE NAME: Return DBCS display infromation for current mode  */
/*                                                                      */
/*  FUNCTION: GetDBCSDisplayInfo is called by BVS to get information    */
/*            about the display width of characters on the screen.      */
/*            Characters may be one or two cells wide.  This routine    */
/*            returns the range of characters that require two cells    */
/*            on the screen to be displayed.                            */
/*                                                                      */
/*  ENTRY POINT: GetDBCSDisplayInfo                                     */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 261 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     USHORT TableLen                                  */
/*                     USHORT TableOffset                               */
/*             ULONG Function ( Call vector table entry = 261 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               The DBCS display information is returned               */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY GetDBCSInfo( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_DBCS far *ParmBlock;
ULONG Function;
{
USHORT rc;
USHORT EnvBufferPassed;                                                 //J-TS00

    /*SaveRegs();*/ /*@B14*/                /* Preserve registers and flags */

EnvBufferPassed = SEG( Environment );       /* Non-zero = TRUE */       //J-TS00
                                                                        //J-TS00
if ( EnvBufferPassed &&                                                 //J-TS00
     !Environment->NATIVE_MODE &&                                       //J-TS00
     VGA_PRESENT )       {                  /* VGA Mode */              //J-TS00
                                                                        //J-TS00
    rc = ChainedVDHGetDBCSInfo(                                         //J-TS00
                (ENVIRONMENT far *)&Environment->VGAEnvironment,        //J-TS00
                 ParmBlock, Function );                                 //J-TS00
                                                                        //J-TS00
} else {                                    /* ATLAS Mode */            //J-TS00

    rc = ERROR_VIO_INVALID_PARMS;                   /* Initialize to error */
    if ( ( Function == FnGetDBCSDisplayInfo ) ) {   /* Valid function request */
        rc = NO_ERROR;                          /* Initialize no error */
        if ( ParmBlock->Length == 2 )           /* user wants table size */
             ParmBlock->Length = 6;             /* return ParmBlock size */
        else
            if ( ParmBlock->Length >= 6 )       /* user wants info */
                 ParmBlock->CellTableLen = 0;   /* no Double cell characters */
            else
                 rc = ERROR_VIO_INVALID_PARMS;  /* Initialize to error */
    }

}                                                                       //J-TS00

    /*RestoreRegs();*/ /*@B14*/          /* Restore registers and flags */
    return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: GetLVBInfo                                         */
/*                                                                      */
/*  DESCRIPTIVE NAME: Return LVB size and default attribute information */
/*                                                                      */
/*  FUNCTION: GetLVBInfo is called by BVS to determine the size of the  */
/*            LVB required for a given display size.  Based on the      */
/*            mode and its associated LVB format the allocation size    */
/*            of the LVB is returned.  Also a default attribute may     */
/*            be returned.                                              */
/*                                                                      */
/*  ENTRY POINT: GetLVBInfo                                             */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 2xx )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     UCHAR  FormatId;                                 */
/*                     UCHAR  AttrCount;                                */
/*                     USHORT LVBWidth;                                 */
/*                     USHORT LVBHeight;                                */
/*                     ULONG  LVBSize;                                  */
/*                     USHORT AttrBufSize;                              */
/*                     UCHAR far *AttrBufAddr;                          */
/*             ULONG Function ( Call vector table entry = 261 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               The correct size of the LVB is returned                */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY GetLVBInfo( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_LVB far *ParmBlock;
ULONG Function;
{
USHORT rc;
USHORT i;
USHORT size;        
USHORT EnvBufferPassed;                                                 //J-TS00

    /*SaveRegs();*/ /*@B14*/                /* Preserve registers and flags */

EnvBufferPassed = SEG( Environment );       /* Non-zero = TRUE */       //J-TS00
                                                                        //J-TS00
if ( EnvBufferPassed &&                                                 //J-TS00
     !Environment->NATIVE_MODE &&                                       //J-TS00
     VGA_PRESENT )       {                  /* VGA Mode */              //J-TS00
                                                                        //J-TS00
    rc = ChainedVDHGetLVBInfo(                                          //J-TS00
                (ENVIRONMENT far *)&Environment->VGAEnvironment,        //J-TS00
                 ParmBlock, Function );                                 //J-TS00
                                                                        //J-TS00
} else {                                    /* ATLAS Mode */            //J-TS00

 if ( EnvBufferPassed ) {                                              /*J-KKJ*/
    rc = ERROR_VIO_INVALID_PARMS;                   /* Initialize to error */
    if ( ( Function == FnGetLVBInfo ) ) {   /* Valid function request */
        if ( ParmBlock->Length >= MinGLen_LVB_Size ){/* user wants LVB size */
            rc = NO_ERROR;                      /* Initialize no error */
            size = 2;
            if (((ParmBlock->FormatId == WorldFormat) &&
                 (ParmBlock->AttrCount == WorldAttrCount)) ||
                ((ParmBlock->FormatId == 0) &&
                 (ParmBlock->AttrCount == 0) &&
                 (Environment->ModeData.fmt_ID == WorldFormat) &&
                 (Environment->ModeData.attrib == WorldAttrCount))) {
                        size = 4;
            }
            if (ParmBlock->LVBWidth == 0)   /* cells width of LVB */
                size *= Environment->ModeData.col;
            else
                size *= ParmBlock->LVBWidth;
            if (ParmBlock->LVBHeight == 0)  /* cells width of LVB */
                size *= Environment->ModeData.row;
            else
                size *= ParmBlock->LVBHeight;

            if   ( ParmBlock->FormatId == 0 )            /*j-ys1027*/
                size += 0x1000;                          /*j-ys1027*/

            ParmBlock->LVBSize = size;      /* size of the LVB */

            if ( ParmBlock->Length >= MinGLen_LVB_AttrBufSize ) {
                if ( ParmBlock->AttrBufSize == 0 )
                  ParmBlock->AttrBufSize = Environment->AttrBufSize;
                else
                  if ( ParmBlock->AttrBufSize >= Environment->AttrBufSize ) {
                    if ( ParmBlock->Length >= MinGLen_LVB_AttrBufAddr) {
                      for (i = 0; i < Environment->AttrBufSize; i++)
                        ParmBlock->AttrBufAddr[i] = Environment->AttrBuf[i];
                    }
                    else
                      rc = ERROR_VIO_INVALID_LENGTH;
                  }
                  else
                      rc = ERROR_VIO_INVALID_PARMS;
            }

            if ((Environment->ModeIndex == ModeIndex_ATLASMONOVGA) || //j-ys1102
                (Environment->ModeIndex == ModeIndex_ATLASMONOEGA) || //j-ys1102
                (Environment->ModeIndex == ModeIndex_EPOCHMONO80 ) || //j-ys1102
                (Environment->FormatIndex == ATLAS3_Format))          //j-ys1102
                      rc = ERROR_VIO_INVALID_PARMS;                   //j-ys1102

        }
    }
 }                                                                      //J-TS00

}                                                                       //J-TS00

    /*RestoreRegs();*/ /*@B14*/                /* Restore registers and flags */
    return( rc );

}
