/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) Microsoft Corporation, 1989                                 */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/*****************************************************************************
 *
 * SOURCE FILE NAME = VDHFONT.c
 *
 * DESCRIPTIVE NAME = Base video device handlers - Font support
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION   This source file contains VDH entry points which get/set
 *               the current font.
 *
 * FUNCTIONS     GetCurrentFont, SetCurrentFont, SetHWFont, FindFont
 *
 * NOTES
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES  SaveRegs, RestoreRegs
 *                      AccessHardware, AccessFont
 *
 * EXTERNAL FUNCTIONS
 *
*/

/*
 ͻ
   Include files                                                             
 ͼ
*/
#define INCL_BASE               /* ALL of OS/2 Base                */
#include <os2.h>

#include "vdhctl.h"             /* Conditional compilation control */
#include "vdh.h"                /* Type definitions                */
#include "fntcalls.h"           /* FNTCALLS definition         */   //J-TS00

/*
 ͻ
   Externally defined global variables                                       
 ͼ
*/
extern USHORT VGA_PRESENT;      /* TRUE if VGA VDH has been installed */
extern USHORT (APIENTRY *ChainedCallVectorTable[MaxFn])();           /* @T24 */

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: GetCurrentFont                                     */
/*                                                                      */
/*  DESCRIPTIVE NAME: Get color lookup table                            */
/*                                                                      */
/*  FUNCTION: GetCurrentFont is called by BVS to return the current     */
/*            active video font.                                        */
/*                                                                      */
/*  ENTRY POINT: GetCurrentFont                                         */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 267 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     UCHAR far *FontBuffer                            */
/*                     USHORT     FontLength                            */
/*                     USHORT     PelColumns                            */
/*                     USHORT     PelRows                               */
/*             ULONG Function ( Call vector table entry = 267 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Current font is returned to caller                     */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: PhysToUVirt, FreePhysToUVirt                            */
/*              AccessFont                                              */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/

USHORT EXPENTRY GetCurrentFont( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_FONT far *ParmBlock;
ULONG Function;
{
USHORT rc,
       FontSize,
       EnvBufferPassed;

  /*SaveRegs();*/ /*@B14*/

  rc = ERROR_VIO_INVALID_PARMS;
  EnvBufferPassed = SEG( Environment );             /* Non-zero = TRUE     */

  if ( EnvBufferPassed &&
      !Environment->NATIVE_MODE &&
       VGA_PRESENT               ) {
                                                    /* VGA Mode */
    rc = ChainedVDHGetCurrentFont(
                   (ENVIRONMENT far *)&Environment->VGAEnvironment,
                   ParmBlock, Function );

  } else {                                          /* ATLAS Mode */

    if (   EnvBufferPassed &&
         ( Function == FnGetFont ) &&                   /* Valid function request */
         ( ParmBlock->Length >= sizeof( VDH_FONT ) ) && /* Valid Length */
         ( ParmBlock->Flags <= 3 ) ) {                  /* Valid flags  */

      rc = NO_ERROR;                                    /* Parameters are OK */
      if ( Environment->ModeData.fbType & GRAPHICS )    /* Graphic mode?  */
        rc = ERROR_VIO_MODE;
      else if ( ParmBlock->Flags & GET_ROM_FONT )       /* Get ROM font? */
        rc = ERROR_VIO_FONT;

      if ( !rc ) {

        FontSize = 256 * 19;

        if ( ( ParmBlock->FontLength == NULL ) &&   /* Return font size only */
             ( ParmBlock->FontBuffer == NULL ) ) {  /*                  @T11 */

          ParmBlock->FontLength = FontSize;

        } else if ( ParmBlock->FontLength < FontSize ) {

          rc = ERROR_VIO_INVALID_LENGTH;

        } else {    /* Return complete font information */

          ParmBlock->PelColumns =  8;
          ParmBlock->PelRows    = 19;
          ParmBlock->FontLength = FontSize; /* in case too much specified */

          if ((rc = FNTGETFONT ( 0, 256, ParmBlock->FontBuffer )))
            rc = ERROR_VIO_FONT;

        }
      }
    }
  } /* Device Chaining */

  /*RestoreRegs();*/ /*@B14*/
  return( rc );
}

/**********************  START OF SPECIFICATIONS  ***********************/
/*                                                                      */
/*  SUBROUTINE NAME: SetCurrentFont                                     */
/*                                                                      */
/*  DESCRIPTIVE NAME: Set color lookup table                            */
/*                                                                      */
/*  FUNCTION: SetCurrentFont is called by BVS to load the specified     */
/*            video text font                                           */
/*                                                                      */
/*  ENTRY POINT: SetCurrentFont                                         */
/*    LINKAGE:   CALL FAR ( via BVS-DDI call vector table entry 268 )   */
/*                                                                      */
/*  INPUT: (Passed on stack)                                            */
/*             FAR *Environment ( Environment buffer for the session )  */
/*             FAR *ParmBlock                                           */
/*                     USHORT Length = length of this packet            */
/*                     USHORT Flags  = 0 - Environment buffer only      */
/*                                     1 - Hardware also                */
/*                     UCHAR far *FontBuffer                            */
/*                     USHORT     FontLength                            */
/*                     USHORT     PelColumns                            */
/*                     USHORT     PelRows                               */
/*             ULONG Function ( Call vector table entry = 268 )         */
/*                                                                      */
/*  EXIT-NORMAL: AX = 0                                                 */
/*               Current font is set                                    */
/*                                                                      */
/*  EXIT-ERROR: AX = ERROR_VIO_INVALID_PARMS                            */
/*                                                                      */
/*  INTERNAL REFERENCES:                                                */
/*    ROUTINES: NONE                                                    */
/*                                                                      */
/*  EXTERNAL REFERENCES:                                                */
/*    ROUTINES: PhysToUVirt, FreePhysToUVirt                            */
/*              AccessFont                                              */
/*                                                                      */
/***********************  END OF SPECIFICATIONS  ************************/
USHORT EXPENTRY SetCurrentFont( Environment, ParmBlock, Function )
ENVIRONMENT far *Environment;
VDH_FONT far *ParmBlock;
ULONG Function;
{
FarAddress VideoFontBuffer;
USHORT rc,
       FontLength,
       EnvBufferPassed;

  /*SaveRegs();*/ /*@B14*/                  /* Preserve registers and flags */

  rc = ERROR_VIO_INVALID_PARMS;
  EnvBufferPassed = SEG( Environment );             /* Non-zero = TRUE     */

  if ( EnvBufferPassed &&
      !Environment->NATIVE_MODE &&
       VGA_PRESENT               ) {
                                                    /* VGA Mode */
    rc = ChainedVDHSetCurrentFont(
                   (ENVIRONMENT far *)&Environment->VGAEnvironment,
                    ParmBlock, Function );

  } else {                                          /* ATLAS Mode */

    if ( ( Function == FnSetFont ) &&                   /* Valid function request */
         ( ParmBlock->Length >= sizeof( VDH_FONT ) ) && /* Valid Length */
         ( ParmBlock->Flags <= 1 )                   && /* Valid flags  */
         ( EnvBufferPassed ) ) {

      rc = ERROR_VIO_MODE;                                  /* Preset invalid mode */
      if ( !(Environment->ModeData.fbType & GRAPHICS) ) {   /* Text mode only      */

        rc = NO_ERROR;                                     /* Parameters are OK */
        if ( ParmBlock->FontLength < 256 * 19 )
          rc = ERROR_VIO_INVALID_LENGTH;
        else
          if (  ParmBlock->PelRows    != 19   ||
                ParmBlock->PelColumns != 8 )
            rc = ERROR_VIO_USER_FONT;

        if ( !rc ) {

          if ((rc = FNTSETFONT ( 0, 256, ParmBlock->FontBuffer ))) {
            rc = ERROR_VIO_FONT;
          } else {
            Environment->UserFont = USER_FONT_SELECTABLE;
          }
        }
      }
    }
  } /* Device Chaining */

  /*RestoreRegs();*/ /*@B14*/           /* Restore registers and flags */
  return( rc );
}
