;*DDK*************************************************************************/
;
; COPYRIGHT    Copyright (C) 1995 IBM Corporation
;
;    The following IBM OS/2 WARP source code is provided to you solely for
;    the purpose of assisting you in your development of OS/2 WARP device
;    drivers. You may use this code in accordance with the IBM License
;    Agreement provided in the IBM Device Driver Source Kit for OS/2. This
;    Copyright statement may not be removed.;
;*****************************************************************************/
        PAGE    ,132
        TITLE   Virtual Video SVGA Identification Processing

;/*****************************************************************************
;*
;* SOURCE FILE NAME = VVSVGAID.ASM
;*
;* DESCRIPTIVE NAME = Virtual Video SVGA Identification Processing
;*
;*
;* VERSION      V2.0
;*
;* DATE         02/06/92
;*
;* DESCRIPTION  This module contains the VVD's SVGA ID support code.
;*
;* FUNCTIONS    GetPortBase          Return port base address in DX (3b0 or 3d0)
;*              IsTseng              Look for Tseng chip
;*              IsVideo7             Look for Headland chip (Video7)
;*              Video7EnableExt      Enable Video7 extended registers
;*              Video7DisableExt     Disable Video7 extended registers
;*              IsTrident            Look for Trident chip set
;*              IsWesternDig         Look for Western Digital chip set
;*              IsATI                Look for an ATI Wonder.
;*              IsSpeedWay           Look for IBM 'SVGA'
;*              WhichTseng           Determine which Tseng chip we're using ET3000/ET4000
;*              WhichVideo7          Determine which Video7 chip we're using
;*              WhichTrident         Determine Trident chip version
;*              WhichWesternDig      Determine which Western Digital chip is in place.
;*              WhichATI             Determine ATI chip version
;*              IdentifySVGA         Identify a super VGA adapter in the system.
;*
;* NOTES        NONE
;*
;* STRUCTURES   NONE
;*
;* EXTERNAL REFERENCES
;*
;*              NONE
;*
;* EXTERNAL FUNCTIONS
;*
;*              NONE
;*
;* CHANGE ACTIVITY =
;*   DATE      FLAG        APAR   CHANGE DESCRIPTION
;*   --------  ----------  -----  --------------------------------------
;*   mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
;*
;*   02/06/92                     WKB - Ported from SVGAUTIL
;*
;*   03/08/92                     B734223
;*
;*   07/09/92                     IBM SVGA enhancements
;*
;*   11/05/92                     Don't lock Video 7 regs if previously unlocked
;*
;*****************************************************************************/


        .xlist
        include mvdm.inc
        include vddseg.inc
        .list
        include vvd.inc
        include vvdp.inc

        DefData     PRIVATE,SWAP,C
        public      WDSig,ATISig

        WDSig       db      'VGA='
        ATISig      db      ' 761295520'

        EndData

        DefCode     PRIVATE,SWAP,PASCAL

        DefFn       GetPortBase
        DefFn       IdentifySVGA
        DefFn       IsATI
        DefFn       IsTrident
        DefFn       IsTseng
        DefFn       IsVideo7
        DefFn       IsWesternDig
        DefFn       Video7DisableExt
        DefFn       Video7EnableExt
        DefFn       WhichATI
        DefFn       WhichTrident
        DefFn       WhichTseng
        DefFn       WhichVideo7
        DefFn       WhichWesternDig


IODelay         MACRO
                jmp     $+2
                ENDM

;               PUBLIC  IdentifySVGA


;/***************************************************************************
;*
;* FUNCTION NAME = GetPortBase
;*
;* DESCRIPTION   = Return port base address in DX (3b0 or 3d0)
;*                 depending on mono/colour mode.
;*
;* INPUT         = None
;*
;* OUTPUT        = Port base address in DX
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       GetPortBase

                EnterProc
                mov     dx, 03cch               ; 
                in      al, dx                  ; 
                mov     dx, 03b0h               ;assume mono
                test    al, 1                   ; 
                jz      @F                      ; 
                add     dl, 20h                 ;bump up to colour (3d0)
@@:             ExitProc
EndProc         GetPortBase


;****************************************************************************
;*
;* FUNCTION NAME = IsTseng
;*
;* DESCRIPTION   = Look for Tseng chip
;*
;*                 read old reg value
;*                 write a new value
;*                 read the new value
;*                 write old value back
;*                 if new value written equals value read, chip is Tseng
;*
;* INPUT         = None
;*
;* OUTPUT        = ZR => Tseng, NZ => not
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       IsTseng

                call    GetPortBase             ; 
                mov     cx, dx                  ; 
                add     dx, 4                   ; 
                mov     ax, 3333h               ; 
                out     dx, al                  ; 
                IODelay                         ; 
                in      al, dx                  ;Extra sanity check for L40
                cmp     al, ah                  ; which ignores index set,
                jnz     IsTsengExit             ; so check it worked.
                inc     dl                      ; 
                in      al, dx                  ; 
                IODelay                         ; 
                mov     bl, al                  ;save previous contents
                xor     al, 0fh                 ; 
                out     dx, al                  ;set some new value
                IODelay                         ; 
                mov     ah, al                  ; 
                in      al, dx                  ;read it back
                sub     ah, al                  ; 
                mov     al, bl                  ;get old value
                IODelay                         ; 
                out     dx, al                  ;restore it

               ;mov     dx, cx                  ;Get port base address
               ;add     dx, 0ah                 ;
               ;in      al, dx                  ;dummy read to set index
               ;mov     dx, 3c0h                ;
               ;mov     al, 20h                 ;
               ;out     dx, al                  ;turn palette back on
IsTsengExit:
                or      ah, ah                  ;ZR => Tseng, NZ => not
                jz      MustBeTseng
                mov     dx, cx                  ;could be ET3000
                add     dx, 0ah                 ;WPWong put it this portion
                in      al, dx                  ;on 10/28/92, 26129 (r206).
                mov     dx, 3c0h
                IODelay                         ; 
                mov     al, 16h
                out     dx, al
                inc     dx
                IODelay                         ; 
                in      al, dx
                mov     bl, al                  ;old value in bl
                push    bx

                mov     dx, cx
                add     dx, 0ah
                in      al, dx
                mov     dx, 3c0h
                IODelay                         ; 
                mov     al, 16h
                out     dx, al
                mov     al, bl
                xor     al, 10h
                mov     bl, al
                out     dx, al
                IODelay                         ; 

                mov     dx, cx
                add     dx, 0ah
                in      al, dx
                mov     dx, 3c0h
                mov     al, 16h
                out     dx, al
                IODelay                         ; 
                inc     dx
                in      al, dx
                mov     bh, al                  ;new value in bh

                mov     dx, cx
                add     dx, 0ah
                in      al, dx
                mov     dx, 3c0h
                mov     al, 16h
                out     dx, al
                IODelay                         ; 
                pop     ax                      ;original value from bl
                out     dx, al

                cmp     bh, bl                  ;set zero flag properly
MustBeTseng:
                pushf                           ;save flags
                mov     dx, cx                  ;Get port base address
                add     dx, 0ah                 ; 
                in      al, dx                  ;dummy read to set index
                mov     dx, 3c0h                ; 
                mov     al, 20h                 ; 
                out     dx, al                  ;turn palette back on
                popf                            ;restore flags
                ExitProc                        ; 
EndProc         IsTseng


;/***************************************************************************
;*
;* FUNCTION NAME = IsVideo7
;*
;* DESCRIPTION   = Look for Headland chip (Video7)
;*
;*                 read value of Start Address High (SAH) register
;*                 read Identification register (ID field)
;*                 if (SAH ^ 0EAH == ID reg) its a Video7 chip
;*
;* INPUT         = None
;*
;* OUTPUT        = ZR => Video7, NZ => no
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       IsVideo7

                mov     dx, 03c4h               ;Extension Control register
                mov     al, 6                   ; 
                out     dx, al                  ;(TS reg 6 gets destroyed)
                inc     dl                      ; 
                in      al, dx                  ; 
                mov     cl, al                  ;save current contents

                call    Video7EnableExt         ;enable extended registers

                call    GetPortBase             ; 
                add     dl, 4                   ;port 3x4
                mov     bx, dx                  ;save copy

                mov     al, 0ch                 ; 
                out     dx, al                  ;select Start Address High reg
                inc     dl                      ; 
                IODelay                         ; 
                in      al, dx                  ; 
                mov     bl, al                  ;save previous value

                mov     al, 055h                ; 
                out     dx, al                  ;write in a new value
                IODelay                         ; 
                in      al, dx                  ; 

                dec     dl                      ;to 3x4
                mov     al, 01fh                ; 
                out     dx, al                  ;select ID register
                inc     dl                      ; 
                IODelay                         ; 
                in      al, dx                  ;read ID register

                xchg    al, bl                  ;restore old value
                dec     dl                      ; 
                push    ax
                mov     al, 0ch                 ;select SAH register
                out     dx, al                  ; 
                inc     dl                      ; 
                IODelay                         ; 
                pop     ax
                out     dx, al                  ;write old value back
		or	cl, cl			;if its zero	                
		jnz	short @F		; they were locked              
                call    Video7DisableExt        ;disable extended registers
@@:             mov     al, 055h                ; 
                xor     al, 0eah                ; 
                sub     al, bl                  ;ZR => Video7, NZ => not
                jz      @F                      ; 
                mov     dx, 03c4h               ;Extension Control register
                mov     al, 6                   ; 
                mov     ah, cl                  ; 
                out     dx, ax                  ; 
@@:             ExitProc
EndProc         IsVideo7


;/***************************************************************************
;*
;* FUNCTION NAME = Video7EnableExt
;*
;* DESCRIPTION   = Enable Video7 extended registers
;*
;* INPUT         = None
;*
;* OUTPUT        = None
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       Video7EnableExt
                mov     dx, 03c4h               ;Extension Control register
                mov     ax, 0ea06h              ;write 0eah to reg 6
                out     dx, ax                  ; 
                ExitProc
EndProc         Video7EnableExt


;/***************************************************************************
;*
;* FUNCTION NAME = Video7DisableExt
;*
;* DESCRIPTION   = Disable Video7 extended registers
;*
;* INPUT         = None
;*
;* OUTPUT        = None
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       Video7DisableExt
                mov     dx, 03c4h               ;Extension Control register
                mov     ax, 0ae06h              ;write 0aeh to reg 6
                out     dx, ax                  ; 
                ExitProc
EndProc         Video7DisableExt


;/***************************************************************************
;*
;* FUNCTION NAME = IsTrident
;*
;* DESCRIPTION   = Look for Trident chip set
;*
;*                 check for presence of inverting bit field in
;*                 Mode Control register 1
;*
;* INPUT         = None
;*
;* OUTPUT        = ZR => Trident, NZ => not
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       IsTrident
                mov     dx, 03c4h               ;Mode Control register
                mov     al, 0eh                 ; 
                out     dx, al                  ; 
                inc     dl                      ; 
                IODelay                         ; 
                in      al, dx                  ; 
                mov     ah, al                  ;save old value
                IODelay                         ; 
                xor     al, al                  ; 
                out     dx, al                  ;write all zeros
                IODelay                         ; 
                in      al, dx                  ;read new value
                and     al, 0fh                 ; 
                IODelay                         ; 
                xchg    al, ah                  ;get old value back
                out     dx, al                  ;restore it
                cmp     ah, 2                   ;ZR => Trident, NZ => not
                ExitProc
EndProc         IsTrident


;/***************************************************************************
;*
;* FUNCTION NAME = IsWesternDig
;*
;* DESCRIPTION   = Look for Western Digital chip set.
;*
;*                 Look for string 'VGA=' at BIOS location C000:007D
;*
;* INPUT         = None
;*
;* OUTPUT        = ZR => Western Digital, NZ => not
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       IsWesternDig
                push    esi
                push    edi
                mov     edi, 0c007dh            ; 
                mov     esi, FLAToffset WDSig   ; 
                mov     ecx, 4                  ; 
                repe    cmpsb                   ; 
                pop     edi
                pop     esi
                ExitProc
EndProc         IsWesternDig


;/***************************************************************************
;*
;* FUNCTION NAME = IsATI
;*
;* DESCRIPTION   = Look for an ATI Wonder.
;*
;*                 Check for product signature string at C000:30
;*                 which should read '761295520'
;*
;* INPUT         = Assume DS -> DGROUP on entry
;*
;* OUTPUT        = ZR => ATI, NZ => not
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       IsATI
                push    esi
                push    edi
                mov     edi, 0c0030h            ; 
                mov     esi, FLAToffset ATISig  ; 
                mov     ecx, 10                 ; 
                repe    cmpsb                   ; 
                pop     edi
                pop     esi
                ExitProc
EndProc         IsATI


;/***************************************************************************
;*
;* FUNCTION NAME = IsSpeedWay                              Start
;*
;* DESCRIPTION   = Look for IBM 'SVGA'
;*
;* INPUT         = None
;*
;* OUTPUT        = ZR => found, NZ => not
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

IsSpeedWay      PROC
                mov     dx, 2170h               ; start from 8th instance OpMode Reg.
SWayNext:       in      al, dx                  ; get the OpMode setting
                cmp     al, 0f0h                ; valid value?
                jae     SWayNextPort            ; 
                xor     al, al                  ; indicate found
                jmp     short @F                ; indicate found
SWayNextPort:   sub     dx, 10h                 ; next configuration
                cmp     dx, 2100h               ; 
                jae     SWayNext                ; still more ports to check
@@:             or      al, al                  ; ZR => found, NZ => not
                ret
IsSpeedWay      ENDP                            ;            End


;/***************************************************************************
;*
;* FUNCTION NAME = WhichTseng
;*
;* DESCRIPTION   = Determine which Tseng chip we're using ET3000/ET4000
;*
;*                 Distinguish the two by availability of Extended Start
;*                 Address Register (CRTC index 33h) which only exists
;*                 on ET4000.
;*
;* INPUT         = None
;*
;* OUTPUT        = AX = 0  ET4000
;*                 AX = 1  ET3000
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       WhichTseng
                call    GetPortBase             ; 
                mov     bx, dx                  ;keep copy
                add     dl, 4                   ;Extended Start Address (3d4)
                mov     al, 33h                 ; 
                out     dx, al                  ;CRTC reg index 33h
                inc     dl                      ; 
                in      al, dx                  ; 
                mov     bl, al                  ;save previous value
                xor     al, 0fh                 ;flip some bits
                out     dx, al                  ;write to 3x5
                IODelay                         ; 
                mov     ah, al                  ;save copy of byte written
                in      al, dx                  ;read it back
                sub     ah, al                  ; 
                mov     al, bl                  ;restore previous value
                out     dx, al                  ;write back to 3x5
                or      ah, ah                  ;ZR => ET4000, NZ => ET3000
                mov     ax, 1                   ;assume ET3000
                jnz     WTExit                  ; 
                inc     ax                      ; 
WTExit:         ExitProc
EndProc         WhichTseng


;/***************************************************************************
;*
;* FUNCTION NAME = WhichVideo7
;*
;* DESCRIPTION   = Determine which Video7 chip we're using
;*
;*                 Chip Revision Register contents
;*                    HT205   71H                     up to 800x600x16
;*                    HT208   42H     0111xxxxB       up to 1024x768x16
;*                    HT209   52H/51H 0101xxxxB       up to 1024x768x256
;*                                                         (1Meg memory)
;*
;* INPUT         = None
;*
;* OUTPUT        = AX = 1  (HT205)
;*                 AX = 2  (HT208)
;*                 AX = 3  (HT209)
;*                 AX = 0  (don't know)
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       WhichVideo7
		mov	dx, 03c4h		;          
		mov	al, 6			;          
		out	dx, al			;          
		inc	dl			;          
		in	al, dx			;          
		mov	cl, al			;          

                call    Video7EnableExt         ;enable extended registers
                mov     al, 08eh                ; 
                mov     dx, 03c4h               ; 
                out     dx, al                  ; 
                IODelay                         ; 
                inc     dl                      ; 
                in      al, dx                  ; 
                shr     al, 4                   ;has value 04h to 0fh
                mov     bl, al                  ; 
		or	cl, cl			;          
		jnz	short @F		;          
                call    Video7DisableExt        ;disable extended registers
@@:             mov     ax, 1                   ;AX = 1  (HT205)
                cmp     bl, 7                   ; 
                je      WhichV7Exit             ; 
                inc     ax                      ;AX = 2  (HT208)
                cmp     bl, 4                   ; 
                je      WhichV7Exit             ; 
                inc     ax                      ;AX = 3  (HT209)
                cmp     bl, 5                   ; 
                je      WhichV7Exit             ; 
                xor     ax, ax                  ;AX = 0 (don't know)
WhichV7Exit:    ExitProc
EndProc         WhichVideo7


;/***************************************************************************
;*
;* FUNCTION NAME = WhichTrident
;*
;* DESCRIPTION   = Determine Trident chip version
;*
;*                 read Chip Version Register
;*                 value in AL >= 3   8900 chip version
;*
;*                 convert AL to:  AL = 1 => 8800
;*                                 AL = 2 => 8900
;*
;*                 Check memory on 8900 from 'Software Programming' register
;*
;* INPUT         = None
;*
;* OUTPUT        = AX = 0  (8800)
;*                 AX = 1  (8900)
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       WhichTrident

                mov     dx, 03c4h               ; 
                mov     ax, 0bh                 ;write 0 to version reg (0bh)
                out     dx, ax                  ; 
                inc     dl                      ; 
                IODelay                         ; 
                in      al, dx                  ; 
                mov     dl, al                  ; 
                mov     ax, 1                   ;assume 8800
                cmp     dl, 3                   ; 
                jl      @F                      ; 
                inc     ax                      ;make it 8900

@@:             ExitProc
EndProc         WhichTrident


;/***************************************************************************
;*
;* FUNCTION NAME = WhichWesternDig
;*
;* DESCRIPTION   = Determine which Western Digital chip is in place.
;*
;* INPUT         = None
;*
;* OUTPUT        = AX = 1     WD90c00
;*                 AX = 2     WD90c11
;*                 AX = 3     WD90c30
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       WhichWesternDig

                mov     bx, 1                   ;BX has value to return

                mov     al, 1eh                 ; 
                mov     dx, 3c3h                ; 
                out     dx, al                  ;Enter setup mode

                mov     al, 80h                 ; 
                mov     dx, 0103h               ; 
                out     dx, al                  ;Enable extended registers

                mov     al, 0eh                 ; 
                mov     dx, 3c3h                ; 
                out     dx, al                  ;Exit setup mode

                mov     dx, 03d4h               ;Unlock extended CRT regs
                mov     ax, 8529h               ; 
                out     dx, ax                  ; 

                mov     al, 2bh                 ; 
                out     dx, al                  ; 
                inc     dl                      ; 
                in      al, dx                  ; 
                mov     ah, al                  ;save current contents
                mov     al, 0aah                ;write new value
                out     dx, al                  ; 
                IODelay                         ; 
                in      al, dx                  ; 
                cmp     al, 0aah                ;if value read != 0AAH
                jne     WhichWDExit             ;  its PVGA1A

                xchg    al, ah                  ;save return, get old value
                IODelay                         ; 
                out     dx, al                  ; 
                inc     bx                      ;Assume WD90c00

                mov     dx, 03c4h               ;Unlock
                mov     ax, 4806h               ; extended
                out     dx, ax                  ; sequencers
                                                ;Distinguish between
                mov     al, 12h                 ; WD90c00 and WD90c10
                out     dx, al                  ; by using bit 6 of
                IODelay                         ; Misc Control Reg
                inc     dl                      ;If chip doesn't respond to
                in      al, dx                  ; changes in the bit, its a
                IODelay                         ; WD90c00
                mov     ah, al                  ; 
                and     al, NOT 01000000B       ; 
                out     dx, al                  ;Replace with bit 6 set to 0
                IODelay                         ; 
                in      al, dx                  ;read back value set
                xchg    al, ah                  ; 
                out     dx, al                  ;Replace original value
                xchg    ah, al                  ; 
                test    al, 01000000B           ;if bit IS NOT zero,
                jnz     WhichWDExit             ;  its a 90c00
                or      al, 01000000B           ; 
                out     dx, al                  ;Replace with bit 6 set to 1
                IODelay                         ; 
                in      al, dx                  ;read back value set
                xchg    al, ah                  ; 
                out     dx, al                  ;Replace original value
                xchg    ah, al                  ; 
                test    al, 01000000B           ;if bit IS zero,
                jz      WhichWDExit             ;  its a 90c00

                inc     bx                      ;Assume WD90c11

                mov     dx, 03c4h               ;Look for scratch pad reg
                mov     ax, 0aa08h              ;  at index 08h, found on
                out     dx, ax                  ;  90c30, not 90c11
                IODelay                         ; 
                inc     dl                      ; 
                in      al, dx                  ; 
                cmp     al, 0aah                ; 
                jnz     WhichWDExit             ;Its probably a 90c11

                inc     bx                      ;Call it a WD90c30

WhichWDExit:
                push    bx                      ;Save return code

                mov     al, 1eh                 ; 
                mov     dx, 3c3h                ; 
                out     dx, al                  ;Enter setup mode

                mov     al, 0                   ; 
                mov     dx, 0103h               ; 
                out     dx, al                  ;Disable extended registers

                mov     al, 0eh                 ; 
                mov     dx, 3c3h                ; 
                out     dx, al                  ;Exit setup mode

                pop     ax                      ; 

                ExitProc

EndProc         WhichWesternDig


;/***************************************************************************
;*
;* FUNCTION NAME = WhichATI
;*
;* DESCRIPTION   = Determine ATI chip version
;*
;*                  ATI Wonder boards V3, V4, V5 use ATI18800-x chips
;*                                    V6, +, XL  use ATI28800-x chips
;*
;* INPUT         = None
;*
;* OUTPUT        = AX = 1     ATI18800-x chip
;*                 AX = 2     ATI28800-x chip
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       WhichATI
                mov     ebx, 0c0043h            ; 
                mov     bl, byte ptr [ebx]      ;Chip version number
                sub     bl, '0'                 ; 
                mov     ax, 1                   ; 
                cmp     bl, 5                   ;          
                jb      @F                      ; 
                inc     ax                      ; 
@@:             ExitProc
EndProc         WhichATI


;/***************************************************************************
;*
;* FUNCTION NAME = IdentifySVGA
;*
;* DESCRIPTION   = Identify a super VGA adapter in the system, if one exists.
;*
;*                 extern pascal USHORT IdentifySVGA(VOID);
;*
;*                 Call from 'C' as:
;*
;*                    USHORT svga, ChipType, AdapterType;
;*
;* INPUT         = svga        = IdentifySVGA();
;*                 ChipType    = svga & 0x0f;
;*                 AdapterType = svga >> 8;
;*
;* OUTPUT        =                                  Return in AH        AL
;*                                                  (AdapterType)   (ChipType)
;*                 o Indeterminate chip set               0              0
;*
;*                 o Headland/Video 7     HT205           1              1
;*                                        HT208           1              2
;*                                        HT209           1              3
;*
;*                 o Trident Microsystems 8800            2              1
;*                                        8900            2              2
;*
;*                 o Tseng                ET3000          3              1
;*                                        ET4000          3              2
;*
;*                 o Western Digital      WD90c00         4              1
;*                                        WD90c11         4              2
;*                                        WD90c30         4              3
;*
;*                 o ATI                  ATI18800        5              1
;*                                        ATI28800        5              2
;*
;* RETURN-NORMAL =
;* RETURN-ERROR  =
;*
;**************************************************************************/

Procedure       IdentifySVGA

                LocalVar AdapterType,USHORT
                LocalVar ChipType,USHORT

                EnterProc

                push    ds
                pushad                  ; 

                mov     word ptr [AdapterType], 0

                call    IsSpeedWay      ;                               ;          
                jnz     @F              ;                               ;          
                mov     word ptr [AdapterType], 6                       ;          
                mov     ax, 1           ;                               ;          
                jmp     short SetChipType                               ;          

@@:             call    IsVideo7        ; 
                jnz     @F              ; 
                mov     word ptr [AdapterType], 1
                call    WhichVideo7     ; 
                jmp     short SetChipType

@@:             call    IsTrident       ; 
                jnz     @F              ; 
                mov     word ptr [AdapterType], 2
                call    WhichTrident    ; 
                jmp     short SetChipType

@@:             call    IsTseng         ; 
                jnz     @F              ; 
                mov     word ptr [AdapterType], 3
                call    WhichTseng      ; 
                jmp     short SetChipType

@@:             call    IsWesternDig    ; 
                jnz     @F              ; 
                mov     word ptr [AdapterType], 4
                call    WhichWesternDig ; 
                jmp     short SetChipType

@@:             call    IsATI           ; 
                jnz     @F              ; 
                mov     word ptr [AdapterType], 5
                call    WhichATI        ; 

SetChipType:    mov     ChipType, ax    ; 
@@:

SVGAExit:       popad                   ; 
                pop     ds              ; 

                mov     al, byte ptr [ChipType]
                mov     ah, byte ptr [AdapterType]

                ExitProc

EndProc         IdentifySVGA


        EndCode     PRIVATE,SWAP,PASCAL


        END
