;*DDK*************************************************************************/
;
; COPYRIGHT    Copyright (C) 1995 IBM Corporation
;
;    The following IBM OS/2 WARP source code is provided to you solely for
;    the purpose of assisting you in your development of OS/2 WARP device
;    drivers. You may use this code in accordance with the IBM License
;    Agreement provided in the IBM Device Driver Source Kit for OS/2. This
;    Copyright statement may not be removed.;
;*****************************************************************************/
;***********************************************************************
;    Module	      : EDDJTHNK.INC
;
;    Description      : PM Display Driver DBCS handling routine
;			Macro Definitions for 32->16 Thunk
;
;    Created	      : 11/29/91
;    Author	      : Yohji Nakamura (JL04328 at YMTVM6)
;
;    Notes	      :
;	      - In the Usage description for each macro, <> means
;		optional parameters.  For example,
;			SegPtr reg, src <, rpl, nochk>
;		means 'reg' and 'src' are required, but 'rpl' and
;		'nochk' are optional.  If you want to supply
;		'nochk' but not 'rpl', specify:
;			SegPtr EAX, EBX, <>, NOCHK
;
;	      - Macros defined in this file are:
;			T32Begin
;			T32End
;			GetRPL
;			T32Stack
;			T32EndStack
;			SegPtr
;			FlatPtr
;			PushArg/PushArgD/PushArgW
;			T32Call
;
;		3 macros below are private macros also defined here:
;			SetRegs
;			SetLocalVar
;
;    History	      :
;		Created, based on THKMACS.INC.		(11/29/91,YN)
;
;			OCO Source Materials
;			XXXX-XXX
;***********************************************************************
	.386p

CODE32	SEGMENT DWORD USE32 PUBLIC 'CODE'
EXTRN	THK32ALLOCSTACK:NEAR
EXTRN	THK32FREESTACK:NEAR
CODE32	ENDS


;======================================================================
; SetRegs
;		Set name of registers to local vars (?r1, ?r2...)
;	Usage:
;		SetRegss reglist
;			reglist : list of regs e.g. <AX, BX>
;======================================================================
SetRegs macro r1, r2, r3, r4, r5

	?r1 equ <r1>
	?r2 equ <r2>
	?r3 equ <r3>
	?r4 equ <r4>
	?r5 equ <r5>
endm

;======================================================================
; SetLocalVar
;		Set local variable value with default handling
;		(value and default should be string.)
;	Usage :
;		SetLocalVar local, value, default
;
;		if 'value' is non-blank, value is set to local,
;		otherwise default is set to local.
;======================================================================
SetLocalVar macro localv, value, default
	.errb <localv>
	ifb <value>		       ;; if given value is blank
	  localv equ <default>	       ;; set default value
	else			       ;;
	  localv equ <value>	       ;;
	endif			       ;;
endm				       ;;

;======================================================================
; GetRPL
;		move RPL bits in reg for later use
;
;	Usage :
;		GetRPL reg, src
;
;			reg :	16 bit reg or memory variable to hold
;				RPL bits.  (should be reg for speed)
;			src :	Selector from where RPL is retrieved
;======================================================================
GetRPL	macro	reg,src 	       ;; get RPL from source selector
	.errb	<reg>		       ;;
	.errb	<src>

	mov	reg,src
	and	reg,3
	or	reg,4		       ;; force LDT bit on
	?rplreg equ <reg>
endm


;======================================================================
; SegPtr
;		Convert 0:32 ptr to 16:16 ptr
;	Usage :
;		SegPtr reg, src <, rpl, nochk>
;
;			reg :	32-bit reg where 16:16 ptr is returned
;			src :	0:32 pointer.  May be memory.
;				May be equal to reg.
;			rpl :	RPL holder set with GetRPL macro.
;				default is reg most recently used as
;				destination of GetRPL macro.
;			nochk : if any value is supplied, NULL ptr check
;				is suppressed.	default is not-suppress.
;======================================================================
SegPtr	macro	reg,src,rpl,nochk
	local	lab1
	.errb	<reg>
	.errb	<src>

	SetLocalVar <?rpl>, <rpl>, ?rplreg
	.errb	<?rpl>		       ;; when rpl & ?rplreg are blank

	?r substr <reg>,2

	ifdifi	<src>, <reg>
	  mov	reg, src
	endif

	ifb    <nochk>
	  or	reg, reg
	  jz	short lab1
	endif

	ror   reg,16			  ;;do CRMA on pointer
	shl   ?r,3
	or    ?r,?rpl
	rol   reg,16
lab1:
endm


;======================================================================
; FlatPtr
;		Convert segmented 16:16 ptr to flag 0:32 ptr
;
;	Usage:	FlatPtr    reg, hi <,lo>
;			reg :	destination 32-bit reg
;			hi  :	source 16:16 ptr,
;				  or segment part if lo is supplied
;				  may be identical to reg.
;			lo  :	offset part
;======================================================================
FlatPtr macro	reg,hi,lo
	  ?r substr <reg>,2
	  ifb <lo>
	    % ifdifi <reg>, <hi>
	    mov   reg,hi		  ;;swap high/low words of eax
	    endif
	    ror   reg,16		  ;;swap high/low words of eax
	    shr   ?r,3			  ;;discard LDT and RPL bits
	    rol   reg,16		  ;;unswap high/low words of eax
	  else
	    % ifidni <lo>,<?r>
	    rol   reg,16
	    endif
	    % ifdifi <hi>,<?r>
	    mov   ?r,hi
	    endif
	    shr   ?r,3			  ;;discard LDT and RPL bits
	    rol   reg,16		  ;;unswap high/low words of eax
	    % ifdifi <lo>,<?r>
	    mov   ?r,lo
	    endif
	  endif
endm


;======================================================================
; PushArg/PushArgW/PushArgD
;		Push argument
;	Usage:
;		PushArg  size, arg
;		PushArgW arg
;		PushArgD arg
;
;			size :	size of each item in arg (e.g. WORD)
;			arg  :	item or list of items to be pushed
;				items in list are pushed in L to R, and
;				must be in same size.
;======================================================================
PushArg macro	size,arg
	?size = type ( size ptr 0 )
	irp entry, <arg>
	  push entry		       ;; push each entry

	  ?argbyte=?argbyte + ?size    ;; and remember total bytes
	endm
endm

PushArgW macro	arg
	PushArg WORD, <arg>
endm

PushArgD macro	arg
	PushArg DWORD, <arg>
endm

;======================================================================
; T32Begin
;		Prepare for 32->16 thunk.  Just to define segments.
;
;	Usage:
;		T32Begin cseg, c32seg, dseg, d32seg
;
;			cseg  : name of 16-bit code segment
;			c32seg: name of 32-bit code segment
;			dseg  : name of 16-bit data segment
;			d32seg: name of 32-bit data segment
;	Note:
;	      - Since this macro only defines segment names, it can
;		also be called in the middle of procedure to switch
;		16 bit segment.  For example,
;		----------------------------------
;		T32Begin C16, C32, D16, D32
;		T32Stack ....
;		T32Call  ....
;
;     ===>	T32End			      (<- Just for consistency)
;     ===>	T32Begin C16_2, C32, D16_2, D32
;
;		T32Call  ....
;
;		T32EndStack ....
;		T32End
;		-------------------------------------
;		(32-bit segment should not be changed, since it is used
;		 in T32Stack macro.)
;
;======================================================================
T32Begin  macro   cseg,c32seg,dseg,d32seg
	.errb	<cseg>
	.errb	<c32seg>
	.errb	<dseg>
	.errb	<d32seg>

	?cseg	equ <cseg>
	?c32seg equ <c32seg>
	?dseg	equ <dseg>
	?d32seg equ <d32seg>

	?argbyte = 0
endm


;======================================================================
; T32Stack
;		Switch to 16-bit stack guaranteeing minimum amount of
;		space for 16-bit proc's use.  Old SS:ESP is saved on
;		the top of new stack.  This will be restored by
;		T32EndStack macro automagically.
;
;	Usage :
;		T32Stack src <, opt, workreg, minstk, rpl, oldss>
;			src :	bottom of new stack.  see opt also.
;			opt :	type of src.
;					PTRFLAT --> 0:32 address
;					PTRSEG	--> 16:16 address
;				default is PTRFLAT.
;			workreg :
;				32-bit work registers to be used.
;				default is <EAX,ECX,EDX>. if NOREG
;				is given, all regs are preserved.
;			minstk: minimum size of stack that should be
;				guaranteed. default is 1024 (bytes)
;				Only valid when PTRFLAT is given.
;			rpl :	RPL holder set with GetRPL macro.
;				default is reg most recently used as
;				destination of GetRPL macro.
;			oldss : name of segment register which will
;				contain old SS value to access vars on
;				stack.	default is none.
;
;	Note :
;		When NOREG is specified, this macro generates the stack
;		frame below:
;
;      32-bit stack		 <---- ESP before this macro is called
;	       Ĵ		(-> ?work1)
;	        work reg save	
;	        (8 or 12 bytes)<---- if ESP is given as src and if no
;	       Ĵ      bumping happens, new SS:SP points
;	        new 16 bit	      here.
;	           SS:SP	<---- Current ESP
;	       Ĵ
;
;		Then stack switching is done with 'LSS SP,[ESP]'.
;		Original SS:ESP is pushed on the top of the 16 bit stack.
;
;	16-bit stack		 <---- SS:SP just after stack switching
;	       Ĵ
;	        original SS:ESP<---- SS:SP when this macro exits
;	       Ĵ
;
;		At last, all work registers are restored through original
;		ESP stored in ?work1.  Since SS:ESP on the bottom of
;		the new stack is one BEFORE work regs are pushed,
;		loading it in T32EndStack macro effectively removes
;		all work regs saved in the stack.
;======================================================================
T32Stack macro	src,opt,workreg,minstk,rpl,oldss
	local	l1
	.errb	<src>

	SetLocalVar <?opt>, <opt>, <PTRFLAT>
	SetLocalVar <?oldss>, <oldss>, <NONE>

%	ifidni <?opt>, <PTRFLAT>
	  SetLocalVar <?rpl>, <rpl>, %?rplreg
	  .errb <?rpl>
	endif

	SetLocalVar <?minstk>, <minstk>, <1024>
				       ; 
	.errnb	<workreg>
				       ; 
	  push	0000		       ;; used for later stack comp.
	  mov	ecx, esp

	  mov	ax, ss
%	ifdifi	<?oldss>, <NONE>       ;; if OLDSS is given
	  mov	?oldss, ax	       ;; save this reg for later use
	endif

	  cmp	ax, seg FLAT:?d32seg   ;; standard 32 bit stack?
	  jne	short l1	       ;; if not, skip bumping code

	  mov	eax, esp
	  cmp	ax, ?minstk	       ;; check lower 16 bit only
	  jae	short l1	       ; 

	  pop	  edx			  ;old stack comparator
	  push	  ecx
	  call	  THK32ALLOCSTACK	  ; bump the stack down
	  pop	  ecx
	  push	  eax			  ;old stack comparator on old stack
	  mov	  esp,eax


  ;Insure that 1 DWORD exists on new stack
	  push	  eax
l1:
	  push	  es		       ; 
	  push	  ecx
	  push	  ebx
	  push	  esi
	  push	  edi
	  push	  ebp			  ; save ebp
	  mov	  eax,esp		  ; save current esp
	  push	  ss
	  push	  eax


	  mov	  edx,ss		  ; get cpl bits to use as dpl bits
	  and	  edx,3 		  ; mask off rest of register

	  or	  dl,4			  ;Force LDT bit on

;* Convert SS:ESP to 16-bit SS:SP.

	mov	eax,esp 		; use CRMA on ESP to get SS
	ror	eax,16
	shl	eax,3
	or	al,dl
	mov	ss,eax
endm

;======================================================================
; T32EndStack
;		switch back to 32 bit stack
;	Usage :
;		T32StackEnd
;======================================================================
T32EndStack macro			;;switch back to 0:32 stack
;* Restore 32-bit SS:ESP - it is on top of stack.

	movzx	esp,sp			; make sure that esp is correct
	lss	esp,[esp]
	pop	ebp
	pop	edi
	pop	esi
	pop	ebx
	pop	ecx
	pop	es

;* 32-bit return code.

	mov	esp, ecx	; get the old stack
	pop	ecx			;get the new stack
	cmp	ecx,0			;is it new
	jz	short @f	       ; 
	push	edx
	call	THK32FREESTACK		;deallocate the memory
	pop	edx
@@:
endm

;======================================================================
; T32Call
;		call 16:16 far proc through thunking code
;
;	Usage :
;		T32Call name <, lang, opt, cseg16>
;
;			name :	name of 16 bit proc
;				  e.g. some16proc
;				or name of ptr to 16 bit proc
;				  e.g. [ptr_some16proc]
;
;			lang :	calling convention of 16-bit proc
;				(PASCAL or C) Note if 'C' is specified,
;				all arguments in stack should be pushed
;				with PushArg macro.  default is PASCAL.
;				Also note that when 'C', all values
;				which are PushArg-ed after most recent
;				T32Begin or T32Call.
;
;			opt  :	valid option values are:
;				  DSSET  : Change DS to dseg before
;					   call, and restore to dseg32
;					   after call (default)
;				  DSKEEP : Do not alter DS value
;
;			csegR2 : name of Ring-2 16 bit code segment.
;				If given, it is used to transfer from
;				32-bit to 16-bit code when current level
;				is ring 2.  If current level is ring 3,
;				or if this option is not given,
;				'cseg' specified with T32Begin is used.
;
;======================================================================
T32Call macro	name, lang, opt, csegR2
	local	C16R2, C16R3, C32R2, C32RET

	.errb	<name>
	SetLocalVar <?lang>, <lang>, <PASCAL>
	SetLocalVar <?opt>, <opt>, <DSSET>

% ?cseg segment
  C16R3 label far
% ?cseg ends

	ifnb	<csegR2>	       ;; if R2 CS is given, use it too
% csegR2 segment
  C16R2 label far
% csegR2 ends

	.errb	<?rplreg>	       ;; if RPL holder must exist
	test	?rplreg, 1	       ;; assumes Ring 2 or 3
	je	short C32R2
	endif			       ;; end of if: R2 CS is given

;;	  ifb	  <csegR2>
;;	  ifdef   FIREWALLS
;;	  test	  ?rplreg, 1
;;	  jnz	  @f
;;	  int	  3
;;@@:
;;	  endif
;;	  endif

	jmp	far ptr C16R3		;;jump to 16-bit code segment

	ifnb	<csegR2>
C32R2:
	jmp	far ptr C16R2
	endif

C32RET: 			       ;; come back to here

% ?cseg segment
      % assume	cs:?cseg
	.errnz	($ - C16R3)		;;make sure label points here

%	ifidni	<?opt>, <DSSET>
	push	seg ?dseg
	pop	ds
      % assume	ds:?dseg
	endif

	call	name			;;call 16-bit API

%	ifidni	<?lang>, <C>
	  if	?argbyte	       ;; only if any var was pushed
	    add   sp, ?argbyte
	  endif
	endif

%	ifidni	<?opt>, <DSSET>
	push	seg FLAT:?d32seg
	pop	ds
	assume	ds:FLAT
	endif

	jmp	far ptr FLAT:C32RET    ;;jump back to 32-bit code segment
% ?cseg ends

	ifnb	<csegR2>	       ;; only if Ring2 CS is also given

% csegR2 segment
      % assume	cs:csegR2
	.errnz	($ - C16R2)		;;make sure label points here

%	ifidni	<?opt>, <DSSET>
	push	seg ?dseg
	pop	ds
      % assume	ds:?dseg
	endif

	call	name			;;call 16-bit API

%	ifidni	<?lang>, <C>
	  if	?argbyte	       ;; only if any var was pushed
	    add   sp, ?argbyte
	  endif
	endif

%	ifidni	<?opt>, <DSSET>
	push	seg FLAT:?d32seg
	pop	ds
	assume	ds:FLAT
	endif

	jmp	far ptr FLAT:C32RET    ;;jump back to 32-bit code segment
% csegR2 ends
	endif

	?argbyte = 0		       ;; reset argument count

	assume	cs:FLAT
endm


T32End	macro				;;switch back to 0:32 stack
endm
