/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**********************************************************************/
/*                                                                    */
/*   Module          = EDDJFONT.C                                     */
/*                                                                    */
/*   Description     = PM Display Driver DBCS handling routine        */
/*                     Subfunctions for DBCS Font Managing            */
/*                                                                    */
/*   Function        =                                                */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/*   CHANGE HISTORY:                                                  */
/*                                                                    */
/**********************************************************************/



#define INCL_DOSMODULEMGR
#define INCL_ERRORS
#include <eddinclt.h>

#include <memman.h>
#include <eddjdef.h>
#include <eddjfm.h>
#include <fontdrvr.h>                  // for LOAD_FONT_DRIVER flags
#include <eddjfont.h>

extern ULONG QueryCapsTable[];         // driver's capability table
extern PFOCAFONT pResourceFont;        // ptr to SBCS res. font

#define ROUNDDIV( x, y )  ( ((x) + (y)/2) / (y) )

/*====================================================================*/
/* This macro converts return code from DBCS Font Manager (i.e.       */
/* FM_SUCCESS or FM_ERROR) into our format (i.e. TRUE or FALSE)       */
/*====================================================================*/
#if FM_SUCCESS+1 != TRUE
  #error Assertion on SUCCESS code failed!  (Thought 0, was FM_SUCCESS)
#endif

#if FM_ERROR+1 != FALSE
  #error Assertion on ERROR code failed!  (Thought -1, was FM_ERROR)
#endif

#define ISSUCCESS( code )       (code+1)

/*===================== Exported Routine =============================*/
/*      eddj_FixFontMetrics                                           */
/*                                                                    */
/* Fix point sizes and resolutions stored in font metrics when that   */
/* font has diffrent resolution from display's resolution.            */
/*                                                                    */
/* Entry       :        Pointer to FONTMETRICS structure              */
/*                                                                    */
/* Returns     :        None.  Values in given font metrics are       */
/*                      changed                                       */
/*                                                                    */
/* Error Returns :      None                                          */
/*                                                                    */
/* Note        :                                                      */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
VOID eddj_FixFontMetrics( PFONTMETRICS pMetrics )
{                                      /* top of func: eddj_FixFontMet*/
  ULONG ulDispRes;                     /* display's resolution (Y)    */
  ULONG ulFontRes;                     /* font's resolution (Y)       */
                                       /*                             */
  ulDispRes = QueryCapsTable[CAPS_VERTICAL_FONT_RES];
  ulFontRes = (ULONG)pMetrics->sYDeviceRes;

  if ( ulDispRes == ulFontRes )
    {                                  /* if 2 resolution is identical*/
      return;                          /* then need to do nothing     */
    }                                  /* end of if:                  */

  /*------------------------------------------------------------------*/
  /* resolution                                                       */
  /*------------------------------------------------------------------*/
  pMetrics->sYDeviceRes = (SHORT)ulDispRes;
  pMetrics->sXDeviceRes =
    (SHORT) ROUNDDIV( (ULONG)pMetrics->sXDeviceRes * ulDispRes,
                      ulFontRes );     /* adjust X-res and round it   */

  /*------------------------------------------------------------------*/
  /* point sizes...                                                   */
  /*------------------------------------------------------------------*/
  pMetrics->sNominalPointSize =
    (SHORT) (ROUNDDIV( (ULONG)pMetrics->sNominalPointSize * ulFontRes,
                       ulDispRes * 10 ) * 10);

  pMetrics->sMinimumPointSize =
    (SHORT) (ROUNDDIV( (ULONG)pMetrics->sMinimumPointSize * ulFontRes,
                       ulDispRes * 10 ) * 10);

  pMetrics->sMaximumPointSize =
    (SHORT) (ROUNDDIV( (ULONG)pMetrics->sMaximumPointSize * ulFontRes,
                       ulDispRes * 10 ) * 10);

}                                      /* end of func: eddj_FixFontMet*/

/*===================== Exported Routine =============================*/
/*      eddj_LoadFontManager                                          */
/*                                                                    */
/* Load DBCS Font Manager and create dispatch table for later call    */
/*                                                                    */
/* Entry       :        None                                          */
/*                                                                    */
/* Returns     :        Returns TRUE if completes.                    */
/*                                                                    */
/* Error Returns :      Returns FALSE if error                        */
/*                                                                    */
/* Note        :                                                      */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
CHAR szFMname[] = "PMNLSFM";
CHAR szFMDispatchTable[] = "DDFM_DISPATCH_TABLE32";
FMDISPATCH32 FMDispatchTable;          /* copy of FM dispatch table   */
#ifndef _8514                                               
BOOL eddj_LoadFontManager( VOID )
{                                      /* top of func: eddj_LoadFontMa*/
  HMODULE hmod;                        /*                             */
  CHAR szErrBuf[100];                  /*                             */
  PFMDISPATCH32 pTable;                /*                             */
                                       /*                             */
  if ((NO_ERROR != DosLoadModule( szErrBuf,
                                  sizeof(szErrBuf),
                                  szFMname,
                                  &hmod ) ) ||
      (NO_ERROR != DosQueryProcAddr( hmod,
                                     0,
                                     szFMDispatchTable,
                                     &(PFN)pTable ) ))
    {
      return FALSE;                    /*                             */
    }                                  /* end of if:                  */

  FMDispatchTable = *pTable;           /* copy dispatch table         */
  return TRUE;                         /*                             */
}                                      /* end of func: eddj_LoadFontMa*/

#else 

// eddj_LoadFontManger is replaced by DbcsLoadFontManager (dbcsfont.asm).
// For 8514 and S3 driver, this function is called with ring-2
// segment, in order to do in/out operation (KlugeReset).
// Hence we cannot load the font manager at this context.

BOOL eddj_LoadFontManager (VOID)
{
    extern BOOL APIENTRY DbcsLoadFontManager (VOID);

    return !DbcsLoadFontManager ();
}
#endif 

/*===================== Exported Routine =============================*/
/*      eddj_GetFontList                                              */
/*                                                                    */
/* Returns array of FMFONTINFO structure containing all available     */
/* fonts managed by DBCS font manager.                                */
/*                                                                    */
/* Entry       :        FM_PUBLIC and/or FM_PRIVATE                   */
/*                      Specify if public or private font should      */
/*                      be returned.                                  */
/*                                                                    */
/* Returns     :        Returns number of fonts managed by Font Mgr.  */
/*                      Flat address of FMFONTINFO array is returned  */
/*                      where ppFMFontInfo points.                    */
/*                                                                    */
/* Error Returns :      When any error is found, 0 is returned.       */
/*                      *ppFMFontInfo is not altered in this case.    */
/*                                                                    */
/* Note        :        FMFONTINFO array is stored in memory allocated*/
/*                      using AllocateMemory() with SHARED option.    */
/*                      Caller should free it using FreeMemory().     */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
ULONG eddj_GetFontList( PFMFONTINFO *ppFMFontInfo, ULONG flOptions )
{                                      /* top of func: eddj_GetFontLis*/
  ULONG ulFontNum;                     /*                             */
  PFMFONTINFO pFInfo;                  /*                             */
                                       /*                             */
  ulFontNum = FMDispatchTable.Fm32QueryFonts( flOptions,
                                              NULL,
                                              NULL,
                                              1,
                                              0xFFFFFFFF );
                                       /* query #fonts at first       */
  /*------------------------------------------------------------------*/
  /* allocate buffer for font info                                    */
  /*------------------------------------------------------------------*/
  if (0 == ulFontNum)                  /* if no fonts found           */
    {                                  /*                             */
      return 0;                        /* return ERROR                */
    }                                  /* end of if:                  */

  pFInfo = AllocateMemory( sizeof(FMFONTINFO) * ulFontNum,
                           MT_FONTLIST, MO_SHARED );
  if (NULL == pFInfo)                  /* if cannot allocate buffer   */
    {                                  /*                             */
      return 0;                        /*                             */
    }                                  /* end of if:                  */

  /*------------------------------------------------------------------*/
  /* get detail information for fonts                                 */
  /*------------------------------------------------------------------*/
  if (FM_SUCCESS !=
        FMDispatchTable.Fm32QueryFonts( flOptions,
                                        pFInfo,
                                        NULL,
                                        1,
                                        ulFontNum ))
    {                                  /* from VGA source, it seems   */
                                       /* that this routine returns   */
                                       /* 0/non-0 (error), rather than*/
                                       /* # of fonts like when buffer */
                                       /* address given is NULL...    */
      FreeMemory( pFInfo );            /*                             */
      return 0;                        /*                             */
    }                                  /* end of if:                  */
                                       /* get details on all fonts    */
  *ppFMFontInfo = pFInfo;              /*                             */
  return ulFontNum;                    /*                             */
}                                      /* end of func: eddj_GetFontLis*/

/*===================== Exported Routine =============================*/
/*      eddj_ValidateSystemFont                                       */
/*                                                                    */
/* Prepare DBCS Font Manager and DBCS Fonts for caller's process so   */
/* that they can be accessed in this process.                         */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/* Entry       :        None                                          */
/*                                                                    */
/* Returns     :        None                                          */
/*                                                                    */
/* Error Returns :      None.                                         */
/*                                                                    */
/* Note        :                                                      */
/*            - This function does no error checking, assuming that   */
/*              all operation, i.e. loading font manager and querying */
/*              all public fonts, have successfully completed in other*/
/*              process.                                              */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
#ifndef _8514                                               
VOID eddj_ValidateSystemFont( VOID )
{                                      /* top of func: eddj_ValidateSy*/
  HMODULE hmod;                        /*                             */
  CHAR szErrBuf[100];                  /*                             */
                                       /*                             */

  DosLoadModule( szErrBuf, sizeof(szErrBuf), szFMname, &hmod );
                                       /* to ensure DLL is mapped in  */
                                       /* this process's memory space */
  FMDispatchTable.Fm32QueryFonts( FM_PUBLIC,
                                  NULL,
                                  NULL,
                                  1,
                                  0xFFFFFFFF );
                                       /* to ensure all fonts are     */
                                       /* mapped in process's memory  */
}                                      /* end of func: eddj_ValidateSy*/

#else 

VOID eddj_ValidateSystemFont (VOID)
{
    HMODULE hmod;

    extern APIRET APIENTRY DbcsDosLoadModule (PSZ pszName, PHMODULE pHMod);


    DbcsDosLoadModule (szFMname, &hmod);

    FMDispatchTable.Fm32QueryFonts (FM_PUBLIC,
                                    NULL,
                                    NULL,
                                    1,
                                    0xFFFFFFFF);
}
#endif 

/*===================== Exported Routine =============================*/
/*      eddj_RealizeFont                                              */
/*                                                                    */
/* Realize DBCS/SBCS device font using given font attributes.         */
/*                                                                    */
/* Entry       :        Flat pointer to font attribute structure.     */
/*                                                                    */
/* Returns     :        Flat pointer to FOCAFONT structure found.     */
/*                                                                    */
/* Error Returns :      Returns NULL if no matched font is found or   */
/*                      FontMgr returns some error.                   */
/*                                                                    */
/* Note        :                                                      */
/*            - Before calling this routine, caller must set-up       */
/*              contents of pResourceFont so that it points to        */
/*              FOCAFONT structure of SBCS resource font.             */
/*                                                                    */
/*            - In positive lMatch case, this routine should not be   */
/*              called.  (Caller should immediately returns NO_FOUND  */
/*              to engine.)                                           */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
PFOCAFONT eddj_RealizeFont( PFATTRS pFattrs )
{                                      /* top of func: eddj_RealizeFon*/
  FMFONTINFO FInfo;                    /*                             */
  LONG lRc;                            /*                             */

  /*------------------------------------------------------------------*/
  /* First we will check if requested font is SBCS resource font. I.e.*/
  /*    case 1:  Special match value and facename match               */
  /*    case 2:  Match value is 0, and other parms (facename and Max  */
  /*             baseline ext) match                                  */
  /*------------------------------------------------------------------*/
  if ( ((pFattrs->lMatch == LMATCH_SYS_SBCS) &&
        !strcmp(pFattrs->szFacename,
                pResourceFont->fmMetrics.szFacename)) ||
                                       /* case 1:                     */
       ((pFattrs->lMatch == 0) &&
        (pFattrs->lMaxBaselineExt ==
            pResourceFont->fmMetrics.yMaxBaselineExt) &&
        !strcmp(pFattrs->szFacename,
                pResourceFont->fmMetrics.szFacename)) )
    {                                  /* case 2:                     */
      return pResourceFont;            /* SBCS res font matched       */
    }                                  /* end of if:                  */

  /*------------------------------------------------------------------*/
  /* Now make DBCS Font Manager search the font                       */
  /*------------------------------------------------------------------*/
  lRc = FMDispatchTable.Fm32OpenFont( pFattrs, &FInfo );
  if (FM_FONT_FOUND == lRc)            /*                             */
    {                                  /*                             */
#ifdef OMIT
      /*--------------------------------------------------------------*/
      /* This code is to circumvent DBCS font manager's bug.          */
      /*--------------------------------------------------------------*/
      if (0 == FInfo.pFont->fdDefinitions.pCellBaseOffset)
        {                              /* if value is not given       */
          FInfo.pFont->fdDefinitions.pCellBaseOffset =
            FInfo.pFont->fmMetrics.yMaxAscender;
                                       /* we have to guess it!        */
        }                              /* end of if:                  */
      /*--------------------------------------------------------------*/
      /* This code is to circumvent DBCS font manager's bug.          */
      /*--------------------------------------------------------------*/
#endif /* OMIT */                      /*                             */

      return FInfo.pFont;              /* Spec says hfm is always same*/
                                       /*   to pFont.  So we may use  */
                                       /*   either...                 */
    }                                  /* end of if:                  */
  else                                 /*                             */
    {                                  /*                             */
      return NULL;                     /* error or not-found case     */
    }                                  /* end of else:                */
}                                      /* end of func: eddj_RealizeFon*/

/*===================== Exported Routine =============================*/
/*      eddj_UnrealizeFont                                            */
/*                                                                    */
/* Unrealize DBCS/SBCS device font having given FOCAFONT ptr.         */
/*                                                                    */
/* Entry       :        Flat pointer to FOCAFONT structure            */
/*                                                                    */
/* Returns     :        None                                          */
/*                                                                    */
/* Error Returns :      None                                          */
/*                                                                    */
/* Note        :                                                      */
/*            - If given font is engine font, do nothing              */
/*            - If given font is SBCS font, do nothing assuming that  */
/*              it is SBCS resource font                              */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
VOID eddj_UnrealizeFont( PFOCAFONT pFont )
{                                      /* top of func: eddj_UnrealizeF*/
  if (!(pFont->fmMetrics.fsDefn & FM_DEFN_GENERIC) &&
      (pFont->fmMetrics.fsTypeFlags & (FM_TYPE_MBCS | FM_TYPE_DBCS)))
    {                                  /*                             */
      FMDispatchTable.Fm32CloseFont( pFont );
    }                                  /* end of if:                  */
}                                      /* end of func: eddj_UnrealizeF*/

/*===================== Exported Routine =============================*/
/*      eddj_QueryFontProfile                                         */
/*                                                                    */
/* Query font characteristics flag (NLSCA_*) and number of glyphs Font*/
/* Mgr can return for that font at 1 call.                            */
/*                                                                    */
/* Entry       :        Flat pointer to FOCAFONT structure            */
/*                                                                    */
/* Returns     :        Font flags and cache size are returned.       */
/*                                                                    */
/* Error Returns :      None                                          */
/*                                                                    */
/* Note        :                                                      */
/*          - Assumes that no SBCS fonts are managed by Font Mgr, and */
/*            all DBCS fonts are managed by Font Mgr.                 */
/*          - If NLSCA_FONT_CACHED is not set, returns 256 as cache   */
/*            size, since it is generic limit for Font Manager call.  */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
VOID pascal eddj_QueryFontProfile( PFOCAFONT pFont,
                                   PUSHORT pFlags,
                                   PUSHORT pCacheSize )
{                                      /* top of func: eddj_QueryFontP*/
  FMFONTINFO FInfo;                    /*                             */
                                       /*                             */
  *pCacheSize = FM_CHARSTR_LIMIT;      /* this is general limit       */
                                       /*                             */
  /*------------------------------------------------------------------*/
  /* set font type flag                                               */
  /*------------------------------------------------------------------*/
  switch (pFont->fmMetrics.fsTypeFlags & (FM_TYPE_DBCS | FM_TYPE_MBCS))
    {
    case FM_TYPE_MBCS:                 /*                             */
      *pFlags = NLSCA_FONT_MBCS;       /*                             */
      break;                           /* end of case:                */

    case FM_TYPE_DBCS:                 /*                             */
      *pFlags = NLSCA_FONT_DBCS;       /*                             */
      break;                           /* end of case:                */

    default:                           /*                             */
      *pFlags = NLSCA_FONT_SBCS;       /* for SBCS font, we need do   */
      return;                          /*  nothing any more           */
      break;                           /* end of default:             */
    }                                  /* end of switch:              */

  /*------------------------------------------------------------------*/
  /* get info about cache                                             */
  /*------------------------------------------------------------------*/
  if (pFont->fmMetrics.fsDefn & FM_DEFN_GENERIC)
    {                                  /* if engine font,             */
      return;
    }                                  /* end of if:                  */

  if ((FM_SUCCESS == FMDispatchTable.
                       Fm32QueryFonts( FM_PUBLIC | FM_PRIVATE,
                                       &FInfo,
                                       pFont,
                                       0,
                                       0 )) &&
                                       /* if that font was found      */
      (FInfo.flType & FM_CACHE_FONT))  /* & it has own cache limit    */
    {                                  /*                             */
      *pFlags |= NLSCA_FONT_CACHED;    /*                             */
      *pCacheSize = min((USHORT)FInfo.cbCache, *pCacheSize);
    }                                  /* end of if:                  */
}                                      /* end of func: eddj_QueryFontP*/

/*===================== Exported Routine =============================*/
/*      eddj_LoadFontDriver                                           */
/*                                                                    */
/* Load/Unload DBCS Font Driver                                       */
/*                                                                    */
/* Entry       :      - Function ID                                   */
/*                    - Ptr to font driver's pathname                 */
/*                                                                    */
/* Returns     :        Returns TRUE if completes.                    */
/*                                                                    */
/* Error Returns :      Returns FALSE if error                        */
/*                                                                    */
/* Note        :                                                      */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
BOOL eddj_LoadFontDriver( ULONG ulFuncID, PSZ pszFontDriver )
{                                      /* top of func: eddj_LoadFontDr*/
  LONG lRc;                            /*                             */

  switch (ulFuncID)                    /*                             */
    {                                  /*                             */
    case LOAD_FD_PRIVATE:              /*                             */
      lRc = FMDispatchTable.           /*                             */
              Fm32LoadFontDriver( FM_PRIVATE, pszFontDriver );
      break;                           /* end of case:                */

    case LOAD_FD_PUBLIC:               /*                             */
      lRc = FMDispatchTable.           /*                             */
              Fm32LoadFontDriver( FM_PUBLIC, pszFontDriver );
      break;                           /* end of case:                */

    case UNLOAD_FD_PRIVATE:            /*                             */
      lRc = FMDispatchTable.           /*                             */
              Fm32UnloadFontDriver( FM_PRIVATE, pszFontDriver );
      break;                           /* end of case:                */

    case UNLOAD_FD_PUBLIC:             /*                             */
      lRc = FMDispatchTable.           /*                             */
              Fm32UnloadFontDriver( FM_PUBLIC, pszFontDriver );
      break;                           /* end of case:                */

    default:                           /*                             */
      lRc = FM_ERROR;                  /*                             */
      break;                           /* end of default:             */
    }                                  /* end of switch:              */

  return ISSUCCESS( lRc );             /* return TRUE/FALSE code      */
}                                      /* end of func: eddj_LoadFontDr*/

/*===================== Exported Routine =============================*/
/*      eddj_ValidateFontSeg                                          */
/*                                                                    */
/* Get DBCS font image and/or font information from DBCS Font Manager */
/*                                                                    */
/* Entry       :                                                      */
/*            - Number of glyph index.                                */
/*                      if <0 only font definintion is returned       */
/*                      if >0 font definition and font image returned */
/*            - Flat ptr to FOCAFONT structure                        */
/*            - Flat ptr to Glyph Index array                         */
/*                                                                    */
/*                                                                    */
/* Returns     :        Returns TRUE if completes.                    */
/*                                                                    */
/* Error Returns :      Returns FALSE if error                        */
/*                                                                    */
/* Note        :                                                      */
/*            - Number of glyph index must be <= cache size returned  */
/*              from QueryFontProfile                                 */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
BOOL eddj_ValidateFontSeg (LONG lCount,
                           PFOCAFONT pFont,
                           PUSHORT pusGlyphIndex)
{
    ULONG ulFlags;

    if (lCount > 0)
    {
        // font def & image requested

        ulFlags = FM_WITH_FONT;
    }
    else
    {
        // only font def requested

        ulFlags = FM_WITHOUT_FONT;

        // get abs value
        lCount = -lCount;
    }

    if (FM_SUCCESS == FMDispatchTable.Fm32CharStr (pFont,
                                                   ulFlags,
                                                   pusGlyphIndex,
                                                   lCount))
    {
        return 1L;
    }
    else
    {
        // Indicate that the caller should do the default substitution.

        return (BOOL) -1L;
    }
}

/*===================== Exported Routine =============================*/
/*      eddj_MapCharGlyph                                             */
/*                                                                    */
/* Convert MBCS codepoints into glyph index arary                     */
/*                                                                    */
/* Entry       :                                                      */
/*            - Current codepage                                      */
/*            - Length of string in bytes                             */
/*            - Flat ptr to FOCAFONT structure                        */
/*            - Flat ptr to MBCS string                               */
/*                                                                    */
/* Returns     :        Returns TRUE if completes.                    */
/*            - Input string is also replaced by glyph index array    */
/*                                                                    */
/* Error Returns :      Returns FALSE if error                        */
/*                                                                    */
/* Note        :                                                      */
/*            - We use single buffer for both source string input and */
/*              destination glyph-index output.  So this function     */
/*              need to be called when CP is DBCS!  (Never call with  */
/*              SBCS or MBCS CP)  Otherwise source buffer may be over-*/
/*              written with ourput.                                  */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
BOOL eddj_MapCharGlyph( ULONG ulCodePage,
                        ULONG ulLength,
                        PFOCAFONT pFont,
                        PVOID pchString )
{                                      /* top of func: eddj_MapCharGly*/
  return ISSUCCESS( FMDispatchTable.
                      Fm32MapCharGlyph( pFont,
                                        ulCodePage,
                                        pchString,
                                        pchString,
                                        ulLength ) );
}                                      /* end of func: eddj_MapCharGly*/

/*===================== Exported Routine =============================*/
/*      eddj_QueryMatchNum                                            */
/*                                                                    */
/* Returns match number for given SBCS/DBCS font                      */
/*                                                                    */
/* Entry       :        Flat ptr to FOCAFONT structure                */
/*                                                                    */
/* Returns     :        Match number (negative) is returned.          */
/*                                                                    */
/* Error Returns :      Returns 0 if not found                        */
/*                                                                    */
/* Note        :                                                      */
/*                                                                    */
/* Calls       :                                                      */
/*                                                                    */
/*====================================================================*/
LONG eddj_QueryMatchNum( PFOCAFONT pFont )
{                                      /* top of func: eddj_QueryMatch*/
  FMFONTINFO FInfo;                    /*                             */
                                       /*                             */
  /*------------------------------------------------------------------*/
  /* handle engine font and SBCS font first                           */
  /*------------------------------------------------------------------*/
  if (pFont->fmMetrics.fsDefn & FM_DEFN_GENERIC)
    {                                  /* if engine font,             */
      return 0;                        /*   then we do not this font  */
    }                                  /* end of if:                  */

  if (!(pFont->fmMetrics.fsTypeFlags & (FM_TYPE_MBCS | FM_TYPE_DBCS)))
    {                                  /* if SBCS font                */
      if (pFont == pResourceFont)      /*   and it is our res font    */
        {                              /*                             */
          return LMATCH_SYS_SBCS;      /*                             */
        }                              /* end of if:                  */
      else                             /* otherwise unknown device fnt*/
        {                              /*                             */
          return 0;                    /*                             */
        }                              /* end of else:                */
    }                                  /* end of if:                  */

  /*------------------------------------------------------------------*/
  /* now we know it is DBCS Font Mgr font...                          */
  /*------------------------------------------------------------------*/
  if (FM_SUCCESS ==
        FMDispatchTable.
          Fm32QueryFonts( FM_PUBLIC | FM_PRIVATE,
                          &FInfo,
                          pFont,
                          1, 1 ))
    {                                  /* if the font was found       */
      return FInfo.lMatch;             /*                             */
    }                                  /* end of if:                  */
  else                                 /*                             */
    {                                  /*                             */
      return 0;                        /*                             */
    }                                  /* end of else:                */
}                                      /* end of func: eddj_QueryMatch*/

